// ===========================================================================
// $Source: d:/source/master/crusher/src/md2.cpp,v $
// $Revision: 1.21 $
// $Date: 1998/08/04 02:55:42 $
// ===========================================================================
// Copyright (C) 1998 Tom Conder <blitz@gazpacho.net>. All Rights Reserved.
//
// BY USING ANY PORTION OF THIS SOFTWARE, YOU AGREE TO THE FOLLOWING
// TERMS AND CONDITIONS:
// 
// Tom Conder, "THE AUTHOR", grants you, "THE USER", a non-exclusive,
// royalty free, license to use this software in source and binary code
// form, provided the user does not utilize the software in a manner
// which is disparaging to the author and the user acknowledges the
// author in any derivative work.
// 
// This software is provided "AS IS," without a warranty of any kind. ALL
// EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND WARRANTIES, INCLUDING
// ANY IMPLIED WARRANTY OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
// PURPOSE OR NON-INFRINGEMENT, ARE HEREBY EXCLUDED. THE AUTHOR SHALL NOT
// BE LIABLE FOR ANY DAMAGES SUFFERED BY THE USER AS A RESULT OF USING,
// MODIFYING OR DISTRIBUTING THE SOFTWARE OR ITS DERIVATIVES. IN NO EVENT
// WILL THE AUTHOR BE LIABLE FOR ANY LOST REVENUE, PROFIT OR DATA, OR FOR
// DIRECT, INDIRECT, SPECIAL, CONSEQUENTIAL, INCIDENTAL OR PUNITIVE
// DAMAGES, HOWEVER CAUSED AND REGARDLESS OF THE THEORY OF LIABILITY,
// ARISING OUT OF THE USE OF OR INABILITY TO USE SOFTWARE, EVEN IF THE
// AUTHOR HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
// ===========================================================================
// Project:    Crusher md2 viewer
//
// File:       md2.cpp
//
// Written:    16 Jun. 1998 - Tom Conder <blitz@gazpacho.net>
//
// Description:
//    This modules handles models stored in md2 format.
//
// Modifications:
// $Log$
// ===========================================================================
#include "stdafx.h"
#include "defs.h"
#include "errors.h"
#include "3dmath.h"
#include "pcx.h"
#include "md2.h"
#include "render.h"
#include <stdio.h>
#include <stdlib.h>
#include <malloc.h>

POINT_3D	*g_points;

static int IsBackface	(triangle_t vertices[]);

// ===========================================================================
// Name.......: CheckTexturetoMd2()
// Description:	This function checks the texture to the md2 model file.
// Parameters.: pTexMap				- a pointer to the texture map
//				pMdl				- a pointer to the model
// Returns....: UINT				- a return code
// ===========================================================================
UINT
CheckTexturetoMd2 (const pcx_t *pTexMap, const model_t *pMdl)
{
	UINT rc;

	rc = RC_OK;

	if (pTexMap->iHeight < pMdl->skinheight)
	{
		rc = ERR_MD2_TEXHEIGHT;
	}
	else if (pTexMap->iWidth < pMdl->skinwidth)
	{
		rc = ERR_MD2_TEXWIDTH;
	}

	return rc;
} // CheckTexturetoMd2

// ===========================================================================
// Name.......: CompareZValues()
// Description:	This function is used by the sort function to compare the
//				z-coordinate values of two triangles.
// Parameters.: arg1				- the first triangle
//				arg2				- the second triangle
// Returns....: int					- -1 for less than, 0 for equals, 1 for
//									  greater than
// ===========================================================================
int
CompareZValues (const void *arg1, const void *arg2)
{
	dtriangle_t *tri;
	double dValZ1;
	double dValZ2;

	// use the average z value of the first triangle
	tri = (dtriangle_t *) arg1;
	dValZ1 = (g_points[tri->index_xyz[0]].z +
				g_points[tri->index_xyz[1]].z + 
				g_points[tri->index_xyz[2]].z) / 3;

	// use the average z value of the second triangle
	tri = (dtriangle_t *) arg2;
	dValZ2 = (g_points[tri->index_xyz[0]].z +
				g_points[tri->index_xyz[1]].z + 
				g_points[tri->index_xyz[2]].z) / 3;

	if (dValZ1 < dValZ2)
	{
		return 1;
	}
	else if (dValZ1 > dValZ2)
	{
		return -1;
	}
	else
	{
		return 0;
	}
} // CompareZValues

// ===========================================================================
// Name.......: DrawMd2()
// Description:	This function draws the md2 model.
// Parameters.: pTexMap				- a pointer to the texture map
//				pMall				- a pointer to the mult matrix
//				pMdl				- a pointer to the model
// Returns....: UINT				- a return code
// ===========================================================================
UINT
DrawMd2 (const pcx_t *pTexMap, Matrix3D *pMall, const model_t *pMdl)
{
	int			i;
	int			j;
	triangle_t	pTriangle[3];
	POINT_3D	pt;

	if (pMdl->num_frames == 0)
	{
		return RC_OK;
	}

	// transform all points
	for (i=0; i < pMdl->num_xyz; i++)
	{
		pt.x = pMdl->framelist[pMdl->curr_frame].tris[i].x;
		pt.y = pMdl->framelist[pMdl->curr_frame].tris[i].y;
		pt.z = pMdl->framelist[pMdl->curr_frame].tris[i].z;

		pt = pMall->transform(pt);

		pMdl->points[i].x = pt.x;
		pMdl->points[i].y = pt.y;
		pMdl->points[i].z = pt.z;
	}


	g_points = pMdl->points;

	// sort triangles by z-coordinate
	qsort ((void *) pMdl->tri_index,
			(size_t) pMdl->num_tris,
			sizeof (dtriangle_t),
			CompareZValues);

	// draw triangles
	for (i=0; i < pMdl->num_tris; i++)
	{
		for (j=0; j < 3; j++)
		{
			pTriangle[j].x = (long) pMdl->points[pMdl->tri_index[i].index_xyz[j]].x;
			pTriangle[j].y = (long) pMdl->points[pMdl->tri_index[i].index_xyz[j]].y;

			pTriangle[j].s = pMdl->texture_st[pMdl->tri_index[i].index_st[j]].s;
			pTriangle[j].t = pMdl->texture_st[pMdl->tri_index[i].index_st[j]].t;
		}

		if (! IsBackface (pTriangle))
		{
		    DrawTriangle (pTriangle, pTexMap);
		}
	}

	return RC_OK;
} // DrawMd2

// ===========================================================================
// Name.......: FreeMd2()
// Description:	This function frees the memory allocated to the model.
// Parameters.: pMdl				- a pointer to the model
// Returns....: NIL
// ===========================================================================
void
FreeMd2 (model_t *pMdl)
{
	int i;

	if (pMdl->num_frames == 0)
	{
		return;
	}

	pMdl->num_frames = 0;

	free (pMdl->points);
	free (pMdl->commands);
	free (pMdl->tri_index);

	for (i=0; i < pMdl->num_frames; i++)
	{
		free (pMdl->framelist[i].tris);
	}

	free (pMdl->framelist);
} // FreeMd2

// ===========================================================================
// Name.......: GetFrame()
// Description:	This function returns the current frame number.
// Parameters.: pMdl			- a pointer to the model
// Returns....: int				- frame number
// ===========================================================================
int
GetFrame (const model_t *pMdl)
{
	return pMdl->curr_frame;
} // GetFrame

// ===========================================================================
// Name.......: LoadMd2FromFile()
// Description:	This function loads a md2 model from a file.
// Parameters.: sFilename			- a string containing the filename
//				pMdl				- a pointer to the model
// Returns....: UINT				- a return code
// ===========================================================================
UINT
LoadMd2FromFile (LPCTSTR sFilename, model_t *pMdl)
{
	FILE			*pMd2File;
	byte			buffer[MAX_VERTS*4+128];
	daliasframe_t	*out;
	int				i;
	int				j;
	
	pMdl->curr_frame = 0;

	pMd2File = fopen(sFilename, "rb");
	if (pMd2File == (FILE *) 0)
	{
		// Unable to open file for reading
		return ERR_MD2_OPENFILE;
	}

	// read model header
	fread(pMdl, sizeof(dmdl_t), 1, pMd2File);
	
	// check file header
	if (pMdl->ident != IDALIASHEADER)
	{
		// Invalid md2 header
		return ERR_MD2_IDENT;
	}

	// read skin names
	fseek(pMd2File, pMdl->ofs_skins, SEEK_SET);
	if (pMdl->num_skins)
	{
		for (i=0; i < pMdl->num_skins; i++)
		{
			fread(pMdl->skins[i], MAX_SKINNAME, 1, pMd2File);
		}
	}

	// read texture coordinates (dstvert_t)
	pMdl->texture_st = (dstvert_t *) calloc(pMdl->num_st, sizeof(dstvert_t));
	if (pMdl->texture_st == (dstvert_t *) 0)
	{
		return ERR_MD2_ST;
	}

	fseek (pMd2File, pMdl->ofs_st, SEEK_SET);
	fread (pMdl->texture_st, pMdl->num_st * sizeof(dstvert_t), 1, pMd2File);

	// read triangle array indexes
	pMdl->tri_index = (dtriangle_t *) calloc(pMdl->num_tris, sizeof(dtriangle_t));
	if (pMdl->tri_index == (dtriangle_t *) 0)
	{
		return ERR_MD2_TRIS;
	}

	fseek(pMd2File, pMdl->ofs_tris, SEEK_SET);
	for (i=0; i < pMdl->num_tris; i++)
	{
		fread (&pMdl->tri_index[i], sizeof(dtriangle_t), 1, pMd2File);
	}

	pMdl->framelist = (framelist_t *) calloc(pMdl->num_frames, sizeof (frame_t));
	if (pMdl->framelist == (framelist_t *) 0)
	{
		return ERR_MD2_FRAMELIST;
	}

	pMdl->points = (POINT_3D *) calloc(pMdl->num_xyz, sizeof (POINT_3D));
	if (pMdl->points == (POINT_3D *) 0)
	{
		return ERR_MD2_POINTS;
	}

	// read triangle vertices
	fseek(pMd2File, pMdl->ofs_frames, SEEK_SET);
	for (i=0; i < pMdl->num_frames; i++)
	{
		pMdl->framelist[i].tris = (POINT_3D *) calloc(pMdl->num_xyz, sizeof (POINT_3D));
		if (pMdl->framelist[i].tris == (POINT_3D *) 0)
		{
			return ERR_MD2_TRIS;
		}

		out = (daliasframe_t *) buffer;

		fread(out, pMdl->framesize, 1, pMd2File);

		for (j=0; j < pMdl->num_xyz; j++)
		{
			pMdl->framelist[i].tris[j].x = out->verts[j].v[0] * out->scale[0] + out->translate[0];
			pMdl->framelist[i].tris[j].y = out->verts[j].v[1] * out->scale[1] + out->translate[1];
			pMdl->framelist[i].tris[j].z = out->verts[j].v[2] * out->scale[2] + out->translate[2];
		}
	}

	// read gl commands
	pMdl->commands = (int *) calloc(pMdl->num_glcmds, sizeof(long));
	if (pMdl->commands == (int *) 0)
	{
		return ERR_MD2_GLCMDS;
	}
	fseek(pMd2File, pMdl->ofs_glcmds, SEEK_SET);
	fread(pMdl->commands, pMdl->num_glcmds * sizeof(long), 1, pMd2File);
	
	fclose(pMd2File);
	
	return RC_OK;
} // LoadMd2FromFile

// ===========================================================================
// Name.......: NextFrame()
// Description:	This function advances the model one frame.
// Parameters.: pMdl				- a pointer to the model
// Returns....: NIL
// ===========================================================================
void
NextFrame (model_t *pMdl)
{
	pMdl->curr_frame = (pMdl->curr_frame+1) % pMdl->num_frames;
} // NextFrame

// ===========================================================================
// Name.......: PrevFrame()
// Description:	This function retreats the model one frame.
// Parameters.: pMdl				- a pointer to the model
// Returns....: NIL
// ===========================================================================
void
PrevFrame (model_t *pMdl)
{
	if (pMdl->curr_frame == 0)
	{
		pMdl->curr_frame = pMdl->num_frames - 1;
	}
	else
	{
		pMdl->curr_frame = pMdl->curr_frame - 1;
	}
} // PrevFrame

// ===========================================================================
// Name.......: SetFrame()
// Description:	This function sets the model frame number.
// Parameters.: nNewFrame	- new value for frame
//				pMdl				- a pointer to the model
// Returns....: NIL
// ===========================================================================
void
SetFrame (int nNewFrame, model_t *pMdl)
{
	pMdl->curr_frame = nNewFrame % pMdl->num_frames;
} // SetFrame

// ===========================================================================
// Name.......: IsBackface()
// Description:	This function determines if a polygon is a backface is
//				therefore not visible. It returns 0 if the polygon is
//              visible, 1 if not visible. Note: this only works for a
//              convex model.
// Parameters.: POINT *pPoints	-	array of 2D vertices
// Returns....: int				-	boolean for backface
// ===========================================================================
static int
IsBackface (triangle_t pPoints[])
{
	int z;

	z = (pPoints[0].x - pPoints[1].x) * (pPoints[2].y - pPoints[1].y) -
		(pPoints[0].y - pPoints[1].y) * (pPoints[2].x - pPoints[1].x);
	
	return (z > 0);
} // IsBackface
