// ===========================================================================
// $Source: d:/source/master/crusher/src/crusher.cpp,v $
// $Revision: 1.43 $
// $Date: 1998/08/04 02:55:42 $
// ===========================================================================
// Copyright (C) 1998 Tom Conder <blitz@gazpacho.net>. All Rights Reserved.
//
// BY USING ANY PORTION OF THIS SOFTWARE, YOU AGREE TO THE FOLLOWING
// TERMS AND CONDITIONS:
// 
// Tom Conder, "THE AUTHOR", grants you, "THE USER", a non-exclusive,
// royalty free, license to use this software in source and binary code
// form, provided the user does not utilize the software in a manner
// which is disparaging to the author and the user acknowledges the
// author in any derivative work.
// 
// This software is provided "AS IS," without a warranty of any kind. ALL
// EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND WARRANTIES, INCLUDING
// ANY IMPLIED WARRANTY OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
// PURPOSE OR NON-INFRINGEMENT, ARE HEREBY EXCLUDED. THE AUTHOR SHALL NOT
// BE LIABLE FOR ANY DAMAGES SUFFERED BY THE USER AS A RESULT OF USING,
// MODIFYING OR DISTRIBUTING THE SOFTWARE OR ITS DERIVATIVES. IN NO EVENT
// WILL THE AUTHOR BE LIABLE FOR ANY LOST REVENUE, PROFIT OR DATA, OR FOR
// DIRECT, INDIRECT, SPECIAL, CONSEQUENTIAL, INCIDENTAL OR PUNITIVE
// DAMAGES, HOWEVER CAUSED AND REGARDLESS OF THE THEORY OF LIABILITY,
// ARISING OUT OF THE USE OF OR INABILITY TO USE SOFTWARE, EVEN IF THE
// AUTHOR HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
// ===========================================================================
// Project:    Crusher md2 viewer
//
// File:       crusher.cpp
//
// Written:    16 Jun. 1998 - Tom Conder <blitz@gazpacho.net>
//
// Description:
//    This is the source for main portion the md2 viewer.
//
// Modifications:
// $Log$
// ===========================================================================
#include "stdafx.h"
#include "defs.h"
#include "errors.h"
#include "3dmath.h"
#include "pcx.h"
#include "md2.h"
#include "render.h"
#include "crusher.h"
#include <stdio.h>
#include <regstr.h>
#include <commdlg.h>
#include <shlobj.h>
#include <time.h>

#define	NAME			"Crusher"
#define	TITLE			"Crusher md2 Viewer"

// prototypes
LRESULT	CALLBACK		WndProc	(HWND, UINT, WPARAM, LPARAM);

static BOOL CALLBACK	AboutDialogProc		(HWND, UINT, WPARAM, LPARAM);
static UINT				Create256Palette	(HDC);
static int				DoAboutBox			(HWND);
static void				ILFree				(LPITEMIDLIST);
static BOOL CALLBACK	SettingsDialogProc	(HWND, UINT, WPARAM, LPARAM);
static void				SetWindowTitle		(HWND, const model_t *);
static BOOL				ValidateQuake2Path	(LPCSTR);

static BOOL				bWireframe = FALSE;
static HINSTANCE		hMainInstance = 0;
static HPALETTE			hPalette;
static HPALETTE			hpalOld;
static model_t			mdlModel;
static pcx_t			pcxTexMap;
model_t					*pMdl = &mdlModel;
pcx_t					*pTexMap = &pcxTexMap;


// ===========================================================================
// Name.......: WinMain()
// Description:	This function is the enter point for the program and the main
//				initialization and message loop.
// Parameters.: hInstance				- handle to current instance
//				hPrevInstance			- handle to previous instance
//				lpCmdLine				- pointer to comand line
//				nCmdShow				- show state of window
// Returns....: int						- program return status
// ===========================================================================
int WINAPI
WinMain	(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR lpCmdLine, int nCmdShow)
{
	HACCEL		hAccel;
	HWND		hwnd;
	LPCTSTR 	pszWindowClass = TEXT(NAME);
	LPCTSTR 	pszWindowTitle = TEXT(TITLE);
	MSG 		msg;
	UINT		uExStyle, uStyle, xPos, yPos;
	WNDCLASSEX	wc;
	
	hMainInstance = hInstance;
	
	// set up and register window class
	wc.cbSize			= sizeof (wc);
	wc.hCursor			= LoadCursor (NULL, IDC_ARROW);
	wc.hIcon			= LoadIcon (hMainInstance, MAKEINTRESOURCE (IDI_CRUSHER));
	wc.hIconSm			= LoadIcon (hMainInstance, MAKEINTRESOURCE (IDI_CRUSHER));
	wc.lpszMenuName 	= MAKEINTRESOURCE(IDR_MENU);
	wc.lpszClassName	= pszWindowClass;
	wc.hbrBackground	= (HBRUSH) GetStockObject(BLACK_BRUSH);
	wc.hInstance		= hMainInstance;
	wc.lpfnWndProc		= WndProc;
	wc.cbClsExtra		= 0;
	wc.cbWndExtra		= 0;
	wc.style			= CS_HREDRAW | CS_VREDRAW | CS_DBLCLKS | CS_OWNDC;
	
	xPos		= (GetSystemMetrics(SM_CXSCREEN) - iWIDTH ) / 2;
	yPos		= (GetSystemMetrics(SM_CYSCREEN) - iHEIGHT ) / 2;
	uStyle		= WS_VISIBLE | WS_OVERLAPPEDWINDOW | WS_POPUP | WS_CAPTION;
	uExStyle	= 0;
	
	if (RegisterClassEx (&wc))
	{
		// create a window
		hwnd = CreateWindowEx (uExStyle,
								pszWindowClass,
								pszWindowTitle,
								uStyle,
								xPos,
								yPos,
								iWIDTH,
								iHEIGHT,
								(HWND) NULL,
								(HMENU) NULL,
								hMainInstance,
								(LPVOID) NULL);
	}
	
	srand ((unsigned) time(NULL));
	
	if (hwnd)
	{
		hAccel = LoadAccelerators (hInstance, MAKEINTRESOURCE(IDR_ACCELERATOR));

		while (1)
		{
			if (PeekMessage (&msg, NULL, 0, 0, PM_NOREMOVE))
			{
				if (!GetMessage( &msg, NULL, 0, 0))
				{
					break;
				}
				
				if (!TranslateAccelerator (hwnd, hAccel, &msg))
				{
					TranslateMessage(&msg); 
					DispatchMessage(&msg);
				}
			}
		}
		
		return msg.wParam;
	}
	
	return 0;
} // WinMain

// ===========================================================================
// Name.......: CommandHandler()
// Description:	This function handles WM_COMMAND messages.
// Parameters.: hWnd					- handle for window
//				wParam					- first message parameter
//				lParam					- second message parameter
// Returns....: LONG WINAPI				- return status
// ===========================================================================
LONG WINAPI
CommandHandler (HWND hWnd, WPARAM wParam, LPARAM lParam)
{
	LPOPENFILENAME	lpofn;
	OPENFILENAME	ofn;
	LPTSTR			lpstrFile;
	TCHAR			ch;
	TCHAR			tPath[iMAX_PATH_LEN];
	TCHAR			tTemp[iMAX_PATH_LEN];
	UINT			rc;
	static TCHAR	tInitDir[iMAX_PATH_LEN] = "\0";

	switch (LOWORD(wParam))
	{
	case IDM_OPEN_MODEL:
		if (*tInitDir == NULL)
		{
			GetRegPath (tInitDir);

			if (*tInitDir != NULL)
			{
				ch = *(tInitDir + strlen(tInitDir) - 1);
				if (ch != '\\' && ch != '/')
				{
					strcat (tInitDir, "\\");
				}

				strcat (tInitDir, tQUAKE2_PLAYER_DIR);

				if (LoadRegValue (tREG_KEY_PATH, tPath) == RC_OK)
				{
					strcat (tInitDir, tPath);
					strcat (tInitDir, "\\");

					if (LoadRegValue (tREG_KEY_MODEL, tTemp) == RC_OK)
					{
						strcat (tInitDir, tTemp);
					}
					else
					{
						strcat (tInitDir, tQUAKE2_MD2_FILENAME);
					}
				}
				else
				{
					strcat (tInitDir, tQUAKE2_MD2_PATHNAME);
					strcat (tInitDir, "\\");
					strcat (tInitDir, tQUAKE2_MD2_FILENAME);
				}
			}
		}

		lpstrFile = (TCHAR *) calloc(iMAX_PATH_LEN, sizeof(TCHAR));

		lpofn = &ofn;

		ofn.lStructSize = sizeof (OPENFILENAME); 
		ofn.hwndOwner = hWnd;
		ofn.hInstance = NULL;
		ofn.lpstrFilter = "Quake 2 Model Files (.md2)\0*.md2\0"
							"All Files (*.*)\0*.*\0\0";
		ofn.lpstrCustomFilter = NULL; 
		ofn.nMaxCustFilter = NULL;
		ofn.nFilterIndex = 0; 
		ofn.lpstrFile = lpstrFile;
		ofn.nMaxFile = iMAX_PATH_LEN; 
		ofn.lpstrFileTitle = NULL;
		ofn.nMaxFileTitle = NULL; 
		ofn.lpstrInitialDir = tInitDir;
		ofn.lpstrTitle = NULL; 
		ofn.Flags = NULL;
		ofn.nFileOffset = 0; 
		ofn.nFileExtension = 0;
		ofn.lpstrDefExt = NULL; 
		ofn.lCustData = NULL ;
		ofn.lpfnHook = NULL; 
		ofn.lpTemplateName = NULL;

		if (GetOpenFileName (lpofn) > 0)
		{
			// set new init dir to chosen dir
			strncpy (tInitDir, lpofn->lpstrFile, lpofn->nFileOffset);
			*(tInitDir + lpofn->nFileOffset) = '\0';

			if (lpofn->nFileExtension != 0)
			{
				if (stricmp (lpofn->lpstrFile + lpofn->nFileExtension, "md2") == 0)
				{
					// load model
					if ((rc = LoadModel (GetDC(hWnd), lpofn->lpstrFile, pTexMap, pMdl, TRUE)) != RC_OK)
					{
						DisplayErrorMessage (rc);
					}
					else
					{
						SetWindowTitle (hWnd, pMdl);
					}
				}
			}
		}

		free (lpstrFile);

		SendMessage (hWnd, WM_PAINT, 0, 0);
		break;

	case IDM_OPEN_TEX:
		if (*tInitDir == NULL)
		{
			GetRegPath (tInitDir);

			if (*tInitDir != NULL)
			{
				ch = *(tInitDir + strlen(tInitDir) - 1);
				if (ch != '\\' && ch != '/')
				{
					strcat (tInitDir, "\\");
				}

				strcat (tInitDir, tQUAKE2_PLAYER_DIR);

				if (LoadRegValue (tREG_KEY_PATH, tPath) == RC_OK)
				{
					strcat (tInitDir, tPath);
					strcat (tInitDir, "\\");

					if (LoadRegValue (tREG_KEY_SKIN, tTemp) == RC_OK)
					{
						strcat (tInitDir, tTemp);
					}
					else
					{
						strcat (tInitDir, tQUAKE2_PCX_FILENAME);
					}
				}
				else
				{
					strcat (tInitDir, tQUAKE2_MD2_PATHNAME);
					strcat (tInitDir, "\\");
					strcat (tInitDir, tQUAKE2_PCX_FILENAME);
				}
			}
		}

		lpstrFile = (TCHAR *) calloc(iMAX_PATH_LEN, sizeof(TCHAR));

		lpofn = &ofn;

		ofn.lStructSize = sizeof (OPENFILENAME); 
		ofn.hwndOwner = hWnd;
		ofn.hInstance = NULL;
		ofn.lpstrFilter = "PCX Texture Files (.pcx)\0*.pcx\0"
							"All Files (*.*)\0*.*\0\0";
		ofn.lpstrCustomFilter = NULL; 
		ofn.nMaxCustFilter = NULL;
		ofn.nFilterIndex = 0; 
		ofn.lpstrFile = lpstrFile;
		ofn.nMaxFile = iMAX_PATH_LEN; 
		ofn.lpstrFileTitle = NULL;
		ofn.nMaxFileTitle = NULL; 
		ofn.lpstrInitialDir = tInitDir;
		ofn.lpstrTitle = NULL; 
		ofn.Flags = NULL;
		ofn.nFileOffset = 0; 
		ofn.nFileExtension = 0;
		ofn.lpstrDefExt = NULL; 
		ofn.lCustData = NULL ;
		ofn.lpfnHook = NULL; 
		ofn.lpTemplateName = NULL;

		if (GetOpenFileName (lpofn) > 0)
		{
			// set new init dir to chosen dir
			strncpy (tInitDir, lpofn->lpstrFile, lpofn->nFileOffset);
			*(tInitDir + lpofn->nFileOffset) = '\0';

			if (lpofn->nFileExtension != 0)
			{
				if (stricmp (lpofn->lpstrFile + lpofn->nFileExtension, "pcx") == 0)
				{
					// load pcx
					if ((rc = LoadTexture (GetDC(hWnd), lpofn->lpstrFile, pTexMap, pMdl)) != RC_OK)
					{
						DisplayErrorMessage (rc);
					}
					else
					{
						if ((rc = CheckTexture (pTexMap, pMdl)) != RC_OK)
						{
							DisplayErrorMessage (rc);
						}
						else
						{
							SetWindowTitle (hWnd, pMdl);
						}
					}
				}
			}
		}

		free (lpstrFile);

		SendMessage (hWnd, WM_PAINT, 0, 0);
		break;

	case IDM_ABOUT:
		DoAboutBox(hWnd);
		break;

	case IDM_WIREFRAME:
		HMENU	hMenu;

		hMenu = GetMenu(hWnd);
		hMenu = GetSubMenu(hMenu, 1);

		if (bWireframe)
		{
			CheckMenuItem(hMenu, 0, MF_BYPOSITION|MF_UNCHECKED);
			bWireframe = FALSE;
		}
		else
		{
			CheckMenuItem(hMenu, 0, MF_BYPOSITION|MF_CHECKED);
			bWireframe = TRUE;
		}

		SendMessage (hWnd, WM_PAINT, 0, 0);
		break;

	case IDM_SETTINGS:
		DoSettingsBox(hWnd);
		break;

	case IDM_EXIT:
		SendMessage (hWnd, WM_CLOSE, 0, 0);
		break;

	default:
		return FALSE;
	}

	return TRUE;
} // CommandHandler

// ===========================================================================
// Name.......: WndProc()
// Description:	This function is the main callback for windows procedures.
// Parameters.: hWnd					- handle for window
//				message					- message identifier
//				wParam					- first message parameter
//				lParam					- second message parameter
// Returns....: LRESULT CALLBACK		- return status
// ===========================================================================
LRESULT	CALLBACK
WndProc (HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	static BOOL		bLeftDown = FALSE;
	static BOOL		bRightDown = FALSE;
	static int		iNewX, iNewY;
	HDC				hDC;
	int				iLastX, iLastY;
	POINT			ptCursor;
	UINT			rc;

	switch (message)
	{
	case WM_CREATE:
		if ((rc = Init3D(hWnd, pTexMap, pMdl)) != RC_OK)
		{
			if (DoSettingsBox(hWnd) == TRUE)
			{
				if (CheckRegForQuake2Path() == TRUE)
				{
					if ((rc = Init3D(hWnd, pTexMap, pMdl)) != RC_OK)
					{
						DisplayErrorMessage (rc);
					}
					else
					{
						SetWindowTitle (hWnd, pMdl);
						break;
					}
				}
			}

			SendMessage (hWnd, WM_CLOSE, 0, 0);
			break;
		}

		SetWindowTitle (hWnd, pMdl);

		hDC = GetDC(hWnd);
		if (GetDeviceCaps(hDC, RASTERCAPS) & RC_PALETTE)
		{
			// 256 color palette
			if ((rc = Create256Palette (hDC)) != RC_OK)
			{
				DisplayErrorMessage (rc);
			}
		}

		ReleaseDC (hWnd, hDC);
		break;
		
	case WM_DESTROY:
		End3D(pTexMap, pMdl);

		PostQuitMessage(0);
		break;
		
	case WM_COMMAND:
		return CommandHandler (hWnd, wParam, lParam);
		break;

	case WM_PAINT:
		hDC = GetDC(hWnd);

		if (!(GetDeviceCaps(hDC, RASTERCAPS) & RC_PALETTE))
		{
			DrawFrame(hDC, pTexMap, pMdl);
		}
		else
		{
			hpalOld = SelectPalette (hDC, hPalette, FALSE);
			RealizePalette (hDC);

			DrawFrame(hDC, pTexMap, pMdl);

			if (hpalOld)
			{
				SelectPalette(hDC, hpalOld, FALSE);
			}
		}

		ReleaseDC(hWnd, hDC);
		break;

	case WM_QUERYNEWPALETTE:
        if(hPalette)
        {
            hDC = GetDC(hWnd);

            // Select and realize the palette
            hpalOld = SelectPalette(hDC, hPalette, FALSE);
            RealizePalette(hDC);

			DrawFrame(hDC, pTexMap, pMdl);

            if (hpalOld)
			{
                SelectPalette(hDC, hpalOld, FALSE);
			}

            ReleaseDC(hWnd, hDC);
        }
		break;

	case WM_PALETTECHANGED:
		if (wParam != (WPARAM)hWnd)
		{
			hDC = GetDC(hWnd);

			hpalOld = SelectPalette (hDC, hPalette, FALSE);
			RealizePalette (hDC);

			UpdateColors(hDC);
			DrawFrame(hDC, pTexMap, pMdl);

			if (hpalOld)
			{
				SelectPalette(hDC, hpalOld, FALSE);
			}

			ReleaseDC(hWnd, hDC);
		}
		break;

	case WM_LBUTTONDOWN:
		if (!bLeftDown && !bRightDown)
		{
			SetCapture (hWnd);
		}

		bLeftDown = TRUE;

		GetCursorPos (&ptCursor);

		iNewX = ptCursor.x;
		iNewY = ptCursor.y;
		break;

	case WM_LBUTTONUP:
		if (bLeftDown && !bRightDown)
		{
			ReleaseCapture ();
		}

		bLeftDown = FALSE;
		break;

	case WM_RBUTTONDOWN:
		if (!bLeftDown && !bRightDown)
		{
			SetCapture (hWnd);
		}

		bRightDown = TRUE;

		GetCursorPos (&ptCursor);

		iNewX = ptCursor.x;
		iNewY = ptCursor.y;
		break;

	case WM_RBUTTONUP:
		if (!bLeftDown && bRightDown)
		{
			ReleaseCapture ();
		}

		bRightDown = FALSE;
		break;

	case WM_KEYDOWN:
		switch (wParam)
		{
		case VK_LEFT:
			PrevFrame(pMdl);

			hDC = GetDC(hWnd);

			if (!(GetDeviceCaps(hDC, RASTERCAPS) & RC_PALETTE))
			{
				DrawFrame(hDC, pTexMap, pMdl);
			}
			else
			{
				hpalOld = SelectPalette (hDC, hPalette, FALSE);
				RealizePalette (hDC);

				DrawFrame(hDC, pTexMap, pMdl);

				if (hpalOld)
				{
					SelectPalette(hDC, hpalOld, FALSE);
				}
			}

			ReleaseDC(hWnd, hDC);
			break;

		case VK_RIGHT:
			NextFrame(pMdl);

			hDC = GetDC(hWnd);

			if (!(GetDeviceCaps(hDC, RASTERCAPS) & RC_PALETTE))
			{
				DrawFrame(hDC, pTexMap, pMdl);
			}
			else
			{
				hpalOld = SelectPalette (hDC, hPalette, FALSE);
				RealizePalette (hDC);

				DrawFrame(hDC, pTexMap, pMdl);

				if (hpalOld)
				{
					SelectPalette(hDC, hpalOld, FALSE);
				}
			}

			ReleaseDC(hWnd, hDC);
			break;
		}
		break;

	case WM_MOUSEMOVE:
		iLastX = iNewX;
		iLastY = iNewY;

		GetCursorPos (&ptCursor);

		iNewX = ptCursor.x;
		iNewY = ptCursor.y;

		if (bLeftDown)
		{
			AdjustXRot ((int) (iNewY - iLastY));
			AdjustZRot ((int) (iLastX - iNewX));
		}

		if (bRightDown)
		{
			AdjustHorzOffset ((int) (iNewX - iLastX));
			AdjustVertOffset ((int) (iNewY - iLastY));
		}

		hDC = GetDC(hWnd);

		if (!(GetDeviceCaps(hDC, RASTERCAPS) & RC_PALETTE))
		{
			DrawFrame(hDC, pTexMap, pMdl);
		}
		else
		{
			hpalOld = SelectPalette (hDC, hPalette, FALSE);
			RealizePalette (hDC);

			DrawFrame(hDC, pTexMap, pMdl);

			if (hpalOld)
			{
				SelectPalette(hDC, hpalOld, FALSE);
			}
		}

		ReleaseDC(hWnd, hDC);
		break;

	case WM_SIZE:
		if ((rc = Resize(hWnd, LOWORD(lParam), HIWORD(lParam), pTexMap)) != RC_OK)
		{
			DisplayErrorMessage(rc);
		}
		break;
	}
	
	return DefWindowProc (hWnd, message, wParam, lParam);
} // WndProc

// ========================================================================
// Name.......: CheckRegForQuake2Path()
// Description: This function checks the registry to see if it contains the
//				Quake 2 directory name.
// Parameters.: NIL
// Returns....: BOOL					- boolean for exists in registry
// ========================================================================
BOOL
CheckRegForQuake2Path ()
{
	TCHAR	tSelection[iREG_KEY_BUF_SIZE];

	if (LoadRegValue (tREG_KEY_Q2PATH, tSelection) == RC_OK)
	{
		return TRUE;
	}
	else
	{
		return FALSE;
	}
} // CheckRegForQuake2Path

// ========================================================================
// Name.......: DoSettingsBox()
// Description: This function launches the 'settings' dialog box.
// Parameters.: hWnd					- handle to current window
// Returns....: int						- message parameter
// ========================================================================
int
DoSettingsBox (HWND hWnd)
{
	return DialogBox (hMainInstance,
						MAKEINTRESOURCE(IDD_SETTINGS),
						hWnd,
						(DLGPROC) SettingsDialogProc);
} // DoSettingsBox

// ========================================================================
// Name.......: GetRegPath()
// Description: This function returns the path stored in the registry.
// Parameters.: tFilename				- string to contain filename
// Returns....: NIL
// ========================================================================
void
GetRegPath (LPTSTR tFilename)
{
	TCHAR	tSelection[iREG_KEY_BUF_SIZE];

	tFilename[0] = '\0';

	if (LoadRegValue (tREG_KEY_Q2PATH, tSelection) == RC_OK)
	{
		strcpy (tFilename, tSelection);
	}
}

// ===========================================================================
// Name.......: GetWireframeState()
// Description:	This function returns a boolean indicating whether the
//				software is displaying in wireframe mode.
// Parameters.: NIL
// Returns....: BOOL					- boolean for wireframe state
// ===========================================================================
BOOL
GetWireframeState ()
{
	return bWireframe;
}

// ========================================================================
// Name.......: LoadRegValue()
// Description: This function loads a value from the registry.
// Parameters.: tValue					- the given value
//				tData					- the data to be returned
// Returns....: UINT					- a return code
// ========================================================================
UINT
LoadRegValue (LPCTSTR tValue, LPTSTR tData)
{
	DWORD	dwDisp = iREG_KEY_BUF_SIZE; 
	DWORD	dwType;
	HKEY	hkCrusher;
	LONG	lResult;

	lResult = RegOpenKeyExA(HKEY_LOCAL_MACHINE,
							tREG_KEY_CRUSHER,
							0,
							KEY_QUERY_VALUE,
							&hkCrusher);

	if (lResult == ERROR_SUCCESS)
	{
		lResult = RegQueryValueEx (hkCrusher,
									tValue,
									0,
									(LPDWORD) &dwType,
									(LPBYTE) tData,
									(LPDWORD) &dwDisp);
		
		RegCloseKey (hkCrusher);

		if (lResult != ERROR_SUCCESS)
		{
			return ERR_RENDER_REG_QUERY;
		}
	}
	else
	{
		return ERR_RENDER_REG_OPEN;
	}

	return RC_OK;
} // LoadRegValue

// ========================================================================
// Name.......: SaveRegValue()
// Description: This function saves the given value in the registry.
// Parameters.: tValue					- the given value
//				tData					- the data to be stored
// Returns....: NIL
// ========================================================================
void
SaveRegValue (LPCTSTR tValue, LPCTSTR tData)
{
	DWORD	dwDisp = iREG_KEY_BUF_SIZE; 
	HKEY	hkClass;
	HKEY	hkCrusher;

	RegCreateKeyEx (HKEY_LOCAL_MACHINE,
					tREG_KEY_CLASS,
					0,
					"",
					REG_OPTION_NON_VOLATILE,
					KEY_ALL_ACCESS,
					NULL,
					&hkClass,
					&dwDisp);
	RegCreateKeyEx (hkClass,
					tREG_KEY_NAME,
					0,
					"",
					REG_OPTION_NON_VOLATILE,
					KEY_ALL_ACCESS,
					NULL,
					&hkCrusher,
					&dwDisp);
	RegSetValueEx (hkCrusher,
					tValue,
					0,
					REG_SZ,
					(CONST BYTE *) tData,
					lstrlen (tData) + 1);

	RegCloseKey (hkClass);
	RegCloseKey (hkCrusher);
} // SaveRegValue

// ========================================================================
// Name.......: AboutDialogProc()
// Description: This function provides the processing of window messages for
//              the About dialog box.
// Parameters.: hWnd					- handle to the window
//              uMsg					- the window message
//              wParam					- first message parameter
//              lParam					- second message parameter
// Returns....: BOOL CALLBACK			- result of message processing
// ========================================================================
static BOOL CALLBACK
AboutDialogProc(HWND hDlg, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
	switch (uMsg)
	{
	case WM_INITDIALOG:
		EnableWindow (hDlg, TRUE);
		ShowWindow (hDlg, SW_SHOW);
		return TRUE;
		
	case WM_COMMAND:
		switch (LOWORD(wParam))
		{
		case IDOK:
		case IDCANCEL:
			EndDialog (hDlg, 0);
			return TRUE;
		}
		break;
	}
	return FALSE;
} // AboutDialogProc

// ===========================================================================
// Name.......: Create256Palette()
// Description:	This function creates a 256 color palette.
// Parameters.: hDC						- handle to the device context
// Returns....: UINT					- a return code
// ===========================================================================
static UINT
Create256Palette (HDC hDC)
{
	LOGPALETTE		*pLogPal;
	int				i;

	pLogPal = (LOGPALETTE *) malloc(sizeof(LOGPALETTE) +
	    		256 * sizeof(PALETTEENTRY));

	if (pLogPal == (LOGPALETTE *) NULL)
	{
		return ERR_RENDER_ALLOC_PAL;
	}

	pLogPal->palVersion = 0x300;
	pLogPal->palNumEntries = 256;

	for (i=0; i < 256; i++)
	{
		pLogPal->palPalEntry[i].peRed = GetRValue(pTexMap->pPalette[i]);
		pLogPal->palPalEntry[i].peBlue = GetBValue(pTexMap->pPalette[i]);
		pLogPal->palPalEntry[i].peGreen = GetGValue(pTexMap->pPalette[i]);
		pLogPal->palPalEntry[i].peFlags = PC_NOCOLLAPSE;
	}

	if ((hPalette = CreatePalette (pLogPal)) == NULL)
	{
		return ERR_RENDER_PAL_CREATE;
	}

	if ((hpalOld = SelectPalette(hDC, hPalette, FALSE)) == NULL)
	{
		return ERR_RENDER_PAL_SEL;
	}

	if (RealizePalette(hDC) == GDI_ERROR)
	{
		return ERR_RENDER_PAL_REALIZE;
	}

	if (hpalOld)
	{
		SelectPalette(hDC, hpalOld, FALSE);
	}

	free (pLogPal);

	return RC_OK;
} // Create256Color

// ========================================================================
// Name.......: DoAboutBox()
// Description: This function launches the 'about' dialog box.
// Parameters.: hWnd					- handle to current window
// Returns....: int						- message parameter
// ========================================================================
static int
DoAboutBox (HWND hWnd)
{
	return DialogBox (hMainInstance,
						MAKEINTRESOURCE(IDD_ABOUT),
						hWnd,
						(DLGPROC) AboutDialogProc);
} // DoAboutBox

// ========================================================================
// Name.......: ILFree()
// Description: This function frees the memory allocated to an item list.
// Parameters.: pidl				- an item list
// Returns....: NIL
// ========================================================================
static void
ILFree(LPITEMIDLIST pidl)
{
	LPMALLOC pMalloc;
	
	if (pidl)
	{ 
		SHGetMalloc(&pMalloc);
		pMalloc->Free(pidl); 
		pMalloc->Release();      
	}
} // ILFree

// ========================================================================
// Name.......: SettingsDialogProc()
// Description: This function provides the processing of window messages for
//              the Settings dialog box.
// Parameters.: hDlg					- handle to the window
//              uMsg					- the window message
//              wParam					- first message parameter
//              lParam					- second message parameter
// Returns....: BOOL CALLBACK			- result of message processing
// ========================================================================
static BOOL CALLBACK
SettingsDialogProc(HWND hDlg, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
	BOOL			bSuccess;
	HWND			hTextBox;
	TCHAR			tSelection[iREG_KEY_BUF_SIZE];
	TCHAR			szTempDir[MAX_PATH];
	TCHAR			szTempName[MAX_PATH];
	BROWSEINFO		biInfo;
	LPITEMIDLIST	pidl;

	switch (uMsg)
	{
	case WM_INITDIALOG:
		hTextBox = GetDlgItem(hDlg, IDC_Q2PATH);
		SendMessage(hTextBox, WM_SETTEXT, 0, (LPARAM) " ");

		if (LoadRegValue (tREG_KEY_Q2PATH, tSelection) == RC_OK)
		{
			SendMessage(hTextBox, WM_SETTEXT, 0, (LPARAM) tSelection);
		}
		else
		{
			SendMessage(hTextBox, WM_SETTEXT, 0, (LPARAM) tQUAKE2_PATH);
		}

		EnableWindow (hDlg, TRUE);
		ShowWindow (hDlg, SW_SHOW);
		return TRUE;
		
	case WM_COMMAND:
		switch (LOWORD(wParam))
		{
		case IDBROWSE:
			biInfo.hwndOwner = GetParent(hDlg);
			biInfo.pidlRoot = NULL;
			biInfo.pszDisplayName = szTempName;
			biInfo.lpszTitle = "Please select the Quake 2 directory:";
			biInfo.ulFlags = BIF_RETURNONLYFSDIRS;
			biInfo.lpfn = NULL;
			biInfo.lParam = NULL;
			biInfo.iImage = 0;
			pidl = SHBrowseForFolder (&biInfo);
			if (pidl != NULL)
			{
				bSuccess = SHGetPathFromIDList(pidl, szTempDir);
				if (bSuccess == TRUE)
				{
					hTextBox = GetDlgItem(hDlg, IDC_Q2PATH);
					SendMessage(hTextBox, WM_SETTEXT, 0, (LPARAM) szTempDir);
				}

				ILFree (pidl);
			}
			break;

		case IDOK:
			SendDlgItemMessage(hDlg,
								IDC_Q2PATH,
								WM_GETTEXT, 
								sizeof(tSelection),
								(LPARAM) tSelection);

			if (ValidateQuake2Path((LPCSTR)tSelection) == TRUE)
			{
				SaveRegValue (tREG_KEY_Q2PATH, tSelection);

				EndDialog (hDlg, TRUE);
				return TRUE;
			}
			else
			{
				MessageBox (hDlg, "Unable to find the 'players' subfolder. Please check the folder name.", "Crusher Error", MB_ICONWARNING | MB_OK);
				return FALSE;
			}

		case IDCANCEL:
			EndDialog (hDlg, FALSE);
			return TRUE;
		}
		break;
	}
	return FALSE;
} // SettingsDialogProc

// ========================================================================
// Name.......: SetWindowTitle()
// Description: This function sets the window title.
// Parameters.: hWnd					- handle to window
//				pMdl					- a pointer to the model
// Returns....: NIL
// ========================================================================
static void
SetWindowTitle (HWND hWnd, const model_t *pMdl)
{
	TCHAR			tWindowTitle[255];

	sprintf (tWindowTitle, "%s - [%s : %s / %s]", 
				TITLE, 
				pMdl->model_pathname, 
				pMdl->model_name, 
				pMdl->skin_name);

	SetWindowText (hWnd, tWindowTitle);
} // SetWindowTitle

// ========================================================================
// Name.......: ValidateQuake2Path()
// Description: This function checks to see if the given directory is a
//				valid Quake 2 directory.
// Parameters.: tQuake2Path				- the given Quake 2 path
// Returns....: BOOL					- boolean for valid path
// ========================================================================
static BOOL
ValidateQuake2Path (LPCSTR tQuake2Path)
{
	TCHAR	tQ2Player[iMAX_PATH_LEN];
	TCHAR	ch;

	strcpy (tQ2Player, tQuake2Path);

	if (SetCurrentDirectory (tQuake2Path) == FALSE)
	{
		return FALSE;
	}

	ch = *(tQ2Player + strlen(tQ2Player));
	if (ch != '\\' && ch != '/')
	{
		strcat (tQ2Player, "\\");
	}

	strcat (tQ2Player, tQUAKE2_PLAYER_DIR);

	if (SetCurrentDirectory (tQ2Player) == TRUE)
	{
		return TRUE;
	}
	else
	{
		return FALSE;
	}
} // ValidateQuake2Path
