#define	LIBQTOOLS_CORE
#include "../include/libqtools.h"
#include "../include/libqbuild.h"

/*
 * Script-System:
 * 
 * commands:
 * "process <file>.<ext>"                               -> take this file upto
 * "finish <file>.<ext>"                                -> here
 * "list <file>.<ext>"					-> list the contents
 * "view <file>.<ext>"					-> view the pictures or texts in a window, or whatever
 * 
 * sub-commands <PAK>:
 * "OP_UPDATE <file>.<ext> as <name>"                   -> OP_ADD if not exists, OP_REPLACE if newer
 * "OP_REPLACE <file>.<ext> as <name>"                  -> OP_REPLACE if exists, do NOT OP_ADD
 * "OP_ADD <file>.<ext> as <name>"                      -> OP_ADD if not exists, do NOT OP_REPLACE
 * "OP_DELETE <name>"                                   -> OP_DELETE if exists
 * "OP_EXTRACT <name> as <file>.<ext>"                  -> OP_EXTRACT if exists
 * 
 * sub-commands <WAD>:
 * -multiple same names are allowed, but with different types
 * "OP_UPDATE <file>.<ext> as <name> as <type>"         -> OP_ADD if not exists, OP_REPLACE if newer
 * "OP_REPLACE <file>.<ext> as <name> as <type>"        -> OP_REPLACE if exists, do NOT OP_ADD
 * "OP_ADD <file>.<ext> as <name> as <type>"            -> OP_ADD if not exists, do NOT OP_REPLACE
 * "OP_DELETE <name> as <type>"                         -> OP_DELETE if exists
 * "OP_EXTRACT <name> as <file>.<ext> as <type>"        -> OP_EXTRACT if exists
 * 
 * sub-commands <BSP>:
 * sub-commands <MDL>:
 * 
 */

/*
 * basic main-skeleton
 *
 * extension-type system:
 *  .wad -> wad-archive
 *  .bsp -> bsp
 *  .pak -> pak-archive
 *  .mdl -> models
 *  .spr -> sprites
 *
 *  .ppm -> ppm-picture
 *  .mip -> miptexture
 *  .lmp -> console-picture
 *  .stb -> statusbar-picture
 *  .pal -> palettes
 *  .wav -> wave-sounds
 *  .dat -> pseudo-code
 *  .rc  -> resource
 *  .cfg -> config
 *  .tri -> triangle
 *  .map -> map
 *  .skn -> skin-picture (lmp)
 *  .frm -> frame-picture (lmp)
 *
 * hierachy-system:
 *   game                                               -> pak?.pak
 *   game/pak?.dir                                      -> ?.dat, ...
 *   game/pak?.dir/gfx                                  -> ?.lmp
 *   game/pak?.dir/maps                                 -> ?.bsp
 *   game/pak?.dir/maps/?.dir                           -> ?.map, ?.wad
 *   game/pak?.dir/maps/?.dir/?.dir                     -> ?.mip/ppm, ?.lmp, ?.pal, ?.stb
 *   game/pak?.dir/progs                                -> ?.mdl, ?.spr
 *   game/pak?.dir/progs/?.dir                          -> ?-?.tri, ?-?.skn
 *   game/pak?.dir/progs/?.dir                          -> ?-?.frm
 *   game/pak?.dir/sounds                               -> ?.wav
 * example:
 *   game/pak0.pak
 *   game/pak0.dir/default.cfg                          -> pak0.pak  (makpak)
 *   game/pak0.dir/gfx/colormap.lmp                     -> pak0.pak  (makpak)
 *   game/pak0.dir/gfx/conback.lmp                      -> pak0.pak  (repak)
 *   game/pak0.dir/gfx/palette.lmp                      -> pak0.pak  (repak)
 *   game/pak0.dir/maps/start.bsp                       -> pak0.pak  (repak)
 *   game/pak0.dir/maps/start.dir/start.map             -> start.bsp (qbsp)
 *   game/pak0.dir/maps/start.dir/start.wad             -> start.bsp (qbsp)
 *   game/pak0.dir/maps/start.dir/start.dir/text1.mip   -> start.wad (rewad)
 *   game/pak0.dir/maps/start.dir/start.dir/text2.mip   -> start.wad (rewad)
 *   game/pak0.dir/progs/ogre.mdl                       -> pak0.pak  (repak)
 *   game/pak0.dir/progs/ogre.dir/ogre-0.tri            -> ogre.mdl  (mdlgen)
 *   game/pak0.dir/progs/ogre.dir/ogre-1.tri            -> ogre.mdl  (mdlgen)
 *   game/pak0.dir/progs/ogre.dir/ogre-0.skn            -> ogre.mdl  (mdlgen)
 *   game/pak0.dir/progs/fire.spr                       -> pak0.pak  (repak)
 *   game/pak0.dir/progs/fire.dir/fire-0.frm            -> fire.spr  (sprgen)
 *   game/pak0.dir/progs/fire.dir/fire-1.frm            -> fire.spr  (sprgen)
 */

bool processType(char *procName, filetype procType,
		 char *destDir,
		 char *outName, filetype outType,
		 char *arcName, filetype arcType,
		 operation procOper, 
		 bool script, bool recurse) {
  FILE *destFile = 0, *scrpFile = 0;
  FILE *procFile = 0;

#ifdef	PRINTCALLS
  mprintf("processType(%s, %d, %s, %s, %d, %s, %d, %d, %d, %d)\n", procName, procType, destDir, outName, outType, arcName, arcType, procOper, script, recurse);
#endif

  if ((procFile = fopen(procName, READ_BINARY))) {
    /*
     * get extension 
     */
    if ((procType == TYPE_PPM) && (!outType)) {
      eprintf("you must specify an output if i should convert from an ppm\n");
    }
    else {
      struct palpic *inPic = 0;
      struct rawdata *inData = 0;
      char idxName[NAMELEN_PATH + 1];

      strncpy(idxName, destDir, NAMELEN_PATH - 1);
      strncat(idxName, procName, NAMELEN_PATH - 1);
      ReplaceExt(idxName, "idx");

      /*
       * script-based execution 
       */
      if (procType == TYPE_INDEX) {
	char command[NAMELEN_PATH], parameter[NAMELEN_PATH];
	char *nameStack[NAMELEN_PATH]; short int nameCount = 0;
	char *thisName;
	nameStack[nameCount] = 0;

	/*
	 * process are now stackable -> proces ... process ... finish ... finish
	 */
	while (fscanf(procFile, "%256s %256[^\n]\n", command, parameter) != EOF) {
	  if (!strcmp(command, "process")) {
	    if(!(thisName = nameStack[++nameCount] = smalloc(parameter)))
	      eprintf("cannot allocate process-stack");
	  }
	  else if ((!strcmp(command, "dither")) && (thisName)) {
	    char Bool;
	    int Value;

	    sscanf(parameter, "%c with %d", &Bool, &Value);
	    if (Bool == 1) {
	      /*
	       * activate dithering with specified value 
	       */
	      dither = TRUE;
	      dithervalue = Value;
	    }
	    else if (Bool == 0) {
	      /*
	       * deactivate dithering and reset dithervalue 
	       */
	      dither = FALSE;
	      dithervalue = 16;
	    }
	  }
	  else if ((!strcmp(command, "compress")) && (thisName)) {
	    char Bool;
	    int Value;

	    sscanf(parameter, "%c with %d", &Bool, &Value);
	    if (Bool == 1)
	      Compression = Value;
	    else if (Bool == 0)
	      Compression = CMP_NONE;
	  }
	  else if ((!strcmp(command, "update")) && (thisName)) {
	    char fileName[NAMELEN_PATH], entryName[NAMELEN_MAXQUAKE], entryType;

	    sscanf(parameter, "%256s as %56s as %c", fileName, entryName, &entryType);
	    processName(fileName, destDir, thisName, outType, entryName, entryType, OP_UPDATE, script, recurse);
	  }
	  else if ((!strcmp(command, "replace")) && (thisName)) {
	    char fileName[NAMELEN_PATH], entryName[NAMELEN_MAXQUAKE], entryType;

	    sscanf(parameter, "%256s as %56s as %c", fileName, entryName, &entryType);
	    processName(fileName, destDir, thisName, outType, entryName, entryType, OP_REPLACE, script, recurse);
	  }
	  else if ((!strcmp(command, "add")) && (thisName)) {
	    char fileName[NAMELEN_PATH], entryName[NAMELEN_MAXQUAKE], entryType;

	    sscanf(parameter, "%256s as %56s as %c", fileName, entryName, &entryType);
	    processName(fileName, destDir, thisName, outType, entryName, entryType, OP_ADD, script, recurse);
	  }
	  else if ((!strcmp(command, "delete")) && (thisName)) {
	    char entryName[NAMELEN_MAXQUAKE], entryType;

	    sscanf(parameter, "%56s as %c", entryName, &entryType);
	    processName(thisName, 0, 0, outType, entryName, entryType, OP_DELETE, script, recurse);
	  }
	  else if ((!strcmp(command, "extract")) && (thisName)) {
	    char fileName[NAMELEN_PATH], entryName[NAMELEN_MAXQUAKE], entryType;

	    sscanf(parameter, "%256s as %56s as %c", entryName, fileName, &entryType);
	    processName(thisName, destDir, fileName, outType, entryName, entryType, OP_EXTRACT, script, recurse);
	  }
	  else if ((!strcmp(command, "finish")) && (thisName)) {
	    tfree(nameStack[nameCount]);
	    thisName = nameStack[--nameCount];
	  }
	  else
	    eprintf("unknown command, cannot execute line \"%s\" ... \n", command);
	}
	/*
	 * tfree resources in case of unfinished processes 
	 */
	while(nameCount)
	  tfree(nameStack[nameCount--]);
      }
      /*
       * directory-based execution 
       */
      else if (procType == TYPE_DIRECTORY) {
	DIR *procDir = 0;
	struct dirent *dirEnt = 0;

	if((procDir = opendir(procName))) {
	  while((dirEnt = readdir(procDir))) {
	    if(strcmp(dirEnt->d_name, ".") && strcmp(dirEnt->d_name, "..")) {
	      char dirName[NAMELEN_PATH];
	    
	      strncpy(dirName, procName, NAMELEN_PATH - 1);
	      strncat(dirName, "/", NAMELEN_PATH - 1);
	      strncat(dirName, dirEnt->d_name, NAMELEN_PATH - 1);
	      /*
	       * if outName == 0, process the dir entirely
	       * else add the complete to outName
	       */
	      processName(dirName, destDir, outName, outType, 0, arcType, procOper, script, recurse);
	    }
	  }
	  closedir(procDir);
	}
	else
	  eprintf("cannot access directory %s\n", procName);
      }
      /*
       * parts to archive or ppm 
       */
      else if ((procType == TYPE_PALETTE) ||
	       (procType == TYPE_CONFIG) ||
	       (procType == TYPE_DEMO) ||
	       (procType == TYPE_FRAME) ||
	       (procType == TYPE_ILBM) ||
	       (procType == TYPE_JPEG) ||
	       (procType == TYPE_LIT) ||
	       (procType == TYPE_LUMP) ||
	       (procType == TYPE_MAP) ||
	       (procType == TYPE_MIPMAP) ||
	       (procType == TYPE_PNG) ||
	       (procType == TYPE_PPM) ||
	       (procType == TYPE_RESOURCE) ||
	       (procType == TYPE_SKIN) ||
	       (procType == TYPE_STATUSBAR) ||
	       (procType == TYPE_TRIANGLE) ||
	       (procType == TYPE_VIS) ||
	       (procType == TYPE_WAL) ||
	       (procType == TYPE_WAVE)) {
	if (procType == TYPE_MIPMAP) {
	  mprintf("read mipmap %s\t-> ", procName);
	  inPic = GetMipMap(procFile, 0);
	}
	if (procType == TYPE_WAL) {
	  mprintf("read wal %s\t-> ", procName);
	  inPic = 0;	/* GetWal(procFile, 0); */
	}
	else if (procType == TYPE_LUMP) {
	  mprintf("read lump %s\t-> ", procName);
	  inPic = GetLMP(procFile, arcName);
	}
	else if (procType == TYPE_IMAGE) {
	  short int alignX = 1, alignY = 1;
	  
	  /*
	   * special size-rules if we convert to a mipmap
	   */
	  if ((outType == TYPE_MIPMAP) || (outType == TYPE_WAD2) || (outType == TYPE_BSP)) {
	    alignX = alignY = 16;
	    if (outName[0] == WARP_MIPMAP)
	      alignX = alignY = WARP_X;
	    else if (!strncmp(outName, SKY_MIPMAP, 3)) {
	      alignX = -(SKY_X);
	      alignY = -(SKY_Y);
	    }
	  }
	  mprintf("read image %s\t-> ", procName);
	  inPic = GetImage(procFile, arcName, alignX, alignY);
	}
	else if (procType == TYPE_TRIANGLE) {
	  mprintf("read tri %s\t-> ", procName);
	  inData = 0;
	}
	else if (procType == TYPE_IMAGINE) {
	  mprintf("read iob %s\t-> ", procName);
	  inData = GetRaw(procFile, procName, 0);
	}
	else if (procType == TYPE_MAP) {
	  mprintf("read map %s\t-> ", procName);
	  inData = GetRaw(procFile, procName, 0);
	}
	else {
	  mprintf("read raw %s\t-> ", procName);
	  inData = GetRaw(procFile, procName, 0);
	}

	if (inPic || inData) {
	  /*
	   * convert to ppm 
	   */
	  if (((outType == TYPE_PPM)
	    || (outType == TYPE_JPEG)
	    || (outType == TYPE_ILBM)
	    || (outType == TYPE_PNG)) && inPic) {
	    mprintf("write image %s\n", outName);
	    if ((destFile = fopen(outName, WRITE_BINARY))) {
	      if (PutImage(destFile, inPic, outType) == FALSE)
		eprintf("failed to convert pic\n");
	      fclose(destFile);
	    }
	    else
	      eprintf("cannot open %s\n", outName);
	  }
	  /*
	   * convert to mip 
	   */
	  else if ((outType == TYPE_MIPMAP) && inPic) {
	    mprintf("write mip %s\n", outName);
	    if ((destFile = fopen(outName, WRITE_BINARY))) {
	      if (PutMipMap(destFile, inPic) == FALSE)
		eprintf("failed to convert to mip\n");
	      fclose(destFile);
	    }
	    else
	      eprintf("cannot open %s\n", outName);
	  }
	  /*
	   * convert to lump
	   */
	  else if ((outType == TYPE_LUMP) && inPic) {
	    mprintf("write lump %s\n", outName);
	    if ((destFile = fopen(outName, WRITE_BINARY))) {
	      if (PutLMP(destFile, inPic) == FALSE)
		eprintf("failed to convert to lump\n");
	      fclose(destFile);
	    }
	    else
	      eprintf("cannot open %s\n", outName);
	  }
	  /*
	   * convert to imagine
	   */
	  else if (((outType == TYPE_IMAGINE) || (outType == TYPE_MAP)) && ((procType == TYPE_MAP) || (procType == TYPE_IMAGINE)) && inData) {
	    struct memory bspStatic;
	    __memBase = &bspStatic;
	    bool retval = FALSE;
	    
	    mprintf("write to %s\n", outName);
	    if ((destFile = fopen(outName, outType == TYPE_MAP ? "w" : WRITE_BINARY))) {
	      AllocClusters(bspMem, ALL_MAPS);
	      
	      if (procType == TYPE_MAP)
	        retval = LoadMapFile(bspMem, inData->rawdata);
	      else
	        retval = LoadTDDDFile(bspMem, inData->rawdata);
printf("loaded\n");
	      if (retval == TRUE) {
	        if (outType == TYPE_MAP)
	          retval = SaveMapFile(bspMem, destFile);
	        else
	          retval = SaveTDDDFile(bspMem, destFile);
printf("saved\n");
	        if (retval != TRUE)
	          eprintf("failed to save %s\n", outName);
	      }
	      else
	        eprintf("failed to parse %s\n", procName);

	      FreeClusters(bspMem, 0);
	      fclose(destFile);
	    }
	    else
	      eprintf("cannot open %s\n", outName);
	  }
	  /*
	   * OP_ADD to pak-file 
	   */
	  else if (outType == TYPE_PACK) {
	    mprintf("write to %s to pak %s\n", procName, outName);
	    if (AddPAK(inPic, inData, outName, procOper) != TRUE)
	      eprintf("failed to add %s to pak %s\n", procName, outName);
	  }
	  /*
	   * OP_ADD to wad2-file 
	   */
	  else if (outType == TYPE_WAD2) {
	    mprintf("write to wad %s\n", outName);
	    if (AddWAD2(inPic, inData, outName, procOper, arcType) != TRUE)
	      eprintf("failed to add %s to wad %s\n", procName, outName);
	  }
	  /*
	   * OP_ADD to bsp-file 
	   */
	  else if (outType == TYPE_BSP) {
	    mprintf("write to bsp %s\n", outName);
	    if (AddBSP(inPic, inData, outName, procOper, arcType) != TRUE)
	      eprintf("failed to add %s to bsp %s\n", procName, outName);
	  }
	  /*
	   * OP_ADD to mdl-file 
	   */
	  else if (outType == TYPE_MODEL) {
	  }
	  /*
	   * OP_ADD to spr-file 
	   */
	  else if (outType == TYPE_SPRITE) {
	  }
	  else
	    eprintf("dont know how to convert %s to %s\n", procName, outName);

	  /*
	   * tfree resources 
	   */
	  if (inPic)
	    pfree(inPic);
	  else if (inData)
	    rfree(inData);
	}
	else
	  eprintf("failed to read %s\n", procName);
      }
      else if (procType == TYPE_QUAKEC) {
        char srcDir[NAMELEN_PATH], *tmp;
          
        strncpy(srcDir, procName, NAMELEN_PATH - 1);
        /* find parent dir */
        if((tmp = (char *)rindex(srcDir, '/')))
          tmp[1] = '\0';
        else
          srcDir[0] = '\0';
        if(qcc(procFile, srcDir, procOper) != TRUE)
          eprintf("cannot compile full %s\n", procName);
      }
      /*
       * archives to archive 
       */
      else if ((outType == TYPE_PACK) && (procType != TYPE_PACK)) {
	mprintf("add %s %s to pak %s\n", procType, procName, outName);
	if ((inData = GetRaw(procFile, arcName, 0))) {
	  if (AddPAK(0, inData, outName, procOper) != TRUE)
	    eprintf("failed to add %s to pak %s\n", procName, outName);
	  rfree(inData);
	}
	else
	  eprintf("failed to read %s\n", procName);
      }
      /*
       * archive to parts 
       */
      else if ((procType == TYPE_PACK) ||
	       (procType == TYPE_WAD2) ||
	       (procType == TYPE_SPRITE) ||
	       (procType == TYPE_MODEL) ||
	       (procType == TYPE_CODE) ||
	       (procType == TYPE_BSP)) {
	if (script == TRUE) {
	  if ((scrpFile = fopen(idxName, WRITE_BINARY))) {
	    fprintf(scrpFile, "process %s\n", procName);
	    if (dither == TRUE)
	      fprintf(scrpFile, "dither %1d with %3d\n", 1, dithervalue);
	    else
	      fprintf(scrpFile, "dither %1d with %3d\n", 0, dithervalue);
	    if (Compression != CMP_NONE)
	      fprintf(scrpFile, "compress %1d with %3d\n", 1, Compression);
	    else
	      fprintf(scrpFile, "compress %1d with %3d\n", 0, Compression);
	  }
	}
	if ((!destDir) || (*destDir == '\0')) {
	  ReplaceExt(idxName, "dir/");
	  destDir = idxName;
	}

	/*
	 * pak to parts 
	 */
	if (procType == TYPE_PACK) {
	  mprintf("extract %s from pak %s\n", arcName, procName);
	  if (ExtractPAK(procFile, scrpFile, destDir, arcName, outType, procOper, recurse) != TRUE)
	    eprintf("failed to extract full pak %s\n", procName);
	}
	/*
	 * wad to parts 
	 */
	else if (procType == TYPE_WAD2) {
	  mprintf("extract %s from wad %s\n", arcName, procName);
	  if (ExtractWAD2(procFile, scrpFile, destDir, arcName, outType, procOper, arcType) != TRUE)
	    eprintf("failed to extract full wad %s\n", procName);
	}
	/*
	 * bsp to parts 
	 */
	else if (procType == TYPE_BSP) {
	  mprintf("extract %s from bsp %s\n", arcName, procName);
	  if (ExtractBSP(procFile, scrpFile, destDir, arcName, outType, procOper, recurse) != TRUE)
	    eprintf("failed to extract full bsp %s\n", procName);
	}
	/*
	 * mdl to parts 
	 */
	else if (procType == TYPE_MODEL) {
	}
	/*
	 * spr to parts 
	 */
	else if (procType == TYPE_SPRITE) {
	}
	/*
	 * code to parts 
	 */
	else if (procType == TYPE_CODE) {
          if(unqcc(procFile, destDir, procOper) != TRUE)
            eprintf("cannot decompile full %s to %s\n", procName, destDir);
	}

	if (scrpFile) {
	  fprintf(scrpFile, "finish %s\n", procName);
	  fclose(scrpFile);
	  scrpFile = 0;
	}
      }
      else
	eprintf("dont know what to convert %s\n", procName);
    }
    fclose(procFile);
  }
  else
    eprintf("cannot open %s\n", procName);

  return TRUE;
}

/*
 * one of outName and outType MUST be given
 */
bool processName(char *procName, char *destDir, char *outName, filetype outType, char *arcName, filetype arcType,
		 operation procOper, bool script, bool recurse) {
  char *procExt = GetExt(procName);
  char *outExt = GetExt(outName);
  filetype procType = TYPE_UNKNOWN;
  char newName[NAMELEN_PATH + 1];
  
  /*
   * set default proctype
   */
  if(!strcmp(procExt, "pal"))      { procType = TYPE_PALETTE; }
  else if(!strcmp(procExt, "bsp")) { procType = TYPE_BSP; }
  else if(!strcmp(procExt, "cfg")) { procType = TYPE_CONFIG; }
  else if(!strcmp(procExt, "dat")) { procType = TYPE_CODE; }
  else if(!strcmp(procExt, "dem")) { procType = TYPE_DEMO; }
  else if(!strcmp(procExt, "dir")) { procType = TYPE_DIRECTORY; }
  else if(!strcmp(procExt, "frm")) { procType = TYPE_LUMP; }
  else if(!strcmp(procExt, "idx")) { procType = TYPE_INDEX; }
  else if(!strcmp(procExt, "iff")) { procType = TYPE_IMAGE; }
  else if(!strcmp(procExt, "iob")) { procType = TYPE_IMAGINE; }
  else if(!strcmp(procExt, "jpg")) { procType = TYPE_IMAGE; }
  else if(!strcmp(procExt, "lbm")) { procType = TYPE_IMAGE; }
  else if(!strcmp(procExt, "lit")) { procType = TYPE_LIT; }
  else if(!strcmp(procExt, "lmp")) { procType = TYPE_LUMP; }
  else if(!strcmp(procExt, "map")) { procType = TYPE_MAP; }
  else if(!strcmp(procExt, "mdl")) { procType = TYPE_MODEL; }
  else if(!strcmp(procExt, "mip")) { procType = TYPE_MIPMAP; }
  else if(!strcmp(procExt, "pak")) { procType = TYPE_PACK; }
  else if(!strcmp(procExt, "png")) { procType = TYPE_IMAGE; }
  else if(!strcmp(procExt, "ppm")) { procType = TYPE_IMAGE; }
  else if(!strcmp(procExt, "pnm")) { procType = TYPE_IMAGE; }
  else if(!strcmp(procExt, "prt")) { procType = TYPE_PRT; }
  else if(!strcmp(procExt, "rc"))  { procType = TYPE_RESOURCE; }
  else if(!strcmp(procExt, "skn")) { procType = TYPE_LUMP; }
  else if(!strcmp(procExt, "spr")) { procType = TYPE_SPRITE; }
  else if(!strcmp(procExt, "src")) { procType = TYPE_QUAKEC; }
  else if(!strcmp(procExt, "stb")) { procType = TYPE_LUMP; }
  else if(!strcmp(procExt, "tri")) { procType = TYPE_TRIANGLE; }
  else if(!strcmp(procExt, "vis")) { procType = TYPE_VIS; }
  else if(!strcmp(procExt, "wad")) { procType = TYPE_WAD2; }
  else if(!strcmp(procExt, "wal")) { procType = TYPE_WAL; }
  else if(!strcmp(procExt, "wav")) { procType = TYPE_WAVE; }

  /*
   * set default outtype
   */
  if(outName) {
    if (outType == TYPE_NONE) {
      if(!strcmp(outExt, "pal"))      { outType = TYPE_PALETTE; }
      else if(!strcmp(outExt, "bsp")) { outType = TYPE_BSP; }
      else if(!strcmp(outExt, "cfg")) { outType = TYPE_CONFIG; }
      else if(!strcmp(outExt, "dat")) { outType = TYPE_CODE; }
      else if(!strcmp(outExt, "dem")) { outType = TYPE_DEMO; }
      else if(!strcmp(outExt, "dir")) { outType = TYPE_DIRECTORY; }
      else if(!strcmp(outExt, "frm")) { outType = TYPE_FRAME; }
      else if(!strcmp(outExt, "idx")) { outType = TYPE_INDEX; }
      else if(!strcmp(outExt, "iff")) { outType = TYPE_ILBM; }
      else if(!strcmp(outExt, "iob")) { outType = TYPE_IMAGINE; }
      else if(!strcmp(outExt, "jpg")) { outType = TYPE_JPEG; }
      else if(!strcmp(outExt, "lbm")) { outType = TYPE_ILBM; }
      else if(!strcmp(outExt, "lit")) { outType = TYPE_LIT; }
      else if(!strcmp(outExt, "lmp")) { outType = TYPE_LUMP; }
      else if(!strcmp(outExt, "map")) { outType = TYPE_MAP; }
      else if(!strcmp(outExt, "mdl")) { outType = TYPE_MODEL; }
      else if(!strcmp(outExt, "mip")) { outType = TYPE_MIPMAP; }
      else if(!strcmp(outExt, "pak")) { outType = TYPE_PACK; }
      else if(!strcmp(outExt, "png")) { outType = TYPE_PNG; }
      else if(!strcmp(outExt, "ppm")) { outType = TYPE_PPM; }
      else if(!strcmp(outExt, "pnm")) { outType = TYPE_PPM; }
      else if(!strcmp(outExt, "prt")) { outType = TYPE_PRT; }
      else if(!strcmp(outExt, "rc"))  { outType = TYPE_RESOURCE; }
      else if(!strcmp(outExt, "skn")) { outType = TYPE_SKIN; }
      else if(!strcmp(outExt, "spr")) { outType = TYPE_SPRITE; }
      else if(!strcmp(outExt, "src")) { outType = TYPE_QUAKEC; }
      else if(!strcmp(outExt, "stb")) { outType = TYPE_STATUSBAR; }
      else if(!strcmp(outExt, "tri")) { outType = TYPE_TRIANGLE; }
      else if(!strcmp(outExt, "vis")) { outType = TYPE_VIS; }
      else if(!strcmp(outExt, "wad")) { outType = TYPE_WAD2; }
      else if(!strcmp(outExt, "wal")) { outType = TYPE_WAL; }
      else if(!strcmp(outExt, "wav")) { outType = TYPE_WAVE; }
    }
    strncpy(newName, outName, NAMELEN_PATH - 1);
  }
  /*
   * set default outname
   */
  else {
    strncpy(newName, destDir, NAMELEN_PATH - 1);
    strncat(newName, procName, NAMELEN_PATH - 1);
    switch (outType) {
      case TYPE_PPM:    ReplaceExt(newName, "ppm"); break;
      case TYPE_JPEG:   ReplaceExt(newName, "jpg"); break;
      case TYPE_ILBM:   ReplaceExt(newName, "iff"); break;
      case TYPE_PNG:    ReplaceExt(newName, "png"); break;
      case TYPE_WAL:    ReplaceExt(newName, "wal"); break;
      case TYPE_MIPMAP: ReplaceExt(newName, "mip"); break;
      case TYPE_LUMP:   ReplaceExt(newName, "lmp"); break;
      default:					    break;
    }
  }

  /*
   * set default arcname
   *
   * behaviour:
   *  * / *.* -> *.pak	arcName = * / *.*
   *  * / *.* -> *.wad	arcName =     *
   *  * / *.* -> *.bsp	arcName =     *
   *  *.pak -> * / *.*	arcName = * / *.*
   *  *.wad -> * / *.*	arcName =     *
   *  *.bsp -> * / *.*	arcName =     *
   *  *.bsp -> *.wad	arcName = *.wad
   */
  if(!arcName) {
    if((outType == TYPE_PACK) ||
	((outType == TYPE_WAD2) && (procType != TYPE_WAD2)) ||
	(outType == TYPE_SPRITE) ||
	(outType == TYPE_MODEL) ||
	(outType == TYPE_BSP)) {
      arcName = smalloc(procName);
      if(arcType == TYPE_NONE)
        arcType = procType;
    }
    else {
      arcName = smalloc(outName);
      if(arcType == TYPE_NONE)
        arcType = outType;
    }
  }
  if(!( ((outType == TYPE_PACK) || (procType == TYPE_PACK)) ||
        (((outType == TYPE_MAP) || (outType == TYPE_IMAGINE) || (outType == TYPE_LIT) || (outType == TYPE_VIS) || (outType == TYPE_WAD2)) && (procType == TYPE_BSP)) ||
        ((outType == TYPE_WAD2) && (procType == TYPE_WAD2)) )) {
    arcName = GetFile(arcName);
    StripExt(arcName);
  }
  
  /*
   * if the dirname doesn't end with / we must add it, hmpf
   */
  if(destDir) {
    int len = strlen(destDir);
  
    if(!((destDir[len - 1] == '/') || (destDir[len - 1] == ':') || (destDir[len - 1] == '\0'))) {
      char *newDir;
      
      if((newDir = (char *)tmalloc(len + 2))) {
        strcpy(newDir, destDir);
        newDir[len] = '/';
        destDir = newDir;
printf("%s\n", destDir);
      }
    }
  }

  outName = newName;
  return processType(procName, procType, destDir, outName, outType, arcName, arcType, procOper, script, recurse);
}
