#define	LIBQTOOLS_CORE
#include "../include/libqtools.h"
#include "../include/libqbuild.h"

/* light */
bool waterlit = FALSE;
float scale = 0, range = 0;
/* qbsp */
bool watervis = FALSE, slimevis = FALSE;
bool nofill = FALSE, notjunc = FALSE, noclip = FALSE, onlyents = FALSE, usehulls = FALSE;
int subdivide = 0, hullnum = 0;
/* vis */
bool fastvis = FALSE;
int vislevel = 2;

bool AddBSP(struct palpic *inPic, struct rawdata *inData, char *bspName, operation procOper, filetype inType)
{
  bool retval = FALSE;
  bool appendPic = FALSE;
  char *procName = 0;
  char *srcName;
  FILE *bspFile = 0;
  __memBase = 0;

#ifdef	MEM_SIZETRACK
  mprintf(" memory  :        %7i (       %7i)\n", memcounter, mempeak);
#ifdef	MEM_ANALYSE
  mprintf("                  %7i (       %7i)\n", memallocs, mempeakallocs);
  mprintf("                  %7i (       %7i)\n", memcounter/memallocs, mempeak/mempeakallocs);
#endif
#endif

  if (inPic) {
    procName = inPic->name;
    appendPic = TRUE;
  }
  else if (inData) {
    procName = inData->name;
    appendPic = FALSE;
  }
  srcName = GetExt(procName);

  if (!strcmp(srcName, "wad") || (inType == TYPE_WAD2))
    appendPic = TRUE;

  if (appendPic == TRUE) {
  }
  else {
    bool newBsp = FALSE, newLit = FALSE, newVis = FALSE;
    bool oldBsp = FALSE, oldLit = FALSE, oldVis = FALSE;

    if (!strcmp(srcName, "map") || (inType == TYPE_MAP)) {
      newBsp = TRUE;
      inType = TYPE_MAP;
    }
    else if (!strcmp(srcName, "iob") || (inType == TYPE_IMAGINE)) {
      newBsp = TRUE;
      inType = TYPE_IMAGINE;
    }
    else if (!strcmp(srcName, "prt") || (inType == TYPE_PRT)) {
      newVis = TRUE;
      inType = TYPE_PRT;
    }
    else if (!strcmp(srcName, "vis") || (inType == TYPE_VIS)) {
      oldVis = oldBsp = TRUE;
      inType = TYPE_VIS;
    }
    else if (!strcmp(srcName, "lit") || (inType == TYPE_LIT)) {
      oldLit = oldBsp = TRUE;
      inType = TYPE_LIT;
    }
    else if (inType == TYPE_LIT) {
      newLit = TRUE;
      inType = TYPE_LIT;
    }

    while (1) {
      if (!setjmp(eabort)) {
	if (oldBsp == TRUE) {
	  if (!(bspFile = fopen(bspName, READWRITE_BINARY_OLD))) {
	    eprintf("failed to open file %s\n", bspName);
	    break;
	  }
	  if (!(bspMem = LoadBSP(bspFile, ALL_LUMPS))) {
	    eprintf("failed to load file %s\n", bspName);
	    break;
	  }
	  
	  bspMem->bspOptions |= QBSP_NOTEXTURES;
	  
	  fclose(bspFile);
	  if (!(bspFile = fopen(bspName, WRITE_BINARY))) {
	    eprintf("failed to re-open file %s\n", bspName);
	    break;
	  }
	}
	else {
	  if (!(bspFile = fopen(bspName, WRITE_BINARY))) {
	    eprintf("failed to open new file %s\n", bspName);
	    break;
	  }
	  if (!(bspMem = (struct memory *)tmalloc(sizeof(struct memory)))) {
	    eprintf("failed to allocate bspMem\n");
	    break;
	  }

	  // init the tables to be shared by all models
	  BeginBSPFile(bspMem);
	  bspMem->mapOptions |= ((newLit == TRUE) ? MAP_LOADLIGHTS : 0);
	  bspMem->bspOptions |= ((watervis == TRUE) ? QBSP_WATERVIS : 0);
	  bspMem->bspOptions |= ((slimevis == TRUE) ? QBSP_SLIMEVIS : 0);
	  bspMem->bspOptions |= ((nofill == TRUE) ? QBSP_NOFILL : 0);
	  bspMem->bspOptions |= ((notjunc == TRUE) ? QBSP_NOTJUNC : 0);
	  bspMem->bspOptions |= ((noclip == TRUE) ? QBSP_NOCLIP : 0);
	  bspMem->bspOptions |= ((onlyents == TRUE) ? QBSP_ONLYENTS : 0);
	  bspMem->bspOptions |= ((usehulls == TRUE) ? QBSP_USEHULLS : 0);

	  // load brushes and bspMem->mapentities
	  if (inType == TYPE_IMAGINE) {
	    if((retval = LoadTDDDFile(bspMem, inData->rawdata)) == FALSE) {
	      eprintf("failed to load TDDD\n");
	      break;
	    }
	  }
	  else if (inType == TYPE_MAP) {
	    if ((retval = LoadMapFile(bspMem, inData->rawdata)) == FALSE) {
	      eprintf("failed to load map\n");
	      break;
	    }
	  }
	  if ((retval = qbsp(bspMem, hullnum, subdivide, bspName)) == FALSE) {
	    eprintf("failed to calculate bsp-tree\n");
	    break;
	  }
	}

	bspMem->litOptions |= ((newLit == TRUE) ? LIGHT_MEM : 0);
	bspMem->litOptions |= ((waterlit == TRUE) ? LIGHT_WATERLIT : 0);
	bspMem->visOptions |= ((fastvis == TRUE) ? VIS_FAST : 0);
	bspMem->visOptions |= ((verbose == TRUE) ? VIS_VERBOSE : 0);
   
	if (newVis == TRUE) {
	  mprintf("build vis-data\n");
	  if ((retval = vis(bspMem, vislevel, inData->rawdata)) == FALSE) {
	    eprintf("failed to calculate vis-data\n");
	    break;
	  }
	}
	else if(oldVis == TRUE) {
	  mprintf("replace vis-data\n");
	  FreeClusters(bspMem, LUMP_VISIBILITY);
	  bspMem->dvisdata = inData->rawdata;
	  bspMem->visdatasize = inData->size;
	  WriteBSP(bspFile, bspMem);
	  retval = TRUE;
	  break;
	}
	
	if (newLit == TRUE) {
	  mprintf("build lit-data\n");
	  if ((retval = light(bspMem, scale, range)) == FALSE) {
	    eprintf("failed to calculate lit-data\n");
	    break;
	  }
	}
	else if(oldLit == TRUE) {
	  mprintf("replace lit-data\n");
	  FreeClusters(bspMem, LUMP_LIGHTING);
	  bspMem->dlightdata = inData->rawdata;
	  bspMem->lightdatasize = inData->size;
	  WriteBSP(bspFile, bspMem);
	  retval = TRUE;
	  break;
	}

	FinishBSPFile(bspMem, bspFile);
	retval = TRUE;
      }
      break;
    }
  }
  if (bspFile)
    fclose(bspFile);
  if (bspMem) {
    FreeClusters(bspMem, 0);
    tfree(bspMem);
  }

#ifdef	MEM_SIZETRACK
  mprintf(" memory  :        %7i (       %7i)\n", memcounter, mempeak);
#ifdef	MEM_ANALYSE
  mprintf("                  %7i (       %7i)\n", memallocs, mempeakallocs);
  mprintf("                  %7i (       %7i)\n", memcounter/memallocs, mempeak/mempeakallocs);
#endif
#endif
  return retval;
}

/*
 * BSP-tools
 * destDir == 0 -> OP_EXTRACT to current directory
 */
bool ExtractBSP(FILE * bspFile, FILE * script, char *destDir, char *entryName, filetype outType, operation procOper, bool recurse)
{
  FILE *outFile;
  bool retval = FALSE;
  __memBase = 0;

#ifdef	PRINTCALLS
  mprintf("ExtractBSP(%lx, %lx, %s, %s, %d, %d, %d)\n", bspFile, script, destDir, entryName, outType, procOper, recurse);
#endif

  if (!setjmp(eabort)) {
    if ((bspMem = LoadBSP(bspFile, LUMP_TEXTURES			       // BSPtoMap/BSPtoWAD/BSPtoMip
			   | LUMP_ENTITIES				       // BSPtoMap
			   | LUMP_PLANES				       // BSPtoMap
			   | LUMP_FACES					       // BSPtoMap
			   | LUMP_EDGES					       // GetEdges
			   | LUMP_SURFEDGES				       // GetEdges
			   | LUMP_MODELS				       // BSPtoMap
			   | LUMP_VERTEXES				       // BSPtoMap
			   | LUMP_TEXINFO				       // BSPtoMap
			   | LUMP_VISIBILITY				       // BSPtoVis
			   | LUMP_LIGHTING				       // BSPtoLit
	 ))) {
      /*
       * save mip-textures 
       */
      int i;
      char destPath[NAMELEN_PATH], *destName;
      bool toWad = FALSE;
      bool toMap = FALSE;
      bool toVis = FALSE;
      bool toLit = FALSE;
      bool toIob = FALSE;

      strncpy(destPath, destDir, NAMELEN_PATH - 1);
      if (!entryName) {
        destName = smalloc(destDir);
        destName[strlen(destName) - 1] = '\0';
        if (outType == TYPE_WAD2)     	  toWad = TRUE;
        else if (outType == TYPE_MAP) 	  toMap = TRUE;
        else if (outType == TYPE_IMAGINE) toMap = toIob = TRUE;
        else if (outType == TYPE_VIS) 	  toVis = TRUE;
        else if (outType == TYPE_LIT)	  toLit = TRUE;
        else				  toWad = toMap = toVis = toLit = TRUE;	/* default: extract all */
        strncat(destPath, GetFile(destName), NAMELEN_PATH - 1);
        tfree(destName);
      }
      else {
        destName = GetExt(entryName);
        strncat(destPath, GetFile(entryName), NAMELEN_PATH - 1);
        if (!strcmp(destName, "wad") || (outType == TYPE_WAD2))	        { entryName = 0; toWad = TRUE; }
        else if (!strcmp(destName, "map") || (outType == TYPE_MAP))     { entryName = 0; toMap = TRUE; }
        else if (!strcmp(destName, "iob") || (outType == TYPE_IMAGINE)) { entryName = 0; toMap = toIob = TRUE; }
        else if (!strcmp(destName, "vis") || (outType == TYPE_VIS))     { entryName = 0; toVis = TRUE; }
        else if (!strcmp(destName, "lit") || (outType == TYPE_LIT))     { entryName = 0; toLit = TRUE; }
      }

      if (procOper == OP_EXTRACT)
        CreatePath(destPath);

      if (bspMem->dtexdata) {
	int *MipOffsets = (int *)bspMem->dtexdata;
	int MipNums = *MipOffsets++;

	/*
	 * decode mips from bsp
	 */
	ReplaceExt(destPath, "wad");
	for (i = 0; i < MipNums; i++) {
	  struct mipmap *Texture = (struct mipmap *)(MipOffsets[i] + bspMem->dtexdata);

	  if (!(entryName && strncmp(entryName, Texture->name, NAMELEN_MIP))) {
	    char fileName[NAMELEN_PATH];

	    strncpy(fileName, destDir, NAMELEN_PATH - 1);
	    strncat(fileName, Texture->name, NAMELEN_PATH - 1);

	    switch (outType) {
	      case TYPE_PPM:	strncat(fileName, ".ppm", NAMELEN_PATH - 1); break;
	      case TYPE_JPEG:	strncat(fileName, ".jpg", NAMELEN_PATH - 1); break;
	      case TYPE_ILBM:	strncat(fileName, ".iff", NAMELEN_PATH - 1); break;
	      case TYPE_PNG:	strncat(fileName, ".png", NAMELEN_PATH - 1); break;
	      case TYPE_NONE:
	      default:		strncat(fileName, ".mip", NAMELEN_PATH - 1); break;
	    }

	    switch (procOper) {
	      case OP_EXTRACT:{
		  struct palpic *MipMap;

		  if ((MipMap = ParseMipMap(Texture, MIPMAP_0))) {
		    if (toWad == TRUE) {
		      mprintf("extract %s to wad %s\n", MipMap->name, destPath);
		      AddWAD2(MipMap, 0, destPath, OP_UPDATE, WAD2_MIPMAP);
		    }
		    else {
		      FILE *fileDst;

		      mprintf("extract %s to %s ...\n", MipMap->name, fileName);
		      CreatePath(fileName);

		      if ((fileDst = fopen(fileName, WRITE_BINARY))) {
			if (outType != TYPE_NONE)
			  retval = PutImage(fileDst, MipMap, outType);
			else
			  retval = PutMipMap(fileDst, MipMap);
			fclose(fileDst);
		      }
		      else
			eprintf("cannot open %s\n", fileName);
		    }
		    pfree(MipMap);
		  }
		  else
		    eprintf("cannot read mipmap %s\n", MipMap->name);
		}
		break;
	      case OP_LIST:
	      case OP_DEFAULT:
	      default:{
		  mprintf("%16s (offset: %8d)\n", Texture->name, MipOffsets[i]);
		  retval = TRUE;
		}
		break;
	    }
	    if (script)
	      fprintf(script, "update %s as %s as %c\n", fileName, Texture->name, WAD2_MIPMAP);
	  }
	}
	if ((toWad == TRUE) && (recurse == TRUE))
	  retval = processName(destPath, 0, 0, outType, 0, 0, procOper, script ? TRUE : FALSE, recurse);
      }

      if ((bspMem->dentdata) && (toMap == TRUE)) {
	/*
	 * decode map from bsp
	 */
	LoadMapFile(bspMem, bspMem->dentdata);
	if (procOper == OP_EXTRACT) {
	  ReplaceExt(destPath, toIob == TRUE ? "iob" : "map");

	  if ((outFile = fopen(destPath, toIob == TRUE ? WRITE_BINARY : "w"))) {
	    LoadBSPFile(bspMem);

	    if(toIob == TRUE)
	      SaveTDDDFile(bspMem, outFile);
	    else
	      SaveMapFile(bspMem, outFile);

	    fclose(outFile);
	  }
	}
	retval = TRUE;
      }

      if ((bspMem->dvisdata) && (toVis == TRUE) && (procOper == OP_EXTRACT)) {
	/*
	 * decode vis from bsp
	 */
	ReplaceExt(destPath, "vis");
	if ((outFile = fopen(destPath, WRITE_BINARY))) {
	  fwrite((void *)bspMem->dvisdata, 1, bspMem->visdatasize * sizeof(unsigned char), outFile);
	  fclose(outFile);
	  retval = TRUE;
	}
      }

      if ((bspMem->dlightdata) && (toLit == TRUE) && (procOper == OP_EXTRACT)) {
	/*
	 * decode lit from bsp
	 */
	ReplaceExt(destPath, "lit");
	if ((outFile = fopen(destPath, WRITE_BINARY))) {
	  fwrite((void *)bspMem->dlightdata, 1, bspMem->lightdatasize * sizeof(unsigned char), outFile);
	  fclose(outFile);
	  retval = TRUE;
	}
      }

      if ((procOper == OP_LIST) || (procOper == OP_DEFAULT))
	PrintClusters(bspMem, 0, FALSE);
      FreeClusters(bspMem, 0);
    }
    else
      eprintf("cannot read bspfile\n");
  }

  /*
   * outstanding:
   *  scripting for bsps
   *  maptotri/iob?
   */

  return retval;
}

/*
 * =============
 * SwapBSPFile
 * 
 * Byte swaps all data in a bsp file.
 * =============
 */
void SwapBSPFile(__memBase, bool toDisk)
{
  int i, c;
  short int j;
  struct dmodel_t *d;
  struct dmiptexlump_t *mtl;

  /*
   * planes
   */
  if (bspMem->availHeaders & LUMP_PLANES)
    for (i = 0; i < bspMem->numplanes; i++) {
      for (j = 0; j < 3; j++)
	bspMem->dplanes[i].normal[j] = LittleFloat(bspMem->dplanes[i].normal[j]);
      bspMem->dplanes[i].dist = LittleFloat(bspMem->dplanes[i].dist);
      bspMem->dplanes[i].type = LittleLong(bspMem->dplanes[i].type);
    }

  /*
   * miptex
   */
  if (bspMem->availHeaders & LUMP_TEXTURES)
    if (bspMem->texdatasize) {
      mtl = (struct dmiptexlump_t *)bspMem->dtexdata;
      if (toDisk == TRUE)
	c = mtl->nummiptex;
      else
	c = LittleLong(mtl->nummiptex);
      mtl->nummiptex = LittleLong(mtl->nummiptex);
      for (i = 0; i < c; i++)
	mtl->dataofs[i] = LittleLong(mtl->dataofs[i]);
    }

  /*
   * vertexes
   */
  if (bspMem->availHeaders & LUMP_VERTEXES)
    for (i = 0; i < bspMem->numvertexes; i++) {
      for (j = 0; j < 3; j++)
	bspMem->dvertexes[i].point[j] = LittleFloat(bspMem->dvertexes[i].point[j]);
    }

  /*
   * nodes
   */
  if (bspMem->availHeaders & LUMP_NODES)
    for (i = 0; i < bspMem->numnodes; i++) {
      bspMem->dnodes[i].planenum = LittleLong(bspMem->dnodes[i].planenum);
      for (j = 0; j < 3; j++) {
	bspMem->dnodes[i].mins[j] = LittleShort(bspMem->dnodes[i].mins[j]);
	bspMem->dnodes[i].maxs[j] = LittleShort(bspMem->dnodes[i].maxs[j]);
      }
      bspMem->dnodes[i].children[0] = LittleShort(bspMem->dnodes[i].children[0]);
      bspMem->dnodes[i].children[1] = LittleShort(bspMem->dnodes[i].children[1]);
      bspMem->dnodes[i].firstface = LittleShort(bspMem->dnodes[i].firstface);
      bspMem->dnodes[i].numfaces = LittleShort(bspMem->dnodes[i].numfaces);
    }

  /*
   * texinfos
   */
  if (bspMem->availHeaders & LUMP_TEXINFO)
    for (i = 0; i < bspMem->numtexinfo; i++) {
      for (j = 0; j < 8; j++)
	bspMem->texinfo[i].vecs[0][j] = LittleFloat(bspMem->texinfo[i].vecs[0][j]);
      bspMem->texinfo[i].miptex = LittleLong(bspMem->texinfo[i].miptex);
      bspMem->texinfo[i].flags = LittleLong(bspMem->texinfo[i].flags);
    }

  /*
   * faces
   */
  if (bspMem->availHeaders & LUMP_FACES)
    for (i = 0; i < bspMem->numfaces; i++) {
      bspMem->dfaces[i].texinfo = LittleShort(bspMem->dfaces[i].texinfo);
      bspMem->dfaces[i].planenum = LittleShort(bspMem->dfaces[i].planenum);
      bspMem->dfaces[i].side = LittleShort(bspMem->dfaces[i].side);
      bspMem->dfaces[i].lightofs = LittleLong(bspMem->dfaces[i].lightofs);
      bspMem->dfaces[i].firstedge = LittleLong(bspMem->dfaces[i].firstedge);
      bspMem->dfaces[i].numedges = LittleShort(bspMem->dfaces[i].numedges);
    }

  /*
   * clipnodes
   */
  if (bspMem->availHeaders & LUMP_CLIPNODES)
    for (i = 0; i < bspMem->numclipnodes; i++) {
      bspMem->dclipnodes[i].planenum = LittleLong(bspMem->dclipnodes[i].planenum);
      bspMem->dclipnodes[i].children[0] = LittleShort(bspMem->dclipnodes[i].children[0]);
      bspMem->dclipnodes[i].children[1] = LittleShort(bspMem->dclipnodes[i].children[1]);
    }

  /*
   * leafs
   */
  if (bspMem->availHeaders & LUMP_LEAFS)
    for (i = 0; i < bspMem->numleafs; i++) {
      bspMem->dleafs[i].contents = LittleLong(bspMem->dleafs[i].contents);
      for (j = 0; j < 3; j++) {
	bspMem->dleafs[i].mins[j] = LittleShort(bspMem->dleafs[i].mins[j]);
	bspMem->dleafs[i].maxs[j] = LittleShort(bspMem->dleafs[i].maxs[j]);
      }
      bspMem->dleafs[i].firstmarksurface = LittleShort(bspMem->dleafs[i].firstmarksurface);
      bspMem->dleafs[i].nummarksurfaces = LittleShort(bspMem->dleafs[i].nummarksurfaces);
      bspMem->dleafs[i].visofs = LittleLong(bspMem->dleafs[i].visofs);
    }

  /*
   * marksurfaces
   */
  if (bspMem->availHeaders & LUMP_MARKSURFACES)
    for (i = 0; i < bspMem->nummarksurfaces; i++)
      bspMem->dmarksurfaces[i] = LittleShort(bspMem->dmarksurfaces[i]);

  /*
   * edges
   */
  if (bspMem->availHeaders & LUMP_EDGES)
    for (i = 0; i < bspMem->numedges; i++) {
      bspMem->dedges[i].v[0] = LittleShort(bspMem->dedges[i].v[0]);
      bspMem->dedges[i].v[1] = LittleShort(bspMem->dedges[i].v[1]);
    }

  /*
   * surfedges
   */
  if (bspMem->availHeaders & LUMP_SURFEDGES)
    for (i = 0; i < bspMem->numsurfedges; i++)
      bspMem->dsurfedges[i] = LittleLong(bspMem->dsurfedges[i]);

  /*
   * models
   */
  if (bspMem->availHeaders & LUMP_MODELS)
    for (i = 0; i < bspMem->nummodels; i++) {
      d = &bspMem->dmodels[i];
      for (j = 0; j < MAX_MAP_HULLS; j++)
	d->headnode[j] = LittleLong(d->headnode[j]);
      d->visleafs = LittleLong(d->visleafs);
      d->firstface = LittleLong(d->firstface);
      d->numfaces = LittleLong(d->numfaces);
      for (j = 0; j < 3; j++) {
	d->mins[j] = LittleFloat(d->mins[j]);
	d->maxs[j] = LittleFloat(d->maxs[j]);
	d->origin[j] = LittleFloat(d->origin[j]);
      }
    }
}

int GetBlock(register FILE *bspFile, register struct dpair *dPair, register void **store, register int partSize)
{
  register int blockSize = LittleLong(dPair->size);

  if ((*store = (void *)tmalloc(blockSize))) {
    fseek(bspFile, LittleLong(dPair->offset), SEEK_SET);
    fread(*store, 1, blockSize, bspFile);
    return (blockSize / partSize);
  }
  else
    return 0;
}

void PutBlock(register FILE *bspFile, register struct dpair *dPair, register void *store, register int blockSize)
{
  if (blockSize) {
    dPair->size = LittleLong(blockSize);
    dPair->offset = LittleLong(ftell(bspFile));
    fwrite(store, 1, (blockSize + 3) & ~3, bspFile);
    /*
     * probably we want to use it after this
     * tfree(store);
     */
  }
  else {
    dPair->size = 0;
    dPair->offset = LittleLong(ftell(bspFile));
  }
}

/*
 * =============
 * LoadBSPFile
 * =============
 */
struct memory *LoadBSP(FILE * bspFile, int availLoad)
{
  __memBase = 0;

  if ((bspMem = (struct memory *)tmalloc(sizeof(struct memory)))) {
    struct bspheader Header;
    memset(&Header, 0, sizeof(struct bspheader));
    memset(bspMem, 0, sizeof(struct memory));

    /*
     * load the file header
     */
    fseek(bspFile, 0, SEEK_SET);
    fread(&Header, 1, sizeof(struct bspheader), bspFile);

    if (Header.version == LittleLong(BSP_VERSION)) {
      bspMem->availHeaders = availLoad;

      if (availLoad & LUMP_ENTITIES)
	bspMem->entdatasize = bspMem->max_entdatasize = GetBlock(bspFile, &Header.entities, (void **)&bspMem->dentdata, sizeof(char));

      if (availLoad & LUMP_PLANES)
	bspMem->numplanes = bspMem->max_numplanes = GetBlock(bspFile, &Header.planes, (void **)&bspMem->dplanes, sizeof(struct dplane_t));

      if (availLoad & LUMP_TEXTURES)
	bspMem->texdatasize = bspMem->max_texdatasize = GetBlock(bspFile, &Header.miptex, (void **)&bspMem->dtexdata, sizeof(unsigned char));

      if (availLoad & LUMP_VERTEXES)
	bspMem->numvertexes = bspMem->max_numvertexes = GetBlock(bspFile, &Header.vertices, (void **)&bspMem->dvertexes, sizeof(struct dvertex_t));

      if (availLoad & LUMP_VISIBILITY)
	bspMem->visdatasize = bspMem->max_visdatasize = GetBlock(bspFile, &Header.visilist, (void **)&bspMem->dvisdata, sizeof(unsigned char));

      if (availLoad & LUMP_NODES)
	bspMem->numnodes = bspMem->max_numnodes = GetBlock(bspFile, &Header.nodes, (void **)&bspMem->dnodes, sizeof(struct dnode_t));

      if (availLoad & LUMP_TEXINFO)
	bspMem->numtexinfo = bspMem->max_numtexinfo = GetBlock(bspFile, &Header.texinfo, (void **)&bspMem->texinfo, sizeof(struct texinfo));

      if (availLoad & LUMP_FACES)
	bspMem->numfaces = bspMem->max_numfaces = GetBlock(bspFile, &Header.faces, (void **)&bspMem->dfaces, sizeof(struct dface_t));

      if (availLoad & LUMP_LIGHTING)
	bspMem->lightdatasize = bspMem->max_lightdatasize = GetBlock(bspFile, &Header.lightmaps, (void **)&bspMem->dlightdata, sizeof(unsigned char));

      if (availLoad & LUMP_CLIPNODES)
	bspMem->numclipnodes = bspMem->max_numclipnodes = GetBlock(bspFile, &Header.clipnodes, (void **)&bspMem->dclipnodes, sizeof(struct dclipnode_t));

      if (availLoad & LUMP_LEAFS)
	bspMem->numleafs = bspMem->max_numleafs = GetBlock(bspFile, &Header.leaves, (void **)&bspMem->dleafs, sizeof(struct dleaf_t));

      if (availLoad & LUMP_MARKSURFACES)
	bspMem->nummarksurfaces = bspMem->max_nummarksurfaces = GetBlock(bspFile, &Header.lface, (void **)&bspMem->dmarksurfaces, sizeof(unsigned short int));

      if (availLoad & LUMP_EDGES)
	bspMem->numedges = bspMem->max_numedges = GetBlock(bspFile, &Header.edges, (void **)&bspMem->dedges, sizeof(struct dedge_t));

      if (availLoad & LUMP_SURFEDGES)
	bspMem->numsurfedges = bspMem->max_numsurfedges = GetBlock(bspFile, &Header.ledges, (void **)&bspMem->dsurfedges, sizeof(int));

      if (availLoad & LUMP_MODELS)
	bspMem->nummodels = bspMem->max_nummodels = GetBlock(bspFile, &Header.models, (void **)&bspMem->dmodels, sizeof(struct dmodel_t));

      /*
       * swap everything
       */
      SwapBSPFile(bspMem, FALSE);
    }
    else {
      tfree(bspMem);
      bspMem = 0;
      eprintf("no valid bsp-file\n");
    }
  }

  return bspMem;
}

/*
 * =============
 * WriteBSPFile
 * =============
 */
void WriteBSP(FILE * bspFile, __memBase)
{
  if (bspMem) {
    struct bspheader Header;
    memset(&Header, 0, sizeof(struct bspheader));

    Header.version = LittleLong(BSP_VERSION);

    /*
     * save the file header
     */
    fseek(bspFile, 0, SEEK_SET);
    fwrite(&Header, 1, sizeof(struct bspheader), bspFile);

    /*
     * swap everything
     */
    SwapBSPFile(bspMem, TRUE);

    if (bspMem->availHeaders & LUMP_PLANES)
      PutBlock(bspFile, &Header.planes, (void *)bspMem->dplanes, bspMem->numplanes * sizeof(struct dplane_t));

    if (bspMem->availHeaders & LUMP_LEAFS)
      PutBlock(bspFile, &Header.leaves, (void *)bspMem->dleafs, bspMem->numleafs * sizeof(struct dleaf_t));

    if (bspMem->availHeaders & LUMP_VERTEXES)
      PutBlock(bspFile, &Header.vertices, (void *)bspMem->dvertexes, bspMem->numvertexes * sizeof(struct dvertex_t));

    if (bspMem->availHeaders & LUMP_NODES)
      PutBlock(bspFile, &Header.nodes, (void *)bspMem->dnodes, bspMem->numnodes * sizeof(struct dnode_t));

    if (bspMem->availHeaders & LUMP_TEXINFO)
      PutBlock(bspFile, &Header.texinfo, (void *)bspMem->texinfo, bspMem->numtexinfo * sizeof(struct texinfo));

    if (bspMem->availHeaders & LUMP_FACES)
      PutBlock(bspFile, &Header.faces, (void *)bspMem->dfaces, bspMem->numfaces * sizeof(struct dface_t));

    if (bspMem->availHeaders & LUMP_CLIPNODES)
      PutBlock(bspFile, &Header.clipnodes, (void *)bspMem->dclipnodes, bspMem->numclipnodes * sizeof(struct dclipnode_t));

    if (bspMem->availHeaders & LUMP_MARKSURFACES)
      PutBlock(bspFile, &Header.lface, (void *)bspMem->dmarksurfaces, bspMem->nummarksurfaces * sizeof(unsigned short int));

    if (bspMem->availHeaders & LUMP_SURFEDGES)
      PutBlock(bspFile, &Header.ledges, (void *)bspMem->dsurfedges, bspMem->numsurfedges * sizeof(int));

    if (bspMem->availHeaders & LUMP_EDGES)
      PutBlock(bspFile, &Header.edges, (void *)bspMem->dedges, bspMem->numedges * sizeof(struct dedge_t));

    if (bspMem->availHeaders & LUMP_MODELS)
      PutBlock(bspFile, &Header.models, (void *)bspMem->dmodels, bspMem->nummodels * sizeof(struct dmodel_t));

    if (bspMem->availHeaders & LUMP_LIGHTING)
      PutBlock(bspFile, &Header.lightmaps, (void *)bspMem->dlightdata, bspMem->lightdatasize * sizeof(unsigned char));

    if (bspMem->availHeaders & LUMP_VISIBILITY)
      PutBlock(bspFile, &Header.visilist, (void *)bspMem->dvisdata, bspMem->visdatasize * sizeof(unsigned char));

    if (bspMem->availHeaders & LUMP_ENTITIES)
      PutBlock(bspFile, &Header.entities, (void *)bspMem->dentdata, bspMem->entdatasize * sizeof(char));

    if (bspMem->availHeaders & LUMP_TEXTURES)
      PutBlock(bspFile, &Header.miptex, (void *)bspMem->dtexdata, bspMem->texdatasize * sizeof(unsigned char));

    /*
     * save the file header
     */
    fseek(bspFile, 0, SEEK_SET);
    fwrite(&Header, 1, sizeof(struct bspheader), bspFile);

    /*
     * tfree(bspMem); 
     */
  }
}
