#define	LIBQSYS_DRIVER
#define	LIBQTOOLS_CORE
#include "../../../include/libqsys.h"
/*
 * needed for tmalloc/.../Error
 * maybe move them to this one (generic?)
 */
#include "../../../include/libqtools.h"

/*
 * display
 */

static struct DisplayDimension dim;

extern struct ExecBase *SysBase;
extern struct DOSBase *DOSBase;
static struct GfxBase *GfxBase;
static struct IntuitionBase *IntuitionBase;
static struct Library *AslBase;
static struct Library *CyberGfxBase;

static int ScreenWidth = 0;
static int ScreenHeight = 0;
static int ScreenDepth = 0;
static int ScreenLog = 0;
static int ScreenLeft = 0;
static int ScreenTop = 0;

static void *FrameBuffer = NULL;		/* actual framebufferaddress */
static int FrameSize = 0;			/* size of updateregion in bytes */
static unsigned char FrameFormat = 0;		/* default RECTFMT_LUT8   ? */
static unsigned short int BytesPerRow = 0;	/* default            8/8 ? */
static unsigned short int BytesPerPel = 0;	/* default            8/8 ? */

static struct Screen *Screen = NULL;
static struct Window *Window = NULL;
static struct ViewPort *ViewPort = NULL;
static struct RastPort *RastPort = NULL;

#define SAVEBACK_COLORS (256 * 3)
static unsigned short int oldNFree, oldShareable;
static long unsigned int OldColorMap[1 + (256 * 3) + SAVEBACK_COLORS];
static long unsigned int NewColorMap[1 + (256 * 3) + SAVEBACK_COLORS];	/* LoadRGB32-Palette */

static void *InitDisplay(void) {
  /*
   * Works with Windows and Screens!!!
   */
  if (ScreenDepth <= 8) {
    BytesPerPel = 1;
    FrameFormat = RECTFMT_LUT8;
  }
  else if (ScreenDepth <= 16) {
    BytesPerPel = 2;
    FrameFormat =         NULL;
  }
  else if (ScreenDepth <= 32) {
    BytesPerPel = 4;
    FrameFormat = RECTFMT_RGB;
  }
  BytesPerRow = ScreenWidth * BytesPerPel;
  FrameSize = BytesPerRow * ScreenHeight;

  if(FrameBuffer != NULL)
    tfree(FrameBuffer);
  if((FrameBuffer = (void *)tmalloc(FrameSize)) == NULL)	// Bitmap allocation failed, what now??
    Error("Cannot allocate FrameBuffer\n");

  return FrameBuffer;
}

struct DisplayDimension *OpenDisplay(int width, int height, int depth, struct rgb *Palette) {
  atexit(CloseDisplay);

  GfxBase = (struct GfxBase *)OpenLibrary((unsigned char *)"graphics.library", (unsigned long)39);
  IntuitionBase = (struct IntuitionBase *)OpenLibrary((unsigned char *)"intuition.library", (unsigned long)39);
  AslBase = OpenLibrary((unsigned char *)"asl.library", (unsigned long)37);
  CyberGfxBase = OpenLibrary((unsigned char *)"cybergraphics.library", (unsigned long)39);

  if((GfxBase == NULL) || (IntuitionBase == NULL) || (AslBase == NULL))
    Error("Cannot open one or more of the required libraries\n");

  ScreenWidth  = width;
  ScreenHeight = height;
  ScreenDepth  = depth;

  Screen = LockPubScreen(NULL);
  if ((ScreenDepth > 8) && (GetBitMapAttr(Screen->RastPort.BitMap,BMA_DEPTH) <= 8))
    Error("Screen must have more than 8bits\n");
  else if ((ScreenDepth = 8) && (GetBitMapAttr(Screen->RastPort.BitMap,BMA_DEPTH) < 8))
    Error("Screen must have at least 8bits\n");
  else {
    if ((Window = OpenWindowTags(NULL,
				 WA_InnerWidth, ScreenWidth,
 				 WA_InnerHeight, ScreenHeight,
				 WA_MinWidth, BASEWIDTH,
				 WA_MinHeight, BASEHEIGHT,
				 WA_MaxWidth, MAXWIDTH,
				 WA_MaxHeight, MAXHEIGHT,
 				 WA_Left, (Screen->Width - ScreenWidth) / 2,
 				 WA_Top, (Screen->Height - ScreenHeight) / 2,
 				 WA_Flags, WFLG_GIMMEZEROZERO | WFLG_SIMPLE_REFRESH | WFLG_DRAGBAR | WFLG_DEPTHGADGET | WFLG_CLOSEGADGET | WFLG_SIZEGADGET | WFLG_SIZEBBOTTOM | WFLG_ACTIVATE,
 				 WA_PubScreen, (long unsigned int)Screen,
  				 WA_RMBTrap, TRUE,
  				 WA_IDCMP, IDCMP_RAWKEY | IDCMP_NEWSIZE | IDCMP_CLOSEWINDOW | IDCMP_CHANGEWINDOW,
  				 WA_Title, (long unsigned int)"aQView (by Niels Frhling)",
  				 WA_ScreenTitle, (long unsigned int)"QMap v1.0",
  				 TAG_DONE, NULL)) == NULL)	// can't open window, what now??
      Error("Cannot open window\n");

    RastPort = Window->RPort;
    ViewPort = &Window->WScreen->ViewPort;
    GetRGB32(ViewPort->ColorMap, 0, 256, &OldColorMap[1]);
    oldNFree = ViewPort->ColorMap->PalExtra->pe_NFree;
    ViewPort->ColorMap->PalExtra->pe_NFree = 0;
    oldShareable = ViewPort->ColorMap->PalExtra->pe_SharableColors;
    ViewPort->ColorMap->PalExtra->pe_SharableColors = 0;
    OldColorMap[0] = 0x01000000;
  }

  ChangeDisplay(width, height, depth, Palette);

  dim.X = (int)Window->LeftEdge;
  dim.Y = (int)Window->TopEdge;
  dim.changedOffset = TRUE;
  dim.Width = ScreenWidth = (int)Window->GZZWidth;
  dim.Height = ScreenHeight = (int)Window->GZZHeight;
  dim.changedSize = TRUE;
  dim.dtX = (int)Window->WScreen->LeftEdge;
  dim.dtY = (int)Window->WScreen->TopEdge;
  dim.changedDesktopOffset = TRUE;
  dim.dtWidth = (int)Window->WScreen->Width;
  dim.dtHeight = (int)Window->WScreen->Height;
  dim.changedDesktopSize = TRUE;
  dim.frameBuffer = InitDisplay();
  dim.frameSize = ScreenWidth * ScreenHeight;
  dim.changedBuffer = TRUE;

  return &dim;
}

void *SwapDisplay(void *oldBuffer) {
  if(RastPort) {
    if(CyberGfxBase) {
      if(FrameFormat != 0) {
        //WaitBOVP(ViewPort);
        WritePixelArray(oldBuffer, 0, 0, BytesPerRow, RastPort, 0, 0, ScreenWidth, ScreenHeight, FrameFormat);
      }
    }
    else
      WriteChunkyPixels(RastPort, 0, 0, ScreenWidth, ScreenHeight, oldBuffer, BytesPerRow);
  }
  return oldBuffer;
}

void *UpdateDisplay(void *oldBuffer, int x, int y, int width, int height) {
  if(RastPort) {
    if(CyberGfxBase) {
      if(FrameFormat != 0) {
        //WaitBOVP(ViewPort);
        WritePixelArray(oldBuffer, x, y, width, RastPort, 0, 0, width, height, FrameFormat);
      }
    }
    else
      WriteChunkyPixels(RastPort, 0, 0, ScreenWidth, ScreenHeight, oldBuffer, BytesPerRow);
  }
  return oldBuffer;
}

struct DisplayDimension *ChangeDisplay(int width, int height, int depth, struct rgb *Palette) {
  if(ViewPort) {
    long unsigned int *ColorMap = NewColorMap;
    int c = 256-1;

    *ColorMap++ = 0x01000000;
    for(; c >= 0; c--) {
      *ColorMap++ = ((long unsigned int) (*((unsigned char *)Palette)++)) << 24;	/* red   */
      *ColorMap++ = ((long unsigned int) (*((unsigned char *)Palette)++)) << 24;	/* green */
      *ColorMap++ = ((long unsigned int) (*((unsigned char *)Palette)++)) << 24;	/* blue  */
    }
    LoadRGB32(ViewPort, NewColorMap);
  }
  return NULL;	/* not fully functional */
}

void CloseDisplay(void) {
  if (Window) {
    LoadRGB32(ViewPort, &OldColorMap[0]);
    ViewPort->ColorMap->PalExtra->pe_NFree = oldNFree;
    ViewPort->ColorMap->PalExtra->pe_SharableColors = oldShareable;
    CloseWindow(Window);
    Window = NULL;
    UnlockPubScreen(NULL, Screen);
    Screen = NULL;
    tfree(FrameBuffer);
    FrameBuffer = NULL;
  }
  
  if (CyberGfxBase != NULL) {
    CloseLibrary(CyberGfxBase);
    CyberGfxBase = NULL;
  }
  if (IntuitionBase != NULL) {
    CloseLibrary((struct Library *)IntuitionBase);
    IntuitionBase = NULL;
  }
  if (GfxBase != NULL) {
    CloseLibrary((struct Library *)GfxBase);
    GfxBase = NULL;
  }
  if (AslBase != NULL) {
    CloseLibrary(AslBase);
    AslBase = NULL;
  }
}

/*
 * input
 */

static struct MsgPort *inputPort = NULL;

void OpenKeys(void) {
  if(Window)
    inputPort = Window->UserPort;
}

bool GetKeys(struct keyEvent *eventBuffer) {
  struct IntuiMessage *intuiMsg = NULL;

  eventBuffer->pressed = RAWKEY_NOTHING;

  if(inputPort) {
    Forbid();
    while ((intuiMsg = (struct IntuiMessage *)GetMsg(inputPort)) != NULL) {
      long unsigned int Class    = intuiMsg->Class;
      unsigned short int Code    = intuiMsg->Code;
      unsigned short int Qual    = intuiMsg->Qualifier;
      struct Window *IDCMPWindow = intuiMsg->IDCMPWindow;

      ReplyMsg((struct Message *)intuiMsg);

      switch (Class) {
        case IDCMP_RAWKEY:
	  eventBuffer->pressed = (unsigned char)Code;
	  eventBuffer->qualifier = (unsigned char)Qual;
	  break;
        case IDCMP_CLOSEWINDOW:
	  eventBuffer->pressed = RAWKEY_ESCAPE;
	  break;
        case IDCMP_NEWSIZE: {
	    dim.changedSize = TRUE;
	    dim.Width = ScreenWidth = (int)IDCMPWindow->GZZWidth;
	    dim.Height = ScreenHeight = (int)IDCMPWindow->GZZHeight;
	    dim.changedBuffer = TRUE;
	    dim.frameBuffer = InitDisplay();
            dim.frameSize = FrameSize;

            SetDisplay(&dim);
	  } break;
        case IDCMP_CHANGEWINDOW: {
	    dim.changedOffset = TRUE;
            dim.X = IDCMPWindow->LeftEdge;
            dim.Y = IDCMPWindow->TopEdge;

            SetDisplay(&dim);
	  } break;
      }
    }
    Permit();
    intuiMsg = (struct IntuiMessage *)-1;
  }
  
  return (intuiMsg != NULL);
}

void CloseKeys(void) {
  if(inputPort) {
    struct IntuiMessage *intuiMsg;

    Forbid();
    while ((intuiMsg = (struct IntuiMessage *)GetMsg(inputPort)) != NULL)
      ReplyMsg((struct Message *)intuiMsg);
    Permit();
  }
}
