/*
 * Copyright (C) 1996 by Raphael Quinet.  All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby
 * granted, provided that the above copyright notice appear in all
 * copies and that both that copyright notice and this permission
 * notice appear in supporting documentation.  If more than a few
 * lines of this code are used in a program which displays a copyright
 * notice or credit notice, the following acknowledgment must also be
 * displayed on the same screen: "This product includes software
 * developed by Raphael Quinet for use in the Quake Editing Utilities
 * project."  THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR
 * IMPLIED WARRANTY.
 *
 * More information about the QEU project can be found on the WWW:
 * "http://www.montefiore.ulg.ac.be/~quinet/games/editing.html" or by
 * mail: Raphael Quinet, 9 rue des Martyrs, B-4550 Nandrin, Belgium.
 */

#ifndef _F_SPRITE_H_
#define _F_SPRITE_H_
/*
 * F_SPRITE.H - Read and write Quake sprite files.
 */
#include "f_bitmap.h"

/*
 * NOTE: There are still some parts of the sprite format which are a
 *       bit strange for me.  I could be wrong for the multi-image
 *       stuff, even if it works rather well.  The test version of
 *       Quake doesn't contain enough "special" sprites like the
 *       torches (s_torch*.spr) and the shots (shots.spr), so I'm
 *       not sure if my guesses are correct.
 */

/*
 * The SpriteImage structure defines an individual image in a sprite.
 * This is a flat bitmap with an optional offset.
 */
typedef struct
{
  Int16   xoffset;        /* horizontal offset of picture */
  Int16   yoffset;        /* vertical offset of picture */
  BitMap  bitmap;         /* bitmap data (incl. width and height) */
} SpriteImage;

/*
 * (...)
 * If numimages = 0, the frame contains only one image and no data in
 * "unknown".  If numimages > 0, the frame contains several images and
 * each one has a corresponding value in "unknown".
 */
typedef struct
{
  UInt16       numimages; /* number of sub-frames (images) */
  Float32     *unknown;   /*! array of ? (probably scale factors) */
  SpriteImage *images;    /* array of images */
} SpriteFrame;

/*
 * The SpriteInfo structure defines a simple sprite.  It is is stored
 * at the beginning of a Quake sprite file (extension *.spr, magic
 * "IDSP") and defines the size of the sprite and its number of
 * frames.
 *
 * Thanks to Olivier Montanuy for the information about the meaning of
 * some of the fields in this structure.
 */
typedef struct SpriteInfo *SpritePtr;
struct SpriteInfo
{
  UInt32       unknown1;   /*! version number?  always 1 */
  UInt32       unknown2;   /*! ? 1 or 2 */
  Float32      radius;     /* maximum radius of all frames */
  UInt16       maxwidth;   /* maximum width of all frames. */
  UInt16       maxheight;  /* maximum height of all frames. */
  UInt16       numframes;  /* number of frames */
  SpriteFrame *frames;     /* array of frames */
  UInt32       unknown4;   /*! ? always 0*/
  UInt32       unknown5;   /*! ? 1 or 0 */
};

/*
 * Prototypes.
 */
SpritePtr NewSprite();
void      FreeSprite(SpritePtr sprite);
UInt16    AddSpriteImage(SpritePtr sprite, Int16 framenum, Bool multi,
			 Float32 unknown, Int16 xoffset, Int16 yoffset,
			 BitMap *bmptr);
SpritePtr ReadSprite(FILE *file, UInt32 offset);
void      DumpSprite(FILE *outf, SpritePtr sprite);

UInt32    SaveSprite(FILE *file, SpritePtr sprite);

#endif /* _F_SPRITE_H_ */
/* end of file */
