/*
==============================================================================
Title: "Quake BSP Texture Renamer"
(c) 1998 by Matthias "Maddes" Buecher

Authors:
Matthias "Maddes" Buecher <maddes@bigfoot.com>

Additional credits:
Stefan Schwoon <ssch0098@rz.uni-hildesheim.de>
Stefan Schwoon <ssch0098@rz.uni-hildesheim.de> (initial idea)
Martin "Lunatic" Willers <M.Willers@tu-bs.de> (unix help)
Paul "PaK" Koehler <koehler@globalserve.net> (unix help)

Notes:
This program was created to provide a
"GL_LoadTexture: cache mismatch" workaround for OpenGL Quake/QuakeWorld

Most of this code is easy to comprehend and there should not be too many
questions about how it works.  The heart of the program has been slighty
commented out to help you understand the technical operations being done.

Important:
It should compile under Win32 and Unix with no errors, some warnings may still occur.
If you use a processor which has another byte order than Intel (lowest to highest byte),
you have to reprogram all offset functions, etc.

DISCLAIMER WARNING:
WE DO NOT ACCEPT RESPONSIBILITY FOR ANY EFFECTS, POSITIVE OR NEGATIVE,
THAT THIS CODE MAY CAUSE TOWARDS YOUR COMPUTER.
WE DO NOT SUPPORT MODIFICATIONS OF THIS CODE, USE ONLY AT YOUR OWN RISK.
THIS PATCH HAS NOT BEEN ENDORSED BY THE ID SOFTWARE CORPORATION.
THIS PROGRAM IS FREE OF CHARGE. IF YOU HAVE RECEIVED THIS PROGRAM THROUGH
PAYMENT, PLEASE CONTACT US.
THANK YOU.
==============================================================================
*/

/* Compatible headers */
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <errno.h>

/* needed to compile under Linux */
#ifndef __WIN32__
#define stricmp strcasecmp
#endif

/* constants */
#define VERSION "1.00 (1998-11-01)"

#define BUFLEN 256
#define STRLEN 128
#define TEXLEN 16

#define PARM_CHAR	'-'

#define COMMENT_CHAR	';'

#define PATH_BEGIN_CHAR	'['
#define PATH_END_CHAR	']'

/*
structure definitions taken from Quake tool source codes
*/

/* BSP file */
#define BSP_VERSION	0x1d	/* Classic Quake */

#define LUMP_ENTITIES		0
#define LUMP_PLANES		1
#define LUMP_TEXTURES		2
#define LUMP_VERTEXES		3
#define LUMP_VISIBILITY		4
#define LUMP_NODES		5
#define LUMP_TEXINFO		6
#define LUMP_FACES		7
#define LUMP_LIGHTING		8
#define LUMP_CLIPNODES		9
#define LUMP_LEAFS		10
#define LUMP_MARKSURFACES 	11
#define LUMP_EDGES		12
#define LUMP_SURFEDGES		13
#define LUMP_MODELS		14

#define	HEADER_LUMPS	15

typedef struct {
	long fileofs, filelen;		/* info about lump */
} lump_t;

typedef struct {			/* BSP file header */
	long version;			/* BSP file version */
	lump_t lumps[HEADER_LUMPS];	/* info table for lumps */
} dheader_t;

typedef struct {			/* texture lump header */
	long nummiptex;			/* number of textures in lump */
	long dataofs[1];			/* [nummiptex], info table for textures */
} dmiptexlump_t;

#define	MIPLEVELS	4

typedef struct miptex_s {		/* texture header */
	char name[16];			/* texture name */
	unsigned width, height;
	unsigned offsets[MIPLEVELS];	/* four mip maps stored */
} miptex_t;

/* PAK file */
#define PAK_HEADER 0x4b434150		/* 'PACK' */

typedef struct {			/* PAK file header */
	char id[4];			/* 'PACK' */
	long dirofs, dirlen;		/* info about directory table */
} packheader_t;

typedef struct {			/* directory entry */
	char name[56];			/* filename with path */
	long filepos, filelen;		/* info about file */
} packfile_t;

/*
structure definitions of lists for pathes -> maps -> textures
*/
typedef struct s_texentry {
	char oldname[TEXLEN], newname[TEXLEN];
	int orgnamelength;
	int found;
	int changed;
	struct s_texentry *next;
} texentry;

typedef struct s_mapentry {
	char name[STRLEN];
	int found;
	struct s_texentry *textures;
	struct s_mapentry *next;
} mapentry;

typedef struct s_pathentry {
	char full[STRLEN];
	char *file;
	int type;
	int found;
	struct s_mapentry *maps;
	struct s_pathentry *next;
} pathentry;

/* global data definition */
char *default_config_file = "glfix.ini";	/* pointer on default config file name */
char *config_file;				/* pointer on used config file name */
char *start_directory;				/* pointer on start directory */

FILE *cfg;		/* config file handle */

int flag_error;					/* error occured */
int flag_warning;				/* warning occured */

int flag_undo;					/* undo flag */
int flag_yes;					/* assume yes flag */
int flag_test;					/* test flag */

int outredirect;				/* stdout redirected to file flag */

/*
******************************************************************************
*
*   G E N E R A L   F U N C T I O N S
*
******************************************************************************
*/

/*
==============================================================================
displays the given text and asks for confirmation
==============================================================================
*/
int confirm(char *question)
{
	char answer;

	printf("%s [Y/N] ", question);
	if (outredirect)
		fprintf(stderr, "%s [Y/N] ", question);

	do
	{
		if (flag_yes)
		{
			answer = 'Y';
			printf("Y\n");
			if (outredirect)
				fprintf(stderr, "Y\n");
		}
		else
			answer=toupper(getchar());
	}
	while ((answer != 'Y') && (answer != 'N'));	/* loop until valid answer */

	if (!flag_yes && outredirect)
		printf("%c\n", answer);

	if (answer == 'Y')
		return 1;
	else
		return 0;
}

/*
==============================================================================
trims the string from all leading spaces/tabs
and all trailing spaces/tabs/carriage return
returns the new length
==============================================================================
*/
int strtrim(char *text)
{
	int i;
	int length;

	/* leading spaces */
	length = strlen(text);
	for (i=0; (i<length) && ((text[i]==' ') || (text[i]=='\t') || (text[i]==(char)0x0A) || (text[i]==(char)0x0D)); i++);
	if (i)
		strcpy(text, text+i);

	/* trailing spaces */
	length = strlen(text);
	for (i=length-1; (i>=0) && ((text[i]=='\n') || (text[i]==' ') || (text[i]=='\t') || (text[i]==(char)0x0A) || (text[i]==(char)0x0D)); i--);
	i++;
	text[i] = 0;

	return i;
}

/*
==============================================================================
transforms a string to lowercase
==============================================================================
*/
void strlower(char *text)
{
	int i;
	int length;

	length = strlen(text);
	for (i=0; i<length; i++)
		text[i] = tolower(text[i]);
}

/*
==============================================================================
frees all the memory used
==============================================================================
*/
void freemen(pathentry *path)
{
	pathentry *delpath;
	mapentry *map;
	mapentry *delmap;
	texentry *tex;
	texentry *deltex;

	while (path)
	{
		map = path->maps;
		while (map)
		{
			tex = map->textures;
			while (tex)
			{
				deltex = tex;
				tex = tex->next;
				free(deltex);
			}
			delmap = map;
			map = map->next;
			free(delmap);
		}
		delpath = path;
		path = path->next;
		free(delpath);
	}
}

/*
******************************************************************************
*
*   U S E R   I N T E R F A C E
*
******************************************************************************
*/

/*
==============================================================================
displays the help text
==============================================================================
*/
void display_help(char *call)
{
	/* syntax */
	printf("Usage: BSPTxRen [-?] [-t] [-u] [-y] [-d <directory>] [-c <config file>]\n");
	printf("  -?              : displays this help text\n");
	printf("  -t              : testmode\n");
	printf("  -u              : undo previous changes done with the config file\n");
	printf("  -y              : assumes yes on all queries\n");
	printf("  -d <directory>  : defines the start directory\n");
	printf("  -c <config file>: defines the config file to be used\n");
	printf("                    default is \"%s\"\n", default_config_file);
	if (outredirect)
	{
		fprintf(stderr, "Usage: BSPTxRen [-?] [-u] [-y] [-c <config file>]\n\n");
		fprintf(stderr, "  -?              : displays this help text\n");
		fprintf(stderr, "  -t              : testmode\n");
		fprintf(stderr, "  -u              : undo previous changes done with the config file\n");
		fprintf(stderr, "  -y              : assumes yes on all queries\n");
		fprintf(stderr, "  -d <directory>  : defines the start directory\n");
		fprintf(stderr, "  -c <config file>: defines the config file to be used\n");
		fprintf(stderr, "                    default is \"%s\"\n", default_config_file);
	}

	/* example */
	printf("\nExample:\n");
	printf("\"%s -u\"\n", call);
	printf("This would undo all previous changes done with the default config file\n");
	if (outredirect)
	{
		fprintf(stderr, "\nExample:\n");
		fprintf(stderr, "\"%s -u\"\n", call);
		fprintf(stderr, "This would undo all previous changes done with the default config file\n");
	}

	/* contact */
	printf("\nHow to contact the author:\n");
	printf("Matthias \"Maddes\" Buecher <maddes@bigfoot.com>\n");
	if (outredirect)
	{
		fprintf(stderr, "\nHow to contact the author:\n");
		fprintf(stderr, "Matthias \"Maddes\" Buecher <maddes@bigfoot.com>\n");
	}

	printf("\nThis program was created to provide a\n\"GL_LoadTexture: cache mismatch\" workaround for OpenGL Quake/QuakeWorld\n");
	if (outredirect)
		fprintf(stderr, "\nThis program was created to provide a\n\"GL_LoadTexture: cache mismatch\" workaround for OpenGL Quake/QuakeWorld\n");

}

/*
==============================================================================
checks if string begins with a parameter character
==============================================================================
*/
int chk_paramchar(char *text)
{
	/* check for parameter character at beginning */
	if (text[0] == PARM_CHAR)
	{
		strlower(text);
		return 1;
	}
	else
		return 0;
}

/*
==============================================================================
checks all parameters
when help parameter is found it displays the help text and leaves the function
checks if the config file is present
displays all used parameters and asks for confirmation
==============================================================================
*/
int chk_param(int argc, char *argv[])
{
	/* return values: 0 = confirmed, 1 = help called, 2 = error in parameters, 3 = declined confirmation */

	/* define local variables */
	int i;			/* iteration variable */

	int dif;		/* difference between statement and parameter number, used for correct output */

	int flag_help;		/* display help */

	/* initialize local varaibles */
	dif = 0;
	flag_error = 0;
	flag_warning = 0;
	flag_help = 0;

	/* parse parameters */
	for (i=1; i<argc; i++)
	{
		if (chk_paramchar(argv[i]))
		{
			if (!strcmp(argv[i],"-y"))	/* assume yes on all queries parameter */
			{
				/* check already set */
				if (flag_yes)
				{
					printf("Warning on parameter %i: assume yes already defined\n", i-dif);
					if (outredirect)
						fprintf(stderr, "Warning on parameter %i: assume yes already defined\n", i-dif);
					flag_warning = 1;
				}

				flag_yes=1;
			}
			else if (!strcmp(argv[i],"-u"))	/* undo parameter */
			{
				/* check already set */
				if (flag_undo)
				{
					printf("Warning on parameter %i: undo already defined\n", i-dif);
					if (outredirect)
						fprintf(stderr, "Warning on parameter %i: undo already defined\n", i-dif);
					flag_warning = 1;
				}

				flag_undo=1;
			}
			else if (!strcmp(argv[i],"-t") || !strcmp(argv[i],"-test"))	/* test parameter */
			{
				/* check already set */
				if (flag_test)
				{
					printf("Warning on parameter %i: test already defined\n", i-dif);
					if (outredirect)
						fprintf(stderr, "Warning on parameter %i: test already defined\n", i-dif);
					flag_warning = 1;
				}

				flag_test=1;
			}
			else if (!strcmp(argv[i],"-c"))	/* config file parameter */
			{
				i++;	/* next statement (normally filename) */

				if ( (i<argc) && (!chk_paramchar(argv[i])) )	/* statement available and not a *new parameter* */
				{
					/* check already set */
					if (config_file != default_config_file)
					{
						printf("Warning on parameter %i: a custom config file was already named\n", i-1-dif);
						if (outredirect)
							fprintf(stderr, "Warning on parameter %i: a custom config file was already named\n", i-1-dif);
						flag_warning = 1;
					}

					/* set pointer of config file name to statement string */
					config_file = argv[i];

					/* as this parameter has two statements, decrease the parameter number by one */
					dif++;
				}
				else
				{
					i--;

					printf("Error on parameter %i: custom config file not named\n", i-dif);
					if (outredirect)
						fprintf(stderr, "Error on parameter %i: custom config file not named\n", i-dif);
					flag_help = 1;
					flag_error = 2;
				}
			}
			else if (!strcmp(argv[i],"-d"))	/* directory parameter */
			{
				i++;	/* next statement (normally directory) */

				if ( (i<argc) && (!chk_paramchar(argv[i])) )	/* statement available and not a *new parameter* */
				{
					/* check already set */
					if (start_directory)
					{
						printf("Warning on parameter %i: a start directory was already named\n", i-1-dif);
						if (outredirect)
							fprintf(stderr, "Warning on parameter %i: a start directory was already named\n", i-1-dif);
						flag_warning = 1;
					}

					/* set pointer of directory to statement string */
					start_directory = argv[i];

					/* as this parameter has two statements, decrease the parameter number by one */
					dif++;
				}
				else
				{
					i--;

					printf("Error on parameter %i: start directory not named\n", i-dif);
					if (outredirect)
						fprintf(stderr, "Error on parameter %i: start directory not named\n", i-dif);
					flag_help = 1;
					flag_error = 2;
				}
			}
			else if (!strcmp(argv[i],"-?") || !strcmp(argv[i],"-h") || !strcmp(argv[i],"-help"))	/* help parameter */
			{
				flag_help = 1;
				flag_error = 1;
			}
			else				/* unknown parameter */
			{
				printf("Error on parameter %i: \"%s\" is an unknown parameter\n", i-dif, argv[i]);
				if (outredirect)
					fprintf(stderr, "Error on parameter %i: \"%s\" is an unknown parameter\n", i-dif, argv[i]);
				flag_help = 1;
				flag_error = 2;
			}
		}
		else	/* not a parameter */
		{
			printf("Error on position %i: \"%s\" is not a parameter\n", i-dif, argv[i]);
			if (outredirect)
				fprintf(stderr, "Error on position %i: \"%s\" is not a parameter\n", i-dif, argv[i]);
			flag_help = 1;
			flag_error = 2;
		}
	}

	/* check if config file exists */
	cfg = fopen(config_file,"r");
	if (!cfg)
	{
		printf("Error: could not open configuration file \"%s\"\n", config_file);
		if (outredirect)
			fprintf(stderr, "Error: could not open configuration file \"%s\"\n", config_file);
		flag_error = 2;
	}

	/* any errors occured? */
	if (flag_error > 1)
	{
		printf("\n");
		if (outredirect)
			fprintf(stderr, "\n");
	}
	if (flag_help) display_help(argv[0]);	/* asked for help */
	if (flag_error) return 1;

	/* display all data */
	if (flag_warning) printf("\n");
	printf("The following settings will be used:\n------------------------------------\n");
	if (flag_undo) printf("UNDO enabled, all files will be reverted to original\n");
	if (flag_test) printf("TESTMODE enabled, no changes will be done to any file\n");
	if (start_directory) printf("Start directory: \"%s\"\n", start_directory);
	printf("Configuration file: \"%s\"\n", config_file);
	if (outredirect)
	{
		if (flag_warning) fprintf(stderr, "\n");
		fprintf(stderr, "The following settings will be used:\n------------------------------------\n");
		if (flag_undo) fprintf(stderr, "UNDO enabled, all files will be reverted to original\n");
		if (flag_test) fprintf(stderr, "TESTMODE enabled, no changes will be done to any file\n");
		if (start_directory) fprintf(stderr, "Start directory: \"%s\"\n", start_directory);
		fprintf(stderr, "Configuration file: \"%s\"\n", config_file);
	}

	/* ask for confirmation and return */
	if (!confirm("\nDo you wish to proceed?"))
		return 1;

	printf("\n");
	if (outredirect)
		fprintf(stderr, "\n");

	return 0;
}

/*
******************************************************************************
*
*   C O N F I G   F I L E   R E A D I N G   &   P A R S I N G
*
******************************************************************************
*/

/*
==============================================================================
gets the filename out of the path
==============================================================================
*/
char *get_filename(pathentry *path)
{
	char *slash;
	char *backslash;

	slash = strrchr(path->full, '/');
	backslash = strrchr(path->full, '\\');

	if ((slash) && (slash > backslash))
	{
		slash++;
		return slash;
	}
	else if ((backslash) && (backslash > slash))
	{
		backslash++;
		return backslash;
	}
	return path->full;
}

/*
==============================================================================
checks for .BSP or .PAK fileformat
==============================================================================
*/
int chk_fileformat(FILE *data, int type)
{
	long id;

	id = 0;
	if ( fread(&id, 4, 1, data) != 1 )
	{
		printf("Error: couldn't read fileformat header\n");
		if (outredirect)
			fprintf(stderr, "Error: couldn't read fileformat header\n");
		type = 0;
	}
	else
	{
		switch ( type )
		{
			case 1:		/* BSP fileformat */
				if (!(id == (long)BSP_VERSION)) type = 0;
				break;
			case 2:		/* PAK fileformat */
				if (!(id == (long)PAK_HEADER)) type = 0;
				break;
			default:	/* unknown	*/
				type = 0;
				break;
		}
	}

	return type;
}

/*
==============================================================================
checks for .BSP or .PAK extension
==============================================================================
*/
int chk_extension(char *filename)
{
	char *extension;
	char *slash;
	char *backslash;

	/* get position of last point */
	extension = strrchr(filename, '.');

	/* check if no other dir seperator follows */
	if (extension)
	{
		slash = strchr(extension, '/');
		backslash = strchr(extension, '\\');
		if ((slash) || (backslash))
			extension = NULL;
	}

	if (extension)
	{
		if (!stricmp(extension, ".pak"))
			return 2;		/* PAK extension */
		if (!stricmp(extension, ".bsp"))
			return 1;		/* BSP extension */
	}

	return 0;			/* unknown */
}

/*
==============================================================================
reads the next line of the file
checks if line did not exceed buffer
converts the line for Unix
removes any comment
trims the line
checks if not an empty line, otherwise reads next line
==============================================================================
*/
char *read_cfgline(char *buffer, int buflen, FILE *cfg, int *line_count)
{
	int length;
	char *comment;

	while (fgets(buffer, buflen, cfg))
	{
		(*line_count)++;
		length = strlen(buffer);
		if ((length == buflen-1) && (buffer[length-1]!='\n'))
		{
			printf("Error in line %i: line too long, only %i characters allowed", *line_count, buflen-2);
			if (outredirect)
				fprintf(stderr, "Error in line %i: line too long, only %i characters allowed", *line_count, buflen-2);
			flag_error = 2;
			break;
		}

		if (buffer[0] == (char)0x0A)			/* make Unix read DOS/Windows files */
			buffer[0] = ' ';
		if (buffer[0] == (char)0x0D)			/* make Unix read DOS/Windows files */
			buffer[0] = ' ';

		comment = strchr(buffer, COMMENT_CHAR);	/* remove any comment */
		if (comment)
			comment[0] = 0;

		length = strtrim(buffer);		/* remove leading/trailing whitespaces */

		if (!length)
			continue;

		return buffer;
	}

	return NULL;
}

/*
==============================================================================
checks if line is a new path setting
removes the beginning and ending char
trims the line
==============================================================================
*/
int chk_pathline(char *line, int line_count)
{
	int i;

	if (line[0] == PATH_BEGIN_CHAR)
	{
		i = strlen(line) - 1;

		if (line[i] != PATH_END_CHAR)
		{
			printf("Error in line %i: path string not defined correctly", line_count);
			if (outredirect)
				fprintf(stderr, "Error in line %i: path string not defined correctly", line_count);
			flag_error = 2;
		}
		else
		{
			line[0] = ' ';
			line[i] = 0;
			strtrim(line);
			return 1;
		}
	}

	return 0;
}

/*
==============================================================================
parses all data lines of the config file
and creates lists out of the read data
==============================================================================
*/
pathentry *read_cfgfile(void)
{
	FILE *exist;		/* test handle */

	int line_count;		/* line number */
	char buffer[BUFLEN];	/* read buffer */

	int type;		/* type of pathline */
	int temp_type;		/* type of pathline */

	char mapname[BUFLEN];
	char oldname[BUFLEN], newname[BUFLEN];
	char unknown[BUFLEN];
	int oldlength, newlength;

	pathentry *start;
	pathentry *path;
	pathentry *lastpath;
	mapentry *map;
	mapentry *lastmap;
	texentry *tex;
	texentry *lasttex;

	/* initialize local varaibles */
	start = NULL;
	path = NULL;
	lastpath = NULL;
	line_count = 0;
	flag_warning = 0;

	/* reading config file */
	while (!feof(cfg) && read_cfgline(buffer, BUFLEN, cfg, &line_count))
	{
		if (chk_pathline(buffer, line_count))	/* pathline */
		{
			if (flag_error) break;

			if (strlen(buffer) > (STRLEN-1))
			{
				printf("Error on line %i: path too long, only %i characters allowed\n", line_count, STRLEN-1);
				if (outredirect)
					fprintf(stderr, "Error on line %i: path too long, only %i characters allowed\n", line_count, STRLEN-1);
				flag_error = 2;
				break;
			}

			/* check if file exists */
			exist = fopen(buffer, "r+b");
			if (!exist)
			{
				if (errno == EACCES)
				{
					printf("Warning on line %i: can not access file \"%s\"- write protected or a directory?\n", line_count, buffer);
					if (outredirect)
						fprintf(stderr, "Warning on line %i: can not access file \"%s\"- write protected or a directory?\n", line_count, buffer);
					flag_warning = 1;
				}
				else
				{
					printf("Warning on line %i: file \"%s\" not found\n", line_count, buffer);
					if (outredirect)
						fprintf(stderr, "Warning on line %i: file \"%s\" not found\n", line_count, buffer);
					flag_warning = 1;
				}
				type = 0;
			}
			else
			{
				/* check file extension */
				type = chk_extension(buffer);
				if (type)
				{
					/* check fileformat */
					temp_type = chk_fileformat(exist, type);
					if (!temp_type)
					{
						switch ( type )
						{
							case 1:		/* BSP fileformat */
								printf("Warning on line %i: \"%s\" is not a BSP file of Classic Quake\n", line_count, buffer);
								if (outredirect)
									fprintf(stderr, "Warning on line %i: \"%s\" is not a BSP file of Classic Quake\n", line_count, buffer);
								break;
							case 2:		/* PAK fileformat */
								printf("Warning on line %i: \"%s\" is not a pak file\n", line_count, buffer);
								if (outredirect)
									fprintf(stderr, "Warning on line %i: \"%s\" is not a pak file\n", line_count, buffer);
								break;
							default:	/* unknown	*/
								printf("Warning on line %i: fileformat of \"%s\" doesn't match extension\n", line_count, buffer);
								if (outredirect)
									fprintf(stderr, "Warning on line %i: fileformat of \"%s\" doesn't match extension\n", line_count, buffer);
								break;
						}
						flag_warning = 1;
						type = 0;
					}
					else
						type = temp_type;
				}
				else
				{
					printf("Warning on line %i: unknown file type (only .BSP and .PAK)\n", line_count);
					if (outredirect)
						fprintf(stderr, "Warning on line %i: unknown file type (only .BSP and .PAK)\n", line_count);
					flag_warning = 1;
				}
				fclose(exist);
			}

			/* create path list entry */
			path = malloc(sizeof(pathentry));
			if (!path)
			{
				printf("Error on line %i: couldn't allocate memory for new path\n", line_count);
				if (outredirect)
					fprintf(stderr, "Error on line %i: couldn't allocate memory for new path\n", line_count);
				flag_error = 2;
				break;
			}
			strcpy(path->full, buffer);
			path->type = type;
			path->found = 0;
			path->file = NULL;
			path->maps = NULL;
			path->next = NULL;

			if (type == 1)		/* BSP file */
				path->file = get_filename(path);

			if (!start) start = path;
			if (lastpath) lastpath->next = path;

			lastpath = path;
		}
		else if (path)		/* texture line */
		{
			if (!flag_undo)
			{
				sscanf(buffer, "%s %s %s %s", mapname, oldname, newname, unknown);
				oldlength = strlen(oldname);
				newlength = strlen(newname);
			}
			else
			{
				sscanf(buffer, "%s %s %s %s", mapname, newname, oldname, unknown);
				oldlength = strlen(newname);
				newlength = strlen(oldname);
			}

			if (strlen(mapname) > (STRLEN-1))
			{
				printf("Error on line %i: mapname too long, only %i characters allowed\n", line_count, STRLEN-1);
				if (outredirect)
					fprintf(stderr, "Error on line %i: mapname too long, only %i characters allowed\n", line_count, STRLEN-1);
				flag_error = 2;
				break;
			}
			if (oldlength > (TEXLEN-1))
			{
				printf("Error on line %i: old texture name too long, only %i characters allowed\n", line_count, TEXLEN-1);
				if (outredirect)
					fprintf(stderr, "Error on line %i: texture name too long, only %i characters allowed\n", line_count, TEXLEN-1);
				flag_error = 2;
				break;
			}
			if (newlength > (TEXLEN-1))
			{
				printf("Error on line %i: new texture name too long, only %i characters allowed\n", line_count, TEXLEN-1);
				if (outredirect)
					fprintf(stderr, "Error on line %i: texture name too long, only %i characters allowed\n", line_count, TEXLEN-1);
				flag_error = 2;
				break;
			}

			if (newlength > oldlength)
			{
				printf("Error on line %i: new texture name can not be longer than the original one\n", line_count);
				if (outredirect)
					fprintf(stderr, "Error on line %i: new texture name can not be longer than the old one\n", line_count);
				flag_error = 2;
				break;
			}

			if (strlen(unknown))
			{
				printf("Error on line %i: unexpected data following texture names\n", line_count);
				if (outredirect)
					fprintf(stderr, "Error on line %i: unexpected data following texture names\n", line_count);
				flag_error = 2;
				break;
			}

			lastmap = NULL;
			lasttex = NULL;

			if ((path->type == 1) && (stricmp(path->file, mapname)))
			{
				printf("Error on line %i: only textures of the same map allowed for single BSP files\n", line_count);
				if (outredirect)
					fprintf(stderr, "Error on line %i: only textures of the same map allowed for single BSP files\n", line_count);
				flag_error = 2;
				break;
			}

			/* search map list entry */
			map = path->maps;
			while ((map) && (stricmp(map->name, mapname)))
			{
				lastmap = map;
				map = map->next;
			}

			if (map)
			{
				/* search texture list entry */
				tex = map->textures;
				while ((tex) && (stricmp(tex->oldname, oldname)))
				{
					lasttex = tex;
					tex = tex->next;
				}
			}
			else
			{
				/* create map list entry */
				map = malloc(sizeof(mapentry));
				if (!map)
				{
					printf("Error on line %i: couldn't allocate memory for new map\n", line_count);
					if (outredirect)
						fprintf(stderr, "Error on line %i: couldn't allocate memory for new map\n", line_count);
					flag_error = 2;
					break;
				}
				strcpy(map->name, mapname);
				map->found = 0;
				map->textures = NULL;
				map->next = NULL;

				if (lastmap) lastmap->next = map;
				else if (!path->maps) path->maps = map;
				else
				{
					printf("Error on line %i: map entry impossible, please contact the authors\n", line_count);
					if (outredirect)
						fprintf(stderr, "Error on line %i: map entry impossible, please contact the authors\n", line_count);
					flag_error = 2;
					break;
				}

				tex = NULL;
			}

			/* create texture list entry */
			if (!tex)
			{
				tex = malloc(sizeof(texentry));
				if (!tex)
				{
					printf("Error on line %i: couldn't allocate memory for new textures\n", line_count);
					if (outredirect)
						fprintf(stderr, "Error on line %i: couldn't allocate memory for new textures\n", line_count);
					flag_error = 2;
					break;
				}
				strcpy(tex->oldname, oldname);
				strcpy(tex->newname, newname);
				tex->orgnamelength = oldlength;
				tex->found = 0;
				tex->changed = 0;
				tex->next = NULL;

				if (lasttex) lasttex->next = tex;
				else if (!map->textures) map->textures = tex;
				else
				{
					printf("Error on line %i: texture entry impossible, please contact the authors\n", line_count);
					if (outredirect)
						fprintf(stderr, "Error on line %i: texture entry impossible, please contact the authors\n", line_count);
					flag_error = 2;
					break;
				}
			}
		}
		else
		{
			printf("Error on line %i: map and textures defined before any path\n", line_count);
			if (outredirect)
				fprintf(stderr, "Error on line %i: map and textures defined before any path\n", line_count);
			flag_error = 2;
			break;
		}
	}
	if (flag_warning)
	{
		printf("\n");
		if (outredirect)
			fprintf(stderr, "\n");
	}

	return start;
}

/*
******************************************************************************
*
*   O T H E R   F I L E S   R E A D I N G   &   W R I T I N G
*
******************************************************************************
*/

void process_bspfile(FILE *data, long bspofs, long bsplen, mapentry *map)
{
	dheader_t bspheader;
	long nummiptex;			/* number of textures in lump */
	dmiptexlump_t *texofs;
	miptex_t *texture;

	long i, j, texdif;

	texentry *tex;

	texture = NULL;
	texofs = NULL;

	do			/* just to use break for errors */
	{
		if ( fseek(data, bspofs+0, SEEK_SET) != 0 || ftell(data) != bspofs+0 )
		{
			printf("\n- Error: couldn't read bsp header\n");
			if (outredirect)
				fprintf(stderr, "Error: couldn't read bsp header\n");
			flag_error = 2;
			break;
		}
		if ( fread(&bspheader, sizeof(dheader_t), 1, data) != 1 )
		{
			printf("\n- Error: couldn't read bsp header\n");
			if (outredirect)
				fprintf(stderr, "Error: couldn't read bsp header\n");
			flag_error = 2;
			break;
		}
		/* checking bsp version / fileformat */
		if (bspheader.version != (long)BSP_VERSION)
		{
			printf("\n- Error: unknown BSP version\n");
			if (outredirect)
				fprintf(stderr, "Error: unknown BSP version\n");
			flag_error = 2;
			break;
		}
		map->found = 1;
		/* read number of textures */
		if ( fseek(data, bspofs+bspheader.lumps[LUMP_TEXTURES].fileofs, SEEK_SET) || ftell(data) != bspofs+bspheader.lumps[LUMP_TEXTURES].fileofs )
		{
			printf("\n- Error: couldn't read number of textures\n");
			if (outredirect)
				fprintf(stderr, "Error: couldn't read number of textures\n");
			flag_error = 2;
			break;
		}
		if ( fread(&nummiptex, 4, 1, data) != 1 )
		{
			printf("\n- Error: couldn't read number of textures\n");
			if (outredirect)
				fprintf(stderr, "Error: couldn't read number of textures\n");
			flag_error = 2;
			break;
		}

		/* read texture offsets */
		texofs = malloc(4+nummiptex*4);
		if (!texofs)
		{
			printf("\n- Error: couldn't allocate memory for texture offsets\n");
			if (outredirect)
				fprintf(stderr, "Error: couldn't allocate memory for texture offsets\n");
			flag_error = 2;
			break;
		}
		if ( fseek(data, bspofs+bspheader.lumps[LUMP_TEXTURES].fileofs, SEEK_SET) != 0 || ftell(data) != bspofs+bspheader.lumps[LUMP_TEXTURES].fileofs )
		{
			printf("\n- Error: couldn't read texture offsets\n");
			if (outredirect)
				fprintf(stderr, "Error: couldn't read texture offsets\n");
			flag_error = 2;
			break;
		}
		if ( fread(texofs, 4+nummiptex*4, 1, data) != 1 )
		{
			printf("\n- Error: couldn't read texture offsets\n");
			if (outredirect)
				fprintf(stderr, "Error: couldn't read texture offsets\n");
			flag_error = 2;
			break;
		}

		/* read texture headers */
		texture = malloc(nummiptex*sizeof(miptex_t));
		if (!texture)
		{
			printf("\n- Error: couldn't allocate memory for texture headers\n");
			if (outredirect)
				fprintf(stderr, "Error: couldn't allocate memory for texture headers\n");
			flag_error = 2;
			break;
		}

		texdif = 0;
		for (i=0; i<nummiptex; i++)
		{
			if (texofs->dataofs[i] == -1)
			{
				texdif++;
				continue;
			}

			if ( fseek(data, bspofs+bspheader.lumps[LUMP_TEXTURES].fileofs+texofs->dataofs[i], SEEK_SET) != 0 || ftell(data) != bspofs+bspheader.lumps[LUMP_TEXTURES].fileofs+texofs->dataofs[i] )
			{
				printf("\n- Error: couldn't read texture headers\n");
				if (outredirect)
					fprintf(stderr, "Error: couldn't read texture headers\n");
				flag_error = 2;
				break;
			}
			if ( fread(&texture[i-texdif], sizeof(miptex_t), 1, data) != 1 )
			{
				printf("\n- Error: couldn't read texture headers\n");
				if (outredirect)
					fprintf(stderr, "Error: couldn't read texture headers\n");
				flag_error = 2;
				break;
			}
		}
		if (flag_error) break;

		nummiptex -= texdif;
		printf("(%li textures)\n", nummiptex);

		/* search textures */
		tex = map->textures;
		while (tex)
		{
			/* search texture with oldname */
			i=0;
			while (i<nummiptex && stricmp(texture[i].name, tex->oldname)) i++;
			if (i<nummiptex)
			{
				tex->found = 1;
				/* check if newname not already exists*/
				j=0;
				while (j<nummiptex && stricmp(texture[j].name, tex->newname)) j++;

				if (j<nummiptex)
				{
					printf("- Warning: Can *not* rename \"%s\" as new texture name \"%s\" already exists\n", texture[i].name, texture[j].name);
					if (outredirect)
						fprintf(stderr, "- Warning: Can *not* rename \"%s\" as new texture name \"%s\" already exists\n", texture[i].name, texture[j].name);
				}
				else
				{
					printf("- Renaming \"%s\" to \"%s\"\n", texture[i].name, tex->newname);
					if (outredirect)
						fprintf(stderr, "- Renaming \"%s\" to \"%s\"\n", texture[i].name, tex->newname);
					/* copying new name into texture header */
					strncpy(texture[i].name, tex->newname, tex->orgnamelength);
					texture[i].name[tex->orgnamelength] = 0;
					if (!flag_test)
					{
						tex->changed = 1;
						/* writing texture header back into file*/
						if ( fseek(data, bspofs+bspheader.lumps[LUMP_TEXTURES].fileofs+texofs->dataofs[i], SEEK_SET) != 0 || ftell(data) != bspofs+bspheader.lumps[LUMP_TEXTURES].fileofs+texofs->dataofs[i] )
						{
							printf("- Error: couldn't write texture header\n");
							if (outredirect)
								fprintf(stderr, "Error: couldn't write texture header\n");
							flag_error = 2;
							break;
						}
						if ( fwrite(&texture[i], sizeof(miptex_t), 1, data) != 1)
						{
							printf("- Error: couldn't write texture header\n");
							if (outredirect)
								fprintf(stderr, "Error: couldn't write texture header\n");
							flag_error = 2;
							break;
						}
					}
				}
			}

			tex = tex->next;
		}
	} while (0);

	if (texture) free(texture);
	if (texofs) free(texofs);
}

void process_list(pathentry *path)
{
	FILE *data;

	packheader_t pakheader;
	packfile_t *directory;
	long direntries;
	long i;

	mapentry *map;

	while (path)
	{
		/* read files and process */
		if (!path->type)
		{
			printf("*** Skipping \"%s\": file not found or unsupported file format\n\n", path->full);
			path = path->next;
			continue;
		}

		printf("*** Processing \"%s\"\n", path->full);

		/* open data file */
		data = fopen(path->full, "r+b");
		if (!data)
		{
			if (errno == EACCES)
				printf("Warning: can not access file - write protected or a directory?\n");
			else
				printf("Warning: file not found\n");
			path = path->next;
			continue;
		}

		path->found = 1;
		switch ( path->type )
		{
			case 1:		/* BSP file */
				process_bspfile(data, 0, 0, path->maps);
				break;
			case 2:		/* PAK file */
				if (chk_fileformat(data, path->type) != path->type)
				{
					printf("Error: mismatch of extension and fileformat (something has changed during run)\n");
					if (outredirect)
						fprintf(stderr, "Error: mismatch of extension and fileformat (something has changed during run)\n");
					flag_error = 2;
					break;
				}
				/* read pack header */
				if ( fseek(data, 0, SEEK_SET) != 0 || ftell(data) != 0 )
				{
					printf("Error: couldn't read pack header\n");
					if (outredirect)
						fprintf(stderr, "Error: couldn't read pack header\n");
					flag_error = 2;
					break;
				}
				if ( fread(&pakheader, sizeof(packheader_t), 1, data) != 1 )
				{
					printf("Error: couldn't read pack header\n");
					if (outredirect)
						fprintf(stderr, "Error: couldn't read pack header\n");
					flag_error = 2;
					break;
				}
				/* read directory */
				directory = malloc(pakheader.dirlen);
				if (!directory)
				{
					printf("Error: couldn't allocate memory for directory structure\n");
					if (outredirect)
						fprintf(stderr, "Error: couldn't allocate memory for directory structure\n");
					flag_error = 2;
					break;
				}
				direntries = pakheader.dirlen / sizeof(packfile_t);
				if ( fseek(data, pakheader.dirofs, SEEK_SET) != 0 || ftell(data) != pakheader.dirofs )
				{
					printf("Error: couldn't read pack directory\n");
					if (outredirect)
						fprintf(stderr, "Error: couldn't read pack directory\n");
					flag_error = 2;
					free(directory);
					break;
				}
				if ( fread(directory, pakheader.dirlen, 1, data) != 1 )
				{
					printf("Error: couldn't read pack directory\n");
					if (outredirect)
						fprintf(stderr, "Error: couldn't read pack directory\n");
					flag_error = 2;
					free(directory);
					break;
				}
				/* search directory entries in list */
				for (i=0; i<direntries; i++)
				{
					map = path->maps;
					while (map && stricmp(directory[i].name, map->name))
					{
						map = map->next;
					}
					if (map)
					{
						printf("Found map \"%s\" ", directory[i].name);
						process_bspfile(data, directory[i].filepos, directory[i].filelen, map);
					}
				}
				break;
			default:		/* unknown	*/
				printf("Error: fileformat #%i can not be processed, please contact the authors\n", path->type);
				break;
		}
		fclose(data);

		printf("\n");
		if (flag_error) break;
		path = path->next;
	}
}


/*
******************************************************************************
*
*   M A I N   F U N C T I O N   ( S T A R T   O F   P R O G R A M )
*
******************************************************************************
*/

/*
==============================================================================
here the program starts and all sub functions are called
==============================================================================
*/
int main(int argc, char *argv[])
{
	int stop;
	pathentry *path;
	mapentry *map;
	texentry *tex;
	int pathes, maps, texsfound, texschanged;

	pathentry *start;

	/* initialize variables */
	config_file = default_config_file;	/* pointer on used config file name */
	start_directory = NULL;
	start = NULL;
	flag_undo = 0;
	flag_yes = 0;

	/* check if standard output is re-directed to a file */
	if (isatty(fileno(stdout)))
		outredirect = 0;
	else
		outredirect = 1;

	/* display program header lines */
	printf("Quake BSP Texture Renamer\n");
	printf("(c) 1998 by Matthias \"Maddes\" Buecher\n");
	printf("Version %s\n\n", VERSION);
	if (outredirect)
	{
		fprintf(stderr, "Quake BSP Texture Renamer\n");
		fprintf(stderr, "(c) 1998 by Matthias \"Maddes\" Buecher\n");
		fprintf(stderr, "Version %s\n\n", VERSION);
	}

	/* check parameters and let them confirm */
	stop = chk_param(argc, argv);
	if (stop)
	{
		exit(flag_error);
	}

	/* here we go with the "real" program */
	if (start_directory)
	{
		if (chdir(start_directory))
		{
			printf("Error: couldn't find start directory \"%s\"\n", start_directory);
			if (outredirect)
				fprintf(stderr, "Error: couldn't find start directory \"%s\"\n", start_directory);
			flag_error = 2;
		}
	}

	if (!flag_error)
	{
		start = read_cfgfile();
	}

	if (!flag_error)
	{
		process_list(start);
		path = start;
		pathes = 0;
		maps = 0;
		texsfound = 0;
		texschanged = 0;
		while (path)
		{
			pathes += path->found;
			map = path->maps;
			while (map)
			{
				maps += map->found;
				tex = map->textures;
				while (tex)
				{
					texsfound += tex->found;
					texschanged += tex->changed;
					tex = tex->next;
				}
				map = map->next;
			}
			path = path->next;
		}
		printf("--- Summary ---\n");
		printf("Files found: %i\n", pathes);
		printf("Maps  found: %i\n", maps);
		printf("Textures found/changed: %i / %i\n\n", texsfound, texschanged);
		if (maps && !texsfound && !flag_undo)
		{
			printf("No textures found, if you want to undo previous changes then use \"-u\"\n\n");
		}
	}

	if (start) freemen(start);

	if (cfg) fclose(cfg);

	if (flag_error)
	{
		exit(flag_error);
	}

	printf("Finished\n");
	if (outredirect)
		fprintf(stderr, "Finished\n");

	return 0;
}
