/*
==============================================================================
Title: "Quake BSP Texture Lister"
(c) 1998 by Matthias "Maddes" Buecher

Authors:
Matthias "Maddes" Buecher <maddes@bigfoot.com>

Important:
It should compile under Win32 and Unix with no errors, some warnings may still occur.
If you use a processor which has another byte order than Intel (lowest to highest byte),
you have to reprogram all offset functions, etc.

DISCLAIMER WARNING:
WE DO NOT ACCEPT RESPONSIBILITY FOR ANY EFFECTS, POSITIVE OR NEGATIVE,
THAT THIS CODE MAY CAUSE TOWARDS YOUR COMPUTER.
WE DO NOT SUPPORT MODIFICATIONS OF THIS CODE, USE ONLY AT YOUR OWN RISK.
THIS PATCH HAS NOT BEEN ENDORSED BY THE ID SOFTWARE CORPORATION.
THIS PROGRAM IS FREE OF CHARGE. IF YOU HAVE RECEIVED THIS PROGRAM THROUGH
PAYMENT, PLEASE CONTACT US.
OTHERWISE ENJOY THE PROGRAM. THANK YOU.
==============================================================================
*/

/* Compatible headers */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>

/* needed to compile under Linux */
#ifndef __WIN32__
#define stricmp strcasecmp
#endif

/*
structure definitions taken from Quake tool source codes
*/

/* BSP file */
#define BSP_VERSION 0x1d	/* Classic Quake */

#define LUMP_ENTITIES		0
#define LUMP_PLANES		1
#define LUMP_TEXTURES		2
#define LUMP_VERTEXES		3
#define LUMP_VISIBILITY		4
#define LUMP_NODES		5
#define LUMP_TEXINFO		6
#define LUMP_FACES		7
#define LUMP_LIGHTING		8
#define LUMP_CLIPNODES		9
#define LUMP_LEAFS		10
#define LUMP_MARKSURFACES	11
#define LUMP_EDGES		12
#define LUMP_SURFEDGES		13
#define LUMP_MODELS		14

#define	HEADER_LUMPS	15

typedef struct {
	long fileofs, filelen;		/* info about lump */
} lump_t;

typedef struct {			/* BSP file header */
	long version;			/* BSP file version */
	lump_t lumps[HEADER_LUMPS];	/* info table for lumps */
} dheader_t;

typedef struct {			/* texture lump header */
	long nummiptex;			/* number of textures in lump */
	long dataofs[1];		/* [nummiptex], info table for textures */
} dmiptexlump_t;

#define	MIPLEVELS	4

typedef struct miptex_s {		/* texture header */
	char name[16];			/* texture name */
	unsigned width, height;
	unsigned offsets[MIPLEVELS];	/* four mip maps stored */
} miptex_t;

/* PAK file */
#define PAK_HEADER	0x4b434150		/* 'PACK' */

typedef struct {			/* PAK file header */
	char id[4];			/* 'PACK' */
	long dirofs, dirlen;		/* info about directory table */
} packheader_t;

typedef struct {			/* directory entry */
	char name[56];			/* filename with path */
	long filepos, filelen;		/* info about file */
} packfile_t;

/*
==============================================================================
checks for .BSP or .PAK fileformat
==============================================================================
*/
int chk_fileformat(FILE *data, int type)
{
	long id;

	id = 0;
	fread(&id, 4, 1, data);
	switch ( type )
	{
		case 1:			/* BSP fileformat */
			if (!(id == (long)BSP_VERSION)) type = 0;
			break;
		case 2:			/* PAK fileformat */
			if (!(id == (long)PAK_HEADER)) type = 0;
			break;
		default:		/* unknown	*/
			type = 0;
			break;
	}

	return type;
}

/*
==============================================================================
checks for .BSP or .PAK extension
==============================================================================
*/
int chk_extension(char *filename)
{
	char *extension;
	char *slash;
	char *backslash;

	/* get position of last point */
	extension = strrchr(filename, '.');

	/* check if no other dir seperator follows */
	if (extension)
	{
		slash = strchr(extension, '/');
		backslash = strchr(extension, '\\');
		if ((slash) || (backslash))
			extension = NULL;
	}

	if (extension)
	{
		if (!stricmp(extension, ".pak"))
			return 2;	/* PAK extension */
		if (!stricmp(extension, ".bsp"))
			return 1;	/* BSP extension */
	}

	return 0;			/* unknown */
}

void process_bspfile(FILE *data, long bspofs)
{
	dheader_t bspheader;
	long nummiptex;			/* number of textures in lump */
	dmiptexlump_t *texofs;
	miptex_t *texture;

	long i, texdif;

	fseek(data, bspofs+0, SEEK_SET);
	if ( fread(&bspheader, sizeof(dheader_t), 1, data) != 1)
	{
		printf("Error: couldn't read bsp header\n");
	}
	else
	{
		/* checking bsp version / fileformat */
		if (bspheader.version != (long)BSP_VERSION)
		{
			printf("Error: unknown BSP version\n");
		}
		else
		{
			/* read number of textures */
			fseek(data, bspofs+bspheader.lumps[LUMP_TEXTURES].fileofs, SEEK_SET);
			fread(&nummiptex, 4, 1, data);

			/* read texture offsets */
			texofs = malloc(4+nummiptex*4);
			fseek(data, bspofs+bspheader.lumps[LUMP_TEXTURES].fileofs, SEEK_SET);
			fread(texofs, 4+nummiptex*4, 1, data);
			/* read texture headers */
			texture = malloc(nummiptex*sizeof(miptex_t));
			texdif = 0;
			for (i=0; i<nummiptex; i++)
			{
				if (texofs->dataofs[i] == -1)
				{
					texdif++;
					continue;
				}

				fseek(data, bspofs+bspheader.lumps[LUMP_TEXTURES].fileofs+texofs->dataofs[i], SEEK_SET);
				fread(&texture[i-texdif], sizeof(miptex_t), 1, data);
			}

			/* list textures */
			printf("%li texture entries with %li textures\n", nummiptex, nummiptex-texdif);
			for (i=0; i<(nummiptex-texdif); i++)
			{
				printf("\"%-56s\" %5i %5i\n", texture[i].name, texture[i].width, texture[i].height);
			}

			free(texofs);
			free(texture);
		}
	}
}

void process_file(char *path)
{
	FILE *data;

	packheader_t pakheader;
	packfile_t *directory;
	long direntries;
	long i;

	int type;

	/* read files and process */
	type = chk_extension(path);

	printf("*** Processing \"%s\"\n", path);

	/* open data file */
	data = fopen(path, "rb");
	if (!data)
	{
		if (errno == EACCES)
			printf("Warning: can not access file - write protected or a directory?\n");
		else
			printf("Warning: file not found\n");
	}
	else
	{
		switch ( type )
		{
			case 1:		/* BSP file */
				process_bspfile(data, 0);
				break;
			case 2:		/* PAK file */
				if (chk_fileformat(data, type) != type)
				{
					printf("Error: mismatch of extension and fileformat\n");
					break;
				}
				/* read pack header */
				fseek(data, 0, SEEK_SET);
				if ( fread(&pakheader, sizeof(packheader_t), 1, data) != 1)
				{
					printf("Error: couldn't read pack header\n");
					break;
				}
				/* read directory */
				directory = malloc(pakheader.dirlen);
				direntries = pakheader.dirlen / sizeof(packfile_t);
				fseek(data, pakheader.dirofs, SEEK_SET);
				if ( fread(directory, pakheader.dirlen, 1, data) != 1)
				{
					printf("Error: couldn't read pack directory\n");
					free(directory);
					break;
				}
				/* search directory entries in list */
				for (i=0; i<direntries; i++)
				{
					if ( chk_extension(directory[i].name) == 1)
					{
						printf("Found map \"%s\" ", directory[i].name);
						process_bspfile(data, directory[i].filepos);
						printf("\n");
					}
				}
				break;
			default:	/* unknown	*/
				printf("Error: fileformat #%i can not be processed, please contact the author\n", type);
				break;
		}
		fclose(data);
	}
}


/*
==============================================================================
here the program starts and all sub functions are called
==============================================================================
*/
int main(int argc, char *argv[])
{
	/* display program header lines */
	printf("Quake BSP Texture Lister\n");
	printf("(c) 1998 by Matthias \"Maddes\" Buecher\n\n");

	if (argc != 2)
	{
		/* syntax */
		printf("Usage: BSPTxLst <bsp or pak file>\n");
		/* contact */
		printf("\nHow to contact the author:\n");
		printf("Matthias \"Maddes\" Buecher <maddes@bigfoot.com>\n");
	}
	else
	{
		process_file(argv[1]);
	}

	return 0;
}
