/* tcp.c */
/* Manages the TCP client/server connection */

#include "header.h"


/* Build a connection from an fd */
tcp_conn_t *build_conn_from_fd(int fd) {
  tcp_conn_t *r = malloc(sizeof(tcp_conn_t));
  int ret;

  if (!r) { 
    return NULL;
  }
  
  r->fd = fd; 
  ret = fcntl(r->fd, F_SETFL, O_NONBLOCK);
  if (ret < 0) {
    warn(1, "Can't make socket non blocking"); free(r); return NULL;
  }
  r->top = TCP_PACKET_BUF_SIZ-1;
  r->firstFree =0; r->nextToWrite = 0;
  return r;
}

/* Accept a connection and return it */
tcp_conn_t *accept_on(tcp_conn_t *svr) {
  tcp_conn_t *r = malloc(sizeof(tcp_conn_t));
  int ret;
  int sin = sizeof(struct sockaddr_in);
  
  if (!r) { return NULL; }

  ret = accept(svr->fd, (struct sockaddr *)&r->remote, &sin);
  if (ret < 0) {
    if (errno == EINTR || errno==EAGAIN) {
      return NULL;
    } else {
      fatal(1, "Can't accept()");
    }
  }
  /* r->remote.sin_addr.s_addr = htonl(r->remote.sin_addr.s_addr); */
  r->fd = ret;

  ret = fcntl(r->fd, F_SETFL, O_NONBLOCK);
  if (ret < 0) {
    warn(1, "Can't make socket non blocking"); close(r->fd); free(r); return NULL;
  }
  r->top = TCP_PACKET_BUF_SIZ-1;
  r->firstFree =0; r->nextToWrite = 0;
  return r;
}

/* Send a packet if necessary */
void tcp_write_buffer(tcp_conn_t *conn) {
  int ret;
  
  if (conn->nextToWrite < conn->firstFree) { 
    /*printf("Writing %d bytes to TCP\n", conn->firstFree-conn->nextToWrite);*/
    ret = write(conn->fd, &conn->wbuffer[conn->nextToWrite],
		conn->firstFree-conn->nextToWrite);
    if (ret==0) {
      close(conn->fd);
      fatal(0, "Someone closed the TCP connection");
    }
    if (ret < 0) {
      if (errno != EAGAIN && errno != EINTR) {
        fatal(1, "Can't write() to TCP port");
      }
      return;
    }
    conn->nextToWrite += ret;
  }
  if (conn->firstFree-conn->nextToWrite < MOVE_TCP_BUFFER_WINDOW) { 
    memcpy(conn->wbuffer, &conn->wbuffer[conn->nextToWrite], 
	   conn->firstFree-conn->nextToWrite);
    conn->firstFree -= conn->nextToWrite;
    conn->nextToWrite = 0;
  }
}

/* Recieve some data and add it to the end of the read buffer. 
 * If there's a packet in the read buffer, and the read buffer is full, toss the packet.
 */
void tcp_read_buffer(tcp_conn_t *conn) {
  int ret;

  if (conn->rptr >= TCP_PACKET_BUF_SIZ) {
    int len = ntohl(((unsigned int *)conn->rbuffer)[0]);
    int toss;

    if (len > TCP_PACKET_BUF_SIZ) {
      warn(0, "Help ! Packet of length %d (> max buffer size). Tossing as much as I can.", len);
      toss = TCP_PACKET_BUF_SIZ;
    } else {
      toss = len;
    }
    memcpy(conn->rbuffer, &conn->rbuffer[len+4], TCP_PACKET_BUF_SIZ-(len+4));
    conn->rptr = TCP_PACKET_BUF_SIZ-(len+4);
  }
  
  ret = read(conn->fd, &conn->rbuffer[conn->rptr], TCP_PACKET_BUF_SIZ-conn->rptr);
  /* printf("Read %d gets %d", ret, TCP_PACKET_BUF_SIZ-conn->rptr); */
  if (ret==0) {
    close(conn->fd);
    fatal(0, "Someone closed the TCP connection");  
  }
  if (ret < 0) {
    if (errno != EAGAIN && errno != EINTR) {
      fatal(1, "Error reading data from connection");
    } else {
      return;
    }
  } else {
    /* printf("TCP_READ %d\n", ret); */
    conn->rptr += ret;
  }
}
	

/* Connect to a server */
tcp_conn_t *connect_to_svr_socket(ipv4_addr_t server, ipv4_port_t svrport) {
  tcp_conn_t *conn = malloc(sizeof(tcp_conn_t));
  int ret;
  struct sockaddr_in addr;

  if (!conn) {
    fatal(0, "Can't allocate %d bytes for connection", sizeof(tcp_conn_t));
  }

  conn->fd  = socket(AF_INET, SOCK_STREAM, 0);
  if (conn->fd < 0) {
    fatal(1, "Can't create socket");
  }
  
  addr.sin_family = AF_INET;
  addr.sin_addr.s_addr = 0;
  addr.sin_port = 0;

  ret = bind(conn->fd, (struct sockaddr *)&addr, sizeof(struct sockaddr_in));
  if (ret < 0) {
    fatal(1, "Can't bind() client port to anything");
  }
  conn->local = addr;

  addr.sin_family = AF_INET;
  addr.sin_addr.s_addr = server;
  addr.sin_port = htons(svrport);
  
  ret = connect(conn->fd, (struct sockaddr *)&addr, sizeof(struct sockaddr_in));
  conn->remote = addr;
  if (ret < 0) {
    fatal(1, "Can't connect()");
  }

  ret = fcntl(conn->fd, F_SETFL, O_NONBLOCK);
  if (ret < 0) {
    fatal(1, "Can't make socket non blocking");
  }

  conn->top = TCP_PACKET_BUF_SIZ-1;
  conn->firstFree =0; conn->nextToWrite = 0;
  conn->rptr = 0;

  return conn;
}


/* Create a server socket */
tcp_conn_t *create_svr_socket(ipv4_port_t port) {
  tcp_conn_t *conn = malloc(sizeof(tcp_conn_t));
  int ret;
  int onoff;
  struct sockaddr_in addr;
  
  if (!conn) {
    fatal(0, "Can't allocate %d bytes for connection", sizeof(tcp_conn_t));
  }
  
  conn->fd = socket(AF_INET, SOCK_STREAM,0);
  if (conn->fd < 0) {
    fatal(1, "Can't create a socket");
  }

  addr.sin_family = AF_INET;
  addr.sin_addr.s_addr = INADDR_ANY;
  addr.sin_port = htons(port);
  
  ret = bind(conn->fd, (struct sockaddr *)&addr, sizeof(struct sockaddr_in));
  conn->local = addr;
  if (ret < 0) {
    fatal(1, "Can't bind() to socket");
  }

  /* Set SO_REUSEADDR, so we don't end up in FIN_WAITs all the time */
  onoff = 1;
  ret = setsockopt(conn->fd, SOL_SOCKET, 
		   SO_REUSEADDR, &onoff, sizeof(onoff));
  if (ret < 0) {
    fatal(1, "Can't set SO_REUSEADDR");
  }
  

  ret = listen(conn->fd, 2);
  if (ret < 0) {
    fatal(1, "Can't listen()");
  }
  
  return conn;
}

int tcp_blocking_read(int fd, char *buf, int len) {
  int done = 0, ret;

  while (done < len) {
    ret = read(fd, &buf[done], len-done);
    if (ret==0) { return -2; } /* Close */
    if (ret < 0) {
      if (errno != EAGAIN && errno != EINTR) return -1;
    } else {
      done += ret;
    }
  }
  
  return 0;
}

int tcp_blocking_write(int fd, char *buf, int len) {
  int done = 0, ret;

  while (done < len) {
    /* printf("Attempting write %d \n", len-done); */
    ret = write(fd, &buf[done], len-done);
    /* printf("Ret %d\n", ret); */
    if (ret==0) { return -2; } /* Close */
    if (ret < 0) {
      /* printf("W err %d\n", errno);*/
      if (errno != EAGAIN && errno != EINTR) return -1;
    } else {
      done += ret;
    }
  }
  
  return 0;
}
  

  
