/* server.c */

#include "header.h"
#include "helptext.h"
#ifdef _POSIX_PRIORITY_SCHEDULING
#include <sched.h>
#endif

extern FILE *yyin;
extern int yyparse(void);

static tcp_conn_t *tconn, *sconn;
extern int test_only, log_all_packets, do_fork, verbose;
extern int from_inetd, use_mlock, use_rt;

static void cleanup(void);
static void handle_reread(int t);
int main(int argn, char *args[]);

static void cleanup(void) {
  if (sconn != NULL) { close(sconn->fd); }
  if (tconn != NULL) { close(tconn->fd); }
  close_all_routes();
}

static void handle_fatal(int thingy) {
  info(0, "pid %d: signal %d recieved: shutting down.", getpid(), thingy);
  cleanup();
  exit(1);
}

static void handle_reread(int thingy) {
  info(0, "Reread on SIGHUP not implemented yet. Use SIGINT to kill server.");
}

int main(int argn, char *args[]) {
  int fb=1, b0,b1;
  int port_nbo=0, ret;
  FILE *f;
  char *cookie;
  fd_set all_udpfds;
  int max_udpfd = 0, max_fd = 0;
  struct timeval last_prune;
  char *pnam = args[0];
  char *config_file = DEFAULT_CONFIG_FILE;
  struct timeval nt;

#ifdef HAVE_SYSLOG
  openlog(LOG_IDENT_SRV, 0, LOG_FACILITY);
#endif

  FD_ZERO(&all_udpfds);

  svr_static_flags = 0;
  use_mlock = DEFAULT_USE_MLOCK; use_rt = DEFAULT_USE_RT;

  args++;argn--;
  while (argn > 0) {
    if (!strcmp(*args, "--version")) {
      printf("rqpserver version " VERSION " built for " TARGET " on " __DATE__ " at " __TIME__ ".\n");
      printf("Do %s --help for help.\n", pnam);
      exit(0);
    }
    if (!strcmp(*args, "--help")) {
      printf(SERVER_HELP_TEXT);
      exit(0);
    }
    if (!strcmp(*args, "--no-mlock")) {
      use_mlock = 0;svr_static_flags |= SVR_FLAGS_OPTIONS_MLOCK;
      goto end_arg_loop;
    }
    if (!strcmp(*args, "--no-rt")) {
      use_rt = 0; svr_static_flags |= SVR_FLAGS_OPTIONS_RT;
      goto end_arg_loop;
    }
    if (!strcmp(*args, "--use-mlock")) {
      use_mlock = 1; svr_static_flags |= SVR_FLAGS_OPTIONS_MLOCK;
      goto end_arg_loop;
    }
    if (!strcmp(*args, "--use-rt")) {
      use_rt = 1; svr_static_flags |= SVR_FLAGS_OPTIONS_RT;
      goto end_arg_loop;
    }
    if (!strcmp(*args, "--test-only")) {
      test_only++; svr_static_flags |= SVR_FLAGS_OPTIONS_TEST;
      goto end_arg_loop;
    }
    if (!strcmp(*args, "--log-all-packets")) {
      log_all_packets++; svr_static_flags |= SVR_FLAGS_OPTIONS_LOG;
      goto end_arg_loop;
    }
    if (!strcmp(*args, "--do-fork")) {
      do_fork = 1;
      from_inetd = 0; svr_static_flags |= SVR_FLAGS_OPTIONS_NOFORK;
      goto end_arg_loop;
    }
    if (!strcmp(*args, "--no-fork")) {
      do_fork = 0; svr_static_flags |= SVR_FLAGS_OPTIONS_NOFORK;
      goto end_arg_loop;
    }
    if (!strcmp(*args, "--from-inetd")) {
      from_inetd = 1; do_fork = 0; svr_static_flags |= SVR_FLAGS_OPTIONS_INETD;
      goto end_arg_loop;
    }
    if (!strcmp(*args, "-f")) {
      if (argn < 2) {
	fatal(0, "Filename must follow '-f'");
      }
      config_file = *(++args); argn--;
      goto end_arg_loop;
    }
    if (!strcmp(*args, "-p")) {
      if (argn < 2) { 
	fatal(0, "Port number must follow -p");
      }
      port_nbo = htons(atoi(*(++args))); argn--;
      goto end_arg_loop;
    }
    if (!strcmp(*args, "-v")) {
      verbose++; svr_static_flags |= SVR_FLAGS_OPTIONS_VERBOSE;
      goto end_arg_loop;
    }
    fatal(0,"Unrecognised option %s", *args);
  end_arg_loop:
    argn--;args++;
  }
      
  route_init();


  tconn = NULL; sconn= NULL;
  /* Install the signal handlers .. */
  signal(SIGINT, handle_fatal);
  signal(SIGHUP, handle_reread);
  signal(SIGPIPE, handle_fatal);
  signal(SIGQUIT, handle_fatal);

  if (from_inetd) { safe_stdout = 0; safe_stderr = 0; }

  if (!(f=fopen(config_file, "rb"))) {
    fatal(1, "Can't open configuration file %s", config_file);
  }
#if USE_YACC
  yyin = f;
  if (yyparse()) { 
    fatal(1, "Parse error in %s", config_file);
  }
#else
  if (readConfigFile(f)) {
    fatal(0, "Can't read configuration file");
  }
#endif
  fclose(f);
  
  if ((svr_static_flags&0x07) != 0x07) {
    fatal(0, "Configuration file didn't contain client name, port and cookie");
  }
  
  cookie = svr_static_cookie;
  port_nbo = port_nbo ? port_nbo : svr_static_client_port;

  if (test_only) {
    info(0, "Configuration file read:");
    info(0,"Listening on port %d, cookie is %s", ntohs(port_nbo), cookie);
    info(0,"do_fork is %d , from_inetd is %d\n", do_fork, from_inetd);
    
    dump_acls(route_acl_table.entries, route_acl_table.occ);
    dump_redirs(route_config_table.entries, route_config_table.occ,1);
    exit(0);
  }

  sconn = create_svr_socket(port_nbo);
  
  info(0,"pid %d created socket %d. Waiting for client to connect...", 
       getpid(),
       ntohs(port_nbo));

  if (!from_inetd) { 
    if (do_fork) {
       reap_children_as_they_die();
    }      
    do {
      while (1) {
	ipv4_addr_t rem;
	int done = 0,i;
	
	tconn = accept_on(sconn);
	if (tconn!=NULL) {
	  rem  = tconn->remote.sin_addr.s_addr;

	  for (i =0;i<client_acl_table.occ;i++) {
	    if (matchACL(client_acl_table.entries[i], 
			 rem, tconn->remote.sin_port)) {
	      done =1;
	      break;
	    }
	  }
	  if (done) break; 
	  rem = ntohl(tconn->remote.sin_addr.s_addr);
	  warn(1, "Potentially bogus connection (not right client IP) from %d.%d.%d.%d",
	       (rem>>24)&0xff, (rem>>16)&0xff, (rem>>8)&0xff, (rem>>0)&0xff);
	  close(tconn->fd);
	  free(tconn);
	}
      }
    
      if (do_fork) {
	pid_t pid;
	
	pid = fork();
	if (pid==0) {
	  if (verbose) {
	    info(0, "Child, pid %d, forked to deal with client.", getpid());
	  }
	  /* Child */
	  break;
	} else {
	  /* Parent - continue */
	  close(tconn->fd); free(tconn);
	}
      }
   } while (do_fork);
  } else {
    tconn = build_conn_from_fd(0);
  }

  ret = ft_server_handle_client_connect(tconn, cookie); 
  if (ret < 0) {
    fatal(0, "Can't initiate connection with client");
  }
  switch (ret) {
  case CLIENT_FN_CODE_PROXY: /* Nothing */
    break;
  default:
    fatal(0, "Client requested invalid function code %d", ret);
  }
  
  info(0,"Client connected");
  
  close(sconn->fd); free(sconn);
  max_udpfd = -1;
  max_fd = tconn->fd;
  gettimeofday(&last_prune, NULL);
  gettimeofday(&nt, NULL);

  timeval_adds(nt, 10);
  
  /* Set the scheduler */
  lock_and_set_scheduler(use_mlock, use_rt, 1);

  while (1) {
    int ret, bad = 0;
    fd_set rs, ws, xs;
    int cur_fd,i;
    char udp_buffer[UDP_BUFFER_SIZE];
    struct timeval tout;

back_to_select:
    memcpy(&rs, &all_udpfds, sizeof(fd_set));
    FD_SET(tconn->fd, &rs);
    FD_ZERO(&ws);
    if (tconn->nextToWrite < tconn->firstFree) { FD_SET(tconn->fd, &ws); }
    FD_ZERO(&xs);
    
    gettimeofday(&tout, NULL);
    timeval_sub(nt, tout);
    /* info(0, "Wait %d,%d", nt.tv_sec, nt.tv_usec); */
    /* Just in case .. */

#ifdef _POSIX_PRIORITY_SCHEDULING
#ifdef HAVE_RTSCHED 
   if (use_rt) { 
      sched_yield();
    }
#endif
#endif
    ret = select(max_fd+1, &rs,&ws,&xs,&nt);
    if (ret<0) {
      if (bad >= SELECT_BOMBS_HOWMANY) {
	warn(1, "Select() bombed out 8 times in a row. Sleeping ..");
	usleep(SELECT_BOMBS_USLEEP_FOR);
	/* bad=0; */
      }
      if (errno != EINTR && errno != EAGAIN) { 
	warn(1, "Help ! Select() exited.");
      }
      warn(0, "Select() bombed");
      bad++;
      goto back_to_select;
    }
    
    bad = 0;
    gettimeofday(&tout, NULL);
    if (tout.tv_sec - last_prune.tv_sec > PRUNE_AFTER) {
      for (i=0;i<routing_table.length;i++) {
	server_route_t *cur = routing_table.fwd[i];
	server_route_t *n;

	while (cur) {
	  n = cur->next;
#if DROP_CONN_ASYMMETRIC
	  if (tout.tv_sec - cur->last_in.tv_sec > PRUNE_AFTER ||
	      tout.tv_sec - cur->last_out.tv_sec > PRUNE_AFTER) {
#else
	    if (tout.tv_sec - cur->last_pckt.tv_sec > PRUNE_AFTER) { 
#endif
	    warn(0, "Deleting unused route to %lx, %d from %lx, %d via port %d (fd %d)",
		 ntohl(cur->svr_name), ntohs(cur->svr_port),
		 ntohl(cur->orig_name), ntohs(cur->orig_port),
		 ntohs(cur->local_port), cur->fd_destination);
	    /* Remove it from the packet thingies */
	    discard_packet_for(cur, &IQ); discard_packet_for(cur, &OQ);
	    close(cur->fd_destination);
	    FD_CLR(cur->fd_destination, &all_udpfds);
	    FD_CLR(cur->fd_destination, &rs);
	    if (max_fd == cur->fd_destination) { 
	      max_fd--; /* Not sure if this does any good, but it can't hurt to try */
	    }
	    delete_route(cur);
	    break;
	  }
	  cur = n;
	}
      }
    }
    
      if (ret != 0) { 
	fb = -fb;
	/* info(0, "fb = %d", fb); */
	if (fb>0) { b0 = 0; b1 = max_fd+1; } else { b0 = max_fd; b1 = -1; }
	for (cur_fd = b0; cur_fd != b1; cur_fd += fb) {
	  SEND_TOP_PCKT;
	  if (FD_ISSET(cur_fd, &rs)) { 
	    if (cur_fd == tconn->fd) {
	      /* TCP connection ready for reading */
	      tcp_read_buffer(tconn);
	      /* Is there a full packet in there ? */
	      {
		unsigned long len;
		unsigned long *rbuf = ((unsigned long *)(tconn->rbuffer));
		
		len = ntohl(rbuf[0]);
		while (len < tconn->rptr) {
		  server_route_t *rt;
		  
		  if (log_all_packets) {
		    info(0,"Incoming packet from port %d (originally %lx, %d, length %ld)",
			 ntohs(rbuf[3]),ntohl(rbuf[1]), ntohs(rbuf[4]), len);
		  }
		  rt = getIncomingConnection(rbuf[2], rbuf[3],
					     rbuf[1], rbuf[4]);
		  if (!rt ) {
		    /* Open a connection ourselves */
		    if (routing_table.occ >= ROUTE_MAX) {
		      /* Too many connections */
		      warn(0, "Can't build route: too many (%d) connections", ROUTE_MAX);
		      goto discard_packet;
		    }
		    rt = getFreeBlock();
		    if (!rt) {
		      warn(0, "Can't allocate memory for connection");
		      goto discard_packet;
		    }
		    {
		      int done = 0;
		      

		      rt->orig_name = rbuf[1];
		      rt->orig_port = rbuf[4];
		      rt->clnt_name = rbuf[2];
		      rt->clnt_port = rbuf[3];
		      
		      /* Find out who we're supposed to connect to */
		      for (i=0;i<route_config_table.occ;i++) {
			if (route_config_table.entries[i].client_name == 
			    rbuf[2] &&
			    route_config_table.entries[i].client_port == 
			    rbuf[3]) {
			  if (route_config_table.entries[i].u.assocacl == 
			      NULL ||
			      matchACL((*(route_config_table.entries[i].u.assocacl)),
				       rt->orig_name, rt->orig_port)) { 
			    rt->svr_name = route_config_table.entries[i].svr_ip;
			    rt->svr_port = route_config_table.entries[i].svr_port;
			    rt->delay_in = route_config_table.entries[i].delay_in;
			    rt->delay_out = route_config_table.entries[i].delay_out;
			    done = 1;
			    break;
			}
			}
		      }
		      if (!done) {
			warn(0, "Can't find destination server for %lx, %ld",
			     ntohl(rt->clnt_name), ntohs(rt->clnt_port));
			putFreeBlock(rt);
			goto discard_packet;
		      }
		    }
		    rt->pckt_out_len = 0;
		    rt->pckt_in_len = 0; 
		    rt->destination.sin_family = AF_INET;
		    rt->destination.sin_addr.s_addr = rt->svr_name;
		    rt->destination.sin_port = rt->svr_port;
		    rt->ttl = TTL_USED;
		    rt->fd_destination = socket(AF_INET, SOCK_DGRAM, 0);
		    if (rt->fd_destination < 0) {
		      warn(1, "Can't open socket to svr %x, port %d", ntohl(rt->svr_name),
			   ntohs(rt->svr_port));
		      putFreeBlock(rt);
		      goto discard_packet;
		    }
		    {
		      struct sockaddr_in saddr;
		      
		      saddr.sin_family = AF_INET;
		      saddr.sin_addr.s_addr = INADDR_ANY;
		      saddr.sin_port = 0;
		      
		      ret = bind(rt->fd_destination, 
				 (struct sockaddr *)&saddr, sizeof(struct sockaddr_in));
		      if (ret < 0) {
			warn(1, "Can't bind() socket to svr %x, port %d", ntohl(rt->svr_name),
			     ntohs(rt->svr_port));
			close(rt->fd_destination);
			putFreeBlock(rt);
			goto discard_packet;
		      }
		      rt->local_port = saddr.sin_addr.s_addr;
		      /* And register it */
		      info(0, "New route from %lx, %ld to %lx, %ld (fd %d)",
			   ntohl(rt->orig_name), ntohs(rt->orig_port),
			   ntohl(rt->svr_name), ntohs(rt->svr_port),
			   rt->fd_destination);
		      add_route(rt);
		      if (rt->fd_destination > max_udpfd) { max_udpfd = rt->fd_destination; }
		      if (max_udpfd > max_fd) { max_fd = max_udpfd; }
		      FD_SET(rt->fd_destination, &all_udpfds);
		      gettimeofday(&rt->last_in, NULL);
		      gettimeofday(&rt->last_out, NULL);
		    }
		  }
		  /* OK. Now rt contains the current connection, simply write the packet to it */
		  if (log_all_packets) {
		    info(0,"Sending packet to fd %d (%lx, %d). Len %ld", rt->fd_destination,
			 ntohl(rt->destination.sin_addr.s_addr),
			 ntohs(rt->destination.sin_port), len-16);
		  }
		  {
#if ENABLE_THROTTLE
	      struct timeval f2;
	      
	      ret = gettime
		ofday(&f2, NULL);
	      
	      if (TIMEVAL_DIF(f2, rt->last_in) < THROTTLE_INCOMING) {
		warn(0, "Throttling packet to %lx, %d",
		     ntohl(rt->destination.sin_addr.s_addr), ntohl(rt->destination.sin_port));
	      } else {
#endif
		rt->last_in = tout; 
		  rt->last_pckt = tout;
		  if (rt->pckt_in_len == 0 || INDUCED_LATENCY_DROP_POLICY) { 
		    if (len < SERVER_PACKET_BUFFER_SZ) {
		      if (rt->pckt_in_len) {
			discard_packet_for(rt, &IQ);
		      }
		      timeval_add(rt->due_in, rt->delay_in, tout);
		      rt->pckt_in_len = len-16;
		      /* We simply replace any pending packet */
		      memcpy(rt->pckt_in, &tconn->rbuffer[20], len-16); 
		      insert_packet_for_in(rt);
		    } else {
		      warn(0, "Packet from %lx, %d too big (> %d)",
			   ntohl(rt->orig_name), ntohl(rt->orig_port),
			   SERVER_PACKET_BUFFER_SZ);
		    }
		  }
#if ENABLE_THROTTLE
	      }
#endif
		  }
		discard_packet:
	      memcpy(&tconn->rbuffer, &tconn->rbuffer[len+4], tconn->rptr-(len+4));
	      tconn->rptr -= len+4;
	      len = ntohl(rbuf[0]);
		  }
	      }	       
	    } else {
	int fl, r;
	struct sockaddr_in saddr;
	
	
	/* It's a UDP connection ready for reading. */
	r = sizeof(struct sockaddr_in);
	fl = recvfrom(cur_fd, &udp_buffer[20], UDP_BUFFER_SIZE-20, 0, 
		       (struct sockaddr *)&saddr, &r);
	if (fl < 0) {
	  if (errno==EBADF || errno==ECONNREFUSED) {
	    server_route_t *rt;

	    
	    rt = searchForOutgoingConnection(cur_fd);
	    if (!rt) {
	      warn(0, "Can't find outgoing connection for fd %d", cur_fd);
	    } else {
	      warn(0, "Shutting down unconnected file descriptor %d", cur_fd);
	      discard_packet_for(rt, &IQ); discard_packet_for(rt, &OQ);
	      close(rt->fd_destination);
	      FD_CLR(rt->fd_destination, &all_udpfds);
	      FD_CLR(rt->fd_destination, &rs);
	      if (max_fd == rt->fd_destination) { max_fd--; }
	      delete_route(rt);
	    }
	  } else {
	    warn(1, "recvfrom() failed (fd %d) ?", cur_fd);
	  }
	}
	
	/* Check if we're going to route this packet */
	if (fl > 0 && fl+20 < SERVER_PACKET_BUFFER_SZ) {
	  server_route_t *rt;
	  unsigned long *rbuf = ((unsigned long *)udp_buffer);
	  
	  if (log_all_packets) {
	    info(0,"Incoming packet from UDP %lx, %d to fd %d, len %d",
		   ntohl(saddr.sin_addr.s_addr), ntohs(saddr.sin_port),
		   cur_fd, 
		   fl);
	    }


	  rt = getOutgoingConnection(saddr.sin_addr.s_addr, saddr.sin_port, cur_fd);
	  if (!rt) {
	    warn(1, "No outgoing connection for %lx, %ld", 
		 ntohl(saddr.sin_addr.s_addr), ntohs(saddr.sin_port));
	  } else {
#if ENABLE_THROTTLE
	    struct timeval f2;
	    
	    ret = gettimeofday(&f2, NULL);
	    if (TIMEVAL_DIF(f2,rt->last_out) < THROTTLE_OUTGOING) {
	      warn(0, "Throttling packet from %lx, %ld", 
		   ntohl(saddr.sin_addr. s_addr), ntohs(saddr.sin_port));
	    } else {
#endif	      
	      rt->last_out = tout;	
	     rt->last_pckt = tout;
	     
	     rbuf[0] = htonl(fl+16);
	    rbuf[1] = rt->orig_name;
	    rbuf[2] = rt->clnt_name;
	    rbuf[3] = rt->clnt_port;
	    rbuf[4] = rt->orig_port;
	    
	    if (rt->pckt_out_len == 0 || INDUCED_LATENCY_DROP_POLICY) {
	      if (rt->pckt_out_len) {
		discard_packet_for(rt, &OQ);
	      }
	      timeval_add(rt->due_out, rt->delay_out, tout);
	      /* This is a pseudo-random delay addition so we don't get synchronous 
		 packet starvation in the output queue. */
	      timeval_addus(rt->due_out, tout.tv_usec%16); 
	      memcpy(rt->pckt_out, udp_buffer, fl+20);
	      rt->pckt_out_len = fl+20;
	      insert_packet_for_out(rt);
	    }
	    }
#if ENABLE_THROTTLE
	  }
#endif
	} else {
	  if (fl >= 0) { 
	    warn(0, "Tossing %d length packet from %lx, %ld", 
		 fl, ntohl(saddr.sin_addr. s_addr), ntohs(saddr.sin_port));
	  }
	}
      }
	}
      }
      }
    
  

    /* If there are any packets due for writing, write them. If there are any packets more than
     PRUNE_PACKETS_AT us old, discard them. */
    
      /*    gettimeofday(&tout, NULL); */
    tout.tv_usec++; /* HACK */
    nt = tout;
    timeval_adds(nt, 10);

    /* Process the incoming queue .. */
    {
      int i,j;
      struct timeval cutoff;

      i = 0;
      while (i < IQ.occ) {
	if (timeval_after(tout, IQ.entries[i]->due_in)) {
	  server_route_t *rt = IQ.entries[i];
	  
	  if (log_all_packets) {
	    info(0, "Sending delayed packet to %lx, %d", ntohl(rt->svr_name), 
		 ntohs(rt->svr_port));
	  }
	  sendto(rt->fd_destination, rt->pckt_in, rt->pckt_in_len, 0,
		 (struct sockaddr *)&rt->destination, sizeof(struct sockaddr_in));
	  rt->pckt_in_len = 0; i++;
	} else {
	  if (timeval_before(IQ.entries[i]->due_in, nt)) { nt = IQ.entries[i]->due_in; }
	  break;
	}
      }

      /* Yes, this could be combined with the other reap loop, but I really don't want to do that .. */
      for (j = i;j<IQ.occ;j++) { 
	IQ.entries[j-i] = IQ.entries[j];
      }
      IQ.occ -= i;
      
      i = IQ.occ-1;
      while (i >= 0) {
	cutoff = IQ.entries[i]->due_in;
	
	timeval_addus(cutoff, DISCARD_AFTER_US);
	if (timeval_after(tout, cutoff)) i--; else break;
      }
      for (j=i+1;j<IQ.occ;j++) { IQ.entries[j]->pckt_out_len = 0; }
      IQ.occ = i+1;   
    }
    /* Now, the outgoing queue. I'm not sure whether we should discard due packets that
     don't fit in the buffer, or just delay them: on the one hand, this stops synchronicity
     problems, but on the other it can cause back-run. I think we'll discard and rely on local
    network jitter to randomise things (indeed, let's add the connection # to tout later on).
    */
    {
      int i,j;
      struct timeval cutoff;

      i = 0;
      while (i < OQ.occ) {
	if (timeval_after(tout, OQ.entries[i]->due_out)) {
	  server_route_t *rt = OQ.entries[i];
	  
	  if (log_all_packets) {
	    info(0, "Sending delayed packet from %lx, %d to TCP", ntohl(rt->orig_name), 
		 ntohs(rt->orig_port));
	  }
	  if (tconn->top - tconn->firstFree >= rt->pckt_out_len) {
	    memcpy(&tconn->wbuffer[tconn->firstFree], rt->pckt_out, rt->pckt_out_len);
	    tconn->firstFree += rt->pckt_out_len;
	    /* the write stuff for TCP's select() will happen automagically on the next
	       schedule */
	  } 
	  i++;
	  rt->pckt_out_len = 0;
	} else {  
	  if (timeval_before(OQ.entries[i]->due_out, nt)) { nt = OQ.entries[i]->due_out; }
	  break;
	}
      }
      /* Just try it .. */
      if (FD_ISSET(tconn->fd, &ws)) {
	  tcp_write_buffer(tconn);
	}
      /*  tcp_write_buffer(tconn);   */
 /* Yes, this could be combined with the other reap loop, but I really don't want to do that .. */
      for (j = i;j<OQ.occ;j++) { 
	OQ.entries[j-i] = OQ.entries[j];
      }
      OQ.occ -= i;
      
      i = OQ.occ-1;
      while (i >= 0) {
	cutoff = OQ.entries[i]->due_out;
	
	timeval_addus(cutoff, DISCARD_AFTER_US);
	if (timeval_after(tout, cutoff)) i--; else break;
      }
      for (j=i+1;j<OQ.occ;j++) { OQ.entries[j]->pckt_out_len = 0; }
      OQ.occ = i+1;  
    }
   }
   

  return 0;
}

  
 
  
