/* route.h */

#ifndef __ROUTE_H
#define __ROUTE_H

#include "header.h"


#define TIMEVAL_DIF(t1,t2) (((t1.tv_sec-t2.tv_sec)*1000000)+(t1.tv_usec-t2.tv_usec))

/* ACL hash table entry */
typedef struct acl_ht_struct {
  ipv4_addr_t allowed;
  ipv4_addr_t mask;
  int pstart, pend;
  int deny;
} acl_ht_entry_t;


/* Table mapping (clnt name, p2) to (server, port) */
typedef struct server_conf_elem_struct {
  ipv4_addr_t client_is;
  ipv4_addr_t client_name;
  ipv4_port_t client_port;
  ipv4_addr_t svr_ip;
  ipv4_port_t svr_port;
  struct timeval delay_in, delay_out;
  /* The history of this pointer is long and convoluted. Basically, the client gets
     routing entries by direct write()s of this structure, and does linear searches,
     so we don't want to gratuitously increase the length of the structure. So we
     put a pointer here that the client uses as an fd storage area. */
  union {
    acl_ht_entry_t *assocacl;   
    int client_fd;
  } u;
   int applies;
} server_conf_elem_t;

/* Table mapping client fds to client ports */
typedef struct client_fd_entry_struct {
  server_conf_elem_t *origin;
  int client_fd, used;
  ipv4_port_t clnt_port;
  ipv4_addr_t clnt_name;
} client_fd_entry_t;

/* Table mapping (clnt name, a.b.c.d, p1, p2) -> (struct inaddr_in *tgt, int fd, int ttl, int htback) */
#define TTL_CLEARED 0
#define TTL_USED 1
#define TTL_EMPTY 2
typedef struct server_routing_struct {
  ipv4_addr_t orig_name;
  ipv4_port_t orig_port;
  ipv4_addr_t clnt_name;
  ipv4_port_t clnt_port;
  ipv4_addr_t svr_name;
  ipv4_port_t svr_port;
  ipv4_port_t local_port;
  struct sockaddr_in destination;
  int fd_destination;
  int ttl;
  struct server_routing_struct *next, *prev;
  struct server_routing_struct *next2, *prev2;
  struct timeval last_out, last_in;
  struct timeval delay_in, delay_out, due_out, due_in;
  char *pckt_out;
  char *pckt_in;
  int pckt_in_len, pckt_out_len;
  struct timeval last_pckt;
} server_route_t;


/* Table types. */
typedef struct acl_table_struct {
  acl_ht_entry_t *entries;
  int length;
  int occ; /* Occupancy - 1 => one ACL entry */
} acl_table_t;

typedef struct client_fd_table_struct {
  client_fd_entry_t *entries;
  int length;
  int occ;
} client_fd_table_t;

typedef struct svr_conf_table_struct {
  server_conf_elem_t *entries;
  int length;
  int occ; /* Occupancy - 1 => one config structure */
} svr_conf_table_t;

typedef struct svr_route_table_struct {
  server_route_t **fwd;
  server_route_t **bwd;
  int length, occ;
} svr_route_table_t;

typedef struct svr_route_freelist_struct {
  server_route_t *lst;
  int howmany;
} svr_route_freelist_t;

typedef struct svr_packet_queue_struct {
  server_route_t **entries;
  int length;
  int occ;
} svr_packet_queue_t;


/* Static variables containing routing tables. These are allocated by route_init() */
extern acl_table_t route_acl_table, client_acl_table;
extern svr_conf_table_t route_config_table;
extern svr_route_table_t routing_table;
extern svr_route_freelist_t freeList;
extern client_fd_table_t client_fdtable;
extern svr_packet_queue_t outgoing_queue, incoming_queue;
  
/* Horrible convenience defines */
#define IQ incoming_queue
#define OQ outgoing_queue

/* Packet queue stuff */
void discard_packet_for(server_route_t *e, svr_packet_queue_t *qt);
int insert_packet_for_in(server_route_t *e);
int insert_packet_for_out(server_route_t *e);

void close_all_routes();


void route_init(void);
int access_allowed(ipv4_addr_t addr, ipv4_port_t port);
void delete_route(server_route_t *rt);
void add_route(server_route_t *rt);
void manage_freelist();
void putFreeBlock(server_route_t *t);
server_route_t *searchForOutgoingConnection(int fd);
server_route_t *getFreeBlock(void);
server_route_t *getIncomingConnection(ipv4_addr_t client, ipv4_port_t cl_port,
				      ipv4_addr_t orig, ipv4_port_t orig_port);
server_route_t *getOutgoingConnection(ipv4_addr_t svr, ipv4_port_t svr_port, 
				     int fd);




int getHashBucketIncoming(ipv4_addr_t clnt, ipv4_port_t cl_port, 
			  ipv4_addr_t orig, ipv4_port_t orig_port);
int getHashBucketOutgoing(ipv4_addr_t svr, ipv4_port_t svr_port, int fd);


#endif
