/* io.c */
/* Handle I/O. */

#include "header.h"

static int readACLData(FILE *f, acl_ht_entry_t *a, int b, char *l);
static int readProxyData(FILE *f, server_conf_elem_t *a, int b, char *l);

int safe_stdout = 1, safe_stderr = 1;

void info(int perr, char *fmt, ...) { 
  va_list lst;
  char buf[2048];

  va_start(lst, fmt);
  vsnprintf(buf, 2048, fmt, lst);
  va_end(lst);

#ifdef LOG_INFO
#ifdef USE_SYSLOG
  syslog(LOG_INFO, "%s", buf);
#endif
#endif
#ifdef USE_STDIO_LOGGING
  if (perr) { 
    if (safe_stderr) { 
      fprintf(stderr, "(%d) ", (int)getpid());
      perror(buf); 
    } else {
      if (safe_stdout) { 
	fprintf(stdout, "(%d) ", (int)getpid());
	fprintf(stdout, "%s (errno %d)\n", buf, errno);
      }
    }
  } else { 
    if (safe_stdout) { 
      fprintf(stdout, "(%d) ", (int)getpid());
      fprintf(stdout, "%s\n", buf); 
    } else {
      if (safe_stderr) {
	fprintf(stderr, "(%d) ", (int)getpid());
	fprintf(stderr, "Warning: %s\n", buf); 
      }
    }
  }
#endif
}
  
void fatal(int perr,char *fmt, ...) {
  va_list lst;
  char buf[2048];
  
  va_start(lst, fmt);
  vsnprintf(buf, 2048, fmt, lst);
  va_end(lst);

#ifdef LOG_FATAL
#ifdef USE_SYSLOG
  syslog(LOG_FATAL, "%s", buf);
#endif
#endif
#ifdef USE_STDIO_LOGGING
  if (perr) { 
    if (safe_stderr) { 
      fprintf(stderr, "(%d) ", (int)getpid());
      perror(buf); 
    } else {
      if (safe_stdout) { 
	fprintf(stdout, "(%d) ", (int)getpid());
	fprintf(stdout, "%s (errno %d)\n", buf, errno);
      }
    }
  } else { 
    if (safe_stderr) { 
      fprintf(stderr, "(%d) ", (int)getpid());
      fprintf(stderr, "%s\n", buf); 
    } else {
      if (safe_stdout) {
	fprintf(stdout, "(%d) ", (int)getpid());
	fprintf(stdout, "Warning: %s\n", buf); 
      }
    }
  }
#endif
  exit(2);
}

  void warn(int perr,char *fmt, ...) {
  va_list lst;
  char buf[2048];
  
  va_start(lst, fmt);
  vsnprintf(buf, 2048, fmt, lst);
  va_end(lst);

#ifdef LOG_WARN
#ifdef USE_SYSLOG
  syslog(LOG_WARN, "%s", buf);
#endif
#endif
#ifdef USE_STDIO_LOGGING
  if (perr) { 
    if (safe_stderr) { 
      perror(buf); 
    } else {
      if (safe_stdout) { 
	fprintf(stdout, "%s (errno %d)\n", buf, errno);
      }
    }
  } else { 
    if (safe_stdout) { 
      fprintf(stdout, "%s\n", buf); 
    } else {
      if (safe_stderr) {
	fprintf(stderr, "Warning: %s\n", buf); 
      }
    }
  }
#endif
}


int readConfigFile(FILE *f) { 
  char buf[1024];
  int ret;

  route_acl_table.occ = 0;
  route_config_table.occ = 0;

  while (!ferror(f)) {
    ret = fscanf(f, "%1024s", buf);
    


    if (ret==1) {
      if (buf[0] !='#') { 
	while (buf[0] == '[') {
	  if (!strncmp(CONFIG_FILE_START_ACLS, buf, 
		       CONFIG_FILE_START_ACLS_LENGTH)) {
	    ret = readACLData(f, &route_acl_table.entries[route_acl_table.occ],
			      route_acl_table.length-route_acl_table.occ, buf);
	    if (ret < 0) { return ret; }
	    route_acl_table.occ  += ret;
	  } else {
	    if (!strncmp(CONFIG_FILE_START_REDIR, buf, 
			 CONFIG_FILE_START_REDIR_LENGTH)) {
	      ret = readProxyData(f, &route_config_table.entries[route_config_table.occ], 
				  route_config_table.length-route_config_table.occ,
				  buf);
	      if (ret < 0) { return ret; }
	      route_config_table.occ += ret;
	    } else {
	      fatal(0, "Invalid configuration file directive %s", buf);
	    }
	  }
	}
      }
    } else {
      if (ret==EOF) {
	break;
      } 
      fatal(0, "Invalid format in configuration file");
    }
  }

  if (!feof(f)) {
      fatal(1,"Can't read ACL data");
  }

    return 0;
}

/* Returns n (=occ) if all goes well, -1 if no space, -2 for other error */
int readProxyData(FILE *f, server_conf_elem_t *arr, int sz, char *line) {
  char buf[1024];
  int ret;
  int ptr = 0;
  char *q;
  
  while (!ferror(f)) {
    ret = fscanf(f, "%1024s", buf);
    
    if (buf[0] == '[') {
      strncpy(line, buf, 1024);
      return ptr;
    }
      

    if (ret==1 && buf[0] != '#') {
      char *a, *b, *c;

      if (ptr >= sz) { return -1; }
      
 
     a = strchr(buf, ':');
     if (a==NULL) { fatal(0, "Invalid format for redirect(1): %s", buf); }
      a[0] = '\0';
      b = strchr(&a[1], '-');
     if (b==NULL) { fatal(0, "Invalid format for redirect(2): %s", buf); }
      if (b[1] != '>') { 
	fatal(0, "Invalid format for redirect(3): %s", buf);
      }
      b[0] = '\0';
      c = strchr(&b[1], ':');
      c[0] = '\0';
      if (c==NULL) { fatal(0, "Invalid format for redirect(4): %s", buf); }
      arr[ptr].client_name = host_to_ipaddr(buf);
      switch (arr[ptr].client_name) {
      case 0: fatal(0, "Host %s not found (1)", a+1); break;
      case 1: fatal(0, "Reverse lookup doesn't match for %s (DNS poisoning ?) (1)", a+1); break;
      default: /* Nothing */
      }
      arr[ptr].svr_ip = host_to_ipaddr(&b[2]);
      switch (arr[ptr].svr_ip) {
      case 0: fatal(0, "Host %s not found (2)", b+2); break;
      case 1: fatal(0, "Reverse lookup doesn't match for %s (DNS poisoning ?) (2)", b+2); break;
      default: /* Nothing */
      }
           
      arr[ptr].client_port = strtol(&a[1], &q, 10);
      if (arr[ptr].client_port < 0 || arr[ptr].client_port > 65535 || q==NULL) {
	fatal(0, "Error in client port specifier - %s", &a[1]);
      }

      arr[ptr].svr_port = strtol(&c[1], &q, 10);
      if (arr[ptr].svr_port < 0 || arr[ptr].svr_port > 65535 || q==NULL) {
	fatal(0, "Error in server port specifier - %s", &c[1]);
      }
      arr[ptr].client_port = htons(arr[ptr].client_port);
      arr[ptr].svr_port = htons(arr[ptr].svr_port);     
      ptr++;
    } else { 
      if (ret != 1) { 
	if (ret==EOF) break;
	fatal(1, "Invalid format for redirect - %s", buf);
      }
    }
  }

  if (!feof(f)) {
    fatal(1, "Can't read redirect data");
  }

  *line='\0';
  return ptr;
}


/* Returns n (=occ) if all goes well, -1 if no space in array, -2 if other error. line must be at least
 1024 characters long. */
int readACLData(FILE *f, acl_ht_entry_t *arr, int sz, char *line) {
  char buf[1024];
  int ret;
  int ptr = 0;
  /* ACL entries consist of a name:port1:port2 triple */
  
  while (!ferror(f)) {
    ret = fscanf(f, "%1024s", buf);
    
    if (ret == 1 && buf[0] != '#') {
      char *s, *p1, *p2;
 
      if (buf[0] == '[') {
	strncpy(line, buf, 1024);
	return ptr;
      }

      if (ptr >= sz) { 
	return -1;
      }
      
      s = strchr(buf, '/');
      if (s==NULL) {
	s  = buf-1;
      } else {
	*s = '\0';
      }
   

      p1 = strchr(s+1, ':');
      if (p1 != NULL) { 
	char *q;

	p2 = strchr(&p1[1], '-');
	
	*p1 = '\0';
	
	if (p2 > (p1+1)) { 
	  arr[ptr].pstart = strtol(&p1[1], &q, 10);
	  
	  if (arr[ptr].pstart < 0 || arr[ptr].pstart > 65535 || (q==NULL)) {
	    fatal(0, "Error in start port specifier - %s", buf);
	  }
	} else {
	  arr[ptr].pstart = 0;
	}
	if (p2 != NULL) { 
	  if (p2[1] == '\0') { 
	    arr[ptr].pend = 65535;
	  } else { 
	    arr[ptr].pend = strtol(&p2[1], &q, 10);
	    if (arr[ptr].pend < 0 || arr[ptr].pend > 65535 || (q==NULL)) {
	      fatal(0, "Error in end port specifier - %s", buf);
	    }
	  }
	} else {
	  arr[ptr].pend = arr[ptr].pstart;
	}
      } else {
	p2 = strchr(s+1, '-');
	if (p2 != NULL) { 
	  *p2 = '\0';
	  if (p2[1] == '\0') { 
	    arr[ptr].pend = 65535;
	  } else {
	    char *q;
	    
	    arr[ptr].pend = strtol(&p2[1], &q, 10);
	    if (arr[ptr].pend < 0 || arr[ptr].pend > 65535 || (q==NULL)) {
	      fatal(0, "Error in end port specifier - %s", &p2[1]);
	  }
	  }
	  arr[ptr].pstart = 0;
	} else {
	  arr[ptr].pstart = 0; arr[ptr].pend = 65535;
	}
      }
      
      /* Read address */
      arr[ptr].allowed = host_to_ipaddr(buf);
      if (arr[ptr].allowed == 0) {
	fatal(0, "Host %s not found", buf);
      }
      if (arr[ptr].allowed == 1) {
	fatal(0, "Help ! Reverse lookup doesn't match for %s (DNS poisoning ?)", buf);
      }
      if (s > buf) { 
	arr[ptr].mask = inet_addr(s+1);
      } else {
	arr[ptr].mask = 0xffffffff;
      }
      if (arr[ptr].pstart > arr[ptr].pend) {
	warn(0,"ACL element %d has empty range.", ptr);
      
      }
      arr[ptr].pstart = htons(arr[ptr].pstart);
      arr[ptr].pend = htons(arr[ptr].pend);

      ptr++;
    } else { 
      if (ret != 1) { 
	if (ret==EOF) {
	  break;
	}
	fatal(1, "Invalid ACL file format");
      }
    }
  }

  if (!feof(f)) {
    fatal(1,"Can't read ACL data");
  }
  
  *line = '\0';
  return ptr;
}

/* End */
