/* -------------------------------------------------------------------------
 *  idcin - Routines for encoding and decoding Id Software's .cin sequences.
 *  This uses a Huffman based compression routine achieving approximatly
 *  3:1 compression on 8-bit images sequences.
 *
 *  Written by Tim Ferguson.  1998.
 *  timf@dgs.monash.edu.au           http://www.dgs.monash.edu.au/~timf/
 *
 *  For updates and further info: http://www.dgs.monash.edu.au/~timf/bottim/
 * ------------------------------------------------------------------------- */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <ctype.h>
#include "imglib.h"

/* ------------------------------------------------------------------------- */
#define HUF_TOKENS 256

typedef struct
{
	long rate;
	long width;
	long channels;
} wavinfo_t;

typedef struct
{
	int count;
	byte used;
	int children[2];
} hnode_t;

typedef struct
{
	int bit_count;
	unsigned long bits;
} hcbits_t;

static long width, height;
static byte *huff_data = NULL;
static FILE *fp = NULL;

wavinfo_t wavinfo;
hcbits_t huff_char_bits[256][HUF_TOKENS];
hnode_t huff_nodes[256][HUF_TOKENS*2];
int num_huff_nodes[256];
static byte r_cmap[260], g_cmap[260], b_cmap[260];
static int huff_pass;

/* -------------------------------------------------------------------------
 *  Read 1/14th of a second of audio from the sequence for the current frame.
 *  If there is no audio, nothing will be read.
 *  Silent movies at the moment...  :)
 */
void read_audio(FILE *fp, int frame)
{
long width, start, end, count, empty, i;

	width = wavinfo.width * wavinfo.channels;
	start = frame * wavinfo.rate/14;
	end = (frame+1) * wavinfo.rate/14;
	count = end - start;

	for(i=0; i < count; i++) fread(&empty, 1, width, fp);
}


/* -------------------------------------------------------------------------
 *  Write 1/14th of a second of audio to the sequence for the current frame.
 *  If there is no audio, nothing will be written.
 *  Silent movies at the moment...  :)
 */
void write_audio(FILE *fp, int frame)
{
long width, start, end, count, empty, i;

	width = wavinfo.width * wavinfo.channels;
	start = frame * wavinfo.rate/14;
	end = (frame+1) * wavinfo.rate/14;
	count = end - start;

	for(i=0; i < count; i++) fwrite(&empty, 1, width, fp);
}


/* -------------------------------------------------------------------------
 *  Decodes input Huffman data using the Huffman table.
 *
 *  Input:   data = encoded data to be decoded.
 *           len = length of encoded data.
 *           image = a buffer for the decoded image data.
 */
void huff_decode(byte *data, long len, byte *image)
{
hnode_t *hnodes;
long i, dec_len;
int prev;
register byte v = 0;
register int bit_pos, node_num, dat_pos;

		/* read count */
	dec_len  = (*data++ & 0xff);
	dec_len |= (*data++ & 0xff) << 8;
	dec_len |= (*data++ & 0xff) << 16;
	dec_len |= (*data++ & 0xff) << 24;

	prev = bit_pos = dat_pos = 0;
	for(i = 0; i < dec_len; i++)
		{
		node_num = num_huff_nodes[prev];
		hnodes = huff_nodes[prev];

		while(node_num >= HUF_TOKENS)
			{
			if(!bit_pos)
				{
				if(dat_pos > len)
					{
					printf("Huffman decode error.\n");
					return;
					}
				bit_pos = 8;
				v = data[dat_pos++];
				}

			node_num = hnodes[node_num].children[v & 0x01];
			v = v >> 1;
			bit_pos--;
			}

		*image++ = prev = node_num;
		}
}


/* -------------------------------------------------------------------------
 *  Encode input image data using the Huffman dictionary.
 *
 *  Input:   data = a buffer for returned encoded data.
 *           len = length of image data.
 *           image = image data to be encoded.
 *  Return:  Number of bytes in encoded data buffer.
 */
long huff_encode(byte *data, long len, byte *image)
{
int i, out_bits, c, prev, v;
unsigned long bits;
byte *out_p;

	memset(data, 0, width * height * 2 +1000);

		/* write bits */
	out_bits = 0;
	prev = 0;
	out_p = data;

		/* write count */
	*out_p++ = len & 255;
	*out_p++ = (len >> 8) & 255;
	*out_p++ = (len >> 16) & 255;
	*out_p++ = (len >> 24) & 255;

	for(i = 0; i < len; i++)
		{
		v = image[i];

		c = huff_char_bits[prev][v].bit_count;
		bits = huff_char_bits[prev][v].bits;
		if(!c) printf("Error in huff_encode: no bits\n");
		while(c)
			{
			c--;
			if(bits & (1 << c))
				out_p[out_bits >> 3] |= 1 << (out_bits & 7);
			out_bits++;
			}

		prev = v;
		}

	out_p += (out_bits + 7) >> 3;

	return((long)(out_p - data));
}


/* -------------------------------------------------------------------------
 *  Find the lowest probability node in a Huffman table, and mark it as
 *  being assigned to a higher probability.
 *  Returns the node index of the lowest unused node, or -1 if all nodes
 *  are used.
 */
int huff_smallest_node(hnode_t *hnodes, int num_hnodes)
{
int i;
int best, best_node;

	best = 99999999;
	best_node = -1;
	for(i = 0; i < num_hnodes; i++)
		{
		if(hnodes[i].used) continue;
		if(!hnodes[i].count) continue;
		if(hnodes[i].count < best)
			{
			best = hnodes[i].count;
			best_node = i;
			}
		}

	if(best_node == -1) return -1;
	hnodes[best_node].used = 1;
	return best_node;
}


/* -------------------------------------------------------------------------
 *  Build the Huffman tree using the generated/loaded probabilities histogram.
 *
 *  On completion:
 *   huff_nodes[prev][i < HUF_TOKENS] - are the nodes at the base of the tree.
 *   huff_nodes[prev][i >= HUF_TOKENS] - are used to construct the tree.
 *   num_huff_nodes[prev] - contains the index to the root node of the tree.
 *     That is: huff_nodes[prev][num_huff_nodes[prev]] is the root node.
 */
void huff_build_tree(int prev)
{
hnode_t *node, *hnodes;
int num_hnodes, i;

	num_hnodes = HUF_TOKENS;
	hnodes = huff_nodes[prev];
	for(i = 0; i < HUF_TOKENS * 2; i++) hnodes[i].used = 0;

	while (1)
		{
		node = &hnodes[num_hnodes];		/* next free node */

			/* pick two lowest counts */
		node->children[0] = huff_smallest_node(hnodes, num_hnodes);
		if(node->children[0] == -1) break;	/* reached the root node */

		node->children[1] = huff_smallest_node(hnodes, num_hnodes);
		if(node->children[1] == -1) break;	/* reached the root node */

			/* combine nodes probability for new node */
		node->count = hnodes[node->children[0]].count + 
			hnodes[node->children[1]].count;
		num_hnodes++;
		}

	num_huff_nodes[prev] = num_hnodes - 1;
}


/* -------------------------------------------------------------------------
 *  Recursively build a variable length code dictionary from the
 *  Huffman tree.
 *  Inital call with:
 *     prev = previous pixel line to build.
 *     node_num = root node index of Huffman tree (num_huff_nodes[prev]).
 *     bits and bit_count = 0.
 */
void huff_build_chars(int prev, int node_num, unsigned bits, int bit_count)
{
hnode_t *node;

	if(node_num < HUF_TOKENS)		/* if at base of tree */
		{
		if(bit_count > 32) printf("Huffman build chars error: bit_count > 32\n");
		huff_char_bits[prev][node_num].bits = bits;
		huff_char_bits[prev][node_num].bit_count = bit_count;
		return;
		}

	node = &huff_nodes[prev][node_num];
	bits <<= 1;
	huff_build_chars(prev, node->children[0], bits, bit_count+1);
	bits |= 1;
	huff_build_chars(prev, node->children[1], bits, bit_count+1);
}


/* -------------------------------------------------------------------------
 *  Normalise the frequency histogram between 0 and 255 (one byte), save
 *  this histogram to the output file, and create the Huffman table and
 *  encoding dictionary.
 */
void huff_build(void)
{
hnode_t *hnodes;
int i, j, max, total, v;

	for(i = 0; i < 256; i++)
		{
		hnodes = huff_nodes[i];		/* select the prev row of the freq table */

			/* normalize and save the counts */
		max = 0;
		for(j = 0; j < HUF_TOKENS; j++)
			{
			if(hnodes[j].count > max)
				max = hnodes[j].count;
			}

		if(max == 0) max = 1;

		total = 0;
		for(j = 0; j < HUF_TOKENS; j++)
			{
				/* easy to overflow 32 bits here! */
			v = (hnodes[j].count * (double)255 + max-1)/max;
			if(v > 255) printf("Error in Huffman build: v > 255\n");
			hnodes[j].count = v;
			if(v) total++;
			}

		if(total == 1)
			{
				/* must have two tokens */
			if(!hnodes[0].count) hnodes[0].count = 1;
			else hnodes[1].count = 1;
			}

		huff_build_tree(i);
		huff_build_chars(i, num_huff_nodes[i], 0, 0);

		for(j = 0; j < HUF_TOKENS; j++)		/* save normalised histogram */
			putc(hnodes[j].count, fp);
		}
}

/* =========================== Public Functions =========================== */

/* ------------------------------------------------------------------------
 *  Open an existing .cin file of name `fname' and return a picture
 *  information structure.  No image data will be read, however the sequence
 *  dimensions are set.
 */
pic_info *open_idcin(char *fname)
{
pic_info *pinfo;
int i, j;

	if(fp != NULL) fclose(fp);

	if((fp = fopen(fname, "rb")) == NULL)
		{
		printf("Error opening IDCIN file `%s'\n", fname);
		return(NULL);
		}

	width = stm_read_lsb_long(fp);
	height = stm_read_lsb_long(fp);
	wavinfo.rate = stm_read_lsb_long(fp);
	wavinfo.width = stm_read_lsb_long(fp);
	wavinfo.channels = stm_read_lsb_long(fp);

	if((pinfo = create_pic(width, height, C_LOOKUP)) == NULL)
		{
		printf("Memory allocation error in reading Id cin file.\n");
		return(NULL);
		}

	pinfo->type = T_IDCIN;
	pinfo->frame_num = 0;

	if(huff_data != NULL) free(huff_data);
	if((huff_data = (byte *)malloc(width * height * 2 + 1024)) == NULL) return(NULL);
	huff_pass = -1;

	for(i = 0; i < 256; i++)
		{
		for(j = 0; j < HUF_TOKENS; j++)
			huff_nodes[i][j].count = getc(fp);
		huff_build_tree(i);
		}

	return(pinfo);
}


/* ------------------------------------------------------------------------
 *  Read a frame from a .cin sequence.  The resulting frame will be put into
 *  the passed picture info structure including the current colour map.  If
 *  the colour map changes for that frame, the colour_update flag will be set.
 *  Returns 0 when no more frames are present.
 */
int read_idcin_frame(pic_info *pinfo)
{
long command, huff_count;
int i;

	command = stm_read_lsb_long(fp);
	if(command == 2) return(0);
	if(command == 1)
		{
		for(i = 0; i < 256; i++)
			{
			r_cmap[i] = pinfo->r_cmap[i] = getc(fp);
			g_cmap[i] = pinfo->g_cmap[i] = getc(fp);
			b_cmap[i] = pinfo->b_cmap[i] = getc(fp);
			}
		pinfo->colour_update = 1;
		}
	else
		{
		memcpy(pinfo->r_cmap, r_cmap, 255);
		memcpy(pinfo->g_cmap, g_cmap, 255);
		memcpy(pinfo->b_cmap, b_cmap, 255);
		pinfo->colour_update = 0;
		}

	pinfo->width = width;
	pinfo->height = height;
	pinfo->chroma_width = pinfo->chroma_height = 0;
	pinfo->colour_format = C_LOOKUP;

	huff_count = stm_read_lsb_long(fp);
	fread(huff_data, 1, huff_count, fp);
	huff_decode(huff_data, huff_count, pinfo->pic_a);

	read_audio(fp, pinfo->frame_num);
	pinfo->frame_num++;

	return(1);
}


/* ------------------------------------------------------------------------
 *  Create a new .cin sequence.  This just opens the file and initalises
 *  variables.
 */
int create_idcin(char *fname)
{
	if(fp != NULL) fclose(fp);

	if((fp = fopen(fname, "wb")) == NULL)
		{
		printf("Error creating IDCIN file `%s'\n", fname);
		return(0);
		}

	memset(&wavinfo, 0, sizeof(wavinfo));

	huff_pass = -1;
	memset(huff_nodes, 0, sizeof(huff_nodes));
	memset(r_cmap, 0, 256);
	memset(g_cmap, 0, 256);
	memset(b_cmap, 0, 256);

	return(1);
}


/* ------------------------------------------------------------------------
 *  Write a frame to a .cin sequence.  This must be called twice for the
 *  entire sequence.  On the first run through the frames, pass must be
 *  set to 0, and on the second run through the same frames, pass must be
 *  set to 1.  The first pass generates the Huffman table, the second pass
 *  encodes the sequence to the output file.
 */
int write_idcin_frame(pic_info *pinfo, int pass)
{
static int frame = 0;
int i, prev, v;
long command, hbytes;

	if(huff_pass == -1)		/* first call to this since file was created */
		{
		stm_write_lsb_long(fp, pinfo->width);		/* write header */
		stm_write_lsb_long(fp, pinfo->height);
		stm_write_lsb_long(fp, wavinfo.rate);
		stm_write_lsb_long(fp, wavinfo.width);
		stm_write_lsb_long(fp, wavinfo.channels);

		width = pinfo->width;
		height = pinfo->height;
		if(huff_data != NULL) free(huff_data);
		if((huff_data = (byte *)malloc(width * height * 2 + 1024)) == NULL) return(0);
		}

	if(pass == 0)		/* perform frequency counts on first pass */
		{
		prev = 0;
		for(i = 0; i < pinfo->width * pinfo->height; i++)
			{
			v = pinfo->pic[i];
			huff_nodes[prev][v].count++;
			prev = v;
			}
		}
	else if(pass == 1)		/* encode and save image data on second pass */
		{
		if(huff_pass == 0)		/* if previous pass was the first pass, */
			{
			frame = 0;				/* build and save the Huffman table */
			huff_build();
			}
		command = 0;

			/* see if the palette has changed */
		for(i = 0; i < 255; i++)
			if(r_cmap[i] != pinfo->r_cmap[i] || g_cmap[i] != pinfo->g_cmap[i] ||
				b_cmap[i] != pinfo->b_cmap[i])
				{
				command = 1;
				break;
				}

		stm_write_lsb_long(fp, command);

		if(command == 1)		/* write a palette change */
			{
			printf("Palette change.\n");
			memcpy(r_cmap, pinfo->r_cmap, 255);
			memcpy(g_cmap, pinfo->g_cmap, 255);
			memcpy(b_cmap, pinfo->b_cmap, 255);
			for(i = 0; i < 256; i++)
				{
				putc(r_cmap[i], fp);
				putc(g_cmap[i], fp);
				putc(b_cmap[i], fp);
				}
			}

			/* encode and save image data */
		hbytes = huff_encode(huff_data, pinfo->width * pinfo->height, pinfo->pic);
		stm_write_lsb_long(fp, hbytes);		/* write Huffman count */
		fwrite(huff_data, 1, hbytes, fp);

			/* save some sound samples */
		write_audio(fp, frame);
		frame++;
		}

	huff_pass = pass;
	return(1);
}


/* ------------------------------------------------------------------------
 *  Close and free memory.
 */
void close_idcin(pic_info *pinfo)
{
	if(huff_data != NULL) free(huff_data);
	if(fp != NULL)
		{
		if(huff_pass != -1) stm_write_lsb_long(fp, 2);	/* end of file */
		fclose(fp);
		}

	if(pinfo != NULL) free_pic(pinfo);
	huff_data = NULL;
	fp = NULL;
}

