/* BSP2WAL by Chrisopher Wise */
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <ctype.h>

#include "bsp2wal.h"

unsigned char *output_pal;
unsigned char *trans_table;

int main(int argc, char *argv[]) {
    FILE *infile;
    dheader_t bsp_header;
    long numtex;          //number of textures in the bsp file
    long mip_offset[256]; //offsets to the mip textures in bsp file
    long count, i, j, pixel;
    char tex_name[32];
    char out_file_name[40];
    u_long width;                // width of picture
    u_long height;               // height of picture
    u_long offset1;              // offset to u_char Pix[width   * height]
    unsigned char *image_data;   //pointer to the array that holds the pixels
    int table_width, table_height;

    if (argc != 2) {
       printf("bsp2wal v1.2 by Christopher Wise\n");
       printf("Usage: bsp2wal filename.bsp\n");
       exit(0);
    }
    load_pcx_file ("trans.pcx", &trans_table, &output_pal, &table_width, &table_height);
    if (table_width * table_height != 256) {
       printf("Error: the file trans.pcx is the wrong size");
       exit(1);
    }
    infile = fopen(argv[1], "rb");
    if (infile == NULL) {
       printf ("Error opening %s: %s",argv[1],strerror(errno));
       exit(0);
    }
    fread(&bsp_header, sizeof(dheader_t), 1, infile);
    fseek(infile, bsp_header.miptex.offset , SEEK_SET);
    fread(&numtex, sizeof(long), 1, infile);
    if (numtex > 256)
       numtex = 256;
    for (count=0; count<numtex; count++) {
        fread(mip_offset+count, sizeof(long), 1, infile);
    }

    // read in the mips
    for (count=0; count<numtex; count++) {
        fseek(infile, bsp_header.miptex.offset+mip_offset[count] , SEEK_SET);
        fread(tex_name, sizeof(char), 16, infile);
        fread(&width, sizeof(long), 1, infile);
        fread(&height, sizeof(long), 1, infile);
        fread(&offset1, sizeof(long), 1, infile);
        image_data = malloc(width*height);
        if (!image_data) {
           printf("Error: could not allocate image memory\n");
           exit(1);
        }
        fseek(infile, bsp_header.miptex.offset+mip_offset[count]+offset1, SEEK_SET);
        fread(image_data, width, height, infile);
        // convert the palette
        for (pixel=0; pixel<width*height; pixel++) {
            image_data[pixel] = trans_table[image_data[pixel]];
        }

        // remove bad characters from the name
        // and put a .pcx at the end of the texture name
        j = 0;
        for (i=0; i<strlen(tex_name);i++){
            if (isalnum(tex_name[i]) || (tex_name[i] == '_') || (tex_name[i] == '+')) {
               out_file_name[j] = tex_name[i];
               j++;
            }
        }
        out_file_name[j] = '\0';
        strlwr(out_file_name);

        if ( (width & 15) || (height & 15) ) {
            strcat(out_file_name, ".pcx");
            printf("Error: texture %s not converted because it is not a multiple of 16\n", tex_name);
            printf("writing pcx file instead: %s", out_file_name);
            write_pcx_file (out_file_name, image_data, width, height, output_pal);
        }
        else {
             strcpy(tex_name, out_file_name);
             strcat(out_file_name, ".wal");
             printf("Writing texture %s as file %s\n", tex_name, out_file_name);
             write_mip_file(out_file_name, image_data, width, height, tex_name);
        }
        free(image_data);
    }
    return (0);
}



/*
====================================================
LoadPCX  based on the iD source code by John Carmack
=====================================================
*/

void load_pcx_file (char *filename, byte **pic, byte **palette, int *width, int *height) {

    byte	*raw;
    pcx_t	*pcx;
    int		x, y;
    int		len;
    int		dataByte, runLength;
    byte	*out, *pix;
    FILE    *f;
    int     pos;

	// load the file
    f = fopen(filename, "rb");
    if (!f) {
       printf("Error opening %s: %s",filename,strerror(errno));
       exit(1);
    }
    pos = ftell (f);
    fseek (f, 0, SEEK_END);
    len = ftell (f);
    fseek (f, pos, SEEK_SET);
    raw = malloc (len+1);
    if (!raw) {
       printf("could not allocate memory for pcx file\n");
       exit(1);
    }

    ((char *)raw)[len] = 0;
    fread (raw, 1, len, f);
    fclose (f);

    // parse the PCX file
	pcx = (pcx_t *)raw;
	raw = &pcx->data;

	pcx->xmin = pcx->xmin;
	pcx->ymin = pcx->ymin;
	pcx->xmax = pcx->xmax;
	pcx->ymax = pcx->ymax;
	pcx->hres = pcx->hres;
	pcx->vres = pcx->vres;
	pcx->bytes_per_line = pcx->bytes_per_line;
	pcx->palette_type = pcx->palette_type;

	if (pcx->manufacturer != 0x0a
		|| pcx->version != 5
		|| pcx->encoding != 1
		|| pcx->bits_per_pixel != 8
		|| pcx->xmax >= 640
		|| pcx->ymax >= 480)
    {
   		printf("Bad pcx file %s", filename);
        exit (1);
    }

    if (palette) {
	*palette = malloc(768);
	memcpy (*palette, (byte *)pcx + len - 768, 768);
    }

    if (width)
	*width = pcx->xmax+1;
    if (height)
	*height = pcx->ymax+1;
    if (!pic)
	return;
    out = malloc ( (pcx->ymax+1) * (pcx->xmax+1) );
    if (!out) {
   	printf("Error: couldn't allocate memory for pcx");
        exit (1);
    }

    *pic = out;

    pix = out;

    for (y=0 ; y<=pcx->ymax ; y++, pix += pcx->xmax+1) {
	for (x=0 ; x<=pcx->xmax ; ) {
       	    dataByte = *raw++;
	    if((dataByte & 0xC0) == 0xC0) {
		runLength = dataByte & 0x3F;
		dataByte = *raw++;
            }
	    else
		runLength = 1;

	while(runLength-- > 0)
            pix[x++] = dataByte;
	}
    }

    if ( raw - (byte *)pcx > len) {
		printf("PCX file %s was malformed", filename);
        exit (1);
    }
    free (pcx);
}



/* 
=====================================================
WritePCXfile from the iD source code by John Carmack
=====================================================
*/

void write_pcx_file (char *filename, byte *data, int width, int height, byte *palette) {
    int i, j, length;
    pcx_t *pcx;
    byte *pack;
    FILE    *f;

	  
    pcx = malloc (width*height*2+1000);
    if ( !pcx ) {
       printf("could not allocate memory for pcx\n");
       exit(1);
    }

    memset (pcx, 0, sizeof(*pcx));

    pcx->manufacturer = 0x0a;	// PCX id
    pcx->version = 5;			// 256 color
    pcx->encoding = 1;		// uncompressed
    pcx->bits_per_pixel = 8;		// 256 color
    pcx->xmin = 0;
    pcx->ymin = 0;
    pcx->xmax = (short)(width-1);
    pcx->ymax = (short)(height-1);
    pcx->hres = (short)width;
    pcx->vres = (short)height;
    pcx->color_planes = 1;		// chunky image
    pcx->bytes_per_line = (short)width;
    pcx->palette_type = (2);		// not a grey scale

    // pack the image
    pack = &pcx->data;
	
    for (i=0 ; i<height ; i++)	{
        for (j=0 ; j<width ; j++) {
            if ( (*data & 0xc0) != 0xc0)
                *pack++ = *data++;
            else {
                *pack++ = 0xc1;
		*pack++ = *data++;
	    }
        }
    }
			
    // write the palette
    *pack++ = 0x0c;	// palette ID byte
    for (i=0 ; i<768 ; i++)
        *pack++ = *palette++;
		
    // write output file
    length = pack - (byte *)pcx;
    f = fopen(filename, "wb");
    if (!f)
        printf("Error creating %s: %s",filename,strerror(errno));
    if (fwrite (pcx, 1, length, f) != (size_t)length)
        printf("File write failure");
    fclose(f);
    free (pcx);
} 


/*
==============
closest_colour
==============
*/

unsigned char closest_colour(int red, int green, int blue) {

    unsigned char best_colour = 0; //default to black
    unsigned char colour = 1;
    int error;
    int best_error = 1000000;

    while ((colour < 255) && (best_error > 0)) {
        error = (red   - output_pal[3*colour]  ) * (red   - output_pal[3*colour])
              + (green - output_pal[3*colour+1]) * (green - output_pal[3*colour+1])
              + (blue  - output_pal[3*colour+2]) * (blue  - output_pal[3*colour+2]);
        if (error < best_error) {
            best_error = error;
            best_colour = colour;
        }
        colour++;
    }
    return (best_colour);

}
/*
==============
write_mip_file
==============
*/

void write_mip_file(char * filename, byte *data, int width, int height, char *tex_name) {

    int n, level;
    FILE    *f;
    wal_miptex_t miptex;
    unsigned char *buffer;
    int r, g, b;
    int i, j, x, y;
    int  mip_height, mip_width;
    strcpy(miptex.name, tex_name); 
    miptex.width = width;
    miptex.height = height;
    memset(miptex.animname, 0, 1);
    miptex.flags = 0;
    miptex.contents = 0;
    miptex.contents = 0;
    miptex.value = 0;
    miptex.offsets[0] = sizeof(wal_miptex_t);
    level = 1;
    for (n=1; n<4; n++) {
        miptex.offsets[n] = miptex.offsets[n-1] + ((width/level) * (height/level));
        level *= 2;
    }
    f = fopen(filename, "wb");
    if (!f) {
        printf("Error creating %s: %s",filename,strerror(errno));
        return;
    }
    fwrite (&miptex, sizeof(wal_miptex_t), 1, f); //write header

    fwrite (data, width, height, f);  //write first mip
    level = 1;
    for (n=1; n<4; n++) {
        level *= 2;

        mip_width = width/level;
        mip_height = height/level;

        buffer = malloc(mip_width*mip_height);
        if (!buffer) {
           printf("could not allocate palette memory for mip image\n");
           exit(1);
        }


        for (x=0; x<mip_width; x++) {           //for each pixel in the mip
            for (y=0; y<mip_height; y++) {
                r=g=b=0;
                for (i=0; i<level; i++) {         //average over a square of
                    for (j=0; j<level; j++) {     //level*level from the main image
                        r += output_pal[3*data[(y*level+j)*width+(x*level)+i]];
                        g += output_pal[3*data[(y*level+j)*width+(x*level)+i]+1];
                        b += output_pal[3*data[(y*level+j)*width+(x*level)+i]+2];
                    }
                }
                r /= (level*level);
                g /= (level*level);
                b /= (level*level);
                buffer[y*mip_width+x] = closest_colour(r, g, b);
            }
        }

        fwrite (buffer, mip_width, mip_height, f);
        free(buffer);
    }
    fclose(f);
}
