#include "globals.h" 


// handles to the Game class
static jclass class_Game;
static jobject object_Game;
static jmethodID method_GameListener_init;
static jmethodID method_GameListener_shutdown;
static jmethodID method_GameListener_startLevel;
static jmethodID method_GameListener_writeGame;
static jmethodID method_GameListener_readGame;
static jmethodID method_GameListener_writeLevel;
static jmethodID method_GameListener_readLevel;
static jmethodID method_GameListener_runFrame;
static jmethodID method_GameListener_serverCommand;
static jmethodID method_GameListener_getPlayerClass;
static jmethodID method_GameListener_consoleOutput;
static jmethodID method_GameListener_playerConnect;


void Game_javaInit()
    {
    cvar_t *gameclass_cvar;
    jclass interface_GameListener;
    jmethodID method_GameListener_ctor;
    char *p;
    char buffer[128];

    interface_GameListener = (*java_env)->FindClass(java_env, "q2java/GameListener");
    if (CHECK_EXCEPTION() || !interface_GameListener)
        {
        java_error = "Can't find q2java.GameListener interface\n";
        return;
        }

    gameclass_cvar = gi.cvar("q2java_game", "q2jgame.Game", CVAR_NOSET);

    // convert classname to the strange internal format Java
    // uses, where the periods are replaced with
    // forward slashes
    strcpy(buffer, gameclass_cvar->string);
    for (p = buffer; *p; p++)
        if (*p == '.')
            *p = '/';

    class_Game = (*java_env)->FindClass(java_env, buffer);
    if (CHECK_EXCEPTION() || !class_Game)
        {
        java_error = "Couldn't find the specified Game class\n";
        return;
        }

    if (!((*java_env)->IsAssignableFrom(java_env, class_Game, interface_GameListener)))
        {
        java_error = "The specified game class doesn't implement q2java.GameListener\n";
        return;
        }

    // drop the local reference to the GameListener interface
    (*java_env)->DeleteLocalRef(java_env, interface_GameListener);

    // perhaps we should be GameListener interface to get the method ids instead of 
    // the game class....not sure about this
    method_GameListener_init = (*java_env)->GetMethodID(java_env, class_Game, "init", "()V");
    method_GameListener_shutdown = (*java_env)->GetMethodID(java_env, class_Game, "shutdown", "()V");
    method_GameListener_startLevel = (*java_env)->GetMethodID(java_env, class_Game, "startLevel", "(Ljava/lang/String;Ljava/lang/String;Ljava/lang/String;)V");
    method_GameListener_writeGame = (*java_env)->GetMethodID(java_env, class_Game, "writeGame", "(Ljava/lang/String;)V");
    method_GameListener_readGame = (*java_env)->GetMethodID(java_env, class_Game, "readGame", "(Ljava/lang/String;)V");
    method_GameListener_writeLevel = (*java_env)->GetMethodID(java_env, class_Game, "writeLevel", "(Ljava/lang/String;)V");
    method_GameListener_readLevel = (*java_env)->GetMethodID(java_env, class_Game, "readLevel", "(Ljava/lang/String;)V");
    method_GameListener_runFrame = (*java_env)->GetMethodID(java_env, class_Game, "runFrame", "()V");
    method_GameListener_serverCommand = (*java_env)->GetMethodID(java_env, class_Game, "serverCommand", "()V");
    method_GameListener_getPlayerClass = (*java_env)->GetMethodID(java_env, class_Game, "getPlayerClass", "()Ljava/lang/Class;");
    method_GameListener_consoleOutput = (*java_env)->GetMethodID(java_env, class_Game, "consoleOutput", "(Ljava/lang/String;)V");
    method_GameListener_playerConnect = (*java_env)->GetMethodID(java_env, class_Game, "playerConnect", "(Lq2java/NativeEntity;Z)V");
    method_GameListener_ctor = (*java_env)->GetMethodID(java_env, class_Game, "<init>", "()V");
    if (CHECK_EXCEPTION())
        {
        java_error = "Problem getting handle for one or more of the game methods\n";
        return;
        }

    object_Game = (*java_env)->NewObject(java_env, class_Game, method_GameListener_ctor);
    if(CHECK_EXCEPTION())
        {
        java_error = "Couldn't create instance of game object\n";
        return;
        }
    }


void Game_javaFinalize()
    {
    (*java_env)->DeleteLocalRef(java_env, class_Game);
    (*java_env)->DeleteLocalRef(java_env, object_Game);
    }


jclass Game_getPlayerClass()
    {
    jclass result = (*java_env)->CallObjectMethod(java_env, object_Game, method_GameListener_getPlayerClass);
    if (CHECK_EXCEPTION())
        return NULL;
    else
        return result;
    }

void Game_consoleOutput(const char *msg)
    {
    jstring jmsg;

    jmsg = (*java_env)->NewStringUTF(java_env, msg);
    (*java_env)->CallVoidMethod(java_env, object_Game, method_GameListener_consoleOutput, jmsg);
    CHECK_EXCEPTION();

    (*java_env)->DeleteLocalRef(java_env, jmsg);
    }

int Game_playerConnect(jobject ent, int loadgame)
    {
    (*java_env)->CallVoidMethod(java_env, object_Game, method_GameListener_playerConnect, ent, loadgame);
    return CHECK_EXCEPTION();
    }


//-------  Functions exported back to Quake2 ---------------


static void java_init(void)
    {
    Entity_arrayInit();
    (*java_env)->CallVoidMethod(java_env, object_Game, method_GameListener_init);
    CHECK_EXCEPTION();
    }


static void java_shutdown(void)
    {
    int i;
    gclient_t *client;

    (*java_env)->CallVoidMethod(java_env, object_Game, method_GameListener_shutdown);
    CHECK_EXCEPTION();

    // clear up Player java objects
    for (i = 1; i < global_maxClients + 1; i++)
        if (ge.edicts[i].inuse)
            {
            client = ge.edicts[i].client;
            if (client->listener)
                {
                (*java_env)->DeleteGlobalRef(java_env, client->listener);
                client->listener = NULL;
                }

            if (client->playerInfo)
                {
                (*java_env)->DeleteLocalRef(java_env, client->playerInfo);
                client->playerInfo = NULL;
                }
            }


    stopJava();
    gi.FreeTags (TAG_GAME);
    }


static void java_startLevel(char *mapname, char *entString, char *spawnpoint)
    {
    jstring jmapname;
    jstring jentString;
    jstring jspawnpoint;

    Entity_arrayReset();

    jmapname = (*java_env)->NewStringUTF(java_env, mapname);
    jentString = (*java_env)->NewStringUTF(java_env, entString);
    jspawnpoint = (*java_env)->NewStringUTF(java_env, spawnpoint);

    (*java_env)->CallVoidMethod(java_env, object_Game, method_GameListener_startLevel, jmapname, jentString, jspawnpoint);
    CHECK_EXCEPTION();

    (*java_env)->DeleteLocalRef(java_env, jmapname);
    (*java_env)->DeleteLocalRef(java_env, jentString);
    (*java_env)->DeleteLocalRef(java_env, jspawnpoint);

    global_frameCount = 0;
    }


static void java_writeGame(char *filename)
    {
    jstring jfilename;

    jfilename = (*java_env)->NewStringUTF(java_env, filename);
    (*java_env)->CallVoidMethod(java_env, object_Game, method_GameListener_writeGame, jfilename);
    CHECK_EXCEPTION();

    (*java_env)->DeleteLocalRef(java_env, jfilename);
    }


static void java_readGame(char *filename)
    {
    jstring jfilename;

    jfilename = (*java_env)->NewStringUTF(java_env, filename);
    (*java_env)->CallVoidMethod(java_env, object_Game, method_GameListener_readGame, jfilename);
    CHECK_EXCEPTION();

    (*java_env)->DeleteLocalRef(java_env, jfilename);
    }


static void java_writeLevel(char *filename)
    {
    jstring jfilename;

    jfilename = (*java_env)->NewStringUTF(java_env, filename);
    (*java_env)->CallVoidMethod(java_env, object_Game, method_GameListener_writeLevel, jfilename);
    CHECK_EXCEPTION();

    (*java_env)->DeleteLocalRef(java_env, jfilename);   
    }


static void java_readLevel(char *filename)
    {
    jstring jfilename;

    jfilename = (*java_env)->NewStringUTF(java_env, filename);
    (*java_env)->CallVoidMethod(java_env, object_Game, method_GameListener_readLevel, jfilename);
    CHECK_EXCEPTION();

    (*java_env)->DeleteLocalRef(java_env, jfilename);
    }

static void java_runFrame(void)
    {
    // track the running time to help with entity management
    global_frameCount++;

    (*java_env)->CallVoidMethod(java_env, object_Game, method_GameListener_runFrame);
    CHECK_EXCEPTION();
    }

static void java_serverCommand(void)
    {
    (*java_env)->CallVoidMethod(java_env, object_Game, method_GameListener_serverCommand);
    CHECK_EXCEPTION();
    }


// -------- Hook us up with the Quake II program

void Game_gameInit()
    {
    ge.Init = java_init;
    ge.Shutdown = java_shutdown;
    ge.SpawnEntities = java_startLevel;

    ge.WriteGame = java_writeGame;
    ge.ReadGame = java_readGame;
    ge.WriteLevel = java_writeLevel;
    ge.ReadLevel = java_readLevel;

    ge.RunFrame = java_runFrame;
    ge.ServerCommand = java_serverCommand;
    }
