/*
 *  GoodBot - autonomous client side Quake2 robot
 *  Copyright (C) 1998 Jens Vaasjo <jvaasjo@iname.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
*   This program is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU General Public License for more details.
*
*   You should have received a copy of the GNU General Public License
*   along with this program; if not, write to the Free Software
*   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifdef HAVE_CONFIG_H
#include<config.h>
#endif

#include<stdio.h>
#include<stdlib.h>

#ifdef HAVE_STRING_H
#include<string.h>
#endif

#ifdef HAVE_STRINGS_H
#include<strings.h>
#endif

#include<float.h>
#include<math.h>
#include"vector.h"
#include"misc.h"
#include"bsp.h"
#include"ai.h"

typedef struct
{
	char *model;
	double inv_speed;
	char *cmd;
} weapon_info_t;

enum
{
	BFG10K,
	RAILGUN,
	HYPERBLASTER,
	ROCKET_LAUNCHER,
	GRENADE_LAUNCHER,
	CHAINGUN,
	MACHINEGUN,
	SUPER_SHOTGUN,
	SHOTGUN,
	GRENADES,
	BLASTER,
	UNKNOWN_WEAPON
};

static weapon_info_t weapon_info[] =
{
	{"models/weapons/v_bfg/tris.md2",1.0/400.0,"use BFG10K"},
	{"models/weapons/v_rail/tris.md2",0.0,"use Railgun"},
	{"models/weapons/v_hyperb/tris.md2",1.0/1000.0,"use HyperBlaster"},
	{"models/weapons/v_rocket/tris.md2",1.0/550.0,"use Rocket Launcher"},
	{"models/weapons/v_launch/tris.md2",1.0/600.0,"use Grenade Launcher"},
	{"models/weapons/v_chain/tris.md2",0.0,"use Chaingun"},
	{"models/weapons/v_machn/tris.md2",0.0,"use Machinegun"},
	{"models/weapons/v_shotg2/tris.md2",0.0,"use Super Shotgun"},
	{"models/weapons/v_shotg/tris.md2",0.0,"use Shotgun"},
	{"models/weapons/v_handgr/tris.md2",1.0/400.0,"use grenades"},
	{"models/weapons/v_blast/tris.md2",1.0/1000.0,"use Blaster"},
	{NULL,0.0,NULL}
};

static void set_map_name(ai *state,char *mapname)
{
	strlower(mapname);

	bsp_free(state->map);
	state->map = bsp_load(mapname,state->path);
}

ai *ai_init(shared_data *shared,char **path)
{
	ai *state;

	state = (ai*)xcalloc(1,sizeof(ai));

	state->shared = shared;

	state->shared->smn_arg = (void*)state;
	state->shared->smn = (void (*)(void*,char*))set_map_name;

	state->team = NULL;

	state->weapon_index = 0;
	state->weapon_last = 0.0;

	state->idle_index = 0;
	state->idle_last = 0.0;

	state->path = path;
	state->map = NULL;

	return state;
}

void ai_free(ai *state)
{
	free(state->team);
	free(state);
}

static void send_console_command(ai *state,char *cmd)
{
	state->shared->scc(state->shared->scc_arg,cmd);
}

static vector aim_at(vector target,vector origin)
{
	vector angles,dif;
	float xyd;

	dif = vec_sub(target,origin);
	xyd = vec_len(vec_mk(dif.x,dif.y,0.0));

	if(xyd == 0.0) angles.x = (M_PI / 2.0) * ((dif.z < 0.0) ? 1.0 : -1.0);
	else angles.x = -1.0 * atan(dif.z / xyd);

	if(dif.x == 0.0) angles.y = (M_PI/2.0) * ((dif.y < 0.0) ? -1.0 : 1.0);
	else
	{
		angles.y = atan(dif.y / dif.x);
		if((dif.y <  0.0) && (dif.x < 0.0)) angles.y += M_PI;
		if((dif.y >= 0.0) && (dif.x < 0.0)) angles.y -= M_PI;
	}

	angles.z = 0.0;

	return angles;
}

static vector predict(edict *target,vector origin,double inv_speed,double ping)
{
	float dtime,err,olderr,guess;
	vector target_origin;

	dtime = 0.0;
	err = FLT_MAX;
	do
	{
		target_origin = vec_mul(target->velocity,dtime + 0.5 * ping);
		target_origin = vec_add(target_origin,target->origin);

		guess = vec_len(vec_sub(target_origin,origin)) * inv_speed;

		olderr = err;
		err = fabs(guess - dtime);

		dtime = guess;
	}
	while((err > 0.001) && (err < olderr));

	return target_origin;
}

static unsigned current_weapon(ai *state)
{
	unsigned i,index;
	char *model;

	i = state->shared->player_info.gun_index;
	model = state->shared->configs[CS_MODELS + i];
	if(!model) return UNKNOWN_WEAPON;

	for(index=0;weapon_info[index].model;index++)
	{
		if(!strcmp(weapon_info[index].model,model)) return index;
	}

	return UNKNOWN_WEAPON;
}

static double inv_weapon_speed(ai *state)
{
	return weapon_info[current_weapon(state)].inv_speed;
}

static vector target_predict(ai *state,edict *target,int kill)
{
	return predict(target,state->shared->player_info.origin,
		kill ? inv_weapon_speed(state) : MAX_VELOCITY,
					state->shared->ping);
}

static vector aim_at_edict(ai *state,edict *target,int kill)
{
	return aim_at(target_predict(state,target,kill),
			state->shared->player_info.origin);
}

static void set_best_weapon(ai *state)
{
	unsigned weapon;
	double now;

	now = gettime();
	if((now - state->weapon_last) <= 3.0) return;

	weapon = current_weapon(state);

	if(state->weapon_index >= weapon) state->weapon_index = 0;
	if(weapon == 0) return;

	send_console_command(state,weapon_info[state->weapon_index].cmd);

	state->weapon_index++;
	state->weapon_last = now;
}

static float dist_to(ai *state,vector origin)
{
	return vec_len(vec_sub(origin,state->shared->player_info.origin));
}

/*determined from q2source-3.14/game/m_player.h 183-202*/
static int isdead(unsigned mframe)
{
	return ((mframe >= 178) && (mframe <= 197));
}

static int is_target_valid(ai *state,edict *target)
{
	char *skin,*model;

	if(!target->valid) return 0;
	if(target->modelindex != 255) return 0;

	skin = state->shared->configs[CS_PLAYERSKINS + target->skinnum];
	model = state->shared->configs[CS_MODELS + target->modelindex];

	if(model) return 0;
	if(!skin) return 0;

	if(isdead(target->mframe)) return 0;
	if(dist_to(state,target->origin) < 0.001) return 0;
	if(state->team && strstr(skin,state->team)) return 0;

	return 1;
}

static int need_release(ai *state)
{
	if(current_weapon(state) == GRENADES) return 1;

	return 0;
}

static int is_visible(ai *state,vector origin)
{
	return bsp_isvisible(state->map,origin,
		state->shared->player_info.origin);
}

static int want_item(ai *state,char *model)
{
	(void)state;
	return model ? 1 : 0;
}

static int isbad_idle(ai *state,unsigned index)
{
	edict *target = state->shared->edicts + index;
	vector origin;
	char *model;

	if(!target->valid) return 1;

	model = state->shared->configs[CS_MODELS + target->modelindex];
	if(!want_item(state,model)) return 1;

	origin = target_predict(state,target,0);
	if(dist_to(state,origin) <= 32.0) return 1;

	return !is_visible(state,origin);
}

static int update_idle(ai *state)
{
	double now,dif;
	unsigned old;

	now = gettime();
	dif = now - state->idle_last;

	if(isbad_idle(state,state->idle_index) || (dif >= 3.0))
	{
		old = state->idle_index;

		do
		{
			state->idle_index++;
			if(state->idle_index >= MAX_EDICTS)
				state->idle_index = 0;

			if(state->idle_index == old) return 1;
		}
		while(isbad_idle(state,state->idle_index));

		state->idle_last = now;
	}

	return 0;
}

static usercmd_t no_move(void)
{
	usercmd_t update;

	update.buttons = 0;
	update.angles = vec_mk(0.0,0.0,0.0);
	update.move = vec_mk(0.0,0.0,MAX_VELOCITY);

	return update;
}

static unsigned char rand_attack(ai *state)
{
	return (rand() % 2) ? BUTTON_ATTACK : 0;
}

static unsigned char button_attack(ai *state)
{
	return (need_release(state)) ? rand_attack(state) : BUTTON_ATTACK;
}

static usercmd_t attack_move(ai *state,edict *target,int kill)
{
	usercmd_t update;

	update.buttons = kill ? button_attack(state) : 0;
	update.angles = aim_at_edict(state,target,kill);
	update.move = vec_mk(MAX_VELOCITY,0.0,0.0);

	return update;
}

static usercmd_t idle_move(ai *state)
{
	if(update_idle(state)) return no_move();

	return attack_move(state,state->shared->edicts + state->idle_index,0);
}

static edict *get_target(ai *state)
{
	edict *ent,*target;
	vector origin;
	float d,dist;
	unsigned i;

	target = NULL;
	dist = FLT_MAX;

	for(i=0;i<MAX_EDICTS;i++)
	{
		ent = state->shared->edicts + i;
		if(!is_target_valid(state,ent)) continue;

		origin = target_predict(state,ent,1);
		if(!is_visible(state,origin)) continue;

		d = dist_to(state,origin);
		if(d <= dist)
		{
			dist = d;
			target = ent;
		}
	}

	return target;
}

static int not_alive(ai *state)
{
	return (state->shared->player_info.stats[STAT_HEALTH] <= 0);
}

static usercmd_t dead_move(ai *state)
{
	usercmd_t update;

	update.buttons = rand_attack(state);
	update.angles = vec_mk(0.0,0.0,0.0);
	update.move = vec_mk(0.0,0.0,0.0);

	return update;
}

usercmd_t ai_getmove(ai *state)
{
	edict *target;

	if(not_alive(state)) return dead_move(state);

	set_best_weapon(state);

	target = get_target(state);
	if(target) return attack_move(state,target,1);

	return idle_move(state);
}

void ai_proc_console_command(ai *state,char *line)
{
	if(!strncmp(line,"gb_team",7))
	{
		line += 7;
		strip_head_space(&line);
		strip_tail_space(line);

		if(strlen(line))
		{
			free(state->team);
			state->team = strcmp(line,"null")?xstrdup(line):NULL;
		}
		else printf("\"gb_team\" is \"%s\"\n",state->team);
	}
	else fprintf(stderr,"unknown command: %s",line);
}

