/*//////////////////////////////////////////////////////////
Winwad
Copyright (C) 2000 John Gaughan

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

The full GNU General Public License can be found archived with these
source files as "gpl.txt"
*///////////////////////////////////////////////////////////

#ifndef _CWAD_
#define _CWAD_

// Windows headers
#ifdef WIN32
#include <afxwin.h>
#endif

// Standard headers
#include <vector>
using namespace std;

// Winwad headers

// CWad class
class CWad {
  private:

  // Structures to hold data
  struct CWadEntry {
    char      cName[8];
    INT32     nSize;
    INT32     nOffset;
    bool      bLoaded;
    bool      bZeroLength;
    char     *cData;
  };

  // Data members
  char                 m_cWadType[4];
  INT32                m_nDirectoryOffset;
  vector<CWadEntry *>  m_vEntries;
  char                 m_szFilename[1024];
  bool                 m_bNameChanged;

  public:

  // Default constructor
  CWad (void);

  // Copy constructor
  CWad (const CWad &wOldWad);

  // Destructor
  ~CWad (void);

  // Addition operator, concatenate two wads. "+" is not
  // defined, it would eat too much memory and speed
  // would be very slow.
  CWad operator+= (const CWad &wWad);

  // Assignment operator
  CWad operator= (CWad wWad);

  // Equality comparison
  bool operator== (const CWad &r) const;

  // Inequality comparison
  bool operator!= (const CWad &r) const;

  // Does this wad contain any entries?
  bool IsEmpty (void);

  // Push a new (blank) entry onto the end
  // of the wad
  bool AddEntry (void);

  // Inserts a new (blank) entry into the middle
  // of the wad
  bool InsertEntry (INT32 nIndex);

  // Remove an entry
  bool DeleteEntry (INT32 nIndex);

  // Clear all entries, empty wad
  bool Clear (void);

  // Get an entry's name. Buffer pointed to by szName
  // MUST be at least 9 bytes long, 8 for the name,
  // one for the null. Even if the name is short, the
  // function WILL write all nine bytes, padding with
  // nulls.
  bool GetEntryName (char *szName, INT32 nIndex) const;

  // Get an entry's size
  INT16 GetEntrySize (INT32 nIndex) const;

  // Get a *copy* of an entry's data. IMPORTANT: CWad
  // cannot check the size of your buffer. Make sure
  // you call GetEntrySize first and allocate enough
  // memory in your buffer!
  bool GetEntryData (char *cData, INT32 nIndex) const;

  // Get the type of wad. Make sure the buffer pointed to
  // by szWadType is at least 5 bytes long.
  bool GetWadType (char *szWadType) const;

  // Get the number of entries in the wad
  INT32 GetNumEntries (void) const;

  // Set an entry's name
  bool SetEntryName (char *szName, INT32 nIndex);

  // Set an entry's data
  bool SetEntryData (char *cData, INT16 nEntrySize, INT32 nIndex);

  // Set the wad type, max of four bytes
  bool SetWadType (char *szType) const;

  // Read a wad from disk. bPreCache, if set to true, will
  // load entire wad at once. Otherwise, it loads only the
  // header and directory and this object will load entries
  // on demand (make sure that file doesn't go anywhere!)
  bool LoadWad (bool bPreCache);

  // Save a wad to disk
  bool SaveWad (void);

  // Get the filename to save to/load from
  bool GetFilename (char *szFilename);

  // Set the filename to save to/load from
  bool SetFilename (char *szFilename);

  // Precache an entry from disk
  bool CacheEntry (INT32 nIndex) const;

  // Precache all entries from disk
  bool CacheAllEntries (void);

  // Import an entry from another file (raw)
  bool ImportEntry (char *szFilename, INT32 nIndex);

  // Export an entry to another file (raw)
  bool ExportEntry (char *szFilename, INT32 nIndex);
};
#endif
