#include "patch.h"

//============================================================================
// This program opens two input wads and replaces the common resources
// with those in the second, adding the new ones, keeping all those
// only in the first wad, writing the output to a third wad.
//
// The process proceeds one resource type at a time in order to
// minimize memory allocation, and to properly group resources in the
// output wad. The output wad follows the DeuTex conventions of Xn_
// markers for IWADs and XXn_ markers for PWADs.
//============================================================================

int OpenStreams(void);
int CloseStreams(void);

// Open the files, merge their textures and patches, then write the output wad

int main(int argc,char **argv)
{
	int ky,i,j,k,n;
	char *p,*q;
	char opts[256];
	int first=1;
	int err,cleared=0;
	FILE *tst;

	InitDir(&bdir);			// insure counts and dynamic pointers 0
	InitDir(&pdir);
	InitDir(&odir);
	InitTextures(&bwt);
	InitTextures(&pwt);
	InitTextures(&owt);
	InitRanges();
	InitDTex();
	InitDoomFont();

	iwadname[0]='\0';
	if ((p=getenv("IWAD"))!=NULL)
		strcpy(iwadname,p);

	nclstrs=0;
	clstrs=NULL;
	CopyEnvironmentCLArgs("MER",argc,argv);

	nclopts=nclfiles=0;
	cloptions=clfiles=NULL;
	ParseCommandLine(-1,3);	// Get any number of options, up to three parms

	if (nclfiles!=3)
	{
		printf("\nMER.EXE by Jim Flynn, Copyright 1996 v1.2 -- Freeware --\n");
		printf("Merge base and patch wads to form output wad with resources of both\n");
		printf("\nUsage: MER [options] basefile[.res|.wad] patchfile[.res|.wad] outfile[.wad]\n");
		printf("\n -Basefile is the wad to which resources are added\n");
		printf(" -Patchfile is the wad containing changed and new resources\n");
		printf(" -Outfile is the wad written with the resources of both\n");
		printf("\n Resource selection option:\n");
		printf(" +XYZ does only XYZ resources, -XYZ does all BUT XYZ resources.\n");
		printf(" L LUMPS      W WADS       T TEXTURES\n");
		printf(" E EFFECTS    M MUSICS     G GRAPHICS\n");
		printf(" S SPRITES    P PATCHES    F FLATS\n");
		printf("\n General options, use SET MER=options to set defaults:\n");
		printf(" /1 Convert FF_END to F_END, etc.   /2 Convert F_END to FF_END, etc.\n");
		printf(" /N New only - No replacements      /X extend (+TSF /N /1)\n");
		printf(" /V Verbose mode                    /I write IWAD\n");
		printf(" /Zx x=0 Replace, x=1 Base 1st, x=2 Patch 1st x=:file Read Level Order\n");
		printf("\nSee MER.TXT for details.\n");

		exit(1);
	}
	strcpy(bwadname,clfiles[0]);			// recover the filenames
	strcpy(pwadname,clfiles[1]);
	strcpy(owadname,clfiles[2]);

	cleared=0;
	for (i=0;i<nclopts;i++)
	{
		n = strspn(cloptions[i],"lwtemgspfLWTEMGSPF");
		if (n==strlen(cloptions[i]))
		{
			if (!cleared)	// if first selection option
			{
				if (cloptsense[i])	// clear all if it adds types
				{
					DoLumps=DoWads=DoTextures=DoEffects=DoMusics=0;
					DoGraphics=DoSprites=DoPatches=DoFlats=0;
				}
				else if (!cloptsense[i]) // set all if it subtracts types
				{
					DoLumps=DoWads=DoTextures=DoEffects=DoMusics=1;
					DoGraphics=DoSprites=DoPatches=DoFlats=1;
				}
				cleared=1;	// but don't do it for following selection opts
			}
			for (j=0;j<strlen(cloptions[i]);j++)
			{
				switch(tolower(cloptions[i][j]))
				{
					case 'l':
						DoLumps = cloptsense[i];
						break;
					case 'w':
						DoWads = cloptsense[i];
						break;
					case 't':
						DoTextures = cloptsense[i];
						break;
					case 'e':
						DoEffects = cloptsense[i];
						break;
					case 'm':
						DoMusics = cloptsense[i];
						break;
					case 'g':
						DoGraphics = cloptsense[i];
						break;
					case 's':
						DoSprites = cloptsense[i];
						break;
					case 'p':
						DoPatches = cloptsense[i];
						break;
					case 'f':
						DoFlats = cloptsense[i];
						break;
				}
			}
		}
		else if (tolower(cloptions[i][0])=='v')
			Verbose = 1;
		else if (tolower(cloptions[i][0])=='i')
		{
			MakeIwad = 1;
			SingleMarker = 1;
		}
		else if (tolower(cloptions[i][0])=='n')
			NewOnly = 1;
		else if (tolower(cloptions[i][0])=='x')
		{
			NewOnly = 1;
			SingleMarker = 1;
			DoLumps=DoWads=DoEffects=DoMusics=DoGraphics=DoPatches=0;
			DoTextures=DoSprites=DoFlats=1;
		}
		else if (cloptions[i][0]=='1')
			SingleMarker = 1;
 		else if (cloptions[i][0]=='2')
			DoubleMarker = 1;
		else if (tolower(cloptions[i][0]=='z'))
		{
			switch(cloptions[i][1])
			{
				case '0':
					ReplaceLevels=1;
					MapLevels=0;
					BaseFirst=0;
					break;
				case '1':
					ReplaceLevels=0;
					BaseFirst=1;
					MapLevels=0;
					break;
				case '2':
					ReplaceLevels=0;
					BaseFirst=0;
					MapLevels=0;
					break;
				case ':':
					strcpy(levmapfil,cloptions[i]+2);
					MapLevels=1;
					ReplaceLevels=0;
					BaseFirst=0;
					break;
			}
		}
		else printf("Warning: command line option: %s not understood\n",cloptions[i]);
	}

	FreeDynStr(&clstrs,&nclstrs);
	FreeDynStr(&clfiles,&nclfiles);
	FreeDynStr(&cloptions,&nclopts);

	if (Verbose)
	{
		printf("Heap at program start\n");
		heap_dump();
	}

	segs = (Segment *)realloc(segs,sizeof(Segment)*256);
	assert(segs);
	nsegs = 0;

    if ((err=OpenStreams())!=NOERROR)	// open the files
    {
        switch(err)
        {
            case BASEWONTOPEN:
                printf("Base wad: %s won't open\n",bwadname);
                exit(1);
            case PATCHWONTOPEN:
                printf("Patch wad: %s won't open\n",pwadname);
                exit(1);
            case OUTWADWONTOPEN:
                printf("Output wad: %s won't open\n",owadname);
                exit(1);
            case ERRBASEDIR:
                printf("Error reading base wad:%s directory\n",bwadname);
                exit(1);
            case ERRPATCHDIR:
                printf("Error reading patch wad:%s directory\n",pwadname);
                exit(1);
        }
    }

	printf("Patching %s with %s to %s\n",bwadname,pwadname,owadname);

	ReadPNames(&bdir,bst,&bwt);					// read base PNAMES TEXTURE1
	ReadTextures(&bdir,bst,&bwt);

	ReadPNames(&pdir,pst,&pwt);					// read patch PNAMES TEXTURE1
	ReadTextures(&pdir,pst,&pwt);

	MergePNames();								// merge the PNAMES
	MergeTextures();							// merge the TEXTURE1's
		
	WriteOutputWad();							// write base with additions

	CloseStreams();								// close 'em

	if (Verbose)
	{
		printf("Heap at program end\n");
		heap_dump();
	}
	return 0;
}

// Open base, patch, and output wads

int OpenStreams(void)
{
    char *p,cmd[128];

    // force .WAD extension on all files passed
	// if patchwad has a .RES extension, convert control file to .WAD first

	p=strrchr(bwadname,'.');
    if (p!=NULL)
	{
		if (strnicmp(p+1,"RES",3)==0)
			ReadResourceControl(bwadname,bwadname);
		p=strrchr(bwadname,'.');
		if (p!=NULL) *p='\0';
	}
	strcat(bwadname,".WAD");

	p=strrchr(pwadname,'.');
    if (p!=NULL)
	{
		if (strnicmp(p+1,"RES",3)==0)
			ReadResourceControl(pwadname,pwadname);
		p=strrchr(pwadname,'.');
		if (p!=NULL) *p='\0';
	}
	strcat(pwadname,".WAD");

    if ((p=strrchr(owadname,'.'))!=NULL) *p='\0';
    strcat(owadname,".WAD");

    // try to open the wad files
    bst = fopen(bwadname,"rb");
    if (bst==NULL)
    {
        pst=NULL;
        ost=NULL;
        return BASEWONTOPEN;
    }
	fread(&bhdr,sizeof(WadHeader),1,bst);
	if (strncmp(bhdr.wadtag,"IWAD",4)==0 || strncmp(bhdr.wadtag,"PWAD",4)==0)
	{
		if (ReadDir(&bhdr,&bdir,bst))
		{
			fclose(bst);
			bst=pst=ost=NULL;
			return ERRBASEDIR;
		}
	}
	nblr = nbsr = nbfr = nbpr = 0;
	GetRanges(&bdir,&nblr,blevRng,&nbsr,bsprRng,&nbfr,bflaRng,&nbpr,bpatRng);

    pst = fopen(pwadname,"rb");
    if (pst==NULL)
    {
        fclose(bst);
        bst=NULL;
        ost=NULL;
        return PATCHWONTOPEN;
    }
	fread(&phdr,sizeof(WadHeader),1,pst);
	if (strncmp(phdr.wadtag,"IWAD",4)==0 || strncmp(phdr.wadtag,"PWAD",4)==0)
	{
		if (ReadDir(&phdr,&pdir,pst))
		{
			fclose(pst);
			fclose(bst);
			bst=pst=ost=NULL;
			return ERRBASEDIR;
		}
	}
	nplr = npsr = npfr = nppr = 0;
	GetRanges(&pdir,&nplr,plevRng,&npsr,psprRng,&npfr,pflaRng,&nppr,ppatRng);

    ost = fopen(owadname,"wb");
    if (ost==NULL)
    {
        fclose(bst);
        bst=NULL;
        fclose(pst);
        pst=NULL;
        return OUTWADWONTOPEN;
    }
    // all files opened correctly
    return NOERROR;
}

// close all streams

int CloseStreams(void)
{
    fclose(bst);
    fclose(pst);
    fclose(ost);
	return 0;
}

