/*
	DooM MaP StaTistics, by Frans P. de Vries.

Derived from:

	DooM PostScript Maps Utility, by Frans P. de Vries.

And thus from:

	Doom Editor Utility, by Brendon Wyber and Raphaël Quinet.

	You are allowed to use any parts of this code in another program, as
	long as you give credits to the authors in the documentation and in
	the program itself.  Read the file README for more information.

	This program comes with absolutely no warranty.

	LEVELS.C - Level loading and saving routines.
*/

#include "dmmpst.h"
#include "levels.h"
#include "things.h"


/*
	the global variables
*/
char  *LevelName = NULL;   /* official name for the level */

UBCLNG NumThings = 0;      /* number of things */
TPtr      Things = NULL;   /* things data */
UBCLNG NumLineDefs = 0;    /* number of line defs */
LDPtr     LineDefs = NULL; /* line defs data */
UBCLNG NumSideDefs = 0;    /* number of side defs */
SDPtr     SideDefs = NULL; /* side defs data */
UBCLNG TotVertexes = 0;    /* total number of vertexes */
UBCLNG NumVertexes = 0;    /* number of used vertexes */
VPtr      Vertexes = NULL; /* vertex data */
/* UBCINT NumSegs = 0;        /* number of segments */
/* SEPtr     Segs = NULL;     /* list of segments */
/* UBCINT NumSSectors = 0;    /* number of subsectors */
/* SSPtr     SSectors = NULL; /* list of subsectors */
UBCLNG NumSectors = 0;     /* number of sectors */
SPtr      Sectors = NULL;  /* sectors data */

BCLNG MapMaxX;             /* maximum X value of map */
BCLNG MapMaxY;             /* maximum Y value of map */
BCLNG MapMinX;             /* minimum X value of map */
BCLNG MapMinY;             /* minimum Y value of map */

Bool UDMFmap;              /* UDMF map flag */
Bool UDMFall;              /* UDMF all skills flag */


/*
	the local function prototypes
*/
void ReadTextMap( MDirPtr);


/*
	official names for all Ultimate DOOM levels
*/
char *LevelNames1[4][9] = {
	{ "Hangar", "Nuclear Plant", "Toxin Refinery", "Command Control", "Phobos Lab",
	  "Central Processing", "Computer Station", "Phobos Anomaly", "Military Base" },
	{ "Deimos Anomaly", "Containment Area", "Refinery", "Deimos Lab", "Command Center",
	  "Halls of the Damned", "Spawning Vats", "Tower of Babel", "Fortress of Mystery" },
	{ "Hell Keep", "Slough of Despair", "Pandemonium", "House of Pain",
	  "Unholy Cathedral", "Mt. Erebus", "Limbo", "Dis", "Warrens" },
	{ "Hell Beneath", "Perfect Hatred", "Sever the Wicked", "Unruly Evil", "They Will Repent",
	  "Against Thee Wickedly", "And Hell Followed", "Unto the Cruel", "Fear" }
};

/*
	official names for all DOOM II levels
*/
char *LevelNames2[32] = {
	"Entryway", "Underhalls", "The Gantlet", "The Focus", "The Waste Tunnels", "The Crusher",
	"Dead Simple", "Tricks and Traps", "The Pit", "Refueling Base", "'O' of Destruction!",
	"The Factory", "Downtown", "The Inmost Dens", "Industrial Zone", "Suburbs", "Tenements",
	"The Courtyard", "The Citadel", "Gotcha!", "Nirvana", "The Catacombs", "Barrels o' Fun",
	"The Chasm", "Bloodfalls", "The Abandoned Mines", "Monster Condo", "The Spirit World",
	"The Living End", "Icon of Sin", "Wolfenstein", "Grosse"
};

/*
	official names for all TNT: Evilution levels
*/
char *LevelNamesT[32] = {
	"System Control", "Human BBQ", "Power Control", "Wormhole", "Hanger", "Open Season",
	"Prison", "Metal", "Stronghold", "Redemption", "Storage Facility", "Crater",
	"Nukage Processing", "Steel Works", "Dead Zone", "Deepest Reaches", "Processing Area",
	"Mill", "Shipping/Respawning", "Central Processing", "Administration Center",
	"Habitat", "Lunar Mining Project", "Quarry", "Baron's Den", "Ballistyx",
	"Mount Pain", "Heck", "River Styx", "Last Call", "Pharaoh", "Caribbean"
};

/*
	official names for all Plutonia Experiment levels
*/
char *LevelNamesP[32] = {
	"Congo", "Well of Souls", "Aztec", "Caged", "Ghost Town", "Baron's Lair",
	"Caughtyard", "Realm", "Abattoire", "Onslaught", "Hunted", "Speed", "The Crypt",
	"Genesis", "The Twilight", "The Omen", "Compound", "Neurosphere", "NME",
	"The Death Domain", "Slayer", "Impossible Mission", "Tombstone", "The Final Frontier",
	"The Temple of Darkness", "Bunker", "Anti-Christ", "The Sewers", "Odyssey of Noises",
	"The Gateway of Hell", "Cyberden", "Go 2 It"
};

/*
	official names for all Heretic: Shadow of the Serpent Riders levels
*/
char *LevelNamesH[6][9] = {
	{ "The Docks", "The Dungeons", "The Gatehouse", "The Guard Tower", "The Citadel",
	  "The Cathedral", "The Crypts", "Hell's Maw", "The Graveyard" },
	{ "The Crater", "The Lava Pits", "The River of Fire", "The Ice Grotto", "The Catacombs",
	  "The Labyrinth", "The Great Hall", "The Portals of Chaos", "The Glacier" },
	{ "The Storehouse", "The Cesspool", "The Confluence", "The Azure Fortress",
	  "The Ophidian Lair", "The Halls of Fear", "The Chasm", "D'Sparil's Keep", "The Aquifer" },
	{ "Catafalque", "Blockhouse", "Ambulatory", "Sepulcher", "Great Stair",
	  "Halls of the Apostate", "Ramparts of Perdition", "Shattered Bridge", "Mausoleum" },
	{ "Ochre Cliffs", "Rapids", "Quay", "Courtyard", "Hydratyr",
	  "Colonnade", "Foetid Manse", "Field of Judgement", "Skein of D'Sparil" },
	{ "Raven's Lair", "The Water Shrine", "American's Legacy", "", "", "", "", "", "" }
};

/*
	official names for all Hexen levels
*/
char *LevelNamesX[40] = {
	"Hub 1: Winnowing Hall", "Hub 1: Seven Portals", "Hub 1: Guardian of Ice",
	"Hub 1: Guardian of Fire", "Hub 1: Guardian of Steel", "Hub 1: Bright Crucible",
	"" /* 7 */, "Hub 2: Darkmere", "Hub 2: Caves of Circe", "Hub 2: Wastelands",
	"Hub 2: Sacred Grove", "Hub 2: Hypostyle", "Hub 2: Shadow Wood", "" /* 14 */,
	"" /* 15 */, "" /* 16 */, "" /* 17 */, "" /* 18 */, "" /* 19 */, "" /* 20 */,
	"Hub 4: Forsaken Outpost", "Hub 4: Castle of Grief", "Hub 4: Gibbet",
	"Hub 4: Effluvium", "Hub 4: Dungeons", "Hub 4: Desolate Garden",
	"Hub 3: Heresiarch's Seminary", "Hub 3: Dragon Chapel", "" /* 29 */,
	"Hub 3: Griffin Chapel", "Hub 3: Deathwind Chapel", "Hub 3: Orchard of Lamentations",
	"Hub 3: Silent Refectory", "Hub 3: Wolf Chapel", "Hub 5: Necropolis",
	"Hub 5: Zedek's Tomb", "Hub 5: Menelkir's Tomb", "Hub 5: Traductus' Tomb",
	"Hub 5: Vivarium", "Hub 5: Dark Crucible"
};

/*
	official names for all Hexen: Deathkings of the Dark Citadel levels
*/
char *LevelNamesD[30] = {
	"" /* 31 */, "" /* 32 */, "Hub 4: Transit", "Hub 4: Over N Under", "Hub 4: Deathfog",
	"Hub 4: Castle of Pain", "Hub 4: Sewer Pit", "Hub 4: The Rose", "" /* 39 */, "" /* 40 */,
	"Hub 1: Ruined Village", "Hub 1: Blight", "Hub 1: Sump", "Hub 1: Catacomb",
	"Hub 1: Badlands", "Hub 1: Brackenwood", "Hub 1: Pyre", "Hub 2: Constable's Gate",
	"Hub 2: Treasury", "Hub 2: Market Place", "Hub 2: Locus Requiescat", "Hub 2: Ordeal",
	"Hub 2: Armory", "Hub 3: Nave", "Hub 3: Chantry", "Hub 3: Abattoir", "Hub 3: Dark Watch", "Hub 3: Cloaca", "Hub 3: Ice Hold", "Hub 3: Dark Citadel"
};

/*
	official names for all Strife levels
*/
char *LevelNamesS[38] = {
	"Sanctuary", "Town", "Front Base", "Power Station", "Prison", "Sewers", "Castle",
	"Audience Chamber", "Castle: Programmer's Keep", "New Front Base", "Borderlands",
	"The Temple of the Oracle", "Catacombs", "Mines", "Fortress: Administration",
	"Fortress: Bishop's Tower", "Fortress: The Bailey", "Fortress: Stores",
	"Fortress: Security Complex", "Factory: Receiving", "Factory: Manufacturing",
	"Factory: Forge", "Order Commons", "Factory: Conversion Chapel", "Catacombs: Ruined Temple",
	"Proving Grounds", "The Lab", "Alien Ship", "Entity's Lair", "Abandoned Front Base",
	"Training Facility", "Sanctuary", "Town", "Movement Base",
	/* Veteran Edition */
	"Factory: Production", "Castle Clash", "Killing Grounds", "Ordered Chaos"
};

/*
	official names for all Doom 64 levels
*/
char *LevelNames64[40] = {
	"Staging Area", "The Terraformer", "Main Engineering", "Holding Area", "Tech Center",
	"Alpha Quadrant", "Research Lab", "Final Outpost", "Even Simpler", "The Bleeding",
	"Terror Core", "Altar of Pain", "Dark Citadel", "Eye of the Storm", "Dark Entries",
	"Blood Keep", "Watch Your Step", "Spawned Fear", "The Spiral", "Breakdown",
	"Pitfalls", "Burnt Offerings", "Unholy Temple", "No Escape", "Cat And Mouse",
	"Hardcore", "Playground", "The Absolution", "Outpost Omega", "The Lair",
	"In The Void", "Hectic", "Title",
	/* The Lost Levels */
	"Plant Ops", "Evil Sacrifice", "Cold Grounds", "Wretched Vats", "Thy Glory",
	"Final Judgement", "Panic"
};


/*
	check the input level parameters
*/
void CheckLevelParams( BCINT episode, BCINT mission)
{
	BCINT MaxEpisode, MinMission, MaxMission;

	if (GameVersion == 0x02 || GameVersion >= 0x20)
	{
		if (episode != 0)
			ProgError( "invalid game episode number (%d)", episode);
		switch (GameVersion)
		{
			case 0x02: MinMission =  0; MaxMission = 299; break; // Doom II + megawads
			case 0x20: MinMission = 32; MaxMission = 34; break; // Strife demo
			case 0x22: MinMission =  1; MaxMission = 38; break; // Strife full
			case 0x28: MinMission =  1; MaxMission = 38; break; // Strife VE
			case 0x40: MinMission =  1; MaxMission =  4; break; // Hexen demo
			case 0x42: MinMission =  1; MaxMission = 40; break; // Hexen full
			case 0x48: MinMission = 33; MaxMission = 60; break; // Hexen DDC
			case 0x82: MinMission =  1; MaxMission = 99; break; // Doom 64
			case 0x88: MinMission =  1; MaxMission = 99; break; // Doom 64: tLL
			default: ProgError( "unsupported game version (%02x)", GameVersion);
		}
		if (mission < MinMission || mission > MaxMission)
			ProgError( "invalid or missing game mission number (%d)", mission);
	}
	else
	{
		MaxEpisode = 1;
		MinMission = 1;
		MaxMission = 19;
		switch (GameVersion)
		{
			case 0x00: break; // Doom shareware
			case 0x01: MaxEpisode = 3; break; // Doom reg
			case 0x04: MaxEpisode = 6; MinMission = 0; break; // Doom ult + megawads
			case 0x10: break; // Heretic shareware
			case 0x11: MaxEpisode = 4;
			           /* check for & allow Heretic E4M1 */
			           if (episode == MaxEpisode) MaxMission = 1; break; // Heretic reg
			case 0x14: MaxEpisode = 6; MinMission = 0;
			           /* check for & allow Heretic:SotSR E6M1-3 */
			           if (episode == MaxEpisode) MaxMission = 3; break; // Heretic SSR + megawads
			default: ProgError( "unsupported game version (%02x)", GameVersion);
		}
		if (episode < 0 || episode > MaxEpisode)
			ProgError( "invalid or missing game episode number (%d)", episode);
		if (mission < MinMission || mission > MaxMission)
			ProgError( "invalid or missing game mission number (%d)", mission);
	}
}


/*
	read in the level data
*/
BCINT ReadLevelData( BCINT episode, BCINT mission)
{
	MDirPtr dir, level;
	char name[15];
	BCLNG n, m, lval;
	BCINT val, lumpver = GameVersion;
	Bool *VertexUsed;

	/* reset maximum/minimum X/Y values of map */
	MapMaxX = MapMaxY = -32767;
	MapMinX = MapMinY = 32767;

	/* find the various level information from the master directory */
	if (GameVersion == 0x02 || GameVersion >= 0x20)
	{
		if (UDMFmark != NULL)
			sprintf( name, "%s%02d", UDMFmark, mission);
		else
			sprintf( name, "MAP%02d", mission);
	}
	else
	{
		if (UDMFmark != NULL)
			sprintf( name, "%s%d", UDMFmark, mission);
		else
			sprintf( name, "E%dM%d", episode, mission);
	}
	level = FindMasterDir( MasterDir, name);
	if (!level)
		ProgError( "level data not found for %d %d", episode, mission);

	/* check for Hexen-format PWAD loaded with Doom-series / Heretic / Strife IWAD */
	if ((GameVersion & 0x40) != 0x40 && FindLevelDir( level, "BEHAVIOR") != NULL) {
		lumpver += 0x40;
		if (Verbose)
			printf( "Map lump version: 0x%02x.\n", lumpver);
	}

	/* check for Doom64-format PWAD loaded with Doom-series IWAD */
	if (GameVersion < 0x10 && FindLevelDir( level, "LEAFS") != NULL) {
		lumpver += 0x80;
		if (Verbose)
			printf( "Map lump version: 0x%02x.\n", lumpver);
	}

	if (GameVersion == 0x02 || GameVersion >= 0x20)
	{
		if ((GameVersion & 0x80) == 0x80)
			if (mission <= 40)
				LevelName = LevelNames64[mission-1]; /* Doom 64 */
			else
				LevelName = ""; /* wads */
		else if ((GameVersion & 0x40) == 0x40)
			if (GameVersion == 0x48)
				LevelName = LevelNamesD[mission-31]; /* Hexen: DotDC */
			else /* GameVersion == 0x40 or 0x42 */
				LevelName = LevelNamesX[mission-1]; /* Hexen */
		else if ((GameVersion & 0x20) == 0x20)
			LevelName = LevelNamesS[mission-1]; /* Strife */
		else /* GameVersion == 0x02 */
			if (mission <= 32)
				LevelName = LevelNames2[mission-1]; /* Doom II */
			else
				LevelName = ""; /* megawads */
	}
	else
	{
		if ((GameVersion & 0x10) == 0x10)
		{
			if (GameVersion == 0x11 && episode == 4 && mission == 1) /* Heretic E4M1 */
				LevelName = LevelNamesH[5][2]; /* == Heretic: SotSR E6M3 */
			else /* GameVersion == 0x10, 0x11, or 0x14 */
				LevelName = LevelNamesH[episode-1][mission-1];
		}
		else { /* GameVersion == 0x00, 0x01, or 0x04 */
			if (episode <= 4 && mission <= 9)
				LevelName = LevelNames1[episode-1][mission-1]; /* Ultimate Doom */
		}
	}

	/* check for UDMF data */
	UDMFmap = UDMFall = FALSE;
	dir = FindLevelDir( level, "TEXTMAP");
	if (dir)
	{
		ReadTextMap( dir);
		UDMFmap = TRUE;
		return lumpver;
	}

	/* read in the Things data */
	dir = FindLevelDir( level, "THINGS");
	if (dir)
	{
		if ((lumpver & 0x80) == 0x80) // Doom64 format
			NumThings = (UBCLNG) (dir->dir.size / sizeof(struct Thing64));
		else if ((lumpver & 0x40) == 0x40) // Hexen format
			NumThings = (UBCLNG) (dir->dir.size / sizeof(struct ThingH));
		else // Doom format
			NumThings = (UBCLNG) (dir->dir.size / sizeof(struct ThingD));
	}
	else
		NumThings = 0;
	if (NumThings > 0)
	{
		Things = (TPtr) GetMemory( NumThings * sizeof(struct Thing));
		BasicWadSeek( dir->wadfile, dir->dir.start);
		for (n = 0; n < NumThings; n++)
		{
			if ((lumpver & 0x40) == 0x40) // Hexen format
			{
				BasicWadRead( dir->wadfile, &(Things[n].tid), 2); // tid
			}
			BasicWadRead( dir->wadfile, &val, 2); // xpos
			BasicWadRead( dir->wadfile, &val, 2); // ypos
			if ((lumpver & 0x40) == 0x40 || (lumpver & 0x80) == 0x80) // Hexen/Doom64 format, skip what's not needed
			{
				BasicWadRead( dir->wadfile, &val, 2); // zpos
			}
			BasicWadRead( dir->wadfile, &(Things[n].angle), 2);
			swapint( &(Things[n].angle));
			BasicWadRead( dir->wadfile, &(Things[n].type), 2);
			swapint( &(Things[n].type));
			BasicWadRead( dir->wadfile, &(Things[n].when), 2);
			swapint( &(Things[n].when));
			if ((lumpver & 0x40) == 0x40) // Hexen format
			{
				BasicWadRead( dir->wadfile, &(Things[n].special), 1);
				BasicWadRead( dir->wadfile, &(Things[n].arg1), 1);
				BasicWadRead( dir->wadfile, &val, 2); // arg2, arg3
				BasicWadRead( dir->wadfile, &val, 2); // arg4, arg5
			}
			if ((lumpver & 0x80) == 0x80) // Doom64 format
			{
				BasicWadRead( dir->wadfile, &val, 2); // tid
			}
		}
	}

	/* get the number of Vertices */
	dir = FindLevelDir( level, "VERTEXES");
	if (dir)
		if ((lumpver & 0x80) == 0x80) // Doom64 format
			TotVertexes = (UBCLNG) (dir->dir.size / sizeof(struct Vertex64));
		else // Doom/Hexen format
			TotVertexes = (UBCLNG) (dir->dir.size / sizeof(struct Vertex));
	else
		TotVertexes = 0;
	if (TotVertexes > 0)
	{
		VertexUsed = (Bool *) GetMemory( TotVertexes * sizeof(Bool));
		for (n = 0; n < TotVertexes; n++)
			VertexUsed[n] = FALSE;
	}

	/* read in the LineDef information */
	dir = FindLevelDir( level, "LINEDEFS");
	if (dir)
	{
		if ((lumpver & 0x80) == 0x80) // Doom64 format
			NumLineDefs = (UBCLNG) (dir->dir.size / sizeof(struct LineDef64));
		else if ((lumpver & 0x40) == 0x40) // Hexen format
			NumLineDefs = (UBCLNG) (dir->dir.size / sizeof(struct LineDefH));
		else
			NumLineDefs = (UBCLNG) (dir->dir.size / sizeof(struct LineDefD));
	}
	else
		NumLineDefs = 0;
	if (NumLineDefs > 0)
	{
		LineDefs = (LDPtr) GetMemory( NumLineDefs * sizeof(struct LineDef));
		BasicWadSeek( dir->wadfile, dir->dir.start);
		for (n = 0; n < NumLineDefs; n++)
		{
			BasicWadRead( dir->wadfile, &(LineDefs[n].start), 2);
			swapint( &(LineDefs[n].start));
			VertexUsed[LineDefs[n].start] = TRUE;
			BasicWadRead( dir->wadfile, &(LineDefs[n].end), 2);
			swapint( &(LineDefs[n].end));
			VertexUsed[LineDefs[n].end] = TRUE;
			if ((lumpver & 0x80) == 0x80) // Doom64 format
			{
				BasicWadRead( dir->wadfile, &(LineDefs[n].flags), 4);
				swaplong( &(LineDefs[n].flags));
			}
			else // Doom/Hexen format
			{
				BasicWadRead( dir->wadfile, &(LineDefs[n].flags), 2);
				swapint( &(LineDefs[n].flags));
			}
			if ((lumpver & 0x40) == 0x40) // Hexen format, skip what's not needed
			{
				BasicWadRead( dir->wadfile, &(LineDefs[n].special), 1);
				BasicWadRead( dir->wadfile, &(LineDefs[n].arg1), 1);
				BasicWadRead( dir->wadfile, &val, 2); // arg2, arg3
				BasicWadRead( dir->wadfile, &val, 2); // arg4, arg5
				LineDefs[n].type = 0;
				LineDefs[n].tag = 0;
			}
			else // Doom format
			{
				BasicWadRead( dir->wadfile, &(LineDefs[n].type), 2);
				swapint( &(LineDefs[n].type));
				BasicWadRead( dir->wadfile, &(LineDefs[n].tag), 2);
				swapint( &(LineDefs[n].tag));
				if ((lumpver & 0x80) == 0x80) // Doom64 format
					LineDefs[n].type &= 0xFF;
			}
			BasicWadRead( dir->wadfile, &(LineDefs[n].sidedef1), 2);
			swapint( &(LineDefs[n].sidedef1));
			BasicWadRead( dir->wadfile, &(LineDefs[n].sidedef2), 2);
			swapint( &(LineDefs[n].sidedef2));
		}
	}

	/* read in the SideDef information */
	dir = FindLevelDir( level, "SIDEDEFS");
	if (dir)
		if ((lumpver & 0x80) == 0x80) // Doom64 format
			NumSideDefs = (UBCLNG) (dir->dir.size / sizeof(struct SideDef64));
		else
			NumSideDefs = (UBCLNG) (dir->dir.size / sizeof(struct SideDef));
	else
		NumSideDefs = 0;
	if (NumSideDefs > 0)
	{
		SideDefs = (SDPtr) GetMemory( NumSideDefs * sizeof(struct SideDef));
		BasicWadSeek( dir->wadfile, dir->dir.start);
		for (n = 0; n < NumSideDefs; n++)
		{
			BasicWadRead( dir->wadfile, &(SideDefs[n].xoff), 2);
			swapint( &(SideDefs[n].xoff));
			BasicWadRead( dir->wadfile, &(SideDefs[n].yoff), 2);
			swapint( &(SideDefs[n].yoff));
			if ((lumpver & 0x80) == 0x80) // Doom64 format
			{
				/* discard unused indices */
				BasicWadRead( dir->wadfile, &val, 8); // tex1
				BasicWadRead( dir->wadfile, &val, 8); // tex2
				BasicWadRead( dir->wadfile, &val, 8); // tex3
				SideDefs[n].tex1[0] = '\0';
				SideDefs[n].tex2[0] = '\0';
				SideDefs[n].tex3[0] = '\0';
			}
			else // Doom/Hexen format
			{
				BasicWadRead( dir->wadfile, &(SideDefs[n].tex1), 8);
				BasicWadRead( dir->wadfile, &(SideDefs[n].tex2), 8);
				BasicWadRead( dir->wadfile, &(SideDefs[n].tex3), 8);
			}
			BasicWadRead( dir->wadfile, &(SideDefs[n].sector), 2);
			swapint( &(SideDefs[n].sector));
		}
	}

	/* read in the Vertices which are all the corners of the level, but ignore the */
	/* Vertices not used in any LineDef (they usually are at the end of the list). */
	NumVertexes = 0;
	for (n = 0; n < TotVertexes; n++)
		if (VertexUsed[n])
			NumVertexes++;
	if (NumVertexes > 0)
	{
		Vertexes = (VPtr) GetMemory( NumVertexes * sizeof(struct Vertex));
		dir = FindLevelDir( level, "VERTEXES");
		BasicWadSeek( dir->wadfile, dir->dir.start);
		m = 0;
		for (n = 0; n < TotVertexes; n++)
		{
			if ((lumpver & 0x80) == 0x80) // Doom64 format
			{
				BasicWadRead( dir->wadfile, &lval, 4);
				swaplong( &lval);
				val = lval >> 16; // truncate decimal part
			}
			else
			{
				BasicWadRead( dir->wadfile, &val, 2);
				swapint( &val);
			}
			if (VertexUsed[n])
			{
				if (MapMinX > val)
					MapMinX = val;
				if (MapMaxX < val)
					MapMaxX = val;
				Vertexes[m].x = val;
			}
			if ((lumpver & 0x80) == 0x80) // Doom64 format
			{
				BasicWadRead( dir->wadfile, &lval, 4);
				swaplong( &lval);
				val = lval >> 16; // truncate decimal part
			}
			else
			{
				BasicWadRead( dir->wadfile, &val, 2);
				swapint( &val);
			}
			if (VertexUsed[n])
			{
				if (MapMinY > val)
					MapMinY = val;
				if (MapMaxY < val)
					MapMaxY = val;
				Vertexes[m].y = val;
				m++;
			}
		}
		if (m != NumVertexes)
			ProgError("inconsistency in the Vertexes data");
	}

	if (TotVertexes > 0)
	{
		/* update the Vertex numbers in the LineDefs (not really necessary, but...) */
		m = 0;
		for (n = 0; n < TotVertexes; n++)
			if (VertexUsed[n])
				VertexUsed[n] = m++;
		for (n = 0; n < NumLineDefs; n++)
		{
			LineDefs[n].start = VertexUsed[LineDefs[n].start];
			LineDefs[n].end = VertexUsed[LineDefs[n].end];
		}
		FreeMemory( VertexUsed);
		VertexUsed = NULL;
	}

	/* ignore the Segs, SSectors and Nodes */

	/* read in the Sectors information */
	dir = FindLevelDir( level, "SECTORS");
	if (dir)
		if ((lumpver & 0x80) == 0x80) // Doom64 format
			NumSectors = (UBCLNG) (dir->dir.size / sizeof(struct Sector64));
		else
			NumSectors = (UBCLNG) (dir->dir.size / sizeof(struct Sector));
	else
		NumSectors = 0;
	if (NumSectors > 0)
	{
		Sectors = (SPtr) GetMemory( NumSectors * sizeof(struct Sector));
		BasicWadSeek( dir->wadfile, dir->dir.start);
		for (n = 0; n < NumSectors; n++)
		{
			BasicWadRead( dir->wadfile, &(Sectors[n].floorh), 2);
			swapint( &(Sectors[n].floorh));
			BasicWadRead( dir->wadfile, &(Sectors[n].ceilh), 2);
			swapint( &(Sectors[n].ceilh));
			if ((lumpver & 0x80) == 0x80) // Doom64 format
			{
				/* discard unused indices */
				BasicWadRead( dir->wadfile, &val, 2); // floort
				BasicWadRead( dir->wadfile, &val, 2); // ceilt
				BasicWadRead( dir->wadfile, &val, 2); // color[0]
				BasicWadRead( dir->wadfile, &val, 2); // color[1]
				BasicWadRead( dir->wadfile, &val, 2); // color[2]
				BasicWadRead( dir->wadfile, &val, 2); // color[3]
				BasicWadRead( dir->wadfile, &val, 2); // color[4]
				Sectors[n].floort[0] = '\0';
				Sectors[n].ceilt[0] = '\0';
				Sectors[n].light = 0;
			}
			else
			{
				BasicWadRead( dir->wadfile, &(Sectors[n].floort), 8);
				BasicWadRead( dir->wadfile, &(Sectors[n].ceilt), 8);
				BasicWadRead( dir->wadfile, &(Sectors[n].light), 2);
			}
			swapint( &(Sectors[n].light));
			BasicWadRead( dir->wadfile, &(Sectors[n].special), 2);
			swapint( &(Sectors[n].special));
			BasicWadRead( dir->wadfile, &(Sectors[n].tag), 2);
			swapint( &(Sectors[n].tag));
			if ((lumpver & 0x80) == 0x80) // Doom64 format
			{
				/* supersede unused special with flags */
				BasicWadRead( dir->wadfile, &(Sectors[n].special), 2);
				swapint( &(Sectors[n].special));
			}
		}
	}

	/* ignore the Reject, BlockMap, and Hexen/Doom64/other lumps */

	return lumpver;
}


/*
	process the TEXTMAP in this level's directory section
*/
void ReadTextMap( MDirPtr textmap)
{
	int idx, trim, cursize = 0, totsize = textmap->dir.size;
	char *pos, buf[BUFSIZE];
	enum {
		ENT_THING,
		ENT_VERTX,
		ENT_LINED,
		ENT_SIDED,
		ENT_SECTR
	} entry_type = ENT_THING;

	/* read lines from lump and process by type */
	BasicWadSeek( textmap->wadfile, textmap->dir.start);
	memset( buf, 0, BUFSIZE);
	while (fgets( buf, BUFSIZE, textmap->wadfile->fileinfo) != NULL)
	{
		/* tally size before trimming */
		cursize += strlen( buf);
		/* trim leading whitespace */
		for (idx = 0; buf[idx] == ' ' || buf[idx] == '\t'; idx++)
			;
		for (trim = 0; buf[idx]; idx++)
			buf[trim++] = buf[idx];
		buf[trim] = '\0';

		/* check start/type of entry */
		if (strstr( buf, "thing") != NULL && buf == strstr( buf, "thing") && strchr( buf, '=') == NULL)
		{
			entry_type = ENT_THING;
			Things = (TPtr) ResizeMemory( Things, (NumThings+1) * sizeof(struct Thing));
			Things[NumThings].when = Things[NumThings].tid = 0;
			Things[NumThings].special = Things[NumThings].arg1 = 0;
			Things[NumThings].secret = FALSE;
			for (idx = 0; idx < MAX_UDMF; idx++)
			{
				Things[NumThings].skill[idx] = FALSE;
				Things[NumThings].class[idx] = FALSE;
			}
		}
		else if (strstr( buf, "vertex") != NULL && buf == strstr( buf, "vertex") && strchr( buf, '=') == NULL)
		{
			entry_type = ENT_VERTX;
			Vertexes = (VPtr) ResizeMemory( Vertexes, (NumVertexes+1) * sizeof(struct Vertex));
		}
		else if (strstr( buf, "linedef") != NULL && buf == strstr( buf, "linedef") && strchr( buf, '=') == NULL)
		{
			entry_type = ENT_LINED;
			LineDefs = (LDPtr) ResizeMemory( LineDefs, (NumLineDefs+1) * sizeof(struct LineDef));
			LineDefs[NumLineDefs].type = 0;
			LineDefs[NumLineDefs].special = LineDefs[NumLineDefs].arg1 = 0;
		}
		else if (strstr( buf, "sidedef") != NULL && buf == strstr( buf, "sidedef") && strchr( buf, '=') == NULL)
		{
			entry_type = ENT_SIDED;
			SideDefs = (SDPtr) ResizeMemory( SideDefs, (NumSideDefs+1) * sizeof(struct SideDef));
		}
		else if (strstr( buf, "sector") != NULL && buf == strstr( buf, "sector") && strchr( buf, '=') == NULL)
		{
			entry_type = ENT_SECTR;
			Sectors = (SPtr) ResizeMemory( Sectors, (NumSectors+1) * sizeof(struct Sector));
			Sectors[NumSectors].special = 0;
		}

		/* check end of entry */
		else if (strchr( buf, '}') != NULL && buf == strchr( buf, '}'))
		{
			switch (entry_type)
			{
				case ENT_THING: NumThings++; break;
				case ENT_VERTX: NumVertexes++; break;
				case ENT_LINED: NumLineDefs++; break;
				case ENT_SIDED: NumSideDefs++; break;
				case ENT_SECTR: NumSectors++; break;
			}
		}

		else
		{
			/* collect entry-specific fields */
			switch (entry_type)
			{
				case ENT_THING:
					if (strstr( buf, "angle") != NULL && buf == strstr( buf, "angle") && (pos = strchr( buf, '=')) != NULL)
						Things[NumThings].angle = atoi(pos+1);
					if (strstr( buf, "type") != NULL && buf == strstr( buf, "type") && (pos = strchr( buf, '=')) != NULL)
						Things[NumThings].type = atoi(pos+1);
					if (strstr( buf, "countsecret") != NULL && buf == strstr( buf, "countsecret") && strchr( buf, '=') != NULL &&
					    strstr( buf, "true") != NULL)
						Things[NumThings].secret = TRUE;

					/* use Hexen mode flags */
					if (strstr( buf, "single") != NULL && buf == strstr( buf, "single") && strchr( buf, '=') != NULL &&
					    strstr( buf, "true") != NULL)
						Things[NumThings].when |= TF_HXNSP;
					if (strstr( buf, "dm") != NULL && buf == strstr( buf, "dm") && strchr( buf, '=') != NULL &&
					    strstr( buf, "true") != NULL)
						Things[NumThings].when |= TF_HXNDM;
					if (strstr( buf, "coop") != NULL && buf == strstr( buf, "coop") && strchr( buf, '=') != NULL &&
					    strstr( buf, "true") != NULL)
						Things[NumThings].when |= TF_HXNCP;

					if (strstr( buf, "skill") != NULL && buf == strstr( buf, "skill") && strchr( buf, '=') != NULL)
					{
						idx = atoi(buf+5);
						if (idx >= 1 && idx <= MAX_UDMF && strstr( buf, "true") != NULL)
							Things[NumThings].skill[idx-1] = TRUE;
					}
					if (strstr( buf, "class") != NULL && buf == strstr( buf, "class") && strchr( buf, '=') != NULL)
					{
						idx = atoi(buf+5);
						if (idx >= 1 && idx <= MAX_UDMF && strstr( buf, "true") != NULL) {
							Things[NumThings].class[idx-1] = TRUE;
							if ((GameVersion & 0x40) == 0x40) { // Hexen games
								if (idx == 1)
									Things[NumThings].when |= TF_HFGHT;
								else if (idx == 2)
									Things[NumThings].when |= TF_HCLRC;
								else if (idx == 3)
									Things[NumThings].when |= TF_HMAGE;
							}
						}
					}

					if (strstr( buf, "id") != NULL && buf == strstr( buf, "id") && (pos = strchr( buf, '=')) != NULL)
						Things[NumThings].tid = atoi(pos+1);
					if (strstr( buf, "special") != NULL && buf == strstr( buf, "special") && (pos = strchr( buf, '=')) != NULL)
						Things[NumThings].special = atoi(pos+1);
					if (strstr( buf, "arg0") != NULL && (pos = strchr( buf, '=')) != NULL)
						Things[NumThings].arg1 = atoi(pos+1);
					break;

				case ENT_VERTX:
					if (strstr( buf, "x") != NULL && buf == strstr( buf, "x") && (pos = strchr( buf, '=')) != NULL)
					{
						Vertexes[NumVertexes].x = (BCINT) atof(pos+1);
						if (MapMinX > Vertexes[NumVertexes].x)
							MapMinX = Vertexes[NumVertexes].x;
						if (MapMaxX < Vertexes[NumVertexes].x)
							MapMaxX = Vertexes[NumVertexes].x;
					}
					if (strstr( buf, "y") != NULL && buf == strstr( buf, "y") && (pos = strchr( buf, '=')) != NULL)
					{
						Vertexes[NumVertexes].y = (BCINT) atof(pos+1);
						if (MapMinY > Vertexes[NumVertexes].y)
							MapMinY = Vertexes[NumVertexes].y;
						if (MapMaxY < Vertexes[NumVertexes].y)
							MapMaxY = Vertexes[NumVertexes].y;
					}
					break;

				case ENT_LINED:
					if (strstr( buf, "type") != NULL && buf == strstr( buf, "type") && (pos = strchr( buf, '=')) != NULL)
						LineDefs[NumLineDefs].type = atoi(pos+1);
					if (strstr( buf, "special") != NULL && buf == strstr( buf, "special") && (pos = strchr( buf, '=')) != NULL)
						LineDefs[NumLineDefs].special = atoi(pos+1);
					if (strstr( buf, "arg0") != NULL && (pos = strchr( buf, '=')) != NULL)
						LineDefs[NumLineDefs].arg1 = atoi(pos+1);
					break;

				case ENT_SIDED:
					/* no fields needed here */
					break;

				case ENT_SECTR:
					if (strstr( buf, "special") != NULL && buf == strstr( buf, "special") && (pos = strchr( buf, '=')) != NULL)
						Sectors[NumSectors].special = atoi(pos+1);
					if (strstr( buf, "secret") != NULL && buf == strstr( buf, "secret") && strchr( buf, '=') != NULL &&
					    strstr( buf, "true") != NULL)
						Sectors[NumSectors].special |= B_GENSECH;
					break;
			}
		}

		/* check for end of lump */
		if (cursize >= totsize)
			break;
	}

	TotVertexes = NumVertexes;
}


/*
	forget the level data
*/
void ForgetLevelData( void)
{
	/* forget the Things */
	if (Things)
		FreeMemory( Things);
	Things = NULL;
	NumThings = 0;

	/* forget the Vertices */
	if (Vertexes)
		FreeMemory( Vertexes);
	Vertexes = NULL;
	NumVertexes = 0;

	/* forget the LineDefs */
	if (LineDefs)
		FreeMemory( LineDefs);
	LineDefs = NULL;
	NumLineDefs = 0;

	/* forget the SideDefs */
	if (SideDefs)
		FreeMemory( SideDefs);
	SideDefs = NULL;
	NumSideDefs = 0;

	/* forget the Sectors */
	if (Sectors)
		FreeMemory( Sectors);
	Sectors = NULL;
	NumSectors = 0;

	/* forget the level name */
	LevelName = NULL;
}

/* vim:set noexpandtab tabstop=2 softtabstop=2 shiftwidth=2: */
