#ifndef __WAD__LNL_H__
#define __WAD__LNL_H__

/**********************************************************************************************************************************/
/********************************************************* Documentation **********************************************************/
/**********************************************************************************************************************************/

/*

Wad lump name list

*/

/**********************************************************************************************************************************/
/*********************************************************** Systemics ************************************************************/
/**********************************************************************************************************************************/

// Includes
#include "services.h"
#include "waddef.h"

/**********************************************************************************************************************************/
/********************************************************* Lump Name List *********************************************************/
/**********************************************************************************************************************************/

// Lump name list entry
typedef struct
{
	char name[9];												// name of lump
	size_t indx;												// index of entry within directory
	type_t type;												// type of lump (by name)
	list_t list;												// type of lump (by list)
	bool_t isempty;												// lump has zero size
	bool_t detected;											// lump type was auto-detected **not valid after call to LNLScan**
	bool_t eraseit;												// lump should be deleted (after certain types of scan)
	kind_t kind;												// Kind (sub-list type for intra-list sortation)
	char sortname[9];											// name field for sorting
	size_t sortindx;											// index field for sorting
} LNLentry_t;

// Lump name list
typedef struct
{
	size_t limit;												// maximum number of entries
	size_t count;												// current number of entries
	LNLentry_t *entries;										// entries
}
LNL_t;

/**********************************************************************************************************************************/
/************************************************** Lump Name List Maintainance ***************************************************/
/**********************************************************************************************************************************/

// Initialise a lump name list
void LNLInit (
	LNL_t **LNLp);												// lump name list to be initialised

// Deinitialise a lump name list
void LNLDone (
	LNL_t **LNLp);												// lump name list to be deinitialised

/**********************************************************************************************************************************/
/****************************************************** Lump Name List Usage ******************************************************/
/**********************************************************************************************************************************/

// Add a lump name to a lump name list at the next entry
void LNLAdd (
	LNL_t *LNL,													// lump name list to which to add lump name
	const char *name,											// name of lump
	size_t indx,												// index of entry within directory
	type_t type,												// lump name type
	list_t list,												// list that lump name belongs to
	bool_t isempty,												// lump has zero size
	bool_t detected);											// lump type was auto-detected

// Scan a pre-scanned lump name list and identify each entry
void LNLScan (
	LNL_t *LNL,													// lump name list whose contents are to be identified
	LST_t *LST,													// symbol table to be created from this list
	bool_t recognised_names,									// identify sounds/musics/dialogs/conversations by recognised names
	bool_t tolerate_multiples,									// do not treat multiple instances of structured lists as an error
	bool_t quiet_multiples,										// do not treat multiple instances of structured lists as a problem
	bool_t declassify_pnames,									// treat PNAMES as an unclassified lump (separate from TEXTURES)
	bool_t loose_headers,										// allow nonstandard map name headers (not just E\?M\? and MAP\?\?)
	bool_t quiet_headers,										// do not warn about non-empty map name headers
	game_t game);												// game for which (WAD whose lump names these are) was designed

// Normalise a lump name list
void LNLNormalise (
	LNL_t *LNL,													// lump name list to normalise
	LST_t *LST,													// symbol table resulting from the scan of that list
	bool_t remove_duplicates,									// remove all but the first of entries having the same name
	bool_t identify_voices,										// identify loose STRIFE VOICES by name (READ MANUAL BEFORE USING!)
	bool_t keep_wintex,											// do not remove _DEUTEX_ lump
	bool_t keep_platform,										// do not remove PLATFORM lump
	bool_t keep_history,										// do not remove HISTORY lump
	bool_t keep_tagdesc,										// do not remove TAGDESC lump
	bool_t keep_pcsfx,											// do not remove PC speaker sound effects
	bool_t keep_doubles,										// do not normalise double-letter list markers (e.g., PP_START)
	bool_t keep_borders,										// do not remove sub-list border markers (e.g., F1_START)
	bool_t keep_empties,										// do not remove empty structured lists
	bool_t remove_scripts,										// remove all SCRIPTS and SCRIPTnn entries
	bool_t force_removal,										// remove duplicate entries even if in different lists
	size_t *deleted);											// number of entries logically deleted by normalising

// Sort a lump name list by list (only)
void LNLSortListOnly (
	LNL_t *LNL);												// lump name list to sort

// Sort a lump name list by list and alphabetically within list
void LNLSortListAlpha (
	LNL_t *LNL,													// lump name list to sort
	bool_t sort_intrinsic,										// sort order is: list then intrinsic (use alphabetic if none)
	game_t game);												// game for which (WAD whose lump names these are) was designed

// Get a lump name from a lump name list by index number
void LNLGet (
	const LNL_t *LNL,											// lump name list from which to get lump name
	size_t which,												// which entry in the lump name list to get
	char *name,													// lump name
	size_t *indx,												// index of entry within directory
	type_t *type,												// lump name type
	list_t *list,												// list that lump name belongs to
	bool_t *eraseit);											// lump should be deleted (after certain types of scan)

/**********************************************************************************************************************************/
/********************************************************** End of File ***********************************************************/
/**********************************************************************************************************************************/

#endif // __WAD__LNL_H__
