/**********************************************************************************************************************************/
/********************************************************* Documentation **********************************************************/
/**********************************************************************************************************************************/

/*

Wad lump name list

*/

/**********************************************************************************************************************************/
/*********************************************************** Systemics ************************************************************/
/**********************************************************************************************************************************/

// Includes
#include <stdlib.h>
#include <malloc.h>
#include <string.h>
#include "services.h"
#include "waddef.h"
#include "wadlin.h"
#include "wadlst.h"
#include "wadlps.h"
#include "wadlnl.h"

/**********************************************************************************************************************************/
/************************************************** Lump Name List Maintainance ***************************************************/
/**********************************************************************************************************************************/

#define LNL_APPORTION 32
#define LNL_INCREMENT 32

// Initialise a lump name list
void LNLInit (
	LNL_t **LNLp)													// lump name list to be initialised
{
	// VARIABLES
	size_t sizeb;
	LNL_t *objptr;
	LNLentry_t *entptr;

	// PRIMARY ERROR CHECK
	if (LNLp==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to initialise lump name list with NULL argument pointer");
	}
	if (*LNLp!=NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to initialise lump name list with non-NULL argument");
	}

	// ALLOCATE MEMORY FOR THE LNL
	sizeb=sizeof(LNL_t);
	objptr=malloc(sizeb);
	if (objptr==NULL)
	{
		ErrorAbort(1,__FILE__,__LINE__,"Could not allocate %ld bytes for lump name list",sizeb);
	}
	(void) memset(objptr,0,sizeb);

	// ALLOCATE MEMORY FOR THE LNL ENTRIES
	sizeb=LNL_APPORTION*sizeof(LNLentry_t);
	entptr=malloc(sizeb);
	if (entptr==NULL)
	{
		ErrorAbort(1,__FILE__,__LINE__,"Could not allocate %ld bytes for lump name list entries",sizeb);
	}
	(void) memset(entptr,0,sizeb);

	// INITIALISE THE LNL STATIC FIELDS
	objptr->limit=LNL_APPORTION;
	objptr->count=0;
	objptr->entries=entptr;

	// RETURN RESULT
	*LNLp=objptr;
}

// Deinitialise a lump name list
void LNLDone (
	LNL_t **LNLp)													// lump name list to be deinitialised
{
	// PRIMARY ERROR CHECK
	if (LNLp==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to deinitialise lump name list with NULL argument pointer");
	}
	if (*LNLp==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to deinitialise lump name list with NULL argument");
	}
	if ((*LNLp)->entries==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to deinitialise lump name list with NULL entries");
	}
#ifdef PARANOID
	if ((*LNLp)->count>(*LNLp)->limit)
	{
		FatalAbort(1,__FILE__,__LINE__,"Lump name list is corrupt");
	}
#endif

	// DEALLOCATE MEMORY FOR THE LNL ENTRIES
	free((*LNLp)->entries);

	// DEALLOCATE MEMORY FOR THE LNL
	free(*LNLp);

	// RETURN RESULT
	*LNLp=NULL;
}

/**********************************************************************************************************************************/
/************************************************* Lump Name List Usage (Utility) *************************************************/
/**********************************************************************************************************************************/

// QSort-compatible comparison function for list-only sorting
int LNLSortListOnlyCompare (const void *x, const void *y)
{
	// VARIABLES
	LNLentry_t *p=(LNLentry_t*)x;
	LNLentry_t *q=(LNLentry_t*)y;

	//
	// No idea
	//

	// CHECK LIST EQUALITY
	if (p->list<q->list)
	{
		return -1;
	}
	if (p->list>q->list)
	{
		return 1;
	}

	//
	// Lists are equal
	//

	// CHECK INDEX EQUALITY
	if (p->sortindx<q->sortindx)
	{
		return -1;
	}
	if (p->sortindx>q->sortindx)
	{
		return 1;
	}

	//
	// Items are equal
	//

	// SOMETHING WRONG HERE; WE DELIBERATELY SET THIS UP AS
	// A STABLE SORT, SO NO TWO ITEMS SHOULD EVER BE EQUAL.
	FatalAbort(1,__FILE__,__LINE__,"Sort preprocessing failed (found equal items in stable sort)");
	return 0;
}

// QSort-compatible comparison function for list+alpha sorting
int LNLSortListAlphaCompare (const void *x, const void *y)
{
	// VARIABLES
	LNLentry_t *p=(LNLentry_t*)x;
	LNLentry_t *q=(LNLentry_t*)y;
	int namecomp;

	//
	// No idea
	//

	// CHECK LIST EQUALITY
	if (p->list<q->list)
	{
		return -1;
	}
	if (p->list>q->list)
	{
		return 1;
	}

	//
	// Lists are equal
	//

	// CHECK LIST MEMBERSHIP INDEX EQUALITY
	if (p->kind<q->kind)
	{
		return -1;
	}
	if (p->kind>q->kind)
	{
		return 1;
	}

	//
	// Kinds are equal
	//

	// CHECK NAME EQUALITY
	namecomp=strcmp(p->sortname,q->sortname);
	if (namecomp<0)
	{
		return -1;
	}
	if (namecomp>0)
	{
		return 1;
	}

	//
	// Names are equal
	//

	// CHECK INDEX EQUALITY
	if (p->sortindx<q->sortindx)
	{
		return -1;
	}
	if (p->sortindx>q->sortindx)
	{
		return 1;
	}

	//
	// Items are equal
	//

	// SOMETHING WRONG HERE; WE DELIBERATELY SET THIS UP AS
	// A STABLE SORT, SO NO TWO ITEMS SHOULD EVER BE EQUAL.
	FatalAbort(1,__FILE__,__LINE__,"Sort preprocessing failed (found equal items in stable sort)");
	return 0;
}

// Check for list nesting errors while parsing structured lists
static void CheckForListNestingErrors (
	LPS_t *LPS,													// parse stack used to parse this list
	LST_t *LST)													// symbol table created from this list
{
	// DOCUMENTATION
	/*
		Used only by IdentifyListMembershipOfEachEntry()
	*/

	// VARIABLES
	size_t stack_depth;
	list_t curr_list;
	list_t prev_list;
	char curr_lumpname[9];
	char prev_lumpname[9];
	char temp_lumpname[9];
	char curr_listname[LIST_MAXNAMELENGTH+1];
	char prev_listname[LIST_MAXNAMELENGTH+1];
	bool_t curr_corrected;
	bool_t prev_corrected;
	bool_t curr_is_main;
	bool_t prev_is_main;

	// PRIMARY ERROR CHECK
	if (LPS==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to check for nesting errors with NULL list parse stack argument");
	}
	if (LPS->entries==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to check for nesting errors with NULL list parse stack entries");
	}
#ifdef PARANOID
	if (LPS->count>LPS->limit)
	{
		FatalAbort(1,__FILE__,__LINE__,"List parse stack is corrupt");
	}
#endif
	if (LST==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to check for nesting errors with NULL list symbol table argument");
	}
	if (LST->entries==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to check for nesting errors with NULL list symbol table entries");
	}
#ifdef PARANOID
	if (LST->count>LST->limit)
	{
		FatalAbort(1,__FILE__,__LINE__,"List symbol table is corrupt");
	}
#endif

	// CHECKS FOR WHEN STACK DEPTH IS LESS THAN 2
	stack_depth=LPSGetStackDepth(LPS);
	if (stack_depth<2)
	{
		// There IS no nesting
		// to check for errors.
		return;
	}

	// GET DETAILS OF TOP TWO LIST STACK ENTRIES
	LPSGetStackTopTwoLists (
		LPS,													// list parse stack to get details from
		&curr_list,												// list on top
		&curr_corrected,										// corrected flag for lump name associated with list on top
		&prev_list,												// list below top
		&prev_corrected);										// corrected flag for lump name associated with list below top

	// GET LUMP NAMES OF THOSE ENTRIES
	LSTGetLumpName(LST,curr_list,/*start=*/TRUE,temp_lumpname);
	if (!curr_corrected)
	{
		(void) strcpy(curr_lumpname,temp_lumpname);
	}
	else
	{
		curr_lumpname[0]=temp_lumpname[0];
		curr_lumpname[1]=NUL;
		(void) strcat(curr_lumpname,temp_lumpname);
	}
	LSTGetLumpName(LST,prev_list,/*start=*/TRUE,temp_lumpname);
	if (!prev_corrected)
	{
		(void) strcpy(prev_lumpname,temp_lumpname);
	}
	else
	{
		prev_lumpname[0]=temp_lumpname[0];
		prev_lumpname[1]=NUL;
		(void) strcat(prev_lumpname,temp_lumpname);
	}

	// LET LIST NAMES OF THOSE ENTRIES
	(void) strcpy(curr_listname,LST->entries[curr_list].name);
	(void) strcpy(prev_listname,LST->entries[prev_list].name);

	// GET MAIN/SUB LIST STATUS OF THOSE ENTRIES
	curr_is_main=TRUE;
	if (
		 (curr_list==LIST_SPRITESSUB1) ||
		 (curr_list==LIST_SPRITESSUB2) ||
		 (curr_list==LIST_SPRITESSUB3) ||
		 (curr_list==LIST_PATCHESSUB1) ||
		 (curr_list==LIST_PATCHESSUB2) ||
		 (curr_list==LIST_PATCHESSUB3) ||
		 (curr_list==LIST_FLATSSUB1)   ||
		 (curr_list==LIST_FLATSSUB2)   ||
		 (curr_list==LIST_FLATSSUB3)
	   )
	{
		curr_is_main=FALSE;
	}
	prev_is_main=TRUE;
	if (
		 (prev_list==LIST_SPRITESSUB1) ||
		 (prev_list==LIST_SPRITESSUB2) ||
		 (prev_list==LIST_SPRITESSUB3) ||
		 (prev_list==LIST_PATCHESSUB1) ||
		 (prev_list==LIST_PATCHESSUB2) ||
		 (prev_list==LIST_PATCHESSUB3) ||
		 (prev_list==LIST_FLATSSUB1)   ||
		 (prev_list==LIST_FLATSSUB2)   ||
		 (prev_list==LIST_FLATSSUB3)
	   )
	{
		prev_is_main=FALSE;
	}

	// CHECKS FOR WHEN STACK DEPTH IS 2
	if (stack_depth==2)
	{
		if (!curr_is_main && !prev_is_main)
		{
			// sub-in-sub: always invalid
			EventState(VERBOSITY_ERRORS,"Parse error at entry %s (nesting of one sub-list inside another)",curr_lumpname);
			if (
			     ( (prev_list==LIST_SPRITESSUB1) && (curr_list==LIST_SPRITESSUB2) ) ||
			     ( (prev_list==LIST_SPRITESSUB2) && (curr_list==LIST_SPRITESSUB3) )
			   )
			{
				EventState(VERBOSITY_ERRORS,"File attempts to nest list [%s] inside list [%s]",curr_listname,prev_listname);
				ErrorAbort(1,__FILE__,__LINE__,NULL);
			}
			if (
			     ( (prev_list==LIST_PATCHESSUB1) && (curr_list==LIST_PATCHESSUB2) ) ||
			     ( (prev_list==LIST_PATCHESSUB2) && (curr_list==LIST_PATCHESSUB3) )
			   )
			{
				EventState(VERBOSITY_ERRORS,"File attempts to nest list [%s] inside list [%s]",curr_listname,prev_listname);
				ErrorAbort(1,__FILE__,__LINE__,NULL);
			}
			if (
			     ( (prev_list==LIST_FLATSSUB1) && (curr_list==LIST_FLATSSUB2) ) ||
			     ( (prev_list==LIST_FLATSSUB2) && (curr_list==LIST_FLATSSUB3) )
			   )
			{
				EventState(VERBOSITY_ERRORS,"File attempts to nest list [%s] inside list [%s]",curr_listname,prev_listname);
				ErrorAbort(1,__FILE__,__LINE__,NULL);
			}
			EventState(VERBOSITY_ERRORS,"The COMBINATION is ALSO invalid (sub lists do not match)",curr_lumpname);
			EventState(VERBOSITY_ERRORS,"File attempts to nest list [%s] inside list [%s]",curr_listname,prev_listname);
			ErrorAbort(1,__FILE__,__LINE__,NULL);
		}
		if (!curr_is_main && prev_is_main)
		{
			// sub-in-main: depends on the lists
			if (
			     (prev_list==LIST_SPRITES)
			    &&
			     (
			       (curr_list==LIST_SPRITESSUB1) ||
			       (curr_list==LIST_SPRITESSUB2) ||
			       (curr_list==LIST_SPRITESSUB3)
			     )
			   )
			{	// ok
				return;
			}
			if (
			     (prev_list==LIST_PATCHES)
			    &&
			     (
			       (curr_list==LIST_PATCHESSUB1) ||
			       (curr_list==LIST_PATCHESSUB2) ||
			       (curr_list==LIST_PATCHESSUB3)
			     )
			   )
			{	// ok
				return;
			}
			if (
			     (prev_list==LIST_FLATS)
			    &&
			     (
			       (curr_list==LIST_FLATSSUB1) ||
			       (curr_list==LIST_FLATSSUB2) ||
			       (curr_list==LIST_FLATSSUB3)
			     )
			   )
			{	// ok
				return;
			}
			EventState(VERBOSITY_ERRORS,"Parse error at entry %s (main list does not match sub-list)",curr_lumpname);
			EventState(VERBOSITY_ERRORS,"File attempts to nest list [%s] inside list [%s]",curr_listname,prev_listname);
			ErrorAbort(1,__FILE__,__LINE__,NULL);
		}
		if (curr_is_main && !prev_is_main)
		{
			// main-in-sub: always invalid
			EventState(VERBOSITY_ERRORS,"Parse error at entry %s (nesting of main list inside sub-list)",curr_lumpname);
			EventState(VERBOSITY_ERRORS,"File attempts to nest list [%s] inside list [%s]",curr_listname,prev_listname);
			ErrorAbort(1,__FILE__,__LINE__,NULL);
		}
		if (curr_is_main && prev_is_main)
		{
			// main-in-main: always invalid
			EventState(VERBOSITY_ERRORS,"Parse error at entry %s (nesting of main list inside main list)",curr_lumpname);
			EventState(VERBOSITY_ERRORS,"File attempts to nest list [%s] inside list [%s]",curr_listname,prev_listname);
			ErrorAbort(1,__FILE__,__LINE__,NULL);
		}
	}

	// CHECKS FOR WHEN STACK DEPTH IS GREATER THAN 2
	if (stack_depth>2)
	{
		// Invalid DEPTH of nesting
		EventState(VERBOSITY_ERRORS,"Parse error at entry %s (list nesting depth)",curr_lumpname);
		EventState(VERBOSITY_ERRORS,"File attempts to nest more than two lists (of ANY combination)");
		// Invalid COMBINATIONS of nesting
		if (!curr_is_main && !prev_is_main)
		{
			// sub-in-sub: always invalid
			EventState(VERBOSITY_ERRORS,"This case takes the form of one sub-list inside another");
			if (
			     ( (prev_list==LIST_SPRITESSUB1) && (curr_list==LIST_SPRITESSUB2) ) ||
			     ( (prev_list==LIST_SPRITESSUB2) && (curr_list==LIST_SPRITESSUB3) )
			   )
			{
				EventState(VERBOSITY_ERRORS,"File attempts to nest list [%s] inside list [%s]",curr_listname,prev_listname);
				ErrorAbort(1,__FILE__,__LINE__,NULL);
			}
			if (
			     ( (prev_list==LIST_PATCHESSUB1) && (curr_list==LIST_PATCHESSUB2) ) ||
			     ( (prev_list==LIST_PATCHESSUB2) && (curr_list==LIST_PATCHESSUB3) )
			   )
			{
				EventState(VERBOSITY_ERRORS,"File attempts to nest list [%s] inside list [%s]",curr_listname,prev_listname);
				ErrorAbort(1,__FILE__,__LINE__,NULL);
			}
			if (
			     ( (prev_list==LIST_FLATSSUB1) && (curr_list==LIST_FLATSSUB2) ) ||
			     ( (prev_list==LIST_FLATSSUB2) && (curr_list==LIST_FLATSSUB3) )
			   )
			{
				EventState(VERBOSITY_ERRORS,"File attempts to nest list [%s] inside list [%s]",curr_listname,prev_listname);
				ErrorAbort(1,__FILE__,__LINE__,NULL);
			}
			EventState(VERBOSITY_ERRORS,"The COMBINATION is ALSO invalid (sub lists do not match)",curr_lumpname);
			EventState(VERBOSITY_ERRORS,"File attempts to nest list [%s] inside list [%s]",curr_listname,prev_listname);
			ErrorAbort(1,__FILE__,__LINE__,NULL);
		}
		if (!curr_is_main && prev_is_main)
		{
			// sub-in-main: depends on lists
			EventState(VERBOSITY_ERRORS,"This case takes the form of a sub-list inside a main list");
			if (
			     (prev_list==LIST_SPRITES)
			    &&
			     (
			       (curr_list==LIST_SPRITESSUB1) ||
			       (curr_list==LIST_SPRITESSUB2) ||
			       (curr_list==LIST_SPRITESSUB3)
			     )
			   )
			{
				EventState(VERBOSITY_ERRORS,"File attempts to nest list [%s] inside list [%s]",curr_listname,prev_listname);
				ErrorAbort(1,__FILE__,__LINE__,NULL);
			}
			if (
			     (prev_list==LIST_PATCHES)
			    &&
			     (
			       (curr_list==LIST_PATCHESSUB1) ||
			       (curr_list==LIST_PATCHESSUB2) ||
			       (curr_list==LIST_PATCHESSUB3)
			     )
			   )
			{
				EventState(VERBOSITY_ERRORS,"File attempts to nest list [%s] inside list [%s]",curr_listname,prev_listname);
				ErrorAbort(1,__FILE__,__LINE__,NULL);
			}
			if (
			     (prev_list==LIST_FLATS)
			    &&
			     (
			       (curr_list==LIST_FLATSSUB1) ||
			       (curr_list==LIST_FLATSSUB2) ||
			       (curr_list==LIST_FLATSSUB3)
			     )
			   )
			{
				EventState(VERBOSITY_ERRORS,"File attempts to nest list [%s] inside list [%s]",curr_listname,prev_listname);
				ErrorAbort(1,__FILE__,__LINE__,NULL);
			}
			EventState(VERBOSITY_ERRORS,"The COMBINATION is ALSO invalid (main list does not match sub list)",curr_lumpname);
			EventState(VERBOSITY_ERRORS,"File attempts to nest list [%s] inside list [%s]",curr_listname,prev_listname);
			ErrorAbort(1,__FILE__,__LINE__,NULL);
		}
		if (curr_is_main && !prev_is_main)
		{
			// main-in-sub: always invalid
			EventState(VERBOSITY_ERRORS,"The COMBINATION is ALSO invalid (main list inside sub-list)",curr_lumpname);
			EventState(VERBOSITY_ERRORS,"File attempts to nest list [%s] inside list [%s]",curr_listname,prev_listname);
			ErrorAbort(1,__FILE__,__LINE__,NULL);
		}
		if (curr_is_main && prev_is_main)
		{
			// main-in-main: always invalid
			EventState(VERBOSITY_ERRORS,"The COMBINATION is ALSO invalid (main list inside main list)",curr_lumpname);
			EventState(VERBOSITY_ERRORS,"File attempts to nest list [%s] inside list [%s]",curr_listname,prev_listname);
			ErrorAbort(1,__FILE__,__LINE__,NULL);
		}
	}
}

// Detect nonstandard map names by map sequence structure
static void DetectNonStandardMapNames (
	LNL_t *LNL)													// lump name list to detect nonstandard map names in
{
	// DOCUMENTATION
	/*
		Used only by LNLScan()

		The algorithm is too complicated to detail here
		and is documented in the manual, in Appendix 1.
	*/

	// VARIABLES
	type_t previous,current;
	size_t i;

	// PRIMARY ERROR CHECK
	if (LNL==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to detect nonstandard map names in lump name list with NULL argument");
	}
	if (LNL->entries==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to detect nonstandard map names in lump name list with NULL entries");
	}
#ifdef PARANOID
	if (LNL->count>LNL->limit)
	{
		FatalAbort(1,__FILE__,__LINE__,"Lump name list is corrupt");
	}
#endif

	// HANDLE ANY NONSTANDARD NORMAL MAP NAMES
	previous=TYPE_UNKNOWN;
	for (i=0;i<LNL->count;i++)
	{
		current=LNL->entries[i].type;
		if ((i==0) && (current==TYPE_MAPDATA))
		{
			ErrorAbort(1,__FILE__,__LINE__,"Map data %s at start of file has no map header",LNL->entries[i].name);
		}
		if (i>0)
		{
			if ( (current==TYPE_MAPDATA) && (previous!=TYPE_MAPDATA) && (previous!=TYPE_MAPNAME))
			{
				// We hit either a WAD error or else a nonstandard map name.
				// The latter can happen with SkullTag (e.g., D2ST1), EDGE
				// (e.g., QD01 in QDOOM) or ZDOOM (basically as SkullTag).
				//
				if (!LNL->entries[i-1].isempty)
				{
					ErrorAbort(1,__FILE__,__LINE__,"Entry \"%s\" has non-zero length where map header expected",LNL->entries[i-1].name);
				}
				LNL->entries[i-1].type=TYPE_MAPNAME;
			}
		}
		// store the (now) previous entry name type
		previous=current;
	}

	// HANDLE ANY NONSTANDARD OPENGL MAP NAMES
	previous=TYPE_UNKNOWN;
	for (i=0;i<LNL->count;i++)
	{
		current=LNL->entries[i].type;
		if ((i==0) && (current==TYPE_GLMAPDATA))
		{
			ErrorAbort(1,__FILE__,__LINE__,"OpenGL map data %s at start of file has no OpenGL map header",LNL->entries[i].name);
		}
		if (i>0)
		{
			if ( (current==TYPE_GLMAPDATA) && (previous!=TYPE_GLMAPDATA) && (previous!=TYPE_GLMAPNAME))
			{
				// We hit either a WAD error or else a nonstandard OpenGL
				// map name. The latter could happen with SkullTag if it
				// ever acuires OpenGL support (e.g., GL_D2ST1), EDGE
				// (e.g., QD01 in QDOOM) or ZDOOM (basically as SkullTag).
				//
				if (!LNL->entries[i-1].isempty)
				{
					ErrorAbort(1,__FILE__,__LINE__,"Entry \"%s\" has non-zero length where OpenGL map header expected",LNL->entries[i-1].name);
				}
				LNL->entries[i-1].type=TYPE_GLMAPNAME;
			}
		}
		// store the (now) previous entry name type
		previous=current;
	}
}

// Assign OpenGL map items to parent maps where applicable
static void AssignOpenGLMapItemsToParentMaps (
	LNL_t *LNL)													// lump name list to assign OpenGL map items to parent maps in
{
	// DOCUMENTATION
	/*
		Used only by LNLScan()
	*/

	// VARIABLES
	type_t previous,current;
	char last_normal_map_name[9];
	char tempname[6];
	size_t i,j;

	// PRIMARY ERROR CHECK
	if (LNL==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to assign OpenGL map items to parent maps in lump name list with NULL argument");
	}
	if (LNL->entries==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to assign OpenGL map items to parent maps in lump name list with NULL entries");
	}
#ifdef PARANOID
	if (LNL->count>LNL->limit)
	{
		FatalAbort(1,__FILE__,__LINE__,"Lump name list is corrupt");
	}
#endif

	// PERFORM OPERATION
	previous=TYPE_UNKNOWN;
	(void) strcpy(last_normal_map_name,"");
	for (i=0;i<LNL->count;i++)
	{
		current=LNL->entries[i].type;
		if (current==TYPE_MAPNAME)
		{
			(void) strcpy(last_normal_map_name,LNL->entries[i].name);
		}
		if ( (i>0) && (current==TYPE_GLMAPNAME) && (previous==TYPE_MAPDATA) )
		{
			// We hit an OpenGL map name header right after a
			// normal map data entry. If the GL map name is not
			// the same as the main map name then it is an error.
			// For that matter, if the normal map name is more
			// than five characters in length then it is an error
			// as a match would be impossible anyway and it would
			// violate the DOOM family games OpenGL standard.
			if (strlen(last_normal_map_name)>5)
			{
				ErrorAbort(1,__FILE__,__LINE__,"Map name \"%s\" preceding OpenGL map name \"%s\" is more than 5 characters in length",
											   last_normal_map_name,LNL->entries[i].name);
			}
			(void) strcpy(tempname,&LNL->entries[i].name[3]);
			if (strcmp(last_normal_map_name,tempname)!=0)
			{
				ErrorAbort(1,__FILE__,__LINE__,"OpenGL map name \"%s\" does not match previous normal map name \"%s\"",
											   LNL->entries[i].name,last_normal_map_name);
			}
			// Now recode the OpenGL items to be TYPE_MAPDATA
			LNL->entries[i].type=TYPE_MAPDATA;
			for (j=i+1;(j<LNL->count) && (LNL->entries[j].type==TYPE_GLMAPDATA);j++)
			{
				LNL->entries[j].type=TYPE_MAPDATA;
			}
		}
		// store the (now) previous entry name type
		previous=current;
	}
}

// Log markers with invalid non-zero sizes where applicable
static void LogMarkersWithInvalidNonZeroSizes (
	LNL_t *LNL,													// lump name list to log markers with invalid non-zero sizes in
	bool_t quiet_headers)										// do not warn about non-empty map name headers
{
	// DOCUMENTATION
	/*
		Used only by LNLScan()
	*/

	// VARIABLES
	size_t i;

	// PRIMARY ERROR CHECK
	if (LNL==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to log markers with invalid non-zero sizes in lump name list with NULL argument");
	}
	if (LNL->entries==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to log markers with invalid non-zero sizes in lump name list with NULL entries");
	}
#ifdef PARANOID
	if (LNL->count>LNL->limit)
	{
		FatalAbort(1,__FILE__,__LINE__,"Lump name list is corrupt");
	}
#endif

	// PERFORM OPERATION
	for (i=0;i<LNL->count;i++)
	{
		if (!LNL->entries[i].isempty)
		{
			// VARIABLES
			type_t type=LNL->entries[i].type;
			// CHECK FOR NON-ZERO-SIZED OPENGL MAP NAME HEADERS
			if (type==TYPE_GLMAPNAME)
			{
				// This is always a cause for concern
				EventState(VERBOSITY_WARNINGS,"OpenGL map name header %s (entry %ld) has non-zero size",LNL->entries[i].name,i);
				continue;
			}
			// CHECK FOR NON-ZERO-SIZED MAP NAME HEADERS
			if ( !quiet_headers && (type==TYPE_MAPNAME) )
			{
				EventState(VERBOSITY_WARNINGS,"Map name header %s (entry %ld) has non-zero size",LNL->entries[i].name,i);
				continue;
			}
			// CHECK FOR NON-ZERO-SIZED LIST MARKERS
			if (
		        (type==TYPE_GOODSTART)        ||				// S_START,P_START,F_START,C_START,A_START,TX_START,V_START
		        (type==TYPE_POORSTART)        ||				// SS_START,PP_START,FF_START
		        (type==TYPE_SUBSTART)         ||				// S[1-3]_START,P[1-3]_START,F[1-3]_START
		        (type==TYPE_SOMESTART)        ||				// A START, Jim,but not as we know it ...
		        (type==TYPE_HERIANFONTSTART)  ||				// heretic/hexen font start FONTA_S,FONTAY_S,FONTB_S,
		        (type==TYPE_HERETICFONTSTART) ||				// hexen font start FONTAY_S
		        (type==TYPE_HEXENFONTSTART)   ||				// heretic/hexen font start FONTA_S,FONTAY_S,FONTB_S,
		        (type==TYPE_GOODEND)          ||				// S_START,P_START,F_START,C_START,A_END,TX_END,V_START
		        (type==TYPE_POOREND)          ||				// SS_START,PP_START,FF_START
		        (type==TYPE_SUBEND)           ||				// S[1-3]_START,P[1-3]_START,F[1-3]_START
		        (type==TYPE_SOMEEND)          ||				// An END, Jim,but not as we know it ...
		        (type==TYPE_HERIANFONTEND)    ||				// heretic/hexen font end FONTA_E,FONTAY_E,FONTB_E,
		        (type==TYPE_HERETICFONTEND)   ||				// hexen font end FONTAY_E
		        (type==TYPE_HEXENFONTEND)						// heretic/hexen font end FONTA_E,FONTAY_E,FONTB_E,
			   )
			{
				EventState(VERBOSITY_WARNINGS,"List marker %s (entry %ld) has non-zero size",LNL->entries[i].name,i);
				continue;
			}
		}
	}
}

// Identify list membership of each entry
static void IdentifyListMembershipOfEachEntry (
	LNL_t *LNL,													// lump name list to identify list membership of each entry in
	LST_t *LST,													// symbol table created from this list
	bool_t declassify_pnames,									// treat PNAMES as an unclassified lump (separate from TEXTURES)
	bool_t *foundsub)											// found remove sub-list border marker
{
	// DOCUMENTATION
	/*
		Used only by LNLScan()
	*/

	// VARIABLES
	size_t i;
	bool_t in_map=FALSE;
	LPS_t *LPS=NULL;

	// PRIMARY ERROR CHECK
	if (LNL==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to identify list membership of each entry with NULL lump name list argument");
	}
	if (LNL->entries==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to identify list membership of each entry with NULL lump name list entries");
	}
#ifdef PARANOID
	if (LNL->count>LNL->limit)
	{
		FatalAbort(1,__FILE__,__LINE__,"Lump name list is corrupt");
	}
#endif
	if (LST==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to identify list membership of each entry with NULL list symbol table argument");
	}
	if (LST->entries==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to identify list membership of each entry with NULL list symbol table entries");
	}
#ifdef PARANOID
	if (LST->count>LST->limit)
	{
		FatalAbort(1,__FILE__,__LINE__,"List symbol table is corrupt");
	}
#endif

	// INITIALISE
	*foundsub=FALSE;
	LPSInit(&LPS);

	// PERFORM THE OPERATION
	for (i=0;i<LNL->count;i++)
	{
		// Variables
		bool_t corrected;
		LNLentry_t *p;
		char name[9];

		// Initialise (part 1)
		p=&LNL->entries[i];

		// Ignore if we know what it is
		if (p->list!=LIST_UNKNOWN)
		{
			continue;
		}

		// Initialise (part 2)
		corrected=FALSE;
		(void) strcpy(name,p->name);

		// Preprocess
		//
		// If it is a standard and required (but not graphic
		// font) list start or end then preprocess it to handle
		// the double flag letter problem found in standard and
		// required list markers (it cannot occur occur in the
		// sub-list border markers, nor in the Heretic and HeXen
		// graphic font markers). We do not change the markers
		// on exit; it is only during the scan.
		//
		if (
			 (p->type==TYPE_GOODSTART)        ||
			 (p->type==TYPE_POORSTART)        ||
			 (p->type==TYPE_SOMESTART)        ||
			 (p->type==TYPE_GOODEND)          ||
			 (p->type==TYPE_POOREND)          ||
			 (p->type==TYPE_SOMEEND)
		 )
		{
			if  (
					( (name[0]=='S') && (name[1]=='S') && (name[2]=='_') ) ||
					( (name[0]=='P') && (name[1]=='P') && (name[2]=='_') ) ||
					( (name[0]=='F') && (name[1]=='F') && (name[2]=='_') )
				)
			{
				// Must correct internally
				corrected=TRUE;
				(void) strcpy(name,&p->name[1]);
			}
		}

		// If found at least one sub-list border marker then remember
		if ( (p->type==TYPE_SUBSTART) || (p->type==TYPE_SUBEND) )
		{
			*foundsub=TRUE;
		}

		// If it is a start then identify and push
		if (
			 (p->type==TYPE_GOODSTART)        ||
			 (p->type==TYPE_POORSTART)        ||
			 (p->type==TYPE_SUBSTART)         ||
			 (p->type==TYPE_SOMESTART)        ||
			 (p->type==TYPE_HERIANFONTSTART)  ||
			 (p->type==TYPE_HERETICFONTSTART) ||
			 (p->type==TYPE_HEXENFONTSTART)
		   )
		{
			// VARIABLES
			bool_t start;
			// IDENTIFY THE LIST THAT THIS START BELONGS TO
			LSTAddSymbolAndGetMetrics(LST,name,p->type,&p->list,&start);
			EventState(VERBOSITY_DETAILS,"Classified entry %s (entry %ld) in %s",p->name,i,LST->entries[p->list].name);
			// THIS IS THE LIST WE ARE NOW PARSING
			LPSPush(LPS,p->list,start,corrected);

			// CHECK FOR LIST NESTING ERRORS
			CheckForListNestingErrors (
				LPS,											// parse stack used to parse this list
				LST);											// symbol table created from this list

			// WE NOW KNOW WHAT THIS IS
			continue;
		}
		// If it is an end then identify, pop and validate
		if (
			 (p->type==TYPE_GOODEND)        ||
			 (p->type==TYPE_POOREND)        ||
			 (p->type==TYPE_SUBEND)         ||
			 (p->type==TYPE_SOMEEND)        ||
			 (p->type==TYPE_HERIANFONTEND)  ||
			 (p->type==TYPE_HERETICFONTEND) ||
			 (p->type==TYPE_HEXENFONTEND)
		   )
		{
			// VARIABLES
			list_t templist;
			bool_t start;
			bool_t tempstart;
			bool_t tempcorrected;
			// PRIMARY ERROR CHECK
			if (LPSGetStackDepth(LPS)==0)
			{
				// USED TO BE PREDICATED ON IGNORE_LIST_SYNTAX
				// Parse error means a broken WAD file so is always an abort-causing error
				ErrorAbort(1,__FILE__,__LINE__,"Unmatched list end marker (%s) in the input file",p->name);
			}
			// IDENTIFY THE LIST THAT THIS END BELONGS TO
			LSTAddSymbolAndGetMetrics(LST,name,p->type,&p->list,&start);
			EventState(VERBOSITY_DETAILS,"Classified entry %s (entry %ld) in %s",p->name,i,LST->entries[p->list].name);
			// GET BACK THE LIST THAT WE SHOULD BE PARSING
			LPSPop(LPS,&templist,&tempstart,&tempcorrected);
			// SECONDARY ERROR CHECK
			if (templist!=p->list)
			{
				char tempname[9];
				/***************/
				LSTGetLumpName(LST,templist,tempstart,tempname);
				if (tempcorrected)
				{	char temptempname[9];
					/*******************/
					temptempname[0]=tempname[0];
					temptempname[1]=NUL;
					(void) strcat(temptempname,tempname);
					(void) strcpy(tempname,temptempname);
				}
				ErrorAbort(1,__FILE__,__LINE__,"Entry %s (entry %ld) does not match %s",p->name,tempname);
			}
			// WE NOW KNOW WHAT THIS IS
			continue;
		}

		// If this is not a marker it may be part of a list
		if (LPSGetStackDepth(LPS)!=0)
		{
			// SET THE LIST THAT THIS ENTRY BELONGS TO
			LPSGetStackTopList(LPS,&p->list);
			// If physically a graphic then update type flag
			if (
				 (p->list==LIST_HERIANFONTA)  ||				// heretic/hexen font "A"
				 (p->list==LIST_HERIANFONTAY) ||				// hexen font "AY"
				 (p->list==LIST_HERIANFONTB)  ||				// heretic/hexen font "B"
				 (p->list==LIST_HERETICFONTA) ||				// heretic font "A"
				 (p->list==LIST_HERETICFONTB) ||				// heretic font "B"
				 (p->list==LIST_HEXENFONTA)   ||				// hexen font "A"
				 (p->list==LIST_HEXENFONTAY)  ||				// hexen font "AY"
				 (p->list==LIST_HEXENFONTB)   ||				// hexen font "B"
				 (p->list==LIST_SPRITES)      ||				// actor sprites
				 (p->list==LIST_PATCHES)      ||				// wall patches (as listed in PNAMES)
//				 (p->list==LIST_FLATS)		  ||				// foors and ceilings
//				 (p->list==LIST_AUTOTEXTURES) ||				// single-patch textures (independent of PNAMES) (from ZDOOM)
				 (p->list==LIST_SPRITESSUB1)  ||				// sprites sub-list #1
				 (p->list==LIST_SPRITESSUB2)  ||				// sprites sub-list #2
				 (p->list==LIST_SPRITESSUB3)  ||				// sprites sub-list #3
				 (p->list==LIST_PATCHESSUB1)  ||				// patches sub-list #1
				 (p->list==LIST_PATCHESSUB2)  ||				// patches sub-list #2
				 (p->list==LIST_PATCHESSUB3)					// patches sub-list #3
//				 (p->list==LIST_FLATSSUB1)    ||				// flats sub-list #1
//				 (p->list==LIST_FLATSSUB2)    ||				// flats sub-list #2
//				 (p->list==LIST_FLATSSUB3)						// flats sub-list #3
			   )
			{
				// RECLASSIFY TYPE TO GRAPHIC
				p->type=TYPE_GRAPHIC;
				EventState(VERBOSITY_DETAILS,"Classified entry %s (entry %ld)in %s <type updated to GRAPHIC>",p->name,i,LST->entries[p->list].name);
				// WE NOW KNOW WHAT THIS IS
				continue;
			}
			// If physically a flat then update type flag
			if (
				 (p->list==LIST_FLATS)		  ||				// foors and ceilings
				 (p->list==LIST_FLATSSUB1)    ||				// flats sub-list #1
				 (p->list==LIST_FLATSSUB2)    ||				// flats sub-list #2
				 (p->list==LIST_FLATSSUB3)						// flats sub-list #3
			   )
			{
				// RECLASSIFY TYPE TO FLAT
				p->type=TYPE_FLAT;
				EventState(VERBOSITY_DETAILS,"Classified entry %s (entry %ld)in %s <type updated to FLAT>",p->name,i,LST->entries[p->list].name);
				// WE NOW KNOW WHAT THIS IS
				continue;
			}
			// If part of an ACS library then update type flag
			if (p->list==LIST_ACSLIBRARIES)
			{
				// RECLASSIFY TYPE AS NEEDED
				if (wnmatch(p->name,"SCRIPTS"))
				{
					p->type=TYPE_SCRIPTS;
					EventState(VERBOSITY_DETAILS,"Classified entry %s (entry %ld)in %s <type updated to SCRIPTS>",p->name,i,LST->entries[p->list].name);
				}
				else
				{
					p->type=TYPE_BEHAVIOR;
					EventState(VERBOSITY_DETAILS,"Classified entry %s (entry %ld)in %s <type updated to BEHAVIOR>",p->name,i,LST->entries[p->list].name);
				}
				// WE NOW KNOW WHAT THIS IS
				continue;
			}
			// LEAVE IT AS IS
			EventState(VERBOSITY_DETAILS,"Classified entry %s (entry %ld) in %s",p->name,i,LST->entries[p->list].name);
			// WE NOW KNOW WHAT THIS IS
			continue;
		}

		// The entry is not part of a structured list
		if (LPSGetStackDepth(LPS)==0)
		{
			// Clear "in map" flag if we just left (or were never in) map DATA
			if ( (p->type!=TYPE_MAPDATA) && (p->type!=TYPE_GLMAPDATA) )
			{
				in_map=FALSE;
			}
			// Set "in map" flag if we see a map NAME.
			if ( (p->type==TYPE_MAPNAME) || (p->type==TYPE_GLMAPNAME) )
			{
				in_map=TRUE;
			}
			//
			// Thus if we see "stray" map DATA entries (i.e., before a map NAME
			// entry or between, but not included in, the most recent map data
			// sequence and the [start of the] next) then we shunt them out to
			// the LUMPS subdirectory as we are not sure what to do with them.
			// This is why the flag in_map has to be initialised to FALSE.
			//
			if (in_map)
			{
				p->list=LIST_MAPS;
			}
			else if (p->type==TYPE_GRAPHIC)
			{
				if (p->detected)
				{
					EventState(VERBOSITY_WARNINGS,"Type of entry %s (entry %ld) resolved as GRAPHIC",p->name,i);
				}
				p->list=LIST_GRAPHICS;
			}
			else if (p->type==TYPE_PAGE)
			{
				p->list=LIST_LUMPS;
			}
			else if (p->type==TYPE_AUTO)
			{
				FatalAbort(1,__FILE__,__LINE__,"Graphic and page image auto-detection should have been performed by now");
			}
			else if (p->type==TYPE_FLAT)
			{
				FatalAbort(1,__FILE__,__LINE__,"Floor and ceiling tile graphics can only exist in a structured list");
			}
			else if (p->type==TYPE_PCSFX)
			{
				p->list=LIST_PCSFXS;
			}
			else if (p->type==TYPE_SOUND)
			{
				if (p->detected)
				{
					EventState(VERBOSITY_WARNINGS,"Type of entry %s (entry %ld) resolved as SOUND",p->name,i);
				}
				p->list=LIST_SOUNDS;
			}
			else if (p->type==TYPE_VOICE)
			{
				if (p->detected)
				{
					EventState(VERBOSITY_WARNINGS,"Type of entry %s (entry %ld) resolved as SOUND",p->name,i);
				}
				p->list=LIST_SOUNDS;
			}
			else if (p->type==TYPE_MUSIC)
			{
				if (p->detected)
				{
					EventState(VERBOSITY_WARNINGS,"Type of entry %s (entry %ld) resolved as MUSIC",p->name,i);
				}
				p->list=LIST_MUSICS;
			}
			else if (p->type==TYPE_TEXTURE)
			{
				p->list=LIST_TEXTURES;
			}
			else if (!declassify_pnames && (p->type==TYPE_PNAMES))
			{
				p->list=LIST_PNAMES;
			}
			else if (p->type==TYPE_DIALOG)
			{
				p->list=LIST_DIALOGS;
			}
			else if (p->type==TYPE_CONVERSATION)
			{
				p->list=LIST_CONVERSATIONS;
			}
			else if (p->type==TYPE_BEHAVIOR)
			{
				FatalAbort(1,__FILE__,__LINE__,"Non-map-related BEHAVIOR lumps can only exist in an ACS library");
			}
			else if (p->type==TYPE_SCRIPTS)
			{
				FatalAbort(1,__FILE__,__LINE__,"Non-map-related SCRIPTS lumps can only exist in an ACS library");
			}
			else
			{
				p->list=LIST_LUMPS;
			}
			EventState(VERBOSITY_DETAILS,"Classified entry %s (entry %ld) in %s",p->name,i,LST->entries[p->list].name);
			// We now know what this is
			continue;
		}
	}

	// DEINITIALISE
	LPSDone(&LPS);
}

// Reclassify sub-list items in parent lists
static void ReclassifySubListItemsInParentLists (
	LNL_t *LNL,													// lump name list to reclassify sub-list items in parent lists in
	LST_t *LST)													// symbol table created from this list
{
	// DOCUMENTATION
	/*
		Used only by LNLScan()

		The sub-lists have no meaning to CleanWAD, but are used
		by some WAD designers and by some WAD tools, usually to
		mark a boundary between shareware and commercial WAD
		resources. However, they are not strictly part of the
		WAD file format and the games themselves ignore them.
	*/

	// VARIABLES
	size_t i;

	// PRIMARY ERROR CHECK
	if (LNL==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to reclassify sub-list items in parent lists with NULL lump name list argument");
	}
	if (LNL->entries==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to reclassify sub-list items in parent lists with NULL lump name list entries");
	}
#ifdef PARANOID
	if (LNL->count>LNL->limit)
	{
		FatalAbort(1,__FILE__,__LINE__,"Lump name list is corrupt");
	}
#endif
	if (LST==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to reclassify sub-list items in parent lists with NULL list symbol table argument");
	}
	if (LST->entries==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to reclassify sub-list items in parent lists with NULL list symbol table entries");
	}
#ifdef PARANOID
	if (LST->count>LST->limit)
	{
		FatalAbort(1,__FILE__,__LINE__,"List symbol table is corrupt");
	}
#endif

	// PERFORM OPERATION
	for (i=0;i<LNL->count;i++)
	{
		LNLentry_t *p;
		/************/
		p=&LNL->entries[i];
		if (
			 (p->list==LIST_SPRITESSUB1) ||
			 (p->list==LIST_SPRITESSUB2) ||
			 (p->list==LIST_SPRITESSUB3)
		   )
		{
			EventState(VERBOSITY_DETAILS,"Reclassified entry %s (entry %ld) in %s",p->name,i,LST->entries[LIST_SPRITES].name);
			p->list=LIST_SPRITES;
		}
		if (
			 (p->list==LIST_PATCHESSUB1) ||
			 (p->list==LIST_PATCHESSUB2) ||
			 (p->list==LIST_PATCHESSUB3)
		   )
		{
			EventState(VERBOSITY_DETAILS,"Reclassified entry %s (entry %ld) in %s",p->name,i,LST->entries[LIST_PATCHES].name);
			p->list=LIST_PATCHES;
		}
		if (
			 (p->list==LIST_FLATSSUB1) ||
			 (p->list==LIST_FLATSSUB2) ||
			 (p->list==LIST_FLATSSUB3)
		   )
		{
			EventState(VERBOSITY_DETAILS,"Reclassified entry %s (entry %ld) in %s",p->name,i,LST->entries[LIST_FLATS].name);
			p->list=LIST_FLATS;
		}
	}
}

// Verify that all entries are in a known list
static void VerifyThatAllEntriesAreInAKnownList (
	LNL_t *LNL)													// lump name list to verify that all entries are in a known list in
{
	// DOCUMENTATION
	/*
		Used only by LNLScan()
	*/

	// VARIABLES
	size_t i;

	// PRIMARY ERROR CHECK
	if (LNL==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to verify that all entries are in a known list in lump name list with NULL argument");
	}
	if (LNL->entries==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to verify that all entries are in a known list in lump name list with NULL entries");
	}
#ifdef PARANOID
	if (LNL->count>LNL->limit)
	{
		FatalAbort(1,__FILE__,__LINE__,"Lump name list is corrupt");
	}
#endif

	// PERFORM OPERATION
	for (i=0;i<LNL->count;i++)
	{
		if (LNL->entries[i].list==LIST_UNKNOWN)
		{
			FatalAbort(1,__FILE__,__LINE__,"Entry %s (entry %ld) is not in any list",LNL->entries[i].name,i);
		}
	}
}

// Identify lists having more than one occurence
static void IdentifyListsHavingMoreThanOneOccurence(
	LST_t *LST,													// symbol table created from this list
	bool_t tolerate_multiples,									// do not treat multiple instances of structured lists as an error
	bool_t quiet_multiples)										// do not treat multiple instances of structured lists as a problem
{
	// DOCUMENTATION
	/*
		Used only by LNLScan()
	*/

	// VARIABLES
	size_t i,multiples;
	bool_t logged;

	// PRIMARY ERROR CHECK
	if (LST==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to identify lists having more than one occurence in list symbol table with NULL argument");
	}
	if (LST->entries==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to identify lists having more than one occurence in list symbol table with NULL entries");
	}
#ifdef PARANOID
	if (LST->count>LST->limit)
	{
		FatalAbort(1,__FILE__,__LINE__,"List symbol table is corrupt");
	}
#endif

	// IF TOLERATED THEN ISSUE WARNINGS ONLY AND RETURN
	if (tolerate_multiples || quiet_multiples)
	{
		if (!quiet_multiples)
		{
			size_t j;
			/*******/
			for (j=0;j<LST->count;j++)
			{
				if (LST->entries[j].seen>1)
				{
					EventState(VERBOSITY_WARNINGS,"List %s (%s) occurs more than once",LST->entries[j].flag,LST->entries[j].name);
				}
			}
		}
		return;
	}

	// COUNT MULTIPLE OCCURENCES
	multiples=0;
	for (i=0;i<LST->count;i++)
	{
		if (LST->entries[i].seen>1)
		{
			multiples++;
		}
	}

	// HANDLE TRIVIAL CASE
	if (multiples==0)
	{
		return;
	}

	// PERFORM OPERATION
	logged=FALSE;
	for (i=0;i<LST->count;i++)
	{
		if (LST->entries[i].seen>1)
		{
			logged=TRUE;
			EventState(VERBOSITY_ERRORS,"List %s (%s) occurs more than once",LST->entries[i].flag,LST->entries[i].name);
		}
	}
	if (logged)
	{
		ErrorAbort(1,__FILE__,__LINE__,NULL);
	}
}

/**********************************************************************************************************************************/
/****************************************************** Lump Name List Usage ******************************************************/
/**********************************************************************************************************************************/

// Add a lump name to a lump name list at the next entry
void LNLAdd (
	LNL_t *LNL,													// lump name list to which to add lump name
	const char *name,											// name of lump
	size_t indx,												// index of entry within directory
	type_t type,												// lump name type
	list_t list,												// list that lump name belongs to
	bool_t isempty,												// lump has zero size
	bool_t detected)											// lump type was auto-detected
{
	// PRIMARY ERROR CHECK
	if (LNL==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to add to lump name list with NULL argument");
	}
	if (LNL->entries==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to add to lump name list with NULL entries");
	}
#ifdef PARANOID
	if (LNL->count>LNL->limit)
	{
		FatalAbort(1,__FILE__,__LINE__,"Lump name list is corrupt");
	}
#endif

	// SECONDARY ERROR CHECK
	if (strlen(name)>8)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to use name (\"%s\") as a WAD entry name (longer than 8 characters)",name);
	}

	// REALLOCATE [IF NECESSARY] MEMORY FOR THE LNL ENTRIES
	if (LNL->count==LNL->limit)
	{
		LNLentry_t *newptr;
		size_t size,sizeb,incrb;
		/**********************/
		size=(LNL->limit)+LNL_INCREMENT;
		sizeb=size*sizeof(LNLentry_t);
		incrb=LNL_INCREMENT*sizeof(LNLentry_t);
		newptr=realloc(LNL->entries,sizeb);
		if (newptr==NULL)
		{
			ErrorAbort(1,__FILE__,__LINE__,"Could not grow lump name list entries by %ld bytes (from %ld bytes)",incrb,sizeb);
		}
		LNL->entries=newptr;
		(void) memset(&LNL->entries[LNL->limit],0,incrb);
		LNL->limit=size;
	}

	// PERFORM THE OPERATION
	(void) memmove(LNL->entries[LNL->count].name,name,1+strlen(name));
	LNL->entries[LNL->count].indx=indx;
	LNL->entries[LNL->count].type=type;
	LNL->entries[LNL->count].list=list;
	LNL->entries[LNL->count].isempty=isempty;
	LNL->entries[LNL->count].detected=detected;
	LNL->entries[LNL->count].eraseit=FALSE;
	LNL->entries[LNL->count].kind=KIND_PUTINLIST;
	LNL->entries[LNL->count].sortname[0]=NUL;
	LNL->entries[LNL->count].sortindx=0;
	LNL->count++;
}

// Scan a pre-scanned lump name list and identify each entry
void LNLScan (
	LNL_t *LNL,													// lump name list whose contents are to be identified
	LST_t *LST,													// symbol table to be created from this list
	bool_t recognised_names,									// identify sounds/musics/dialogs/conversations by recognised names
	bool_t tolerate_multiples,									// do not treat multiple instances of structured lists as an error
	bool_t quiet_multiples,										// do not treat multiple instances of structured lists as a problem
	bool_t declassify_pnames,									// treat PNAMES as an unclassified lump (separate from TEXTURES)
	bool_t loose_headers,										// allow nonstandard map name headers (not just E\?M\? and MAP\?\?)
	bool_t quiet_headers,										// do not warn about non-empty map name headers
	game_t game)												// game for which (WAD whose lump names these are) was designed
{
	// DOCUMENTATION
	/*
		The lump name list should have already been pre-scanned
		at the WAD directory level to identify by name where
		possible and to resolve any requests for auto-detection
		between page and graphic types.
	*/

	// VARIABLES
	bool_t foundsub=FALSE;

	// PRIMARY ERROR CHECK
	if (LNL==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to scan lump name list with NULL argument");
	}
	if (LNL->entries==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to scan lump name list with NULL entries");
	}
#ifdef PARANOID
	if (LNL->count>LNL->limit)
	{
		FatalAbort(1,__FILE__,__LINE__,"Lump name list is corrupt");
	}
#endif
	if (LST==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to scan lump name list with NULL list symbol table argument");
	}
	if (LST->entries==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to scan lump name list with NULL list symbol table entries");
	}
#ifdef PARANOID
	if (LST->count>LST->limit)
	{
		FatalAbort(1,__FILE__,__LINE__,"List symbol table is corrupt");
	}
#endif

	// PHASE 1
	// DETECT NONSTANDARD MAP NAMES
	if (loose_headers)
	{
		EventState(VERBOSITY_DETAILS,">>>>>>>> Detecting nonstandard map names");
		DetectNonStandardMapNames(LNL);
	}

	// PHASE 2
	// ASSIGN OPENGL MAP ITEMS TO PARENT MAPS WHERE APPLICABLE
	EventState(VERBOSITY_DETAILS,">>>>>>>> Assigning OpenGL map items to parent maps where applicable");
	AssignOpenGLMapItemsToParentMaps(LNL);

	// PHASE 3
	// LOG MARKERS WITH INVALID NON-ZERO SIZES
	EventState(VERBOSITY_DETAILS,">>>>>>>> Logging markers with invalid non-zero sizes");
	LogMarkersWithInvalidNonZeroSizes(LNL,quiet_headers);

	// PHASE 4 - IDENTIFY LIST MEMBERSHIP OF EACH ENTRY
	EventState(VERBOSITY_DETAILS,">>>>>>>> Identifying list membership of entries by markers");
	IdentifyListMembershipOfEachEntry(LNL,LST,declassify_pnames,&foundsub);

	// PHASE 5 - RECLASSIFY SUB-LIST ITEMS IN PARENT LISTS
	if (foundsub)
	{
		EventState(VERBOSITY_DETAILS,">>>>>>>> Reclassify sub-list items in parent lists");
		ReclassifySubListItemsInParentLists(LNL,LST);
	}

	// PHASE 6 - VERIFY THAT ALL ENTRIES ARE IN A KNOWN LIST
	EventState(VERBOSITY_DETAILS,">>>>>>>> Verifying that all entries are in a known list");
	VerifyThatAllEntriesAreInAKnownList(LNL);

	// PHASE 7 - Identify lists having more than one occurence
	EventState(VERBOSITY_DETAILS,">>>>>>>> Identifying lists having more than one occurence");
	IdentifyListsHavingMoreThanOneOccurence(LST,tolerate_multiples,quiet_multiples);
}

// Normalise a lump name list
void LNLNormalise (
	LNL_t *LNL,													// lump name list to normalise
	LST_t *LST,													// symbol table resulting from the scan of that list
	bool_t remove_duplicates,									// remove all but the first of entries having the same name
	bool_t identify_voices,										// identify loose STRIFE VOICES by name (READ MANUAL BEFORE USING!)
	bool_t keep_wintex,											// do not remove _DEUTEX_ lump
	bool_t keep_platform,										// do not remove PLATFORM lump
	bool_t keep_history,										// do not remove HISTORY lump
	bool_t keep_tagdesc,										// do not remove TAGDESC lump
	bool_t keep_pcsfx,											// do not remove PC speaker sound effects
	bool_t keep_doubles,										// do not normalise double-letter list markers (e.g., PP_START)
	bool_t keep_borders,										// do not remove sub-list border markers (e.g., F1_START)
	bool_t keep_empties,										// do not remove empty structured lists
	bool_t remove_scripts,										// remove all SCRIPTS and SCRIPTnn entries
	bool_t force_removal,										// remove duplicate entries even if in different lists
	size_t *deleted)											// number of entries logically deleted by normalising
{
	// VARIABLES
	size_t i;
	LNLentry_t *entry;
	type_t type_mapname,type_mapdata;

	// PRIMARY ERROR CHECK
	if (LNL==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to normalise lump name list with NULL argument");
	}
	if (LNL->entries==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to normalise lump name list with NULL entries");
	}
#ifdef PARANOID
	if (LNL->count>LNL->limit)
	{
		FatalAbort(1,__FILE__,__LINE__,"Lump name list is corrupt");
	}
#endif

	// SCAN AND PROCESS
	*deleted=0;
	for (i=0;i<LNL->count;i++)
	{
		// Create a pointer to this entry
		entry=&LNL->entries[i];

		// Delete this entry if its name is the NULL string
		if (strlen(entry->name)==0)
		{
			entry->eraseit=TRUE;
			(*deleted)++;
			continue;
		}

		// Delete this entry if it is _DEUTEX_ and must do so
		if ( (!keep_wintex) && wnmatch(entry->name,"_DEUTEX_") )
		{
			EventState(VERBOSITY_CORRECTIONS,"Deleting editing leftover entry %s (entry %ld)",entry->name,i);
			entry->eraseit=TRUE;
			(*deleted)++;
			continue;
		}

		// Delete this entry if it is PLATFORM and must do so
		if ( (!keep_platform) && wnmatch(entry->name,"PLATFORM") )
		{
			EventState(VERBOSITY_CORRECTIONS,"Deleting editing leftover entry %s (entry %ld)",entry->name,i);
			entry->eraseit=TRUE;
			(*deleted)++;
			continue;
		}

		// Delete this entry if it is HISTORY and must do so
		if ( (!keep_history) && wnmatch(entry->name,"HISTORY") )
		{
			EventState(VERBOSITY_CORRECTIONS,"Deleting editing leftover entry %s (entry %ld)",entry->name,i);
			entry->eraseit=TRUE;
			(*deleted)++;
			continue;
		}

		// Delete this entry if it is TAGDESC and must do so
		if ( (!keep_tagdesc) && wnmatch(entry->name,"TAGDESC") )
		{
			EventState(VERBOSITY_CORRECTIONS,"Deleting editing leftover entry %s (entry %ld)",entry->name,i);
			entry->eraseit=TRUE;
			(*deleted)++;
			continue;
		}

		// Delete this entry if it is a PCSFX and must do so
		if ( (!keep_pcsfx) && (entry->type==TYPE_PCSFX) )
		{
			EventState(VERBOSITY_CORRECTIONS,"Deleting PCSFX entry %s (entry %ld)",entry->name,i);
			entry->eraseit=TRUE;
			(*deleted)++;
			continue;
		}

		// Delete this entry if it is a SCRIPT and must do so
		if (
		     (remove_scripts     ) &&
		     (entry->name[0]=='S') &&
		     (entry->name[1]=='C') &&
		     (entry->name[2]=='R') &&
		     (entry->name[3]=='I') &&
		     (entry->name[4]=='P') &&
		     (entry->name[5]=='T') &&
		     (entry->name[6]=='S') &&
		     (entry->name[7]==NUL)
		   )
		{
			EventState(VERBOSITY_CORRECTIONS,"Deleting SCRIPT entry SCRIPTS (entry %ld)",i);
			entry->eraseit=TRUE;
			(*deleted)++;
			continue;
		}
		if (
		     (remove_scripts         ) &&
		     (entry->name[0]=='S'    ) &&
		     (entry->name[1]=='C'    ) &&
		     (entry->name[2]=='R'    ) &&
		     (entry->name[3]=='I'    ) &&
		     (entry->name[4]=='P'    ) &&
		     (entry->name[5]=='T'    ) &&
		     (ISDIGIT(entry->name[6])) &&
		     (ISDIGIT(entry->name[7])) &&
		     (entry->name[8]==NUL    )
		   )
		{
			EventState(VERBOSITY_CORRECTIONS,"Deleting SCRIPT entry %s (entry %ld)",entry->name,i);
			entry->eraseit=TRUE;
			(*deleted)++;
			continue;
		}

		// If it is a standard (but not graphic font) list
		// start or end then preprocess it to handle the double
		// flag letter and sub-list border marker problems
		// found in standard list markers (they cannot occur
		// in Heretic and HeXen graphic font markers).
		if (
			 (entry->type==TYPE_GOODSTART)        ||
			 (entry->type==TYPE_POORSTART)        ||
			 (entry->type==TYPE_SUBSTART)         ||
			 (entry->type==TYPE_SOMESTART)        ||
			 (entry->type==TYPE_GOODEND)          ||
			 (entry->type==TYPE_POOREND)          ||
			 (entry->type==TYPE_SUBEND)           ||
			 (entry->type==TYPE_SOMEEND)
		 )
		{
			char *namep=entry->name;
			if  (
					(  !keep_borders                                        ) &&
					( (namep[0]=='S') || (namep[0]=='P') || (namep[0]=='F') ) &&
					( (namep[1]=='1') || (namep[1]=='2') || (namep[1]=='3') ) &&
					(  namep[2]=='_' )
				)
			{
				EventState(VERBOSITY_CORRECTIONS,"Deleting sub-list border marker %s (entry %ld)",namep,i);
				entry->eraseit=TRUE;
				(*deleted)++;
				continue;
			}
			if  (
					( !keep_doubles )
					&&
					(
						( (namep[0]=='S') && (namep[1]=='S') && (namep[2]=='_') ) ||
						( (namep[0]=='P') && (namep[1]=='P') && (namep[2]=='_') ) ||
						( (namep[0]=='F') && (namep[1]=='F') && (namep[2]=='_') )
					)
				)
			{
				char newname[9];
				//
				(void) strcpy(newname,&entry->name[1]);
				EventState(VERBOSITY_CORRECTIONS,"Renaming list marker %s (entry %ld) to %s",entry->name,i,newname);
				(void) strcpy(entry->name,newname);
			}
		}

		// Identify map entry type for comparisons
		// ** refer to Appendix 4 in the manual **
		if ((entry->type==TYPE_GLMAPNAME) || (entry->type==TYPE_GLMAPDATA))
		{	// GL MAP
			type_mapname=TYPE_GLMAPNAME;
			type_mapdata=TYPE_GLMAPDATA;
		}
		else
		{	// Normal map (or not map at all)
			type_mapname=TYPE_MAPNAME;
			type_mapdata=TYPE_MAPDATA;
		}

		// Check for duplicate entries if required
		if (remove_duplicates && (i>0))
		{
			// Variables
			size_t j,k,index;
			bool_t match,atmap,found;
			char *p1,*p2;
			char v1,v2;
			LNLentry_t *entries;
			// Find any previous same-named entry
			entries=LNL->entries;
			atmap=FALSE;
			found=FALSE;
			index=0;
			j=i;
			while (j>0)
			{
				--j;
				if (!entries[j].eraseit)
				{
					// Stop if map (maps are handled specially)
					if (entries[i].type==type_mapdata)
					{
						if (entries[j].type==type_mapname)
						{
							atmap=TRUE;
							index=j;
							break;
						}
						if (entries[j].type!=type_mapdata)
						{
							EventState(VERBOSITY_ERRORS,"Corrupt map sequence %s..%s: is %s a map name?", entries[j].name,entries[i].name,entries[j].name);
							EventState(VERBOSITY_ERRORS,"To allow generic map names, include the +gh option in the command line.");
							ErrorAbort(1,__FILE__,__LINE__,NULL);
						}
					}
					// Compare the entry names
					match=TRUE;
					p1=entries[i].name;
					p2=entries[j].name;
					for (k=0;;k++)
					{
						v1=*p1++;
						v2=*p2++;
						if ( (v1==NUL) && (v2==NUL) )
						{
							break;
						}
						if (v1!=v2)
						{
							match=FALSE;
							break;
						}
						if (k==7)
						{
							break;
						}
					}
					// If they are the same then found a match
					if (match)
					{
						found=TRUE;
						index=j;
						break;
					}
				}
			}
			// If this is map data [i.e., WE HIT THE MAP START
			// WITHOUT FINDING A DUPLICATE OF THIS SPECIFIC MAP
			// DATA ITEM], then this entry does not need to be
			// considered further as it should only be checked
			// for duplicates in its parent map; if not, AND
			// THERE IS A MATCH, then delete the duplicate.
			if (!atmap && found)
			{
				// If we found a second occurrence (going backward) of this
				// entry in the same map then this entry must be deleted as
				// the DOOM backward override search in PWADS is not used
				// for map entries; DOOM assumes that there is a specific
				// offset from the map name header for each map data entry.
				// Righly or wrongly, DOOM would just ignore it; so we might
				// as well get rid of it, especially as it would confuse
				// several map editing utilities and, of course, end users.
				if (entry->type==type_mapdata)
				{
					// Treat this as a warning (it should never happen)
					EventState(VERBOSITY_WARNINGS,"Replacing duplicate map data %s (entry %ld)",LNL->entries[index].name,index);
					LNL->entries[index].eraseit=TRUE;
					(*deleted)++;
					continue;
				}
				// If we found a map name that was duplicated, we delete
				// the previous occurrence of the entire map, as a whole.
				// This includes any internal map data entry duplicates;
				// we are discarding that entire map, so we don't care
				// about any errors occuring inside that map anyway.
				if (entry->type==type_mapname)
				{
					// Variables
					size_t l,s,e;
					// Identify range of entries in map
					s=index;
					e=index;
					// Get upper bound on range
					l=index+1;
					while ( (l<i) && (LNL->entries[l].type==type_mapdata) )
					{
						e=l;
						l++;
					}
					// Delete the map entries
					EventState(VERBOSITY_CORRECTIONS,"Replacing duplicate map %s (entries %ld through %ld)",entry->name,s,e);
					for (l=s;l<=e;l++)
					{
						LNL->entries[l].eraseit=TRUE;
						(*deleted)++;
					}
					continue;
				};
				// If it is a SCRIPTS lump in an ACS library,
				// then the situation is like that of maps, but
				// much simpler. The structure of these is a
				// named BEHAVIOR lump followed by a contiguous
				// stream of SCRIPTS lumps, of which the last
				// is the "real" one (there should, properly,
				// only be one). Thus if the matching entry of
				// a SCRIPTS lump is not the entry IMMEDIATELY
				// before it then it isn't a match, as the
				// matching entry belongs to a BEHAVIOR lump in
				// a map or elswhere in that ACS library.
				if (entry->type==TYPE_SCRIPTS)
				{
					// Variables
					size_t l,s,e;
					// Check for name consistency
					if (!wnmatch(entry->name,"SCRIPTS"))
					{
						FatalAbort(1,__FILE__,__LINE__,"Failure in SCRIPTS identification for %s (entry %ld)",entry->name,i);
					}
					// If not really a match then leave it
					if (index!=(i-1))
					{
						continue;
					}
					// Identify range of entries in ACS module
					s=index;
					e=index;
					// Get lower bound on range
					while ( (s>0) && (LNL->entries[s].type==TYPE_SCRIPTS) )
					{
						s--;
					}
					// Get upper bound on range
					l=index+1;
					while ( (l<LNL->count) && (LNL->entries[l].type==TYPE_SCRIPTS) )
					{
						e=l;
						l++;
					}
					// Delete the entries
					EventState(VERBOSITY_CORRECTIONS,"Replacing duplicate %s (entry %ld) in acs module %s (entries %ld through %ld)",
												     entry->name,index,LNL->entries[s].name,s,e);
					LNL->entries[index].eraseit=TRUE;
					(*deleted)++;
					continue;
				}
				// If it is a named BEHAVIOR lump in an ACS
				// library, then the situation is like that of
				// maps, but much simpler. The structure of
				// these is a named BEHAVIOR lump followed by a
				// contiguous stream of SCRIPTS lumps, of which
				// the last is the "real" one (there should,
				// properly, only be one). Remove the matching
				// lump and any trailing SCRIPTS lumps.
				if (entry->type==TYPE_BEHAVIOR)
				{
					// Variables
					size_t l,s,e;
					// Identify range of entries in ACS module
					s=index;
					e=index;
					l=index+1;
					while ( (l<i) && (LNL->entries[l].type==TYPE_SCRIPTS) )
					{
						e=l;
						l++;
					}
					// Delete the entries
					EventState(VERBOSITY_CORRECTIONS,"Replacing duplicate acs module %s (entries %ld through %ld)",entry->name,s,e);
					for (l=s;l<=e;l++)
					{
						LNL->entries[l].eraseit=TRUE;
						(*deleted)++;
					}
					continue;
				}
				// If it is a list marker, then do not remove
				// it as a duplicate as that would corrupt the
				// list that it marks. We already know by now
				// that any lists are correct in themselves and
				// that duplicate lists are handled elsewhere.
				if (
				     (entry->type==TYPE_GOODSTART)        ||			// S_START,P_START,F_START,C_START,A_START,TX_START,V_START
				     (entry->type==TYPE_POORSTART)        ||			// SS_START,PP_START,FF_START
				     (entry->type==TYPE_SUBSTART)         ||			// S[1-3]_START,P[1-3]_START,F[1-3]_START
				     (entry->type==TYPE_SOMESTART)        ||			// A START, Jim,but not as we know it ...
				     (entry->type==TYPE_HERIANFONTSTART)  ||			// heretic/hexen font start FONTA_S,FONTAY_S,FONTB_S,
				     (entry->type==TYPE_HERETICFONTSTART) ||			// hexen font start FONTAY_S
				     (entry->type==TYPE_HEXENFONTSTART)   ||			// heretic/hexen font start FONTA_S,FONTAY_S,FONTB_S,
				     (entry->type==TYPE_GOODEND)          ||			// S_START,P_START,F_START,C_START,A_END,TX_END,V_START
				     (entry->type==TYPE_POOREND)          ||			// SS_START,PP_START,FF_START
				     (entry->type==TYPE_SUBEND)           ||			// S[1-3]_START,P[1-3]_START,F[1-3]_START
				     (entry->type==TYPE_SOMEEND)          ||			// An END, Jim,but not as we know it ...
				     (entry->type==TYPE_HERIANFONTEND)    ||			// heretic/hexen font end FONTA_E,FONTAY_E,FONTB_E,
				     (entry->type==TYPE_HERETICFONTEND)   ||			// hexen font end FONTAY_E
				     (entry->type==TYPE_HEXENFONTEND)					// heretic/hexen font end FONTA_E,FONTAY_E,FONTB_E,
				   )
				{
					continue;
				}
				// If not a map then it is a single entry to replace
				if ( (entry->type!=TYPE_MAPNAME) && (entry->type!=TYPE_GLMAPNAME) )
				{
					// Same lists?
					if (entry->list==LNL->entries[index].list)
					{
						EventState(VERBOSITY_CORRECTIONS,"Replacing duplicate %s (entry %ld)",entry->name,index);
						LNL->entries[index].eraseit=TRUE;
						(*deleted)++;
						continue;
					}
					// Different lists but permitted?
					if (
					     (force_removal)
					    ||
					     (
					       (
							(identify_voices)                       &&
					        (entry->list==LIST_SOUNDS)              &&
					        (entry->type==TYPE_VOICE)               &&
					        (LNL->entries[index].list==LIST_VOICES) &&
					        (LNL->entries[index].type==TYPE_VOICE)
					       )
					      ||
					       (
							(identify_voices)                       &&
					        (entry->list==LIST_VOICES)              &&
					        (entry->type==TYPE_VOICE)               &&
					        (LNL->entries[index].list==LIST_SOUNDS) &&
					        (LNL->entries[index].type==TYPE_VOICE)
					       )
					     )
					   )
					{
						EventState(VERBOSITY_WARNINGS,"Replacing (different lists) duplicate %s (entry %ld)",entry->name,index);
						LNL->entries[index].eraseit=TRUE;
						(*deleted)++;
						continue;
					}
					// Different lists but not permitted
					EventState(VERBOSITY_WARNINGS,"Not replacing (different lists) duplicate %s (entry %ld)",entry->name,index);
					continue;
				}
				// If we get here then something has gone wrong
				FatalAbort(1,__FILE__,__LINE__,"Unknown duplicate type %s (entry %ld)",entry->name,index);
			}
		}
	}

	// REMOVE EMPTY LISTS IF REQUIRED TO
	if (!keep_empties)
	{
		// Variables
		size_t list_start,list_quantity;
		bool_t list_in;

		// Initialise
		list_in=FALSE;
		list_start=0;
		list_quantity=0;

		// Perform function for sub-lists
		for (i=0;i<LNL->count;i++)
		{
			// Create a pointer to this entry
			entry=&LNL->entries[i];

			// Ignore logically deleted entries
			if (entry->eraseit)
			{
				continue;
			}

			// Handle list starts
			if (entry->type==TYPE_SUBSTART)						// S[1-3]_START,P[1-3]_START,F[1-3]_START
			{
				list_in=TRUE;
				list_start=i;
				list_quantity=0;
				continue;
			}

			// Handle list ends
			if (entry->type==TYPE_SUBEND)						// S[1-3]_END,P[1-3]_END,F[1-3]_END
			{
				list_in=FALSE;
				if (list_quantity==0)
				{
					// Variables
					list_t list_actual;
					bool_t dummy_bool_t=FALSE;
					// Get actual list in case reclassified as
					// parent list (want the actual list name).
					LSTGetMetricsViaLumpName (
						LST,									// list symbol table to get list metrics from
						entry->name,							// lump name from which to identify list
						entry->type,							// lump name type
						&list_actual,							// list identifier associated with lump name
						&dummy_bool_t);							// start of list flag (FALSE means end of list) for lump name
					// Print the message
					EventState(VERBOSITY_CORRECTIONS,"Deleting empty structured list \"%s\" (entries %ld through %ld)",
													 LST->entries[list_actual].name,list_start,i);
					// Delete the (sub) list
					LNL->entries[list_start].eraseit=TRUE;
					(*deleted)++;
					LNL->entries[i].eraseit=TRUE;
					(*deleted)++;
				}
				continue;
			}

			// Handle non-marker entries
			if (list_in)
			{
				list_quantity++;
			}
//			else
//			{
//				Not a special case
//				No action needed
//			}
		}

		// Perform function for main lists
		for (i=0;i<LNL->count;i++)
		{
			// Create a pointer to this entry
			entry=&LNL->entries[i];

			// Ignore logically deleted entries
			if (entry->eraseit)
			{
				continue;
			}

			// Handle list starts
			if (
			     (entry->type==TYPE_GOODSTART)        ||			// S_START,P_START,F_START,C_START,A_START,TX_START,V_START
			     (entry->type==TYPE_POORSTART)        ||			// SS_START,PP_START,FF_START
			     (entry->type==TYPE_SOMESTART)        ||			// A START, Jim,but not as we know it ...
			     (entry->type==TYPE_HERIANFONTSTART)  ||			// heretic/hexen font start FONTA_S,FONTAY_S,FONTB_S,
			     (entry->type==TYPE_HERETICFONTSTART) ||			// hexen font start FONTAY_S
			     (entry->type==TYPE_HEXENFONTSTART)					// heretic/hexen font start FONTA_S,FONTAY_S,FONTB_S,
			   )
			{
				list_in=TRUE;
				list_start=i;
				list_quantity=0;
				continue;
			}

			// Handle list ends
			if (
			     (entry->type==TYPE_GOODEND)          ||			// S_START,P_START,F_START,C_START,A_END,TX_END,V_START
			     (entry->type==TYPE_POOREND)          ||			// SS_START,PP_START,FF_START
			     (entry->type==TYPE_SOMEEND)          ||			// An END, Jim,but not as we know it ...
			     (entry->type==TYPE_HERIANFONTEND)    ||			// heretic/hexen font end FONTA_E,FONTAY_E,FONTB_E,
			     (entry->type==TYPE_HERETICFONTEND)   ||			// hexen font end FONTAY_E
			     (entry->type==TYPE_HEXENFONTEND)					// heretic/hexen font end FONTA_E,FONTAY_E,FONTB_E,
			   )
			{
				list_in=FALSE;
				if (list_quantity==0)
				{
					EventState(VERBOSITY_CORRECTIONS,"Deleting empty structured list \"%s\" (entries %ld through %ld)",
													 LST->entries[entry->list].name,list_start,i);
					LNL->entries[list_start].eraseit=TRUE;
					(*deleted)++;
					LNL->entries[i].eraseit=TRUE;
					(*deleted)++;
				}
				continue;
			}

			// Handle non-marker entries
			if (list_in)
			{
				list_quantity++;
			}
//			else
//			{
//				Not a special case
//				No action needed
//			}
		}
	}
}

// Sort a lump name list by list (only)
void LNLSortListOnly (
	LNL_t *LNL)													// lump name list to sort
{
	// DOCUMENTATION
	/*
		The list must have already been scanned, but need not
		have been normalised or sorted.

		The sequence number ensures a stable sort (i.e., items
		with equal keys come out of the sort in the order that
		they went in.

		The key is thus LIST:SEQUENCE and the QSORT function
		from stdlib is used to perform the sort.
	*/

	// VARIABLES
	size_t i;

	// PRIMARY ERROR CHECK
	if (LNL==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to sort (list only) lump name list with NULL argument");
	}
	if (LNL->entries==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to sort (list only) lump name list with NULL entries");
	}
#ifdef PARANOID
	if (LNL->count>LNL->limit)
	{
		FatalAbort(1,__FILE__,__LINE__,"Lump name list is corrupt");
	}
#endif

	// ASSIGN SORT KEY FIELDS
	for (i=0;i<LNL->count;i++)
	{
		LNLentry_t *entry;
		/****************/
		entry=&LNL->entries[i];
		entry->sortindx=i;
		(void) memmove(entry->sortname,entry->name,9);
	}

	// PERFORM THE SORT
	qsort(LNL->entries,LNL->count,sizeof(LNLentry_t),LNLSortListOnlyCompare);
}

// Sort a lump name list by list and alphabetically within list
void LNLSortListAlpha (
	LNL_t *LNL,													// lump name list to sort
	bool_t sort_intrinsic,										// sort order is: list then intrinsic (use alphabetic if none)
	game_t game)												// game for which (WAD whose lump names these are) was designed
{
	// DOCUMENTATION
	/*
		The list must have already been scanned, but need not
		have been normalised or sorted. The list is pre-scanned
		and for map items, the sort name is given as the name
		of the map, otherwise it is the name of that entry.

		The sequence number ensures a stable sort (i.e., items
		with equal keys come out of the sort in the order that
		they went in) and that map data items are kept with the
		map of which they are a part.

		The key is thus LIST:NAME:SEQUENCE and the QSORT
		function from stdlib is used to perform the sort.
	*/

	// VARIABLES
	size_t i;
	char owner[9];

	// PRIMARY ERROR CHECK
	if (LNL==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to sort (list+alpha) lump name list with NULL argument");
	}
	if (LNL->entries==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to sort (list+alpha) lump name list with NULL entries");
	}
#ifdef PARANOID
	if (LNL->count>LNL->limit)
	{
		FatalAbort(1,__FILE__,__LINE__,"Lump name list is corrupt");
	}
#endif

	// ASSIGN SORT KEY FIELDS
	(void) strcpy(owner,"");
	for (i=0;i<LNL->count;i++)
	{
		// Variables
		LNLentry_t *entry;

		// Get entry pointer
		entry=&LNL->entries[i];

		// Set entry index
		entry->sortindx=i;

		// Set entry kind
		if (
		     (entry->type==TYPE_GOODSTART)        ||			// S_START,P_START,F_START,C_START,A_START,TX_START,V_START
		     (entry->type==TYPE_POORSTART)        ||			// SS_START,PP_START,FF_START
		     (entry->type==TYPE_SUBSTART)         ||			// S[1-3]_START,P[1-3]_START,F[1-3]_START
		     (entry->type==TYPE_SOMESTART)        ||			// A START, Jim,but not as we know it ...
		     (entry->type==TYPE_HERIANFONTSTART)  ||			// heretic/hexen font start FONTA_S,FONTAY_S,FONTB_S,
		     (entry->type==TYPE_HERETICFONTSTART) ||			// hexen font start FONTAY_S
		     (entry->type==TYPE_HEXENFONTSTART)					// heretic/hexen font start FONTA_S,FONTAY_S,FONTB_S,
		   )
		{
			entry->kind=KIND_BEGINLIST;							// entry is the START marker of a list
		}
		else if (
		     (entry->type==TYPE_GOODEND)          ||			// S_START,P_START,F_START,C_START,A_END,TX_END,V_START
		     (entry->type==TYPE_POOREND)          ||			// SS_START,PP_START,FF_START
		     (entry->type==TYPE_SUBEND)           ||			// S[1-3]_START,P[1-3]_START,F[1-3]_START
		     (entry->type==TYPE_SOMEEND)          ||			// An END, Jim,but not as we know it ...
		     (entry->type==TYPE_HERIANFONTEND)    ||			// heretic/hexen font end FONTA_E,FONTAY_E,FONTB_E,
		     (entry->type==TYPE_HERETICFONTEND)   ||			// hexen font end FONTAY_E
		     (entry->type==TYPE_HEXENFONTEND)					// heretic/hexen font end FONTA_E,FONTAY_E,FONTB_E,
		   )
		{
			entry->kind=KIND_ENDOFLIST;							// entry is the END marker of a list
		}
		else if (sort_intrinsic)
		{
			entry->kind=LINIdentifyLumpKindByMetrics (
				entry->name,									// name of the lump whose kind is to be determined
				entry->list,									// list membership of the lump
				game);											// game in which this combination of lump name and list appears
		}
		else
		{
			entry->kind=KIND_PUTINLIST;							// entry is an ENTRY in a list
		}

		// Set sort name key
		if (entry->list==LIST_MAPS)
		{
			if (entry->type==TYPE_MAPNAME)
			{
				(void) memmove(owner,entry->name,9);
			}
			(void) memmove(entry->sortname,owner,9);
		}
		else if (entry->list==LIST_ACSLIBRARIES)
		{
			if (entry->type==TYPE_BEHAVIOR)
			{
				(void) memmove(owner,entry->name,9);
			}
			(void) memmove(entry->sortname,owner,9);
		}
		else
		{
			(void) memmove(entry->sortname,entry->name,9);
		}
	}

	// PERFORM THE SORT
	qsort(LNL->entries,LNL->count,sizeof(LNLentry_t),LNLSortListAlphaCompare);
}

// Get a lump name from a lump name list by index number
void LNLGet (
	const LNL_t *LNL,											// lump name list from which to get lump name
	size_t which,												// which entry in the lump name list to get
	char *name,													// lump name
	size_t *indx,												// index of entry within directory
	type_t *type,												// lump name type
	list_t *list,												// list that lump name belongs to
	bool_t *eraseit)											// lump should be deleted (after certain types of scan)
{
	// PRIMARY ERROR CHECK
	if (LNL==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to get from lump name list with NULL argument");
	}
	if (LNL->entries==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to get from lump name list with NULL entries");
	}
#ifdef PARANOID
	if (LNL->count>LNL->limit)
	{
		FatalAbort(1,__FILE__,__LINE__,"Lump name list is corrupt");
	}
#endif

	// SECONDARY ERROR CHECK
	if (which>LNL->count)
	{
		FatalAbort(1,__FILE__,__LINE__,"Unrecognised value %ld in variable \"which\"",which);
	}

	// PERFORM THE OPERATION
	(void) strcpy(name,LNL->entries[which].name);
	*indx=LNL->entries[which].indx;
	*type=LNL->entries[which].type;
	*list=LNL->entries[which].list;
	*eraseit=LNL->entries[which].eraseit;
}

/**********************************************************************************************************************************/
/********************************************************** End of File ***********************************************************/
/**********************************************************************************************************************************/
