/**********************************************************************************************************************************/
/********************************************************* Documentation **********************************************************/
/**********************************************************************************************************************************/

/*

Wad file header

*/

/**********************************************************************************************************************************/
/*********************************************************** Systemics ************************************************************/
/**********************************************************************************************************************************/

// Includes
#include <malloc.h>
#include <string.h>
#include "services.h"
#include "waddef.h"
#include "wadhdr.h"

/**********************************************************************************************************************************/
/*********************************************************** Structures ***********************************************************/
/**********************************************************************************************************************************/

// Raw WAD file header
typedef struct
{
	char raw_magic[4];											// magic string ("IWAD" or "PWAD")
	SINT32 raw_num_entries;										// entry count (Little-Endian Signed 32-Bits)
	SINT32 raw_dir_start;										// start of directory (Little-Endian Signed 32-Bits)
}
RWH_t;

/**********************************************************************************************************************************/
/************************************************************ Utility *************************************************************/
/**********************************************************************************************************************************/

// Convert magic string as raw (4 bytes null-padded) to C string format
static void strfrommagic (char *str, const char *magic)
{
	(void) memmove(str,magic,4);
	str[4]=NUL;
}

// Convert entry name as C string to raw (4 bytes null-padded) format
static void magicfromstr (char *magic, const char *str)
{
	size_t str_len;
	/*************/
	str_len=strlen(str);
	if (str_len>4)
	{
		FatalAbort(1,__FILE__,__LINE__,"Magic string (\"%s\") in WAD could not have been longer that 4 characters",str);
	}
	(void) memmove(magic,str,str_len);
	(void) memset(&magic[str_len],0,4-str_len);
}

/**********************************************************************************************************************************/
/************************************************** WAD File Header Maintanance ***************************************************/
/**********************************************************************************************************************************/

// Initialise a WAD file header
void WADInitHeader (
	WADHDR_t **WADHDRp)											// WAD file header to be initialised
{
	// VARIABLES
	size_t sizeb;
	WADHDR_t *objptr;

	// PRIMARY ERROR CHECK
	if (WADHDRp==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to initialise WAD file header with NULL argument pointer");
	}
	if (*WADHDRp!=NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to initialise WAD file header with non-NULL argument");
	}

	// ALLOCATE MEMORY FOR THE WADHDR
	sizeb=sizeof(WADHDR_t);
	objptr=malloc(sizeb);
	if (objptr==NULL)
	{
		ErrorAbort(1,__FILE__,__LINE__,"Could not allocate %ld bytes for WAD file header",sizeb);
	}
	(void) memset(objptr,0,sizeb);

	// INITIALISE THE WADHDR STATIC FIELDS
	(void) strcpy(objptr->magic,"");
	objptr->num_entries=0;
	objptr->dir_start=0;

	// RETURN RESULT
	*WADHDRp=objptr;
}

// Deinitialise a WAD file header
void WADDoneHeader (
	WADHDR_t **WADHDRp)											// WAD file header to be deinitialised
{
	// PRIMARY ERROR CHECK
	if (WADHDRp==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to deinitialise WAD file header with NULL argument pointer");
	}
	if (*WADHDRp==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to deinitialise WAD file header with NULL argument");
	}

	// DEALLOCATE MEMORY FOR THE WADHDR
	free(*WADHDRp);

	// RETURN RESULT
	*WADHDRp=NULL;
}

/**********************************************************************************************************************************/
/***************************************************** WAD File Header Usage ******************************************************/
/**********************************************************************************************************************************/

// Read a WAD file header into memory from a file
void WADReadHeader (
	WADHDR_t *WADHDR,											// WAD file header to read from file
	FILE *infile)												// file to read WAD file header from
{
	// DOCUMENTATION
	/*
		PRE:
		File must be positioned at BOF [which is where the WAD
		file format specifies the header to be located].

		POST:
		File is positioned just after the header. By common
		usage this file position is the start of the data lumps
		in the file, but according to the WAD file format this
		need not be so. Any file CREATED with this library will
		always follow the convention.
	*/

	// VARIABLES
	RWH_t RWH;
	SINT32 SINT32_num_entries;
	SINT32 SINT32_dir_start;

	// READ THE RAW WAD HEADER
	if (fread(&RWH,sizeof(RWH_t),1,infile)<1)
	{
		ErrorAbort(1,__FILE__,__LINE__,"Could not read from file");
	}

	// SET THE MAGIC STRING
	strfrommagic(WADHDR->magic,RWH.raw_magic);

	// SET THE NUMBER OF ENTRIES
	SINT32_num_entries=RDLend32SV(RWH.raw_num_entries);
	if (SINT32_num_entries<0)
	{
		ErrorAbort(1,__FILE__,__LINE__,"Negative entry count (%ld) in WAD file header",SINT32_num_entries);
	}
	WADHDR->num_entries=(size_t)(SINT32_num_entries);

	// SET THE DIRECTORY OFFSET
	SINT32_dir_start=RDLend32SV(RWH.raw_dir_start);
	if (SINT32_dir_start<0)
	{
		ErrorAbort(1,__FILE__,__LINE__,"Negative directory offset (%ld) in WAD file header",SINT32_dir_start);
	}
	WADHDR->dir_start=(fofs_t)SINT32_dir_start;
}

// Write a WAD file header from memory out to a file
void WADWriteHeader (
	const WADHDR_t *WADHDR,										// WAD file header to write to file
	FILE *outfile)												// file to write WAD file header to
{
	// DOCUMENTATION
	/*
		PRE:
		File must be positioned at BOF [which is where the WAD
		file format specifies the header to be located].

		POST:
		File is positioned just after the header. By common
		usage this file position is the start of the data lumps
		in the file, but according to the WAD file format this
		need not be so. Any file CREATED with this library will
		always follow the convention.
	*/

	// VARIABLES
	RWH_t RWH;

	// SET THE MAGIC STRING
	magicfromstr(RWH.raw_magic,WADHDR->magic);

	// SET THE NUMBER OF ENTRIES
	if (WADHDR->num_entries>(size_t)(SINT32_MAX))
	{
		ErrorAbort(1,__FILE__,__LINE__,"Entry count (%lu) will not fit into a 32-bit signed integer",WADHDR->num_entries);
	}
	RWH.raw_num_entries=RDLend32SV((SINT32)(WADHDR->num_entries));

	// SET THE DIRECTORY OFFSET
	if (WADHDR->dir_start>SINT32_MAX)
	{
		ErrorAbort(1,__FILE__,__LINE__,"Directory offset (%lu) will not fit into a 32-bit signed integer",WADHDR->dir_start);
	}
	RWH.raw_dir_start=RDLend32SV((SINT32)(WADHDR->dir_start));

	// WRITE THE RAW WAD HEADER
	if (fwrite(&RWH,sizeof(RWH_t),1,outfile)<1)
	{
		ErrorAbort(1,__FILE__,__LINE__,"Could not write to file");
	}
}

/**********************************************************************************************************************************/
/********************************************************** End of File ***********************************************************/
/**********************************************************************************************************************************/
