/**********************************************************************************************************************************/
/********************************************************* Documentation **********************************************************/
/**********************************************************************************************************************************/

/*

Wad directory

*/

/**********************************************************************************************************************************/
/*********************************************************** Systemics ************************************************************/
/**********************************************************************************************************************************/

// Includes
#include <malloc.h>
#include <string.h>
#include <stdio.h>
#include "services.h"
#include "waddef.h"
#include "wadlin.h"
#include "wadlst.h"
#include "wadlnl.h"
#include "waddir.h"

/**********************************************************************************************************************************/
/*********************************************************** Structures ***********************************************************/
/**********************************************************************************************************************************/

// Raw WAD directory entry
typedef struct
{
	SINT32 raw_start;											// start of entry (Little-Endian Signed 32-Bits)
	SINT32 raw_length;											// length of entry (Little-Endian Signed 32-Bits)
	packed_name_t raw_name;										// name of entry (NULL padded to 8 characters)
}
RWDentry_t;

// Raw WAD directory
typedef struct
{
	SINT32 raw_count;											// number of entries (Little-Endian Signed 32-Bits)
	RWDentry_t *raw_entries;									// entries
}
RWD_t;

/**********************************************************************************************************************************/
/************************************************************ Utility *************************************************************/
/**********************************************************************************************************************************/

// Convert entry name as raw (8 bytes null-padded) to C string format
static void strfromname (char *str, const char *name)
{
	(void) memmove(str,name,8);
	str[8]=NUL;
}

// Convert entry name as C string to raw (8 bytes null-padded) format
static void namefromstr (char *name, const char *str)
{
	size_t str_len;
	/*************/
	str_len=strlen(str);
	if (str_len>8)
	{
		FatalAbort(1,__FILE__,__LINE__,"Entry name (\"%s\") in WAD could not have been longer that 8 characters",str);
	}
	(void) memmove(name,str,str_len);
	(void) memset(&name[str_len],0,8-str_len);
}

/**********************************************************************************************************************************/
/************************************************* WAD File Directory Maintanance *************************************************/
/**********************************************************************************************************************************/

#define WADDIR_APPORTION 64
#define WADDIR_INCREMENT 64

// Initialise a WAD file directory
void WADInitDirectory (
	WADDIR_t **WADDIRp)											// WAD file directory to be initialised
{
	// VARIABLES
	size_t sizeb;
	WADDIR_t *objptr;
	WADDIRentry_t *entptr;

	// PRIMARY ERROR CHECK
	if (WADDIRp==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to initialise WAD file directory with NULL argument pointer");
	}
	if (*WADDIRp!=NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to initialise WAD file directory with non-NULL argument");
	}

	// ALLOCATE MEMORY FOR THE WADDIR
	sizeb=sizeof(WADDIR_t);
	objptr=malloc(sizeb);
	if (objptr==NULL)
	{
		ErrorAbort(1,__FILE__,__LINE__,"Could not allocate %ld bytes for WAD file directory",sizeb);
	}
	(void) memset(objptr,0,sizeb);

	// ALLOCATE MEMORY FOR THE WADDIR ENTRIES
	sizeb=WADDIR_APPORTION*sizeof(WADDIRentry_t);
	entptr=malloc(sizeb);
	if (entptr==NULL)
	{
		ErrorAbort(1,__FILE__,__LINE__,"Could not allocate %ld bytes for WAD file directory entries",sizeb);
	}
	(void) memset(entptr,0,sizeb);

	// INITIALISE THE WADDIR STATIC FIELDS
	objptr->limit=WADDIR_APPORTION;
	objptr->count=0;
	objptr->reuse=FALSE;
	objptr->entries=entptr;

	// RETURN RESULT
	*WADDIRp=objptr;
}

// Deinitialise a WAD file directory
void WADDoneDirectory (
	WADDIR_t **WADDIRp)											// WAD file directory to be deinitialised
{
	// PRIMARY ERROR CHECK
	if (WADDIRp==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to deinitialise WAD file directory with NULL argument pointer");
	}
	if (*WADDIRp==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to deinitialise WAD file directory with NULL argument");
	}
	if ((*WADDIRp)->entries==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to deinitialise WAD file directory with NULL entries");
	}
#ifdef PARANOID
	if ((*WADDIRp)->count>(*WADDIRp)->limit)
	{
		FatalAbort(1,__FILE__,__LINE__,"WAD file directory is corrupt");
	}
#endif

	// DEALLOCATE MEMORY FOR THE WADDIR ENTRIES
	free((*WADDIRp)->entries);

	// DEALLOCATE MEMORY FOR THE WADDIR
	free(*WADDIRp);

	// RETURN RESULT
	*WADDIRp=NULL;
}

/**********************************************************************************************************************************/
/**************************************************** WAD File Directory Usage ****************************************************/
/**********************************************************************************************************************************/

// Read a WAD file directory into memory from a file
void WADReadDirectory (
	WADDIR_t *WADDIR,											// WAD file directory to read from file
	FILE *infile,												// file to read WAD file directory from
	size_t count)												// number of entries in WAD file directory
{
	// DOCUMENTATION
	/*
		PRE:
		File must be positioned at the directory offset, which
		is ultimately obtained from the WAD file header.

		POST:
		File is positioned just after the directory. By common
		usage this file position is EOF, but according to the
		WAD file format this need not be so. Any file CREATED
		with this library will always follow the convention.
	*/

	// VARIABLES
	RWDentry_t *rentries;
	WADDIRentry_t *newptr;
	size_t i,size,sizeb,rsizeb;

	// PRIMARY ERROR CHECK
	if (WADDIR==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to read in WAD file directory with NULL argument");
	}
	if (WADDIR->entries==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to read in WAD file directory with NULL entries");
	}
#ifdef PARANOID
	if (WADDIR->count>WADDIR->limit)
	{
		FatalAbort(1,__FILE__,__LINE__,"WAD file directory is corrupt");
	}
#endif
	if (infile==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to read in WAD file directory with NULL input file argument");
	}

	// DEALLOCATE MEMORY FOR THE OLD WADDIR ENTRIES
	free(WADDIR->entries);

	// ALLOCATE MEMORY FOR THE NEW WADDIR ENTRIES
	size=count+WADDIR_INCREMENT;
	sizeb=size*sizeof(WADDIRentry_t);
	newptr=malloc(sizeb);
	if (newptr==NULL)
	{
		ErrorAbort(1,__FILE__,__LINE__,"Could not allocate %ld bytes for replacement WAD file directory entries",sizeb);
	}
	(void) memset(newptr,0,sizeb);
	WADDIR->entries=newptr;
	WADDIR->limit=size;

	// SET THE NEW WADDIR COUNT FIELD
	WADDIR->count=count;

	// ALLOCATE THE MEMORY FOR THE RAW WAD DIRECTORY ENTRIES
	rsizeb=count*sizeof(RWDentry_t);
	rentries=malloc(rsizeb);
	if (rentries==NULL)
	{
		ErrorAbort(1,__FILE__,__LINE__,"Could not allocate %ld bytes for raw WAD file directory entries",rsizeb);
	}

	// READ THE RAW WAD DIRECTORY ENTRIES
	if (fread(rentries,sizeof(RWDentry_t),count,infile)<count)
	{
		ErrorAbort(1,__FILE__,__LINE__,"Could not read from file");
	}

	// COPY RAW WAD DIRECTORY ENTRIES INTO THE WAD DIRECTORY
	for (i=0;i<count;i++)
	{
		// VARIABLES
		SINT32 SINT32_start;
		SINT32 SINT32_length;

		// SET THE ENTRY NAME
		strfromname(WADDIR->entries[i].name,rentries[i].raw_name);

		// SET THE ENTRY START
		SINT32_start=RDLend32SV(rentries[i].raw_start);
		if (SINT32_start<0)
		{
			ErrorAbort(1,__FILE__,__LINE__,"Negative entry start (%ld) in WAD directory entry",SINT32_start);
		}
		WADDIR->entries[i].start=(fofs_t)SINT32_start;

		// SET THE ENTRY LENGTH
		SINT32_length=RDLend32SV(rentries[i].raw_length);
		if (SINT32_length<0)
		{
			ErrorAbort(1,__FILE__,__LINE__,"Negative entry length (%ld) in WAD directory entry",SINT32_length);
		}
		WADDIR->entries[i].length=(size_t)(SINT32_length);

		// INITIALISE  THE OTHER MEMBERS
		WADDIR->entries[i].type=TYPE_UNKNOWN;
		WADDIR->entries[i].list=LIST_UNKNOWN;
	}

	// DEALLOCATE THE MEMORY FOR THE RAW WAD DIRECTORY ENTRIES
	free(rentries);
	rentries=NULL;

	// INITIALISE THE LUMP REUSE FLAG
	WADDIR->reuse=FALSE;
}

// Scan a WAD file directory and identify its entries
void WADScanDirectory (
	WADDIR_t *WADDIR,											// WAD file directory to scan
	LST_t *LST,													// symbol table resulting from the scan of that directory
	bool_t identify_names,										// identify screen images by name (READ MANUAL BEFORE USING!)
	bool_t identify_pages,										// identify screen images as pages (READ MANUAL BEFORE USING!)
	bool_t identify_graphics,									// identify screen images as graphics (READ MANUAL BEFORE USING!)
	bool_t identify_voices,										// identify loose STRIFE VOICES by name (READ MANUAL BEFORE USING!)
	bool_t detect_sounds,										// detect sounds by content (READ MANUAL BEFORE USING!)
	bool_t detect_musics,										// detect musics by content (READ MANUAL BEFORE USING!)
	bool_t detect_graphics,										// detect graphics by content (READ MANUAL BEFORE USING!)
	bool_t recognised_names,									// identify sounds/musics/dialogs/conversations by recognised names
	bool_t loose_markers,										// allow nonstandard list marker characters
	bool_t named_markers,										// loose markers use names only (READ MANUAL BEFORE USING!)
	bool_t tolerate_multiples,									// do not treat multiple instances of structured lists as an error
	bool_t quiet_multiples,										// do not treat multiple instances of structured lists as a problem
	bool_t declassify_pnames,									// treat PNAMES as an unclassified lump (separate from TEXTURES)
	bool_t loose_headers,										// allow nonstandard map name headers (not just E\?M\? and MAP\?\?)
	bool_t quiet_headers,										// do not warn about non-empty map name headers
	game_t game,												// game for which (WAD whose lump names these are) was designed
	FILE *infile)												// file that directory was read from
{
	// VARIABLES
	size_t i,count;
	LNL_t *LNL=NULL;

	// PRIMARY ERROR CHECK
	if (WADDIR==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to scan a WAD file directory with NULL argument");
	}
	if (WADDIR->entries==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to scan a WAD file directory with NULL entries");
	}
#ifdef PARANOID
	if (WADDIR->count>WADDIR->limit)
	{
		FatalAbort(1,__FILE__,__LINE__,"WAD file directory is corrupt");
	}
#endif

	// INITIALISE
	count=WADDIR->count;
	LNLInit(&LNL);

	// PRESCAN DIRECTORY BY NAMES
	EventState(VERBOSITY_DETAILS,">>>>>>>> Prescanning directory by entry names");
	for (i=0;i<count;i++)
	{
		// VARIABLES
		type_t type;
		bool_t maybe_sound;
		bool_t maybe_music;
		bool_t maybe_graphic;

		// ASSUME NOT A SOUND OR MUSIC FOR NOW
		maybe_sound=FALSE;
		maybe_music=FALSE;
		maybe_graphic=FALSE;

		// COULD THE LUMP BE A SOUND, MUSIC OR GRAPHIC?
		if (
		    (detect_sounds && (WADDIR->entries[i].length>=sizeof(sound_header_t))) ||
		    (detect_musics && (WADDIR->entries[i].length>=sizeof(music_header_t))) ||
			(detect_graphics && (WADDIR->entries[i].length>=sizeof(picture_header_t)))
			)
		{
			// VARIABLES
			fofs_t fpos;
			void *lump_header;

			// GET HEADER SIZE
			size_t header_size=0;
			if (sizeof(sound_header_t)>header_size)
			{
				header_size=sizeof(sound_header_t);
			}
			if ((WAVE_MAGIC_SIZE+sizeof(wave_header_t))>header_size)
			{
				header_size=(WAVE_MAGIC_SIZE+sizeof(wave_header_t));
			}
			if (sizeof(music_header_t)>header_size)
			{
				header_size=sizeof(music_header_t);
			}
			if ((MTRK_MAGIC_SIZE+sizeof(midi_header_t))>header_size)
			{
				header_size=(MTRK_MAGIC_SIZE+sizeof(midi_header_t));
			}
			if (sizeof(picture_header_t)>header_size)
			{
				header_size=sizeof(picture_header_t);
			}

			// GET CURRENT FILE POSITION
			fpos=ftell(infile);
			if (fpos<0)
			{
				ErrorAbort(1,__FILE__,__LINE__,"Could not get current position in file");
			}

			// SEEK TO THE ENTRY
			if (fseek(infile,WADDIR->entries[i].start,SEEK_SET)!=0)
			{
				ErrorAbort(1,__FILE__,__LINE__,"Could not seek in file");
			}

			// ALLOCATE THE NECESSARY MEMORY
			lump_header=malloc(header_size);
			if (lump_header==NULL)
			{
				ErrorAbort(1,__FILE__,__LINE__,"Could not allocate %ld bytes for lump header",header_size);
			}

			// READ IN THE LUMP HEADER
			if (fread(lump_header,header_size,1,infile)<1)
			{
				ErrorAbort(1,__FILE__,__LINE__,"Could not read from file");
			}

			// RESTORE PREVIOUS FILE POSITION
			if (fseek(infile,fpos,SEEK_SET)!=0)
			{
				ErrorAbort(1,__FILE__,__LINE__,"Could not seek in file");
			}

			// TRY AND AUTO-DETECT SOUND, GRAPHIC OR MUSIC LUMP
			if (detect_sounds && maybesound(lump_header,WADDIR->entries[i].length))
			{
//				EventState(VERBOSITY_WARNINGS,"Type of entry %s (entry %ld) resolved as SOUND",WADDIR->entries[i].name,i);
//				type=TYPE_SOUND;
				maybe_sound=TRUE;
			}
			else if (detect_musics && maybemusic(lump_header,WADDIR->entries[i].length))
			{
//				EventState(VERBOSITY_WARNINGS,"Type of entry %s (entry %ld) resolved as MUSIC",WADDIR->entries[i].name,i);
//				type=TYPE_MUSIC;
				maybe_music=TRUE;
			}
			else if (detect_graphics && maybegraphic(lump_header,WADDIR->entries[i].length,
													 infile,WADDIR->entries[i].start,fpos))
			{
//				EventState(VERBOSITY_WARNINGS,"Type of entry %s (entry %ld) resolved as GRAPHICS",WADDIR->entries[i].name,i);
//				type=TYPE_MUSIC;
				maybe_graphic=TRUE;
			}
			else
			{
				// Not a special case.
				// No action needed.
			}

			// DEALLOCATE THE NECESSARY MEMORY
			free(lump_header);
		}

		// GET TYPE OF ENTRY BY SYNTAX THEN NAME
		type=LINIdentifyLumpTypeBySyntaxThenName (
			WADDIR->entries[i].name,							// lump to be identified
			i,													// index of entry in WAD directory
			identify_names,										// identify screen images by name (READ MANUAL BEFORE USING!)
			identify_pages,										// identify screen images as pages (READ MANUAL BEFORE USING!)
			identify_graphics,									// identify screen images as graphics (READ MANUAL BEFORE USING!)
			identify_voices,									// identify loose STRIFE VOICES by name (READ MANUAL BEFORE USING!)
			detect_sounds,										// detect sounds by content (READ MANUAL BEFORE USING!)
			detect_musics,										// detect musics by content (READ MANUAL BEFORE USING!)
			detect_graphics,									// detect graphics by content (READ MANUAL BEFORE USING!)
			recognised_names,									// identify sounds/musics/dialogs/conversations by recognised names
			loose_markers,										// allow nonstandard list marker characters
			named_markers,										// loose markers use names only (READ MANUAL BEFORE USING!)
			loose_headers,										// allow nonstandard map name headers (not just E\?M\? and MAP\?\?)
			(WADDIR->entries[i].length==0),						// lump has zero size
			maybe_sound,										// lump data indicates a sound file
			maybe_music,										// lump data indicates a music entry
			maybe_graphic,										// lump data indicates a graphic entry
			game);												// game for which WAD was designed

		// RESOLVE ANY OUTSTANDING PAGE/IMAGE CONFLICTS
		if (type==TYPE_AUTO)
		{
			// VARIABLES
			void *lump;
			fofs_t fpos;

			// GET CURRENT FILE POSITION
			fpos=ftell(infile);
			if (fpos<0)
			{
				ErrorAbort(1,__FILE__,__LINE__,"Could not get current position in file");
			}

			// SEEK TO THE ENTRY
			if (fseek(infile,WADDIR->entries[i].start,SEEK_SET)!=0)
			{
				ErrorAbort(1,__FILE__,__LINE__,"Could not seek in file");
			}

			// ALLOCATE THE NECESSARY MEMORY
			lump=malloc(WADDIR->entries[i].length);
			if (lump==NULL)
			{
				ErrorAbort(1,__FILE__,__LINE__,"Could not allocate %ld bytes for lump buffer",WADDIR->entries[i].length);
			}

			// READ IN THE LUMP
			if (fread(lump,WADDIR->entries[i].length,1,infile)<1)
			{
				ErrorAbort(1,__FILE__,__LINE__,"Could not read from file");
			}

			// RESTORE PREVIOUS FILE POSITION
			if (fseek(infile,fpos,SEEK_SET)!=0)
			{
				ErrorAbort(1,__FILE__,__LINE__,"Could not seek in file");
			}

			// AUTO-DETECT THE TYPE OF LUMP
			if (ispage(lump,WADDIR->entries[i].length))
			{
				EventState(VERBOSITY_WARNINGS,"Type of entry %s (entry %ld) resolved as PAGE",WADDIR->entries[i].name,i);
				type=TYPE_PAGE;
			}
			else
			{
				EventState(VERBOSITY_WARNINGS,"Type of entry %s (entry %ld) resolved as GRAPHIC",WADDIR->entries[i].name,i);
				type=TYPE_GRAPHIC;
			}

			// DEALLOCATE THE NECESSARY MEMORY
			free(lump);
		}

		// SET THE BASIC ENTRY TYPE
		WADDIR->entries[i].type=type;
	}

	//
	// COPY THE DIRECTORY ENTRY NAMES INTO A LUMP NAME LIST
	// AND COMPENSATE FOR ANY AUTO-DETECTION OF ENTRY TYPE.
	//
	for (i=0;i<count;i++)
	{
		WADDIRentry_t *entry;
		bool_t was_detected;
		/*******************/
		entry=&WADDIR->entries[i];
		if (entry->type==TYPE_DETECTED_SOUND)
		{
			entry->type=TYPE_SOUND;
			was_detected=TRUE;
		}
		else if (entry->type==TYPE_DETECTED_MUSIC)
		{
			entry->type=TYPE_MUSIC;
			was_detected=TRUE;
		}
		else if (entry->type==TYPE_DETECTED_GRAPHIC)
		{
			entry->type=TYPE_GRAPHIC;
			was_detected=TRUE;
		}
		else
		{
			was_detected=FALSE;
		}
		LNLAdd (
			LNL,												// lump name list to which to add lump name
			entry->name,										// name of lump
			i,													// index of entry within directory
			entry->type,										// lump name type
			entry->list,										// list that lump name belongs to
			((entry->length==0)?TRUE:FALSE),					// lump has zero size
			was_detected);										// lump type was auto-detected
	}

	// SCAN THE LUMP NAME LIST AND IDENTIFY ITS CONTENTS
	LNLScan (
		LNL,													// lump name list whose contents are to be identified
		LST,													// symbol table to be created from this list
		recognised_names,										// identify sounds/musics/dialogs/conversations by recognised names
		tolerate_multiples,										// do not treat multiple instances of structured lists as an error
		quiet_multiples,										// do not treat multiple instances of structured lists as a problem
		declassify_pnames,										// treat PNAMES as an unclassified lump (separate from TEXTURES)
		loose_headers,											// allow nonstandard map name headers (not just E\?M\? and MAP\?\?)
		quiet_headers,											// do not warn about non-empty map name headers
		game);													// game for which (WAD whose lump names these are) was designed

	// COPY THE DIRECTORY ENTRY METRICS FROM THE LUMP NAME LIST
	for (i=0;i<count;i++)
	{
		WADDIRentry_t *entry;
		char dummy_char[9]="";
		size_t dummy_size_t=0;
		bool_t dummy_bool_t=FALSE;
		/************************/
		entry=&WADDIR->entries[i];
		LNLGet (
			LNL,												// lump name list from which to get lump name
			i,													// which entry in the lump name list to get
			dummy_char,											// lump name
			&dummy_size_t,										// index of entry within directory
			&entry->type,										// lump name type
			&entry->list,										// list that lump name belongs to
			&dummy_bool_t);										// lump should be deleted (after certain types of scan)
	}

	// DEINITIALISE
	LNLDone(&LNL);
}

// Copy a WAD file directory
void WADCopyDirectory (
	WADDIR_t *outWADDIR,										// WAD file directory that is the copy
	const WADDIR_t *inWADDIR)									// WAD file directory to copy
{
	// VARIABLES
	size_t sizeb;
	WADDIRentry_t *newptr;

	// PRIMARY ERROR CHECK
	if (inWADDIR==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to copy WAD file directory with NULL input argument");
	}
	if (inWADDIR->entries==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to copy WAD file directory with NULL input entries");
	}
#ifdef PARANOID
	if (inWADDIR->count>inWADDIR->limit)
	{
		FatalAbort(1,__FILE__,__LINE__,"Input WAD file directory is corrupt");
	}
#endif
	if (outWADDIR==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to copy WAD file directory with NULL output argument");
	}
	if (outWADDIR->entries==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to copy WAD file directory with NULL output entries");
	}
#ifdef PARANOID
	if (outWADDIR->count>outWADDIR->limit)
	{
		FatalAbort(1,__FILE__,__LINE__,"Output WAD file directory is corrupt");
	}
#endif

	// DEALLOCATE MEMORY FOR THE OLD WADDIR ENTRIES
	free(outWADDIR->entries);

	// COPY THE DIRECTORY STATIC FIELDS
	(void) memmove(outWADDIR,inWADDIR,sizeof(WADDIR_t));

	// CLEAR NEW ENTRIES POINTER AND LIMIT
	outWADDIR->entries=NULL;

	// ALLOCATE MEMORY FOR THE NEW ENTRIES
	sizeb=(outWADDIR->limit)*sizeof(WADDIRentry_t);
	newptr=malloc(sizeb);
	if (newptr==NULL)
	{
		ErrorAbort(1,__FILE__,__LINE__,"Could not allocate %ld bytes for replacement WAD file directory entries",sizeb);
	}
	(void) memset(&newptr[outWADDIR->count],0,((outWADDIR->limit)-(outWADDIR->count))*sizeof(WADDIRentry_t));
	outWADDIR->entries=newptr;

	// COPY THE NEW DIRECTORY ENTRIES OVER THE OLD ONES
	(void) memmove(outWADDIR->entries,inWADDIR->entries,((outWADDIR->count)*sizeof(WADDIRentry_t)));
}

// Normalise a WAD file directory
void WADNormaliseDirectory (
	WADDIR_t *WADDIR,											// WAD file directory to normalise
	LST_t *LST,													// symbol table resulting from the scan of that directory
	bool_t remove_duplicates,									// remove all but the first of entries having the same name
	bool_t identify_voices,										// identify loose STRIFE VOICES by name (READ MANUAL BEFORE USING!)
	bool_t keep_wintex,											// do not remove _DEUTEX_ lump
	bool_t keep_platform,										// do not remove PLATFORM lump
	bool_t keep_history,										// do not remove HISTORY lump
	bool_t keep_tagdesc,										// do not remove TAGDESC lump
	bool_t keep_pcsfx,											// do not remove PC speaker sound effects
	bool_t keep_doubles,										// do not normalise double-letter list markers (e.g., PP_START)
	bool_t keep_borders,										// do not remove sub-list border markers (e.g., F1_START)
	bool_t keep_empties,										// do not remove empty structured lists
	bool_t remove_scripts,										// remove all SCRIPTS and SCRIPTnn entries
	bool_t force_removal)										// remove duplicate entries even if in different lists
{
	// VARIABLES
	WADDIRentry_t *entries;
	size_t i,j,count,dummy_deleted;
	LNL_t *LNL=NULL;
	bool_t eraseit;

	// PRIMARY ERROR CHECK
	if (WADDIR==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to normalise WAD file directory with NULL argument");
	}
	if (WADDIR->entries==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to normalise WAD file directory with NULL entries");
	}
#ifdef PARANOID
	if (WADDIR->count>WADDIR->limit)
	{
		FatalAbort(1,__FILE__,__LINE__,"WAD file directory is corrupt");
	}
#endif

	// INITIALISE THE LUMP NAME LIST
	LNLInit(&LNL);

	// INITIALISE OTHER THINGS
	count=WADDIR->count;

	// COPY THE DIRECTORY ENTRY NAMES INTO THE LUMP NAME LIST
	for (i=0;i<count;i++)
	{
		WADDIRentry_t *entry;
		/*******************/
		entry=&WADDIR->entries[i];
		LNLAdd (
			LNL,												// lump name list to which to add lump name
			entry->name,										// name of lump
			i,													// index of entry within directory
			entry->type,										// lump name type
			entry->list,										// list that lump name belongs to
			((entry->length==0)?TRUE:FALSE),					// lump has zero size
			FALSE);												// lump type was auto-detected **not valid after call to LNLScan**
	}

	// NORMALISE THE LUMP NAME LIST
	LNLNormalise (
		LNL,													// lump name list to normalise
		LST,													// symbol table resulting from the scan of that list
		remove_duplicates,										// remove all but the first of entries having the same name
		identify_voices,										// identify loose STRIFE VOICES by name (READ MANUAL BEFORE USING!)
		keep_wintex,											// do not remove _DEUTEX_ lump
		keep_platform,											// do not remove PLATFORM lump
		keep_history,											// do not remove HISTORY lump
		keep_tagdesc,											// do not remove TAGDESC lump
		keep_pcsfx,												// do not remove PC speaker sound effects
		keep_doubles,											// do not normalise doubled-letter list markers (e.g., PP_START)
		keep_borders,											// do not remove sub-list border markers (e.g., F1_START)
		keep_empties,											// do not remove empty structured lists
		remove_scripts,											// remove all SCRIPTS and SCRIPTnn entries
		force_removal,											// remove duplicate entries even if in different lists
		&dummy_deleted);										// number of entries deleted by normalising

	// UPDATE THE DIRECTORY FROM THE LUMP NAME LIST
	j=0;
	entries=WADDIR->entries;
	for (i=0;i<count;i++)
	{
		size_t dummy_size_t=0;
		type_t dummy_type=TYPE_UNKNOWN;
		list_t dummy_list=LIST_UNKNOWN;
		/*****************************/
		LNLGet (
			LNL,												// lump name list from which to get lump name
			i,													// which entry in the lump name list to get
			entries[j].name,									// lump name (may have been changed)
			&dummy_size_t,										// index of entry within directory
			&dummy_type,										// lump name type
			&dummy_list,										// list that lump name belongs to
			&eraseit);											// lump should be deleted (after certain types of scan)
		if (eraseit)
		{
			(void) memmove(&entries[j],&entries[j+1],(((count-1)-j)*sizeof(WADDIRentry_t)));
		}
		else
		{
			j++;
		}
	}
	WADDIR->count=j;

	// DEINITIALISE THE LUMP NAME LIST
	LNLDone(&LNL);

	// REALLOCATE THE DIRECTORY ACCORDING TO THE NEW CONTENTS
	if (WADDIR->limit!=(WADDIR->count+WADDIR_INCREMENT))
	{
		WADDIRentry_t *newptr;
		bool_t grow;
		char growa[7];
		size_t size,sizeb,diffb;
		/**********************/
		size=WADDIR->count+WADDIR_INCREMENT;
		sizeb=size*sizeof(WADDIRentry_t);
		grow=(size>WADDIR->limit);
		(void) strcpy(growa,(grow?"grow":"shrink"));
		diffb=(grow?(size-WADDIR->limit):(WADDIR->limit-size))*sizeof(WADDIRentry_t);
		newptr=realloc(WADDIR->entries,sizeb);
		if (newptr==NULL)
		{
			ErrorAbort(1,__FILE__,__LINE__,"Could not %s WAD file directory entries by %ld bytes (from %ld bytes)",growa,diffb,sizeb);
		}
		WADDIR->entries=newptr;
		if (grow)
		{
			(void) memset(&WADDIR->entries[WADDIR->limit],0,diffb);
		}
		WADDIR->limit=size;
	}
}

// Sort a WAD file directory
void SortWadDirectory (
	WADDIR_t *WADDIR,											// WAD file directory to sort
	const LST_t *LST,											// symbol table resulting from the scan of that directory
	sort_t sort,												// sort order to apply to entries in the output file
	game_t game)												// game for which (WAD whose lump names these are) was designed
{
	// VARIABLES
	size_t sizeb,i,count;
	WADDIRentry_t *entries,*tempentries;
	LNL_t *LNL=NULL;

	// PRIMARY ERROR CHECK
	if (WADDIR==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to sort WAD file directory with NULL argument");
	}
	if (WADDIR->entries==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to sort WAD file directory with NULL entries");
	}
#ifdef PARANOID
	if (WADDIR->count>WADDIR->limit)
	{
		FatalAbort(1,__FILE__,__LINE__,"WAD file directory is corrupt");
	}
#endif

	// HANDLE TRIVIAL CASE
	if (sort==SORT_NONE)
	{
		return;
	}

	// INITIALISE
	count=WADDIR->count;
	entries=WADDIR->entries;
	tempentries=NULL;
	LNLInit(&LNL);

	// INITIALISE OTHER THINGS
	count=WADDIR->count;

	// COPY THE DIRECTORY ENTRY NAMES INTO THE LUMP NAME LIST
	for (i=0;i<count;i++)
	{
		WADDIRentry_t *entry;
		/*******************/
		entry=&WADDIR->entries[i];
		LNLAdd (
			LNL,												// lump name list to which to add lump name
			entry->name,										// name of lump
			i,													// index of entry within directory
			entry->type,										// lump name type
			entry->list,										// list that lump name belongs to
			((entry->length==0)?TRUE:FALSE),					// lump has zero size
			FALSE);												// lump type was auto-detected **not valid after call to LNLScan**
	}

	// SORT THE LUMP NAME LIST
	switch (sort)
	{
		case SORT_NONE:
			FatalAbort(1,__FILE__,__LINE__,"Trivial case (no sortation) should have already been handled by now");
			break;
		case SORT_LIST:
			LNLSortListOnly (
				LNL);											// lump name list to sort
			break;
		case SORT_ALPHABETIC:
			LNLSortListAlpha (
				LNL,											// lump name list to sort
				FALSE,											// sort order is: list then intrinsic (use alphabetic if none)
				game);											// game for which (WAD whose lump names these are) was designed
			break;
		case SORT_INTRINSIC:
			LNLSortListAlpha (
				LNL,											// lump name list to sort
				TRUE,											// sort order is: list then intrinsic (use alphabetic if none)
				game);											// game for which (WAD whose lump names these are) was designed
			break;
		default:
			FatalAbort(1,__FILE__,__LINE__,"Unrecognised value %d in variable \"sort\"",sort);
			break;
		////
	}

	// ALLOCATE NECESSARY MEMORY FOR THE TEMPORARY ENTRY LIST
	sizeb=count*sizeof(WADDIRentry_t);
	tempentries=malloc(sizeb);
	if (tempentries==NULL)
	{
		ErrorAbort(1,__FILE__,__LINE__,"Could not allocate %ld bytes for WAD file directory",sizeb);
	}

	// UPDATE THE DIRECTORY FROM THE LUMP NAME LIST
	entries=WADDIR->entries;
	for (i=0;i<count;i++)
	{
		size_t index;
		char name[9];
		type_t dummy_type=TYPE_UNKNOWN;
		list_t list=LIST_UNKNOWN;
		bool_t dummy_bool=FALSE;
		/*****************************/
		LNLGet (
			LNL,												// lump name list from which to get lump name
			i,													// which entry in the lump name list to get
			name,												// lump name (may have been changed)
			&index,												// index of entry within directory
			&dummy_type,										// lump name type
			&list,												// list that lump name belongs to
			&dummy_bool);										// lump should be deleted (after certain types of scan)
		EventState(VERBOSITY_DETAILS,"Adding entry %s to %s",name,LST->entries[list].name);
		(void) memmove(&tempentries[i],&entries[index],sizeof(WADDIRentry_t));
	}

	// COPY THE TEMPORARY ENTRY LIST BACK INTO THE DIRECTORY
	(void) memmove(entries,tempentries,sizeb);

	// DEINITIALISE
	free(tempentries);
	LNLDone(&LNL);
}

// Write a WAD file directory from memory out to a file
void WADWriteDirectory (
	const WADDIR_t *WADDIR,										// WAD file directory to write to file
	FILE *outfile)												// file to write directory to
{
	/*
		PRE:
		File must be positioned at the directory offset, which
		is ultimately obtained from the WAD file header.

		POST:
		File is positioned just after the directory. By common
		usage this file position is EOF, but according to the
		WAD file format this need not be so. Any file CREATED
		with this library will always follow the convention.
	*/

	// VARIABLES
	size_t i,count,rsizeb;
	RWDentry_t *rentries;

	// PRIMARY ERROR CHECK
	if (WADDIR==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to write out WAD file directory with NULL argument");
	}
	if (WADDIR->entries==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to write out WAD file directory with NULL entries");
	}
#ifdef PARANOID
	if (WADDIR->count>WADDIR->limit)
	{
		FatalAbort(1,__FILE__,__LINE__,"WAD file directory is corrupt");
	}
#endif
	if (outfile==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to write out WAD file directory with NULL output file argument");
	}

	// ALLOCATE THE MEMORY FOR THE RAW WAD DIRECTORY ENTRIES
	count=WADDIR->count;
	rsizeb=count*sizeof(RWDentry_t);
	rentries=malloc(rsizeb);
	if (rentries==NULL)
	{
		ErrorAbort(1,__FILE__,__LINE__,"Could not allocate %ld bytes for raw WAD file directory entries",rsizeb);
	}

	// COPY WAD DIRECTORY INTO THE RAW WAD DIRECTORY ENTRIES
	for (i=0;i<count;i++)
	{
		// SET THE ENTRY NAME
		namefromstr(rentries[i].raw_name,WADDIR->entries[i].name);

		// SET THE ENTRY START
		if (WADDIR->entries[i].start>SINT32_MAX)
		{
			ErrorAbort(1,__FILE__,__LINE__,"Entry start (%lu) will not fit into a 32-bit signed integer",WADDIR->entries[i].start);
		}
		rentries[i].raw_start=RDLend32SV((SINT32)(WADDIR->entries[i].start));

		// SET THE ENTRY LENGTH
		if (WADDIR->entries[i].length>(size_t)(SINT32_MAX))
		{
			ErrorAbort(1,__FILE__,__LINE__,"Entry length (%lu) will not fit into a 32-bit signed integer",WADDIR->entries[i].length);
		}
		rentries[i].raw_length=RDLend32SV((SINT32)(WADDIR->entries[i].length));
	}

	// WRITE THE RAW WAD DIRECTORY ENTRIES
	if (fwrite(rentries,sizeof(RWDentry_t),count,outfile)<count)
	{
		ErrorAbort(1,__FILE__,__LINE__,"Could not write to file");
	}

	// DEALLOCATE THE MEMORY FOR THE RAW WAD DIRECTORY ENTRIES
	free(rentries);
//	rentries=NULL;
}

/**********************************************************************************************************************************/
/********************************************************** End of File ***********************************************************/
/**********************************************************************************************************************************/
