/**********************************************************************************************************************************/
/********************************************************* Documentation **********************************************************/
/**********************************************************************************************************************************/

/*

File system routines

*/

/**********************************************************************************************************************************/
/*********************************************************** Systemics ************************************************************/
/**********************************************************************************************************************************/

// Includes
#include "services.h"
#include "filesys.h"

/**********************************************************************************************************************************/
/*************************************************** File Specification Parsing ***************************************************/
/**********************************************************************************************************************************/

// For some reason this is not in stdio
int fileno(FILE *);

// Initialise platform flag
#undef __SETPLATFORM__

// Platform specifics
#if defined(__WIN32__)
#define __SETPLATFORM__ 1
#include <malloc.h>

// Windows stuff
unsigned long __stdcall GetFullPathNameA (
	const char *lpfilespec,
	unsigned long nBufferLength,
	char *lpBuffer,
	char **lpFilePart);

// Return full filespec (allocated)
void FILEGetFullSpec (const char *infilespec, char **outfilespec)
{
	// METRICS
	#define BUFFER_FIXEDSIZE 17									// includes trailing null

	// VARIABLES
	char *ofspec;												// editable (under construction) copy of output filespec
	char tfspec[BUFFER_FIXEDSIZE];								// temporary filespec
	size_t sizeb;												// temporary variable used during allocation calculations
	unsigned long required_length;								// length required for resolved filespec
	unsigned long allocated_length;								// length allocated for resolved filespec
	char *dummy_string;											// ignores unwanted return parameter from GetFullPathName()

	// INITIALISE
	ofspec=NULL;												// editable (under construction) copy of output filespec
	tfspec[0]=NUL;												// temporary filespec

	// PRIMARY ERROR CHECK
	if (infilespec==NULL)
	{
		ErrorAbort(1,__FILE__,__LINE__,"Attempt to resolve full filespec with null input filespec argument");
	}
	if (outfilespec==NULL)
	{
		ErrorAbort(1,__FILE__,__LINE__,"Attempt to resolve full filepsec with null output filespec argument");
	}

	// HANDLE EXTREME CASES
	sizeb=1;
	if (infilespec[0]==NUL)
	{
		ofspec=malloc(sizeb);
		if (ofspec==NULL)
		{
			ErrorAbort(1,__FILE__,__LINE__,"Could not allocate 1 byte for zero-length output filespec");
		}
		ofspec[0]=NUL;
		*outfilespec=ofspec;
		return;
	}

	// GET REQUIRED LENGTH
	required_length=GetFullPathNameA(infilespec,BUFFER_FIXEDSIZE,tfspec,&dummy_string);
	if (required_length==0)
	{
		ErrorAbort(1,__FILE__,__LINE__,"Could not get length required for output filespec");
	}
	if (required_length>BUFFER_FIXEDSIZE)
	{
		allocated_length=required_length-1;
	}
	else
	{
		allocated_length=required_length;
	}

	// ALLOCATE THE NECESSARY MEMORY
	sizeb=allocated_length+1;
	ofspec=malloc(sizeb);
	if (ofspec==NULL)
	{
		ErrorAbort(1,__FILE__,__LINE__,"Could not allocate %ld bytes for output filespec",sizeb);
	}

	// GET THE FULL FILESPEC
#ifdef _WIN64
	if (sizeb>UINT32_MAX)
	{
		FatalAbort(1,__FILE__,__LINE__,"Value in \"sizeb\" is > 32 bits");
	}
#endif
	#ifdef __VISUALC__
	#pragma warning( push )
	#pragma warning( disable : 4267 )
	#endif
	required_length=GetFullPathNameA(infilespec,sizeb,ofspec,&dummy_string);
	#ifdef __VISUALC__
	#pragma warning( pop )
	#endif
	if (required_length!=allocated_length)
	{
		ErrorAbort(1,__FILE__,__LINE__,"Could not resolve filespec into fully-qualified form");
	}

	// RETURN THE RESULT
	*outfilespec=ofspec;
}

#endif // defined(__WIN32__)

// Platform specifics
#if defined(__CYGWIN__) || defined(__LINUX__)
#define __SETPLATFORM__ 1
#include <unistd.h>
#include <stdlib.h>
#include <malloc.h>
#include <errno.h>

// Return full filespec (allocated)
void FILEGetFullSpec (const char *infilespec, char **outfilespec)
{
	// METRICS
	#define BUFFER_INCREMENT 64									// buffer allocation quantum
	#define LINK_LIMIT 32										// assumed maximum number of symbolic links

	// VARIABLES
	char *i;													// current scan position
	char *ifspec;												// editable copy of input filespec
	char *ofspec;												// editable (under construction) copy of output filespec
	char *tfspec;												// temporary filespec
	char *lfspec;												// link filespec
	size_t ocount;												// current length of ofspec
	size_t olimit;												// current allocation (excluding two spare characters) of lfspec
	size_t lcount;												// current length of lfspec
	size_t llimit;												// current allocation (excluding two spare characters) of ofspec
	size_t nlinks;												// number of symbolic links encountered
	size_t sizeb;												// temporary variable used during allocation calculations

	// INITIALISE
	ifspec=NULL;												// editable copy of input filespec
	ofspec=NULL;												// editable (under construction) copy of output filespec
	tfspec=NULL;												// temporary filespec
	lfspec=NULL;												// link filespec
	ocount=0;													// current length of ofspec
	olimit=0;													// current allocation (excluding two spare characters) of ofspec
	lcount=0;													// current length of lfspec
	llimit=0;													// current allocation (excluding two spare characters) of lfspec
	nlinks=0;													// number of symbolic links encountered

	// PRIMARY ERROR CHECK
	if (infilespec==NULL)
	{
		ErrorAbort(1,__FILE__,__LINE__,"Attempt to resolve full filespec with null input filespec argument");
	}
	if (outfilespec==NULL)
	{
		ErrorAbort(1,__FILE__,__LINE__,"Attempt to resolve full filepsec with null output filespec argument");
	}

	// HANDLE EXTREME CASES
	sizeb=1;
	if (infilespec[0]==NUL)
	{
		ofspec=malloc(sizeb);
		if (ofspec==NULL)
		{
			ErrorAbort(1,__FILE__,__LINE__,"Could not allocate 1 byte for zero-length output filespec");
		}
		ofspec[0]=NUL;
		*outfilespec=ofspec;
		return;
	}

	// CREATE EDITABLE COPY OF INfilespec AND USE IT HEREAFTER
	sizeb=strlen(infilespec)+1;
	ifspec=malloc(sizeb);
	if (ifspec==NULL)
	{
		ErrorAbort(1,__FILE__,__LINE__,"Could not allocate %ld bytes for input filespec copy",sizeb);
	}
	(void) strcpy(ifspec,infilespec);

#if defined(__CYGWIN__)
	// REPLACE ALL BACKWARD SLASHES WITH FORWARD SLASHES
	i=ifspec;
	while (*i!=NUL)
	{
		if (*i==BSLASH)
		{
			*i=FSLASH;
		}
		i++;
	}

	// REPLACE "D:" WITH "/CYGDRIVE/D" (LCOUNT+=9)
	if ( (strlen(ifspec)>=2) && (ISALPHA(ifspec[0])) && (ifspec[1]==':') )
	{
		// Variables
		size_t icount,isizeb;
		// Initialise
		icount=strlen(ifspec);
		isizeb=icount+9+1;
		// Reallocate with extra space
		tfspec=realloc(ifspec,isizeb);
		if (tfspec==NULL)
		{
			ErrorAbort(1,__FILE__,__LINE__,"Could not reallocate %ld bytes for input filespec copy",isizeb);
		}
		ifspec=tfspec;
		// Move up the drive letter
		ifspec[1]=ifspec[0];
		// Move up the data
		(void) memmove(&ifspec[10],&ifspec[1],icount-1+1);
		// Put in the new string over the drive
		(void) strncpy(&ifspec[0],"/cygdrive/",10);
		// Account for the new data
		icount+=9;
	}
#endif

	// CREATE INITIAL OUTPUTFILE SPEC
	if (ifspec[0]==FSLASH)
	{
		// Variables
		size_t isizeb;
		// Allocate with some space for processing
		isizeb=BUFFER_INCREMENT+2;
		ofspec=malloc(isizeb);
		if (ofspec==NULL)
		{
			ErrorAbort(1,__FILE__,__LINE__,"Could not allocate %ld bytes for output filespec buffer",isizeb);
		}
		olimit=BUFFER_INCREMENT;
		// Copy out trailing slash
		ofspec[ocount++]=FSLASH;
		// Terminate ofspec
		ofspec[ocount]=NUL;
		// Start at first character after slash
		i=&ifspec[1];
	}
	else
	{
		// Start with with current directory
		while (TRUE)
		{
			olimit+=BUFFER_INCREMENT;
			tfspec=realloc(ofspec,olimit+1);
			if (tfspec==NULL)
			{
				ErrorAbort(1,__FILE__,__LINE__,"Could not reallocate %ld bytes for output filespec buffer",olimit+1);
			}
			ofspec=tfspec;
			tfspec=getcwd(ofspec,olimit+1);
			if (tfspec!=NULL)
			{
				break;
			}
			if (errno!=ERANGE)
			{
				ErrorAbort(1,__FILE__,__LINE__,"Unexpected error during current directory identification for output filespec");
			}
		}
		ocount=strlen(ofspec);
#if defined(__CYGWIN__)
		// Replace all backward slashes with forward slashes
		i=ofspec;
		while (*i!=NUL)
		{
			if (*i==BSLASH)
			{
				*i=FSLASH;
			}
			i++;
		}
#endif
		// Reallocate with some space for processing
		tfspec=realloc(ofspec,ocount+BUFFER_INCREMENT+2);
		if (tfspec==NULL)
		{
			ErrorAbort(1,__FILE__,__LINE__,"Could not reallocate %ld bytes for output filespec buffer",ocount+BUFFER_INCREMENT+2);
		}
		ofspec=tfspec;
		olimit=ocount+BUFFER_INCREMENT;
		// Ensure trailing slash
		if (ofspec[ocount-1]!=FSLASH)
		{
			ofspec[ocount++]=FSLASH;
		}
		// Terminate ofspec
		ofspec[ocount]=NUL;
		// Start at first character (is not a slash)
		i=&ifspec[0];
	}

	// EXPAND COMPONENTS
	while (*i!=NUL)
	{
		// VARIABLES
		bool_t justnowwaslink;
		// HANDLE DOT-ENDOFSTRING
		if ( (i[0]=='.') && (i[1]==NUL) )
		{
			// skip to end of string
			i++;
			// restart at end of string
			continue;
		}
		// HANDLE DOT-SLASH
		if ( (i[0]=='.') && (i[1]==FSLASH) )
		{
			// skip to next component
			i+=2;
			// restart from that component
			continue;
		}
		// HANDLE DOT-DOT-ENDOFSTRING
		if ( (i[0]=='.') && (i[1]=='.') && (i[2]==NUL) )
		{
			// skip to next component
			i+=2;
			// delete back over current constructed component
			ocount--;
			while ( (ocount>0) && (ofspec[ocount-1]!=FSLASH) )
			{
				ocount--;
			}
			ofspec[ocount]=NUL;
			// restart from next input component
			continue;
		}
		// HANDLE DOT-DOT-SLASH
		if ( (i[0]=='.') && (i[1]=='.') && (i[2]==FSLASH) )
		{
			// skip to next component
			i+=3;
			// delete back over current constructed component
			ocount--;
			while ( (ocount>0) && (ofspec[ocount-1]!=FSLASH) )
			{
				ocount--;
			}
			ofspec[ocount]=NUL;
			// restart from next input component
			continue;
		}
		// APPEND INPUT COMPONENT AFTER CURRENT OUTPUT SLASH
		while ( (*i!=NUL) && (*i!=FSLASH) )
		{
			if (ocount==olimit)
			{
				// Reallocate with some space for processing
				tfspec=realloc(ofspec,ocount+BUFFER_INCREMENT+2);
				if (tfspec==NULL)
				{
					ErrorAbort(1,__FILE__,__LINE__,"Could not reallocate %ld bytes for output filespec buffer",ocount+BUFFER_INCREMENT+2);
				}
				ofspec=tfspec;
				olimit=ocount+BUFFER_INCREMENT;
			}
			ofspec[ocount++]=*i++;
		}
		ofspec[ocount]=NUL;
		// CHECK FOR LINK LOOPS
		if (nlinks==LINK_LIMIT)
		{
			ErrorAbort(1,__FILE__,__LINE__,"Symbolic link chain too big -- probable file system error (ELOOP)");
		}
		// CHECK FOR LINK
		justnowwaslink=FALSE;
		while (TRUE)
		{
			int tlcount;
			/**********/
			llimit+=BUFFER_INCREMENT;
			tfspec=realloc(lfspec,llimit+1);
			if (tfspec==NULL)
			{
				ErrorAbort(1,__FILE__,__LINE__,"Could not reallocate %ld bytes for symbolic link buffer",llimit+1);
			}
			lfspec=tfspec;
			tlcount=readlink(ofspec,lfspec,(int)llimit);
			if (tlcount<0)
			{
				// CHECK FOR INVALID RETURN VALUE
				if (tlcount<(-1))
				{
					ErrorAbort(1,__FILE__,__LINE__,"Unexpected error during symbolic link interrogation for input filespec");
				}
				// CHECK FOR UNEXPECTED ERRNO VALUE
				//
				// For the purposes of this function, we do not
				// care whether this is not a link because (a)
				// it does not exist at all or because (b) it
				// does exist, but is not actually a link.
				//
				if ( (errno!=EINVAL) && (errno!=ENOENT) )
				{
					ErrorAbort(1,__FILE__,__LINE__,"Unexpected error during symbolic link interrogation for input filespec");
				}
				// FILESPEC COMPONENT IS NOT A LINK
				free(lfspec);
				lfspec=NULL;
				lcount=0;
				llimit=0;
				break;
			}
			lcount=(size_t)tlcount;
			if (lcount<llimit)
			{
				// Remove any trailing slash from link
				if (lfspec[lcount-1]==FSLASH)
				{
					lcount--;
				}
				// Add trailing null to link
				lfspec[lcount]=NUL;
#if defined(__CYGWIN__)
				// Replace "d:" with "/cygdrive/d" (lcount+=9)
				if ( (lcount>=2) && (ISALPHA(lfspec[0])) && (lfspec[1]==':') )
				{
					// Reallocate with extra space
					tfspec=realloc(lfspec,llimit+9+1);
					if (tfspec==NULL)
					{
						ErrorAbort(1,__FILE__,__LINE__,"Could not reallocate %ld bytes for symbolic link buffer",llimit+9+1);
					}
					lfspec=tfspec;
					llimit=llimit+9;
					// Move up the drive letter
					lfspec[1]=lfspec[0];
					// Move up the data
					(void) memmove(&lfspec[10],&lfspec[1],lcount-1+1);
					// Put in the new string over the drive
					(void) strncpy(&lfspec[0],"/cygdrive/",10);
					// Account for the new data
					lcount+=9;
				}
#endif
				justnowwaslink=TRUE;
				nlinks++;
				break;
			}
		}
		// HANDLE LINK
		if (lfspec!=NULL)
		{
			// DOCUMENTATION
			/*
				//
				Ifspec  /xxxx/yyyy/zzzz/file.ext ( && yyyy --> "/aaaaa/ppppp/qqqqq" )
				Ofspec  /xxxx/yyyy/
				Lfspec  /aaaaa/ppppp/qqqqq
				------
				Ifspec  /aaaaa/ppppp/qqqqq/zzzz/file.ext
				Ofspec  /
				Lfspec
				------
				THIS LINK IS AN ABSOULTE LINK
				REPLACE Ifspec WITH Lfspec PLUS REST OF Ifspec
				REPLACE Ofspec WITH A SLASH AND START IT AGAIN
				CLEAR Lfspec
				//
				Ifspec  /xxxx/yyyy/zzzz/file.ext ( && yyyy --> "../ppppp/qqqqq" )
				Ofspec  /xxxx/yyyy/
				Lfspec  ../ppppp/qqqqq
				------
				Ifspec  ../ppppp/qqqqq/zzzz/file.ext
				Ofspec  /xxxx/
				Lfspec
				------
				THIS LINK IS A RELATIVE LINK
				REPLACE Ifspec WITH Lfspec PLUS REST OF Ifspec
				REPLACE Ofspec WITH Ofspec MINUS LAST COMPONENT THEREOF
				CLEAR Lfspec
				//
			*/
			// VARIABLES
			bool_t linkisabsolute;
			// LOG ABSOLUTENESS OF LINK
			linkisabsolute=(lfspec[0]==FSLASH);
			// Ifspec BECOMES Lfspec PLUS REMAINS OF Ifspec
			if (*i!=NUL)
			{
				// Should be at a slash if we get here
				if (*i!=FSLASH)
				{
					FatalAbort(1,__FILE__,__LINE__,"Parse of input filespec failed: %s",ifspec);
				}
				// Reallocate lfspec to accommodate
				llimit=lcount+/*includes slash*/strlen(i);
				tfspec=realloc(lfspec,llimit+1);
				if (tfspec==NULL)
				{
					ErrorAbort(1,__FILE__,__LINE__,"Could not reallocate %ld bytes for amended input filespec buffer",llimit+1);
				}
				lfspec=tfspec;
				// Append remainder of ifspec
				(void) memmove(&lfspec[lcount],i,strlen(i)+1);
				// Free current ifspec
				free(ifspec);
				// Replace ifspec with correct data
				ifspec=lfspec;
				// Clear lfspec
				lfspec=NULL;
				lcount=0;
				llimit=0;
			}
			// FIX UP LINK
			if (linkisabsolute)
			{
				// Reset ofspec count
				ocount=0;
				// Reallocate with some space for processing
				tfspec=realloc(ofspec,BUFFER_INCREMENT+2);
				if (tfspec==NULL)
				{
					ErrorAbort(1,__FILE__,__LINE__,"Could not reallocate %ld bytes for ouput filespec buffer",BUFFER_INCREMENT+2);
				}
				ofspec=tfspec;
				olimit=BUFFER_INCREMENT;
				// Copy out trailing slash
				ofspec[ocount++]=FSLASH;
				// Terminate ofspec
				ofspec[ocount]=NUL;
				// Start at first character after slash
				i=&ifspec[1];
			}
			else
			{
				// Delete ofspec back over current constructed component
				ocount--;
				while ( (ocount>0) && (ofspec[ocount-1]!=FSLASH) )
				{
					ocount--;
				}
				// Terminate ofspec
				ofspec[ocount]=NUL;
				// Start at first character (is not a slash)
				i=&ifspec[0];
			}
		}
		// IF NOT LINK THEN: TERMINATE Ofspec FOR NOW AND ALSO,
		// IF NOT AT THE END OF Ifspec, SKIP TO NEXT CHARACTER.
		if (!justnowwaslink)
		{
			// Append (copy out) slash to ofspec
			ofspec[ocount++]=FSLASH;
			// Terminate ofspec
			ofspec[ocount]=NUL;
			// Stop if at end of ifspec
			if (*i==NUL)
			{
				break;
			}
			// Start at next character of ifspec
			i++;
		}
		justnowwaslink=FALSE;
	}

	// REMOVE ANY TRAILING SLASH UNLESS ROOT DIRECTORY
	//
	// We do not use trailing slashes on directories and
	// they don't apply to files. They are automatically
	// added at each stage of parsing and we only know at
	// the last component that is safe to remove them.
	//
	if ( (ocount>1) && (ofspec[ocount-1]==FSLASH) )
	{
		ocount--;
	}

	// ADD NULL TERMINATION
	ofspec[ocount]=NUL;

	// DEINITIALISE
	free(ifspec);

	// RETURN THE RESULT
	*outfilespec=ofspec;
}
#endif // defined(__CYGWIN__) || define(__LINUX__)

// Check platform flag
#ifndef __SETPLATFORM__
#error "You need to set up platform specifics for your platform!"
#endif

// Deinitialise platform flag
#undef __SETPLATFORM__


/**********************************************************************************************************************************/
/******************************************************* File Manipulation ********************************************************/
/**********************************************************************************************************************************/

// Initialise platform flag
#undef __SETPLATFORM__

// Platform specifics
#if defined(__WIN32__)
#define __SETPLATFORM__ 1
#include <io.h>
#include <stdio.h>
// Truncate file to specified size (leave at eof)
void FILETruncateAndSeekEOF (FILE *fp, size_t size)
{
#ifdef _WIN64
	if (size>SINT32_MAX)
	{
		FatalAbort(1,__FILE__,__LINE__,"Value in \"size\" is > 32 bits");
	}
#endif
	if (fflush(fp)!=0)
	{
		ErrorAbort(1,__FILE__,__LINE__,"Could not flush file in order to change its size");
	}
#ifdef __VISUALC__
#pragma warning( push )
#pragma warning( disable : 4267 )
#endif
	if (chsize(fileno(fp),size)!=0)
#ifdef __VISUALC__
#pragma warning( pop )
#endif
	{
		ErrorAbort(1,__FILE__,__LINE__,"Could not change size of file");
	}
#ifdef __VISUALC__
#pragma warning( push )
#pragma warning( disable : 4267 )
#endif
	if (fseek(fp,size,SEEK_SET)!=0)
#ifdef __VISUALC__
#pragma warning( pop )
#endif
	{
		ErrorAbort(1,__FILE__,__LINE__,"Could not seek to EOF in input file");
	}
}
#endif // defined(__WIN32__)

#if defined(__CYGWIN__) || defined(__LINUX__)
#define __SETPLATFORM__ 1
#include <unistd.h>
#include <stdio.h>
// Truncate file to specified size (leave at eof)
void FILETruncateAndSeekEOF (FILE *fp, size_t size)
{
	if (fflush(fp)!=0)
	{
		ErrorAbort(1,__FILE__,__LINE__,"Could not flush file in order to change its size");
	}
	if (ftruncate(fileno(fp),(long int)size)!=0)
	{
		ErrorAbort(1,__FILE__,__LINE__,"Could not change size of file");
	}
	if (fseek(fp,(long int)size,SEEK_SET)!=0)
	{
		ErrorAbort(1,__FILE__,__LINE__,"Could not seek to EOF in input file");
	}
}
#endif // defined(__CYGWIN__) || define(__LINUX__)

// Check platform flag
#ifndef __SETPLATFORM__
#error "You need to set up platform specifics for your platform!"
#endif

// Deinitialise platform flag
#undef __SETPLATFORM__

/**********************************************************************************************************************************/
/********************************************** Master Configuration File Directory ***********************************************/
/**********************************************************************************************************************************/

// Initialise platform flag
#undef __SETPLATFORM__

// Platform specifics
#if defined(__WIN32__)
#define __SETPLATFORM__ 1
#include <stdlib.h>
#include <string.h>

// Windows stuff
unsigned long __stdcall GetModuleFileNameA (
	void *hModule,
	char *lpFilename,
	unsigned long nSize);

// Get the full pathname of the program
static void FILEGetFullProgramSpec (char **outfilespec)
{
	// METRICS
	#define BUFFER_APPORTION 64									// does not include trailing null
	#define BUFFER_INCREMENT 64									// does not include trailing null

	// VARIABLES
	char *ofspec;												// editable (under construction) copy of output filespec
	char *tfspec;												// temporary filespec
	size_t sizeb;												// temporary variable used during allocation calculations
#ifdef _WIN64
	size_t temp_strlen;											// temporary variable used during allocation calculations
#endif
	unsigned long found_length;									// length required for resolved filespec
	unsigned long allocated_length;								// length allocated for resolved filespec

	// INITIALISE
	ofspec=NULL;												// editable (under construction) copy of output filespec

	// PRIMARY ERROR CHECK
	if (outfilespec==NULL)
	{
		ErrorAbort(1,__FILE__,__LINE__,"Attempt to resolve full filepsec with null output filespec argument");
	}

	// ALLOCATE THE NECESSARY MEMORY
	allocated_length=BUFFER_APPORTION;
	sizeb=allocated_length+1;
	ofspec=malloc(sizeb);
	if (ofspec==NULL)
	{
		ErrorAbort(1,__FILE__,__LINE__,"Could not allocate %ld bytes for output filespec",sizeb);
	}

	// KEEP GOING UNTIL BUFFER IS BIG ENOUGH
	found_length=GetModuleFileNameA(NULL,ofspec,allocated_length);
	if (found_length==0)
	{
		ErrorAbort(1,__FILE__,__LINE__,"Could not get length required for output filespec");
	}
	if (found_length>allocated_length)
	{
		FatalAbort(1,__FILE__,__LINE__,"Invalid return value from Win32 API GetModuleFileNameA");
	}
	while (found_length==allocated_length)
	{
		allocated_length+=BUFFER_INCREMENT;
		sizeb=allocated_length+1;
		tfspec=realloc(ofspec,sizeb);
		if (tfspec==NULL)
		{
			ErrorAbort(1,__FILE__,__LINE__,"Could not reallocate %ld bytes for output filespec",sizeb);
		}
		ofspec=tfspec;
		found_length=GetModuleFileNameA(NULL,ofspec,allocated_length);
		if (found_length==0)
		{
			ErrorAbort(1,__FILE__,__LINE__,"Could not get length required for output filespec");
		}
		if (found_length>allocated_length)
		{
			FatalAbort(1,__FILE__,__LINE__,"Invalid return value from Win32 API GetModuleFileNameA");
		}
	}

	// TRIM THE BUFFER TO ONLY THE NECESSARY SIZE
#ifdef _WIN64
	temp_strlen=strlen(ofspec);
	if (temp_strlen>UINT32_MAX)
	{
		FatalAbort(1,__FILE__,__LINE__,"Value in \"temp_strlen\" is > 32 bits");
	}
	allocated_length=(unsigned long)(temp_strlen & 0xFFFFFFFF);
#else
	#ifdef __VISUALC__
	#pragma warning( push )
	#pragma warning( disable : 4267 )
	#endif
	allocated_length=strlen(ofspec);
	#ifdef __VISUALC__
	#pragma warning( pop )
	#endif
#endif
	sizeb=allocated_length+1;
	tfspec=realloc(ofspec,sizeb);
	if (tfspec==NULL)
	{
		ErrorAbort(1,__FILE__,__LINE__,"Could not reallocate %ld bytes for output filespec",sizeb);
	}
	ofspec=tfspec;

	// RETURN THE RESULT
	*outfilespec=ofspec;
}

// Get the master configuration file directory
void FILEGetMasterConfigFileDirectory (const char *envvarname, char **outfilespec)
{
	// DOCUMENTATION
	/*
		Return the master configuration file directory for
		CleanWAD with a trailing directory separator included.
	*/

	// VARIABLES
	char *ofspec;
	char *tfspec;
	char *efspec;
	size_t spos,sizeb;

	// PRIMARY ERROR CHECK
	if (envvarname==NULL)
	{
		ErrorAbort(1,__FILE__,__LINE__,"Attempt to resolve master configuration file directory with null environment variable name argument");
	}
	if (outfilespec==NULL)
	{
		ErrorAbort(1,__FILE__,__LINE__,"Attempt to resolve master configuration file directory with null output filespec argument");
	}

	// CHECK THE ENVIRONMENT
	efspec=getenv(envvarname);

	// IF DEFINED THEN CREATE COPY AND RETURN
	if (efspec!=NULL)
	{
		// VARIABLES
		size_t sizeb;
		// CONVERT TO FULL FILESPEC
		FILEGetFullSpec(efspec,&ofspec);
		// CONVERT TO CORRECT CASE (OR LEAVE IF NONE)
#if __FILENAMESCC__==1											// if can PRESENT filenames case insensitive...
#if FNCASE==1													// lowercase them
		lowercase(ofspec);
#endif
#if FNCASE==2													// uppercase them
		uppercase(ofspec);
#endif
#endif
		// ADD TRAILING BACKSLASH IF NONE IS PRESENT
		sizeb=strlen(ofspec);
		if (ofspec[sizeb-1]!=BSLASH)
		{
			char *tfspec;
			/***********/
			sizeb=sizeb+1+/*for the trailing null*/1;
			tfspec=realloc(ofspec,sizeb);
			if (tfspec==NULL)
			{
				ErrorAbort(1,__FILE__,__LINE__,"Could not reallocate %ld bytes for output filespec",sizeb);
			}
			ofspec=tfspec;
			ofspec[sizeb-2]=BSLASH;
			ofspec[sizeb-1]=NUL;
		}
		// FREE RETURNED ENVIRONMENT STRING
//		Actually, do NOT free efspec; it is managed by getenv/putenv
		// RETURN THE RESULT
		*outfilespec=ofspec;
		return;
	}

	//
	// UNDER WIN32 IT IS THE PROGRAM DIRECTORY
	//
	// Most Win32 programs expect that any master configuration
	// files will reside in the program directory. Thus, if the
	// directory is not specified in the environment, then this
	// is where it should be, according to standard practice.
	//

	// GET PROGRAM FILESPEC
	FILEGetFullProgramSpec(&tfspec);

	// CONVERT TO CORRECT CASE (OR LEAVE IF NONE)
#if __FILENAMESCC__==1											// if can PRESENT filenames case insensitive...
#if FNCASE==1													// lowercase them
	lowercase(tfspec);
#endif
#if FNCASE==2													// uppercase them
	uppercase(tfspec);
#endif
#endif

	// GET INDEX OF LAST CHARACTER IN FILESPEC
	spos=strlen(tfspec);
	if (spos==0)
	{
		FatalAbort(1,__FILE__,__LINE__,"Invalid program filespec (empty)");
	}
	spos--;

	// STRIP OFF NAME AND EXTENSION
	while (TRUE)
	{
		if (tfspec[spos]==BSLASH)
		{
			break;
		}
		if  (spos==0)
		{
			break;
		}
		spos--;
	}

	// VALIDATE THE RESULT
	if (tfspec[spos]!=BSLASH)
	{
		FatalAbort(1,__FILE__,__LINE__,"Invalid program filespec %s",tfspec);
	}

	// TERMINATE THE STRING THERE (LEAVING THE BACKSLASH)
	tfspec[spos+1]=NUL;

	// TRIM THE BUFFER TO ONLY THE NECESSARY SIZE
	sizeb=(spos+1)+/*for the trailing null*/1;
	ofspec=realloc(tfspec,sizeb);
	if (ofspec==NULL)
	{
		ErrorAbort(1,__FILE__,__LINE__,"Could not reallocate %ld bytes for output filespec",sizeb);
	}

	// RETURN THE RESULT
	*outfilespec=ofspec;
}

#endif // defined(__WIN32_)

#if defined(__CYGWIN__) || defined(__LINUX__)
#define __SETPLATFORM__ 1
// Get the master configuration file directory
void FILEGetMasterConfigFileDirectory (const char *envvarname, char **outfilespec)
{
	// DOCUMENTATION
	/*
		Return the master configuration file directory for
		CleanWAD with a trailing directory separator included.
	*/

	// VARIABLES
	char *ofspec;
	char *efspec;

	// PRIMARY ERROR CHECK
	if (envvarname==NULL)
	{
		ErrorAbort(1,__FILE__,__LINE__,"Attempt to resolve master configuration file directory with null environment variable name argument");
	}
	if (outfilespec==NULL)
	{
		ErrorAbort(1,__FILE__,__LINE__,"Attempt to resolve master configuration file directory with null output filespec argument");
	}

	// CHECK THE ENVIRONMENT
	efspec=getenv(envvarname);

	// IF DEFINED THEN CREATE COPY AND RETURN
	if (efspec!=NULL)
	{
		// VARIABLES
		size_t sizeb;
		// CONVERT TO FULL FILESPEC
		FILEGetFullSpec(efspec,&ofspec);
		// CONVERT TO CORRECT CASE (OR LEAVE IF NONE)
#if __FILENAMESCC__==1											// if can PRESENT filenames case insensitive...
#if FNCASE==1													// lowercase them
		lowercase(ofspec);
#endif
#if FNCASE==2													// uppercase them
		uppercase(ofspec);
#endif
#endif
		// ADD TRAILING SLASH IF NONE IS PRESENT
		sizeb=strlen(ofspec);
		if (ofspec[sizeb-1]!=FSLASH)
		{
			char *tfspec;
			/***********/
			sizeb=sizeb+1+/*for the trailing null*/1;
			tfspec=realloc(ofspec,sizeb);
			if (tfspec==NULL)
			{
				ErrorAbort(1,__FILE__,__LINE__,"Could not reallocate %ld bytes for output filespec",sizeb);
			}
			ofspec=tfspec;
			ofspec[sizeb-2]=FSLASH;
			ofspec[sizeb-1]=NUL;
		}
		// FREE RETURNED ENVIRONMENT STRING
//		Actually, do NOT free efspec; it is managed by getenv/putenv
		// RETURN THE RESULT
		*outfilespec=ofspec;
		return;
	}

	//
	// UNDER CYGWIN OR LINUX IT MUST BE AN ENVIRONMENT VARIABLE
	//
	// Most UNIX(TM) like operating systems require the use of
	// environment variables as the standard way of finding
	// special directories or files for a program. Indeed there
	// is no system call to obtain the full path of the current
	// program and attempting to do so is severely deprecated.
	//
	ErrorAbort(1,__FILE__,__LINE__,"Environment variable %s (Master Configuration File Directory) is not set)",envvarname);
}
#endif // defined(__CYGWIN__) || define(__LINUX__)

// Check platform flag
#ifndef __SETPLATFORM__
#error "You need to set up platform specifics for your platform!"
#endif

// Deinitialise platform flag
#undef __SETPLATFORM__

/**********************************************************************************************************************************/
/********************************************************** End of File ***********************************************************/
/**********************************************************************************************************************************/
