/*
   DooM PostScript Maps Utility, by Frans P. de Vries.

Derived from:

   Doom Editor Utility, by Brendon Wyber and Raphal Quinet.

   You are allowed to use any parts of this code in another program, as
   long as you give credits to the authors in the documentation and in
   the program itself.  Read the file README for more information.

   This program comes with absolutely no warranty.

   DMPS.H - Main DMPSMU and DMPSMAP defines.
*/

#ifndef _H_WAD
#define _H_WAD

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <ctype.h>
#include "tcl.h"


extern void swapint( short *i);
extern void swaplong( long *l);

#define LDTYPE_DOOR		1
#define LDTYPE_BLUEDOOR		26
#define LDTYPE_YELLOWDOOR	27
#define LDTYPE_REDDOOR		28
#define LDTYPE_DOOR_NOCLOSE		31
#define LDTYPE_BLUEDOOR_NOCLOSE		32
#define LDTYPE_YELLOWDOOR_NOCLOSE	33
#define LDTYPE_REDDOOR_NOCLOSE		34

#define LDTYPE_DOOR_TURBO		117
#define LDTYPE_DOOR_TURBO_NOCLOSE	118

#define LDTYPE_TELEP1	39
#define LDTYPE_TELEPR	97
#define LDTYPE_TELPM1	125
#define LDTYPE_TELPMR	126

#define LDFLAG_IMPASS	0x01
#define LDFLAG_TWOSIDED	0x04
#define LDFLAG_SECRET	0x20

/* global (re)definitions */
#ifndef max
#define max(x,y)	(((x)>(y))?(x):(y))
#define min(x,y)	(((x)<(y))?(x):(y))
#endif


/*
   The directory structure is the structure used by DOOM to order the
   data in its WAD files.
*/
typedef struct Directory *DirPtr;
struct Directory
{
   long start;		/* offset to start of data */
   long size;		/* byte size of data */
   char name[ 8];	/* name of data block */
};


/*
   The Wad file pointer structure is used for holding the information
   on the Wad files in a linked list.
   The first Wad file is the main WAD file. The rest are patches.
*/
typedef struct WadFileInfo *WadPtr;
struct WadFileInfo
{
   WadPtr next;		/* next file in linked list */
   char   *filename;	/* name of the WAD file */
   FILE   *fileinfo;	/* C file stream information */
   char   type[ 4];	/* type of WAD file (IWAD or PWAD) */
   long   dirsize;	/* directory size of WAD */
   long   dirstart;	/* offset to start of directory */
   DirPtr directory;	/* array of directory information */
};


/*
   The master directory structure is used to build a complete directory
   of all the data blocks from all the various Wad files.
*/
typedef struct MasterDirectory *MDirPtr;
struct MasterDirectory
{
   MDirPtr next;		/* next in list */
   WadPtr  wadfile;		/* file of origin */
   struct Directory dir;	/* directory data */
};


/*
   Description of the command line arguments and config file keywords.
*/
typedef struct
{
   char *short_name;	/* abbreviated command line argument */
   char *long_name;	/* command line arg. or keyword */
   enum	{		/* type of this option */
      OPT_BOOLEAN,	/* boolean (toggle) */
      OPT_INTEGER,	/* integer number */
      OPT_STRING,	/* character string */
      OPT_STRINGACC,	/* character string, but store in a list */
      OPT_STRINGLIST,	/* list of character strings */
      OPT_END		/* end of the options description */
      }  opt_type;
   char *msg_if_true;	/* message printed if option is true */
   char *msg_if_false;	/* message printed if option is false */
   void *data_ptr;	/* pointer to the data */
} OptDesc;


/*
   the interfile global variables
*/

extern char *MainWad;		/* name of the main WAD file */
extern char *UserLvlNm;		/* user defined level name */

extern WadPtr  WadFileList;	/* linked list of Wad files */
extern MDirPtr MasterDir;	/* the master directory */

extern FILE *PSFile;		/* the PostScript output file */


int ProgError( char *, ...);

int OpenMainWad( Tcl_Interp *, char *);
int OpenPatchWad( Tcl_Interp *, char *);
void CloseWadFiles( void);
void CloseUnusedWadFiles( void);
WadPtr BasicWadOpen( Tcl_Interp *, char *);
void BasicWadRead( WadPtr, void *, long);
void BasicWadSeek( WadPtr, long);
MDirPtr FindMasterDir( MDirPtr, char *);
void ListMasterDirectory( FILE *);
void ListFileDirectory( FILE *, WadPtr);
void CopyBytes( FILE *, FILE *, long);
int Exists( char *);
void DumpDirectoryEntry( FILE *, char *);
void SaveEntryToRawFile( FILE *, char *);

short GetThingClass( short);
short GetThingRadius( short);

int ReadLevelData( Tcl_Interp *, int, int);
void ForgetLevelData( void);


/*
   this data structure contains the information about the THINGS
*/
typedef struct Thing *TPtr;
struct Thing
{
   short  xpos;		/* x position */
   short  ypos;		/* y position */
   short  angle;	/* facing angle (0 = east, 90 = north, ...) */
   short  type;		/* thing type */
   short  when;		/* appears when? (bit 4 = multi-player) */
};

/*
   this data structure contains the information about the VERTEXES
*/
typedef struct Vertex *VPtr;
struct Vertex
{
   short  x;		/* X coordinate */
   short  y;		/* Y coordinate */
};

/*
   this data structure contains the information about the LINEDEFS
*/
typedef struct LineDef *LDPtr;
struct LineDef
{
   short  start;	/* from this vertex ... */
   short  end;		/* ... to this vertex */
   short  flags;	/* attributes (bit 0 = impassible, bit 5 = secret) */
   short  type;		/* type to activate special effects */
   short  tag;		/* linedef activates the sector with the same tag */
   short  sidedef1;	/* "right" sidedef */
   short  sidedef2;	/* only if this line adjoins 2 sectors, else -1 */
};

/*
   this data structure contains the information about the SIDEDEFS
*/
typedef struct SideDef *SDPtr;
struct SideDef
{
   short  xoff;		/* X offset for texture */
   short  yoff;		/* Y offset for texture */
   char   tex1[ 8];	/* texture name for the upper part */
   char   tex2[ 8];	/* texture name for the lower part */
   char   tex3[ 8];	/* texture name for the regular part */
   short  sector;	/* adjacent sector */
};

/*
   this data structure contains the information about the SEGS
*/
typedef struct Seg *SEPtr;
struct Seg
{
   SEPtr  next;		/* next Seg in list */
   short  start;	/* from this vertex ... */
   short  end;		/* ... to this vertex */
   unsigned short angle;	/* angle (0 = east, 16384 = north, ...) */
   short  linedef;	/* linedef that this seg goes along */
   short  flip;		/* true if not the same direction as linedef */
   unsigned short dist;		/* distance from starting point */
};

/*
   this data structure contains the information about the SSECTORS
*/
typedef struct SSector *SSPtr;
struct SSector
{
   SSPtr  next;		/* next Sub-Sector in list */
   short  num;		/* number of Segs in this Sub-Sector */
   short  first;	/* first Seg */
};

/*
   this data structure contains the information about the SECTORS
*/
typedef struct Sector *SPtr;
struct Sector
{
   short  floorh;	/* floor height */
   short  ceilh;	/* ceiling height */
   char   floort[ 8];	/* floor texture */
   char   ceilt[ 8];	/* ceiling texture */
   short  light;	/* light level (0-255) */
   short  special;	/* special behaviour (0 = normal, 9 = secret, ...) */
   short  tag;		/* sector activated by a linedef with the same tag */
};

/*
   this data structure contains the information about the NODES
*/
typedef struct Node *NPtr;
struct Node
{
   short  x, y;			/* starting point */
   short  dx, dy;		/* offset to ending point */
   short  miny1, maxy1,
	  minx1, maxx1;		/* bounding rectangle 1 */
   short  miny2, maxy2,
	  minx2, maxx2;		/* bounding rectangle 2 */
   short  child1, child2;	/* Node or SSector (if high bit is set) */
   NPtr   node1, node2;		/* pointer if the child is a Node */
   short  num;			/* number given to this Node */
};

/* Starting points */
#define THING_PLAYER1		1
#define THING_PLAYER2		2
#define THING_PLAYER3		3
#define THING_PLAYER4		4
#define THING_DEATHMATCH	11
#define THING_TELEPORT		14

/* Enhancements & bonuses */
#define THING_BLUECARD		5
#define THING_YELLOWCARD	6
#define THING_REDCARD		13
#define THING_BLUESKULLKEY	40
#define THING_YELLOWSKULLKEY	39
#define THING_REDSKULLKEY	38
#define THING_ARMBONUS1		2015
#define THING_GREENARMOR	2018
#define THING_BLUEARMOR		2019
#define THING_HLTBONUS1		2014
#define THING_STIMPACK		2011
#define THING_MEDIKIT		2012
#define THING_SOULSPHERE	2013
#define THING_MEGASPHERE	83	/* ][ */
#define THING_RADSUIT		2025
#define THING_COMPMAP		2026
#define THING_BLURSPHERE	2024
#define THING_BERSERK		2023
#define THING_INVULN		2022
#define THING_LITEAMP		2045

/* Weapons & ammo */
#define THING_CHAINSAW		2005
#define THING_SHOTGUN		2001
#define THING_COMBATGUN		82	/* ][ */
#define THING_CHAINGUN		2002
#define THING_LAUNCHER		2003
#define THING_PLASMAGUN		2004
#define THING_BFG9000		2006
#define THING_AMMOCLIP		2007
#define THING_AMMOBOX		2048
#define THING_SHELLS		2008
#define THING_SHELLBOX		2049
#define THING_ROCKET		2010
#define THING_ROCKETBOX		2046
#define THING_ENERGYCELL	2047
#define THING_ENERGYPACK	17
#define THING_BACKPACK		8

/* Enemies */
#define THING_HUMAN		3004
#define THING_SERGEANT		9
#define THING_COMMANDO		65	/* ][ */
#define THING_IMP		3001
#define THING_DEMON		3002
#define THING_SPECTRE		58
#define THING_LOSTSOUL		3006
#define THING_CACODEMON		3005
#define THING_PAINELEM		71	/* ][ */
#define THING_MANCUBUS		67	/* ][ */
#define THING_REVENANT		66	/* ][ */
#define THING_KNIGHT		69	/* ][ */
#define THING_BARON		3003
#define THING_ARACHNO		68	/* ][ */
#define THING_ARCHVILE		64	/* ][ */
#define THING_CYBERDEMON	16
#define THING_SPIDERBOSS	7
#define THING_WOLF3DSS		84	/* ][ */

/* Specials */
#define THING_BARREL		2035
#define THING_BOSSBRAIN		88	/* ][ */
#define THING_BOSSSHOOT		89	/* ][ */
#define THING_SPAWNSPOT		87	/* ][ */
#define THING_KEEN		72	/* ][ */

/* Decorations */
#define THING_TECHCOLUMN	48
#define THING_TGREENPILLAR	30
#define THING_TREDPILLAR	32
#define THING_SGREENPILLAR	31
#define THING_SREDPILLAR	33
#define THING_PILLARHEART	36
#define THING_PILLARSKULL	37
#define THING_EYEINSYMBOL	41
#define THING_GREYTREE		43
#define THING_BROWNSTUB		47
#define THING_BROWNTREE		54

/* Decorations 2 */
#define THING_LAMP		2028
#define THING_TALLLAMP		85	/* ][ */
#define THING_SHORTLAMP		86	/* ][ */
#define THING_CANDLE		34
#define THING_CANDELABRA	35
#define THING_TBLUETORCH	44
#define THING_TGREENTORCH	45
#define THING_TREDTORCH		46
#define THING_SBLUETORCH	55
#define THING_SGREENTORCH	56
#define THING_SREDTORCH		57
#define THING_FLAMECAN		70	/* ][ */

/* Decorations 3 */
#define THING_DEADPLAYER	15
#define THING_DEADHUMAN		18
#define THING_DEADSERGEANT	19
#define THING_DEADIMP		20
#define THING_DEADDEMON		21
#define THING_DEADCACODEMON	22
#define THING_DEADLOSTSOUL	23
#define THING_BONES		10
#define THING_BONES2		12
#define THING_POOLOFBLOOD	24
#define THING_POOLOFBLOOD2	79	/* ][ */
#define THING_POOLOFBLOOD3	80	/* ][ */
#define THING_POOLOFBRAINS	81	/* ][ */
#define THING_SKULLTOPPOLE	27
#define THING_HEADSKEWER	28
#define THING_PILEOFSKULLS	29
#define THING_IMPALEDBODY	25
#define THING_IMPALEDBODY2	26
#define THING_SKULLSINFLAMES	42

/* Decorations 4 */
#define THING_HANGINGSWAYING	49
#define THING_HANGINGSWAYING2	63
#define THING_HANGINGARMSOUT	50
#define THING_HANGINGARMSOUT2	59
#define THING_HANGINGONELEG	51
#define THING_HANGINGONELEG2	61
#define THING_HANGINGTORSO	52
#define THING_HANGINGTORSO2	60
#define THING_HANGINGLEG	53
#define THING_HANGINGLEG2	62
#define THING_HANGINGNOGUTS	73	/* ][ */
#define THING_HANGINGNOGUTS2	74	/* ][ */
#define THING_HANGINGLOOKDN	75	/* ][ */
#define THING_HANGINGLOOKUP	77	/* ][ */
#define THING_HANGINGTORSO3	76	/* ][ */
#define THING_HANGINGTORSO4	78	/* ][ */


/* Classes */
#define CLASS_START	0
#define CLASS_ENHANCE	1
#define CLASS_BONUS	2
#define CLASS_WEAPON	3
#define CLASS_ENEMY	4
#define CLASS_DECOR	5
#define CLASS_BARREL	6
#define CLASS_TELEPORT	7
#define CLASS_UNKNOWN	8



/* the external variables from levels.c */
extern MDirPtr  Level;		/* master dictionary entry for the level */
extern char    *LevelName;	/* official name for the level */

extern short NumThings;		/* number of things */
extern TPtr     Things;		/* things data */
extern short NumLineDefs;	/* number of line defs */
extern LDPtr    LineDefs;	/* line defs data */
extern short NumSideDefs;	/* number of side defs */
extern SDPtr    SideDefs;	/* side defs data */
extern short NumVertexes;	/* number of vertexes */
extern VPtr     Vertexes;	/* vertex data */
extern short NumSegs;		/* number of segments */
extern SEPtr    Segs,		/* list of segments */
		LastSeg;	/* last segment in the list */
extern short NumSSectors;	/* number of subsectors */
extern SSPtr    SSectors,	/* list of subsectors */
		LastSSector;	/* last subsector in the list */
extern short NumSectors;	/* number of sectors */
extern SPtr     Sectors;	/* sectors data */

extern short MapMaxX;		/* maximum X value of map */
extern short MapMaxY;		/* maximum Y value of map */
extern short MapMinX;		/* minimum X value of map */
extern short MapMinY;		/* minimum Y value of map */

#endif 
