
/*
  [wad-id] wad open <file name>
  [directory] wad directory <wad-id>
  [level list] wad levels <wad-id>
  [map] wad map <level> ?<sublevel>?
  wad close all
  wad dumpmaps <file> <level> ?<sublevel>? ...
  wad extractmaps <file>
*/

#include "wad.h"
#include <tcl.h>

int Registered;
int Commercial;			/* 0=DOOM, 1=DOOM2 */

WadPtr  WadFileList = NULL;	/* linked list of Wad files */
MDirPtr MasterDir   = NULL;	/* the master directory */

MDirPtr  Level = NULL;		/* master dictionary entry for the level */
char    *LevelName = NULL;	/* official name for the level */

short NumThings = 0;		/* number of things */
TPtr     Things;		/* things data */
short NumLineDefs = 0;		/* number of line defs */
LDPtr    LineDefs;		/* line defs data */
short NumSideDefs = 0;		/* number of side defs */
SDPtr    SideDefs;		/* side defs data */
short NumVertexes = 0;		/* number of vertexes */
VPtr     Vertexes;		/* vertex data */
short NumSegs = 0;		/* number of segments */
SEPtr    Segs = NULL,		/* list of segments */
	 LastSeg = NULL;	/* last segment in the list */
short NumSSectors = 0;		/* number of subsectors */
SSPtr    SSectors = NULL,	/* list of subsectors */
	 LastSSector = NULL;	/* last subsector in the list */
short NumSectors = 0;		/* number of sectors */
SPtr     Sectors;		/* sectors data */

short MapMaxX = -32767;		/* maximum X value of map */
short MapMaxY = -32767;		/* maximum Y value of map */
short MapMinX = 32767;		/* minimum X value of map */
short MapMinY = 32767;		/* minimum Y value of map */

#define LINE_INTERNAL	0
#define LINE_EXTERNAL	1
#define LINE_DOOR	2

static int DumpMap( Tcl_Interp *interp, FILE *file, short level,
	short sublevel);
static int ExtractMap( Tcl_Interp *interp, FILE *file, char *name1,
	char *name2);

/* official names for all DOOM levels */
char *LevelNames[ 3][ 9] = {
{ "Hangar", "Nuclear Plant", "Toxic Refinery", "Command Control", "Phobos Lab",
  "Central Processing", "Computer Station", "Phobos Anomaly", "Military Base" },
{ "Deimos Anomaly", "Containment Area", "Refinery", "Deimos Lab",
  "Command Center", "Halls of the Damned", "Spawning Vats", "Tower of Babel",
  "Fortress of Mystery" },
{ "Hell Keep", "Slough of Despair", "Pandemonium", "House of Pain",
     "Unholy Cathedral", "Mt. Erebus", "Limbo", "Dis", "Warrens" }
};

/* official names for all DOOM ][ levels */
char *LevelNames2[ 32] = {
  "Entryway", "Underhalls", "The Gantlet", "The Focus", "The Waste Tunnels",
  "The Crusher", "Dead Simple", "Tricks and Traps", "The Pit", "Refueling Base",
  "'O' of Destruction!", "The Factory", "Downtown", "The Inmost Dens",
  "Industrial Zone", "Suburbs", "Tenements", "The Courtyard", "The Citadel",
  "Gotcha!", "Nirvana", "The Catacombs", "Barrels o' Fun", "The Chasm",
  "Bloodfalls", "The Abandoned Mines", "Monster Condo", "The Spirit World",
  "The Living End", "Icon of Sin", "Wolfenstein", "Grosse" 
};

Tcl_CmdProc Wad_OpenCmd;
Tcl_CmdProc Wad_DirectoryCmd;
Tcl_CmdProc Wad_LevelsCmd;
Tcl_CmdProc Wad_MapCmd;

int
WadCmd(
    ClientData clientData,
    Tcl_Interp *interp,
    int argc,
    char **argv
)
{
    static char *options= "open directory level map";
 
    if ( argc == 1)  {
	sprintf( interp->result, "wrong # args: must be \"%s option ?args?\"",
		argv[0]);
	return TCL_ERROR;
    }

    if ( strcmp( argv[1], "open") == 0)  {
	return Wad_OpenCmd( clientData, interp, argc, argv);
    }
    else if ( strcmp( argv[1], "directory") == 0)  {
	return Wad_DirectoryCmd( clientData, interp, argc, argv);
    }
    else if ( strcmp( argv[1], "levels") == 0)  {
	return Wad_LevelsCmd( clientData, interp, argc, argv);
    }
    else if ( strcmp( argv[1], "map") == 0)  {
	return Wad_MapCmd( clientData, interp, argc, argv);
    }
    else if ( argc == 3 && strcmp( argv[1], "close") == 0 &&
		strcmp( argv[2], "all") == 0) {
	CloseWadFiles();
	return TCL_OK;
    }
    else if ( strcmp( argv[1], "dumpmaps") == 0)  {
	return Wad_DumpmapsCmd( clientData, interp, argc, argv);
    }
    else if ( strcmp( argv[1], "extractmaps") == 0)  {
	return Wad_ExtractmapsCmd( clientData, interp, argc, argv);
    }

    sprintf( interp->result, "unknown option \"%s\": must be one of %s",
		argv[1], options);
    TCL_ERROR;
}

int
Wad_OpenCmd(
    ClientData clientData,
    Tcl_Interp *interp,
    int argc,
    char **argv
)
{
    FILE *file;
    char type[5];

    file= fopen( argv[2], "r");
    if ( file == NULL)  {
	sprintf( interp->result, "Could not open file %s", argv[2]);
	return TCL_ERROR;
    }
    fread( type, 4, 1, file);
    if ( strncmp( type, "IWAD", 4) == 0)  {
	fclose( file);
	if ( MasterDir != NULL)  {
	    Tcl_AppendResult( interp, "can't open \"", argv[2],
		"\" already opened main wad file", NULL);
	    return TCL_ERROR;
	}
	return OpenMainWad( interp, argv[2]);
    }
    else if ( strncmp( type, "PWAD", 4) == 0)  {
	fclose( file);
	if ( MasterDir == NULL)  {
	    interp->result= "Open main wad file first";
	    return TCL_ERROR;
	}
	return OpenPatchWad( interp, argv[2]);
    }
    
    return TCL_OK;
}

int
Wad_DirectoryCmd(
    ClientData clientData,
    Tcl_Interp *interp,
    int argc,
    char **argv
)
{
    MDirPtr dir;
    WadPtr wad;
    int i;

    char str[12];
    Tcl_DString result;
    Tcl_DStringInit( &result);

    if ( argc == 2)  {
	for (dir= MasterDir; dir != NULL; dir= dir->next)  {
	    Tcl_DStringStartSublist( &result);
	    Tcl_DStringAppendElement( &result, dir->dir.name);
	    Tcl_DStringAppendElement( &result, dir->wadfile->filename);
	    sprintf( str, "%d", dir->dir.size);
	    Tcl_DStringAppendElement( &result, str);
	    sprintf( str, "0x%08x", dir->dir.start);
	    Tcl_DStringAppendElement( &result, str);
	    Tcl_DStringEndSublist( &result);
	}
    }
    else if ( argc == 3)  {
	sscanf( argv[2], "wad%u", & wad);
	for (i= 0; i < wad->dirsize; i++)  {
	    Tcl_DStringStartSublist( &result);
	    Tcl_DStringAppendElement( &result, wad->directory[i].name);
	    sprintf( str, "%d", wad->directory[i].size);
	    Tcl_DStringAppendElement( &result, str);
	    sprintf( str, "0x%08x", wad->directory[i].start);
	    Tcl_DStringAppendElement( &result, str);
	    Tcl_DStringEndSublist( &result);
	}
    }
    else  {
	interp->result= "wad directory ?wad-id?";
	return TCL_ERROR;
    }

    Tcl_DStringResult( interp, &result);
    return TCL_OK;
}

int
Wad_LevelsCmd(
    ClientData clientData,
    Tcl_Interp *interp,
    int argc,
    char **argv
)
{
    MDirPtr dir;
    WadPtr wad;
    int i, level, sublevel;
    char str[12];
    Tcl_DString result;

    if ( argc != 3)  {
	sprintf( interp->result, "wrong # args: must be \"%s levels wad-id\"",
		argv[0]);
	return TCL_ERROR;
    }

    Tcl_DStringInit( &result);

    if ( argc == 3)  {
	sscanf( argv[2], "wad%u", & wad);
	for (i= 0; i < wad->dirsize; i++)  {
	    strncpy( str, wad->directory[i].name, 8);
	    if ( sscanf( str, "MAP%d", &level) == 1)  {
		sprintf( str, "%d", level);
		Tcl_DStringAppendElement( &result, str);
	    }
	    else if ( sscanf( str, "E%dM%d", &level, &sublevel) == 2)  {
		sprintf( str, "%d %d", level, sublevel);
	        Tcl_DStringAppendElement( &result, str);
	    }
	}
    }

    Tcl_DStringResult( interp, &result);
    return TCL_OK;
}

int
Wad_MapCmd(
    ClientData clientData,
    Tcl_Interp *interp,
    int argc,
    char **argv
)
{
    int episode= 0, mission, i;
    int prev_endx= ~0xffff, prev_endy= ~0xffff;
    Tcl_DString map;
    char str[1024], *color, *prev_color= "";
    static char *interior= "i", *exterior= "e", *door= "d";

    if ( argc != 3 && argc != 4)  {
	sprintf( interp->result,
		"wrong # args: must be \"%s map episode ?mission?\"",
		argv[0]);
	return TCL_ERROR;
    }

    if ( Tcl_GetInt( interp, argv[2], &mission) == TCL_ERROR)
	return;
    if ( argc == 4)
	episode= mission;
    if ( argc == 4 && Tcl_GetInt( interp, argv[3], &mission) == TCL_ERROR)
	return;

    if ( ReadLevelData( interp, episode, mission) == TCL_ERROR)
	return TCL_ERROR;

    Tcl_DStringInit( &map);

    sprintf( str, "%d %d %d %d", MapMinX, MapMinY, MapMaxX, MapMaxY);
    Tcl_DStringAppend( &map, str, -1);

    for( i= 0; i < NumLineDefs; i++)  {
	int startx= Vertexes[ LineDefs[i].start].x;
	int starty= MapMaxY-Vertexes[ LineDefs[i].start].y+MapMinY;
	int endx= Vertexes[ LineDefs[i].end].x;
	int endy= MapMaxY-Vertexes[ LineDefs[i].end].y+MapMinY;

	if ( LineDefs[i].flags & LDFLAG_IMPASS)  {
	    if ( LineDefs[i].flags & LDFLAG_TWOSIDED)
		color= "i";
	    else
		color= "e";
	}
	else {
	    switch ( LineDefs[i].type) {
	    case LDTYPE_DOOR:
	    case LDTYPE_BLUEDOOR:
	    case LDTYPE_YELLOWDOOR:
	    case LDTYPE_REDDOOR:
	    case LDTYPE_DOOR_NOCLOSE:
	    case LDTYPE_BLUEDOOR_NOCLOSE:
	    case LDTYPE_YELLOWDOOR_NOCLOSE:
	    case LDTYPE_REDDOOR_NOCLOSE:
	    case LDTYPE_DOOR_TURBO:
	    case LDTYPE_DOOR_TURBO_NOCLOSE:
	        color= "d";
	        break;
	    default: ;
	        color= "i";
	    }
	}
	if ( startx == prev_endx && starty == prev_endy &&
						color == prev_color)  {
	    sprintf( str, " %d %d", endx, endy);
	    Tcl_DStringAppend( &map, str, -1);
	}
	else  {
	    if ( i)  {
	        Tcl_DStringEndSublist( &map);
	        Tcl_DStringAppendElement( &map, prev_color);
	        Tcl_DStringEndSublist( &map);
	    }
	    Tcl_DStringStartSublist( &map);
	    Tcl_DStringStartSublist( &map);
	    sprintf( str, "%d %d %d %d", startx, starty, endx, endy);
	    Tcl_DStringAppend( &map, str, -1);
	}
	prev_endx= endx;
	prev_endy= endy;
	prev_color= color;
    }

    if ( i)  {
        Tcl_DStringEndSublist( &map);
        Tcl_DStringAppendElement( &map, prev_color);
        Tcl_DStringEndSublist( &map);
    }
    Tcl_DStringResult( interp, &map);
    ForgetLevelData();

    return TCL_OK;
}

int
Wad_DumpmapsCmd(
    ClientData clientData,
    Tcl_Interp *interp,
    int argc,
    char **argv
)
{
    FILE *file;
    int arg, rc;

    if ( argc < 4)  {
	interp->result= "wrong # args; wad dumpmaps file level ?sublevel? ...";
	return TCL_ERROR;
    }

    if (Tcl_GetOpenFile (interp, argv[2], 1, 1, &file) != TCL_OK)
	return TCL_ERROR;

    rewind( file);
    for ( arg= 3; arg < argc; arg++)  {
	short level=0, sublevel=0;
	rc= sscanf( argv[arg], "%hd %hd", &level, &sublevel);
	if ( rc == EOF || rc == 0)  {
	    Tcl_AppendResult( interp, "\"", argv[arg], "\" is not a level",
		NULL);
	    return TCL_ERROR;
	}
	if ( DumpMap( interp, file, level, sublevel) == TCL_ERROR)
	    return TCL_ERROR;
    }
    return TCL_OK;
}

int
DumpMap( Tcl_Interp *interp, FILE *file, short level, short sublevel)
{
    int i;
    short prev_endx= ~0x0 >> 1, prev_endy= ~0x0 >> 1;
    short color, prev_color= -1;
    short line[1024];
    short line_length= 0;

    if ( sublevel == 0)  {
	if ( ReadLevelData( interp, 0, (int)level) == TCL_ERROR)
	    return TCL_ERROR;
    }
    else
	if ( ReadLevelData( interp, (int)level, (int)sublevel) == TCL_ERROR)
	    return TCL_ERROR;

    fwrite( &level, sizeof(level), 1, file);
    fwrite( &sublevel, sizeof(sublevel), 1, file);

    fwrite( &MapMinX, sizeof(MapMinX), 1, file);
    fwrite( &MapMinY, sizeof(MapMinY), 1, file);
    fwrite( &MapMaxX, sizeof(MapMaxX), 1, file);
    fwrite( &MapMaxY, sizeof(MapMaxY), 1, file);

    for( i= 0; i < NumLineDefs; i++)  {
	short startx= Vertexes[ LineDefs[i].start].x;
	short starty= MapMaxY-Vertexes[ LineDefs[i].start].y+MapMinY;
	short endx= Vertexes[ LineDefs[i].end].x;
	short endy= MapMaxY-Vertexes[ LineDefs[i].end].y+MapMinY;

	if ( LineDefs[i].flags & LDFLAG_IMPASS)  {
	    if ( LineDefs[i].flags & LDFLAG_TWOSIDED)
		color= LINE_INTERNAL;
	    else
		color= LINE_EXTERNAL;
	}
	else {
	    switch ( LineDefs[i].type) {
	    case LDTYPE_DOOR:
	    case LDTYPE_BLUEDOOR:
	    case LDTYPE_YELLOWDOOR:
	    case LDTYPE_REDDOOR:
	    case LDTYPE_DOOR_NOCLOSE:
	    case LDTYPE_BLUEDOOR_NOCLOSE:
	    case LDTYPE_YELLOWDOOR_NOCLOSE:
	    case LDTYPE_REDDOOR_NOCLOSE:
	    case LDTYPE_DOOR_TURBO:
	    case LDTYPE_DOOR_TURBO_NOCLOSE:
	        color= LINE_DOOR;
	        break;
	    default: ;
	        color= LINE_INTERNAL;
	    }
	}
	if ( startx == prev_endx && starty == prev_endy &&
						color == prev_color)  {
	    line[line_length++]= endx;
	    line[line_length++]= endy;
	}
	else  {
	    if ( i)  {
		fwrite( &line_length, sizeof(line_length), 1, file);
		fwrite( line, sizeof(short), line_length, file);
		fwrite( &prev_color, sizeof(prev_color), 1, file);
		line_length= 0;
	    }
	    line[line_length++]= startx;
	    line[line_length++]= starty;
	    line[line_length++]= endx;
	    line[line_length++]= endy;
	}
	prev_endx= endx;
	prev_endy= endy;
	prev_color= color;
    }

    if ( i)  {
	fwrite( &line_length, sizeof(line_length), 1, file);
	fwrite( line, sizeof(short), line_length, file);
	fwrite( &color, sizeof(color), 1, file);
    }
    line_length= 0;
    fwrite( &line_length, sizeof(line_length), 1, file);
    ForgetLevelData();
}

int
Wad_ExtractmapsCmd(
    ClientData clientData,
    Tcl_Interp *interp,
    int argc,
    char **argv
)
{
    Tcl_DString levels;
    char level_name[12];
    short level, sublevel;
    FILE *file;

    if ( argc != 4)  {
	interp->result= "wrong # args; wad extractmaps <file> <var>";
	return TCL_ERROR;
    }

    if (Tcl_GetOpenFile (interp, argv[2], 0, 0, &file) != TCL_OK)
	return TCL_ERROR;

    rewind( file);
    Tcl_DStringInit( &levels);
    
    while (1)  {
	int rc= fread( &level, sizeof(level), 1, file);
	rc+= fread( &sublevel, sizeof(sublevel), 1, file);
	if ( rc != 2)
	    break;

	if ( sublevel == 0)
	    sprintf( level_name, "%hd", level);
	else
	    sprintf( level_name, "%hd %hd", level, sublevel);

	Tcl_DStringAppendElement( &levels, level_name);

	if ( ExtractMap( interp, file, argv[3], level_name) == TCL_ERROR)
	    return TCL_ERROR;
    }
    Tcl_DStringResult( interp, &levels);
    return TCL_OK;
}

int
ExtractMap( Tcl_Interp *interp, FILE *file, char *name1, char *name2)
{
    Tcl_DString map;
    char str[4096];
    short line_length;
    short line[1024];
    short color;

    fread( &MapMinX, sizeof(MapMinX), 1, file);
    fread( &MapMinY, sizeof(MapMinY), 1, file);
    fread( &MapMaxX, sizeof(MapMaxX), 1, file);
    fread( &MapMaxY, sizeof(MapMaxY), 1, file);

    Tcl_DStringInit( &map);

    sprintf( str, "%hd %hd %hd %hd", MapMinX, MapMinY, MapMaxX, MapMaxY);
    Tcl_DStringAppend( &map, str, -1);

    while ( fread( &line_length, sizeof(line_length), 1, file) == 1 &&
		line_length != 0)  {
	int i, n_output, str_length= 0;
	fread( line, sizeof(short), line_length, file);
	fread( &color, sizeof(color), 1, file);
	for ( i= 0; i < line_length; i++)  {
	    sprintf( str+str_length, "%hd %n", line[i], &n_output);
	    str_length+= n_output;
	}
	Tcl_DStringStartSublist( &map);
	Tcl_DStringAppendElement( &map, str);
	switch (color) {
	    case LINE_INTERNAL: Tcl_DStringAppendElement( &map, "i"); break;
	    case LINE_EXTERNAL: Tcl_DStringAppendElement( &map, "e"); break;
	    case LINE_DOOR: Tcl_DStringAppendElement( &map, "d"); break;
	    default: Tcl_DStringAppendElement( &map, "?"); break;
	}
	Tcl_DStringEndSublist( &map);
    }
    Tcl_SetVar2( interp, name1, name2, Tcl_DStringValue( &map), 0);
    Tcl_DStringFree( &map);
    return TCL_OK;
}

void
swapint( short *i)
{
   short t;

   ((char *) &t)[ 0] = ((char *) i)[ 1];
   ((char *) &t)[ 1] = ((char *) i)[ 0];
   *i = t;
}

void
swaplong( long *l)
{
   long t;

   ((char *) &t)[ 0] = ((char *) l)[ 3];
   ((char *) &t)[ 1] = ((char *) l)[ 2];
   ((char *) &t)[ 2] = ((char *) l)[ 1];
   ((char *) &t)[ 3] = ((char *) l)[ 0];
   *l = t;
}

int
TclError( Tcl_Interp *interp, char *errstr, ...)
{
   va_list args;
 
   va_start( args, errstr);
   vsprintf( interp->result, errstr, args);
   va_end( args);
   return TCL_ERROR;
}

int
ProgError( char *errstr, ...)
{
   va_list args;
 
   va_start( args, errstr);
   vprintf( errstr, args);
   va_end( args);
   return -1;
}

/*
   open the main WAD file, read in its directory and create the
   master directory
*/
int
OpenMainWad( Tcl_Interp *interp, char *filename)
{
   MDirPtr lastp, newp;
   WadPtr wad;
   long n;

   /* open the WAD file */
   wad = BasicWadOpen( interp, filename);
   if ( wad == NULL)
	return TCL_ERROR;
   if (strncmp( wad->type, "IWAD", 4))
      return TclError( interp, "\"%s\" is not the main WAD file", filename);

   /* create the master directory */
   lastp = NULL;
   for (n = 0; n < wad->dirsize; n++)
   {
      newp = (MDirPtr) malloc( sizeof( struct MasterDirectory));
      newp->next = NULL;
      newp->wadfile = wad;
      memcpy( (char *) &(newp->dir), (char *) &(wad->directory[ n]),
		sizeof( struct Directory));
      if (MasterDir)
	 lastp->next = newp;
      else
	 MasterDir = newp;
      lastp = newp;
   }

   /* check if registered/commercial version */
   if (FindMasterDir( MasterDir, "E2M1") != NULL)
      Registered = 1;
   else if (FindMasterDir( MasterDir, "MAP01") != NULL)
      Commercial = 1;
   else {
      /* If you remove this, bad things will happen to you... */
      Registered = 0;
      Commercial = 0;
   }
   sprintf( interp->result, "wad%u", wad);
   return TCL_OK;
}


/*
   open a patch WAD file, read in its directory and alter the master directory
*/
int
OpenPatchWad( Tcl_Interp *interp, char *filename)
{
    MDirPtr mdir;
    WadPtr wad;
    int n, replace, add;
    char entryname[ 12];

    if (! Exists( filename))  {
	sprintf( interp->result, "Warning: patch WAD file \"%s\" doesn't exist."
		"  Ignored.\n", filename);
	return TCL_ERROR;
    }

    wad = BasicWadOpen( interp, filename);
    if ( wad == NULL)
	return TCL_ERROR;
    if ( strncmp( wad->type, "PWAD", 4) != 0)
	return TclError( interp, "\"%s\" is not a patch WAD file", filename);

    replace= 0;
    add= 0;
    for (n = 0; n < wad->dirsize; n++)  {
	int is_level= 0, foo;
	strncpy( entryname, wad->directory[ n].name, 8);
	entryname[8]= '\0';
	if ( sscanf ( entryname, "MAP%d", &foo) == 1 ||
		sscanf ( entryname, "E%dM%d", &foo, &foo) == 2)
	    is_level= 1;

	if (replace == 0 && add == 0)  {
	    mdir= FindMasterDir( MasterDir, wad->directory[ n].name);
	    if (! mdir)  {
		if ( is_level)
		    add= 11;
		else
		    add= 1;
	    }
	    else if ( is_level)
		replace= 10;
	}
	else if ( replace)  {
	    mdir= mdir->next;
	    if (mdir == NULL || strncmp(mdir->dir.name, entryname, 8))
		return TclError( interp,
			"\\%s\" is not an understandable PWAD file "
			"(error with %s)", filename, entryname);
	    replace--;
	}
	if ( add)  {
	    mdir= MasterDir;
	    while (mdir->next) mdir= mdir->next;
	    mdir->next= (MDirPtr) malloc( sizeof( struct MasterDirectory));
	    mdir= mdir->next;
	    mdir->next= NULL;
	    add--;
	}

	mdir->wadfile = wad;
	memcpy( (char *) &(mdir->dir), (char *) &(wad->directory[ n]),
		sizeof( struct Directory));
    }
    sprintf( interp->result, "wad%u", wad);
    return TCL_OK;
}


/*
   close all the Wad files, deallocating the WAD file structures
*/
void
CloseWadFiles()
{
   MDirPtr curd, nextd;
   WadPtr curw, nextw;

   /* close the Wad files */
   curw = WadFileList;
   WadFileList = NULL;
   while (curw)
   {
      nextw = curw->next;
      fclose( curw->fileinfo);
      free( curw->directory);
      free( curw);
      curw = nextw;
   }

   /* delete the master directory */
   curd = MasterDir;
   MasterDir = NULL;
   while (curd)
   {
      nextd = curd->next;
      free( curd);
      curd = nextd;
   }
}


/*
   basic opening of WAD file and creation of node in Wad linked list
*/
WadPtr
BasicWadOpen( Tcl_Interp *interp, char *filename)
{
   WadPtr curw, prevw;
   long i;

   /* find the WAD file in the Wad file list */
   prevw = WadFileList;
   if (prevw)
   {
      curw = prevw->next;
      while (curw && strcmp( filename, curw->filename))
      {
	 prevw = curw;
	 curw = prevw->next;
      }
   }
   else
      curw = NULL;

   /* if this entry doesn't exist, add it to the WadFileList */
   if (! curw)
   {
      curw = (WadPtr) malloc( sizeof( struct WadFileInfo));
      if (! prevw)
	 WadFileList = curw;
      else
	 prevw->next = curw;
      curw->next = NULL;
      curw->filename = strdup( filename);
   }

   /* open the file */
   if ((curw->fileinfo = fopen( filename, "rb")) == NULL)
   {
      if (! prevw)
	 WadFileList = NULL;
      else
	 prevw->next = curw->next;
      free( curw);
      TclError( interp, "error opening \"%s\"", filename);
      return NULL;
   }

   /* read in the WAD directory info */
   BasicWadRead( curw, curw->type, 4);
   if (strncmp( curw->type, "IWAD", 4) && strncmp( curw->type, "PWAD", 4))  {
      TclError( interp, "\"%s\" is not a valid WAD file", filename);
      fclose( curw->fileinfo);
      curw->fileinfo= NULL;
      return NULL;
   }
   BasicWadRead( curw, &curw->dirsize, sizeof( curw->dirsize));
   swaplong( &(curw->dirsize));
   BasicWadRead( curw, &curw->dirstart, sizeof( curw->dirstart));
   swaplong( &(curw->dirstart));

   /* read in the WAD directory itself */
   curw->directory = (DirPtr) malloc( sizeof( struct Directory) *
		curw->dirsize);
   BasicWadSeek( curw, curw->dirstart);
   BasicWadRead( curw, curw->directory, sizeof( struct Directory) *
		curw->dirsize);

   for (i = 0; i < curw->dirsize; i++)
   {
      DirPtr d = &(curw->directory[ i]);
      swaplong( &(d->start));
      swaplong( &(d->size));
   }

   fclose( curw->fileinfo);
   curw->fileinfo= NULL;
   return curw;
}


/*
   read bytes from a file and store it into an address with error checking
*/
void
BasicWadRead( WadPtr wadfile, void *addr, long size)
{
   if (fread( addr, 1, size, wadfile->fileinfo) != size)
      ProgError( "error reading from \"%s\"", wadfile->filename);
}


/*
   go to offset of a file with error checking
*/
void
BasicWadSeek( WadPtr wadfile, long offset)
{
   if (fseek( wadfile->fileinfo, offset, 0))
      ProgError( "error reading from \"%s\"", wadfile->filename);
}


/*
   find an entry in the master directory
*/
MDirPtr
FindMasterDir( MDirPtr from, char *name)
{
   while (from)
   {
      if (! strncmp( from->dir.name, name, 8))
	 break;
      from = from->next;
   }
   return from;
}


/*
   check if a file exists and is readable
*/
int
Exists( char *filename)
{
   FILE *test;

   if (! (test = fopen( filename, "rb")))
      return 0;
   fclose( test);
   return 1;
}


/*
   read in the level data
*/
int
ReadLevelData( Tcl_Interp *interp, int episode, int mission)
{
   MDirPtr dir;
   WadPtr wadfile= NULL;
   char name[ 9];
   short n, m, val;
   short OldNumVertexes;
   short *VertexUsed;

   /* find the various level information from the master directory */
   if (! episode)
      sprintf( name, "MAP%02d", mission);
   else
      sprintf( name, "E%dM%d", episode, mission);
   Level = FindMasterDir( MasterDir, name);
   if (! Level)
      return TclError( interp, "level data not found");
   if (! episode)
      LevelName = LevelNames2[ mission - 1];
   else
      LevelName = LevelNames[ episode - 1][ mission - 1];

   if ( Level->wadfile->fileinfo == NULL)
      Level->wadfile->fileinfo= fopen( Level->wadfile->filename, "r");
   if ( Level->wadfile->fileinfo == NULL)
      return TclError( interp, "could not open wad file");

   /* read in the Things data */
   dir = FindMasterDir( Level, "THINGS");
   if (dir)
      NumThings = (short) (dir->dir.size / sizeof( struct Thing));
   else
      NumThings = 0;

   if (NumThings > 0)
   {
      Things = (TPtr) malloc( (unsigned long) NumThings *
		sizeof( struct Thing));
      BasicWadSeek( dir->wadfile, dir->dir.start);
      for (n = 0; n < NumThings; n++)
      {
	 BasicWadRead( dir->wadfile, &(Things[ n].xpos), 2);
	 swapint( &(Things[ n].xpos));
	 BasicWadRead( dir->wadfile, &(Things[ n].ypos), 2);
	 swapint( &(Things[ n].ypos));
	 BasicWadRead( dir->wadfile, &(Things[ n].angle), 2);
	 swapint( &(Things[ n].angle));
	 BasicWadRead( dir->wadfile, &(Things[ n].type), 2);
	 swapint( &(Things[ n].type));
	 BasicWadRead( dir->wadfile, &(Things[ n].when), 2);
	 swapint( &(Things[ n].when));
      }
   }

   /* get the number of Vertices */
   dir = FindMasterDir( Level, "VERTEXES");
   if (dir)
      OldNumVertexes = (short) (dir->dir.size / sizeof( struct Vertex));
   else
      OldNumVertexes = 0;
   if (OldNumVertexes > 0)
   {
      VertexUsed = (short *) malloc( OldNumVertexes * sizeof( short));
      for (n = 0; n < OldNumVertexes; n++)
	 VertexUsed[ n] = 0;
   }

   /* read in the LineDef information */
   dir = FindMasterDir( Level, "LINEDEFS");
   if (dir)
      NumLineDefs = (short) (dir->dir.size / sizeof( struct LineDef));
   else
      NumLineDefs = 0;
   if (NumLineDefs > 0)
   {
      LineDefs = (LDPtr) malloc( (unsigned long) NumLineDefs *
		sizeof( struct LineDef));
      BasicWadSeek( dir->wadfile, dir->dir.start);
      for (n = 0; n < NumLineDefs; n++)
      {
	 BasicWadRead( dir->wadfile, &(LineDefs[ n].start), 2);
	 swapint( &(LineDefs[ n].start));
	 VertexUsed[ LineDefs[ n].start] = 1;
	 BasicWadRead( dir->wadfile, &(LineDefs[ n].end), 2);
	 swapint( &(LineDefs[ n].end));
	 VertexUsed[ LineDefs[ n].end] = 1;
	 BasicWadRead( dir->wadfile, &(LineDefs[ n].flags), 2);
	 swapint( &(LineDefs[ n].flags));
	 BasicWadRead( dir->wadfile, &(LineDefs[ n].type), 2);
	 swapint( &(LineDefs[ n].type));
	 BasicWadRead( dir->wadfile, &(LineDefs[ n].tag), 2);
	 swapint( &(LineDefs[ n].tag));
	 BasicWadRead( dir->wadfile, &(LineDefs[ n].sidedef1), 2);
	 swapint( &(LineDefs[ n].sidedef1));
	 BasicWadRead( dir->wadfile, &(LineDefs[ n].sidedef2), 2);
	 swapint( &(LineDefs[ n].sidedef2));
      }
   }

   /* read in the SideDef information */
   dir = FindMasterDir( Level, "SIDEDEFS");
   if (dir)
      NumSideDefs = (short) (dir->dir.size / sizeof( struct SideDef));
   else
      NumSideDefs = 0;
   if (NumSideDefs > 0)
   {
      SideDefs = (SDPtr) malloc( (unsigned long) NumSideDefs *
		sizeof( struct SideDef));
      BasicWadSeek( dir->wadfile, dir->dir.start);
      for (n = 0; n < NumSideDefs; n++)
      {
	 BasicWadRead( dir->wadfile, &(SideDefs[ n].xoff), 2);
	 swapint( &(SideDefs[ n].xoff));
	 BasicWadRead( dir->wadfile, &(SideDefs[ n].yoff), 2);
	 swapint( &(SideDefs[ n].yoff));
	 BasicWadRead( dir->wadfile, &(SideDefs[ n].tex1), 8);
	 BasicWadRead( dir->wadfile, &(SideDefs[ n].tex2), 8);
	 BasicWadRead( dir->wadfile, &(SideDefs[ n].tex3), 8);
	 BasicWadRead( dir->wadfile, &(SideDefs[ n].sector), 2);
	 swapint( &(SideDefs[ n].sector));
      }
   }

   /* read in the Vertices which are all the corners of the level, but ignore
    * the Vertices not used in any LineDef (they usually are at the end of
    * the list).
    */

   NumVertexes = 0;
   for (n = 0; n < OldNumVertexes; n++)
      if (VertexUsed[ n])
	 NumVertexes++;
   if (NumVertexes > 0)
   {
      Vertexes = (VPtr) malloc( (unsigned long) NumVertexes *
		sizeof( struct Vertex));
      dir = FindMasterDir( Level, "VERTEXES");
      BasicWadSeek( dir->wadfile, dir->dir.start);
      MapMaxX = -32767;
      MapMaxY = -32767;
      MapMinX = 32767;
      MapMinY = 32767;
      m = 0;
      for (n = 0; n < OldNumVertexes; n++)
      {
	 BasicWadRead( dir->wadfile, &val, 2);
	 swapint( &val);
	 if (VertexUsed[ n])
	 {
	    if (val < MapMinX)
	       MapMinX = val;
	    if (val > MapMaxX)
	       MapMaxX = val;
	    Vertexes[ m].x = val;
	 }
	 BasicWadRead( dir->wadfile, &val, 2);
	 swapint( &val);
	 if (VertexUsed[ n])
	 {
	    if (val < MapMinY)
	       MapMinY = val;
	    if (val > MapMaxY)
	       MapMaxY = val;
	    Vertexes[ m].y = val;
	    m++;
	 }
      }
      if (m != NumVertexes)
	 TclError(interp, "inconsistency in the Vertexes data");
   }

   if (OldNumVertexes > 0)
   {
      /* update the Vertex numbers in the LineDefs (not really necessary */
      m = 0;
      for (n = 0; n < OldNumVertexes; n++)
	 if (VertexUsed[ n])
	    VertexUsed[ n] = m++;
      for (n = 0; n < NumLineDefs; n++)
      {
	 LineDefs[ n].start = VertexUsed[ LineDefs[ n].start];
	 LineDefs[ n].end = VertexUsed[ LineDefs[ n].end];
      }
      free( VertexUsed);
   }

   /* ignore the Segs, SSectors and Nodes */

   /* read in the Sectors information */
   dir = FindMasterDir( Level, "SECTORS");
   if (dir)
      NumSectors = (short) (dir->dir.size / sizeof( struct Sector));
   else
      NumSectors = 0;
   if (NumSectors > 0)
   {
      Sectors = (SPtr) malloc( (unsigned long) NumSectors *
		sizeof( struct Sector));
      BasicWadSeek( dir->wadfile, dir->dir.start);
      for (n = 0; n < NumSectors; n++)
      {
	 BasicWadRead( dir->wadfile, &(Sectors[ n].floorh), 2);
	 swapint( &(Sectors[ n].floorh));
	 BasicWadRead( dir->wadfile, &(Sectors[ n].ceilh), 2);
	 swapint( &(Sectors[ n].ceilh));
	 BasicWadRead( dir->wadfile, &(Sectors[ n].floort), 8);
	 BasicWadRead( dir->wadfile, &(Sectors[ n].ceilt), 8);
	 BasicWadRead( dir->wadfile, &(Sectors[ n].light), 2);
	 swapint( &(Sectors[ n].light));
	 BasicWadRead( dir->wadfile, &(Sectors[ n].special), 2);
	 swapint( &(Sectors[ n].special));
	 BasicWadRead( dir->wadfile, &(Sectors[ n].tag), 2);
	 swapint( &(Sectors[ n].tag));
      }
   }

   fclose( Level->wadfile->fileinfo);
   Level->wadfile->fileinfo= NULL;
}


/*
   forget the level data
*/
void
ForgetLevelData()
{
   /* forget the Things */
   NumThings = 0;
   if (Things)
      free( Things);
   Things = NULL;

   /* forget the Vertices */
   NumVertexes = 0;
   if (Vertexes)
      free( Vertexes);
   Vertexes = NULL;

   /* forget the LineDefs */
   NumLineDefs = 0;
   if (LineDefs)
      free( LineDefs);
   LineDefs = NULL;

   /* forget the SideDefs */
   NumSideDefs = 0;
   if (SideDefs)
      free( SideDefs);
   SideDefs = NULL;

   /* forget the Sectors */
   NumSectors = 0;
   if (Sectors)
      free( Sectors);
   Sectors = NULL;

   /* forget the level pointers */
   Level = NULL;
   LevelName = NULL;
}



/*
   get the class of a thing
*/
short
GetThingClass( short type)
{
   switch( type)
   {
   /* Starting points */
   case THING_PLAYER1:
   case THING_PLAYER2:
   case THING_PLAYER3:
   case THING_PLAYER4:
   case THING_DEATHMATCH:
      return CLASS_START;
   case THING_TELEPORT:
      return CLASS_TELEPORT;

   /* Enhancements & spheres */
   case THING_BLUECARD:
   case THING_YELLOWCARD:
   case THING_REDCARD:
   case THING_BLUESKULLKEY:
   case THING_YELLOWSKULLKEY:
   case THING_REDSKULLKEY:
   case THING_GREENARMOR:
   case THING_BLUEARMOR:
   case THING_SOULSPHERE:
   case THING_MEGASPHERE:	/* ][ */
   case THING_RADSUIT:
   case THING_COMPMAP:
   case THING_BLURSPHERE:
   case THING_BERSERK:
   case THING_INVULN:
   case THING_LITEAMP:
   case THING_BACKPACK:
      return CLASS_ENHANCE;

   /* Armor & health bonuses & small ammo */
   case THING_ARMBONUS1:
   case THING_HLTBONUS1:
   case THING_STIMPACK:
   case THING_MEDIKIT:
   case THING_AMMOCLIP:
   case THING_SHELLS:
   case THING_ROCKET:
   case THING_ENERGYCELL:
      return CLASS_BONUS;

   /* Weapons & bulk ammo */
   case THING_CHAINSAW:
   case THING_SHOTGUN:
   case THING_COMBATGUN:	/* ][ */
   case THING_CHAINGUN:
   case THING_LAUNCHER:
   case THING_PLASMAGUN:
   case THING_BFG9000:
   case THING_AMMOBOX:
   case THING_SHELLBOX:
   case THING_ROCKETBOX:
   case THING_ENERGYPACK:
      return CLASS_WEAPON;

   /* Enemies */
   case THING_HUMAN:
   case THING_SERGEANT:
   case THING_COMMANDO:		/* ][ */
   case THING_IMP:
   case THING_DEMON:
   case THING_SPECTRE:
   case THING_LOSTSOUL:
   case THING_CACODEMON:
   case THING_PAINELEM:		/* ][ */
   case THING_MANCUBUS:		/* ][ */
   case THING_REVENANT:		/* ][ */
   case THING_KNIGHT:		/* ][ */
   case THING_BARON:
   case THING_ARACHNO:		/* ][ */
   case THING_ARCHVILE:		/* ][ */
   case THING_CYBERDEMON:
   case THING_SPIDERBOSS:
   case THING_WOLF3DSS:		/* ][ */
      return CLASS_ENEMY;

   /* Specials */
   case THING_BARREL:
   case THING_KEEN:		/* ][ */
      return CLASS_BARREL;
   case THING_BOSSBRAIN:	/* ][ */
   case THING_BOSSSHOOT:	/* ][ */
   case THING_SPAWNSPOT:	/* ][ */
      return CLASS_DECOR;

   /* Decorations, et al */
   case THING_TECHCOLUMN:
   case THING_TGREENPILLAR:
   case THING_TREDPILLAR:
   case THING_SGREENPILLAR:
   case THING_SREDPILLAR:
   case THING_PILLARHEART:
   case THING_PILLARSKULL:
   case THING_EYEINSYMBOL:
   case THING_GREYTREE:
   case THING_BROWNSTUB:
   case THING_BROWNTREE:
   case THING_LAMP:
   case THING_TALLLAMP:		/* ][ */
   case THING_SHORTLAMP:	/* ][ */
   case THING_CANDLE:
   case THING_CANDELABRA:
   case THING_TBLUETORCH:
   case THING_TGREENTORCH:
   case THING_TREDTORCH:
   case THING_SBLUETORCH:
   case THING_SGREENTORCH:
   case THING_SREDTORCH:
   case THING_FLAMECAN:		/* ][ */
   case THING_DEADPLAYER:
   case THING_DEADHUMAN:
   case THING_DEADSERGEANT:
   case THING_DEADIMP:
   case THING_DEADDEMON:
   case THING_DEADCACODEMON:
   case THING_DEADLOSTSOUL:
   case THING_BONES:
   case THING_BONES2:
   case THING_POOLOFBLOOD:
   case THING_POOLOFBLOOD2:	/* ][ */
   case THING_POOLOFBLOOD3:	/* ][ */
   case THING_POOLOFBRAINS:	/* ][ */
   case THING_SKULLTOPPOLE:
   case THING_HEADSKEWER:
   case THING_PILEOFSKULLS:
   case THING_IMPALEDBODY:
   case THING_IMPALEDBODY2:
   case THING_SKULLSINFLAMES:
   case THING_HANGINGSWAYING:
   case THING_HANGINGSWAYING2:
   case THING_HANGINGARMSOUT:
   case THING_HANGINGARMSOUT2:
   case THING_HANGINGONELEG:
   case THING_HANGINGONELEG2:
   case THING_HANGINGTORSO:
   case THING_HANGINGTORSO2:
   case THING_HANGINGLEG:
   case THING_HANGINGLEG2:
   case THING_HANGINGNOGUTS:	/* ][ */
   case THING_HANGINGNOGUTS2:	/* ][ */
   case THING_HANGINGLOOKDN:	/* ][ */
   case THING_HANGINGLOOKUP:	/* ][ */
   case THING_HANGINGTORSO3:	/* ][ */
   case THING_HANGINGTORSO4:	/* ][ */
      return CLASS_DECOR;
   }
   return CLASS_UNKNOWN;
}


/*
   get the size of a thing
*/
short
GetThingRadius( short type)
{
   switch( type)
   {
   case THING_SPIDERBOSS:
      return 128;
   case THING_ARACHNO:
      return 64;
   case THING_MANCUBUS:
   case THING_BOSSBRAIN:
   case THING_BOSSSHOOT:
      return 48;
   case THING_CYBERDEMON:
      return 40;
   case THING_BROWNTREE:
   case THING_SPAWNSPOT:
      return 32;
   case THING_CACODEMON:
   case THING_PAINELEM:
      return 31;
   case THING_DEMON:
   case THING_SPECTRE:
      return 30;
   case THING_KNIGHT:
   case THING_BARON:
      return 24;
   case THING_HUMAN:
   case THING_SERGEANT:
   case THING_COMMANDO:
   case THING_IMP:
   case THING_REVENANT:
   case THING_ARCHVILE:
   case THING_WOLF3DSS:
   case THING_BLUECARD:
   case THING_YELLOWCARD:
   case THING_REDCARD:
   case THING_BLUESKULLKEY:
   case THING_YELLOWSKULLKEY:
   case THING_REDSKULLKEY:
   case THING_ARMBONUS1:
   case THING_HLTBONUS1:
   case THING_GREENARMOR:
   case THING_BLUEARMOR:
   case THING_STIMPACK:
   case THING_MEDIKIT:
   case THING_SOULSPHERE:
   case THING_MEGASPHERE:
   case THING_RADSUIT:
   case THING_COMPMAP:
   case THING_BLURSPHERE:
   case THING_BERSERK:
   case THING_INVULN:
   case THING_LITEAMP:
   case THING_CHAINSAW:
   case THING_SHOTGUN:
   case THING_COMBATGUN:
   case THING_CHAINGUN:
   case THING_LAUNCHER:
   case THING_PLASMAGUN:
   case THING_BFG9000:
   case THING_AMMOCLIP:
   case THING_AMMOBOX:
   case THING_SHELLS:
   case THING_SHELLBOX:
   case THING_ROCKET:
   case THING_ROCKETBOX:
   case THING_ENERGYCELL:
   case THING_ENERGYPACK:
   case THING_BACKPACK:
      return 20;
   case THING_BARREL:
      return 10;
   default: /* includes PLAYERs & LOSTSOUL */
      return 16;
   }
}

