/*
 * Copyright (c) 1995 Sun Microsystems, Inc.
 * All rights reserved.
 * 
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL SUN MICROSYSTEMS, INC. BE LIABLE TO ANY PARTY FOR
 * DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING
 * OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF SUN
 * MICROSYSTEMS, INC. HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * SUN MICROSYSTEMS, INC. SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
 * FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT.  THE SOFTWARE PROVIDED
 * HEREUNDER IS ON AN "AS IS" BASIS, AND SUN MICROSYSTEMS, INC. HAS NO
 * OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR
 * MODIFICATIONS.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <tcl.h>
#include <tk.h>
#include "doomarena.h"


/*
 * CmapHack
 */

#define NUM_FREE 19

static unsigned long pixels[256];
static int numPixels = 0;

static int
CMapHackCmd(ClientData clientData, Tcl_Interp *interp,
	    int argc, char **argv)
{
  Tk_Window tkwin = (Tk_Window)clientData;
  char c;
  int i, status, freenum;
  unsigned long *freeptr;
  XColor xcolor;
  Visual *visual;

  /* Make sure we are using an 8-bit PseudoColor visual */
  if (Tk_Depth(tkwin) != 8)
    return TCL_OK;

  visual = Tk_Visual(tkwin);
  if (visual->class != PseudoColor) {
    sprintf(interp->result, "%s: Visual must be PseudoColor\n", argv[0]);
    return TCL_ERROR;
  }

  if (argc != 2 && argc != 3) {
    sprintf(interp->result,
	    "wrong # args: should be \"%s option ?arg?\"", argv[0]);
    return TCL_ERROR;
  }

  c = argv[1][0];
  if (c == 'a' && !strcmp(argv[1], "alloc")) {
    XColor *colorDefaults;
    int i, pixel, colorCount = 0;

    if (argc == 3) {
      XColor dummy;
      int colorArgc, listArgc;
      char **colorArgv, **listArgv;

      if (Tcl_SplitList(interp, argv[2], &colorArgc, &colorArgv) != TCL_OK)
	return TCL_ERROR;
      colorCount = colorArgc;

      if (colorArgc > 0) {
	colorDefaults = (XColor *)malloc(colorArgc * sizeof(XColor));
	if (!colorDefaults) {
	  perror("malloc");
	  exit(1);
	}
      }

      for (i=0; i<colorArgc; i++) {
	if (Tcl_SplitList(interp, colorArgv[i], &listArgc, &listArgv)
	    != TCL_OK)
	  return TCL_ERROR;

	if (listArgc != 2) {
	  if (listArgc > 0)
	    free(listArgv);
	  free(colorDefaults);
	  free(colorArgv);
	  sprintf(interp->result,
		  "%s: Each color list entry must have exactly two items",
		  argv[0]);
	  return TCL_ERROR;
	}

	if (XLookupColor(Tk_Display(tkwin),
			  Tk_Colormap(tkwin),
			  listArgv[1], &dummy, &(colorDefaults[i]))) {
	  if (Tcl_GetInt(interp, listArgv[0], &pixel)
	      != TCL_OK) {
	    return TCL_ERROR;
	  }
	  colorDefaults[i].pixel = (short)pixel;
	  colorDefaults[i].red -= 256;
	  if (colorDefaults[i].red < 0)
	    colorDefaults[i].red = 0;
	  colorDefaults[i].green -= 256;
	  if (colorDefaults[i].green < 0)
	    colorDefaults[i].green = 0;
	  colorDefaults[i].blue -= 256;
	  if (colorDefaults[i].blue < 0)
	    colorDefaults[i].blue = 0;
	} else {
	  colorDefaults[i].pixel = 0;
	}

	free(listArgv);
      }

      if (colorArgc > 0)
	free(colorArgv);
    }

    /* Allocate all but the last 16 entries */
    xcolor.pixel = 0;
    do {
      xcolor.red = (short)(drand48() * 65535.0);
      xcolor.green = (short)(drand48() * 65535.0);
      xcolor.blue = (short)(drand48() * 65535.0);

      if (colorCount > 0) {
	for (i=0; i<colorCount; i++) {
	  if (colorDefaults[i].pixel == (xcolor.pixel + 1)) {
	    xcolor.red = colorDefaults[i].red;
	    xcolor.green = colorDefaults[i].green;
	    xcolor.blue = colorDefaults[i].blue;
	    break;
	  }
	}
      }

      status = XAllocColor(Tk_Display(tkwin), Tk_Colormap(tkwin), &xcolor);
      pixels[numPixels++] = xcolor.pixel;
    } while(status);
    numPixels--;

    if (numPixels > NUM_FREE) {
      freeptr = &(pixels[numPixels-NUM_FREE]);
      freenum = NUM_FREE;
    }
    else {
      freeptr = pixels;
      freenum = numPixels;
    }

    XFreeColors(Tk_Display(tkwin), Tk_Colormap(tkwin),
		freeptr, freenum, 0L);
    numPixels -= freenum;
  }
  else if (c == 'f' && !strcmp(argv[1], "free")) {
    if (argc != 2) {
      sprintf(interp->result,
	      "wrong # args: must be \"%s %s\"", argv[0], argv[1]);
      return TCL_ERROR;
    }

    XFreeColors(Tk_Display(tkwin), Tk_Colormap(tkwin), pixels, numPixels, 0L);
    numPixels = 0;
  }
  else if (c == 'c' && !strcmp(argv[1], "color")) {
    if (argc != 3) {
      sprintf(interp->result,
	      "wrong # args: must be \"%s %s index\"", argv[0], argv[1]);
      return TCL_ERROR;
    }

    if (Tcl_GetInt(interp, argv[2], (int *)&xcolor.pixel) != TCL_OK)
      return TCL_ERROR;

    XQueryColor(Tk_Display(tkwin), Tk_Colormap(tkwin), &xcolor);

    sprintf(interp->result, "#%04X%04X%04X",
	    xcolor.red, xcolor.green, xcolor.blue);
  }
  else {
    sprintf(interp->result,
	    "bad option \"%s\": must be alloc, free, or color", argv[1]);
    return TCL_ERROR;
  }

  return TCL_OK;
}

/*
 *  Disable input for all windows inside the icon window
 */

static int
Tk_DisableInput(ClientData clientData, Tcl_Interp *interp,
		int argc, char *argv[])
{
  Tk_Window mainwin = (Tk_Window)clientData;
  Tk_Window tkwin;
  XSetWindowAttributes atts;

  if (argc != 2) {
    sprintf(interp->result, "wrong # args: should be \"%s window\"",
	    argv[0]);
    return TCL_ERROR;
  }

  tkwin = Tk_NameToWindow(interp, argv[1], mainwin);
  if (!tkwin)
    return TCL_ERROR;

  atts.event_mask = Tk_Attributes(tkwin)->event_mask & ~ButtonPressMask;
  Tk_ChangeWindowAttributes(tkwin, CWEventMask, &atts);

  return TCL_OK;
}

/*
 *  Clip an image
 */

static int
Tk_ClipImage(ClientData clientData, Tcl_Interp *interp,
	     int argc, char *argv[])
{
  char buf[1024];
  char *srcImage, *destImage;
  int i, line, start, x, y, iscyan;
  Tcl_HashEntry *hPtr;
  Tk_PhotoHandle srcHandle, destHandle;
  Tk_PhotoImageBlock srcBlock, destBlock;
  unsigned char *pixelPtr;

  /* Check usage */
  if (argc < 2) {
    sprintf(interp->result,
	    "wrong # args: should be \"%s photoimage ?options?\"",
	    argv[0]);
    return TCL_ERROR;
  }

  /* Find the source image */
  srcImage = argv[1];
  srcHandle = Tk_FindPhoto(srcImage);
  if (srcHandle == NULL) {
    Tcl_AppendResult(interp, "image \"", srcImage, "\" doesn't",
		     " exist or is not a photo image", (char *) NULL);
    return TCL_ERROR;
  }
  Tk_PhotoGetImage(srcHandle, &srcBlock);

  /* Create the destination image */
  sprintf(buf, "image create photo -width %d -height %d ",
	  srcBlock.width, srcBlock.height+1);
  for (i=2; i<argc; i++) {
    sprintf(buf+strlen(buf), "{%s} ", argv[i]);
  }
  if (Tcl_Eval(interp, buf) == TCL_ERROR)
    return TCL_ERROR;

  destImage = strdup(interp->result);
  destHandle = Tk_FindPhoto(destImage);
  Tk_PhotoGetImage(destHandle, &destBlock);


  /* Loop through the image and copy everything that isn't cyan */
  line = 0;
  start = 0;

  for (y=0; y<srcBlock.height; y++) {
    for (x=0; x<srcBlock.width; x++) {
      pixelPtr = srcBlock.pixelPtr + (y * srcBlock.width + x) * 3;
      iscyan = (pixelPtr[0] == 0 && pixelPtr[1] == 255 && pixelPtr[2] == 255);

      if (line) {
	if (iscyan) {
	  sprintf(buf, "%s copy %s -from %d %d %d %d -to %d %d %d %d",
		  destImage, srcImage, start, y, x, y+1, start, y, x, y+1);
	  if (Tcl_Eval(interp, buf) == TCL_ERROR)
	    return TCL_ERROR;

	  line = 0;
	  start = 0;
	}
      }
      else {
	if (!iscyan) {
	  line = 1;
	  start = x;
	}
      }
    }
    if (line) {
      sprintf(buf, "%s copy %s -from %d %d %d %d -to %d %d %d %d",
	      destImage, srcImage, start, y, x, y+1, start, y, x, y+1);
      if (Tcl_Eval(interp, buf) == TCL_ERROR)
	return TCL_ERROR;

      line = 0;
      start = 0;
    }
  }

  sprintf(buf, "image delete %s", srcImage);
  if (Tcl_Eval(interp, buf) == TCL_ERROR)
    return TCL_ERROR;

  Tcl_SetResult(interp, destImage, TCL_DYNAMIC);
  return TCL_OK;
}

/*
 * Change property command
 */

static int
ChangePropertyCmd(ClientData clientData, Tcl_Interp *interp,
		  int argc, char *argv[])
{
  Tk_Window tkwin = (Tk_Window)clientData;
  Atom property, type, value;

  if (argc != 4) {
    sprintf(interp->result,
	    "wrong # args: should be \"%s property type value\"",
	    argv[0]);
    return TCL_ERROR;
  }

  Tk_MakeWindowExist(tkwin);

  property = XInternAtom(Tk_Display(tkwin), argv[1], False);
  type = XInternAtom(Tk_Display(tkwin), argv[2], False);
  value = XInternAtom(Tk_Display(tkwin), argv[3], False);

  XChangeProperty(Tk_Display(tkwin), Tk_WindowId(tkwin),
		  property, type, 32, PropModeReplace,
		  (const unsigned char *)(&value), 1);

  return NULL;
}

/*
 * Resource string command
 */
static int
ResourceStringCmd(ClientData clientData, Tcl_Interp *interp,
		  int argc, char **argv)
{
  Tk_Window tkwin = (Tk_Window)clientData;
  Tcl_SetResult(interp, XResourceManagerString(Tk_Display(tkwin)),
		TCL_DYNAMIC);
  return TCL_OK;
}

/*
 * ScreenCount
 */

static int
ScreenCountCmd(ClientData clientData, Tcl_Interp *interp,
	       int argc, char **argv)
{
  Tk_Window tkwin = (Tk_Window)clientData;

  if (argc != 1) {
    sprintf(interp->result,
	    "wrong # args: should be \"%s\"", argv[0]);
    return TCL_ERROR;
  }

  sprintf(interp->result, "%d", ScreenCount(Tk_Display(tkwin)));
  return TCL_OK;
}

/*
 * KeySym command
 */

#define KEYSYM_TO_STRING 1
#define STRING_TO_KEYSYM 2

static int
KeySymCmd(ClientData clientData, Tcl_Interp *interp,
	  int argc, char **argv)
{
  if (argc != 2) {
    sprintf(interp->result,
	    "wrong # args: should be \"%s keysym\"",
	    argv[0]);
    return TCL_ERROR;
  }

  if ((int)clientData == KEYSYM_TO_STRING) {
    char *name;
    int number;

    if (Tcl_GetInt(interp, argv[1], &number) != TCL_OK)
      return TCL_ERROR;

    name = XKeysymToString(number);
    if (name)
      interp->result = name;
  }
  else {
    int number;
    number = XStringToKeysym(argv[1]);
    sprintf(interp->result, "%d", number);
  }

  return TCL_OK;
}

/*
 * Init
 */

int
Doomarena_Init(Tcl_Interp *interp)
{
  Tk_Window tkwin = Tk_MainWindow(interp);

  Tcl_CreateCommand(interp, "cmaphack", CMapHackCmd,
		    (ClientData)tkwin, NULL);
  Tcl_CreateCommand(interp, "screencount", ScreenCountCmd,
		    (ClientData)tkwin, NULL);
  Tcl_CreateCommand(interp, "KeysymToString", KeySymCmd,
		    (ClientData)KEYSYM_TO_STRING, NULL);
  Tcl_CreateCommand(interp, "StringToKeysym", KeySymCmd,
		    (ClientData)STRING_TO_KEYSYM, NULL);
  Tcl_CreateCommand(interp, "disableinput", Tk_DisableInput,
		    (ClientData)tkwin, NULL);
  Tcl_CreateCommand(interp, "clipimage", Tk_ClipImage,
		    (ClientData)tkwin, NULL);
  Tcl_CreateCommand(interp, "changeproperty", ChangePropertyCmd,
		    (ClientData)tkwin, NULL);
  Tcl_CreateCommand(interp, "resourcestring", ResourceStringCmd,
		    (ClientData)tkwin, NULL);

  Tcl_CreateCommand( interp, "audio", AudioCmd, NULL, NULL);
  Tcl_CreateCommand( interp, "wad", WadCmd, NULL, NULL);

  Tcl_CreateCommand(interp, "tcp_server_start", ServerStartCmd, NULL, NULL);
  Tcl_CreateCommand(interp, "tcp_server_accept", ServerAcceptCmd, NULL, NULL);
  Tcl_CreateCommand(interp, "tcp_server_connect",
		    ServerConnectCmd, NULL, NULL);

  Tcl_CreateCommand( interp, "inet", InetCmd, NULL, NULL);

  Tcl_CreateCommand(interp, "DoomStats", DoomStatsCmd, NULL, NULL);

  return TCL_OK;
}
