
#--------------------------------------------------------------------------
# tkPriv elements used in this file:
#
# afterId -		Token returned by "after" for autoscanning.
# textlistPrev -	The last element to be selected or deselected
#			during a selection operation.
# textlistSelection -	All of the items that were selected before the
#			current selection operation (such as a mouse
#			drag) started;  used to cancel an operation.
#--------------------------------------------------------------------------

proc textlist {} {}
rename textlist {}
Class textlist

# Propagate all the text options
foreach option {
    bbox cget compare configure debug delete dlineinfo
    get index insert mark scan search see tag window xview yview
} {
    textlist instproc $option {args} [format {
	eval $self-win- %s $args
    } $option]
}

textlist instproc init {args} {
    $self next

    set argv(columns) ""
    set argv(selectmode) browse
    set argv(traceselect) ""
    set argv(weights) ""

    aload_argv $args argv

    set filteredArgs ""
    foreach elem [array names argv] {
	if {$elem != "columns" &&
	    $elem != "selectmode" &&
	    $elem != "traceselect" &&
	    $elem != "weights"} {

	    set filteredArgs "$filteredArgs -$elem $argv($elem)"
	}
    }

    rename $self $self-tmp-
    text $self
    rename $self $self-win-
    rename $self-tmp- $self

    setclass $self TextList
    eval $self configure -cursor {{}} -exportselection 0 \
	-height 16 $filteredArgs

    $self-win- mark set active 1.0

    global tkPriv
    set tkPriv(textlistPrev) ""
    set tkPriv(textlistSelection) ""

    $self instvar adjustPads columns weights ignoreColumns

    $self set columns [lindex $argv(columns) 0]
    $self set selectmode $argv(selectmode)
    $self set traceselect [lindex $argv(traceselect) 0]
    $self set tracerecurse 0
    $self set traceafter ""
    $self set traceselection ""

    if {$columns != ""} {
	bind $self <Configure> "$self _windowconfigure %w %h"

	set font [$self-win- cget -font]
	$self set adjustPads ""
	$self set weights ""
	$self set width 0
	$self set height 0

	set initWidth 0
	set prevWeight 1.0
	set length [llength $columns]
	set inputWeights [lindex $argv(weights) 0]
	set textwidth [textwidth "0" $font]

	set i 0
	foreach column $columns {
	    incr initWidth $column
	    lappend adjustPads [expr $column * $textwidth]

	    set weight [lindex $inputWeights $i]
	    if {$weight == ""} {set weight $prevWeight}
	    lappend weights [expr double($weight)]
	    set prevWeight $weight

	    lappend ignoreColumns 0
	    incr i
	}

	$self configure -width $initWidth
    }

    bind $self <Destroy> "$self destroy"
    return $self
}

textlist instproc _windowconfigure {{argW ""} {argH ""}} {
    $self instvar width height

    set force 0
    if {$argW == ""} {
	set force 1
	set argW [lindex [split [winfo geometry $self] x] 0]
    }
    if {$argH == ""} {
	set force 1
	set temp [lindex [split [winfo geometry $self] x] 1]
	set argH [lindex [split $temp +] 0]
    }

    set bd [$self-win- cget -borderwidth]
    set ht [$self-win- cget -highlightthickness]
    set padx [$self-win- cget -padx]
    set pady [$self-win- cget -pady]

    incr argW [expr -($padx+$bd+$ht) * 2]
    incr argH [expr -($pady+$bd+$ht) * 2]

    if {$force || $width != $argW} {
	$self instvar adjustPads columns info

	$self _resize $argW

	# The re-tabulation is done without calling "insertline"
	# for performance reasons.
	$self _savetags
	set state [$self-win- cget -state]
	$self configure -state normal
	$self delete 1.0 end
	
	set font [$self-win- cget -font]
	set spacewidth [textwidth " " $font]
	set size [array size info]

	if {$columns != ""} {
	    for {set i 0} {$i < $size} {incr i} {
		$self insert end \
		    [tabulate $info($i) $adjustPads $font $spacewidth]\n
	    }
	} else {
	    for {set i 0} {$i < $size} {incr i} {
		$self insert end $info($i)\n
	    }
	}

	$self configure -state $state
	$self _restoretags
    }

    set width $argW
    set height $argH
}

textlist instproc _resize {newWidth} {
    $self instvar columns weights adjustPads ignoreColumns

    set tmpweights $weights
    set textwidth [textwidth "0" [$self-win- cget -font]]
    set spacewidth [textwidth " " [$self-win- cget -font]]
    set totalWidth 0
    set totalWeight 0
    set numColumns [llength $columns]

    set i 0
    set ignore $ignoreColumns
    foreach column $columns {
	if {![lindex $ignore $i]} {
	    incr totalWidth [expr $column * $textwidth]
	    set totalWeight [expr $totalWeight + [lindex $tmpweights $i]]
	}
	incr i
    }

    set retry 1

    while {$retry} {
	set i 0
	set lastActive ""
	foreach val $ignore {
	    if {!$val} {
		set lastActive $i
	    }
	    incr i
	}

	if {$totalWeight == 0.0 && $lastActive != ""} {
	    set tmpweights [lreplace $tmpweights $lastActive $lastActive 1.0]
	    set totalWeight 1.0
	}

	set diff [expr $newWidth - $totalWidth]
	set tab 0
	set tabs ""
	set totalPad 0
	set adjustPads ""
	set retry 0
	set i 0

	foreach column $columns {
	    if {[lindex $ignore $i]} {
		lappend adjustPads 0
		incr i
		continue
	    }

	    set weight [lindex $tmpweights $i]
	    if {$i == $lastActive} {
		set adjustPad [expr $newWidth - $totalPad]
		if {$adjustPad < $spacewidth} {set adjustPad 0}
	    } else {
		set adjust [expr $diff * $weight / $totalWeight]
		set adjustPad [expr $column * $textwidth + round($adjust)]
		if {$adjustPad < $spacewidth} {set adjustPad 0}
		incr tab $adjustPad
		lappend tabs $tab
	    }

	    if {$adjustPad == 0} {
		incr totalWidth -[expr $column * $textwidth]
		set totalWeight [expr $totalWeight-$weight]
		set ignore [lreplace $ignore $i $i 1]
		set retry 1
		break
	    }

	    lappend adjustPads $adjustPad
	    incr totalPad $adjustPad
	    incr i
	}
    }

    $self configure -tabs $tabs
}

textlist instproc taginfo {option args} {
    switch $option {
	get {
	    set line [lindex $args 0]
	    set line2 [lindex $args 1]
	    if {$line == ""} {
		set line [lindex [$self curselection] 0]
		if {$line == ""} {return ""}
	    } else {
		if {$line == "end"} {set line last}
		set line [$self lineindex $line]
	    }

	    set result ""

	    if {$line2 == ""} {
		foreach tag [$self-win- tag names $line.0] {
		    set id [lindex $tag 0]
		    if {$id == "TAGINFO"} {
			set result [lindex $tag 1]
			break
		    }
		}
	    } else {
		if {$line2 == "end"} {set line2 last}
		set line2 [$self lineindex $line2]
		for {set i $line} {$i <= $line2} {incr i} {
		    foreach tag [$self-win- tag names $i.0] {
			set id [lindex $tag 0]
			if {$id == "TAGINFO"} {
			    lappend result [lindex $tag 1]
			    break
			}
		    }
		}
	    }
	    return $result
	}

	getlist {
	    set lines $args
	    if {$lines == ""} {
		set lines [$self curselection]
	    } 

	    set result ""
	    foreach line $lines {
		if {$line == "end"} {set line last}
		set line [$self lineindex $line]

		foreach tag [$self-win- tag names $line.0] {
		    set id [lindex $tag 0]
		    if {$id == "TAGINFO"} {
			lappend result [lindex $tag 1]
			break
		    }
		}
	    }
	    return $result
	}

	set {
	    set line [lindex $args 0]
	    set info [lindex $args 1]
	    if {$line == "end"} {set line last}
	    set line [$self lineindex $line]
	    $self-win- tag add [list TAGINFO $info] $line.0
	}

	line {
	    set info [lindex $args 0]
	    set info [list TAGINFO $info]
	    if {[catch {$self-win- tag ranges $info} index] != 0} {
		return ""
	    }
	    return [lindex [split [lindex $index 0] .] 0]
	}
    }
    
}

textlist instproc _savetags {} {
    $self instvar tags

    set tags ""
    foreach tag [$self-win- tag names] {
	set ranges [$self-win- tag ranges $tag]
	if {$ranges == ""} {continue}
	lappend tags "tag add [list $tag] $ranges"
    }
}

textlist instproc _restoretags {} {
    $self instvar tags

    foreach tag $tags {
	eval $self-win- $tag
    }
    $self unset tags
}

textlist instproc showcolumns {clist} {
    $self instvar ignoreColumns columns
    set numColumns [llength $columns]
    set ignore ""

    for {set i 0} {$i < $numColumns} {incr i} {
	if {[lsearch -exact $clist $i] != -1} {
	    lappend ignore 0
	} else {
	    lappend ignore [lindex $ignoreColumns $i]
	}
    }
    set ignoreColumns $ignore
    $self _windowconfigure
}

textlist instproc hidecolumns {clist} {
    $self instvar ignoreColumns columns
    set numColumns [llength $columns]
    set ignore ""

    for {set i 0} {$i < $numColumns} {incr i} {
	if {[lsearch -exact $clist $i] != -1} {
	    lappend ignore 1
	} else {
	    lappend ignore [lindex $ignoreColumns $i]
	}
    }
    set ignoreColumns $ignore
    $self _windowconfigure
}

textlist instproc insertline {line chars {taglist {} } } {
    $self instvar adjustPads columns info

    set line [$self lineindex $line]
    set index $line.0

    set state [$self-win- cget -state]
    set font [$self-win- cget -font]
    set spacewidth [textwidth " " $font]

    $self configure -state normal
    if {$columns != ""} {
	$self insert $index [tabulate $chars $adjustPads $font $spacewidth]\n $taglist
    } else {
	$self insert $index $chars\n $taglist
    }
    set src(0) $chars
    ainsert info [expr $line-1] src
    $self configure -state $state
}

textlist instproc deleteline {line1 {line2 ""}} {
    $self _traceselection

    set line1 [$self lineindex $line1]
    set index1 $line1.0

    if {$line2 != ""} {
	if {$line2 == "end"} {set line2 last}
	set line2 [$self lineindex $line2]
	set index2 [expr $line2+1].0
    } else {
	set index2 $index1
    }

    set state [$self-win- cget -state]
    $self configure -state normal
    $self delete $index1 $index2
    $self configure -state $state

    $self instvar columns info

    if {$line2 == ""} {
	areplace info [expr $line1-1]
    } else {
	areplace info [expr $line1-1] [expr $line2-1]
    }
}

textlist instproc getline {line1 {line2 ""}} {
    set line1 [$self lineindex $line1]
    if {$line2 == ""} {
	set line2 $line1
    } else {
	set line2 [$self lineindex $line2]
    }

    incr line1 -1
    incr line2 -1

    $self instvar info
    set contents ""
    for {set i $line1} {$i <= $line2} {incr i} {
	if {![info exists info($i)]} {
	    break
	}
	lappend contents $info($i)
    }
    return $contents
}

textlist instproc lineindex {line} {
    set end [lindex [split [$self-win- index end] .] 0]
    incr end -1

    if {$line == "end"} {
	set line $end
    } elseif {$line == "last"} {
	set line [expr $end-1]
    } else {
	if {[catch {expr int($line)}] != 0} {
	    set line [lindex [split [$self-win- index $line] .] 0]
	}
	if {$line < 1} {
	    set line 1
	} elseif {$line > $end} {
	    set line $end
	}
    }
    return $line
}

textlist instproc _traceselection {} {
    $self instvar traceselect tracerecurse traceafter

    if {$traceselect != "" && \
	$tracerecurse == 0 && \
	$traceafter == ""} {

	set traceafter [after 1 [format {
	    %s set tracerecurse 1
	    set status [catch {eval [%s set traceselect] %s} message]
	    %s set tracerecurse 0
	    %s set traceafter ""
	    if {$status != 0} {
		error $message $errorInfo $errorCode
	    }
	} $self $self $self $self $self]]
    }
}

textlist instproc curselection {} {
    set curselection ""
    set toggle 0
    foreach elem [$self-win- tag ranges sel] {
	if {$toggle} {
	    set end [lindex [split $elem .] 0]
	    for {set i $start} {$i < $end} {incr i} {
		lappend curselection $i
	    }
	    set toggle 0
	} else {
	    set start [lindex [split $elem .] 0]
	    set toggle 1
	}
    }
    return $curselection
}

textlist instproc selection {option args} {
    set end [$self lineindex end]

    set newArgs ""
    foreach arg $args {
	set arg [$self lineindex $arg]
	if {$arg == $end} {
	    incr arg -1
	}
	lappend newArgs $arg
    }
    set args $newArgs

    switch $option {
	anchor {
	    if {[$self-win- cget -state] == "disabled"} {return}
	    set line [lindex $args 0]
	    $self-win- mark set anchor $line.0
	}
	includes {
	    set line [lindex $args 0]
	    set names [$self-win- tag names $line.0]
	    return [expr [lsearch -exact $names sel] >= 0]
	}
	set -
	clear {
	    if {[$self-win- cget -state] == "disabled"} {return}
	    $self _traceselection
	    if {$option == "set"} {
		set option add
	    } else {
		set option remove
	    }
	    if {[llength $args] == 1} {
		set line [lindex $args 0]
		$self-win- tag $option sel $line.0 [expr $line+1].0
	    } else {
		set line1 [lindex $args 0]
		set line2 [lindex $args 1]
		if {$line1 > $line2} {
		    set tmp $line1
		    set line1 $line2
		    set line2 $tmp
		}
		$self-win- tag $option sel $line1.0 [expr $line2+1].0
	    }
	}
	default {
	    error "unknown option \"$option\": must be anchor, includes, set or clear"
	}
    }
}

#-------------------------------------------------------------------------
# The code below creates the default class bindings for listboxes.
#-------------------------------------------------------------------------

bind TextList <1> {
    tkTextListBeginSelect %W [%W index @%x,%y]
}
bind TextList <B1-Motion> {
    set tkPriv(x) %x
    set tkPriv(y) %y
    tkTextListMotion %W [%W index @%x,%y]
}
bind TextList <ButtonRelease-1> {
    tkCancelRepeat
    %W mark set active [%W index @%x,%y]
}
bind TextList <Shift-1> {
    tkTextListBeginExtend %W [%W index @%x,%y]
}
bind TextList <Control-1> {
    tkTextListBeginToggle %W [%W index @%x,%y]
}
bind TextList <B1-Leave> {
    set tkPriv(x) %x
    set tkPriv(y) %y
    tkTextListAutoScan %W
}
bind TextList <B1-Enter> {
    tkCancelRepeat
}

bind TextList <Up> {
    tkTextListUpDown %W -1
}
bind TextList <Shift-Up> {
    tkTextListExtendUpDown %W -1
}
bind TextList <Down> {
    tkTextListUpDown %W 1
}
bind TextList <Shift-Down> {
    tkTextListExtendUpDown %W 1
}
bind TextList <Left> {
    %W xview scroll -1 units
}
bind TextList <Control-Left> {
    %W xview scroll -1 pages
}
bind TextList <Right> {
    %W xview scroll 1 units
}
bind TextList <Control-Right> {
    %W xview scroll 1 pages
}
bind TextList <Prior> {
    %W yview scroll -1 pages
}
bind TextList <Next> {
    %W yview scroll 1 pages
}
bind TextList <Control-Prior> {
    %W xview scroll -1 pages
}
bind TextList <Control-Next> {
    %W xview scroll 1 pages
}
bind TextList <Home> {
    %W xview moveto 0
}
bind TextList <End> {
    %W xview moveto 1
}
bind TextList <Control-Home> {
    %W mark set active 1.0
    %W see 1.0
    %W selection clear 1 end
    %W selection set 1
}
bind TextList <Shift-Control-Home> {
    tkTextListDataExtend %W 0
}
bind TextList <Control-End> {
    %W mark set active end
    %W see end
    %W selection clear 1 end
    %W selection set end
}
bind TextList <Shift-Control-End> {
    tkTextListDataExtend %W end
}
bind TextList <F16> {
    if {[selection own -displayof %W] == "%W"} {
	clipboard clear -displayof %W
	clipboard append -displayof %W [selection get -displayof %W]
    }
}
bind TextList <space> {
    tkTextListBeginSelect %W [%W index active]
}
bind TextList <Select> {
    tkTextListBeginSelect %W [%W index active]
}
bind TextList <Control-Shift-space> {
    tkTextListBeginExtend %W [%W index active]
}
bind TextList <Shift-Select> {
    tkTextListBeginExtend %W [%W index active]
}
bind TextList <Escape> {
    tkTextListCancel %W
}
bind TextList <Control-slash> {
    tkTextListSelectAll %W
}
bind TextList <Control-backslash> {
    if {[%W cget -selectmode] != "browse"} {
	%W selection clear 1 end
    }
}

# Additional Tk bindings that aren't part of the Motif look and feel:

bind TextList <2> {
    %W scan mark %x %y
}
bind TextList <B2-Motion> {
    %W scan dragto %x %y
}


# tkTextListBeginSelect --
#
# This procedure is typically invoked on button-1 presses.  It begins
# the process of making a selection in the TextList.  Its exact behavior
# depends on the selection mode currently in effect for the TextList;
# see the Motif documentation for details.
#
# Arguments:
# w -		The TextList widget.
# el -		The element for the selection operation (typically the
#		one under the pointer).  Must be in numerical form.

proc tkTextListBeginSelect {w el} {
    global tkPriv

    set line [lindex [split $el .] 0]
    if {[$w set selectmode] == "multiple"} {
	if [$w selection includes $line] {
	    $w selection clear $line
	} else {
	    $w selection set $line
	}
    } else {
	$w selection clear 1 end
	$w selection set $line
	$w selection anchor $line
	set tkPriv(textlistSelection) {}
	set tkPriv(textlistPrev) $line
    }
}

# tkTextListMotion --
#
# This procedure is called to process mouse motion events while
# button 1 is down.  It may move or extend the selection, depending
# on the TextList's selection mode.
#
# Arguments:
# w -		The TextList widget.
# el -		The element under the pointer (must be a number).

proc tkTextListMotion {w el} {
    global tkPriv

    set line [lindex [split $el .] 0]
    if {$line == $tkPriv(textlistPrev)} {
	return
    }

    switch [$w set selectmode] {
	browse {
	    $w selection clear 1 end
	    $w selection set $line
	    set tkPriv(textlistPrev) $line
	}
	extended {
	    set anchor [lindex [split [$w index anchor] .] 0]
	    set i $tkPriv(textlistPrev)
	    if [$w selection includes anchor] {
		$w selection clear $i $line
		$w selection set anchor $line
	    } else {
		$w selection clear $i $line
		$w selection clear anchor $line
	    }
	    while {($i < $line) && ($i < $anchor)} {
		if {[lsearch $tkPriv(textlistSelection) $i] >= 0} {
		    $w selection set $i
		}
		incr i
	    }
	    while {($i > $line) && ($i > $anchor)} {
		if {[lsearch $tkPriv(textlistSelection) $i] >= 0} {
		    $w selection set $i
		}
		incr i -1
	    }
	    set tkPriv(textlistPrev) $line
	}
    }
}

# tkTextListBeginExtend --
#
# This procedure is typically invoked on shift-button-1 presses.  It
# begins the process of extending a selection in the TextList.  Its
# exact behavior depends on the selection mode currently in effect
# for the TextList;  see the Motif documentation for details.
#
# Arguments:
# w -		The TextList widget.
# el -		The element for the selection operation (typically the
#		one under the pointer).  Must be in numerical form.

proc tkTextListBeginExtend {w el} {
    if {([$w set selectmode] == "extended")
	    && [$w selection includes anchor]} {
	tkTextListMotion $w $el
    }
}

# tkTextListBeginToggle --
#
# This procedure is typically invoked on control-button-1 presses.  It
# begins the process of toggling a selection in the TextList.  Its
# exact behavior depends on the selection mode currently in effect
# for the TextList;  see the Motif documentation for details.
#
# Arguments:
# w -		The TextList widget.
# el -		The element for the selection operation (typically the
#		one under the pointer).  Must be in numerical form.

proc tkTextListBeginToggle {w el} {
    global tkPriv
    set line [lindex [split $el .] 0]

    if {[$w set selectmode] == "extended"} {
	set tkPriv(textlistSelection) [$w curselection]
	set tkPriv(textlistPrev) $line
	$w selection anchor $line
	if [$w selection includes $line] {
	    $w selection clear $line
	} else {
	    $w selection set $line
	}
    }
}

# tkTextListAutoScan --
# This procedure is invoked when the mouse leaves an entry window
# with button 1 down.  It scrolls the window up, down, left, or
# right, depending on where the mouse left the window, and reschedules
# itself as an "after" command so that the window continues to scroll until
# the mouse moves back into the window or the mouse button is released.
#
# Arguments:
# w -		The entry window.

proc tkTextListAutoScan {w} {
    global tkPriv
    set x $tkPriv(x)
    set y $tkPriv(y)
    if {$y >= [winfo height $w]} {
	$w yview scroll 1 units
    } elseif {$y < 0} {
	$w yview scroll -1 units
    } elseif {$x >= [winfo width $w]} {
	$w xview scroll 2 units
    } elseif {$x < 0} {
	$w xview scroll -2 units
    } else {
	return
    }
    tkTextListMotion $w [$w index @$x,$y]
    set tkPriv(afterId) [after 50 tkTextListAutoScan $w]
}

# tkTextListUpDown --
#
# Moves the location cursor (active element) up or down by one element,
# and changes the selection if we're in browse or extended selection
# mode.
#
# Arguments:
# w -		The TextList widget.
# amount -	+1 to move down one item, -1 to move back one item.

proc tkTextListUpDown {w amount} {
    global tkPriv

    set activeLine [lindex [split [$w index active] .] 0]
    incr activeLine $amount
    $w mark set active $activeLine.0
    $w see active

    switch [$w set selectmode] {
	browse {
	    $w selection clear 1 end
	    $w selection set active
	}
	extended {
	    $w selection clear 1 end
	    $w selection set active
	    $w selection anchor active
	    set tkPriv(textlistPrev) $activeLine
	    set tkPriv(textlistSelection) {}
	}
    }
}

# tkTextListExtendUpDown --
#
# Does nothing unless we're in extended selection mode;  in this
# case it moves the location cursor (active element) up or down by
# one element, and extends the selection to that point.
#
# Arguments:
# w -		The TextList widget.
# amount -	+1 to move down one item, -1 to move back one item.

proc tkTextListExtendUpDown {w amount} {
    if {[$w set selectmode] != "extended"} {
	return
    }
    set activeLine [lindex [split [$w index active] .] 0]
    incr activeLine $amount
    $w mark set active $activeLine.0
    $w see active
    tkTextListMotion $w [$w index active]
}

# tkTextListDataExtend
#
# This procedure is called for key-presses such as Shift-KEndData.
# If the selection mode isn't multiple or extend then it does nothing.
# Otherwise it moves the active element to el and, if we're in
# extended mode, extends the selection to that point.
#
# Arguments:
# w -		The TextList widget.
# el -		An integer element number.

proc tkTextListDataExtend {w el} {
    set line [lindex [split $el .] 0]
    set mode [$w set selectmode]

    if {$mode == "extended"} {
	$w mark set active $line.0
	$w see $line.0
        if [$w selection includes anchor] {
	    tkTextListMotion $w $el
	}
    } elseif {$mode == "multiple"} {
	$w mark set active $line.0
	$w see $line.0
    }
}

# tkTextListCancel
#
# This procedure is invoked to cancel an extended selection in
# progress.  If there is an extended selection in progress, it
# restores all of the items between the active one and the anchor
# to their previous selection state.
#
# Arguments:
# w -		The TextList widget.

proc tkTextListCancel w {
    global tkPriv
    if {[$w set selectmode] != "extended"} {
	return
    }
    set first [lindex [split [$w index anchor] .] 0]
    set last $tkPriv(textlistPrev)
    if {$first > $last} {
	set tmp $first
	set first $last
	set last $tmp
    }
    $w selection clear $first $last
    while {$first <= $last} {
	if {[lsearch $tkPriv(textlistSelection) $first] >= 0} {
	    $w selection set $first
	}
	incr first
    }
}

# tkTextListSelectAll
#
# This procedure is invoked to handle the "select all" operation.
# For single and browse mode, it just selects the active element.
# Otherwise it selects everything in the widget.
#
# Arguments:
# w -		The TextList widget.

proc tkTextListSelectAll w {
    set mode [$w set selectmode]
    if {($mode == "single") || ($mode == "browse")} {
	$w selection clear 1 end
	$w selection set active
    } else {
	$w selection set 1 end
    }
}
















