# ------------------------------------------------------------------
# Utility Procs
# ------------------------------------------------------------------

# ------------------------------------------------------------------
# ReconfigWidget: This is called by the "setclass" procedure to
# setup the configuration options for the new widget class.
# Configuration options where the old value is different from the
# old default value are not modified.
# ------------------------------------------------------------------

proc ReconfigWidget {w} {
    set opts [$w configure]
    set config ""
    foreach opt $opts {
	if {[llength $opt] != 5} {
	    continue
	}
	set option [lindex $opt 0]
	set name [lindex $opt 1]
	set class [lindex $opt 2]

	set value [option get $w $name $class]
	if {$value != ""} {
	    lappend config $option
	    lappend config $value
	}
    }
    if {$config != ""} {
	eval $w configure $config
    }
}

# ------------------------------------------------------------------
# BrightenColor: used to turn a background color into an active
# background color
# ------------------------------------------------------------------
proc BrightenColor {color percentage} {
    set color [parsecolor $color]
    if {$color == ""} {return ""}
    set lightColor ""
    set max 65535

    set mult [expr int(100 + $percentage)]
    foreach elem $color {
	set val [expr $mult * $elem / 100]
	if {$val > $max} {
	    set val $max
	}
	lappend lightColor $val
    }
    set lightColor [eval format {#%04x%04x%04x} $lightColor]
    return $lightColor
}

# ------------------------------------------------------------------
# DarkenColor: used to turn a background color into a disabled
# background color
# ------------------------------------------------------------------
proc DarkenColor {color percentage} {
    set color [parsecolor $color]
    if {$color == ""} {return ""}
    set darkColor ""

    set mult [expr int(100 - $percentage)]
    foreach elem $color {
	lappend darkColor [expr $mult * $elem / 100]
    }
    set darkColor [eval format {#%04x%04x%04x} $darkColor]
    return $darkColor
}

# ------------------------------------------------------------------
# SetupDefaultOptions
#
# Generate some good options based on four resources:
#   background
#   foreground
#   fontType:	charter, courier, helvetica, lucida,
#		new century schoolbook, times
#   fontSize:   8 10 12 14 18 24
# ------------------------------------------------------------------
proc SetupDefaultOptions {} {
    global argv
    aload_argv $argv argv_array

    #
    # Colors
    #
    global gBackground gForeground

    if {[info exists argv_array(background)]} {
	set background [lindex $argv_array(background) 0]
    } elseif {[info exists argv_array(bg)]} {
	set background [lindex $argv_array(bg) 0]
    } elseif {[info exists gBackground]} {
	set background $gBackground
    } else {
	set background [option get . background Background]
    }

    if {[info exists argv_array(foreground)]} {
	set foreground [lindex $argv_array(foreground) 0]
    } elseif {[info exists argv_array(fg)]} {
	set foreground [lindex $argv_array(fg) 0]
    } elseif {[info exists gForeground]} {
	set foreground $gForeground
    } else {
	set foreground [option get . foreground Foreground]
    }

    if {$background == ""} {
	set background [. cget -background]
    } else {
	. configure -background $background
    }
    if {$foreground == ""} {
	button .fg_clr
	set foreground [.fg_clr cget -foreground]
	destroy .fg_clr
    }

    SetupColorOptions $foreground $background

    #
    # Fonts
    #
    global gFontType gFontSize
    
    if {[info exists argv_array(fonttype)]} {
	set fontType [lindex $argv_array(fonttype) 0]
    } elseif {[info exists gFontType]} {
	set fontType [string tolower $gFontType]
    } else {
	set fontType [option get . fontType FontType]
    }

    if {[info exists argv_array(fontsize)]} {
	set fontSize [lindex $argv_array(fontsize) 0]
    } elseif {[info exists gFontSize]} {
	set fontSize $gFontSize
    } else {
	set fontSize [option get . fontSize FontSize]
    }

    SetupFontOptions $fontType $fontSize
    option add *Menubutton.takeFocus 0
}

# ------------------------------------------------------------------
# SetupColorOptions: setup the colors for a given fg and bg
# ------------------------------------------------------------------
proc SetupColorOptions {foreground background} {
    # Figure out a good highlight color
    set fg_color [parsecolor $foreground]
    if {$fg_color == ""} {error "Invalid color: $foreground"}
    set bg_color [parsecolor $background]
    if {$bg_color == ""} {error "Invalid color: $background"}

    set r [lindex $bg_color 0]
    set g [lindex $bg_color 1]
    set b [lindex $bg_color 2]
    set total [expr ($r * 0.4) + ($g * 0.5) + ($b * 0.1)]
    if {$total > 40000} {
	set highlight [DarkenColor $background 10]
    } else {
	set highlight [BrightenColor $background 15]
    }

    # Set the options in the database
    option add *activeForeground $foreground
    option add *selectForeground $foreground

    option add *background $background
    option add *foreground $foreground
    option add *highlightBackground $background
    option add *troughColor $background
    option add *Scrollbar.foreground $background
    option add *Scrollbar.troughColor $background
    option add *sliderForeground $background

    option add *activeBackground $highlight
    option add *selectBackground $highlight
    option add *Scale.activeForeground $highlight
    option add *Scrollbar.activeForeground $highlight
}

# ------------------------------------------------------------------
# SetupFontOptions: pick some good fonts for the supplied type and size
# ------------------------------------------------------------------
proc SetupFontOptions {fontType fontSize} { 
    # Determine which font to use
    if {$fontType == ""} {
	set fontType -adobe-helvetica
    }
    set fontType [string trim $fontType -]
    set fontType [split $fontType -]
    if {[llength $fontType] == 1} {
	set name [string tolower [lindex $fontType 0]]
	if {$name == "charter" || $name == "lucida"} {
	    set family *
	} else {
	    set family adobe
	}
    } else {
	set family [string tolower [lindex $fontType 0]]
	set name [string tolower [lindex $fontType 1]]
    }
    if {$name == "helvetica" || $name == "courier"} {
	set italic o
    } else {
	set italic i
    }
    set fontType -$family-$name

    set font_types {
	charter courier helvetica lucida
	{new century schoolbook} times
    }
    if {[lsearch -exact $font_types [string tolower $name]] == -1} {
	error "Invalid font type \"$name\": must be one of $font_types"
    }


    if {$fontSize == ""} {
	set fontSize 12
    }
    set font_sizes {8 10 12 14 18 24}
    if {[lsearch -exact $font_sizes $fontSize] == -1} {
	error "Invalid font size \"$fontSize\": must be one of $font_sizes"
    }
    set fontSize ${fontSize}0

    set boldFont $fontType-bold-r-normal-*-*-$fontSize-*
    set mediumFont $fontType-medium-r-normal-*-*-$fontSize-*
    set italicFont $fontType-medium-$italic-normal-*-*-$fontSize-*
    set boldItalicFont $fontType-bold-$italic-normal-*-*-$fontSize-*
    set textFont -*-courier-medium-r-normal-*-*-$fontSize-*

    option add *Scrollbar.placement right

    option add *font $boldFont
    option add *boldFont $boldFont
    option add *italicFont $italicFont
    option add *boldItalicFont $boldItalicFont

    option add *Text.font $textFont
    option add *Entry.font $mediumFont
    option add *TextList.font $mediumFont
}

# ------------------------------------------------------------------
# "ListboxButtonActivate" activates the given button whenever
# something is selected from the list
# ------------------------------------------------------------------

proc ListboxButtonActivate {list button} {
    set old [bind $list <Button-1>]
    if {$old == ""} {
	set old [bind Listbox <Button-1>]
    }
    bind $list <Button-1> [format {
	%s
	set index [lindex [%s curselection] 0]
    	if {$index != ""} {
	    %s configure -state normal
	}
    } $old $list $button]
}

proc ListboxBindVariable {list varname} {
    set bindings {<Button-1> <Shift-Button-1> <B1-Motion> <Shift-B1-Motion>}

    foreach elem $bindings {
	set old [bind $list $elem]
	if {$old == ""} {
	    set old [bind Listbox $elem]
	}
	bind $list $elem [format {
	    %s
	    set index [lindex [%s curselection] 0]
	    if {$index != ""} {
		set %s [%s get $index]
	    } else {
		set %s {}
	    }
	} $old $list $varname $list $varname]
    }
}

proc AdjustLabelState {label var} {
    global label_state_normal label_state_disabled

    if {[info exists label_state_normal] == 0} {
	button .lblb
	set label_state_normal \
	    [lindex [.lblb configure -foreground] 4]
	set label_state_disabled \
	    [lindex [.lblb configure -disabledforeground] 4]
	destroy .lblb
    }

    if {[info command $label] == ""} {return}

    if {$var == ""} {
	$label configure -foreground $label_state_disabled
    } else {
	$label configure -foreground $label_state_normal
    }
}

# ------------------------------------------------------------------
# The HyperLink command is used to either load an object into an
# already running tool, or launch a new tool if necessary.
# ------------------------------------------------------------------

proc HyperLink {name sendCommand execCommand} {
    busy {
    	foreach i [winfo interps] {
	    if [string match *${name}* $i] {
		catch {send $i {expr 999}} retval
		if {$retval == "999"} {
		    set code [catch {send $i $sendCommand} err]
		    if {$code != 0} {
			puts "HyperLink Error: $err"
		    }
		    return
		} else {
		    removeinterp $i
		}
	    }
	}
	eval exec $execCommand &
	after 1200
    }
}

# ------------------------------------------------------------------
# "tabbind" is used to bind two entries together so that they can
# be traversed with the TAB key.
# ------------------------------------------------------------------

proc tabbind {w1 w2} {
    bind $w1 <Tab> "focus $w2 ; $w2 select from 0 ; $w2 select to end"
    bind $w1 <Return> "focus $w2"
    bind $w2 <Shift-Tab> "focus $w1 ; $w1 select from 0 ; $w1 select to end"
}

# ------------------------------------------------------------------
# "PopupMenu" sets bindings on a menu such that it will behave as a
# popup menu on the given parent.
# ------------------------------------------------------------------

proc PopupMenu {menu {parent .}} {
    if {$parent == "."} {set parent all}

    bind $menu <ButtonRelease-1> {
	tk_invokeMenu %W
	if {[set %W-pinned] == 0} { %W unpost }
    }

    bind $parent <Button-2> "$menu post %X %Y ; set $menu-pinned 1"
    bind $parent <B2-Motion> "$menu post %X %Y ; set $menu-pinned 1"

    bind $parent <ButtonPress-3> "$menu post %X %Y ; set $menu-pinned 0"
    bind $parent <B3-Motion> "PopupMenu-B3-Motion $menu %X %Y"
    bind $parent <ButtonRelease-3> "PopupMenu-B3-Release $menu %X %Y"
}

proc PopupMenu-B3-Motion {menu rx ry} {
    scan [winfo geometry $menu] %dx%d+%d+%d w h x y
    if {$rx > $x && $rx < [expr $x+$w] &&
	$ry > $y && $ry < [expr $y+$h]} {
	set index [$menu index @[expr $ry-$y]]
	$menu activate $index
    } else {
	$menu activate none
    }
}

proc PopupMenu-B3-Release {menu rx ry} {
    scan [winfo geometry $menu] %dx%d+%d+%d w h x y
    set margin 3

    if {$x > [expr $rx-$margin] && $x < [expr $rx+$margin] &&
	$y > [expr $ry-$margin] && $y < [expr $ry+$margin]} {
	return
    } elseif {$rx > $x && $rx < [expr $x+$w] &&
    	      $ry > $y && $ry < [expr $y+$h]} {
	set index [$menu index @[expr $ry-$y]]
	$menu invoke $index
    }
    $menu unpost
}
