########################################################################
#
#  TreeBrowser
#
#  The Browser class implements a graphical browser that can be used to
#  browse a hierarchy of items.  It is meant to be a superclass to other
#  classes and should not be directly instantiated.
#
########################################################################

proc TreeBrowser {} {}
rename TreeBrowser {}
Class TreeBrowser

TreeBrowser instproc init {} {
    # Initialize superclasses
    $self next

    #  Create a window with the same name as this object
    rename $self $self-tmp-
    frame $self -class $class
    catch {rename $self-win- ""}
    rename $self $self-win-
    rename $self-tmp- $self

    # Initialize the instance variables
    $self _initvars

    # Initialize all the widgets
    $self _initwidgets

    # Initialize the root
    $self _initroot
}

TreeBrowser instproc destroy {} {
    rename $self-win- {}
    destroy $self
}

TreeBrowser instproc busy {code} {
    $self instvar doBusy
    if {$doBusy} {
	busy {
	    uplevel $code
	}
    } else {
	uplevel $code
    }
}

TreeBrowser instproc update {args} {
    $self instvar doBusy
    if {$doBusy} {
	uplevel update $args
    }
}

TreeBrowser instproc _initvars {} {
    $self set tree(0) 0
    $self unset tree(0)

    $self set activate_label {Activate}
    $self set activate_callback {}
    $self set isActive_callback {}
    $self set initClientData -1
    $self set sparse 0
    $self set title "Tree Browser"

    $self set doBusy 0
    $self set clientDataIndex 3
    $self set dirCodeIndex 2
    $self set displayIndex 4
    $self set expandList {}
    $self set expandedIndex 5
    $self set fileCodeIndex 1
    $self set nameIndex 0
    $self set root {}
    $self set sep {}

    $self set NoSuchEntry {Entry does not exist}
}

TreeBrowser instproc _initwidgets {} {
    set pads {}
    set bttn_pads {}
	
    # Labels
    frame $self.title
    grid $self.title -gridy 0 -gridwidth 0

    # List Boxes
    scrollbar $self.vsbar -relief sunken -command "$self.list yview"

    listbox $self.list -relief sunken -width 30 -height 30 \
	-exportselection 0 \
	-xscrollcommand "$self.hsbar set" \
	-yscrollcommand "$self.vsbar set" \
	-selectmode browse

    scrollbar $self.hsbar -orient horizontal -relief sunken \
	-command "$self.list xview"

    # "bindtags" is needed here to make the <Button-1> binding work.
    # The selection must be changed before the path is adjusted,
    # otherwise the previous selection will be used instead of the
    # current selection.
    bindtags $self.list "Listbox $self.list . all"

    global $self-prevIndex
    set $self-prevIndex -1

    set select [ format {
	set index [lindex [%%W curselection] 0]
	if {$index != [set %s-prevIndex]} {
	    set %s-prevIndex $index
	    %s adjustButtons $index
	    %s adjustPath $index
	}
    } $self $self $self $self]

    set activate [ format {
	%s adjust [lindex [%%W curselection] 0]
    } $self ]

    bind $self.list <Button-1> $select
    bind $self.list <B1-Motion> $select
    bind $self.list <space> $select
    bind $self.list <Double-Button-1> $activate
    bind $self.list <Double-space> $activate

    #  Ignore extra clicks
    bind $self.list <Triple-Button-1> { }

    eval grid $self.vsbar -gridheight 1 -fill y $pads
    eval grid $self.list -gridwidth 0 -fill both -weightx 1 -weighty 1 $pads
    eval grid $self.hsbar -gridx 1 -gridwidth 0 -fill x $pads
	

    # Expand button
    frame $self.bttnFrame
    grid $self.bttnFrame -gridwidth 0 -fill x
    button $self.bttnFrame.expand -width [string length Collapse] \
	-textvariable $self-expandLabel \
	-command [ format {
	    set index [lindex [%s curselection] 0]
	    if {$index != ""} {
		%s adjust $index
	    }
	} $self.list $self ]
    global $self-expandLabel
    set $self-expandLabel Expand
    button $self.bttnFrame.rescan -text Rescan -command "$self rescan 0"
    eval grid $self.bttnFrame.expand $self.bttnFrame.rescan $bttn_pads
    $self _initbuttons $self.bttnFrame $bttn_pads
	
    # Pathname Label and Entry
    frame $self.pathFrame
    label $self.pathLabel -text Path
    entry $self.pathEntry -textvariable $self-path -relief sunken
    focus $self.pathEntry
    bind $self.pathEntry <Return> [ format {
	%s setpath [set %s] } $self $self-path ]

    grid $self.pathFrame -gridwidth 0 -fill x
    eval grid $self.pathLabel -gridy 0 -in $self.pathFrame \
	-padx 2 -pady 2
    eval grid $self.pathEntry -gridy 0 -in $self.pathFrame \
	-fill x -weightx 1 -padx 2 -pady 2
}

#
#  _initbuttons - subclasses can add extra buttons here
#
TreeBrowser instproc _initbuttons {parent pads} {
    # Put extra buttons here
}

TreeBrowser instproc _initroot {} {
    $self instvar tree root initClientData

    $self reset
    set tree(0) [ list {} 0 1 $initClientData $root 0 ]
    $self.list insert 0 $root
    $self.list selection set 0
    $self.list activate 0
    $self adjust 0
    $self set doBusy 1
}

TreeBrowser instproc reset {} {
    $self instvar tree expandList

    if { [array size tree] } {
	areplace tree 0 [expr [array size tree] - 1]
    }
    set expandList {}
    $self.list delete 0 end
}

TreeBrowser instproc path {} {
    $self instvar tree nameIndex sep

    set index [lindex [$self.list curselection] 0]
    set line $tree($index)
    set path $sep[join [lindex $line $nameIndex] $sep]
    return $path
}

TreeBrowser instproc setpath {path} {
    $self instvar tree sparse dirCodeIndex expandList expandedIndex sep
    $self instvar NoSuchEntry

    set contents [split [string trim $path $sep] $sep]

    set valid [$self _validPath $contents]
    if {$valid == 0} {
	error "$NoSuchEntry: $path"
    }

    set pathlist ""
    set expanding 0
    set tryRescan 1

    # Make sure the root directory is opened
    set contents [eval list {{}} $contents]

  $self busy {
    foreach i $contents {
	#  Don't append the empty list to the pathlist
	if {$i != ""} {
	    lappend pathlist $i
	}

	#  Check to see if this path is already expanded.  If it is,
	#  then keep looping.  If it isn't, then start expanding things
	#  from here on.
	if { [lsearch -exact $expandList $pathlist] == -1 } {
	    #  Expand pathlist
	    set index [$self findIndex $pathlist]
	    if { $index == -1 } {
		set depth [llength $pathlist]
		incr depth -2
		if { $depth >= 0 } {
		    set parentPath [lrange $pathlist 0 $depth]
		} else {
		    set parentPath ""
		}
		set thisIndex [$self findIndex $parentPath]

		if { $sparse } {
		    incr thisIndex
		    if { [lindex $tree($thisIndex) $expandedIndex] } {
			$self collapse $thisIndex
			set index [$self findIndex $pathlist]
		    }
		    incr thisIndex -1
		}
		if {$index == -1 && $tryRescan} {
		    $self rescan $thisIndex
		    set index [$self findIndex $pathlist]
		}
	    }

	    #  Give up
	    if {$index == -1 } {
		error "$NoSuchEntry: $path"
	    }

	    #  Expand the path
	    if {[lindex $tree($index) $dirCodeIndex] != 0} {
		$self expand $index
	    }

	    #  Since we just expanded the path, there is no point in
	    #  attempting a rescan on the next iteration
	    set tryRescan 0
	}
    }
  }

    set index [$self findIndex $pathlist]
    $self.list selection clear 0 end
    $self.list selection set $index
    $self.list activate $index
    $self adjustButtons $index
    $self adjustPath $index
    $self _makeSelectedVisible
}

# ------------------------------------------------------------------
#  METHOD:  findIndex - find the index in "array" corresponding to
#  the given path list (the pathlist is compared with the nameIndex
#  element of each list stored in the array).
# ------------------------------------------------------------------
TreeBrowser instproc findIndex {pathlist} {
    $self instvar tree nameIndex

    set size [array size tree]
    set index -1

    for {set i 0} {$i < $size} {incr i} {
	set line $tree($i)
	set namelist [lindex $line $nameIndex]

	if {$namelist == $pathlist} {
	    set index $i
	    break
	}
    }
    return $index
}

TreeBrowser instproc rescan {index} {
    set old_index [lindex [$self.list curselection] 0]
    set index [$self collapse $index]
    $self expand $index

    if {$old_index != "" && $old_index < [$self.list size]} {
	$self.list selection clear 0 end
	$self.list selection set $old_index
	$self.list activate $old_index
	$self adjustButtons $old_index
	$self adjustPath $old_index
    }
}

TreeBrowser instproc _makeSelectedVisible {} {
    set selectedItem [lindex [$self.list curselection] 0]
    if {$selectedItem == ""} {return}

    set topItem [$self.list nearest 0]
    set bottomItem [$self.list nearest 1000000]
    set viewHeight [expr $bottomItem - $topItem]

    if {$selectedItem < $topItem || \
	$selectedItem > ($topItem + $viewHeight)} {
	set yview [expr $selectedItem - ($viewHeight / 2)]
	if {$yview < 0} { set yview 0 }
	$self.list yview $yview
    }
}

TreeBrowser instproc deleteIndex {index} {
    $self instvar tree dirCodeIndex expandedIndex

    set line $tree($index)
    if { [ lindex $line $dirCodeIndex ] != 0 && \
	 [ lindex $line $expandedIndex ] } {
	$self collapse $index
    }
    $self.list delete $index $index

    set bogus(0) 0 ; unset bogus(0)
    areplace tree $index $index bogus

    if {[llength [$self.list curselection]] == 0} {
	set $self-expandLabel Expand
	$self.bttnFrame.expand configure -state disabled
    }
}

# ------------------------------------------------------------------
#  METHOD:  adjust - expand or collapse the given index
# ------------------------------------------------------------------
TreeBrowser instproc adjust {index} {
    $self instvar tree dirCodeIndex expandedIndex

    if { ! [array size tree] } { return }
    set line $tree($index)
    if {$line == "" } {
	error "Bad index passed to TreeBrowser::adjust: $index"
    }

    if { [ lindex $line $dirCodeIndex ] == 0 } {
	set state [lindex [$self.bttnFrame.expand configure -state] 4]
	if {$state != "disabled" } {
	    $self activate $index
	}
    } else {
	if { [lindex $line $expandedIndex] } {
	    $self collapse $index
	} else {
	    $self expand $index
	}
	$self _makeSelectedVisible
    }
}

# ------------------------------------------------------------------
#  METHOD:  adjustButtons - adjust the expand button label
# ------------------------------------------------------------------
TreeBrowser instproc adjustButtons {index} {
    $self instvar tree activate_label dirCodeIndex expandedIndex

    if { ! [array size tree] } { return }

    global $self-expandLabel
    global $self.bttnFrame.expand
    set line $tree($index)
    $self.bttnFrame.expand configure -state normal

    if {$line == "" } { return }
    if { [lindex $line $expandedIndex] } {
	set $self-expandLabel Collapse
    } else {
	if { ! [lindex $line $dirCodeIndex] } {
	    set $self-expandLabel $activate_label
	    if {[$self isActive $index] != 1} {
		$self.bttnFrame.expand configure -state disabled
	    }
	} else {
	    set $self-expandLabel Expand
	}
    }
}

# ------------------------------------------------------------------
#  METHOD:  adjustPath - adjust the path entry
# ------------------------------------------------------------------
TreeBrowser instproc adjustPath {index} {
    $self instvar tree nameIndex sep

    if { ! [array size tree] } { return }
    set line $tree($index)

    global $self-path
    set $self-path $sep[join [lindex $line $nameIndex] $sep]
}

# ------------------------------------------------------------------
#  METHOD:  activate - do something with the entry at the given index
# ------------------------------------------------------------------
TreeBrowser instproc activate {index} {
    $self instvar tree activate_callback nameIndex sep

    set line $tree($index)
    set path $sep[join [lindex $line $nameIndex] $sep]
    if {$activate_callback != ""} {
	uplevel #0 $activate_callback $path
    }
}

# ------------------------------------------------------------------
#  METHOD:  isActive - determine whether index is active
# ------------------------------------------------------------------
TreeBrowser instproc isActive {index} {
    $self instvar tree isActive_callback nameIndex sep

    set line $tree($index)
    set path $sep[join [lindex $line $nameIndex] $sep]
    if {$isActive_callback != ""} {
	uplevel #0 $isActive_callback $path
    }
}

# ------------------------------------------------------------------
#  METHOD:  expand - expand the given index
# ------------------------------------------------------------------
TreeBrowser instproc expand {index} {
    $self update
    $self instvar tree clientDataIndex dirCodeIndex displayIndex doBusy
    $self instvar expandList expandedIndex fileCodeIndex nameIndex sparse

    if { ! [array size tree] } { return }
    set line $tree($index)

    set dircode [lindex $line $dirCodeIndex]
    if {$dircode == 0} {
	if {[$self isActive $index] == 1} {
	    $self activate $index
	}
	return
    }

    set parentPath [lindex $line $nameIndex]
    set clientData [lindex $line $clientDataIndex]


    #
    #  This entire section handles the case where the user
    #  double-clicked for the first time on a symbolic link.
    #
    #  Note:  The symbolic link is an example of a situation
    #  where it is too expensive for the _descendants routine
    #  to decide if a thing is a container (directory) or an
    #  entry (file).  A dirCode with the value of "2" can be
    #  used to flag any entry where this is the case.
    #
    if { [lindex $line $dirCodeIndex] == 2 } {
	# Get the new file and directory codes
	$self busy {
	    set filecode [$self _fileCode $parentPath $clientData]
	    set dircode [$self _dirCode $parentPath $clientData]
	}

	# Get the new display name
	set dpyname \
	    [$self _displayName $parentPath $filecode $dircode $clientData]
		
	# Update the list
	$self.list delete $index
	$self.list insert $index $dpyname
	$self.list selection clear 0 end
	$self.list selection set $index
	$self.list activate $index

	# Mark the node with the new dircode, filecode and displayname
	set line [lreplace $line $fileCodeIndex $fileCodeIndex $filecode]
	set line [lreplace $line $dirCodeIndex $dirCodeIndex $dircode]
	set line [lreplace $line $displayIndex $displayIndex $dpyname]
	set tree($index) $line

	# Don't go any farther if the node isn't a container
	if { $dircode != 1 } {
	    $self adjustButtons $index
	    $self adjustPath $index
	    return
	}
    }	    

    #
    # If we are in sparse mode, move all the tree entries at the
    # current depth into the sparse array and delete them from the
    # tree.  But don't delete the container that we are currently
    # expanding.
    #
    set preOffset 0
    if {$sparse && $index != 0} {
	set depth [llength [lindex $line $nameIndex]]
	set length [array size tree]

	# Get the pre and post indicies
	for {set i [expr $index - 1]} {$i > 0} {incr i -1} {
	    set thisLine $tree($i)
	    set thisDepth [llength [lindex $thisLine $nameIndex]]
	    if {$thisDepth != $depth} {
		break
	    }
	}
	set preIndex [expr $i + 1]

	for {set i [expr $index + 1]} {$i < $length} {incr i} {
	    set thisLine $tree($i)
	    set thisDepth [llength [lindex $thisLine $nameIndex]]
	    if {$thisDepth != $depth} {
		break
	    }
	}
	set postIndex [expr $i - 1]

	# Delete all files from the current depth except the
	# file that we are expanding
	set lineArray(0) $line
	areplace tree $preIndex $postIndex lineArray

	# Keep the display in sync with the tree list
	$self.list delete $preIndex $postIndex
	$self.list insert $preIndex [lindex $line $displayIndex]
	$self.list selection clear 0 end
	$self.list selection set $preIndex
	$self.list activate $preIndex

	# Set the offset here so that the expandables code at the
	# end of this function will work properly
	set preOffset [expr $preIndex - $index]
	set index $preIndex
    }

    #
    #  Get the descendants of the container and the format this list
    #  so that it can be inserted into the tree list and the listbox.
    #
    #  Note:  This needs to be done after the "sparse" code so that the
    #  expandables indexes are set correctly.
    #
    set contents(0) 0 ; unset contents(0)
    $self busy {$self _descendants $parentPath $clientData contents}

    # Initialize "treeInsert" to a zero length array
    set treeInsert(0) 0 ; unset treeInsert(0)

    # Initialize "listInsert" and "expandables" to zero length lists
    set listInsert ""
    set expandables ""

    set j [expr $index+1]
    set k 0
    set size [array size contents]
    for {set i 0} {$i < $size} {incr i} {
	set child $contents($i)

	# Store the full path name
	set fullPath [lindex $child $nameIndex]
	if {$parentPath != ""} {
	    set fullPath [concat $parentPath $fullPath]
	}
	set child [lreplace $child $nameIndex $nameIndex $fullPath]

	# Mark as not exanded
	lappend child 0

	set treeInsert($k) $child
	lappend listInsert [lindex $child $displayIndex]

	if { [lsearch -exact $expandList $fullPath ] != -1 } {
	    lappend expandables $j
	}
	incr j
	incr k
    }

    #
    #  Mark the node as expanded
    #
    set line [lreplace $line $expandedIndex $expandedIndex 1]
    set tree($index) $line

    #
    #  Adjust the current node by updating the buttons and the
    #  current path.  This needs to happen after the node has
    #  been marked as expanded.
    #
    $self adjustButtons $index
    $self adjustPath $index

    #  Insert the children into the tree and into the scrolling list
    incr index
    ainsert tree $index treeInsert
    eval $self.list insert $index $listInsert

    #  Add the current path to the expanded list
    if { [lsearch -exact $expandList $parentPath] == -1 } {
	lappend expandList $parentPath
    }

    #  Expand all additional expansion indexes
    set offset 0
    $self busy {
	foreach i $expandables {
	    incr offset [ $self expand [expr $i+$offset] ]
	}
    }

    if { $sparse } {
	set length 1
    } else {
	set length [array size contents]
    }
    return [expr $offset + $preOffset + $length]
}


# ------------------------------------------------------------------
#  METHOD:  collapse - collapse the given index
# ------------------------------------------------------------------
TreeBrowser instproc collapse {index} {
    $self instvar tree expandList expandedIndex nameIndex sparse

    if { ! [array size tree] } { return -1 }

    set line $tree($index)
    set length [array size tree]

    set fullpath [lindex $line $nameIndex]
    set depth [llength [lindex $line $nameIndex]]

    #  Figure out which elements need to be deleted.
    $self busy {
	for {set i [expr $index+1]} {$i < $length} {incr i} {
	    set thisLine $tree($i)
	    set thisDepth [llength [lindex $thisLine $nameIndex]]
	    if {$thisDepth <= $depth} {
		break
	    }
	}
	incr i -1
    }

    # Mark this node as not expanded
    set line [lreplace $line $expandedIndex $expandedIndex 0]
    set tree($index) $line

    #  Delete the elements that need deleting
    set incrIndex [expr $index+1]
    if {$incrIndex <= $i} {
	areplace tree $incrIndex $i
	$self.list delete $incrIndex $i
    }

    # Delete the current path from the expanded list
    set delIndex [lsearch -exact $expandList $fullpath]
    if {$delIndex != -1} {
	set expandList [lreplace $expandList $delIndex $delIndex]
    }

    #
    #  If we are in sparse mode, re-expand the current level.
    #
    if {$sparse && $index != 0} {
	$self.list delete $index $index
	incr index -1
	$self expand $index

	set index [$self findIndex $fullpath]
	if { $index == -1 } { return -1 }
	$self.list selection clear 0 end
	$self.list selection set $index
	$self.list activate $index
    }

    #
    #  Adjust the current node by updating the buttons and the
    #  current path.  This needs to happen after the node has
    #  been marked as collapsed.
    #
    $self adjustButtons $index
    $self adjustPath $index

    return $index
}

# ------------------------------------------------------------------
#  These are some default methods for sub-classes
# ------------------------------------------------------------------
TreeBrowser instproc _descendants {pathlist clientData array} {
    return -1
}

TreeBrowser instproc _fileCode {pathlist clientData} {
    return 0
}

TreeBrowser instproc _displayName {name filecode dircode clientData} {
    set padding ""
    set length [llength $pathlist]
    incr length -1
    for {set i 0} {$i < $length} {incr i} {
    append padding "    "
    }
    set name $padding[lindex $pathlist $length]

    switch $dircode {
	0 { return $name }
	1 { return $name/ }
    }
}

TreeBrowser instproc _validPath {pathlist} {
    return 1
}


########################################################################
#
#  TreeBrowserPopup
#
#  The TreeBrowserPopup is identical to the TreeBrowser except that it
#  creates the browser as a toplevel window rather than as a frame window.
#
########################################################################

proc TreeBrowserPopup {} {}
rename TreeBrowserPopup {}
Class TreeBrowserPopup -superclass TreeBrowser

TreeBrowserPopup instproc init {} {

    #  Create a window with the same name as this object
    rename $self $self-tmp-
    toplevel $self -class $class
    catch {rename $self-win- ""}
    rename $self $self-win-
    rename $self-tmp- $self

    wm minsize $self 64 64

    # Initialize all the widgets
    $self _initvars
    $self _initwidgets
}


########################################################################
#
#  TreeFileBrowser
#
#  The TreeFileBrowser is used to browse the file system
#
########################################################################

proc TreeFileBrowser {} {}
rename TreeFileBrowser {}
Class TreeFileBrowser -superclass TreeBrowser

TreeFileBrowser instproc _initvars {} {
    $self next

    $self set cancel_callback {}
    $self set root /
    $self set sep /
    $self set sparse 1
    $self set title "File Browser"

    $self set NoSuchEntry {No such file or directory}
}

TreeFileBrowser instproc _initbuttons {parent pads} {
    button $parent.cancel -text Cancel -command "$self cancel"
    eval grid $parent.cancel -gw 0 $pads -fill x
}

TreeFileBrowser instproc cancel {} {
    $self instvar cancel_callback
    uplevel #0 $cancel_callback
}

# ------------------------------------------------------------------
# Override setpath to do tilde expansion
# ------------------------------------------------------------------
TreeFileBrowser instproc setpath {path} {
    $self instvar sep

    set path [string trim $path " "]
    if {[string index $path 0] == "~"} {
	set tildeList [split $path /]
	set path \
	"[glob [lindex $path 0]]${sep}[join [lrange $tildeList 1 end] $sep]"
    }
    $self next $path
}

# ------------------------------------------------------------------
#  METHOD:  _descendants
#  Returns a list of the form
#  { {name filecode dircode displayname} ... }
# ------------------------------------------------------------------
TreeFileBrowser instproc _descendants {pathlist clientData array} {
    upvar $array contents
    set pathname /[join $pathlist /]
    if { $pathname == "/" } { set pathname {} }
    set list [lsort [glob -nocomplain $pathname/*]]

    # Padding
    set padding ""
    set length [llength $pathlist]
    for {set i 0} {$i < $length} {incr i} {
    append padding "    "
    }

    set j 0
    foreach i $list {
	set name [file tail $i]

	if {$name != "." && $name != ".."} {
	    set filecode 0
	    set dircode 0
	    set type [file type $i]

	    if {$type == "link"} {
		set dircode 2
	    } else {
		set dircode [file isdirectory $i]
		if {! $dircode} {
		    if { [file executable $i] } {
			set filecode 1
		    }
		}
	    }
	    set dpyname $padding[$self CodeToName $name $filecode $dircode]
	    set contents($j) [list $name $filecode $dircode -1 $dpyname]
	}
	incr j
    }
}

TreeFileBrowser instproc _fileCode {pathlist clientData} {
    set pathname /[join $pathlist /]
    if { [file executable $pathname] } {
	return 1
    } else {
	return 0
    }
}

TreeFileBrowser instproc _dirCode {pathlist clientData} {
    set pathname /[join $pathlist /]
    return [file isdirectory $pathname]
}

TreeFileBrowser instproc _displayName {pathlist filecode dircode clientData} {
    # Padding
    set padding ""
    set length [llength $pathlist]
    incr length -1
    for {set i 0} {$i < $length} {incr i} {
	append padding "    "
    }

    set name [lindex $pathlist $length]
    return $padding[$self CodeToName $name $filecode $dircode ]
}

TreeFileBrowser instproc _validPath {pathlist} {
    set pathname /[join $pathlist /]
    return [file exists $pathname]
}

TreeFileBrowser instproc CodeToName {name filecode dircode} {
    switch $dircode {
	0 {
	    switch $filecode {
		0 { return $name }
		1 { return $name* }
		2 { return "$name@ #broken link#" }
	    }
	}
	1 { return $name/ }
	2 { return $name@ }
    }
}


########################################################################
#
#  TreeTkBrowser
#
#  The TreeTkBrowser is used to browse the hierarchy of tk widgets
#
########################################################################

proc TreeTkBrowser {} {}
rename TreeTkBrowser {}
Class TreeTkBrowser -superclass TreeBrowser
    
TreeTkBrowser instproc _initvars {} {
    $self next
    $self set title "Tk Window Browser"
    $self set sep .
    $self set root /
}
    
# ------------------------------------------------------------------
#  METHOD:  _descendants
#  Returns a list of the form
#  { {name filecode dircode displayname} ... }
# ------------------------------------------------------------------
TreeTkBrowser instproc _descendants {pathlist clientData array} {
    upvar $array contents

    # Padding
    set padding ""
    set length [llength $pathlist]
    for {set i 0} {$i < $length} {incr i} {
	append padding "    "
    }

    set pathname .[join $pathlist .]
    set list [lsort [winfo children $pathname]]
    set filelist ""

    set j 0
    foreach i $list {
	regexp {\.([^\.]*)$} $i dummy name
	set filecode 0
	set dircode 0
	if { [winfo children $i] != "" } {
	    set dircode 1
	}
	if {$dircode} {
	    set dpyname $padding$name/
	} else {
	    set dpyname $padding$name
	}

	set contents($j) [list $name $filecode $dircode -1 $dpyname]
	incr j
    }
    return $filelist
}

TreeTkBrowser instproc _dirCode {pathlist clientData} {
    set pathname .[join $pathlist .]
    if { [winfo children $pathname] != "" } {
	return 1
    } else {
	return 0
    }
}
