/*----------------------------------------------------------------------------*
 | This file is part of DEU (Doom Editing Utilities), created by the DEU team:|
 | Raphael Quinet, Brendon Wyber, Ted Vessenes and others.  See README.1ST or |
 | the "about" dialog box for full credits.                                   |
 |                                                                            |
 | DEU is an open project: if you think that you can contribute, please join  |
 | the DEU team.  You will be credited for any code (or ideas) included in    |
 | the next version of the program.                                           |
 |                                                                            |
 | If you want to make any modifications and re-distribute them on your own,  |
 | you must follow the conditions of the DEU license.  Read the file LICENSE  |
 | in this directory or README.1ST in the top directory.  If do not have a    |
 | copy of these files, you can request them from any member of the DEU team, |
 | or by mail: Raphael Quinet, Rue des Martyrs 9, B-4550 Nandrin (Belgium).   |
 |                                                                            |
 | This program comes with absolutely no warranty.  Use it at your own risks! |
 *----------------------------------------------------------------------------*

 W_THINGS.C - Things types and properties

*/

/* the includes */
#include "deu.h"
#include "d_main.h"
#include "d_config.h"
/*! the following #include should disappear if GetThingColour is moved in M_???.C */
#include "g_gfx.h" /* color names */
#include "w_things.h"

/* Description of the Things properties. */
typedef struct
{
  Int16  radius;  /* Radius of the Thing for collision detection */
  Int16  height;  /* Height of the Thing: minimum ceiling-floor difference */
  Int16  mass;    /* Mass of the Thing (inertia when the object is hit) */
  Int16  health;  /* Number of health points, 0 = not alive */
  Int16  speed;   /* Speed of the Thing (if alive) */
  Int16  damage;  /* Number of damage points, 0 = no damage */ /*! Explain */
  char  *sprite;  /* Sprite to display - (4 chars for name + 2/4 for frame) */
  char  *name;    /* Name of the Thing (18 chars max) */
  UInt16 props;   /* Properties (flags), see TP_* defines */
} ThingDesc;


/* Doom Things. */
static ThingDesc DThingInfo0[MAXDTHINGINFO0 + 1 - MINDTHINGINFO0] =
{
  {  16,  56,  100,  100,  0,  500, "PLAYA1", /* 1 */
    "Player 1 Start", TP_PLAYER},
  {  16,  56,  100,  100,  0,  500, "PLAYA1", /* 2 */
    "Player 2 Start", TP_PLAYER},
  {  16,  56,  100,  100,  0,  500, "PLAYA1", /* 3 */
    "Player 3 Start", TP_PLAYER},
  {  16,  56,  100,  100,  0,  500, "PLAYA1", /* 4 */
    "Player 4 Start", TP_PLAYER},
  {  20,  16,    0,    0,  0,    0, "BKEYB0", /* 5 */
    "Blue KeyCard", TP_PICK},
  {  20,  16,    0,    0,  0,    0, "YKEYB0", /* 6 */
    "Yellow KeyCard", TP_PICK},
  { 128, 100, 1000, 3000, 12,  350, "SPIDH1", /* 7 */
    "Spider Boss*", TP_BLOCK | TP_KILL},
  {  20,  16,    0,    0,  0, 1080, "BPAKA0", /* 8 */
    "Backpack", TP_PICK | TP_WAMMO | TP_ITEM},
  {  20,  56,  100,   30,  8,   50, "SPOSF1", /* 9 */
    "Shotgun Sargeant*", TP_BLOCK | TP_KILL},
  {  16,  16,    0,    0,  0,    0, "PLAYW0", /* 10 */
    "Guts and Bones", TP_CORPSE},
  {  16,  16,    0,    0,  0,  500, "PLAYF1", /* 11 */
    "Deathmatch Start", TP_PLAYER},
  {  16,  16,    0,    0,  0,    0, "PLAYW0", /* 12 */
    "Guts and Bones 2", TP_CORPSE},
  {  20,  16,    0,    0,  0,    0, "RKEYB0", /* 13 */
    "Red KeyCard", TP_PICK},
  {  16,  16,    0,    0,  0,    0, "TFOGB0", /* 14 */
    "Teleport exit", 0},
  {  16,  16,    0,    0,  0,    0, "PLAYN0", /* 15 */
    "Dead Green Player", TP_CORPSE},
  {  40, 110, 1000, 4000, 16, 1500, "CYBRF1", /* 16 */
    "Cyber Demon*", TP_BLOCK | TP_KILL},
  {  20,  16,    0,    0,  0, 2000, "CELPA0", /* 17 */
    "Energy Pack", TP_PICK | TP_WAMMO},
  {  16,  16,    0,    0,  0,    0, "POSSL0", /* 18 */
    "Dead Trooper", TP_CORPSE},
  {  16,  16,    0,    0,  0,    0, "SPOSL0", /* 19 */
    "Dead Sargeant", TP_CORPSE},
  {  16,  16,    0,    0,  0,    0, "TROOM0", /* 20 */
    "Dead Imp", TP_CORPSE},
  {  16,  16,    0,    0,  0,    0, "SARGN0", /* 21 */
    "Dead Demon", TP_CORPSE},
  {  16,  16,    0,    0,  0,    0, "HEADL0", /* 22 */
    "Dead Cacodemon", TP_CORPSE},
  {  16,  16,    0,    0,  0,    0, "------", /* 23 */
    "Dead Lost Soul", TP_CORPSE},
  {  16,  16,    0,    0,  0,    0, "POL5A0", /* 24 */
    "Pool of Blood", TP_CORPSE},
  {  16,  16,    0,    0,  0,    0, "POL1A0", /* 25 */
    "Impaled body*", TP_CORPSE | TP_BLOCK},
  {  16,  16,    0,    0,  0,    0, "POL6B0", /* 26 */
    "Twitching Impaled*", TP_CORPSE | TP_BLOCK},
  {  16,  16,    0,    0,  0,    0, "POL4A0", /* 27 */
    "Pole with Skull*", TP_CORPSE | TP_BLOCK},
  {  16,  16,    0,    0,  0,    0, "POL2A0", /* 28 */
    "Skewer with Heads*", TP_CORPSE | TP_BLOCK},
  {  16,  16,    0,    0,  0,    0, "POL3A0", /* 29 */
    "Pile of Skulls*", TP_CORPSE | TP_BLOCK},
  {  16,  16,    0,    0,  0,    0, "COL1A0", /* 30 */
    "Tall Green Pillar*", TP_BLOCK},
  {  16,  16,    0,    0,  0,    0, "COL2A0", /* 31 */
    "Short Green Pillar*", TP_BLOCK},
  {  16,  16,    0,    0,  0,    0, "COL3A0", /* 32 */
    "Tall Red Pillar*", TP_BLOCK},
  {  16,  16,    0,    0,  0,    0, "COL4A0", /* 33 */
    "Short Red Pillar*", TP_BLOCK},
  {  16,  16,    0,    0,  0,    0, "CANDA0", /* 34 */
    "Candle", TP_LIGHT},
  {  16,  16,    0,    0,  0,    0, "CBRAA0", /* 35 */
    "Candelabra*", TP_LIGHT | TP_BLOCK},
  {  16,  16,    0,    0,  0,    0, "COL5B0", /* 36 */
    "Pillar w/Heart*", TP_BLOCK},
  {  16,  16,    0,    0,  0,    0, "COL6A0", /* 37 */
    "Red Pillar w/Skul*", TP_BLOCK},
  {  20,  16,    0,    0,  0,    0, "RSKUB0", /* 38 */
    "Red Skull Key", TP_PICK},
  {  20,  16,    0,    0,  0,    0, "YSKUB0", /* 39 */
    "Yellow Skull Key", TP_PICK},
  {  20,  16,    0,    0,  0,    0, "BSKUB0", /* 40 */
    "Blue Skull Key", TP_PICK},
  {  16,  16,    0,    0,  0,    0, "CEYEC0", /* 41 */
    "Eye in Symbol*", TP_BLOCK},
  {  16,  16,    0,    0,  0,    0, "FSKUB0", /* 42 */
    "Skulls in Flames*", TP_CORPSE | TP_BLOCK},
  {  16,  16,    0,    0,  0,    0, "TRE1A0", /* 43 */
    "Grey Tree*", TP_BLOCK},
  {  16,  16,    0,    0,  0,    0, "TBLUA0", /* 44 */
    "Tall Blue Torch*", TP_LIGHT | TP_BLOCK},
  {  16,  16,    0,    0,  0,    0, "TGRNA0", /* 45 */
    "Tall Green Torch*", TP_LIGHT | TP_BLOCK},
  {  16,  16,    0,    0,  0,    0, "TREDA0", /* 46 */
    "Tall Red Torch*", TP_LIGHT | TP_BLOCK},
  {  16,  16,    0,    0,  0,    0, "SMITA0", /* 47 */
    "Stalagmite / Stub*", TP_BLOCK},
  {  16,  16,    0,    0,  0,    0, "ELECA0", /* 48 */
    "Technical Column*", TP_BLOCK},
  {  16,  16,    0,    0,  0,    0, "GOR1A0", /* 49 */
    "Hanging Body*", TP_CORPSE | TP_BLOCK | TP_FLOAT},
  {  16,  16,    0,    0,  0,    0, "GOR2A0", /* 50 */
    "Hanging Arms Out*", TP_CORPSE | TP_BLOCK | TP_FLOAT},
  {  16,  16,    0,    0,  0,    0, "GOR3A0", /* 51 */
    "One-legged Body*", TP_CORPSE | TP_BLOCK | TP_FLOAT},
  {  16,  16,    0,    0,  0,    0, "GOR4A0", /* 52 */
    "Hanging Torso*", TP_CORPSE | TP_BLOCK | TP_FLOAT},
  {  16,  16,    0,    0,  0,    0, "GOR5A0", /* 53 */
    "Hanging Leg*", TP_CORPSE | TP_BLOCK | TP_FLOAT},
  {  32,  16,    0,    0,  0,    0, "TRE2A0", /* 54 */
    "Tall Brown Tree*", TP_BLOCK},
  {  16,  16,    0,    0,  0,    0, "SMBTA0", /* 55 */
    "Short Blue Torch*", TP_LIGHT | TP_BLOCK},
  {  16,  16,    0,    0,  0,    0, "SMGTA0", /* 56 */
    "Short Green Torch*", TP_LIGHT | TP_BLOCK},
  {  16,  16,    0,    0,  0,    0, "SMRTA0", /* 57 */
    "Short Red Torch*", TP_LIGHT | TP_BLOCK},
  {  30,  56,  400,  150, 10,    0, "SARGB5", /* 58 */
    "Specter*", TP_BLOCK | TP_KILL | TP_INVIS},
  {  16,  16,    0,    0,  0,    0, "GOR2A0", /* 59 */
    "Hanging Arms Out", TP_CORPSE | TP_FLOAT},
  {  16,  16,    0,    0,  0,    0, "GOR4A0", /* 60 */
    "Hanging Torso", TP_CORPSE | TP_FLOAT},
  {  16,  16,    0,    0,  0,    0, "GOR3A0", /* 61 */
    "One-legged Body", TP_CORPSE | TP_FLOAT},
  {  16,  16,    0,    0,  0,    0, "GOR5A0", /* 62 */
    "Hanging Leg", TP_CORPSE | TP_FLOAT},
  {  16,  16,    0,    0,  0,    0, "GOR1A0", /* 63 */
    "Hanging Body", TP_CORPSE | TP_FLOAT},
  {  20,  56,  500,  700, 15,  666, "VILEN1", /* 64 */
    "Arch-vile*", TP_BLOCK | TP_KILL},
  {  20,  56,  100,   70,  8,  200, "CPOSF1", /* 65 */
    "Chaingun Sargeant*", TP_BLOCK | TP_KILL},
  {  20,  56,  500,  300, 10,  375, "SKELJ1", /* 66 */
    "Revenant*", TP_BLOCK | TP_KILL},
  {  48,  64, 1000,  600,  8,  260, "FATTH1", /* 67 */
    "Mancubus*", TP_BLOCK | TP_KILL},
  {  64,  64,  600,  500, 12,  600, "BSPIG1", /* 68 */
    "Arachnotron*", TP_BLOCK | TP_KILL},
  {  24,  64, 1000,  500,  8,  333, "BOS2E1", /* 69 */
    "Hell Knight*", TP_BLOCK | TP_KILL},
  {  16,  16,    0,    0,  0,    0, "FCANA0", /* 70 */
    "Burning Barrel*", TP_LIGHT | TP_BLOCK},
  {  31,  56,  400,  400,  8,  250, "PAINF1", /* 71 */
    "Pain Elemental*", TP_BLOCK | TP_FLOAT | TP_KILL},
  {  16,  72, 6666,  100,  0,    0, "KEENA0", /* 72 */
    "Hanging Keen*", TP_BLOCK | TP_KILL},
  {  16,  16,    0,    0,  0,    0, "HDB1A0", /* 73 */
    "Hngng Dead Body 1*", TP_CORPSE | TP_BLOCK | TP_FLOAT},
  {  16,  16,    0,    0,  0,    0, "HDB2A0", /* 74 */
    "Hngng Dead Body 2*", TP_CORPSE | TP_BLOCK | TP_FLOAT},
  {  16,  16,    0,    0,  0,    0, "HDB3A0", /* 75 */
    "Hngng Arms&Head 1*", TP_CORPSE | TP_BLOCK | TP_FLOAT},
  {  16,  16,    0,    0,  0,    0, "HDB4A0", /* 76 */
    "Hngng Arms&Head 2*", TP_CORPSE | TP_BLOCK | TP_FLOAT},
  {  16,  16,    0,    0,  0,    0, "HDB5A0", /* 77 */
    "Hngng Arms&Head 3*", TP_CORPSE | TP_BLOCK | TP_FLOAT},
  {  16,  16,    0,    0,  0,    0, "HDB6A0", /* 78 */
    "Hngng Arms&Head 4*", TP_CORPSE | TP_BLOCK | TP_FLOAT},
  {  16,  16,    0,    0,  0,    0, "??????", /* 79 */ /*! missing... */
    "<! mess 79>", TP_CORPSE},
  {  16,  16,    0,    0,  0,    0, "??????", /* 80 */ /*! missing... */
    "<! mess 80>", TP_CORPSE},
  {  16,  16,    0,    0,  0,    0, "??????", /* 81 */ /*! missing... */
    "<! mess 81>", TP_CORPSE},
  {  20,  16,    0,    0,  0,  760, "SGN2A0", /* 82 */
    "Super Shotgun", TP_PICK | TP_WEAPON},
  {  20,  16,    0,    0,  0,  400, "MEGAA0", /* 83 */
    "Mega Sphere", TP_PICK | TP_ITEM},
  {  20,  56,  100,   50,  8,   45, "SSWVG0", /* 84 */
    "Wolfenstein SS*", TP_BLOCK | TP_KILL},
  {  16,  16,    0,    0,  0,    0, "TLMPA0", /* 85 */
    "Tech. lamp*", TP_LIGHT | TP_BLOCK},
  {  16,  16,    0,    0,  0,    0, "TLP2A0", /* 86 */
    "Tech. lamp 2*", TP_LIGHT | TP_BLOCK},
  {  16,  16,    0,    0,  0,    0, "FIRED0", /* 87 */
    "Spawn Spot", TP_KILL},
  {  16,  16, 6666,  250,  0,    0, "BBRNA0", /* 88 */
    "Boss Brain*", TP_BLOCK | TP_KILL},
  {  16,  16,    0,    0,  0,    0, "BOSFA0", /* 89 */
    "Boss Shooter", 0}
};

static ThingDesc DThingInfo2[MAXDTHINGINFO2 + 1 - MINDTHINGINFO2] =
{
  {  20,  16,    0,    0,  0,  560, "SHOTA0", /* 2001 */
    "Shotgun", TP_PICK | TP_WEAPON},
  {  20,  16,    0,    0,  0,  200, "MGUNA0", /* 2002 */
    "Chaingun", TP_PICK | TP_WEAPON},
  {  20,  16,    0,    0,  0,  600, "LAUNA0", /* 2003 */
    "Rocket Launcher", TP_PICK | TP_WEAPON},
  {  20,  16,    0,    0,  0,  800, "PLASA0", /* 2004 */
    "Plasma Gun", TP_PICK | TP_WEAPON},
  {  20,  16,    0,    0,  0,  150, "CSAWA0", /* 2005 */
    "Chainsaw", TP_PICK | TP_WEAPON},
  {  20,  16,    0,    0,  0, 1000, "BFUGA0", /* 2006 */
    "BFG9000", TP_PICK | TP_WEAPON},
  {  20,  16,    0,    0,  0,  100, "CLIPA0", /* 2007 */
    "Ammo Clip", TP_PICK | TP_WAMMO},
  {  20,  16,    0,    0,  0,  280, "SHELA0", /* 2008 */
    "Shells", TP_PICK | TP_WAMMO},
  { 255,   0,    0,    0,  0,    0, "------", /* 2009 */
    "<UNKNOWN 2009>", TP_BAD},
  {  20,  16,    0,    0,  0,  300, "ROCKA0", /* 2010 */
    "Rocket", TP_PICK | TP_WAMMO},
  {  20,  16,    0,    0,  0,   10, "STIMA0", /* 2011 */
    "Stim Pack", TP_PICK},
  {  20,  16,    0,    0,  0,   25, "MEDIA0", /* 2012 */
    "Medical Kit", TP_PICK},
  {  20,  16,    0,    0,  0,  100, "SOULA0", /* 2013 */
    "Soul Sphere", TP_PICK | TP_ITEM},
  {  20,  16,    0,    0,  0,    1, "BON1D0", /* 2014 */
    "Health Potion", TP_PICK | TP_ITEM},
  {  20,  16,    0,    0,  0,    1, "BON2D0", /* 2015 */
    "Armour Helmet", TP_PICK | TP_ITEM},
  { 255,   0,    0,    0,  0,    0, "------", /* 2016 */
    "<UNKNOWN 2016>", TP_BAD},
  { 255,   0,    0,    0,  0,    0, "------", /* 2017 */
    "<UNKNOWN 2017>", TP_BAD},
  {  20,  16,    0,    0,  0,  100, "ARM1B0", /* 2018 */
    "Green Armour", TP_PICK},
  {  20,  16,    0,    0,  0,  200, "ARM2B0", /* 2019 */
    "Blue Armour", TP_PICK},
  { 255,   0,    0,    0,  0,    0, "------", /* 2020 */
    "<UNKNOWN 2020>", TP_BAD},
  { 255,   0,    0,    0,  0,    0, "------", /* 2021 */
    "<UNKNOWN 2021>", TP_BAD},
  {  20,  16,    0,    0,  0, 2500, "PINVA0", /* 2022 */
    "Invulnerability", TP_PICK | TP_ITEM},
  {  20,  16,    0,    0,  0,  150, "PSTRA0", /* 2023 */
    "Beserk", TP_PICK | TP_ITEM},
  {  20,  16,    0,    0,  0,  500, "PINSA0", /* 2024 */
    "Invisibility", TP_PICK | TP_ITEM},
  {  20,  16,    0,    0,  0,   25, "SUITA0", /* 2025 */
    "Radiation Suit", TP_PICK},
  {  20,  16,    0,    0,  0,    0, "PMAPB0", /* 2026 */
    "Computer Map", TP_PICK | TP_ITEM},
  { 255,   0,    0,    0,  0,    0, "------", /* 2027 */
    "<UNKNOWN 2027>", TP_BAD},
  {  16,  16,    0,    0,  0,    0, "COLUA0", /* 2028 */
    "Lamp*", TP_LIGHT | TP_BLOCK},
  { 255,   0,    0,    0,  0,    0, "------", /* 2029 */
    "<UNKNOWN 2029>", TP_BAD},
  { 255,   0,    0,    0,  0,    0, "------", /* 2030 */
    "<UNKNOWN 2030>", TP_BAD},
  { 255,   0,    0,    0,  0,    0, "------", /* 2031 */
    "<UNKNOWN 2031>", TP_BAD},
  { 255,   0,    0,    0,  0,    0, "------", /* 2032 */
    "<UNKNOWN 2032>", TP_BAD},
  { 255,   0,    0,    0,  0,    0, "------", /* 2033 */
    "<UNKNOWN 2033>", TP_BAD},
  { 255,   0,    0,    0,  0,    0, "------", /* 2034 */
    "<UNKNOWN 2035>", TP_BAD},
  {  10,  42,  100,   20,  0,    0, "BAR1B0", /* 2035 */
    "Barrel*", TP_BLOCK | TP_KILL},
  { 255,   0,    0,    0,  0,    0, "------", /* 2036 */
    "<UNKNOWN 2036>", TP_BAD},
  { 255,   0,    0,    0,  0,    0, "------", /* 2037 */
    "<UNKNOWN 2037>", TP_BAD},
  { 255,   0,    0,    0,  0,    0, "------", /* 2038 */
    "<UNKNOWN 2038>", TP_BAD},
  { 255,   0,    0,    0,  0,    0, "------", /* 2039 */
    "<UNKNOWN 2039>", TP_BAD},
  { 255,   0,    0,    0,  0,    0, "------", /* 2040 */
    "<UNKNOWN 2040>", TP_BAD},
  { 255,   0,    0,    0,  0,    0, "------", /* 2041 */
    "<UNKNOWN 2041>", TP_BAD},
  { 255,   0,    0,    0,  0,    0, "------", /* 2042 */
    "<UNKNOWN 2042>", TP_BAD},
  { 255,   0,    0,    0,  0,    0, "------", /* 2043 */
    "<UNKNOWN 2043>", TP_BAD},
  { 255,   0,    0,    0,  0,    0, "------", /* 2044 */
    "<UNKNOWN 2044>", TP_BAD},
  {  20,  16,    0,    0,  0,   25, "PVISA0", /* 2045 */
    "Lite Amp. Goggles", TP_PICK | TP_ITEM},
  {  20,  16,    0,    0,  0, 1500, "BROKA0", /* 2046 */
    "Box of Rockets", TP_PICK | TP_WAMMO},
  {  20,  16,    0,    0,  0,  400, "CELLA0", /* 2047 */
    "Energy Cell", TP_PICK | TP_WAMMO},
  {  20,  16,    0,    0,  0,  500, "AMMOA0", /* 2048 */
    "Box of Ammo", TP_PICK | TP_WAMMO},
  {  20,  16,    0,    0,  0, 1400, "SBOXA0", /* 2049 */
    "Box of Shells", TP_PICK | TP_WAMMO}
};

static ThingDesc DThingInfo3[MAXDTHINGINFO3 + 1 - MINDTHINGINFO3] =
{
  {  20,  56,  100,   60,  8,  180, "TROOE1", /* 3001 */
    "Imp*", TP_BLOCK | TP_KILL},
  {  30,  56,  400,  150, 10,  225, "SARGF1", /* 3002 */
    "Demon*", TP_BLOCK | TP_KILL},
  {  24,  64, 1000, 1000,  8,  333, "BOSSE1", /* 3003 */
    "Baron of Hell*", TP_BLOCK | TP_KILL},
  {  20,  56,  100,   60,  8,   50, "POSSF1", /* 3004 */
    "Trooper*", TP_BLOCK | TP_KILL},
  {  31,  56,  400,  400,  8,  210, "HEADD1", /* 3005 */
    "Cacodemon*", TP_BLOCK | TP_FLOAT | TP_KILL},
  {  16,  56,   50,  100,  8,   85, "SKULD1", /* 3006 */
    "Lost Soul*", TP_BLOCK | TP_FLOAT | TP_KILL},
};

/* Heretic Things */
static ThingDesc HThingInfo0[MAXHTHINGINFO0 + 1 - MINHTHINGINFO0] =
{
  {  16,  16,    0,    0,  0, 0000, "PLAYA1", /* 1 */ /*! Use Composite bitmap */
    "Player 1 Start", TP_PLAYER},
  {  16,  16,    0,    0,  0, 0000, "PLAYA1", /* 2 */ /*! Use Composite bitmap */
    "Player 2 Start", TP_PLAYER},
  {  16,  16,    0,    0,  0, 0000, "PLAYA1", /* 3 */ /*! Use Composite bitmap */
    "Player 3 Start", TP_PLAYER},
  {  16,  16,    0,    0,  0, 0000, "PLAYA1", /* 4 */ /*! Use Composite bitmap */
    "Player 4 Start", TP_PLAYER},
  {  16,  16,    0,    0,  0, 0000, "IMPXD1", /* 5 */
    "Flying Gargoyle Leader", TP_BLOCK | TP_KILL | TP_FLOAT},
  {  40,  16,    0,    0,  0, 0000, "HEADB1", /* 6 */
    "Ironlich", TP_BLOCK | TP_KILL},
  {  60,  16,    0,    0,  0, 0000, "SRCRS1", /* 7 */ /*! Use Composite bitmap */
    "D'sparil", TP_BLOCK | TP_KILL},
  {  20,  16,    0,    0,  0, 0000, "BAGHA0", /* 8 */
    "Bag of Holding", TP_PICK | TP_WAMMO | TP_ITEM},
  {  40,  16,    0,    0,  0, 0000, "MNTRV1", /* 9 */
    "Maulotaur", TP_BLOCK | TP_KILL},
  {  20,  16,    0,    0,  0, 0000, "AMG1A0", /* 10 */
    "Wand Crystal", TP_PICK | TP_WAMMO},
  {  16,  16,    0,    0,  0, 0000, "PLAYF1", /* 11 */ /*! Use Composite bitmap */
    "Deathmatch Start", TP_PLAYER},
  {  20,  16,    0,    0,  0, 0000, "AMG2A0", /* 12 */
    "Wand Geode", TP_PICK | TP_WAMMO},
  {  20,  16,    0,    0,  0, 0000, "AMM1A0", /* 13 */
    "Mace Spheres", TP_PICK | TP_WAMMO},
  {  10,  16,    0,    0,  0,    0, "TELEH0", /* 14 */
    "Teleport Exit", 0},
  {  24,  16,    0,    0,  0, 0000, "WZRDC1", /* 15 */
    "Disciple", TP_BLOCK | TP_KILL | TP_FLOAT},
  {  20,  16,    0,    0,  0, 0000, "AMM2A0", /* 16 */
    "Pile of Mace Spheres", TP_PICK | TP_WAMMO},
  {  20,  16,    0,    0,  0,    0, "SKH1A0", /* 17 */
    "Hanging Skull 70", TP_BLOCK | TP_FLOAT},
  {  20,  16,    0,    0,  0, 0000, "AMC1A0", /* 18 */
    "Ethereal Arrows", TP_PICK | TP_WAMMO},
  {  20,  16,    0,    0,  0, 0000, "AMC2A0", /* 19 */
    "Ethereal Quiver", TP_PICK | TP_WAMMO},
  {  20,  16,    0,    0,  0, 0000, "AMS1A0", /* 20 */
    "Lesser Runes", TP_PICK | TP_WAMMO},
  {  20,  16,    0,    0,  0, 0000, "AMS2A0", /* 21 */
    "Greater Runes", TP_PICK | TP_WAMMO},
  {  20,  16,    0,    0,  0, 0000, "AMP1C0", /* 22 */
    "Flame Orb", TP_PICK | TP_WAMMO},
  {  20,  16,    0,    0,  0, 0000, "AMP2B0", /* 23 */
    "Inferno Orb", TP_PICK | TP_WAMMO},
  {  20,  16,    0,    0,  0,    0, "SKH2A0", /* 24 */
    "Hanging Skull 60", TP_BLOCK | TP_FLOAT},
  {  20,  16,    0,    0,  0,    0, "SKH3A0", /* 25 */
    "Hanging Skull 45", TP_BLOCK | TP_FLOAT},
  {  20,  16,    0,    0,  0,    0, "SKH4A0", /* 26 */
    "Hanging Skull 35", TP_BLOCK | TP_FLOAT},
  {  20,  16,    0,    0,  0,    0, "SRTCA0", /* 27 */
    "Serpent Torch", TP_LIGHT | TP_BLOCK},
  {  20,  16,    0,    0,  0,    0, "CHDLA0", /* 28 */
    "Chandelier", TP_LIGHT | TP_BLOCK},
  {  20,  16,    0,    0,  0,    0, "SMPLA0", /* 29 */
    "Small Pillar", TP_BLOCK},
  {  20,  16,    0,    0,  0, 0000, "EGGCC0", /* 30 */
    "Morph Ovum", TP_PICK | TP_ITEM},
  {  20,  16,    0,    0,  0, 0000, "SHD2A0", /* 31 */
    "Enchanted Shield", TP_PICK | TP_ITEM},
  {  20,  16,    0,    0,  0, 0000, "SPHLA0", /* 32 */
    "Mystic Urn", TP_PICK | TP_ITEM},
  {  20,  16,    0,    0,  0, 0000, "TRCHA0", /* 33 */
    "Torch", TP_PICK | TP_ITEM},
  {  20,  16,    0,    0,  0, 0000, "FBMBB0", /* 34 */
    "Time Bomb", TP_PICK | TP_ITEM},
  {  20,  16,    0,    0,  0, 0000, "SPMPA0", /* 35 */
    "Map Scroll", TP_PICK | TP_ITEM},
  {  20,  16,    0,    0,  0, 0000, "ARTIATLP", /* 36 */
    "Chaos Device", TP_PICK | TP_ITEM},
  {  20,  16,    0,    0,  0,    0, "STGSA0", /* 37 */
    "Small Stalagmite", TP_BLOCK},
  {  20,  16,    0,    0,  0,    0, "STGLA0", /* 38 */
    "Large Stalagmite", TP_BLOCK},
  {  20,  16,    0,    0,  0,    0, "STCSA0", /* 39 */
    "Small Stalactite", TP_BLOCK | TP_FLOAT},
  {  20,  16,    0,    0,  0,    0, "STCLA0", /* 40 */
    "Large Stalactite", TP_BLOCK | TP_FLOAT},
  {  10,  16,    0,    0,  0,    0, "------", /* 41 */
    "Sound: Waterfall", TP_SOUND},
  {  10,  16,    0,    0,  0,    0, "------", /* 42 */
    "Sound: Wind", TP_SOUND},
  {  10,  16,    0,    0,  0,    0, "PPODG0", /* 43 */ /*! Check props */
    "Pod Spawn Spot", 0},
  {  20,  16,    0,    0,  0,    0, "BARLA0", /* 44 */
    "Barrel", TP_BLOCK},
  {  20,  16,    0,    0,  0, 0000, "MUMMY1", /* 45 */
    "Golem Leader", TP_BLOCK | TP_KILL},
  {  20,  16,    0,    0,  0, 0000, "MUMMY1", /* 46 */
    "Golem Leader Ghost", TP_BLOCK | TP_KILL | TP_INVIS},
  {  20,  16,    0,    0,  0,    0, "BRPLA0", /* 47 */
    "Brown Pillar", TP_BLOCK},
  {  20,  16,    0,    0,  0,    0, "MOS1A0", /* 48 */
    "Moss 1", TP_BLOCK},
  {  20,  16,    0,    0,  0,    0, "MOS2A0", /* 49 */
    "Moss 2", TP_BLOCK},
  {  20,  16,    0,    0,  0,    0, "WTRHA0", /* 50 */
    "Wall Torch", TP_LIGHT | TP_BLOCK},
  {  20,  16,    0,    0,  0,    0, "HCORA0", /* 51 */
    "Hanging Corpse", TP_BLOCK | TP_FLOAT},
  {  10,  16,    0,    0,  0,    0, "TGLTG0", /* 52 */ /*! Use Composite bitmap */
    "Exit Glitter", TP_LIGHT},
  {  20,  16,    0,    0,  0, 0000, "WBLSA0", /* 53 */
    "Dragonclaw", TP_PICK | TP_WEAPON},
  {  20,  16,    0,    0,  0, 0000, "AMB1A0", /* 54 */
    "Claw Orb", TP_PICK | TP_WAMMO},
  {  20,  16,    0,    0,  0, 0000, "AMB2A0", /* 55 */
    "Energy Orb", TP_PICK | TP_WAMMO},
  {  20,  16,    0,    0,  0,    0, "SOR2I0", /* 56 */ /*! Check bitmap */
    "D'sparil Spot", 0},
  { 255,   0,    0,    0,  0,    0, "------", /* 57 */
    "<UNKNOWN 57>", TP_BAD},
  { 255,   0,    0,    0,  0,    0, "------", /* 58 */
    "<UNKNOWN 58>", TP_BAD},
  { 255,   0,    0,    0,  0,    0, "------", /* 59 */
    "<UNKNOWN 59>", TP_BAD},
  { 255,   0,    0,    0,  0,    0, "------", /* 60 */
    "<UNKNOWN 60>", TP_BAD},
  { 255,   0,    0,    0,  0,    0, "------", /* 61 */
    "<UNKNOWN 61>", TP_BAD},
  { 255,   0,    0,    0,  0,    0, "------", /* 62 */
    "<UNKNOWN 62>", TP_BAD},
  { 255,   0,    0,    0,  0,    0, "------", /* 63 */
    "<UNKNOWN 63>", TP_BAD},
  {  24,  16,    0,    0,  0, 0000, "KNIGG1", /* 64 */
    "Undead Warrior", TP_BLOCK | TP_KILL},
  {  24,  16,    0,    0,  0, 0000, "KNIGG1", /* 65 */
    "Undead Warrior Ghost", TP_BLOCK | TP_KILL | TP_INVIS},
  {  16,  16,    0,    0,  0, 0000, "IMPXF1", /* 66 */
    "Flying Gargoyle", TP_BLOCK | TP_KILL | TP_FLOAT},
  { 255,   0,    0,    0,  0,    0, "------", /* 67 */
    "<UNKNOWN 67>", TP_BAD},
  {  20,  16,    0,    0,  0, 0000, "MUMME1", /* 68 */
    "Golem", TP_BLOCK | TP_KILL},
  {  20,  16,    0,    0,  0, 0000, "MUMME1", /* 69 */
    "Golem Ghost", TP_BLOCK | TP_KILL | TP_INVIS},
  {  28,  16,    0,    0,  0, 0000, "BEASI1", /* 70 */
    "Weredragon", TP_BLOCK | TP_KILL},
  { 255,   0,    0,    0,  0,    0, "------", /* 71 */
    "<UNKNOWN 71>", TP_BAD},
  { 255,   0,    0,    0,  0,    0, "------", /* 72 */
    "<UNKNOWN 72>", TP_BAD},
  {  20,  16,    0,    0,  0,    0, "AKYYB0", /* 73 */
    "Green Key", TP_PICK},
  {  10,  16,    0,    0,  0,    0, "TGLTB0", /* 74 */ /*! Use Composite bitmap */
    "Teleport Glitter", TP_LIGHT},
  {  20,  16,    0,    0,  0, 0000, "INVSA0", /* 75 */
    "Shadowsphere", TP_INVIS | TP_PICK | TP_ITEM},
  {  20,  16,    0,    0,  0,    0, "KFR1F0", /* 76 */
    "Fire Brazier", TP_LIGHT | TP_BLOCK},
  { 255,   0,    0,    0,  0,    0, "------", /* 77 */
    "<UNKNOWN 77>", TP_BAD},
  { 255,   0,    0,    0,  0,    0, "------", /* 78 */
    "<UNKNOWN 78>", TP_BAD},
  {  20,  16,    0,    0,  0,    0, "BKYYB0", /* 79 */
    "Blue Key", TP_PICK},
  {  20,  16,    0,    0,  0,    0, "CKYYB0", /* 80 */
    "Yellow Key", TP_PICK},
  {  20,  16,    0,    0,  0, 0000, "PTN1A0", /* 81 */
    "Crystal Vial", TP_PICK},
  {  20,  16,    0,    0,  0, 0000, "PTN2C0", /* 82 */
    "Quartz Flask", TP_PICK | TP_ITEM},
  {  20,  16,    0,    0,  0, 0000, "SOARA0", /* 83 */
    "Wings of Wrath", TP_PICK | TP_ITEM},
  {  20,  16,    0,    0,  0, 0000, "INVUD0", /* 84 */
    "Ring of Invulnerability", TP_PICK | TP_ITEM},
  {  20,  16,    0,    0,  0, 0000, "SHLDA0", /* 85 */
    "Silver Shield", TP_PICK | TP_ITEM},
  {  20,  16,    0,    0,  0, 0000, "PWBKA0", /* 86 */
    "Tome of Power", TP_PICK | TP_ITEM},
  {  20,  16,    0,    0,  0,    0, "VLCOE0", /* 87 */
    "Volcano", TP_BLOCK},
  { 255,   0,    0,    0,  0,    0, "------", /* 88 */
    "<UNKNOWN 88>", TP_BAD},
  { 255,   0,    0,    0,  0,    0, "------", /* 89 */
    "<UNKNOWN 89>", TP_BAD},
  {  28,  16,    0,    0,  0, 0000, "CLNKG1", /* 90 */
    "Sabreclaw", TP_BLOCK | TP_KILL},
  { 255,   0,    0,    0,  0,    0, "------", /* 91 */
    "<UNKNOWN 91>", TP_BAD},
  {  28,  16,    0,    0,  0, 0000, "SNKEF1", /* 92 */
    "Ophidian", TP_BLOCK | TP_KILL},
  { 255,   0,    0,    0,  0,    0, "------", /* 93 */
    "<UNKNOWN 93>", TP_BAD},
  {  20,  16,    0,    0,  0,    0, "KGZBA0", /* 94 */ /*! Use Composite bitmap */
    "Blue Key Statue", TP_BLOCK},
  {  20,  16,    0,    0,  0,    0, "KGZGA0", /* 95 */ /*! Use Composite bitmap */
    "Green Key Statue", TP_BLOCK},
  {  20,  16,    0,    0,  0,    0, "KGZYA0", /* 96 */ /*! Use Composite bitmap */
    "Yellow Key Statue", TP_BLOCK}
};

static ThingDesc HThingInfo1[MAXHTHINGINFO1 + 1 - MINHTHINGINFO1] =
{
  {  10,  16,    0,    0,  0,    0, "------", /* 1200 */
    "Sound: Scream", TP_SOUND},
  {  10,  16,    0,    0,  0,    0, "------", /* 1201 */
    "Sound: Squish", TP_SOUND},
  {  10,  16,    0,    0,  0,    0, "------", /* 1202 */
    "Sound: Water dripping", TP_SOUND},
  {  10,  16,    0,    0,  0,    0, "------", /* 1203 */
    "Sound: Slow footsteps", TP_SOUND},
  {  10,  16,    0,    0,  0,    0, "------", /* 1204 */
    "Sound: Heart beating", TP_SOUND},
  {  10,  16,    0,    0,  0,    0, "------", /* 1205 */
    "Sound: Bells ringing", TP_SOUND},
  {  10,  16,    0,    0,  0,    0, "------", /* 1206 */
    "Sound: Growling", TP_SOUND},
  {  10,  16,    0,    0,  0,    0, "------", /* 1207 */
    "Sound: Magic", TP_SOUND},
  {  10,  16,    0,    0,  0,    0, "------", /* 1208 */
    "Sound: Laughter", TP_SOUND},
  {  10,  16,    0,    0,  0,    0, "------", /* 1209 */
    "Sound: Fast footsteps", TP_SOUND}
};

static ThingDesc HThingInfo2[MAXHTHINGINFO2 + 1 - MINHTHINGINFO2] =
{
  {  20,  16,    0,    0,  0, 0000, "WBOWA0", /* 2001 */
    "Ethereal Crossbow", TP_PICK | TP_WEAPON},
  {  20,  16,    0,    0,  0, 0000, "WMCEA0", /* 2002 */
    "Mace", TP_PICK | TP_WEAPON},
  {  20,  16,    0,    0,  0, 0000, "WPHXA0", /* 2003 */
    "Phoenix Rod", TP_PICK | TP_WEAPON},
  {  20,  16,    0,    0,  0, 0000, "WSKLA0", /* 2004 */
    "Hellstaff", TP_PICK | TP_WEAPON},
  {  20,  16,    0,    0,  0, 0000, "WGNTA0", /* 2005 */
    "Gauntlets of Necromancer", TP_PICK | TP_WEAPON},
  { 255,   0,    0,    0,  0,    0, "------", /* 2006 */
    "<UNKNOWN 2006>", TP_BAD},
  { 255,   0,    0,    0,  0,    0, "------", /* 2007 */
    "<UNKNOWN 2007>", TP_BAD},
  { 255,   0,    0,    0,  0,    0, "------", /* 2008 */
    "<UNKNOWN 2008>", TP_BAD},
  { 255,   0,    0,    0,  0,    0, "------", /* 2009 */
    "<UNKNOWN 2009>", TP_BAD},
  { 255,   0,    0,    0,  0,    0, "------", /* 2010 */
    "<UNKNOWN 2010>", TP_BAD},
  { 255,   0,    0,    0,  0,    0, "------", /* 2011 */
    "<UNKNOWN 2011>", TP_BAD},
  { 255,   0,    0,    0,  0,    0, "------", /* 2012 */
    "<UNKNOWN 2012>", TP_BAD},
  { 255,   0,    0,    0,  0,    0, "------", /* 2013 */
    "<UNKNOWN 2013>", TP_BAD},
  { 255,   0,    0,    0,  0,    0, "------", /* 2014 */
    "<UNKNOWN 2014>", TP_BAD},
  { 255,   0,    0,    0,  0,    0, "------", /* 2015 */
    "<UNKNOWN 2015>", TP_BAD},
  { 255,   0,    0,    0,  0,    0, "------", /* 2016 */
    "<UNKNOWN 2016>", TP_BAD},
  { 255,   0,    0,    0,  0,    0, "------", /* 2017 */
    "<UNKNOWN 2017>", TP_BAD},
  { 255,   0,    0,    0,  0,    0, "------", /* 2018 */
    "<UNKNOWN 2018>", TP_BAD},
  { 255,   0,    0,    0,  0,    0, "------", /* 2019 */
    "<UNKNOWN 2019>", TP_BAD},
  { 255,   0,    0,    0,  0,    0, "------", /* 2020 */
    "<UNKNOWN 2020>", TP_BAD},
  { 255,   0,    0,    0,  0,    0, "------", /* 2021 */
    "<UNKNOWN 2021>", TP_BAD},
  { 255,   0,    0,    0,  0,    0, "------", /* 2022 */
    "<UNKNOWN 2022>", TP_BAD},
  { 255,   0,    0,    0,  0,    0, "------", /* 2023 */
    "<UNKNOWN 2023>", TP_BAD},
  { 255,   0,    0,    0,  0,    0, "------", /* 2024 */
    "<UNKNOWN 2024>", TP_BAD},
  { 255,   0,    0,    0,  0,    0, "------", /* 2025 */
    "<UNKNOWN 2025>", TP_BAD},
  { 255,   0,    0,    0,  0,    0, "------", /* 2026 */
    "<UNKNOWN 2026>", TP_BAD},
  { 255,   0,    0,    0,  0,    0, "------", /* 2027 */
    "<UNKNOWN 2027>", TP_BAD},
  { 255,   0,    0,    0,  0,    0, "------", /* 2028 */
    "<UNKNOWN 2028>", TP_BAD},
  { 255,   0,    0,    0,  0,    0, "------", /* 2029 */
    "<UNKNOWN 2029>", TP_BAD},
  { 255,   0,    0,    0,  0,    0, "------", /* 2030 */
    "<UNKNOWN 2030>", TP_BAD},
  { 255,   0,    0,    0,  0,    0, "------", /* 2031 */
    "<UNKNOWN 2031>", TP_BAD},
  { 255,   0,    0,    0,  0,    0, "------", /* 2032 */
    "<UNKNOWN 2032>", TP_BAD},
  { 255,   0,    0,    0,  0,    0, "------", /* 2033 */
    "<UNKNOWN 2033>", TP_BAD},
  { 255,   0,    0,    0,  0,    0, "------", /* 2034 */
    "<UNKNOWN 2034>", TP_BAD},
  {  10,  16,    0,    0,  0,    0, "PPODA0", /* 2035 */
    "Puff Pod", TP_BLOCK}
};


/*
   Get the name of a Thing.
*/

char *GetThingName(Int16 type)
{
  static char temp[20];

  if (DoomVersion < 16)
    {
      /* Doom Things */
      if (type >= MINDTHINGINFO0 && type <= MAXDTHINGINFO0)
        return DThingInfo0[type - MINDTHINGINFO0].name;
      if (type >= MINDTHINGINFO2 && type <= MAXDTHINGINFO2)
        return DThingInfo2[type - MINDTHINGINFO2].name;
      if (type >= MINDTHINGINFO3 && type <= MAXDTHINGINFO3)
        return DThingInfo3[type - MINDTHINGINFO3].name;
    }
  else
    {
      /* Heretic Things */
      if (type >= MINHTHINGINFO0 && type <= MAXHTHINGINFO0)
        return HThingInfo0[type - MINHTHINGINFO0].name;
      if (type >= MINHTHINGINFO1 && type <= MAXHTHINGINFO1)
        return HThingInfo1[type - MINHTHINGINFO1].name;
      if (type >= MINHTHINGINFO2 && type <= MAXHTHINGINFO2)
        return HThingInfo2[type - MINHTHINGINFO2].name;
    }
  sprintf(temp, "<UNKNOWN %d>", type);
  return temp;
}



/*
   Get the sprite name of a thing.
*/

char *GetThingPicName(Int16 type)
{
  if (DoomVersion < 16)
    {
      /* Doom Things */
      if (type >= MINDTHINGINFO0 && type <= MAXDTHINGINFO0)
        return DThingInfo0[type - MINDTHINGINFO0].sprite;
      if (type >= MINDTHINGINFO2 && type <= MAXDTHINGINFO2)
        return DThingInfo2[type - MINDTHINGINFO2].sprite;
      if (type >= MINDTHINGINFO3 && type <= MAXDTHINGINFO3)
        return DThingInfo3[type - MINDTHINGINFO3].sprite;
    }
  else
    {
      /* Heretic Things */
      if (type >= MINHTHINGINFO0 && type <= MAXHTHINGINFO0)
        return HThingInfo0[type - MINHTHINGINFO0].sprite;
      if (type >= MINHTHINGINFO1 && type <= MAXHTHINGINFO1)
        return HThingInfo1[type - MINHTHINGINFO1].sprite;
      if (type >= MINHTHINGINFO2 && type <= MAXHTHINGINFO2)
        return HThingInfo2[type - MINHTHINGINFO2].sprite;
    }
  return "------";
}



/*
   Get the size of a thing.
*/

Int16 GetThingRadius(Int16 type)
{
  if (DoomVersion < 16)
    {
      /* Doom Things */
      if (type >= MINDTHINGINFO0 && type <= MAXDTHINGINFO0)
        return DThingInfo0[type - MINDTHINGINFO0].radius;
      if (type >= MINDTHINGINFO2 && type <= MAXDTHINGINFO2)
        return DThingInfo2[type - MINDTHINGINFO2].radius;
      if (type >= MINDTHINGINFO3 && type <= MAXDTHINGINFO3)
        return DThingInfo3[type - MINDTHINGINFO3].radius;
    }
  else
    {
      /* Heretic Things */
      if (type >= MINHTHINGINFO0 && type <= MAXHTHINGINFO0)
        return HThingInfo0[type - MINHTHINGINFO0].radius;
      if (type >= MINHTHINGINFO1 && type <= MAXHTHINGINFO1)
        return HThingInfo1[type - MINHTHINGINFO1].radius;
      if (type >= MINHTHINGINFO2 && type <= MAXHTHINGINFO2)
        return HThingInfo2[type - MINHTHINGINFO2].radius;
    }
  return 255;
}



/*
   Get the height of a thing.
*/

Int16 GetThingHeight(Int16 type)
{
  if (DoomVersion < 16)
    {
      /* Doom Things */
      if (type >= MINDTHINGINFO0 && type <= MAXDTHINGINFO0)
        return DThingInfo0[type - MINDTHINGINFO0].height;
      if (type >= MINDTHINGINFO2 && type <= MAXDTHINGINFO2)
        return DThingInfo2[type - MINDTHINGINFO2].height;
      if (type >= MINDTHINGINFO3 && type <= MAXDTHINGINFO3)
        return DThingInfo3[type - MINDTHINGINFO3].height;
    }
  else
    {
      /* Heretic Things */
      if (type >= MINHTHINGINFO0 && type <= MAXHTHINGINFO0)
        return HThingInfo0[type - MINHTHINGINFO0].height;
      if (type >= MINHTHINGINFO1 && type <= MAXHTHINGINFO1)
        return HThingInfo1[type - MINHTHINGINFO1].height;
      if (type >= MINHTHINGINFO2 && type <= MAXHTHINGINFO2)
        return HThingInfo2[type - MINHTHINGINFO2].height;
    }
  return 0;
}



/*
   Get the number of health points that a thing has (0 = not alive).
*/

Int16 GetThingHealth(Int16 type)
{
  if (DoomVersion < 16)
    {
      /* Doom Things */
      if (type >= MINDTHINGINFO0 && type <= MAXDTHINGINFO0)
        return DThingInfo0[type - MINDTHINGINFO0].health;
      if (type >= MINDTHINGINFO2 && type <= MAXDTHINGINFO2)
        return DThingInfo2[type - MINDTHINGINFO2].health;
      if (type >= MINDTHINGINFO3 && type <= MAXDTHINGINFO3)
        return DThingInfo3[type - MINDTHINGINFO3].health;
    }
  else
    {
      /* Heretic Things */
      if (type >= MINHTHINGINFO0 && type <= MAXHTHINGINFO0)
        return HThingInfo0[type - MINHTHINGINFO0].health;
      if (type >= MINHTHINGINFO1 && type <= MAXHTHINGINFO1)
        return HThingInfo1[type - MINHTHINGINFO1].health;
      if (type >= MINHTHINGINFO2 && type <= MAXHTHINGINFO2)
        return HThingInfo2[type - MINHTHINGINFO2].health;
    }
  return 0;
}



/*
   Get the number of damage points that a thing has (0 = no damage).
*/

Int16 GetThingDamage(Int16 type)
{
  if (DoomVersion < 16)
    {
      /* Doom Things */
      if (type >= MINDTHINGINFO0 && type <= MAXDTHINGINFO0)
        return DThingInfo0[type - MINDTHINGINFO0].damage;
      if (type >= MINDTHINGINFO2 && type <= MAXDTHINGINFO2)
        return DThingInfo2[type - MINDTHINGINFO2].damage;
      if (type >= MINDTHINGINFO3 && type <= MAXDTHINGINFO3)
        return DThingInfo3[type - MINDTHINGINFO3].damage;
    }
  else
    {
      /* Heretic Things */
      if (type >= MINHTHINGINFO0 && type <= MAXHTHINGINFO0)
        return HThingInfo0[type - MINHTHINGINFO0].damage;
      if (type >= MINHTHINGINFO1 && type <= MAXHTHINGINFO1)
        return HThingInfo1[type - MINHTHINGINFO1].damage;
      if (type >= MINHTHINGINFO2 && type <= MAXHTHINGINFO2)
        return HThingInfo2[type - MINHTHINGINFO2].damage;
    }
  return 0;
}



/*
   Get the properties (flags) for a type of thing.
*/

UInt16 GetThingProperties(Int16 type)
{
  if (DoomVersion < 16)
    {
      /* Doom Things */
      if (type >= MINDTHINGINFO0 && type <= MAXDTHINGINFO0)
        return DThingInfo0[type - MINDTHINGINFO0].props;
      if (type >= MINDTHINGINFO2 && type <= MAXDTHINGINFO2)
        return DThingInfo2[type - MINDTHINGINFO2].props;
      if (type >= MINDTHINGINFO3 && type <= MAXDTHINGINFO3)
        return DThingInfo3[type - MINDTHINGINFO3].props;
    }
  else
    {
      /* Heretic Things */
      if (type >= MINHTHINGINFO0 && type <= MAXHTHINGINFO0)
        return HThingInfo0[type - MINHTHINGINFO0].props;
      if (type >= MINHTHINGINFO1 && type <= MAXHTHINGINFO1)
        return HThingInfo1[type - MINHTHINGINFO1].props;
      if (type >= MINHTHINGINFO2 && type <= MAXHTHINGINFO2)
        return HThingInfo2[type - MINHTHINGINFO2].props;
    }
  return TP_BAD;
}



/*
   Get the colour of a thing (for the map display).
*/

Int16 GetThingColour(Int16 type)
{
  UInt16 props;

  props = GetThingProperties(type);
  if (props == TP_BAD)
    return WHITE;
  if (Config.colour2 == FALSE)
    {
      if (props & TP_PLAYER)
        return GREEN;
      if (type == THING_BARREL)
        return LIGHTMAGENTA;
      if (props & TP_KILL)
        return LIGHTRED;
      if ((props & TP_WEAPON) || (props & TP_WAMMO))
        return BROWN;
      if (props & TP_PICK)
        return LIGHTGREEN;
      if (type == THING_TELEPORT ||
          (DoomVersion < 0x10 &&
           (type == THING_SPAWNSPOT || type == THING_BOSSSHOOTER)) ||
          (DoomVersion >= 0x10 && type == THING_DSPARILSPOT))
        return YELLOW;
      return WHITE;
    }
  else
    {
      if (props & TP_PLAYER)
        if (type == THING_DEATHMATCH)
          return LIGHTGREEN;
        else
          return GREEN;
      if (type == THING_TELEPORT ||
          (DoomVersion < 0x10 &&
           (type == THING_SPAWNSPOT || type == THING_BOSSSHOOTER)) ||
          (DoomVersion >= 0x10 && type == THING_DSPARILSPOT))
        return YELLOW;
      if (props & TP_KILL)
        if (GetThingHealth(type) >= 250)
          return LIGHTRED;
        else
          return RED;
      if (props & TP_WEAPON)
        if (GetThingDamage(type) >= 600)
          return LIGHTBLUE;
        else
          return BLUE;
      if (props & TP_ITEM)
        if (GetThingHealth(type) >= 25 || GetThingDamage(type) >= 500)
          return LIGHTMAGENTA;
        else
          return MAGENTA;
      if (props & TP_PICK)
        if (GetThingHealth(type) >= 25 || GetThingDamage(type) >= 500)
          return LIGHTCYAN;
        else
          return CYAN;
      return BROWN;
    }
}



/*
   Get the class of a thing (for scrollbox classification).
*/

Int16 GetThingClass(Int16 type)
{
  UInt16 props;

  props = GetThingProperties(type);

  if (props == TP_BAD)
    return -1; /*! Generate a segfault */
  if ((props & TP_PLAYER) || type == THING_TELEPORT)
    return TC_PLAYER;
  if (type == THING_BARREL)
    return TC_DECOR;
  if (props & TP_KILL)
    return TC_ENEMY;
  if (props & TP_WEAPON)
    return TC_WEAPON;
  if (props & TP_ITEM)
    return TC_ARTFCT;
  if (props & TP_PICK)
    return TC_EQPMNT;
  if (props & TP_LIGHT)
    return TC_LIGHT;
  if (props & TP_FLOAT)
    return TC_HNGDEC;
  if (props & TP_CORPSE)
    return TC_CORPSE;
  if (props & TP_SOUND)
    return TC_SOUND;
  return TC_DECOR;
}



/*
   Returns the key type, or 0 for "not-a-key".
*/

Int16 GetKeyType(Int16 type)
{
  if (DoomVersion < 16)
    switch(type)
      {
        case THING_DOOM_BLUEKEY:
        case THING_DOOM_BLUESKL:
          return KEYTYPE_BLUE;
        case THING_DOOM_YELLOWKEY:
        case THING_DOOM_YELLOWSKL:
          return KEYTYPE_YLLW;
        case THING_DOOM_REDKEY:
        case THING_DOOM_REDSKL:
          return KEYTYPE_RED;
        default:
          return KEYTYPE_NONE;
      }
  else
    switch(type)
      {
        case THING_HTIC_BLUEKEY:
          return KEYTYPE_BLUE;
        case THING_HTIC_YELLOWKEY:
          return KEYTYPE_YLLW;
        case THING_HTIC_GREENKEY:
          return KEYTYPE_RED;
        default:
          return KEYTYPE_NONE;
      }
}

/* end of file */
