/*----------------------------------------------------------------------------*
 | This file is part of DEU (Doom Editing Utilities), created by the DEU team:|
 | Raphael Quinet, Brendon Wyber, Ted Vessenes and others.  See README.1ST or |
 | the "about" dialog box for full credits.                                   |
 |                                                                            |
 | DEU is an open project: if you think that you can contribute, please join  |
 | the DEU team.  You will be credited for any code (or ideas) included in    |
 | the next version of the program.                                           |
 |                                                                            |
 | If you want to make any modifications and re-distribute them on your own,  |
 | you must follow the conditions of the DEU license.  Read the file LICENSE  |
 | in this directory or README.1ST in the top directory.  If do not have a    |
 | copy of these files, you can request them from any member of the DEU team, |
 | or by mail: Raphael Quinet, Rue des Martyrs 9, B-4550 Nandrin (Belgium).   |
 |                                                                            |
 | This program comes with absolutely no warranty.  Use it at your own risks! |
 *----------------------------------------------------------------------------*

 G_MOUDOS.C - Mouse driver for GRX

 Functions written by Per Allansson
*/

/* the includes */
#include "deu.h"
#include <dos.h>
#include <mousex.h>
#include "g_gfx.h"
#include "g_colcnv.h"
#include "g_mouse.h"

#ifdef DEBUG_MOUSE
#include "d_misc.h"  /* PlaySound */
#undef ShowMousePointer
#undef HideMousePointer
#endif

/* the global data */
Bool UseMouse = FALSE;                  /* is there a mouse driver? */


/*
   Initialize the mouse driver.
*/

void InitMouseDriver(void)
{
  if (MouseDetect())
    {
      UseMouse = TRUE;
      MouseEventMode(0);
      MouseInit();
      MouseSetColors(TranslateToDoomColor(WHITE), GrNOCOLOR);
    }
  else
    UseMouse = FALSE;
}



/*
   Show the pointer.
*/

void ShowMousePointer(void)
{
  MouseDisplayCursor();
  MouseEventEnable(0, 1);
}



/*
   Hide the pointer.
*/

void HideMousePointer(void)
{
  MouseEraseCursor();
  MouseEventEnable(0, 0);
}


/*
   Read pointer coordinates.
*/

void GetMouseCoords(UInt16 *x, UInt16 *y, UInt16 *buttons)
{
  MouseEvent mevent;

  MouseGetEvent(M_POLL | M_BUTTON_DOWN | M_BUTTON_UP, &mevent);
  *x = mevent.x;
  *y = mevent.y;
  *buttons = mevent.buttons;
}


/*
   Change pointer coordinates.
*/

void SetMouseCoords(UInt16 x, UInt16 y)
{
  MouseWarp(x, y);
}



/*
   Set horizontal and vertical limits (constrain pointer in a box).
*/

void SetMouseLimits(UInt16 x0, UInt16 y0, UInt16 x1, UInt16 y1)
{
  MouseSetLimits(x0, y0, x1, y1);
}



/*
   Reset horizontal and vertical limits.
*/

void ResetMouseLimits(void)
{
  MouseSetLimits(0, 0, ScrMaxX, ScrMaxY);
}


#ifdef DEBUG_MOUSE

static Bool  mouse_shown = FALSE;
static char *old_file = "(NONE)";
static int   old_line = 0;

/*
   Warn the user if the mouse pointer is shown twice.
*/

void DebugShowMouse(char *file, int line)
{
  if (mouse_shown == TRUE)
    {
      PlaySound(50, 100);
      PlaySound(8000, 100);
      LogMessage("\nWarning: Mouse pointer shown twice.\n");
      LogMessage("\tFirst call in %s, line %d\n", old_file, old_line);
      LogMessage("\tSecond call in %s, line %d\n", file, line);
    }
  else
    ShowMousePointer();
  mouse_shown = TRUE;
  old_file = file;
  old_line = line;
}


/*
   Warn the user if the mouse pointer is hidden twice.
*/

void DebugHideMouse(char *file, int line)
{
  if (mouse_shown == FALSE)
    {
      PlaySound(8000, 100);
      PlaySound(50, 100);
      LogMessage("\nWarning: Mouse pointer hidden twice.\n");
      LogMessage("\tFirst call in %s, line %d\n", old_file, old_line);
      LogMessage("\tSecond call in %s, line %d\n", file, line);
    }
  else
    HideMousePointer();
  mouse_shown = FALSE;
  old_file = file;
  old_line = line;
}

#endif /* DEBUG_MOUSE */

/* end of file */
