/*----------------------------------------------------------------------------*
 | This file is part of DEU (Doom Editing Utilities), created by the DEU team:|
 | Raphael Quinet, Brendon Wyber, Ted Vessenes and others.  See README.1ST or |
 | the "about" dialog box for full credits.                                   |
 |                                                                            |
 | DEU is an open project: if you think that you can contribute, please join  |
 | the DEU team.  You will be credited for any code (or ideas) included in    |
 | the next version of the program.                                           |
 |                                                                            |
 | If you want to make any modifications and re-distribute them on your own,  |
 | you must follow the conditions of the DEU license.  Read the file LICENSE  |
 | in this directory or README.1ST in the top directory.  If do not have a    |
 | copy of these files, you can request them from any member of the DEU team, |
 | or by mail: Raphael Quinet, Rue des Martyrs 9, B-4550 Nandrin (Belgium).   |
 |                                                                            |
 | This program comes with absolutely no warranty.  Use it at your own risks! |
 *----------------------------------------------------------------------------*

 G_MOUBGI.C - Mouse driver for Borland C (BGI)

 Functions written by Raphael Quinet
*/

/* the includes */
#include "deu.h"
#include <dos.h>
#include "g_gfx.h"
#include "g_mouse.h"

#ifdef DEBUG_MOUSE
#include "d_misc.h"  /* PlaySound */
#undef ShowMousePointer
#undef HideMousePointer
#endif

/* mouse interrupt number */
#define MOUSE 0x33

/* the global data */
Bool UseMouse = FALSE;                  /* is there a mouse driver? */


/*
   Initialize the mouse driver.
*/

void InitMouseDriver(void)
{
  union REGS regs;

  regs.x.ax = 0x0000;
  int86(MOUSE, &regs, &regs);
  if (regs.x.ax == 0xffff)
    {
      UseMouse = TRUE; /* mouse */
      /*! change the pointer shape & color */
    }
  else
    UseMouse = FALSE; /* no mouse */
}


/*
   Show the pointer.
*/


void ShowMousePointer(void)
{
  union REGS regs;

  regs.x.ax = 0x0001;
  int86(MOUSE, &regs, &regs);
}



/*
   Hide the pointer.
*/

void HideMousePointer(void)
{
  union REGS regs;

  regs.x.ax = 0x0002;
  int86(MOUSE, &regs, &regs);
}


/*
   Read pointer coordinates.
*/

void GetMouseCoords(UInt16 *x, UInt16 *y, UInt16 *buttons)
{
  union REGS regs;

  regs.x.ax = 0x0003;
  int86(MOUSE, &regs, &regs);
  if (x != NULL)
    *x = regs.x.cx;
  if (y != NULL)
    *y = regs.x.dx;
  if (buttons != NULL)
    *buttons = regs.x.bx;
}


/*
   Change pointer coordinates.
*/

void SetMouseCoords(UInt16 x, UInt16 y)
{
  union REGS regs;

  regs.x.ax = 0x0004;
  regs.x.cx = x;
  regs.x.dx = y;
  int86(MOUSE, &regs, &regs);
}



/*
   Set horizontal and vertical limits (constrain pointer in a box).
*/

void SetMouseLimits(UInt16 x0, UInt16 y0, UInt16 x1, UInt16 y1)
{
  union REGS regs;

  regs.x.ax = 0x0007;
  regs.x.cx = x0;
  regs.x.dx = x1;
  int86(MOUSE, &regs, &regs);
  regs.x.ax = 0x0008;
  regs.x.cx = y0;
  regs.x.dx = y1;
  int86(MOUSE, &regs, &regs);
}



/*
   Reset horizontal and vertical limits.
*/

void ResetMouseLimits(void)
{
  union REGS regs;

  regs.x.ax = 0x0007;
  regs.x.cx = 0;
  regs.x.dx = ScrMaxX;
  int86(MOUSE, &regs, &regs);
  regs.x.ax = 0x0008;
  regs.x.cx = 0;
  regs.x.dx = ScrMaxY;
  int86(MOUSE, &regs, &regs);
}


#ifdef DEBUG_MOUSE

static Bool  mouse_shown = FALSE;
static char *old_file = "(NONE)";
static int   old_line = 0;

/*
   Warn the user if the mouse pointer is shown twice.
*/

void DebugShowMouse(char *file, int line)
{
  if (mouse_shown == TRUE)
    {
      PlaySound(50, 100);
      PlaySound(8000, 100);
      LogMessage("\nWarning: Mouse pointer shown twice.\n");
      LogMessage("\tFirst call in %s, line %d\n", old_file, old_line);
      LogMessage("\tSecond call in %s, line %d\n", file, line);
    }
  else
    ShowMousePointer();
  mouse_shown = TRUE;
  old_file = file;
  old_line = line;
}


/*
   Warn the user if the mouse pointer is hidden twice.
*/

void DebugHideMouse(char *file, int line)
{
  if (mouse_shown == FALSE)
    {
      PlaySound(8000, 100);
      PlaySound(50, 100);
      LogMessage("\nWarning: Mouse pointer hidden twice.\n");
      LogMessage("\tFirst call in %s, line %d\n", old_file, old_line);
      LogMessage("\tSecond call in %s, line %d\n", file, line);
    }
  else
    HideMousePointer();
  mouse_shown = FALSE;
  old_file = file;
  old_line = line;
}

#endif /* DEBUG_MOUSE */

/* end of file */
