/*----------------------------------------------------------------------------*
 | This file is part of DEU (Doom Editing Utilities), created by the DEU team:|
 | Raphael Quinet, Brendon Wyber, Ted Vessenes and others.  See README.1ST or |
 | the "about" dialog box for full credits.                                   |
 |                                                                            |
 | DEU is an open project: if you think that you can contribute, please join  |
 | the DEU team.  You will be credited for any code (or ideas) included in    |
 | the next version of the program.                                           |
 |                                                                            |
 | If you want to make any modifications and re-distribute them on your own,  |
 | you must follow the conditions of the DEU license.  Read the file LICENSE  |
 | in this directory or README.1ST in the top directory.  If do not have a    |
 | copy of these files, you can request them from any member of the DEU team, |
 | or by mail: Raphael Quinet, Rue des Martyrs 9, B-4550 Nandrin (Belgium).   |
 |                                                                            |
 | This program comes with absolutely no warranty.  Use it at your own risks! |
 *----------------------------------------------------------------------------*

 F_EXPIMP.C - File Export/Import routines.

*/

#include "deu.h"
#include "d_misc.h"
#include "d_wads.h"
#include "f_sound.h"
#include "f_extins.h"
#include "f_graph.h"
#include "f_color.h"
#include "g_gfx.h"

/* File types */
#define FTYPE_AU   0  /* sound    (*.AU) */ /*! not done yet */
#define FTYPE_BMP  1  /* graphics (*.BMP) */
#define FTYPE_GIF  2  /* graphics (*.GIF) */
#define FTYPE_LMP  3  /* raw file (*.RAW, *.LMP, *.MUS) */
#define FTYPE_MIDI 4  /* music    (*.MID) */ /*! not done yet */
#define FTYPE_PCX  5  /* graphics (*.PCX) */ /*! not done yet */
#define FTYPE_PPM  6  /* graphics (*.PPM, *.PGM, *.PBM) */
#define FTYPE_VOC  7  /* sound    (*.VOC) */
#define FTYPE_WAV  8  /* sound    (*.WAV) */

/*!
NOTE's from Mark Mathews:

MIDI format is not done. Trying to find MUS file format.



In void InsertResource(char * raw_filename, char *wad_filename, char *entryname)
there are 2 test case. You will have to surround code with #ifdef NEVER_DEFINED and #endif
and recompile to test it.

1) load a picture, view picture, and convert to other formats.
   load a sound and convert to other formats

        case FTYPE_AU:
#ifdef NEVER_DEFINED
          AUToWad(file, raw);
#endif
          {
             FILE *temp_file;

             UInt8 far * buffer;
             Int32 buf_size, speed;

             buffer = ReadAU(raw, &buf_size, &speed);
             if (buffer == NULL)
             {
               break;
             }
             temp_file = fopen("testau.au", "wb");
             WriteAU(temp_file, buf_size, speed, buffer);


             temp_file = fopen("testau.voc", "wb");
             WriteVOC(temp_file, buf_size, speed, buffer);


             temp_file = fopen("testau.wav", "wb");
             WriteWAV(temp_file, buf_size, speed, buffer);


             FreeFarMemory(buffer);
             fclose(temp_file);
          }


2) load picture and write to wad
   load sound and write to wad

        case FTYPE_AU:

          AUToWad(file, raw);
#ifdef NEVER_DEFINED
          {
             FILE *temp_file;

             UInt8 far * buffer;
             Int32 buf_size, speed;

             buffer = ReadAU(raw, &buf_size, &speed);
             if (buffer == NULL)
             {
               break;
             }
             temp_file = fopen("testau.au", "wb");
             WriteAU(temp_file, buf_size, speed, buffer);


             temp_file = fopen("testau.voc", "wb");
             WriteVOC(temp_file, buf_size, speed, buffer);


             temp_file = fopen("testau.wav", "wb");
             WriteWAV(temp_file, buf_size, speed, buffer);


             FreeFarMemory(buffer);
             fclose(temp_file);
          }
#endif

end of notes from Mark
*/




/*
   Compare filename extension with format exetensions.
   Return file type number (see File types above).
*/

static int GetFileType(char *filename)
{
  char *ext;

  ext = GetFileExtension(filename);
  if (ext == NULL)
    return -1;
  if (!stricmp(ext,"AU"))
    return FTYPE_AU;
  if (!stricmp(ext,"BMP"))
    return FTYPE_BMP;
  if (!stricmp(ext,"GIF"))
    return FTYPE_GIF;
  if (!stricmp(ext,"LMP"))
    return FTYPE_LMP;
  if (!stricmp(ext,"MID"))
    return FTYPE_MIDI;
  if (!stricmp(ext,"MUS"))
    return FTYPE_LMP;
  if (!stricmp(ext,"PCX"))
    return FTYPE_PCX;
  if (!stricmp(ext,"PBM"))
    return FTYPE_PPM;
  if (!stricmp(ext,"PGM"))
    return FTYPE_PPM;
  if (!stricmp(ext,"PPM"))
    return FTYPE_PPM;
  if (!stricmp(ext,"RAW"))
    return FTYPE_LMP;
  if (!stricmp(ext,"VOC"))
    return FTYPE_VOC;
  if (!stricmp(ext,"WAV"))
    return FTYPE_WAV;
  return -1; /* unknown extension */
}


/*
   ...
*/

void ExtractResource(char *entryname, char *raw_filename)
{
  int      ftype;
  FILE    *file;
  MDirPtr  entry;

  ftype = GetFileType(raw_filename);
  file = fopen(raw_filename, "wb");
  for (entry = MasterDir; entry; entry = entry->next)
    if (!strnicmp(entry->dir.name, entryname, 8))
      break;
  if (entry != NULL)
    {
      BasicWadSeek(entry->wadfile, entry->dir.start);
      switch (ftype)
        {
        case FTYPE_AU:
          WadToAU(file, entry->wadfile->fileinfo);
          break;
        case FTYPE_BMP:
          printf("[Filter not implemented yet.  Sorry!]\n");
          break;
        case FTYPE_GIF:
          printf("[Filter not implemented yet.  Sorry!]\n");
          break;
        case FTYPE_LMP:
          CopyBytes(file, entry->wadfile->fileinfo, entry->dir.size);
          break;
        case FTYPE_MIDI:
          printf("[Filter not implemented yet.  Sorry!]\n");
          break;
        case FTYPE_PCX:
          printf("[Filter not implemented yet.  Sorry!]\n");
          break;
        case FTYPE_PPM:
          printf("[Filter not implemented yet.  Sorry!]\n");
          break;
        case FTYPE_VOC:
          WadToVoc(file, entry->wadfile->fileinfo);
          break;
        case FTYPE_WAV:
          WadToWav(file, entry->wadfile->fileinfo);
          break;
        default: /* error */
          printf("[Unknown format (file extension not recognised).]\n");
          break;
        }
      }
    else
      {
        printf("[Entry not in master directory]\n");
      }
  fclose(file);
}


/*
   ...
*/

void InsertResource(char * raw_filename, char *wad_filename, char *entryname)
{
  int      ftype;
  FILE    *file, *raw;
  MDirPtr  entry;
  UInt32   counter;
  Int32    size;
  char     name8[8];

  ftype = GetFileType(raw_filename);
  file = fopen(wad_filename, "wb");
  raw  = fopen(raw_filename, "rb");

  for (entry = MasterDir; entry; entry = entry->next)
    if (!strnicmp(entry->dir.name, entryname, 8))
      break;
  if (entry != NULL)
    {
      strncpy(name8, entryname, 8);
      WriteBytes(file, "PWAD", 4L);     /* PWAD file */
      counter = 1L;
      WriteInt32(file, &counter);       /* 1 entry */
      counter = 12L;
      WriteInt32(file, &counter);
      counter = 28L;
      WriteInt32(file, &counter);
      if (fseek(raw, 0L, SEEK_END) != 0)
        {
          printf("[Error reading from raw file.]\n");
          goto EXIT;
        }
      size = ftell(raw);
      if (size < 0L)
        {
          printf("[Error reading from raw file.]\n");
          goto EXIT;
        }
      if (fseek(raw, 0L, SEEK_SET) != 0)
        {
          printf("[Error reading from raw file.]\n");
          goto EXIT;
        }
      WriteInt32(file, &size);
      WriteBytes(file, name8, 8L);

      switch (ftype)
        {
        case FTYPE_AU:
          AUToWad(file, raw);
#ifdef NEVER_DEFINED
          {
             FILE *temp_file;

             UInt8 far * buffer;
             Int32 buf_size, speed;

             /* read file and store in buffer */
             buffer = ReadAU(raw, &buf_size, &speed);
             if (buffer == NULL)
             {
               break;
             }
             temp_file = fopen("testau.au", "wb");
             WriteAU(temp_file, buf_size, speed, buffer);


             temp_file = fopen("testau.voc", "wb");
             WriteVOC(temp_file, buf_size, speed, buffer);


             temp_file = fopen("testau.wav", "wb");
             WriteWAV(temp_file, buf_size, speed, buffer);


             FreeFarMemory(buffer);
             fclose(temp_file);
          }
#endif
          break;
        case FTYPE_BMP:
          {
             FILE *temp_file;

             UInt8 far * buffer;
             Int16 x_size, y_size;

             MDirPtr dir;
             UInt8 doompal[768];
             struct PIXEL huge * doompalette;


             dir = FindMasterDir(MasterDir, "PLAYPAL");
             BasicWadSeek(dir->wadfile, dir->dir.start);
             BasicWadRead(dir->wadfile, doompal, 768L);

             /* initialize invisible color */
             /* if invisible color is not correct in bmp file
                than invisible color will NOT be BLACK */
             COLinit(0, 255, 255, doompal, 768);

             doompalette =  COLdoomPalet();


             /* read bmp file and store in buffer */
             buffer = ReadBMP(raw, &x_size, &y_size);
             if (buffer == NULL)
             {
               COLfree();
               break;
             }

             InitGfx();
             DrawScreenBitmap(0, 0, x_size, y_size, buffer);
             WaitForKey();
             TermGfx();
#ifdef NEVER_DEFINED
             temp_file = fopen("testbmp.bmp", "wb");
             WriteBMP(temp_file, x_size, y_size, buffer, doompalette);
             fclose(temp_file);

             temp_file = fopen("testbmp.gif", "wb");
             WriteGIF(temp_file, x_size, y_size, buffer, doompalette);
             fclose(temp_file);

             temp_file = fopen("testbmp.pcx", "wb");
             WritePCX(temp_file, x_size, y_size, buffer, doompalette);
             fclose(temp_file);


             temp_file = fopen("testbmp.ppm", "wb");
             WritePPM(temp_file, x_size, y_size, buffer, doompalette);
             fclose(temp_file);
#endif

             FreeFarMemory(buffer);
             COLfree();
          }

          break;
        case FTYPE_GIF:
          {
             FILE *temp_file;

             UInt8 far * buffer;
             Int16 x_size, y_size;

             MDirPtr dir;
             UInt8 doompal[768];
             struct PIXEL huge * doompalette;

             /* find DOOM palette and store in dpal */
             dir = FindMasterDir(MasterDir, "PLAYPAL");
             BasicWadSeek(dir->wadfile, dir->dir.start);
             BasicWadRead(dir->wadfile, doompal, 768L);

             /* initialize invisible color */
             /* if invisible color is not correct in gif file
                than invisible color will NOT be BLACK */
             COLinit(0, 255, 255, doompal, 768);

             doompalette =  COLdoomPalet();

             /* read gif file and store in buffer */
             buffer = ReadGIF(raw, &x_size, &y_size);
             if (buffer == NULL)
             {
               COLfree();
               break;
             }
             InitGfx();
             DrawScreenBitmap(0, 0, x_size, y_size, buffer);
             WaitForKey();
             TermGfx();
#ifdef NEVER_DEFINED
             temp_file = fopen("testgif.bmp", "wb");
             WriteBMP(temp_file, x_size, y_size, buffer, doompalette);
             fclose(temp_file);

             temp_file = fopen("testgif.gif", "wb");
             WriteGIF(temp_file, x_size, y_size, buffer, doompalette);
             fclose(temp_file);

             temp_file = fopen("testgif.pcx", "wb");
             WritePCX(temp_file, x_size, y_size, buffer, doompalette);
             fclose(temp_file);


             temp_file = fopen("testgif.ppm", "wb");
             WritePPM(temp_file, x_size, y_size, buffer, doompalette);
             fclose(temp_file);
#endif

             FreeFarMemory(buffer);
             COLfree();
          }
          break;
        case FTYPE_LMP:
          CopyBytes(file, raw, size);
          break;
        case FTYPE_MIDI:
          printf("[Filter not implemented yet.  Sorry!]\n");
          break;
        case FTYPE_PCX:
          {
             FILE *temp_file;

             UInt8 far * buffer;
             Int16 x_size, y_size;

             MDirPtr dir;
             UInt8 doompal[768];
             struct PIXEL huge * doompalette;


             dir = FindMasterDir(MasterDir, "PLAYPAL");
             BasicWadSeek(dir->wadfile, dir->dir.start);
             BasicWadRead(dir->wadfile, doompal, 768L);

             /* initialize invisible color */
             /* if invisible color is not correct in pcx file
                than invisible color will NOT be BLACK */
             COLinit(0, 255, 255, doompal, 768);

             doompalette =  COLdoomPalet();


             /* read bmp file and store in buffer */
             buffer = ReadPCX(raw, &x_size, &y_size);
             if (buffer == NULL)
             {
               COLfree();
               break;
             }


             InitGfx();
             DrawScreenBitmap(0, 0, x_size, y_size, buffer);
             WaitForKey();
             TermGfx();
#ifdef NEVER_DEFINED
             temp_file = fopen("testpcx.bmp", "wb");
             WriteBMP(temp_file, x_size, y_size, buffer, doompalette);
             fclose(temp_file);

             temp_file = fopen("testpcx.gif", "wb");
             WriteGIF(temp_file, x_size, y_size, buffer, doompalette);
             fclose(temp_file);

             temp_file = fopen("testpcx.pcx", "wb");
             WritePCX(temp_file, x_size, y_size, buffer, doompalette);
             fclose(temp_file);


             temp_file = fopen("testpcx.ppm", "wb");
             WritePPM(temp_file, x_size, y_size, buffer, doompalette);
             fclose(temp_file);
#endif
             FreeFarMemory(buffer);
             COLfree();
          }
          break;
        case FTYPE_PPM:
          {
             FILE *temp_file;

             UInt8 far * buffer;
             Int16 x_size, y_size;

             MDirPtr dir;
             UInt8 doompal[768];
             struct PIXEL huge * doompalette;


             dir = FindMasterDir(MasterDir, "PLAYPAL");
             BasicWadSeek(dir->wadfile, dir->dir.start);
             BasicWadRead(dir->wadfile, doompal, 768L);

             /* initialize invisible color */
             /* if invisible color is not correct in ppm file
                than invisible color will NOT be BLACK */
             COLinit(0, 255, 255, doompal, 768);

             doompalette =  COLdoomPalet();


             /* read bmp file and store in buffer */
             buffer = ReadPPM(raw, &x_size, &y_size);
             if (buffer == NULL)
             {
               COLfree();
               break;
             }

             InitGfx();
             DrawScreenBitmap(0, 0, x_size, y_size, buffer);
             WaitForKey();
             TermGfx();
#ifdef NEVER_DEFINED
             temp_file = fopen("testppm.bmp", "wb");
             WriteBMP(temp_file, x_size, y_size, buffer, doompalette);
             fclose(temp_file);

             temp_file = fopen("testppm.gif", "wb");
             WriteGIF(temp_file, x_size, y_size, buffer, doompalette);
             fclose(temp_file);

             temp_file = fopen("testppm.pcx", "wb");
             WritePCX(temp_file, x_size, y_size, buffer, doompalette);
             fclose(temp_file);


             temp_file = fopen("testppm.ppm", "wb");
             WritePPM(temp_file, x_size, y_size, buffer, doompalette);
             fclose(temp_file);
#endif

             FreeFarMemory(buffer);
             COLfree();
          }
          break;
        case FTYPE_VOC:
          VocToWad(file, raw);
#ifdef NEVER_DEFINED
          {
             FILE *temp_file;

             UInt8 far * buffer;
             Int32 buf_size, speed;

             /* read file and store in buffer */
             buffer = ReadVOC(raw, &buf_size, &speed);
             if (buffer == NULL)
             {
               break;
             }


             temp_file = fopen("testvoc.au", "wb");
             WriteAU(temp_file, buf_size, 11025, buffer);


             temp_file = fopen("testvoc.voc", "wb");
             WriteVOC(temp_file, buf_size, speed, buffer);


             temp_file = fopen("testvoc.wav", "wb");
             WriteWAV(temp_file, buf_size, speed, buffer);

             FreeFarMemory(buffer);
             fclose(temp_file);
          }
#endif
          break;
        case FTYPE_WAV:
          WavToWad(file, raw);
#ifdef NEVER_DEFINED
          {
             FILE *temp_file;

             UInt8 far * buffer;
             Int32 buf_size, speed;

             /* read file and store in buffer */
             buffer = ReadWAV(raw, &buf_size, &speed);
             if (buffer == NULL)
             {
               break;
             }


             temp_file = fopen("testwav.au", "wb");
             WriteAU(temp_file, buf_size, speed, buffer);


             temp_file = fopen("testwav.voc", "wb");
             WriteVOC(temp_file, buf_size, speed, buffer);


             temp_file = fopen("testwav.wav", "wb");
             WriteWAV(temp_file, buf_size, speed, buffer);

             FreeFarMemory(buffer);
             fclose(temp_file);
          }
#endif
          break;
        default:
          printf("[Unknown format (file extension not recognised).]\n");
          break;
        }
    }
  else
    {
      printf("[Entry not in master directory]\n");
    }

EXIT:
  fclose(file);
  fclose(raw);
}


/* end of file */
