/*----------------------------------------------------------------------------*
 | This file is part of DEU (Doom Editing Utilities), created by the DEU team:|
 | Raphael Quinet, Brendon Wyber, Ted Vessenes and others.  See README.1ST or |
 | the "about" dialog box for full credits.                                   |
 |                                                                            |
 | DEU is an open project: if you think that you can contribute, please join  |
 | the DEU team.  You will be credited for any code (or ideas) included in    |
 | the next version of the program.                                           |
 |                                                                            |
 | If you want to make any modifications and re-distribute them on your own,  |
 | you must follow the conditions of the DEU license.  Read the file LICENSE  |
 | in this directory or README.1ST in the top directory.  If do not have a    |
 | copy of these files, you can request them from any member of the DEU team, |
 | or by mail: Raphael Quinet, Rue des Martyrs 9, B-4550 Nandrin (Belgium).   |
 |                                                                            |
 | This program comes with absolutely no warranty.  Use it at your own risks! |
 *----------------------------------------------------------------------------*

 DEU.H - Main defines for DEU.

*/

/*
   Note: I assume that this file is included first in all *.C files, so
         there is no need to include it in other *.H files.
*/

#ifndef _DEU_H_
#define _DEU_H_

#define DEU_VER_NUM             "5.30 BETA 9x"  /* the version number */


/*
   Common includes (for all systems/compilers).
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <ctype.h>


/*
   Compiler- and OS-specific includes and definitions.
*/

#if !defined(DEU_UNIX) && !defined(DEU_DOS)
/* if the right symbols have not been passed to the compiler from the
   Makefile or project file, then we have to guess them.
*/
#if defined(__TURBOC__)
/* if DEU is compiled with Borland C / Turbo C */
#define DEU_OSCOMPILER          "BC"
#define DEU_DOS                 /* mess-DOS */
#define DEU_GFX_BGI             /* BGI graphics */

#elif defined(__GO32__)
/* if DEU is compiled with DJGPP (GCC + GO32) */
#define DEU_OSCOMPILER          "GCC"
#define DEU_DOS                 /* mess-DOS */
#define DEU_GFX_GRX             /* GRX graphics */
#define far                     /* no far pointers */
#define huge                    /* no huge pointers */

#elif defined(__sgi)
/* if DEU is compiled on an SGI (compiler doesn't matter) */
#define DEU_OSCOMPILER          "SGI"
#define DEU_UNIX                /* UNIX operating system */
#define DEU_GFX_SGI             /* SGI graphics (GL) */
#define far                     /* no far pointers */
#define huge                    /* no huge pointers */
#define FAT_ENDIAN              /* MSB last */

#elif defined(__UNIX__) || defined(__unix)
/* if DEU is compiled on some UNIX system (compiler doesn't matter) */
#define DEU_OSCOMPILER          "X11"
#define DEU_UNIX                /* UNIX operating system */
#define DEU_GFX_X11             /* X Window graphics (X11) */
#define far                     /* no far pointers */
#define huge                    /* no huge pointers */
/*! more big endian systems should be added here... AIX? HP? */
#if defined(_BIG_ENDIAN) || defined(__sun)
#define FAT_ENDIAN              /* MSB last */
#endif

#else
#error Cannot guess your operating system type.  Check the Makefile.
#endif
#endif /* !DEU_UNIX && !DEU_DOS */

#ifndef DEU_OSCOMPILER
#define DEU_OSCOMPILER          "(Unknown)"
#endif

/* the full version number (including OS or compiler) */
#define DEU_VERSION             DEU_VER_NUM " " DEU_OSCOMPILER



/*
   Some useful typedefs for integer types.  The default "int" type should
   be used when the number of bits doesn't matter or when using va_arg with
   integer constants.  Never use "long" or "short" directly in the code,
   because the number of bits may vary (e.g. DEC Alpha: long = 64 bits).
   Always use these typedefs instead.  They will be modified if DEU is
   ported to "exotic" systems.  This is easier than having to modify all
   source files...
*/

typedef char           Int8;
typedef short          Int16;
typedef long           Int32;

typedef unsigned char  UInt8;
typedef unsigned short UInt16;
typedef unsigned long  UInt32;


/*
   The directory structure is the structure used by DOOM to order the
   data in its WAD files.
*/

typedef struct Directory /*! not huge */ *DirPtr;
struct Directory
{
   UInt32 start;                /* offset to start of data */
   UInt32 size;                 /* byte size of data */
   char   name[8];              /* name of data block */
};



/*
   The wad file pointer structure is used for holding the information
   on the wad files in a linked list.

   The first wad file is the main wad file.  The rest are patches.
*/
/*! The main wad file should be the _last_ one.  See comment for master dir*/
/*! The WAD files should not be open all of the time.  The "fileinfo" field
    will be removed as soon as the WAD loading routines are modified. */
typedef struct WadFileInfo /*! not huge */ *WadPtr;
struct WadFileInfo
{
   WadPtr  next;                /* next file in linked list */
   char   *filename;            /* name of the wad file */
   FILE   *fileinfo;            /* C file stream information (open file) */
   char    type[4];             /* type of wad file (IWAD or PWAD) */
   UInt32  dirsize;             /* directory size of WAD */
   UInt32  dirstart;            /* offset to start of directory */
   DirPtr  directory;           /* array of directory information */
};



/*! The master directory should disappear now.  It will be useless if
    we want to be able to edit WAD files as separate documents.  The current
    functions that have to search for an entry in the master directory (e.g.
    when loading textures or sprites) could instead search in all PWADs first,
    then search in the IWAD.  The semantics would be the same, and it would
    be easier to open and close WAD files without interference with the IWAD.
*/

/*
   The master directory structure is used to build a complete directory
   of all the data blocks from all the various wad files.
*/

typedef struct MasterDirectory /*! not huge */ *MDirPtr;
struct MasterDirectory
{
   MDirPtr          next;       /* next in list */
   WadPtr           wadfile;    /* file of origin */
   struct Directory dir;        /* directory data */
};



/*
   The selection list is used when more than one object is selected.
*/

typedef struct SelectionList *SelPtr;
struct SelectionList
{
   SelPtr next;                 /* next in list */
   Int16  objnum;               /* object number */
};


/*
   Syntactic sugar...
*/

typedef int Bool;               /* Boolean data: true or false */


/*
   The macros and constants
*/

/* name of the configuration file */
#define DEU_CONFIG_FILE         "deu.ini"

/* name of the log file (debug mode) */
#define DEU_LOG_FILE            "deu.log"

/* convert screen coordinates to map coordinates */
#define MAPX(x)                (OrigX + (Int16) (((x) - ScrCenterX) / Scale))
#define MAPY(y)                (OrigY + (Int16) ((ScrCenterY - (y)) / Scale))

/* convert map coordinates to screen coordinates */
#define SCREENX(x)             (ScrCenterX + (Int16) (((x) - OrigX) * Scale))
#define SCREENY(y)             (ScrCenterY + (Int16) ((OrigY - (y)) * Scale))

/* get the maximum and minimum of two integers */
#define MAX(a, b)              (((a) > (b)) ? (a) : (b))
#define MIN(a, b)              (((a) < (b)) ? (a) : (b))

/* get a random number */
#ifdef __TURBOC__
#define RANDOM(x)              (random(x))
#define RANDOMIZE()             randomize()
#else
#define RANDOM(x)              (random() % (x))
#define RANDOMIZE()             srand((int) time(NULL))
#endif

/* object types */
#define OBJ_THINGS              1
#define OBJ_LINEDEFS            2
#define OBJ_SIDEDEFS            3
#define OBJ_VERTEXES            4
#define OBJ_SEGS                5
#define OBJ_SSECTORS            6
#define OBJ_NODES               7
#define OBJ_SECTORS             8
#define OBJ_REJECT              9
#define OBJ_BLOCKMAP            10

/* boolean constants */

#ifndef TRUE
#define TRUE                    1
#define FALSE                   0
#endif

#endif /* _DEU_H_ */
/* end of file */
