/*
 *	Name:		MUS Playing kernel
 *	Project:	MUS File Player Library
 *	Version:	1.65
 *	Author:		Vladimir Arnost (QA-Software)
 *	Last revision:	Aug-28-1995
 *	Compiler:	Borland C++ 3.1, Watcom C/C++ 10.0
 *
 */

/*
 * Revision History:
 *
 *	Aug-8-1994	V1.00	V.Arnost
 *		Written from scratch
 *	Aug-9-1994	V1.10	V.Arnost
 *		Some minor changes to improve sound quality. Tried to add
 *		stereo sound capabilities, but failed to -- my SB Pro refuses
 *		to switch to stereo mode.
 *	Aug-13-1994	V1.20	V.Arnost
 *		Stereo sound fixed. Now works also with Sound Blaster Pro II
 *		(chip OPL3 -- gives 18 "stereo" (ahem) channels).
 *		Changed code to handle properly notes without volume.
 *		(Uses previous volume on given channel.)
 *		Added cyclic channel usage to avoid annoying clicking noise.
 *	Aug-17-1994	V1.30	V.Arnost
 *		Completely rewritten time synchronization. Now the player runs
 *		on IRQ 8 (RTC Clock - 1024 Hz).
 *	Aug-28-1994	V1.40	V.Arnost
 *		Added Adlib and SB Pro II detection.
 *		Fixed bug that caused high part of 32-bit registers (EAX,EBX...)
 *		to be corrupted.
 *	Oct-30-1994	V1.50	V.Arnost
 *		Tidied up the source code
 *		Added C key - invoke COMMAND.COM
 *		Added RTC timer interrupt flag check (0000:04A0)
 *		Added BLASTER environment variable parsing
 *		FIRST PUBLIC RELEASE
 *	Apr-16-1995	V1.60	V.Arnost
 *		Moved into separate source file MUSLIB.C
 *	May-01-1995	V1.61	V.Arnost
 *		Added system timer (IRQ 0) support
 *	Jul-12-1995	V1.62	V.Arnost
 *		OPL2/OPL3-specific code moved to module MLOPL.C
 *		Module MUSLIB.C renamed to MLKERNEL.C
 *	Aug-04-1995	V1.63	V.Arnost
 *		Fixed stack-related bug occuring in big-code models in Watcom C
 *	Aug-16-1995	V1.64	V.Arnost
 *		Stack size changed from 256 to 512 words because of stack
 *		underflows caused by AWE32 driver
 *	Aug-28-1995	V1.65	V.Arnost
 *		Fixed a serious bug that caused the player to generate an
 *		exception in AWE32 driver under DOS/4GW: Register ES contained
 *		garbage instead of DGROUP. The compiler-generated prolog of
 *		interrupt handlers doesn't set ES register at all, thus any
 *		STOS/MOVS/SCAS/CMPS instruction used within the int. handler
 *		crashes the program.
 */

#if defined(__WINDOWS__) || defined(_Windows)
  #define STRICT
  #include <windows.h>
  #include <mmsystem.h>
#else
  #include <dos.h>
  #include <mem.h>
  #include <malloc.h>
#endif
#include "muslib.h"


char MLversion[] = "MUS Lib V"MLVERSIONSTR;
char MLcopyright[] = "Copyright (c) 1994,1995 QA-Software";

#ifndef __WINDOWS__

#if defined(__WATCOMC__) && defined(FARDATAPTR)
  #define STATIC_STACK
#endif

#define STACK_SIZE 0x200		/* 512 words of stack */

static void (interrupt far *oldint70h)(void);
static void (interrupt far *oldint08h)(void);

#ifdef STATIC_STACK
static uint	timerstack[STACK_SIZE];
#else
static uint	*timerstack = NULL;
#endif

static uint far *timerstackend = NULL;
static volatile uint far *timersavestack = NULL;

static uint	timersOn = 0;
static uint	timerMode = TIMER_RTC1024;

#else /* __WINDOWS__ */

/*HINSTANCE*/ uint MLinstance = 0;

static uint	timersOn = 0;
static uint	timerMode = TIMER_WIN70;

#endif /* __WINDOWS__ */

volatile ulong	MLtime;
volatile uint	playingChannels = 0;


/* Program */
static void playNote(struct musicBlock *mus, uint channel, uchar note, int volume)
{
    if (mus->channelMask & (1 << channel))
	mus->driver->playNote(mus, channel, note, volume);
}

static void releaseNote(struct musicBlock *mus, uint channel, uchar note)
{
    mus->driver->releaseNote(mus, channel, note);
}

static void pitchWheel(struct musicBlock *mus, uint channel, int pitch)
{
    mus->driver->pitchWheel(mus, channel, pitch);
}

static void changeControl(struct musicBlock *mus, uint channel, uchar controller, int value)
{
    mus->driver->changeControl(mus, channel, controller, value);
}

static int playTick(struct musicBlock *mus)
{
    for(;;)
    {
	uint data = MEMgetchar(&mus->score);
	uint command = (data >> 4) & 7;
	uint channel = data & 0x0F;
	uint last = data & 0x80;

	switch (command) {
	    case 0:	// release note
		mus->playingcount--;
		releaseNote(mus, channel, MEMgetchar(&mus->score));
		break;
	    case 1: {	// play note
		uchar note = MEMgetchar(&mus->score);
		mus->playingcount++;
		if (note & 0x80)	// note with volume
		    playNote(mus, channel, note & 0x7F, MEMgetchar(&mus->score));
		else
		    playNote(mus, channel, note, -1);
		} break;
	    case 2:	// pitch wheel
		pitchWheel(mus, channel, MEMgetchar(&mus->score) - 0x80);
		break;
	    case 3:	// system event (valueless controller)
		changeControl(mus, channel, MEMgetchar(&mus->score), 0);
		break;
	    case 4: {	// change control
		uchar ctrl = MEMgetchar(&mus->score);
		uchar value = MEMgetchar(&mus->score);
		changeControl(mus, channel, ctrl, value);
		} break;
	    case 6:	// end
		return -1;
	    case 5:	// ???
	    case 7:	// ???
		break;
	}
	if (last)
	    break;
    }
    return 0;
}

static ulong delayTicks(struct musicBlock *mus)
{
    ulong time = 0;
    uchar data;

    do {
	time <<= 7;
	time += (data = MEMgetchar(&mus->score)) & 0x7F;
    } while (data & 0x80);

    return time;
}


/*
 * Perform one timer tick using 140 Hz clock
 */
#ifdef __WINDOWS__
  #define PLAYFAST 0
#elif __386__
  #define PLAYFAST (*((BYTE *)0x00000418) & 4) // Alt-SysRq pressed
#else
  #define PLAYFAST (*(BYTE far *)MK_FP(0x0000, 0x0418) & 4) // Alt-SysRq pressed
#endif /* __386__ */

static void playMusic140Hz(void)
{
    uint i;
    for (i = 0; i < MAXMUSBLOCK; i++)
    {
	register struct musicBlock *mus = MLmusicBlocks[i];
	if (mus && mus->state == ST_PLAYING)
	{
	    if (!mus->ticks || PLAYFAST)
	    {
		if (playTick(mus))
		{					// end of song
		    if (mus->loopcount &&
		       (mus->loopcount == -1U || --mus->loopcount)) // -1: loop forever
			MEMrewind(&mus->score);
		    else
			mus->state = ST_STOPPED;
		    continue;
		}
		mus->time += mus->ticks = delayTicks(mus);
	    }
	    mus->ticks--;
	}
    }
    MLtime++;
}

#ifndef __WINDOWS__

/*
 * Perform one timer tick using 18.2 Hz clock
 */
#define RATE140HZ	8523	// 1193180 / 140

static void playMusic18_2Hz(void)
{
    static DWORD ticks = 0;

    while ( !((WORD *)&ticks)[1] )		/* while (ticks < 65536) */
    {
	playMusic140Hz();
	ticks += RATE140HZ;
    }
    ((WORD *)&ticks)[1] = 0;			/* ticks &= 0xFFFF */
}


#ifdef __WATCOMC__

void CMOSwrite(BYTE reg, BYTE value);
BYTE CMOSread(BYTE reg);
void EOIsignal1(void);				/* end-of-interrupt signal */
void EOIsignal2(void);
void setEStoDS(void);
void enterStack(void);
void restoreStack(void);

#define DELAY 0xEB 0x00  			/* jmp $+2 */

#pragma aux CMOSwrite =		\
	"out	70h,al"		\
	DELAY			\
	DELAY			\
	"mov	al,ah"		\
	"out	71h,al"		\
	parm [AL][AH] nomemory	\
	modify exact [AL] nomemory;
		
#pragma aux CMOSread =		\
	"out	70h,al"		\
	DELAY			\
	DELAY			\
	"in	al,71h"		\
	parm [AL] nomemory	\
	modify exact [AL] nomemory;	

#pragma aux EOIsignal1 =	\
	"mov	al,20h"		\
	"out	20h,al"		\
	modify exact [AL] nomemory;

#pragma aux EOIsignal2 =	\
	"mov	al,20h"		\
	"out	0A0h,al"	\
	DELAY			\
	"out	20h,al"		\
	modify exact [AL] nomemory;

#pragma aux setEStoDS =	\
	"mov	ax,ds"		\
	"mov	es,ax"		\
	modify exact [AX] nomemory;

#ifdef __386__
#pragma aux enterStack =	\
	"mov	WORD PTR timersavestack[4],ss"	\
	"mov	DWORD PTR timersavestack[0],esp" \
	"lss	esp,timerstackend"		\
	"pushad";
#else
#pragma aux enterStack =	\
	"mov	WORD PTR timersavestack[2],ss"	\
	"mov	WORD PTR timersavestack[0],sp"	\
	"lss	sp,timerstackend"		\
	"pushad";
#endif

#ifdef __386__
#pragma aux restoreStack =	\
	"popad"			\
	"lss	esp,timersavestack";
#else
#pragma aux restoreStack =	\
	"popad"			\
	"lss	sp,timersavestack";
#endif

#else // __WATCOMC__

#define DELAY db 0EBh,00h  			/* jmp $+2 */

static void FASTCALL CMOSwrite(BYTE reg, BYTE value)
{
    asm {
	mov	al,reg
	out	70h,al
	DELAY
	DELAY
	mov	al,value
	out	71h,al
    }
}

static int FASTCALL CMOSread(BYTE reg)
{
    asm {
	mov	al,reg
	out	70h,al
	DELAY
	DELAY
	in	al,71h
	xor	ah,ah
    }
    return _AX;
}

#define EOIsignal1()		\
    asm {			\
	mov	al,20h;		\
	out	20h,al		\
    }

#define EOIsignal2()		\
    asm {			\
	mov	al,20h;		\
	out	0A0h,al;	\
	DELAY;			\
	out	20h,al		\
    }

#define setEStoDS()		\
    asm {			\
	mov	ax,ds;		\
	mov	es,ax		\
    }

#define enterStack()		\
    asm {			\
	mov	WORD PTR timersavestack[2],ss;	\
	mov	WORD PTR timersavestack[0],sp;	\
	mov	ss,WORD PTR timerstackend[2];	\
	mov	sp,WORD PTR timerstackend[0];	\
	db 66h; pusha	/* *386* pushad */	\
    }

#define restoreStack()		\
    asm {			\
	db 66h; popa;	/* *386* popad */	\
	mov	ss,WORD PTR timersavestack[2];	\
	mov	sp,WORD PTR timersavestack[0]	\
    }

#endif // __WATCOMC__


static void interrupt newint08h_handler(void)
{
    static volatile uint count = 0;
    static volatile ulong ticks = 0;

    setEStoDS();
    if (!count)					/* reentrancy safeguard */
    {
	count++;
	enterStack();				/* swap stacks */
	if (timerMode == TIMER_CNT18_2)
	    playMusic18_2Hz();
	else
	    playMusic140Hz();
	restoreStack();
	count--;
    }

    if (timerMode == TIMER_CNT18_2)
	(*oldint08h)();				/* invoke original handler */
    else {
	ticks += RATE140HZ;
	if ( ((WORD *)&ticks)[1] )		/* if (ticks >= 65536L) */
	{
	    ((WORD *)&ticks)[1] = 0;		/* ticks &= 0xFFFF */
	    (*oldint08h)();
	} else
	    EOIsignal1();			/* end-of-interrupt signal */
    }
}

#define RTC_REG		0x0B
#define RTC_STATUS	0x0C
#define RTC_MASK	0x40

static void interrupt newint70h_handler(void)
{
    static volatile uint count = 0;
    static volatile uint ticks = 0;

    setEStoDS();
    if ( !(CMOSread(RTC_STATUS) & RTC_MASK) )	/* not periodic interrupt */
    {
	(*oldint70h)();				/* invoke original handler */
	return;
    }

    if ( (ticks += 140) >= 1024 )
    {
	ticks -= 1024;
	if (!count)				/* reentrancy safeguard */
	{
	    count++;
	    enterStack();			/* swap stacks */
	    playMusic140Hz();
	    restoreStack();
	    count--;
	}
    }

    EOIsignal2();				/* end-of-interrupt signal */
}

#ifdef __WATCOMC__

void PITset(WORD value);
void PITreset(void);

#pragma aux PITset =		\
	"cli"			\
	"mov	al,36h"		\
	"out	43h,al"		\
	"mov	ax,dx"		\
	"out	40h,al"		\
	"xchg	al,ah"		\
	"out	40h,al"		\
	"sti"			\
	parm [DX]		\
	modify exact [AX] nomemory;
		
#pragma aux PITreset =		\
	"cli"			\
	"mov	al,36h"		\
	"out	43h,al"		\
	"xor	al,al"		\
	"out	40h,al"		\
	"out	40h,al"		\
	"sti"			\
	modify exact [AL] nomemory;
		
#else // __WATCOMC__

#define PITset(value)		\
    asm {			\
	cli;			\
	mov	al,36h;		\
	out	43h,al;		\
	mov	ax,value;	\
	out	40h,al;		\
	xchg	al,ah;		\
	out	40h,al;		\
	sti			\
    }

#define PITreset()		\
    asm {			\
	cli;			\
	mov	al,36h;		\
	out	43h,al;		\
	xor	al,al;		\
	out	40h,al;		\
	out	40h,al;		\
	sti			\
    }

#endif // __WATCOMC__

static int SetupTimer1(uint divisor)
{
    if (timersOn & 1)			/* already initialized? */
	return 0;
#ifndef STATIC_STACK
    if ( (timerstack = calloc(STACK_SIZE, sizeof(int))) == NULL)
	return -1;
#endif
    timerstackend = &timerstack[STACK_SIZE];
    oldint08h = _dos_getvect(0x08);
    _dos_setvect(0x08, newint08h_handler);
    PITset(divisor);
    timersOn |= 1;
    return 0;
}

static int ShutdownTimer1(uint restoreRate)
{
    if (!(timersOn & 1))		/* is initialized? */
	return -1;
    if (restoreRate)
	PITreset();
    _dos_setvect(0x08, oldint08h);
#ifndef STATIC_STACK
    free(timerstack);
#endif
    timersOn &= ~1;
    return 0;
}

#ifdef __WATCOMC__

void IRQ8enable(void);
void IRQ8disable(void);

#pragma aux IRQ8enable =	\
	"in	al,0A1h"	\
	"and	al,0FEh"	\
	"out	0A1h,al"	\
	modify exact [AL] nomemory;
		
#pragma aux IRQ8disable =	\
	"in	al,0A1h"	\
	"or	al,1"		\
	"out	0A1h,al"	\
	modify exact [AL] nomemory;
		
#else // __WATCOMC__

#define IRQ8enable()		\
    asm {			\
	in	al,0A1h;	\
	and	al,not 1;	\
	out	0A1h,al		\
    }

#define IRQ8disable()		\
    asm {			\
	in	al,0A1h;	\
	or	al,1;		\
	out	0A1h,al		\
    }

#endif // __WATCOMC__

#ifdef __386__
#define RTC_TIMER_FLAG	*((BYTE *)0x0000004A0)
#else
#define RTC_TIMER_FLAG	*(BYTE far *)MK_FP(0x0040, 0x00A0)
#endif /* __386__ */

static int SetupTimer2(void)
{
    if (timersOn & 2)			/* already initialized? */
	return 0;
    if (RTC_TIMER_FLAG)				// is the timer busy?
	return -1;
#ifndef STATIC_STACK
    if ( (timerstack = calloc(STACK_SIZE, sizeof(int))) == NULL)
	return -1;
#endif
    timerstackend = &timerstack[STACK_SIZE];
    RTC_TIMER_FLAG = 1;				// mark the timer as busy
    oldint70h = _dos_getvect(0x70);
    _dos_setvect(0x70, newint70h_handler);
    CMOSwrite(RTC_REG, CMOSread(RTC_REG) | RTC_MASK); // enable periodic interrupt
    IRQ8enable();
    timersOn |= 2;
    return 0;
}

static int ShutdownTimer2(void)
{
    if (!(timersOn & 2))		/* is initialized? */
	return -1;
    CMOSwrite(RTC_REG, CMOSread(RTC_REG) & ~RTC_MASK); // disable periodic interrupt
    IRQ8disable();
    _dos_setvect(0x70, oldint70h);
#ifndef STATIC_STACK
    free(timerstack);
#endif
    RTC_TIMER_FLAG = 0;				// mark the timer as unused
    timersOn &= ~2;
    return 0;
}

/*
int MLdetectHardware(void)
{
/ *    DetectBlaster(&SBProPort, NULL, NULL, NULL);
    if (!DetectAdlib(ADLIBPORT))
	return -1;
    return DetectSBProII(SBProPort); * /
    return OPLdetectHardware();
} */

int MLinitTimer(int mode)
{
    MLtime = 0;
    switch (timerMode = mode) {
	case TIMER_CNT18_2:
	    return SetupTimer1(0);
	case TIMER_CNT140:
	    return SetupTimer1(RATE140HZ);	// 1193180 / 140
	case TIMER_RTC1024:
	    return SetupTimer2();
	default:
	    return -1;
    }
}

int MLshutdownTimer(void)
{
    switch (timerMode) {
	case TIMER_CNT18_2:
	    return ShutdownTimer1(0);
	case TIMER_CNT140:
	    return ShutdownTimer1(1);
	case TIMER_RTC1024:
	    return ShutdownTimer2();
	default:
	    return -1;
    }
}

#else /* __WINDOWS__ */
/**************************************************************************/

/* static TIMERPROC lpfnMUSLibTimerProc; */
static LPTIMECALLBACK lpfnMUSLibTimerProc;
static UINT timerID;

#pragma argsused
/* void CALLBACK _export MUSLibTimerProc(HWND hwnd, UINT msg, UINT idTimer, DWORD dwTime) */
void CALLBACK _export MUSLibTimerProc(UINT wtimerID, UINT wMsg, DWORD dwUser, DWORD dw1, DWORD dw2)
{
    switch (timerMode) {
	case TIMER_WIN35:
	    playMusic140Hz();
	    playMusic140Hz();
	case TIMER_WIN70:
	    playMusic140Hz();
	case TIMER_WIN140:
	    playMusic140Hz();
    }
}

static int SetupTimer(uint period)
{
    if (timersOn & 0x100)		/* already initialized? */
	return 0;

/*    lpfnMUSLibTimerProc = (TIMERPROC)MakeProcInstance((FARPROC)MUSLibTimerProc,
						      (HINSTANCE)MLinstance);
    timerID = SetTimer(NULL, NULL, period, lpfnMUSLibTimerProc); */

    lpfnMUSLibTimerProc = (LPTIMECALLBACK)MakeProcInstance(
	(FARPROC)MUSLibTimerProc, (HINSTANCE)MLinstance);
    timerID = timeSetEvent(period, 5, lpfnMUSLibTimerProc, 0, TIME_PERIODIC);
    if (!timerID)
	return -1;

    timersOn |= 0x100;
    return 0;
}

static int ShutdownTimer(void)
{
    if (!(timersOn & 0x100))		/* is initialized? */
	return -1;

/*    KillTimer(NULL, timerID); */
    timeKillEvent(timerID);
    FreeProcInstance((FARPROC)lpfnMUSLibTimerProc);

    timersOn &= ~0x100;
    return 0;
}

int MLinitTimer(int mode)
{
    MLtime = 0;
    switch (timerMode = mode) {
	case TIMER_WIN35:
	    return SetupTimer(28);
	case TIMER_WIN70:
	    return SetupTimer(14);
	case TIMER_WIN140:
	    return SetupTimer(7);
	default:
	    return -1;
    }
}

int MLshutdownTimer(void)
{
    switch (timerMode) {
	case TIMER_WIN35:
	case TIMER_WIN70:
	case TIMER_WIN140:
	    return ShutdownTimer();
	default:
	    return -1;
    }
}

#endif /* __WINDOWS__ */
