//========================================================================
// Quick MUS->MID converter copyright (c) 1995 Sebastien Bacquet
// 11-25-95
// Compiled with Borland C++ 3.1
//========================================================================

//
// Modified for WinTex by O.Montanuy 28/11/95
// - changed char, long, int by portable Int8, Int16, Int32.
// - changed all pointers to huge pointers pInt8, pInt16, pInt32.
// - changed farmalloc and farfree by Malloc and Free (WinTex)
// - changed _fmemcpy by Memcpy (WinTex)
// - qmus2mid now returns a pointer to middata or NULL if error
// Tested on 28/11/95 as part of WinTex 4.1. Works perfectly.
//


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <alloc.h>

#include "lbwintex.h" /*type definitions*/
#include "lbcommon.h" /*Malloc, Free, Memcpy*/
#include "wmus2mid.h"

#include <io.h>


void midcpy( const pVoid src, UInt32 size, pUInt8 middata, pUInt32 midpos )
{
	//_fmemcpy( middata+ (*midpos), src, (size_t) size ) ;
	Memcpy(middata+ (*midpos), src, size );
	(*midpos) += size ;
}

// Little Endian
void midcpyrev( const pUInt32 ptr, UInt32 size, pUInt8 middata, pUInt32 midpos )
{
	UInt32 rev = 0;
	register i = 0 ;

	for( i = 0 ; i < size ; i++ )
		rev = (rev << 8) + (((*ptr) >> (i*8)) & 0xFF) ;

	midcpy( &rev, size, middata, midpos ) ;
}

// Write a byte in track number MIDItrack
Int16 TWriteByte( Int8 MIDItrack, Int8 byte, pTrack track, Int32 BufferSize )
{
	UInt32 pos ;

	pos = track[MIDItrack].current ;
	if( pos >= BufferSize ) return -1 ;
	track[MIDItrack].data[pos] = byte ;
	track[MIDItrack].current++ ;

	return 0 ;
}

// Write a variable-length quantity in track number tracknum
Int16 TWriteVarLen( Int16 tracknum, UInt32 value, pTrack track, Int32 BufferSize )
{
	register UInt32 buffer ;

	buffer = value & 0x7f ;
	while( (value >>= 7) )
	{
		buffer <<= 8 ;
		buffer |= 0x80 ;
		buffer += (value & 0x7f) ;
	}
	while( 1 )
	{
		if( TWriteByte( tracknum, (Int8)buffer, track, BufferSize ) )
			return -1 ;
		if( buffer & 0x80 )
			buffer >>= 8 ;
		else
			break;
	}

	return 0 ;
}

// Read MUS header
void ReadMUSheader( pMUSheader MUSh, const pUInt8 musdata )
{
	Memcpy( &(MUSh->ID), musdata, 4 ) ;
	Memcpy( &(MUSh->ScoreLength), musdata+4, 2 ) ;
	Memcpy( &(MUSh->ScoreStart), musdata+6, 2 ) ;
	Memcpy( &(MUSh->channels), musdata+8, 2 ) ;
	Memcpy( &(MUSh->SecChannels), musdata+10, 2 ) ;
	Memcpy( &(MUSh->InstrCnt), musdata+12, 2 ) ;
	Memcpy( &(MUSh->dummy), musdata+14, 2 ) ;
	MUSh->instruments = (pUInt16) Malloc(MUSh->InstrCnt*sizeof(UInt16));
	if(MUSh->instruments!=NULL)
	{
	  Memcpy( MUSh->instruments, musdata+16, 2*MUSh->InstrCnt ) ;
	}
}

// Write MIDI header in memory
void MWriteMIDheader( UInt16 ntrks, UInt16 division, pUInt8 middata, pUInt32 midpos )
{
	*midpos = 0 ;
	midcpy( "MThd\x00\x00\x00\x06\x00\x01", 10, middata, midpos ) ;
	midcpyrev( (pUInt32)&ntrks, 2, middata, midpos ) ;
	midcpyrev( (pUInt32)&division, 2, middata, midpos ) ;
}

// You don't care.
UInt8 last( Int8 event )
{
	return (event >> 7) ;
}

// You don't care.
UInt8 event_type( Int8 event )
{
	return ((event & 0x7F) >> 4) ;
}

// You don't care.
UInt8 channel( Int8 event )
{
	return (event & 0x0F) ;
}

// Write track number tracknum in memory
void MWriteTrack( Int16 tracknum, pTrack track, pUInt8 middata, pUInt32 midpos )
{
	UInt32 size ;

	size = (Int32) (track[tracknum].current+4) ;
	midcpy( "MTrk", 4, middata, midpos ) ;
	if( !tracknum ) size += 31 ;
	midcpyrev( &size, 4, middata, midpos ) ;
	if( !tracknum)
		midcpy( "\x00\xFF\x03\x1BQuick MUS->MID ! for WinTex", 31, middata, midpos ) ;
	midcpy( track[tracknum].data, track[tracknum].current, middata, midpos ) ;
	midcpy( "\x00\xFF\x2F\x00", 4, middata, midpos ) ;
}

// Write the first track (number 0) in memory
void MWriteFirstTrack( pUInt8 middata, pUInt32 midpos )
{
	Int32 size ;

	size = 43 ;
	midcpy( "MTrk", 4, middata, midpos ) ;
	midcpyrev( (pUInt32)&size, 4, middata, midpos ) ;
	midcpy( "\x00\xFF\x02\x16", 4, middata, midpos ) ;
	midcpy( "QMUS2MID (C) S.BACQUET", 22, middata, midpos ) ;
	midcpy( "\x00\xFF\x59\x02\x00\x00", 6, middata, midpos ) ;
	midcpy( "\x00\xFF\x51\x03\x09\xA3\x1A", 7, middata, midpos ) ;
	midcpy( "\x00\xFF\x2F\x00", 4, middata, midpos ) ;
}

// Read MUS time (similar to MIDI deltatime)
Int32 ReadTime( const pUInt8 musdata, pUInt32 muspos )
{
	register UInt32 time = 0 ;
	int byte ;

	do
	{
		byte = musdata[(*muspos)++] ;
		time = (time << 7) + (byte & 0x7F) ;
	} while( byte & 0x80 ) ;

	return time ;
}

// You don't care.
Int8 FirstChannelAvailable( pInt8 MUS2MIDchannel )
{
	int i ;
	Int8 old15 = MUS2MIDchannel[15], max = -1 ;

	MUS2MIDchannel[15] = -1 ;
	for( i = 0 ; i < 16 ; i++ )
		if( MUS2MIDchannel[i] > max ) max = MUS2MIDchannel[i] ;
	MUS2MIDchannel[15] = old15 ;

	return (max == 8 ? 10 : max+1) ;
}

// Free tracks from memory
void FreeTracks( pTrack track )
{
	int i ;

	for( i = 0 ; i < 16 ; i++ )
		if( track[i].data )
			Free( track[i].data ) ;
}


/***********************************************************************************
* MAIN FUNCTION                                                                    *
* Converts MUS data to MIDI data (MIDI file)                                       *
* Parameters :                                                                     *
*     musdata    = MUS data                                                        *
*     mussize    = size of MUS data                                                *
*     midsize    = size of MID data converted                                      *
*     division   = Ticks per Quater Note (default = 89)                            *
*     BufferSize = track buffers size (default = 64 KB)                            *
*                  In KB.                                                          *
*     nocomp     = No Compression (0 = compression, other = w/o compression)       *
* Returned value :                                                                 *
*     Pointer on MIDI data if no error                                             *
*     NULL if error                                                                *
************************************************************************************/

pInt8 qmus2mid(pInt32 midsize, const pInt8 musdata, Int32 mussize,  Int16 division, Int32 BufferSize, Int16 nocomp )
{
	struct Track track[16] ;
	Int16 TrackCnt = 0 ;
	UInt8 event, et, MUSchannel, MIDIchannel, MIDItrack, data, NewEvent ;
	Int16 i ;
	struct MUSheader MUSh ;
	UInt32 DeltaTime, muspos, midpos ;
	UInt8 MUS2MIDcontrol[15] = {
						0,    // Program change - not a MIDI control change
						0x00, // Bank select
						0x01, // Modulation pot
						0x07, // Volume
						0x0A, // Pan pot
						0x0B, // Expression pot
						0x5B, // Reverb depth
						0x5D, // Chorus depth
						0x40, // Sustain pedal
						0x43, // Soft pedal
						0x78, // All sounds off
						0x7B, // All notes off
						0x7E, // Mono
						0x7F, // Poly
						0x79  // Reset all controllers
					  }, MIDIchan2track[16] ;
	Int8 MUS2MIDchannel[16] ;
	Int32 size ;
	pUInt8 middata;

	ReadMUSheader( &MUSh, (pUInt8)musdata ) ;
	muspos = MUSh.ScoreStart ;
	if( MUSh.channels > 15 )    // <=> MUSchannels+drums > 16
		return NULL ; // TOOMANYCHAN

	for( i = 0 ; i < 16 ; i++ )
	{
		MUS2MIDchannel[i] = -1 ;
		track[i].current = 0 ;
		track[i].vel = 64 ;
		track[i].DeltaTime = 0 ;
		track[i].LastEvent = 0 ;
		track[i].data = NULL ;
	}
	if( BufferSize )
		BufferSize = ((UInt32) BufferSize) << 10 ; // converts
																 // KB -> B
	else
		BufferSize = 65536L ;  // 64 KB

	event = musdata[muspos++] ;
	et = event_type( event ) ;
	MUSchannel = channel( event ) ;
	while( (et != 6) && (muspos < mussize) )
	{
		if( MUS2MIDchannel[MUSchannel] == -1 )
		{
			MIDIchannel = MUS2MIDchannel[MUSchannel ] = (MUSchannel == 15 ? 9 : FirstChannelAvailable( MUS2MIDchannel)) ;
			MIDItrack   = MIDIchan2track[MIDIchannel] = TrackCnt++ ;
			track[MIDItrack].data = (pUInt8) Malloc( BufferSize );
			if( !(track[MIDItrack].data) )
			{
				FreeTracks( track ) ;
				return NULL; // MEMALLOC
			}
		}
		else
		{
			MIDIchannel = MUS2MIDchannel[MUSchannel] ;
			MIDItrack   = MIDIchan2track [MIDIchannel] ;
		}
		if( TWriteVarLen( MIDItrack, track[MIDItrack].DeltaTime, track, BufferSize ) )
		{
			FreeTracks( track ) ;
			return NULL ;
		}
		track[MIDItrack].DeltaTime = 0 ;
		switch( et )
		{
			case 0 :  // release note
				NewEvent = 0x90 | MIDIchannel ;
				if( (NewEvent != track[MIDItrack].LastEvent) || nocomp )
				{
					if( TWriteByte( MIDItrack, NewEvent, track, BufferSize ) )
					{
						FreeTracks( track ) ;
						return NULL ;
					}
					track[MIDItrack].LastEvent = NewEvent ;
				}
				data = musdata[muspos++] ;
				if( TWriteByte( MIDItrack, data, track, BufferSize ) )
				{
					FreeTracks( track ) ;
					return NULL ;
				}
				if( TWriteByte( MIDItrack, 0, track, BufferSize ) )
				{
					FreeTracks( track ) ;
					return NULL ;
				}
				break ;
			case 1 :  // note on
				NewEvent = 0x90 | MIDIchannel ;
				if( (NewEvent != track[MIDItrack].LastEvent) || nocomp )
				{
					if( TWriteByte( MIDItrack, NewEvent, track, BufferSize ) )
					{
						FreeTracks( track ) ;
						return NULL ;
					}
					track[MIDItrack].LastEvent = NewEvent ;
				}
				data = musdata[muspos++] ;
				if( TWriteByte( MIDItrack, data & 0x7F, track, BufferSize ) )
				{
					FreeTracks( track ) ;
					return NULL ;
				}
				if( data & 0x80 )
					track[MIDItrack].vel = musdata[muspos++] ;
				if( TWriteByte( MIDItrack, track[MIDItrack].vel, track, BufferSize ) )
				{
					FreeTracks( track ) ;
					return NULL ;
				}
				break ;
			case 2 :  // pitch wheel
				NewEvent = 0xE0 | MIDIchannel ;
				if( (NewEvent != track[MIDItrack].LastEvent) || nocomp )
				{
					if( TWriteByte( MIDItrack, NewEvent, track, BufferSize ) )
					{
						FreeTracks( track ) ;
						return NULL ;
					}
					track[MIDItrack].LastEvent = NewEvent ;
				}
				data = musdata[muspos++] ;
				if( TWriteByte( MIDItrack, (data & 1) << 6, track, BufferSize ) )
				{
					FreeTracks( track ) ;
					return NULL ;
				}
				if( TWriteByte( MIDItrack, data >> 1, track, BufferSize ) )
				{
					FreeTracks( track ) ;
					return NULL ;
				}
				break ;
			case 3 :  // system event
				NewEvent = 0xB0 | MIDIchannel ;
				if( (NewEvent != track[MIDItrack].LastEvent) || nocomp )
				{
					if( TWriteByte( MIDItrack, NewEvent, track, BufferSize ) )
					{
						FreeTracks( track ) ;
						return NULL ;
					}
					track[MIDItrack].LastEvent = NewEvent ;
				}
				data = musdata[muspos++] ;
				if( TWriteByte( MIDItrack, MUS2MIDcontrol[data], track, BufferSize ) )
				{
					FreeTracks( track ) ;
					return NULL ;
				}
				if( data == 12 )
				{
					if( TWriteByte( MIDItrack, MUSh.channels+1, track, BufferSize ) )
					{
						FreeTracks( track ) ;
						return NULL ;
					}
				}
				else
				{
					if( TWriteByte( MIDItrack, 0, track, BufferSize ) )
					{
						FreeTracks( track ) ;
						return NULL ;
					}
				}
				break ;
			case 4 :  // control change
				data = musdata[muspos++] ;
				if( data )
				{
					NewEvent = 0xB0 | MIDIchannel ;
					if( (NewEvent != track[MIDItrack].LastEvent) || nocomp )
					{
						if( TWriteByte( MIDItrack, NewEvent, track, BufferSize ) )
						{
							FreeTracks( track ) ;
							return NULL ;
						}
						track[MIDItrack].LastEvent = NewEvent ;
					}
					if( TWriteByte( MIDItrack, MUS2MIDcontrol[data], track, BufferSize ) )
					{
						FreeTracks( track ) ;
						return NULL ;
					}
				}
				else
				{
					NewEvent = 0xC0 | MIDIchannel ;
					if( (NewEvent != track[MIDItrack].LastEvent) || nocomp )
					{
						if( TWriteByte( MIDItrack, NewEvent, track, BufferSize ) )
						{
							FreeTracks( track ) ;
							return NULL ;
						}
						track[MIDItrack].LastEvent = NewEvent ;
					}
				}
				data = musdata[muspos++] ;
				if( TWriteByte( MIDItrack, data, track, BufferSize ) )
				{
					FreeTracks( track ) ;
					return NULL ;
				}
				break ;
			case 5 :  // abnormal data
			case 7 :
				FreeTracks( track ) ;
				return NULL; // MUSFILECOR
			default : break ;
		}
		if( last( event ) )
		{
			DeltaTime = ReadTime( (pUInt8)musdata, &muspos ) ;
			for( i = 0 ; i < TrackCnt ; i++ )
				track[i].DeltaTime += DeltaTime ;
		}
		event = musdata[muspos++] ;
		et = event_type( event ) ;
		MUSchannel = channel( event ) ;
	}

	if( !division )
		division = 89 ;

	for( i = 0, size = 0 ; i < TrackCnt ; size += track[i++].current+50 ) ;
	size += 50 ;
	middata = (pUInt8) Malloc( size );
	if( middata == NULL)
	{
		FreeTracks( track ) ;
		return NULL; //MEMALLOC ;
	}

	MWriteMIDheader( TrackCnt+1, division, middata, &midpos ) ;
	MWriteFirstTrack( middata, &midpos ) ;
	for( i = 0 ; i < TrackCnt ; i++ )
	{
		MWriteTrack( i, track, middata, &midpos ) ;
		Free( track[i].data ) ;
	}
	*midsize = midpos ;

	return (pInt8)middata ;
}


/* HOW TO USE IT

void main( void )
{
	FILE *mus, *mid ;
	pUInt8 musdata, middata ;
	UInt32 i ;
	UInt32 mussize, midsize ;
	int r ;
	unsigned Int8 c ;

	mus = fopen( "terminat.mus", "rb" ) ;
	mussize = filelength( fileno( mus ) ) ;
	musdata = (pUInt8) Malloc( mussize ) ;
	printf( "\nReading..." ) ;
	for( i = 0 ; i < mussize ; i++ )
	{
		fread( &c, 1, 1, mus ) ;
		musdata[i] = c ;
	}
	fclose( mus ) ;

	printf( "\nConverting..." ) ;
//======================================================================
	middata = qmus2mid(&midsize, musdata, mussize, 0, 0, 0 ) ;
//======================================================================

	if( middata==NULL )
	{
		printf( "Error when converting.\n") ;
		Free( musdata ) ;
		exit( 1 ) ;
	}
	Free( musdata ) ;
	mid = fopen( "terminat.mid", "wb" ) ;
	printf( "\nWriting..." ) ;
	for( i = 0 ; i < midsize ; i++ )
	{
		c = middata[i] ;
		fwrite( &c, 1, 1, mid ) ;
	}
	fclose( mid ) ;
	// oublie
	Free(middata);
}
*/