/************************************************\
* WinTex, Copyright (c) 1995 Olivier Montanuy
*         (montanuy@lannion.cnet.fr)
* With Technical help from M.Mathews and R.Paquay.
*
* All rights reserved. Any commercial  usage is
* prohibited. Parts of this code can be used in
* freeware programs, provided WinTex is credited.
* This code comes with no guaranty whatsoever.
\************************************************/

#define WINTEXMODULE 'd'

#include "lbwintex.h"
#include "lbcommon.h"
#include <stdlib.h>
#include "lbwad.h"
#include "lbdispl.h"

#include "lbwaddef.h"
#include "lbdoom.h"
#include "lbwaddir.h"
#include "lbwadir.h"
#include "lbwadid.h"

#if (DLLFORDOOM)
/*
** WAD HEADER
*/
struct DMWADHEAD
{
  char Magic[4];  /*PWAD, IWAD, WAD2*/
  Int32 DirNb;
  Int32 DirPos;
};
/*
** WAD Directory Entry
*/
struct DMDIR      /*DOOM WAD directory entry*/
{ Int32 Start;          /*Start of entry*/
  Int32 Size;           /*Size of entry*/
  Int8 Name[8];        /*Name of entry*/
};
typedef struct DMDIR PTR *pDMDIR;
#if (sizeof(struct DMDIR)!=0x10)
#error Error: Wrong size for Wad dir
#endif
#endif /*DLLFORDOOM*/

#if (DLLFORDUKE)
/*
** GRP HEADER
*/
struct DKKENHEAD
{
  char  Magic[12]; /*KenSilverman*/
  Int32 DirNb;
};
/*
** WAD Directory Entry
*/
struct DKDIR      /*DOOM WAD directory entry*/
{ Int8 Name[12];        /*Name of entry*/
  Int32 Size;           /*Size of entry*/
};
typedef struct DKDIR PTR *pDKDIR;
#if (sizeof(struct DKDIR)!=0x10)
#error Error: Wrong size for KenS
#endif
#endif /*DLLFORDUKE*/

#if (DLLFORQUAK)
/*
** PACK HEADER
*/
struct PACKHEAD
{
  char Magic[4];  /*PWAD, IWAD, WAD2*/
  Int32 DirPos;   /*Start of dir*/
  Int32 DirSz;    /*Size of dir = 0x40 * Nb of Entries*/
};
/*
** PACK Directory Entry
*/
struct PKDIR      /*DOOM WAD directory entry*/
{ Int8  Name[0x38];        /*Name of entry*/
  Int32 Start;          /*Size of entry*/
  Int32 Size;           /*Size of entry*/
};
typedef struct PKDIR PTR *pPKDIR;
#if (sizeof(struct PKDIR)!=0x40)
#error Error: Wrong size for Pack
#endif

/*
** WAD2 DIRECTORY ENTRY
*/
struct QKDIR      /*Quake's WAD2 directory entry*/
{ Int32 Start;    /* Start of entry*/
  Int32 Size;     /* Size of entry*/
  Int32 Size2;    /* Always like Size?*/
  Int8  Id;       /* Entry Id */
  Int8  Cmprs;    /* Compression*/
  Int16 Dummy;    /* Not used*/
  Int8  Name[16]; /* Name of entry*/
};
typedef struct QKDIR PTR *pQKDIR;
#endif /*DLLFORQUAKE*/

/****************************************************\
*
*
*  WAD directory
*
*
\****************************************************/
  /*
  ** Give the theorical size of the WAD directory
  */
static Int32 WADDsize(Int16 LstNb);
  /*
  ** Init, resize or free WAD directory
  */
static Int16 WADDinitI(pWADDEF This, Int16 LstNb);
  /*
  ** Resize WAD directory
  */
static Int16 WADDresizeI(pWADDEF This, Int16 LstNb);

/*
** Size of WAD directory
*/
static Int32 WADDsize(Int16 LstNb)
{
  if(LstNb==0) LstNb=1; /*allow WADs of size 0*/
  return (((Int32)LstNb+0xFL)&(~0xFL)) * sizeof(struct WADDIR);
}
/*
** Init directory
*/
static Int16 WADDinitI(pWADDEF This, Int16 LstNb)
{
  if(LstNb<0) { return ERR_SURE;}
  This->LstNb=LstNb;
  This->Lst=(pWADDIR)Malloc(WADDsize(This->LstNb));
  if(This->Lst==NULL) { return ERR_MEM;}
  This->DirDirty=FALSE;
  return 1;
}
/*
** Resize directory
*/
static Int16 WADDresizeI(pWADDEF This, Int16 LstNb)
{
  if(LstNb<0)
  { return ERR_DVOID;}
  if(LstNb > 0x4000)
  { return ERRfaultWad(ERR_DFULL);}
  if(WADDsize(This->LstNb)!=WADDsize(LstNb))
  {
	 This->Lst=Realloc(This->Lst,WADDsize(LstNb));
	 if(This->Lst==NULL)
	 { This->LstNb=0; return ERRfaultWad(ERR_DINS); }
  }
  This->LstNb=LstNb;
  return 1;
}
/*
** Free directory
*/
void WADDfreeI(pWADDEF This)
{
  if(This->Lst!=NULL) { Free(This->Lst);}
  This->Lst=NULL;
  This->LstNb=0;
}
/*
** Read PWAD or IWAD directory
*/
#if (DLLFORDOOM)
Int16 WADDinitFromWadI(pWADDEF This, Int16 LstNb)
{ Int32 Size,res;
  pDMDIR Dir;
  Int16 e;
  /**/
  res=WADDinitI(This, LstNb);
  if(res<0)
  { return (Int16)res; }
  /*
  ** Skip void PWAD
  */
  if(This->LstNb==0)
  { return 0; }
  /**/
  Size=((Int32)sizeof(struct DMDIR))*((Int32)This->LstNb);
  Dir=(pDMDIR)Malloc(Size);
  if(Dir==NULL)
  { WADDfreeI(This); return ERR_MEM;}
  res=FILEreadData((pInt8)Dir,This->OrigDirPos,Size,This->File);
  if(res<0)
  { Free(Dir); WADDfreeI(This); return (Int16)res;}
  for(e=0; e<This->LstNb; e++)
  {
	 WADDsetEntryI(This,e,EWHAT,Dir[e].Name,Dir[e].Start,Dir[e].Size);
  }
  Free(Dir);
  return 1;
}
#endif
/*
** Read WAD2 directory
*/
#if (DLLFORQUAK)
Int16 WADDinitFromWad2I(pWADDEF This, Int16 LstNb)
{
  Int32 Size,res;
  pQKDIR Dir;
  Int16 e;
  /**/
  /**/
  res=WADDinitI(This, LstNb);
  if(res<0)
  { return (Int16)res; }
  /*
  ** Skip void WAD2
  */
  if(This->LstNb==0)
  { return 0; }
  /**/
  Size=((Int32)sizeof(struct QKDIR))*((Int32)This->LstNb);
  Dir=(pQKDIR)Malloc(Size);
  if(Dir==NULL)
  { WADDfreeI(This); return ERR_MEM;}
  res=FILEreadData((pInt8)Dir,This->OrigDirPos,Size,This->File);
  if(res<0)
  { Free(Dir); WADDfreeI(This); return (Int16)res;}
  for(e=0; e<This->LstNb; e++)
  {
	 /*identify*/
	 res=WADidentQuake(Dir[e].Name,Dir[e].Id);
	 /*store*/
	 WADDsetEntryI(This,e,(IDENT)res,Dir[e].Name,Dir[e].Start,Dir[e].Size);
  }
  Free(Dir);
  return 1;
}
#endif /*DLLFORQUAK*/


#if (DLLFORDUKE)
Int16 WADDinitFromKenI(pWADDEF This, Int16 LstNb)
{
  Int32 Start,Size,res;
  pDKDIR Dir,TDir;
  Int16 e;
  IDENT id;
  Int8 Name[NORMALISELEN+2];
  /**/
  res=WADDinitI(This, LstNb);
  if(res<0)
  { return (Int16)res; }
  /**/
  Size=((Int32)sizeof(struct DKDIR))*((Int32)This->LstNb);
  Start= This->OrigDirPos + Size;
  Dir=(pDKDIR)Malloc(Size);
  if(Dir==NULL)
  { WADDfreeI(This); return ERR_MEM;}
  res=FILEreadData((pInt8)Dir,This->OrigDirPos,Size,This->File);
  if(res<0)
  { Free(Dir); WADDfreeI(This); return (Int16)res;}
  for(e=0,TDir=Dir; e<This->LstNb; e++, TDir+=1)
  {
	 /* Ident entry*/
	 id=WADidentDuke(Name,TDir->Name); /*set Name, res=id*/
	 /* Set entry*/
	 WADDsetEntryI(This, e, id, Name, Start, TDir->Size);
	 /* Update entry start*/
	 Start += TDir->Size;
  }
  Free(Dir);
  return 1;
}
#endif /*DLLFORDUKE*/


#if (DLLFORQUAK)
Int16 WADDinitFromPackI(pWADDEF This, Int16 LstNb)
{
  Int32 Size,res;
  pPKDIR Dir,TDir;
  Int16 e;
  IDENT id;
  Int8 Name[NORMALISELEN+2];
  /**/
  res=WADDinitI(This, LstNb);
  if(res<0)
  { return (Int16)res; }
  /**/
  Size=((Int32)sizeof(struct PKDIR))*((Int32)This->LstNb);
  Dir=(pPKDIR)Malloc(Size);
  if(Dir==NULL)
  { WADDfreeI(This); return ERR_MEM;}
  res=FILEreadData((pInt8)Dir,This->OrigDirPos,Size,This->File);
  if(res<0)
  { Free(Dir); WADDfreeI(This); return (Int16)res;}
  for(e=0,TDir=Dir; e<This->LstNb; e++, TDir+=1)
  {
	 /* Ident entry*/
	 id=WADidentPack(Name,TDir->Name); /*set Name, res=id*/
	 /* Set entry*/
	 WADDsetEntryI(This, e, id, Name, TDir->Start, TDir->Size);
	 /* Update entry start*/
  }
  Free(Dir);
  return 1;
}
#endif /*DLLFORQUAKE*/

/*
** Set directory entry
** Set start and size if > 0
** change identification if Id>0
** return entry if ok
*/
Int16 WADDsetEntryI(pWADDEF This,Int16 Entry, IDENT Id, pInt8 Name, Int32 Start,Int32 Size)
{  pWADDIR TLst;
  /**/
#if 0
  if(This==NULL)
  { return ERR_BUG;}
  if(This->Lst==NULL)
  { return ERRfault(ERR_BUG);}
#endif
  if((Entry<0)||(Entry>=This->LstNb))
  { return ERRfault(BAD_PARM);}
  /**/
  TLst = &(This->Lst[Entry]);
  /*change internal directory*/
  if((Name!=NULL)&&(Name[0]!='\0'))
  { Normalise(TLst->Name,Name); }
  if(Start>=0)
  { TLst->Start=Start; }
  if(Size>=0)
  { TLst->Size=Size; }
  /*change Id if requested*/
  if(Id>=0)
  { TLst->Id=Id; }
  /*entry is not dirty*/
  TLst->Dirty=FALSE;
  /*do not change name*/
  This->DirDirty=TRUE;
  return Entry;
}
/*
** Insert an entry to the directory
** Directory os not updated on WAD
** if Id>0 set Identifier
*/
Int16 WADDinsEntryI(pWADDEF This, Int16 Entry, IDENT Id, pInt8 Name, Int32 Start, Int32 Size)
{ Int16 e;
  pWADDIR TLst;
  /**/
  if(This==NULL)
  { return ERR_BUG;}
  if((This->Lst==NULL)||(This->LstNb<0))
  { return ERRfault(ERR_BUG);}
  /*
  ** Resize directory if needed
  */
  if(WADDresizeI(This, This->LstNb+1)<0)
  { return ERR_DINS;}
  /*
  ** Insert at the end if needed
  */
  if((Entry<0)||(Entry>=This->LstNb))
  { Entry=This->LstNb-1; }
  /*
  ** Move the previous entries, starting from the end
  */
  e=This->LstNb-1;
  for(TLst=&(This->Lst[e]);e>Entry;e--,TLst-=1)
  { Memcpy(TLst,(TLst-1),sizeof(struct WADDIR));}
  /*
  ** Declare the new entry, at it's new position
  */
  return WADDsetEntryI(This,Entry,Id,Name,Start,Size);
}
/*
** Delete an entry from directory
*/
Int16 WADDdelEntryI(pWADDEF This, Int16 Entry)
{ Int16 e;
  pWADDIR TLst;
  if((This==NULL)||(This->Lst==NULL))
  { return ERRfault(ERR_BUG);}
  if((Entry<0)||(Entry>=This->LstNb))
  { return ERRfault(BAD_PARM);}
  /*
  ** Check that directory can be sized down
  */
  if(This->LstNb<1)
  { return ERRfaultWad(ERR_DVOID);}
  /*
  ** Move the following entries, overwriting Entry
  */
  e=Entry+1;
  for(TLst=&(This->Lst[e]); e<This->LstNb; e++, TLst+=1)
  { Memcpy((TLst-1),TLst,sizeof(struct WADDIR)); }
  /*
  ** Resize directory if needed
  */
  if(WADDresizeI(This, This->LstNb-1)<0)
  { return ERR_DINS;}
#if 1
  /*
  ** Reduce the wasted space by setting the write pointer
  ** just after the last used entry
  */
  This->WritePtr=(This->OrigFileSz+0xFL)&(~0xFL);
  for(e=0,TLst=This->Lst; e<This->LstNb; e++,TLst+=1)
  { /*TLst=This->Lst[e]*/
	 if((TLst->Start+TLst->Size)>This->WritePtr)
	 { This->WritePtr=(TLst->Start+TLst->Size); }
  }
#endif
  This->DirDirty=TRUE;
  return Entry;
}

/*
** Set name of entry
*/
Int16 WADDchangeNameEntryI(pWADDEF This, Int16 Here, IDENT Id, pInt8 Name, pInt8 OldName)
{
  Int16 Entry;
  Entry = WADDfindEntryI(This,Here,OldName,Id);
  if(Entry<0) return Entry;
  return WADDsetEntryI(This, Entry, Id, Name, -1, -1);
}
/*
** Get start and size of an entry
** returns id if ok
*/
Int16 WADDgetEntryI(pWADDEF This,pInt32 pStart,pInt32 pSize, Int16 Entry)
{  pWADDIR TLst;
#if 0
  if((This==NULL)||(This->Lst==NULL))
  { return ERR_BUG;}
#endif
  if((Entry<0)||(Entry>=This->LstNb))
  { return ERRfault(BAD_PARM);}
  /**/
  TLst= &(This->Lst[Entry]);
  if(pStart!=NULL)
  { *pStart=TLst->Start;}
  if(pSize!=NULL)
  { *pSize=TLst->Size;}
  return TLst->Id;
}
/*
** Get name of entry
** returns a pointer to the actual entry name.
*/
pInt8 WADDgetNameEntryI(pWADDEF This,Int16 Entry)
{  pWADDIR TLst;
  /**/
#if 0
  if((This==NULL)||(This->Lst==NULL))
  { return NULL;}
#endif
  if((Entry<0)||(Entry>=This->LstNb))
  { ERRfault(BAD_PARM); return NULL;}
  /**/
  TLst= &(This->Lst[Entry]);
  return TLst->Name;
}
/*
** Find an entry, from name
** if Id>0, select only entries with ident=Id
*/
Int16 WADDfindEntryI(pWADDEF This,Int16 Here,pInt8  Name, IDENT Id)
{
  Int8 Fname[NORMALISELEN+2];
  Int16 e;
  pWADDIR TLst;
  /**/
#if 0
  if((This==NULL)||(This->Lst==NULL))
  { return ERRfault(ERR_BUG);}
#endif
  /**/
  Normalise(Fname,Name);
  e= ((Here>=0)&&(Here<This->LstNb))? Here:0;
  if(Id>0)
  {
	 for(TLst=&(This->Lst[e]); e<This->LstNb; e++,TLst+=1)
	 { /*TLst=This->Lst[e]*/
		/*search according to Id*/
		if(TLst->Id != Id) continue;
		/*if exactly matches the name, stop*/
		if(Strncmp(TLst->Name,Fname,NORMALISELEN))
		{ break; }
	 }
  }
  else
  {
	 for(TLst=&(This->Lst[e]); e<This->LstNb; e++,TLst+=1)
	 { /*TLst=This->Lst[e]*/
		/*if exactly matches the name, stop*/
		if(Strncmp(TLst->Name,Fname,NORMALISELEN))
		{ break; }
	 }
  }
  if(e>=This->LstNb) return -1; /*not found*/
  return e;
}


/****************************************************\
*
*
*  WAD HEADER API
*
*
\****************************************************/

static struct
{ pInt8 Name;
  Int16 NameSz;
  Int16 Type;
} WADHheaders[]=
{
 {"KenSilverman",12,IS_KENS},
 {"IWAD",4,IS_IWAD},
 {"PWAD",4,IS_PWAD},
 {"WAD2",4,IS_WAD2},
 {"PACK",4,IS_PACK},
};

static Int16 WADHheaderIdI(pInt8 Magic)
{
  Int16 n=-1;
  switch(Magic[0])
  { case 'K': n=0; break;
	 case 'I': n=1; break;
	 case 'P': /*PACK or PWAD*/
		n = (Magic[1]=='A')? 4 : 2;
		break;
	 case 'W': n=3; break;
	 default:
		return ERRfaultWad(ERR_BADWADH);
  }
  if(Strncmp(Magic,WADHheaders[n].Name,WADHheaders[n].NameSz))
  { return WADHheaders[n].Type;}
  return ERRfaultWad(ERR_BADWADH);
}
/*
** Write Header of WAD.
*/
static union
{  struct DMWADHEAD Wad;
	struct DKKENHEAD Ken;
#if (DLLFORQUAK)
	struct PACKHEAD  Pack;
#endif
} Head;
/*
** Set WAD Header
*/
Int16 WADHheaderWriteI(pInt8 File,Int32 DirPos,Int32 DirNb)
{ Int32 res;
  /*
  ** Load old header
  */
  res= FILEreadData((pInt8)&Head.Wad, 0, sizeof(Head.Wad), File);
  if(res<0) { return (Int16)res;}
  /*
  ** Check that it's a PWAD
  */
  switch(WADHheaderIdI(Head.Wad.Magic))
  { case IS_IWAD:
	 case IS_PWAD:
		break;
	 default:
		return 0;
  }
  /*
  ** Check that header needs to be modified
  */
  if((Head.Wad.DirPos==DirPos)&&(Head.Wad.DirNb==DirNb))
  { return 1; }
  /*
  ** modify header
  */
  Head.Wad.DirPos=DirPos;
  Head.Wad.DirNb=DirNb;
  res=FILEwriteData((pInt8)&Head.Wad,0,sizeof(Head.Wad),File);
  /* in case of failure, double write attempt */
  if(res<0)
  { res=FILEwriteData((pInt8)&Head.Wad,0,sizeof(Head.Wad),File);
	 /* definite failure */
	 if(res<0)
	 {	return ERRfaultWad(ERR_DEAD); }
  }
  return 1;
}
/*
** Read Header of WAD.
** returns nb of entries, dir pos.
** returns code IS_IWAD, IS_PWAD, IS_WAD2
*/
Int16 WADHheaderReadI(pInt32 pDirPos,pInt32 pDirNb,pInt8 File, Int32 FileSz)
{ Int32 res,DirPos,DirNb;
  Int16 type=0,err=ERR_BADWADH;
  /**/
  res=FILEreadData((pInt8 )&Head,0,sizeof(Head),File);
  if(res<0)return (Int16)res;
  type=WADHheaderIdI(Head.Wad.Magic);
  switch(type)
  {
	 case IS_KENS:
		DirPos=0x10;
		DirNb=Head.Ken.DirNb;
		err= ERR_BADGRP;
		break;
#if (DLLFORQUAK)
	 case IS_PACK:
#if (sizeof(struct PKDIR)!=0x40)
#error Please update that code
#endif
		DirNb=(Head.Pack.DirSz)>>6; /*Pack dir size= nb * sizeof(PKDIR)*/
		DirPos=Head.Pack.DirPos;      /*Pack dir pos*/
		break;
#endif /*DLLFORQUAK*/
	 case IS_PWAD:
	 case IS_IWAD:
	 case IS_WAD2:
		DirPos=Head.Wad.DirPos;
		DirNb=Head.Wad.DirNb;
		break;
	 default: /*not a PWAD of KEN*/
		return ERR_BADWADH;
  }
  /*
  ** Check header and exit
  */
  if((DirNb>=0)&&(DirNb<0x8000L))
  { if((DirPos>0)&&(DirPos<FileSz))
	 { *pDirNb=DirNb;
		*pDirPos=DirPos;
		return type;
	 }
  }
  return ERRfaultWad(err); /*BAD_WADH or ERR_BADGRP*/
}
/*
** fake WAD
*/
static struct FAKEWAD
{ Int8 Magic[4];
  Int32 DirNb;
  Int32 DirPos;
  Int8 Name[20];
} FakeWad=
{"PWAD",0,0xC,"[WinTex (c)1996 OJM]"};
/*
** create fake WAD
*/
Int16 WADHmakeFakeWadI(pInt8 File)
{ Int16 res;
  res=FILEopen(File,FCREATE);
  if(res<0) return res;
  res=(Int16)FILEwrite((pInt8)&FakeWad,0,sizeof(FakeWad));
  FILEclose();
  if(res<0)return res;
  return 1;
}
/****************************************************\
*
*
*  Save and restore the WAD directory
*
*
\****************************************************/
/*
** save Pack directory
*/
#if (DLLFORQUAK)
static Int16 WADDsavePackDirI(pWADDEF This)
{
  Int32 res,DirSz,DirPos,DirNb,Start,Size;
  pPKDIR Dir,TDir;
  pInt8 Name2;
  Int16 e;
  IDENT id;
  Int8 Name[NORMALISELEN+2];
  /**/
  e=WADHheaderReadI(&DirPos,&DirNb,This->File, This->OrigFileSz);
  if((e!=IS_PACK)||(DirPos<0)||(DirNb<=0))
  { return 0;}
  if(DirNb!=This->LstNb)
  { return 0;}
  /**/
  DirSz=((Int32)sizeof(struct PKDIR))*DirNb; /*Pack dir size= nb * sizeof(PKDIR)*/
  Dir=(pPKDIR)Malloc(DirSz);
  if(Dir==NULL)
  { return ERR_MEM;}
  res=FILEreadData((pInt8)Dir,DirPos,DirSz,This->File);
  if(res<0)
  { Free(Dir); return (Int16)res;}
  for(e=0,TDir=Dir; e<This->LstNb; e++, TDir+=1)
  {
	 Name2=WADDgetNameEntryI(This,e);
	 if(Name2==NULL) continue;
	 /* Ident entry*/
	 id=WADidentPack(Name,TDir->Name); /*set Name, res=id*/
    Normalize(Name);
	 if(Strncmp(Name2,Name,NORMALISELEN))
	 { /* Get entry*/
		if(id==WADDgetEntryI(This, &Start, &Size,e))
		{ TDir->Start=Start; TDir->Size=Size; }
	 }
	 /* Update entry start*/
  }
  res=FILEwriteData((pInt8)Dir,DirPos,DirSz,This->File);
  Free(Dir);
  if(res<0)
  { return (Int16)res; }
  /*
  ** WAD dir need not be saved
  */
  This->DirDirty=FALSE;
  return 1;
}
#endif /*DLLFORQUAKE*/
/*
**
** Save the directory of the WAD
** if Restorable>0, make the WAD restorable, but with warning.
** if Restorable=0, make the WAD restorable, no warning.
** if restorable<0, the WAD is not restorable.
*/
static struct DMDIR HDRdir[5];
Int16 WADHdirSaveI(pWADDEF This, Int16 Restore)
{ Int32 DirPos;
  Int32 Size,res;
  Int16 e;
  pWADDIR TLst;
  pDMDIR Dir;

  if((This==NULL)||(This->Lst==NULL))
  { return ERRfault(ERR_BUG); }
  /*
  ** Check if a save is needed. if not, return
  */
  if((This->DirDirty==FALSE)||(This->Locked!=FALSE))
  { return 0;}
  if(WADmodifiableI(This)<=0) return ERR_SURE;

#if (DLLFORQUAK)
  if(This->Game&GAM_PACK)
  { return WADDsavePackDirI(This);}
#endif /*DLLFORQUAKE*/
  /*
  ** Write all stuff at WritePrt, but do NOT update
  ** WritePtr, so that successive directory saves don't
  ** swallow too much place.
  */
  DirPos=This->WritePtr;
  /*security*/
  if(DirPos<This->OrigFileSz) DirPos=This->OrigFileSz;
  DirPos=(DirPos+3L)&~3L; /*align4*/
  /*
  ** Preserve old status of the file
  */
  if(Restore >= 0)
  {
	 HDRdir[0].Start= 0x24061968L;
	 HDRdir[0].Size = 666L;
	 Normalise(HDRdir[0].Name,"IZNOGOOD");
	 /*Set original WAD DIRECTORY*/
	 HDRdir[1].Start= This->OrigDirPos;
	 HDRdir[1].Size = This->OrigDirNb;
	 Normalise(HDRdir[1].Name,(This->WadType==IS_IWAD)?"DOOM_DIR":"PWAD_DIR");
	 /*Store original WAD size and start*/
	 HDRdir[2].Start= 0;
	 HDRdir[2].Size = This->OrigFileSz;
	 Normalise(HDRdir[2].Name,"ORIGINAL");
	 /*No external WAD*/
	 HDRdir[3].Start= 0;
	 HDRdir[3].Size = 0;
	 Normalise(HDRdir[3].Name,(Restore>0)? "_WINTEX_": "_TEMP_");
	 /*old file time*/
	 HDRdir[4].Size = This->OrigTime;
	 /*do not restore internal PWAD*/
	 HDRdir[4].Start= FALSE;
	 Normalise(HDRdir[4].Name,"TIME");
	 /*
	 ** write some fake entries for restoration
	 */
	 res=FILEwriteData((pInt8)&HDRdir,DirPos,sizeof(HDRdir),This->File);
	 DirPos+=sizeof(HDRdir);
	 if(res<0) return (Int16)res;
  }
  /*
  ** save the new WAD directory
  */
  if(This->LstNb>0)
  {
	 if((This->WadType==IS_IWAD)||(This->WadType==IS_PWAD))
	 {
		Size=((Int32)sizeof(struct DMDIR))*((Int32)This->LstNb);
		Dir=(pDMDIR)Malloc(Size);
		if(Dir==NULL) {return ERR_MEM;}
		/*
		** Translate into WAD directory format
		*/
		for(e=0,TLst=This->Lst; e<This->LstNb; e++,TLst+=1)
		{ /*TLst=This->Lst[e]*/
		  Dir[e].Start =TLst->Start;
		  Dir[e].Size  =TLst->Size;
		  Normalise(Dir[e].Name,TLst->Name);
		}
		/*
		** write
		*/
		res=FILEwriteData((pInt8)Dir,DirPos,Size,This->File);
		Free(Dir);
	 }
	 if(res<0)
	 { return (Int16)res;}
  }
  /*
  ** WAD update header
  */
  if( WADHheaderWriteI(This->File,DirPos,This->LstNb) <0)
  { return (Int16)res;}
  /*
  ** WAD dir need not be saved
  */
  This->DirDirty=FALSE;
  return 1;
}
/*
** Restore WAD
** returns <0 if restore impossible or not needed
** returns 0 if safe, and modifs may be lost.
** returns 1 if restore successful.
*/
Int16 EXPORT WADrestore(pInt8 File, Int16 Safe)
{
  Int32 DirPos, DirNb, FileSz, FileTm;
  /**/
  if(WADHheaderReadI(&DirPos, &DirNb, File, FILEgetSize(File))<0)
  { return BAD_HEAD;}
  DirPos-=sizeof(HDRdir);
  if(DirPos<0)
  { return 0; }
  if(FILEreadData((pInt8)&HDRdir,DirPos,sizeof(HDRdir),File)<0)
  { return ERR_SURE;}
  /*
  ** check format
  */
  if(HDRdir[0].Start!=0x24061968L) return -1;
  if(HDRdir[0].Size != 666L) return -1;
  if(!Strncmp(HDRdir[0].Name,"IZNOGOOD",8)) return -1;
  FileSz= HDRdir[2].Size;
  DirNb=  HDRdir[1].Size;
  DirPos= HDRdir[1].Start;
  FileTm= HDRdir[4].Size;
  /*
  ** check that size, dir is correct
  */
  if((DirNb<0)||(DirNb>0x8000L)) return -1;
  if(DirPos<0) return -1;
  if(FileSz< (DirPos+DirNb*sizeof(struct DMDIR))) return -1;
  /*
  ** Safety check, for WAD modified by WinTex
  */
  if(Safe>0)
  {
	 if(Strncmp(HDRdir[3].Name,"_WINTEX_",8))
		return 0; /*0 means it's a WinTex wad*/
  }
  /*
  ** restore
  */
  if(WADHheaderWriteI(File,DirPos,DirNb)<0)
  { return -1; }
  FILEsetSize(File,FileSz);
  FILEsetTime(File,FileTm);
  return 1;
}



