/************************************************\
* WinTex, Copyright (c) 1995 Olivier Montanuy
*         (montanuy@lannion.cnet.fr)
* With Technical help from M.Mathews and R.Paquay.
*
* All rights reserved. Any commercial  usage is
* prohibited. Parts of this code can be used in
* freeware programs, provided WinTex is credited.
* This code comes with no guaranty whatsoever.
\************************************************/

#define WINTEXMODULE 'Y'


#include "lbwintex.h"
#include "lbcommon.h"
#include <stdlib.h>
#include <math.h>
#include "lbwad.h"
#include "lbdispl.h"
#include "lbtext.h"

#include "lbwaddef.h"
#include "lbquake.h"
#include "lbqkpart.h"
#include "lbqktexu.h"

#if (DLLFORQUAK)
/****************************************************\
*
*
* Caches
*
*
\****************************************************/


Int16 BSPcachInit(pBSPDEF Bsp)
{
 if((Bsp->VrtxNb<=0)||(Bsp->PlanNb<=0))
 {
	Bsp->CachVrtx=NULL;
	Bsp->CachPlan=NULL;
	return ERR_BUG;
 }
 /*
 ** alloc vertices cache
 */
 Bsp->CachVrtx=(pCACHVRTX)Calloc(Bsp->VrtxNb,sizeof(CACHVRTX));
 /*
 ** alloc planes cache
 */
 Bsp->CachPlan=(pInt8)Calloc(Bsp->PlanNb,sizeof(Int8));
 /*
 ** alloc edge color cache
 */
 Bsp->ColEdge=(pInt8)Calloc(Bsp->EdgeNb,sizeof(Int8));
 /*
 ** check allocations
 */
 if((Bsp->CachVrtx==NULL)||(Bsp->CachPlan==NULL))
 { return ERR_MEM;}
 return 1;
}

Int16 BSPcachFree(pBSPDEF Bsp)
{
  if(Bsp->CachVrtx!=NULL) { Free(Bsp->CachVrtx);}
  Bsp->CachVrtx=NULL;
  if(Bsp->CachPlan!=NULL) { Free(Bsp->CachPlan);}
  Bsp->CachPlan=NULL;
  if(Bsp->ColEdge!=NULL) { Free(Bsp->ColEdge);}
  Bsp->ColEdge=NULL;
  return 1;
}
/*
** Calculate cache
*/
void BSPcachCalcRot(pMATRIX Rot, Int32 Phi, Int32 The)
{
  Float32 phi=((Float32)Phi)*(M_PI)/180;
  Float32 the=((Float32)The)*(M_PI)/180;
  Float32 Cphi,Sphi,Cthe,Sthe;
  Cphi = cos(phi);  		Sphi = sin(phi);
  Cthe = cos(the);  		Sthe = sin(the);
  Rot->M11= Cphi*Cthe;	Rot->M12= Sphi*Cthe;	Rot->M13=Sthe;
  Rot->M21=-Sphi;			Rot->M22= Cphi;		Rot->M23=0;
  Rot->M31=-Cphi*Sthe;	Rot->M32=-Sphi*Sthe;	Rot->M33=Cthe;
  return;
/* [ Cphi.Cthe		Sphi.Cthe		Sthe	]
** [ -Sphi			Cphi				0		]
** [ -Cphi.Sthe	-Sphi.Sthe		Cthe	]
*/
}
/*
** Calculate if Observer is in front of plane or not
*/
void BSPcachCalcPlan(pBSPDEF Bsp, pVECTOR Obs)
{
  Int32 n;
  pQKPLANE TPlan;
  pInt8 TChpl;
  Float32 Dot/*,Y,Z*/;
  /*check*/
  if((Bsp->Plan==NULL)||(Bsp->CachPlan==NULL)||(Obs==NULL))
  { ERRfault(ERR_BUG); return; }
  /**/
  for(n=0, TPlan=Bsp->Plan, TChpl=Bsp->CachPlan;
		n<Bsp->PlanNb;  n++, TPlan+=1, TChpl+=1)
  { /* Dot product of plane normal and Observer */
	 Dot =  TPlan->Norm.X* Obs->X;
	 Dot += TPlan->Norm.Y* Obs->Y;
	 Dot += TPlan->Norm.Z* Obs->Z;
	 /* Decide if Observer is in front of plane */
	 *TChpl = (Dot>TPlan->Dist)? 1: 0;
  }
}
/*
** 
*/
/*
** Draw line
*/
#define BSP_SCREEN_DIST (-128)
Int16 BSPcachLineVrtx(pBMPOBJ Bmp, pBOUND Bound, pCACHVRTX TChv0, pCACHVRTX TChv1, Int8 Color)
{
  Float32 a0,a1,u,v;
  Int16 ScrX,ScrY;
  pCACHVRTX Tmp;
  /*
  ** If edge behind screen plane, ok
  */
  if(TChv0->X<=BSP_SCREEN_DIST)
  {
	 /*
	 ** Check Vertex0 and Vertex1 is in front of screen plane
	 */
	 if(TChv1->X<=BSP_SCREEN_DIST)
	 {
		return BMPline(Bmp,TChv0->ScrX,TChv0->ScrY,TChv1->ScrX,TChv1->ScrY,Color);
	 }
  }
  else /*(TChv0->X>BSP_SCREEN_DIST)*/
  {
	 /*
	 ** Vertex0 is in front of screen plane
	 ** Check if Vertex1 is in front of screen plane
	 */
	 if(TChv1->X>BSP_SCREEN_DIST)
	 { return 0;} /*hidden line*/
	 /*
	 ** Exchange TChv1 et TChv0
	 */
	 Tmp=TChv1; TChv1=TChv0; TChv0=Tmp;
  }
  /*
  ** Vertex0 is behind screen plane (valid)
  ** Vertex1 is in front of screen plane
  ** TChv1->X>BSP_SCREEN_DIST
  */

  /*cut A0-A1 by plane X=BSP_SCREEN_DIST*/
  a0= (BSP_SCREEN_DIST - TChv1->X)/(TChv0->X - TChv1->X);
  a1= (TChv0->X - BSP_SCREEN_DIST)/(TChv0->X - TChv1->X);
  /* a0+a1 =1
  /* a0*TChv0->X + a1*TChv1->X = BSP_SCREEN_DIST */
  u= -(a0*TChv0->Y + a1*TChv1->Y)/BSP_SCREEN_DIST;
  v= -(a0*TChv0->Z + a1*TChv1->Z)/BSP_SCREEN_DIST;
  u= u*((Float32)Bound->ScalX) + (Float32)Bound->OfsX;
  v= v*((Float32)Bound->ScalY) + (Float32)Bound->OfsY;
  ScrX= (u<-30000)? -30000: ((u>30000)? 30000: (Int16)u);
  ScrY= (v<-30000)? -30000: ((v>30000)? 30000: (Int16)v);
  return BMPline(Bmp,TChv0->ScrX,TChv0->ScrY,ScrX,ScrY,Color);
}
/*
** Calculate position of one vertex on screen
*/
void BSPcachCalc1Vrtx(pQKVERTEX TVrtx, pCACHVRTX TChvx, pBOUND Bound, pVECTOR Obs, pMATRIX Rot)
{
  Float32 X0,Y0,Z0,X,Y,Z;
  /*
  ** (X0,Y0,Y0) = position relative to the Observer
  **  as expressed in World coordinates
  */
  X0= TVrtx->X -Obs->X;
  Y0= TVrtx->Y -Obs->Y;
  Z0= TVrtx->Z -Obs->Z;
  /*
  ** (X,Y,Z) = position (X0,Y0,Z0)
  **  as expressed in Observer coordinates
  */
  X = X0 * Rot->M11 +  Y0 * Rot->M12 + Z0 * Rot->M13;
  Y = X0 * Rot->M21 +  Y0 * Rot->M22 + Z0 * Rot->M23;
  Z = X0 * Rot->M31 +  Y0 * Rot->M32 + Z0 * Rot->M33;
  /*
  ** Projection of (X,Y,Z) on the Screen plane
  */
  TChvx->X=X; TChvx->Y=Y; TChvx->Z=Z;
  if(X<BSP_SCREEN_DIST) /*distance to screen must be > 0*/
  {
	 X0= -(Y/X)*((Float32)Bound->ScalX) + (Float32)Bound->OfsX;
	 Y0= -(Z/X)*((Float32)Bound->ScalY) + (Float32)Bound->OfsY;
	 TChvx->ScrX= (X0<-32000)? -32000: ((X0>32000)? 32000: (Int16)X0);
	 TChvx->ScrY= (Y0<-32000)? -32000: ((Y0>32000)? 32000: (Int16)Y0);
  }
}



/*
** Calculate position of Vertices on screen
** and distance to observer (or rather, screen plane)
*/
void BSPcachCalcVrtx(pBSPDEF Bsp, pBOUND Bound, pVECTOR Obs, pMATRIX Rot)
{
  Int32 n;
  pQKVERTEX TVrtx;
  pCACHVRTX TChvx;
  /*check*/
  if((Bsp->Vrtx==NULL)||(Bsp->CachVrtx==NULL)||(Rot==NULL)||(Obs==NULL))
  { ERRfault(ERR_BUG); return; }
  /**/
  for(n=0, TVrtx=Bsp->Vrtx, TChvx=Bsp->CachVrtx;
		n<Bsp->VrtxNb;  n++, TVrtx+=1, TChvx+=1)
  {
	 BSPcachCalc1Vrtx(TVrtx, TChvx, Bound, Obs, Rot);
  }
}


/*
** Axis
*/
#define  BSPcachAxisNb (4)
static QKVERTEX BSPcachAxis[]=
{ {0,0,0},{200,0,0},{0,200,0},{0,0,200}};
static Int8 BSPcachAxisColor[]=
{ BMP_COL_BLACK, BMP_COL_BLUE,BMP_COL_RED,BMP_COL_PURPLE};

void BSPcachLineAxis(pBMPOBJ Bmp, pBOUND Bound, pVECTOR Obs, pMATRIX Rot)
{
  Int16 n;
  CACHVRTX Chvx[BSPcachAxisNb];
  for(n=0; n<BSPcachAxisNb; n++)
  { BSPcachCalc1Vrtx(&BSPcachAxis[n], &Chvx[n],Bound, Obs, Rot);}

  for(n=1; n<BSPcachAxisNb; n++)
  {
	 if(BSPcachLineVrtx(Bmp,Bound,&Chvx[0],&Chvx[n],BSPcachAxisColor[n])<0)
	 { break;}
  }
}
/****************************************************\
*
*
* Quake entities
*
*
\****************************************************/

Int16 BSPentiInit(pBSPDEF Bsp, pInt8 Lmp, Int32 LmpSz)
{
 if((Bsp==NULL)||(Lmp==NULL)||(LmpSz<=0))
 { return ERR_BUG; }
 Bsp->EntiSz= LmpSz;
 /*Calloc(Bsp->EntiSz, sizeof(Int8));*/
 Bsp->Enti=(pInt8)Malloc(LmpSz);
 if(Bsp->Enti==NULL)
 { return ERR_MEM;}
 Memcpy(Bsp->Enti,Lmp,LmpSz);
 return 1;
}
/*
** Copy all entities into lump
** if Lmp==NULL or LmpSz<=0, return lump size
*/
Int32 BSPentiCopyAll(pBSPDEF Bsp, pInt8 Lmp, Int32 LmpSz)
{
  Int32 Size;
  if((Bsp==NULL)||(Bsp->Enti==NULL)||(Bsp->EntiSz<0))
  { return ERR_BUG; }
  /**/
  Size= Bsp->EntiSz;
  if(LmpSz<=0)
  { return Size; }
  /**/
  if((LmpSz!=0)&&(LmpSz<Size))
  { Size=LmpSz; }
  Memcpy(Lmp,Bsp->Enti,Size);
  return Size;
}
/*
** Check entities
*/
Int16 BSPentiCheck(pBSPDEF Bsp, pTXTOBJ Txt)
{ Int32 n,bracket;
  pInt8 Str;
  bracket=0;
  TXTprintName(Txt,QBSP_ENTI,"Checking Entities",-1);
  for(Str=Bsp->Enti,n=0; n<Bsp->EntiSz; n++,Str+=1)
  { switch(*Str)
	 { case '{': bracket++;break;
		case '}': bracket--;break;
	 }
  }
  if(bracket==0)
  { return 1;}
  if(bracket<0)
  { TXTprintName(Txt,QBSP_ENTI," Too many '}'",-1);}
  else /* bracket>0 */
  { TXTprintName(Txt,QBSP_ENTI," Too many '{'",-1);}
  return -1;
}
/****************************************************\
*
*
* Quake Planes
*
*
\****************************************************/

Int16 BSPplanInit(pBSPDEF Bsp, pInt8 Lmp, Int32 LmpSz)
{
 Bsp->PlanNb= LmpSz/sizeof(QKPLANE);
 /*Calloc(Bsp->PlanNb, sizeof(QKPLANE));*/
 Bsp->Plan=(pQKPLANE)Malloc(LmpSz);
 if(Bsp->Plan==NULL) { return ERR_MEM;}
 Memcpy((pInt8)Bsp->Plan,Lmp,LmpSz);
 return 1;
}
/*
** Copy all planes into lump
** if Lmp==NULL or LmpSz<0, return lump size
*/
Int32 BSPplanCopyAll(pBSPDEF Bsp, pInt8 Lmp, Int32 LmpSz)
{
  Int32 Size;
  if((Bsp==NULL)||(Bsp->Plan==NULL)||(Bsp->PlanNb<0))
  { return ERR_BUG; }
  /**/
  Size= Bsp->PlanNb*sizeof(QKPLANE);
  if((Lmp==NULL)||(LmpSz<0))
  { return Size; }
  /**/
  if(LmpSz<Size)
  { Size=LmpSz; }
  Memcpy(Lmp,Bsp->Plan,Size);
  return Size;
}
/*
** Check Planes
*/
Int16 BSPplanCheck(pBSPDEF Bsp, pTXTOBJ Txt)
{ Int32 n; Int16 res;
  pQKPLANE TPlan;
  Float32 Norm;
  res=1;
  TXTprintName(Txt,QBSP_PLAN,"Checking Planes",-1);
  for(TPlan=Bsp->Plan,n=0; n<Bsp->PlanNb; n++,TPlan+=1)
  { Norm=  TPlan->Norm.X * TPlan->Norm.X;
	 Norm+= TPlan->Norm.Y * TPlan->Norm.Y;
	 Norm+= TPlan->Norm.Z * TPlan->Norm.Z;
	 if((Norm<0.999)||(Norm>1.001))
	 {
		TXTprintFrm(Txt," Plane %ld: Norm is not 1.",n);
		res=-1;
	 }
	 /*check type, according to normal*/
	 if((TPlan->Type<0)||(TPlan->Type>5))
	 {
		TXTprintFrm(Txt," Plane %ld: bad plane type %ld",n, TPlan->Type);
		res=-1;
	 }
  }
  return res;
}
/*
** Modify planes
*/
Int16 BSPplanModify(pBSPDEF Bsp, pBSPINFOS Infos, Int16 What)
{
  pQKPLANE TPlan;
  Infos->Nb=Bsp->PlanNb;
  if((Infos->N<0)||(Infos->N>=Bsp->PlanNb))
  { return BAD_PARM;}
  TPlan=&(Bsp->Plan[Infos->N]);
  if(What>0)
  {
	 TPlan->Type=  (Infos->PlanType&0x7);
  }
  else
  {
	 Infos->PlanType = TPlan->Type;
  }
  return 1;
}




/****************************************************\
*
*
* Quake Vertices
*
*
\****************************************************/

Int16 BSPvrtxInit(pBSPDEF Bsp, pInt8 Lmp, Int32 LmpSz)
{
 Bsp->VrtxNb= LmpSz/sizeof(QKVERTEX);
 /*Calloc(Bsp->VrtxNb, sizeof(QKVERTEX));*/
 Bsp->Vrtx=(pQKVERTEX)Malloc(LmpSz);
 if(Bsp->Vrtx==NULL) { return ERR_MEM;}
 Memcpy((pInt8)Bsp->Vrtx,Lmp,LmpSz);
 return 1;
}
/*
** Check Vertices
*/
Int16 BSPvrtxCheck(pBSPDEF Bsp, pTXTOBJ Txt)
{
  (void)Bsp;
  (void)Txt;  /*nothing to check*/
  return 1;
}
/*
** Copy all vertices into lump
** if Lmp==NULL or LmpSz<0, return lump size
*/
Int32 BSPvrtxCopyAll(pBSPDEF Bsp, pInt8 Lmp, Int32 LmpSz)
{
  Int32 Size;
  if((Bsp==NULL)||(Bsp->Vrtx==NULL)||(Bsp->VrtxNb<0))
  { return ERR_BUG; }
  /**/
  Size= Bsp->VrtxNb*sizeof(QKVERTEX);
  if((Lmp==NULL)||(LmpSz<0))
  { return Size; }
  /**/
  if(LmpSz<Size)
  { Size=LmpSz; }
  Memcpy(Lmp,Bsp->Vrtx,Size);
  return Size;
}
/*
** Calculate bound box
*/
Int16 BSPvrtxCalcBound(pBSPDEF Bsp, pBOUND Bound)
{
  Int32 n,dum;
  pQKVERTEX TVrtx;
  if((Bsp==NULL)||(Bound==NULL)||(Bsp->Vrtx==NULL))
  { return ERRfault(ERR_BUG);}
  Bound->Xmin=Bound->Ymin=Bound->Zmin= 0x7FFFFFFFL;
  Bound->Xmax=Bound->Ymax=Bound->Zmax=-0x7FFFFFFFL;
  for(n=0, TVrtx=Bsp->Vrtx; n<Bsp->VrtxNb ;n++, TVrtx+=1)
  {
	 dum= TVrtx->X;
	 if(dum<Bound->Xmin) { Bound->Xmin=dum;}
	 if(dum>Bound->Xmax) { Bound->Xmax=dum;}
	 dum= TVrtx->Y;
	 if(dum<Bound->Ymin) { Bound->Ymin=dum;}
	 if(dum>Bound->Ymax) { Bound->Ymax=dum;}
	 dum= TVrtx->Z;
	 if(dum<Bound->Zmin) { Bound->Zmin=dum;}
	 if(dum>Bound->Zmax) { Bound->Zmax=dum;}
  }
  return 1;
}
/****************************************************\
*
*
* Quake Visibility Lists
*
*
\****************************************************/


Int16 BSPvisiInit(pBSPDEF Bsp, pInt8 Lmp, Int32 LmpSz)
{
  Bsp->VisiSz= LmpSz;
  /*Calloc(Bsp->VisiSz, sizeof(Int8));*/
  Bsp->Visi=(pInt8)Malloc(LmpSz);
  if(Bsp->Visi==NULL) { return ERR_MEM;}
  Memcpy(Bsp->Visi,Lmp,LmpSz);
  return 1;
}
/*
** Check visibility lists
*/
Int16 BSPvisiCheck(pBSPDEF Bsp, pTXTOBJ Txt)
{
  (void)Bsp;
  (void)Txt;  /*nothing to check*/
  return 1;
}
/*
** Copy all visilist into lump
** if Lmp==NULL or LmpSz<0, return lump size
*/
Int32 BSPvisiCopyAll(pBSPDEF Bsp, pInt8 Lmp, Int32 LmpSz)
{
  Int32 Size;
  if((Bsp==NULL)||(Bsp->Visi==NULL)||(Bsp->VisiSz<0))
  { return ERR_BUG; }
  /**/
  Size= Bsp->VisiSz;
  if((Lmp==NULL)||(LmpSz<0))
  { return Size; }
  /**/
  if(LmpSz<Size)
  { Size=LmpSz; }
  Memcpy(Lmp,Bsp->Visi,Size);
  return Size;
}
/*
** Paint visilist
*/
Int16 BSPvisiPaint(pBSPDEF Bsp, Int32 Leaf)
{
  Int32 Modl,lnb,lfirst,l,visi;
  pQKMODEL TModl;
  pQKLEAF TLeaf;
  pInt8   TVisi;
  Int16 bit;
  if((Bsp==NULL)||(Bsp->Leaf==NULL)||(Bsp->Modl==NULL))
  { return ERRfault(ERR_BUG);}
  if((Leaf<0)||(Leaf>=Bsp->LeafNb))
  { return ERRfault(BAD_PARM);}
  /*
  ** Find the modl the leaf is in: always leaf 0
  */
  Modl=0;
  lfirst=0; 			/*first leaf*/
  /**/
  TModl=&(Bsp->Modl[Modl]);
  lnb=TModl->LeafNb; /*number of leaves*/
  /*
  ** Check visilists
  */
  TLeaf=&(Bsp->Leaf[Leaf]);
  /*
  **
  */
  if((Bsp->Visi==NULL)||(Bsp->VisiSz<=0)||(TLeaf->Visi<0))
  {
	 BSPmodlPaint(Bsp,Modl,BMP_COL_PURPLE);
  }
  else
  { /*
	 ** Visilists are run-length encoded.
	 ** zero is followed by the number of time it is used
	 */
	 visi=TLeaf->Visi;
	 TVisi=&(Bsp->Visi[visi]);
	 for(l=1; (l<lnb)&&(visi<Bsp->VisiSz);  visi++, TVisi+=1)
	 {
		if(TVisi[0]==0) /*code zero*/
		{ /*
		  ** TVisi[1] indicates how many time zero is used
		  ** so 8*TVisi[1] is the number of leaves to skip
		  */
		  if(visi+1>= Bsp->VisiSz){ break;}
		  l+= 8*TVisi[1]; /* -8 because of l+=8 in the loop */
		  visi+=1; TVisi+=1;
		}
		else /*normal code, decode it*/
		{
		  for(bit=0;bit<8; bit++)
		  {
			 if((TVisi[0])&(1<<bit)) /*(7-bit)*/
			 {
				if(BSPleafPaint(Bsp, lfirst+l,BMP_COL_PURPLE)<0)
				{ return ERR_SURE; }
			 }
          l+=1; /*next leaf*/
		  }
		}
	 }
  }
  return BSPleafPaint(Bsp, Leaf,BMP_COL_YELLOW);
}

/****************************************************\
*
*
* Quake Nodes
*
*
\****************************************************/


Int16 BSPnodeInit(pBSPDEF Bsp, pInt8 Lmp, Int32 LmpSz)
{
 Bsp->NodeNb= LmpSz/sizeof(QKNODE);
 /*Calloc(Bsp->NodeNb, sizeof(QKNODE));*/
 Bsp->Node=(pQKNODE)Malloc(LmpSz);
 if(Bsp->Node==NULL) { return ERR_MEM;}
 Memcpy(Bsp->Node,Lmp,LmpSz);
 return 1;
}
/*
** Copy all nodes into lump
** if Lmp==NULL or LmpSz<0, return lump size
*/
Int32 BSPnodeCopyAll(pBSPDEF Bsp, pInt8 Lmp, Int32 LmpSz)
{
  Int32 Size;
  if((Bsp==NULL)||(Bsp->Node==NULL)||(Bsp->NodeNb<0))
  { return ERR_BUG; }
  /**/
  Size= Bsp->NodeNb*sizeof(QKNODE);
  if((Lmp==NULL)||(LmpSz<0))
  { return Size; }
  /**/
  if(LmpSz<Size)
  { Size=LmpSz; }
  Memcpy(Lmp,Bsp->Node,Size);
  return Size;
}
/*
** Check Child
*/
static BSPnodeCheckChild(pBSPDEF Bsp, UInt16 Child)
{
  Int32 child,childnb;
  if(Child&0x8000)
  { child = (Child^0xFFFFL)&0x7FFFL;
	 childnb = Bsp->LeafNb;
  }
  else
  { child = (Child)&0x7FFFL;
	 childnb = Bsp->NodeNb;
  }
  return ((child<0)||(child>childnb))? -1 : 1;
}
/*
** Check Nodes
*/
Int16 BSPnodeCheck(pBSPDEF Bsp, pTXTOBJ Txt)
{ Int32 n; Int16 res;
  pQKNODE TNode;
  res=1;
  TXTprintName(Txt,QBSP_NODE,"Checking Nodes",-1);
  for(TNode=Bsp->Node,n=0; n<Bsp->NodeNb; n++,TNode+=1)
  {
	 if(TNode->Plan>Bsp->PlanNb)
	 {
		TXTprintFrm(Txt," Node %ld: bad plane index",n);
		res=-1;
	 }
	 if(BSPnodeCheckChild(Bsp,TNode->Front)<0)
	 {
		TXTprintFrm(Txt," Node %ld: bad front child",n);
		res=-1;
	 }
	 if(BSPnodeCheckChild(Bsp,TNode->Back)<0)
	 {
		TXTprintFrm(Txt," Node %ld: bad back child",n);
		res=-1;
	 }
	 /*
	 ** Bound box: ignored
	 */
	 if((TNode->Face+TNode->FaceNb)>Bsp->FaceNb)
	 {
		TXTprintFrm(Txt," Node %ld: bad face list",n);
		res=-1;
	 }
  }
  return res;
}
Int16 BSPnodeModify(pBSPDEF Bsp, pBSPINFOS Infos, Int16 What)
{
  pQKNODE TNode;
  Infos->Nb=Bsp->NodeNb;
  if((Infos->N<0)||(Infos->N>=Bsp->NodeNb))
  { return BAD_PARM;}
  TNode=&(Bsp->Node[Infos->N]);
  if(What>0)
  {
	 TNode->Front= (UInt16)Infos->NodeFront;
	 TNode->Back=  (UInt16)Infos->NodeBack;
  }
  else
  {
	 Infos->NodeFront= (Int32)TNode->Front;
	 Infos->NodeBack=  (Int32)TNode->Back;
  }
  return 1;
}
/*
** Paint Nodes
** Static data to limit growth of stack during recursion
*/
static pBSPDEF BSPnodePBsp=NULL;
static Int8    BSPnodePColor=0;
static Int16 BSPnodePaint1(UInt16 Node)
{
  if(Node&0x8000) /*leaf*/
  {
	 Node=(Node^0xFFFF)&0x7FFF;
	 if(Node>=BSPnodePBsp->LeafNb)
	 { return -1;}
	 if(Node==0) /*ignore leaf 0*/
	 { return 1;}
	 return BSPleafPaint(BSPnodePBsp,Node,BSPnodePColor);
  }
  if(Node>=BSPnodePBsp->NodeNb)
  { return -1;}
  BSPnodePaint1(BSPnodePBsp->Node[Node].Front);
  return BSPnodePaint1(BSPnodePBsp->Node[Node].Back);
}
Int16 BSPnodePaint(pBSPDEF Bsp, Int32 Node)
{
  pQKNODE TNode;
  if((Node<0)||(Node>=Bsp->NodeNb))
  { return -1;}
  TNode = &(Bsp->Node[Node]);
  /*
  ** Paint front child in green, back child in red
  */
  BSPnodePBsp=Bsp;             /*constant during recursion*/
  BSPnodePColor=BMP_COL_RED;   /*constant during recursion*/
  BSPnodePaint1(TNode->Back);  /*recursive*/
  BSPnodePColor=BMP_COL_GREEN; /*constant during recursion*/
  BSPnodePaint1(TNode->Front); /*recursive*/
  /*
  ** Paint split faces in yellow
  */
  return BSPlfacPaint(Bsp,TNode->Face,TNode->FaceNb,BMP_COL_YELLOW);
}

/****************************************************\
*
*
* Quake  TexInfo
*
*
\****************************************************/


Int16 BSPtinfInit(pBSPDEF Bsp, pInt8 Lmp, Int32 LmpSz)
{
 Bsp->TinfNb= LmpSz/sizeof(QKTEXINFO);
 /*Calloc(Bsp->TinfNb, sizeof(QKTEXINFO));*/
 Bsp->Tinf=(pQKTEXINFO)Malloc(LmpSz);
 if(Bsp->Tinf==NULL) { return ERR_MEM;}
 Memcpy(Bsp->Tinf,Lmp,LmpSz);
 return 1;
}
/*
** Copy all tinfaces into lump
** if Lmp==NULL or LmpSz<0, return lump size
*/
Int32 BSPtinfCopyAll(pBSPDEF Bsp, pInt8 Lmp, Int32 LmpSz)
{
  Int32 Size;
  if((Bsp==NULL)||(Bsp->Tinf==NULL)||(Bsp->TinfNb<0))
  { return ERR_BUG; }
  /**/
  Size= Bsp->TinfNb*sizeof(QKTEXINFO);
  if((Lmp==NULL)||(LmpSz<0))
  { return Size; }
  /**/
  if(LmpSz<Size)
  { Size=LmpSz; }
  Memcpy(Lmp,Bsp->Tinf,Size);
  return Size;
}
/*
** Check TexInfo
*/
Int16 BSPtinfCheck(pBSPDEF Bsp, pTXTOBJ Txt)
{ Int32 n,dum; Int16 res;
  pQKTEXINFO TTinf;
  res=1;
  TXTprintName(Txt,QBSP_TINF,"Checking TexInfo",-1);
  for(TTinf=Bsp->Tinf,n=0; n<Bsp->TinfNb; n++,TTinf+=1)
  {
	 /*S vector+off*/
	 /*T vector+off*/
	 /*check texu*/
	 dum=((Int32)TTinf->Texu)&0xFFL;
	 if(dum> Bsp->TexuNb)
	 {
		TXTprintFrm(Txt," Tinf %ld: bad texture index",n);
		res=-1;
	 }
  }
  return res;
}
Int16 BSPtinfModify(pBSPDEF Bsp, pBSPINFOS Infos, Int16 What)
{
  pQKTEXINFO TTinf;
  pQKMIPTEX TTex;
  Infos->Nb=Bsp->TinfNb;
  if((Infos->N<0)||(Infos->N>=Bsp->TinfNb))
  { return BAD_PARM;}
  TTinf=&(Bsp->Tinf[Infos->N]);
  if(What>0)
  {
	 if((Infos->TinfTexu >= 0)&&(Infos->TinfTexu <Bsp->TexuNb))
	 { TTinf->Texu=  Infos->TinfTexu;}
	 TTinf->Soffs= Infos->TinfSoffs;
	 TTinf->Toffs= Infos->TinfToffs;
  }
  else
  {
	 Infos->TinfTexu=  TTinf->Texu;
	 Infos->TinfSoffs= TTinf->Soffs;
	 Infos->TinfToffs= TTinf->Toffs;
	 TTex=BSPtexuGetI(Bsp, (Int16)TTinf->Texu);
	 if(TTex!=NULL)
	 { Strncpy(Infos->TexuName,TTex->Name,sizeof(Infos->TexuName));}
	 else
	 { Memset(Infos->TexuName,' ',sizeof(Infos->TexuName));}
  }
  return 1;
}





/****************************************************\
*
*
* Quake  Faces
*
*
\****************************************************/


Int16 BSPfaceInit(pBSPDEF Bsp, pInt8 Lmp, Int32 LmpSz)
{
 Bsp->FaceNb= LmpSz/sizeof(QKFACE);
 /*Calloc(Bsp->FaceNb, sizeof(QKFACE));*/
 Bsp->Face=(pQKFACE)Malloc(LmpSz);
 if(Bsp->Face==NULL) { return ERR_MEM;}
 Memcpy(Bsp->Face,Lmp,LmpSz);
 return 1;
}
/*
** Copy all faces into lump
** if Lmp==NULL or LmpSz<0, return lump size
*/
Int32 BSPfaceCopyAll(pBSPDEF Bsp, pInt8 Lmp, Int32 LmpSz)
{
  Int32 Size;
  if((Bsp==NULL)||(Bsp->Face==NULL)||(Bsp->FaceNb<0))
  { return ERR_BUG; }
  /**/
  Size= Bsp->FaceNb*sizeof(QKFACE);
  if((Lmp==NULL)||(LmpSz<0))
  { return Size; }
  /**/
  if(LmpSz<Size)
  { Size=LmpSz; }
  Memcpy(Lmp,Bsp->Face,Size);
  return Size;
}
/*
** Check Face
*/
Int16 BSPfaceCheck(pBSPDEF Bsp, pTXTOBJ Txt)
{ Int32 n; Int16 res;
  pQKFACE TFace;
  res=1;
  TXTprintName(Txt,QBSP_FACE,"Checking Faces",-1);
  for(TFace=Bsp->Face,n=0; n<Bsp->FaceNb; n++,TFace+=1)
  {
	 /*plane*/
	 if(TFace->Plan > Bsp->PlanNb)
	 {
		TXTprintFrm(Txt," Face %ld: bad plane index",n);
		res = -1;
	 }
	 /*check texinfo reference*/

	 if(TFace->Tinf > Bsp->TinfNb)
	 {
		TXTprintFrm(Txt," Face %ld: bad texture info",n);
		res=-1;
	 }
	 if(TFace->LEdg > Bsp->LEdgNb)
	 {
		TXTprintFrm(Txt," Face %ld: bad edge list index",n);
		res=-1;
	 }
	 if((TFace->LEdg+((Int32)TFace->LEdgNb)) > Bsp->LEdgNb)
	 {
		TXTprintFrm(Txt," Face %ld: bad edge list nb",n);
		res=-1;
	 }
	 /*check light map index*/
	 if((TFace->LiteMap>=0)&&(TFace->LiteMap > Bsp->LiteSz))
	 {
		TXTprintFrm(Txt," Face %ld: bad ligthmap index",n);
		res=-1;
	 }
	 if(TFace->Lite2!=(UInt8)0xFF)
	 {
		TXTprintFrm(Txt," Face %ld: Light2 is used",n);
		res=-1;
	 }
  }
  return res;
}
Int16 BSPfaceModify(pBSPDEF Bsp, pBSPINFOS Infos, Int16 What)
{
  pQKFACE TFace;
  Int32 N, Nb;
  Infos->Nb=Bsp->FaceNb;
  if((Infos->N<0)||(Infos->N>=Bsp->FaceNb))
  { return BAD_PARM;}
  TFace=&(Bsp->Face[Infos->N]);
  if(What>0)
  {
	 /*TFace->Tinf=  Infos->Tinf;*/
	 TFace->LType= (UInt8)Infos->FaceLType;
	 TFace->Shadow=(UInt8)Infos->FaceShadow;
	 TFace->Lite1= (UInt8)Infos->FaceLite1;
	 TFace->Lite1= (UInt8)Infos->FaceLite2;
  }
  else
  {
	 Infos->FaceLType= (TFace->LType&0xFF);
	 Infos->FaceShadow=(TFace->Shadow&0xFF);
	 Infos->FaceLite1= (TFace->Lite1&0xFF);
	 Infos->FaceLite2= (TFace->Lite2&0xFF);
	 /* Get texture infos*/
	 N = Infos->N; /*save*/
	 Nb= Infos->Nb;
	 Infos->N = TFace->Tinf;
	 BSPtinfModify(Bsp,Infos,What);
	 Infos->N = N;  /*restore*/
	 Infos->Nb= Nb;
  }
  return 1;
}
/*
** Paint one face
*/
static Int16 BSPfacePaint1(pBSPDEF Bsp, pQKFACE TFace, Int8 Color)
{
  Int32 l,edge;
  pQKLEDGE TLEdg;
  /*Int16 Inverted;*/
  /*
  ** Face
  */
  if(((TFace->LEdg+TFace->LEdgNb) > Bsp->LEdgNb))
  { return BAD_PARM;}
  /*
  ** Scan the list of edges
  */
  for(l=0, TLEdg=&(Bsp->LEdg[TFace->LEdg]); l<TFace->LEdgNb; l++, TLEdg+=1)
  {
	 edge=(*TLEdg);
	 /*Inverted = (edge<0)? 1: 0;*/
	 edge     = (edge<0)? -edge: edge;
	 if((edge<0)||(edge>=Bsp->EdgeNb))
	 { continue; }
	 Bsp->ColEdge[edge]=Color;
  }
  return 1;
}
Int16 BSPfacePaint(pBSPDEF Bsp, Int32 Face, Int8 Color)
{
  /**/
  if((Bsp==NULL)||(Bsp->Face==NULL)||(Bsp->LEdg==NULL)||(Bsp->ColEdge==NULL))
  { return ERRfault(ERR_BUG);}
  if((Face<0)||(Face>=Bsp->FaceNb))
  { return ERRfault(BAD_PARM); }
  /*
  ** Face
  */
  return BSPfacePaint1(Bsp,&(Bsp->Face[Face]),Color);
}
/*
** Paint all faces oriented toward Observer
*/
Int16 BSPfacePaintAll(pBSPDEF Bsp, Int8 Color)
{
  Int32 n;
  pQKFACE TFace;
  Bool Show;
  if((Bsp==NULL)||(Bsp->Face==NULL)||(Bsp->LEdg==NULL)||(Bsp->ColEdge==NULL))
  { return ERRfault(ERR_BUG);}
  /**/
  for(n=0,TFace=Bsp->Face; n<Bsp->FaceNb; n++,TFace+=1)
  {
	 Show = (Bsp->CachPlan[TFace->Plan]>0)? TRUE:FALSE;
	 if(TFace->Side!=0) Show= !Show;
	 if(Show)
	 {
		if(BSPfacePaint1(Bsp,TFace,Color)<0)
		{ return ERR_SURE;}
	 }
  }
  return 1;
}
/*
** Paint list of faces
*/
Int16 BSPfacePaintList(pBSPDEF Bsp, Int32 Face, Int32 FaceNb, Int8 Color)
{
  Int32 n;
  pQKFACE TFace;
  if((Bsp==NULL)||(Bsp->Face==NULL)||(Bsp->LEdg==NULL)||(Bsp->ColEdge==NULL))
  { return ERRfault(ERR_BUG);}
  if((Face<0)||((Face+FaceNb)>Bsp->FaceNb))
  { return ERRfault(BAD_PARM);}
  /**/
  for(n=0, TFace=&(Bsp->Face[Face]); n<FaceNb; n++,TFace+=1)
  {
	 if(BSPfacePaint1(Bsp,TFace,Color)<0)
	 { return ERR_SURE;}
  }
  return 1;
}


/*
** Hack, paint a limited number of edges
*/
#if 1
Int16 BSPfacePaintSpecial(pBSPDEF Bsp, Int32 Face, Int32 Nb,Int8 Color)
{
  Int32 l,edge;
  pQKLEDGE TLEdg;
  pQKFACE TFace;
  /**/
  if((Bsp==NULL)||(Bsp->Face==NULL)||(Bsp->LEdg==NULL)||(Bsp->ColEdge==NULL))
  { return ERRfault(ERR_BUG);}
  if((Face<0)||(Face>=Bsp->FaceNb))
  { return ERRfault(BAD_PARM); }
  /*
  ** TexInfo
  */
  TFace=&(Bsp->Face[Face]);
  /*
  ** TexInfo
  */
  if(((TFace->LEdg+TFace->LEdgNb) > Bsp->LEdgNb))
  { return BAD_PARM;}
  /*
  ** Scan the list of edges
  */

  for(l=0, TLEdg=&(Bsp->LEdg[TFace->LEdg]); l<TFace->LEdgNb; l++, TLEdg+=1)
  {
	 /*
	 ** Limit paint
	 */
	 if(l>=Nb) break;

	 edge=(*TLEdg);
	 /*Inverted = (edge<0)? 1: 0;*/
	 edge     = (edge<0)? -edge: edge;
	 if((edge<0)||(edge>=Bsp->EdgeNb))
	 { continue; }
	 Bsp->ColEdge[edge]=Color;
  }
  return 1;
}
#endif



/****************************************************\
*
*
* Quake Lite
*
*
\****************************************************/


Int16 BSPliteInit(pBSPDEF Bsp, pInt8 Lmp, Int32 LmpSz)
{
 Bsp->LiteSz= LmpSz;
 /*Calloc(Bsp->LiteSz, sizeof(Int8));*/
 Bsp->Lite=(pInt8)Malloc(LmpSz);
 if(Bsp->Lite==NULL) { return ERR_MEM;}
 Memcpy(Bsp->Lite,Lmp,LmpSz);
 return 1;
}
/*
** Copy all lightmaps into lump
** if Lmp==NULL or LmpSz<0, return lump size
*/
Int32 BSPliteCopyAll(pBSPDEF Bsp, pInt8 Lmp, Int32 LmpSz)
{
  Int32 Size;
  if((Bsp==NULL)||(Bsp->Lite==NULL)||(Bsp->LiteSz<0))
  { return ERR_BUG; }
  /**/
  Size= Bsp->LiteSz;
  if((Lmp==NULL)||(LmpSz<0))
  { return Size; }
  /**/
  if(LmpSz<Size)
  { Size=LmpSz; }
  Memcpy(Lmp,Bsp->Lite,Size);
  return Size;
}
/*
** Check Lightmaps
*/
Int16 BSPliteCheck(pBSPDEF Bsp, pTXTOBJ Txt)
{
  (void)Bsp;
  (void)Txt;  /*nothing to check*/
  return 1;
}


/****************************************************\
*
*
* Quake Bounding Nodes
*
*
\****************************************************/


Int16 BSPclipInit(pBSPDEF Bsp, pInt8 Lmp, Int32 LmpSz)
{
 Bsp->ClipNb= LmpSz/sizeof(QKCLIPNOD);
 /*Calloc(Bsp->ClipNb, sizeof(QKCLIPNOD));*/
 Bsp->Clip=(pQKCLIPNOD)Malloc(LmpSz);
 if(Bsp->Clip==NULL) { return ERR_MEM;}
 Memcpy(Bsp->Clip,Lmp,LmpSz);
 return 1;
}
/*
** Copy all Clipesinto lump
** if Lmp==NULL or LmpSz<0, return lump size
*/
Int32 BSPclipCopyAll(pBSPDEF Bsp, pInt8 Lmp, Int32 LmpSz)
{
  Int32 Size;
  if((Bsp==NULL)||(Bsp->Clip==NULL)||(Bsp->ClipNb<0))
  { return ERR_BUG; }
  /**/
  Size= Bsp->ClipNb*sizeof(QKCLIPNOD);
  if((Lmp==NULL)||(LmpSz<0))
  { return Size; }
  /**/
  if(LmpSz<Size)
  { Size=LmpSz; }
  Memcpy(Lmp,Bsp->Clip,Size);
  return Size;
}
/*
** Check Child
*/
static BSPclipCheckChild(pBSPDEF Bsp, Int16 Child)
{
  /*0xFFFF external, 0xFFFE internal*/
  if((Child==-1)||(Child==-2))
  { return 1;}
  return ((Child<0)||(Child>Bsp->ClipNb))? -1:1;
}
/*
** Check Clipes
*/
Int16 BSPclipCheck(pBSPDEF Bsp, pTXTOBJ Txt)
{ Int32 n; Int16 res;
  pQKCLIPNOD TClip;
  res=1;
  TXTprintName(Txt,QBSP_CLIP,"Checking Bound Nodes",-1);
  for(TClip=Bsp->Clip,n=0; n<Bsp->ClipNb; n++,TClip+=1)
  {
	 if(TClip->Plan> Bsp->PlanNb)
	 {
		TXTprintFrm(Txt," Clip %ld: bad plane index",n);
		res=-1;
	 }
	 if(BSPclipCheckChild(Bsp,TClip->Front)<0)
	 {
		TXTprintFrm(Txt," Clip %ld: bad front child",n);
		res=-1;
	 }
	 if(BSPclipCheckChild(Bsp,TClip->Back)<0)
	 {
		TXTprintFrm(Txt," Clip %ld: bad back child",n);
		res=-1;
	 }
  }
  return res;
}
Int16 BSPclipModify(pBSPDEF Bsp, pBSPINFOS Infos, Int16 What)
{
  pQKCLIPNOD TClip;
  Infos->Nb=Bsp->ClipNb;
  if((Infos->N<0)||(Infos->N>=Bsp->ClipNb))
  { return BAD_PARM;}
  TClip=&(Bsp->Clip[Infos->N]);
  if(What>0)
  {
	 TClip->Front= (UInt16)Infos->NodeFront;
	 TClip->Back=  (UInt16)Infos->NodeBack;
  }
  else
  {
	 Infos->NodeFront= (Int32)TClip->Front;
	 Infos->NodeBack=  (Int32)TClip->Back;
  }
  return 1;
}

/****************************************************\
*
*
* Quake Leaves
*
*
\****************************************************/


Int16 BSPleafInit(pBSPDEF Bsp, pInt8 Lmp, Int32 LmpSz)
{
 Bsp->LeafNb= LmpSz/sizeof(QKLEAF);
 /*Calloc(Bsp->LeafNb, sizeof(QKLEAF));*/
 Bsp->Leaf=(pQKLEAF)Malloc(LmpSz);
 if(Bsp->Leaf==NULL) { return ERR_MEM;}
 Memcpy(Bsp->Leaf,Lmp,LmpSz);
 return 1;
}
/*
** Copy all leaves into lump
** if Lmp==NULL or LmpSz<0, return lump size
*/
Int32 BSPleafCopyAll(pBSPDEF Bsp, pInt8 Lmp, Int32 LmpSz)
{
  Int32 Size;
  if((Bsp==NULL)||(Bsp->Leaf==NULL)||(Bsp->LeafNb<0))
  { return ERR_BUG; }
  /**/
  Size= Bsp->LeafNb*sizeof(QKLEAF);
  if((Lmp==NULL)||(LmpSz<0))
  { return Size; }
  /**/
  if(LmpSz<Size)
  { Size=LmpSz; }
  Memcpy(Lmp,Bsp->Leaf,Size);
  return Size;
}
/*
** Get color of leaf
*/
Int16 BSPleafColor(Int32 Type)
{ switch(Type)
  { case -1: return BMP_COL_DARKGREY;/*ordinary*/
	 case -2: return BMP_COL_PURPLE;    /*leaf does not exist*/
	 case -3: return BMP_COL_GREEN;		/*water*/
	 case -4: return BMP_COL_DARKGREEN;	/*acid*/
	 case -5: return BMP_COL_RED;			/*lava*/
	 case -6: return BMP_COL_CYAN;		/*water*/
	 case -7: return BMP_COL_CYAN;		/*water*/
	 case -8: return BMP_COL_CYAN;		/*water*/
  }
  return -1;
}
/*
** Check Leaves
*/
Int16 BSPleafCheck(pBSPDEF Bsp, pTXTOBJ Txt)
{ Int32 n; Int16 res;
  pQKLEAF TLeaf;
  res=1;
  TXTprintName(Txt,QBSP_LEAF,"Checking Leaves",-1);
  for(TLeaf=Bsp->Leaf,n=0; n<Bsp->LeafNb; n++,TLeaf+=1)
  {
	 /*check texu*/
	 if(BSPleafColor(TLeaf->Type)<0)
	 {
		TXTprintFrm(Txt," Leaf %ld: invalid type %ld",n,TLeaf->Type);
	 }
	 if((TLeaf->Visi<-1)||(TLeaf->Visi>Bsp->VisiSz))
	 {
		TXTprintFrm(Txt," Leaf %ld: bad visibility list index",n);
		res=-1;
	 }
	 if(TLeaf->LFac>Bsp->LFacNb)
	 {
		TXTprintFrm(Txt," Leaf %ld: bad face list index",n);
		res=-1;
	 }
	 if((TLeaf->LFac+TLeaf->LFacNb)>Bsp->LFacNb)
	 {
		TXTprintFrm(Txt," Leaf %ld: bad face list nb",n);
		res=-1;
	 }
  }
  return res;
}
Int16 BSPleafModify(pBSPDEF Bsp, pBSPINFOS Infos, Int16 What)
{
  pQKLEAF TLeaf;
  Infos->Nb=Bsp->LeafNb;
  if((Infos->N<0)||(Infos->N>=Bsp->LeafNb))
  { return BAD_PARM;}
  TLeaf=&(Bsp->Leaf[Infos->N]);
  if(What>0)
  {
	 TLeaf->Type= Infos->LeafType;
  }
  else
  {
	 Infos->LeafType= TLeaf->Type;
  }
  return 1;
}
Int16 BSPleafPaint(pBSPDEF Bsp, Int32 Leaf, Int8 Color)
{
  pQKLEAF TLeaf;
  if((Bsp==NULL)||(Bsp->Leaf==NULL))
  { return ERRfault(ERR_BUG);}
  if((Leaf<0)||(Leaf>=Bsp->LeafNb))
  { return ERRfault(BAD_PARM); }
  /*
  ** Leaf
  */
  TLeaf=&(Bsp->Leaf[Leaf]);
  return BSPlfacPaint(Bsp, TLeaf->LFac, TLeaf->LFacNb, Color);
}

Int16 BSPleafPaintType(pBSPDEF Bsp)
{
  Int32 n;
  pQKLEAF TLeaf;
  Int8 Color;
  if((Bsp==NULL)||(Bsp->Leaf==NULL))
  { return ERRfault(ERR_BUG);}
  /*
  ** Leaf
  */
  for(n=0,TLeaf=Bsp->Leaf; n<Bsp->LeafNb; n++,TLeaf+=1)
  {
	 if(TLeaf->Type==-1) continue;
	 Color=BSPleafColor(TLeaf->Type);
	 if(BSPlfacPaint(Bsp, TLeaf->LFac, TLeaf->LFacNb, Color)<0)
	 { return ERR_SURE;}
  }
  return 1;
}

/****************************************************\
*
*
* Quake List Tinfs
*
*
\****************************************************/


Int16 BSPlfacInit(pBSPDEF Bsp, pInt8 Lmp, Int32 LmpSz)
{
 Bsp->LFacNb= LmpSz/sizeof(QKLFACE);
 /*Calloc(Bsp->LSrfNb, sizeof(QKLFACE));*/
 Bsp->LFac=(pQKLFACE)Malloc(LmpSz);
 if(Bsp->LFac==NULL) { return ERR_MEM;}
 Memcpy(Bsp->LFac,Lmp,LmpSz);
 return 1;
}
/*
** Copy all LFac into lump
** if Lmp==NULL or LmpSz<0, return lump size
*/
Int32 BSPlfacCopyAll(pBSPDEF Bsp, pInt8 Lmp, Int32 LmpSz)
{
  Int32 Size;
  if((Bsp==NULL)||(Bsp->LFac==NULL)||(Bsp->LFacNb<0))
  { return ERR_BUG; }
  /**/
  Size= Bsp->LFacNb*sizeof(QKLFACE);
  if((Lmp==NULL)||(LmpSz<0))
  { return Size; }
  /**/
  if(LmpSz<Size)
  { Size=LmpSz; }
  Memcpy(Lmp,Bsp->LFac,Size);
  return Size;
}
/*
** Check List TexInfo
*/
Int16 BSPlfacCheck(pBSPDEF Bsp, pTXTOBJ Txt)
{ Int32 n; Int16 res;
  pQKLFACE TLFac;
  res=1;
  TXTprintName(Txt,QBSP_LFAC,"Checking Face List",-1);
  for(TLFac=Bsp->LFac,n=0; n<Bsp->LFacNb; n++,TLFac+=1)
  {
	 if((TLFac[0]<0)||(TLFac[0]>Bsp->FaceNb))
	 {
		TXTprintFrm(Txt," LFac %ld: bad face index",n);
		res=-1;
	 }  
  }
  return res;
}
/*
** Paint a list of tinfaces
*/
Int16 BSPlfacPaint(pBSPDEF Bsp, Int32 LFac, Int32 LFacNb, Int8 Color)
{
  Int32 s;
  pQKLFACE TLFac;
  /**/
  if((Bsp==NULL)||(Bsp->LFac==NULL))
  { return ERRfault(ERR_BUG);}
  if((LFac<0)||((LFac+LFacNb)>Bsp->LFacNb))
  { return ERRfault(BAD_PARM); }
  /*
  ** Paint the list of tinface
  */
  for(s=0, TLFac=&(Bsp->LFac[LFac]); s<LFacNb; s++, TLFac+=1)
  {
	 if(BSPfacePaint(Bsp,(*TLFac),Color)<0)
	 { return ERR_SURE;}
  }
  return 1;
}
/****************************************************\
*
*
* Quake Edges
*
*
\****************************************************/


Int16 BSPedgeInit(pBSPDEF Bsp, pInt8 Lmp, Int32 LmpSz)
{
 Bsp->EdgeNb= LmpSz/sizeof(QKEDGE);
 /*Calloc(Bsp->EdgeNb, sizeof(QKEDGE));*/
 Bsp->Edge=(pQKEDGE)Malloc(LmpSz);
 if(Bsp->Edge==NULL) { return ERR_MEM;}
 Memcpy(Bsp->Edge,Lmp,LmpSz);
 return 1;
}
/*
** Copy all edges into lump
** if Lmp==NULL or LmpSz<0, return lump size
*/
Int32 BSPedgeCopyAll(pBSPDEF Bsp, pInt8 Lmp, Int32 LmpSz)
{
  Int32 Size;
  if((Bsp==NULL)||(Bsp->Edge==NULL)||(Bsp->EdgeNb<0))
  { return ERR_BUG; }
  /**/
  Size= Bsp->EdgeNb*sizeof(QKEDGE);
  if((Lmp==NULL)||(LmpSz<0))
  { return Size; }
  /**/
  if(LmpSz<Size)
  { Size=LmpSz; }
  Memcpy(Lmp,Bsp->Edge,Size);
  return Size;
}
/*
** Check Edges
*/
Int16 BSPedgeCheck(pBSPDEF Bsp, pTXTOBJ Txt)
{ Int32 n; Int16 res;
  pQKEDGE TEdge;
  res=1;
  TXTprintName(Txt,QBSP_EDGE,"Checking Edges",-1);
  for(TEdge=Bsp->Edge,n=0; n<Bsp->EdgeNb; n++,TEdge+=1)
  {
	 if((TEdge->Vrtx[0]>Bsp->VrtxNb)||(TEdge->Vrtx[1]>Bsp->VrtxNb))
	 {
		TXTprintFrm(Txt," Edge %ld: bad vertex index",n);
		res=-1;
	 }
  }
  return res;
}
Int16 BSPedgePaintAll(pBSPDEF Bsp,Int16 Color)
{
  if(Bsp->ColEdge==NULL)
  { return ERRfault(ERR_BUG); }
  /*
  ** 0 = transparent color
  */
  Color= (Color<0)? 0: Color&0xFF;
  Memset(Bsp->ColEdge,(Int8)Color,Bsp->EdgeNb*sizeof(Int8));
  return 1;
}
Int16 BSPedgeLineAll(pBSPDEF Bsp, pBMPOBJ Bmp, pBOUND Bound)
{
  Int32 n;
  pQKEDGE TEdge;
  pInt8 TColor;
  /**/
  if((Bsp->Edge==NULL)||(Bsp->CachVrtx==NULL)||(Bsp->ColEdge==NULL))
  { return ERRfault(ERR_BUG); }
  /*
  ** For all edges
  */
  for(n=0, TEdge=Bsp->Edge, TColor=Bsp->ColEdge;
		n<Bsp->EdgeNb; n++, TEdge+=1, TColor+=1)
  {
	 /*
	 ** Eliminate invisible edges
	 */
	 if(*TColor==0) continue;
	 /*
	 ** show edges
	 */
	 if(BSPcachLineVrtx(Bmp, Bound,
							  &(Bsp->CachVrtx[TEdge->Vrtx[0]]),
							  &(Bsp->CachVrtx[TEdge->Vrtx[1]]),
							  *TColor)<0)
	 { return ERR_SURE;}
  }
  return 1;
}

/****************************************************\
*
*
* Quake List Edges
*
*
\****************************************************/


Int16 BSPledgInit(pBSPDEF Bsp, pInt8 Lmp, Int32 LmpSz)
{
 Bsp->LEdgNb= LmpSz/sizeof(QKLEDGE);
 /*Calloc(Bsp->LEdgNb, sizeof(QKLEDGE));*/
 Bsp->LEdg=(pQKLEDGE)Malloc(LmpSz);
 if(Bsp->LEdg==NULL) { return ERR_MEM;}
 Memcpy(Bsp->LEdg,Lmp,LmpSz);
 return 1;
}
/*
** Copy all into lump
** if Lmp==NULL or LmpSz<0, return lump size
*/
Int32 BSPledgCopyAll(pBSPDEF Bsp, pInt8 Lmp, Int32 LmpSz)
{
  Int32 Size;
  if((Bsp==NULL)||(Bsp->LEdg==NULL)||(Bsp->LEdg<0))
  { return ERR_BUG; }
  /**/
  Size= Bsp->LEdgNb*sizeof(QKLEDGE);
  if((Lmp==NULL)||(LmpSz<0))
  { return Size; }
  /**/
  if(LmpSz<Size)
  { Size=LmpSz; }
  Memcpy(Lmp,Bsp->LEdg,Size);
  return Size;
}
/*
** Check List Edges
*/
Int16 BSPledgCheck(pBSPDEF Bsp, pTXTOBJ Txt)
{ Int32 n,edge; Int16 res;
  pQKLEDGE TLEdg;
  res=1;
  TXTprintName(Txt,QBSP_LEDG,"Checking Edges List",-1);
  for(TLEdg=Bsp->LEdg,n=0; n<Bsp->LEdgNb; n++,TLEdg+=1)
  {
	 edge=(Int32)TLEdg[0];
	 edge = (edge<0)? -edge : edge;
	 if(edge<=0)
	 {
		TXTprintFrm(Txt," LEdg %ld: edge index is 0",n);
		res=-1;
	 }
	 if(edge>= Bsp->EdgeNb)
	 {
		TXTprintFrm(Txt," LEdg %ld: bad edge index",n);
		res=-1;
	 }
  }
  return res;
}

/****************************************************\
*
*
* Quake Modls
*
*
\****************************************************/


Int16 BSPmodlInit(pBSPDEF Bsp, pInt8 Lmp, Int32 LmpSz)
{
 Bsp->ModlNb= LmpSz/sizeof(QKMODEL);
 /*Calloc(Bsp->ModlNb, sizeof(QKMODL));*/
 Bsp->Modl=(pQKMODEL)Malloc(LmpSz);
 if(Bsp->Modl==NULL) { return ERR_MEM;}
 Memcpy(Bsp->Modl,Lmp,LmpSz);
 return 1;
}
/*
** Copy all modls into lump
** if Lmp==NULL or LmpSz<0, return lump size
*/
Int32 BSPmodlCopyAll(pBSPDEF Bsp, pInt8 Lmp, Int32 LmpSz)
{
  Int32 Size;
  if((Bsp==NULL)||(Bsp->Modl==NULL)||(Bsp->ModlNb<0))
  { return ERR_BUG; }
  /**/
  Size= Bsp->ModlNb*sizeof(QKMODEL);
  if((Lmp==NULL)||(LmpSz<0))
  { return Size; }
  /**/
  if(LmpSz<Size)
  { Size=LmpSz; }
  Memcpy(Lmp,Bsp->Modl,Size);
  return Size;
}
/*
** Check Modl
*/
Int16 BSPmodlCheck(pBSPDEF Bsp, pTXTOBJ Txt)
{ Int32 n; Int16 res;
  pQKMODEL TModl;
  res=1;
  TXTprintName(Txt,QBSP_MODL,"Checking Modls",-1);
  for(TModl=Bsp->Modl,n=0; n<Bsp->ModlNb; n++,TModl+=1)
  {
	 if(TModl->Node>Bsp->NodeNb)
	 {
		TXTprintFrm(Txt," Modl %ld: bad node index",n);
		res=-1;
	 }
	 if(TModl->Clip>Bsp->ClipNb)
	 {
		TXTprintFrm(Txt," Modl %ld: bad clip node index",n);
		res=-1;
	 }
	 if(TModl->LeafNb>Bsp->LeafNb)
	 {
		TXTprintFrm(Txt," Modl %ld: bad leaves nb",n);
		res=-1;
	 }
	 if(TModl->Face>Bsp->FaceNb)
	 {
		TXTprintFrm(Txt," Modl %ld: bad face index",n);
		res=-1;
	 }
	 if((TModl->Face+TModl->FaceNb) > Bsp->FaceNb)
	 {
		TXTprintFrm(Txt," Modl %ld: bad face nb",n);
		res=-1;
	 }
  }
  return res;
}
Int16 BSPmodlModify(pBSPDEF Bsp, pBSPINFOS Infos, Int16 What)
{
  pQKMODEL TModl;
  Infos->Nb=Bsp->ModlNb;
  if((Infos->N<0)||(Infos->N>=Bsp->ModlNb))
  { return BAD_PARM;}
  TModl=&(Bsp->Modl[Infos->N]);
  if(What>0)
  {
	 if((Infos->ModlNode>0)&&(Infos->ModlNode<Bsp->NodeNb))
	 { TModl->Node= Infos->ModlNode;}
	 if(Infos->ModlClip<Bsp->ClipNb)
	 { TModl->Clip= Infos->ModlClip;}
	 if(Infos->ModlClip2<Bsp->ClipNb)
	 { TModl->Clip2= Infos->ModlClip2;}
	 if(Infos->ModlClip3<Bsp->ClipNb)
	 { TModl->Clip3= Infos->ModlClip3;}
	 if(Infos->ModlLeafNb>0)
	 { TModl->LeafNb= Infos->ModlLeafNb;}
  }
  else
  {
	 Infos->ModlNode= TModl->Node;
	 Infos->ModlClip= TModl->Clip;
	 Infos->ModlClip2= TModl->Clip2;
	 Infos->ModlClip3= TModl->Clip3;
	 Infos->ModlLeafNb= TModl->LeafNb;
  }
  return 1;
}
/*
** Paint Modl
*/
Int16 BSPmodlPaint(pBSPDEF Bsp, Int32 Modl, Int8 Color)
{
  pQKMODEL TModl;
  if((Bsp==NULL)||(Bsp->Modl==NULL))
  { return ERRfault(ERR_BUG);}
  if((Modl<0)||(Modl>=Bsp->ModlNb))
  { return ERRfault(BAD_PARM); }
  /*
  ** Modl
  */
  TModl=&(Bsp->Modl[Modl]);
  return BSPfacePaintList(Bsp, TModl->Face, TModl->FaceNb, Color);
}
#endif /*DLLFORQUAK*/
