// Emacs style mode select   -*- C++ -*-
//-----------------------------------------------------------------------------
//
// $Id:$
//
// Copyright (C) 1993-1996 by id Software, Inc.
//
// This source is available for distribution and/or modification
// only under the terms of the DOOM Source Code License as
// published by id Software. All rights reserved.
//
// The source is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// FITNESS FOR A PARTICULAR PURPOSE. See the DOOM Source Code License
// for more details.
//
// $Log:$
//
// DESCRIPTION:
//	DOOM selection menu, options, episode etc.
//	Sliders and icons. Kinda widget stuff.
//
// NOTE:
//	All V_DrawPatchDirect () has been replaced by V_DrawScaledPatch ()
//	so that the menu is scaled to the screen size. The scaling is always
//	an integer multiple of the original size, so that the graphics look
//	good.
//
//-----------------------------------------------------------------------------

static const char
rcsid[] = "$Id: m_menu.c,v 1.7 1997/02/03 22:45:10 b1 Exp $";

#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdlib.h>
#include <ctype.h>


#include "doomdef.h"
#include "dstrings.h"

#include "d_main.h"

#include "i_system.h"
#include "i_video.h"
#include "z_zone.h"
#include "v_video.h"
#include "w_wad.h"

#include "r_local.h"


#include "hu_stuff.h"

#include "g_game.h"

#include "m_argv.h"
#include "m_swap.h"

#include "s_sound.h"

#include "doomstat.h"

// Data.
#include "sounds.h"

#include "m_menu.h"

#include "screen.h"	//added:01-02-98:setmodeneeded
#include "g_input.h"	//added:07-02-98:setting game controls at setup menu
#include "d_net.h"	// added 11-2-98 for sendconfigplayer


extern boolean		message_dontfuckwithme;

extern boolean		chat_on;		// in heads-up code


// Show messages has default, 0 = off, 1 = on
int			showMessages;


// Blocky mode, has default, 0 = high, 1 = normal
int			detailLevel;
int			screenblocks;		// has default


//added:10-02-98: some toggles set from the menu
int			speed_locked;	    // added 3-1-98
int			mouseaiming_lock;   // (1) always, (0) momentary, set by menu
int			invert_mouse;	    // only for mouseaiming !


// temp for screenblocks (0-9)
int			screenSize;

// -1 = no quicksave slot picked!
int			quickSaveSlot;

 // 1 = message to be printed
int			messageToPrint;
// ...and here is the message string!
char*			messageString;

// message x & y
int			messx;
int			messy;
int			messageLastMenuActive;

// timed message = no input from user
boolean 		messageNeedsInput;

//added:07-02-98: points the event which responded the message,
//		  this is set before calling the messageRoutine.
static	event_t*   messageEvent;
static	boolean    messageNeedsEvent=false;   //aaarghl it sucks...I'm tired!

void	(*messageRoutine)(int response);

#define SAVESTRINGSIZE	24

char gammamsg[5][26] =
{
    GAMMALVL0,
    GAMMALVL1,
    GAMMALVL2,
    GAMMALVL3,
    GAMMALVL4
};

// we are going to be entering a savegame string
int			saveStringEnter;
int			saveSlot;	// which slot to save in
int			saveCharIndex;	// which char we're editing
// old save description before edit
char			saveOldString[SAVESTRINGSIZE];

boolean 		inhelpscreens;
boolean 		menuactive;

#define SKULLXOFF		-32
#define LINEHEIGHT		16


//added:10-02-98: drawskullcursor is true by default, for most of the
//		  menus which doesn't draw a specific cursor themselves
//		  set this to false in your menu drawer routine if
//		  you draw a cursor yourself
boolean drawskullcursor;

byte*	whitemap;     //set by M_SetupWhiteColormap(), from M_Init()

extern boolean		sendpause;
char			savegamestrings[10][SAVESTRINGSIZE];

char	endstring[160];


//
// MENU TYPEDEFS
//
typedef struct
{
    // 0 = no cursor here, 1 = ok, 2 = arrows ok
    // 3 = custom key handler for this menuitem
    short	status;

    char	name[10];

    // choice = menu item #.
    // if status = 2,
    //	  choice=0:leftarrow,1:rightarrow
    void	(*routine)(int choice);

    // hotkey in menu
    unsigned char   alphaKey;
} menuitem_t;


typedef struct menu_s
{
    short		numitems;	// # of menu items
    struct menu_s*	prevMenu;	// previous menu
    menuitem_t* 	menuitems;	// menu items
    void		(*routine)();	// draw routine
    short		x;
    short		y;		// x,y of menu
    short		lastOn; 	// last item user was on in menu
} menu_t;

short		itemOn; 		// menu item skull is on
short		skullAnimCounter;	// skull animation counter
short		whichSkull;		// which skull to draw

// graphic name of skulls
// warning: initializer-string for array of chars is too long
char	skullName[2][/*8*/9] = {"M_SKULL1","M_SKULL2"};

// current menudef
menu_t* currentMenu;

//
// PROTOTYPES
//
void M_DrawSaveLoadBorder(int x,int y);
void M_SetupNextMenu(menu_t *menudef);

void M_DrawTextBox (int x, int y, int width, int lines);     //added:06-02-98:
void M_DrawThermo(int x,int y,int thermWidth,int thermDot);
void M_DrawEmptyCell(menu_t *menu,int item);
void M_DrawSelCell(menu_t *menu,int item);
void M_CentreText(int y, char* string);        //added:30-01-98:writetext centered

void M_StartControlPanel(void);
void M_StartMessage(char *string,void *routine,boolean input);
void M_StopMessage(void);
void M_ClearMenus (void);


//===========================================================================
//MAIN MENU
//===========================================================================

// protos
void M_DrawMainMenu(void);

void M_SinglePlayer(int choice);
void M_MultiPlayer(int choice);
void M_Options(int choice);
void M_ReadThis(int choice);
void M_ReadThis2(int choice);
void M_QuitDOOM(int choice);

enum
{
    singleplr = 0,
    multiplr,
    options,
    readthis,
    quitdoom,
    main_end
} main_e;

menuitem_t MainMenu[]=
{
    {1,"M_SINGLE",M_SinglePlayer ,'s'},
    {1,"M_MULTI" ,M_MultiPlayer  ,'m'},
    {1,"M_OPTION",M_Options	 ,'o'},
    {1,"M_RDTHIS",M_ReadThis	 ,'r'},  // Another hickup with Special edition.
    {1,"M_QUITG" ,M_QuitDOOM	 ,'q'}
};

menu_t	MainDef =
{
    main_end,
    NULL,
    MainMenu,
    M_DrawMainMenu,
    97,64,
    0
};


//
// M_DrawMainMenu
//
void M_DrawMainMenu(void)
{
    patch_t* p = W_CacheLumpName("M_DOOM",PU_CACHE);
    V_DrawScaledPatch ((BASEVIDWIDTH-p->width)/2,2,0,p);
}


//===========================================================================
//SINGLE PLAYER MENU
//===========================================================================
void M_DrawSinglePlayerMenu(void);

void M_NewGame(int choice);
void M_LoadGame(int choice);
void M_SaveGame(int choice);
void M_EndGame(int choice);

enum
{
    newgame = 0,
    loadgame,
    savegame,
    endgame,
    single_end
} single_e;

menuitem_t SinglePlayerMenu[] =
{
    {1,"M_NGAME" ,M_NewGame ,'n'},
    {1,"M_LOADG" ,M_LoadGame,'l'},
    {1,"M_SAVEG" ,M_SaveGame,'s'},
    {1,"M_ENDGAM",M_EndGame ,'e'}
};

menu_t	SinglePlayerDef =
{
    single_end,
    NULL,
    SinglePlayerMenu,
    M_DrawSinglePlayerMenu,
    97,64,
    0
};

void M_SinglePlayer (int choice)
{
    M_SetupNextMenu(&SinglePlayerDef);
}

//
// M_DrawSinglePlayerMenu
//
void M_DrawSinglePlayerMenu(void)
{
    patch_t* p = W_CacheLumpName("M_SINGLE",PU_CACHE);
    V_DrawScaledPatch ((BASEVIDWIDTH-p->width)/2,2,0,p);
}



//===========================================================================
//			      MULTI PLAYER MENU
//===========================================================================
void M_DrawMultiPlayerMenu(void);
void M_SetupMultiPlayer (int choice);

enum
{
    multisetup = 0,
    multi_end
} single_e;

menuitem_t MultiPlayerMenu[] =
{
    {1,"M_SETUPM",M_SetupMultiPlayer ,'s'}
};

menu_t	MultiPlayerDef =
{
    multi_end,
    NULL,
    MultiPlayerMenu,
    M_DrawMultiPlayerMenu,
    97,64,
    0
};

void M_MultiPlayer (int choice)
{
    M_SetupNextMenu(&MultiPlayerDef);
}

void M_DrawMultiPlayerMenu(void)
{
    patch_t* p = W_CacheLumpName("M_MULTI",PU_CACHE);
    V_DrawScaledPatch ((BASEVIDWIDTH-p->width)/2,2,0,p);
}


//===========================================================================
//MULTI PLAYER SETUP MENU
//===========================================================================
void M_DrawSetupMultiPlayerMenu(void);
void M_HandleSetupMultiPlayer(int choice);

#define setupmulti_end 2
menuitem_t SetupMultiPlayerMenu[] =
{
    //added:08-02-98: BIG HACK : use alphaKey member as coord y for option
    {3,"",M_HandleSetupMultiPlayer,40},
    {3,"",M_HandleSetupMultiPlayer,40+16}
//    {3,"",M_HandleSetupMultiPlayer,40+96}
};

menu_t	SetupMultiPlayerDef =
{
    setupmulti_end,
    &MultiPlayerDef,
    SetupMultiPlayerMenu,
    M_DrawSetupMultiPlayerMenu,
    27,40,
    0
};


#define PLBOXW	  8
#define PLBOXH	  9


// Config of console player
char*	  console_playername;	    // size is MAXPLAYERNAME+1, buffer is
				    // allocated at M_LoadDefaults()
int	  console_skincolor;
int	  console_originalweaponswitch = false;   //boolean
char*	  console_favoriteweapon;   // only used if originalweaponswitch==false


static	int    setupm_cursor = 4;
static	int    setup_oldcolor;


static	int    multi_cursor;
static	int	  multi_tics;
static	state_t*  multi_state;

void M_SetupMultiPlayer (int choice)
{
    multi_cursor = 0;
    multi_state = &states[mobjinfo[MT_PLAYER].seestate];
    multi_tics = multi_state->tics;
    M_SetupNextMenu(&SetupMultiPlayerDef);
}


//
//  Draw the multi player setup menu, had some fun with player anim
//
void M_DrawSetupMultiPlayerMenu(void)
{
    int       mx,my;
    patch_t*  p;

    spritedef_t*    sprdef;
    spriteframe_t*  sprframe;
    int 	    lump;
    patch_t*	    patch;
    int 	    st;
    byte*	    colormap;

    int 	    plboxw,plboxh;

    mx = SetupMultiPlayerDef.x;
    my = SetupMultiPlayerDef.y;

    p = W_CacheLumpName("M_MULTI",PU_CACHE);
    V_DrawScaledPatch ((BASEVIDWIDTH-p->width)/2,2,0,p);

    V_DrawString (mx, my, "Your name");
    M_DrawTextBox(mx+90,my-8,MAXPLAYERNAME,1);
    V_DrawString (mx+98,my,console_playername);

    V_DrawString (mx, my+16, "Your color");

    //V_DrawString (mx, my+96, "Accept Changes");

    // draw text cursor for name
    if (multi_cursor==0 &&
	skullAnimCounter<4)   //blink cursor
	V_DrawMappedPatch (mx+98+V_StringWidth(console_playername),my,0,
			   W_CacheLumpName("STCFN095",PU_CACHE),
			   whitemap);

    // anim the player in the box
    if (--multi_tics<=0)
    {
	st = multi_state->nextstate;
	if (st!=S_NULL)
	    multi_state = &states[st];
	multi_tics = multi_state->tics;
	if (multi_tics==-1)
	    multi_tics=15;
    }

    sprdef    = &sprites[multi_state->sprite];
    sprframe  = &sprdef->spriteframes[ multi_state->frame & FF_FRAMEMASK];
    lump  = sprframe->lump[0];
    patch = W_CacheLumpNum (lump+firstspritelump, PU_CACHE);

    // draw box around guy
    M_DrawTextBox(mx+90,my+8, PLBOXW, PLBOXH);

    if (console_skincolor==0)
	colormap = colormaps;
    else
	colormap = (byte *) translationtables - 256 + (console_skincolor<<8);
    // draw player sprite
    V_DrawMappedPatch (mx+98+(PLBOXW*8/2),my+16+(PLBOXH*8)-8,0,patch,colormap);

// Draw the cursor for the multi player menu
    drawskullcursor = false;
    if (skullAnimCounter<4)  //blink cursor
    {
	V_DrawMappedPatch(currentMenu->x - 10,
			  currentMenu->menuitems[multi_cursor].alphaKey,
			  0,
			  W_CacheLumpName( "STCFN042" ,PU_CACHE),
			  whitemap);
    }
}


//
//
//
void M_HandleSetupMultiPlayer (int choice)
{
    int      l;
    boolean  send = false;

    switch( choice )
    {
      case KEY_DOWNARROW:
	S_StartSound(NULL,sfx_pstop);
	if (multi_cursor+1 >= setupmulti_end)
	    multi_cursor = 0;
	else multi_cursor++;
	break;

      case KEY_UPARROW:
	S_StartSound(NULL,sfx_pstop);
	if (!multi_cursor)
	    multi_cursor = setupmulti_end-1;
	else multi_cursor--;
	break;

      case KEY_LEFTARROW:
	if (multi_cursor==1)		//player color
	{
	    S_StartSound(NULL,sfx_stnmov);
	    console_skincolor--;
	    send = true;
	}
	break;

      case KEY_RIGHTARROW:
	if (multi_cursor==1)		//player color
	{
forward:
	    S_StartSound(NULL,sfx_stnmov);
	    console_skincolor++;
	    send = true;
	}
	break;

      case KEY_ENTER:
	S_StartSound(NULL,sfx_stnmov);
	if (multi_cursor==1)
	    goto forward;
	if (multi_cursor==0)		//player name
	    send = true;
	break;

      case KEY_ESCAPE:
	S_StartSound(NULL,sfx_swtchx);
	if (currentMenu->prevMenu)
	    M_SetupNextMenu (currentMenu->prevMenu);
	else
	    M_ClearMenus ();
	return;

      case KEY_BACKSPACE:
	S_StartSound(NULL,sfx_stnmov);
	if ( (l=strlen(console_playername)) )
	    console_playername[l-1]=0;
	return;

      default:
	if (choice < 32 || choice > 127)
	    break;
	l = strlen(console_playername);
	if (l<MAXPLAYERNAME-1)
	{
	    S_StartSound(NULL,sfx_stnmov);
	    console_playername[l]=choice;
	    console_playername[l+1]=0;
	}
	break;
    }

    // check colors
    if (console_skincolor<0)
	console_skincolor = MAXSKINCOLORS-1;
    if (console_skincolor>MAXSKINCOLORS-1)
	console_skincolor = 0;

    // send new color and/or name if changed
    if (send)
    {
	if(gamestate==GS_LEVEL && !demoplayback)
	   D_SendPlayerConfig(0);
    }

}


//===========================================================================
//				EPISODE SELECT
//===========================================================================

void M_DrawEpisode(void);
void M_Episode(int choice);

enum
{
    ep1,
    ep2,
    ep3,
    ep4,
    ep_end
} episodes_e;

menuitem_t EpisodeMenu[]=
{
    {1,"M_EPI1", M_Episode,'k'},
    {1,"M_EPI2", M_Episode,'t'},
    {1,"M_EPI3", M_Episode,'i'},
    {1,"M_EPI4", M_Episode,'t'}
};

menu_t	EpiDef =
{
    ep_end,		// # of menu items
    &MainDef,		// previous menu
    EpisodeMenu,	// menuitem_t ->
    M_DrawEpisode,	// drawing routine ->
    48,63,		// x,y
    ep1 		// lastOn, flags
};

//===========================================================================
//NEW GAME FOR SINGLE PLAYER
//===========================================================================
void M_DrawNewGame(void);

void M_ChooseSkill(int choice);

enum
{
    killthings,
    toorough,
    hurtme,
    violence,
    nightmare,
    newg_end
} newgame_e;

menuitem_t NewGameMenu[]=
{
    {1,"M_JKILL",	M_ChooseSkill, 'i'},
    {1,"M_ROUGH",	M_ChooseSkill, 'h'},
    {1,"M_HURT",	M_ChooseSkill, 'h'},
    {1,"M_ULTRA",	M_ChooseSkill, 'u'},
    {1,"M_NMARE",	M_ChooseSkill, 'n'}
};

menu_t	NewDef =
{
    newg_end,		// # of menu items
    &EpiDef,		// previous menu
    NewGameMenu,	// menuitem_t ->
    M_DrawNewGame,	// drawing routine ->
    48,63,		// x,y
    violence		// lastOn
};



//===========================================================================
//			       OPTIONS MENU
//===========================================================================
void M_DrawOptions(void);
void M_HandleOptions (int ch);

void M_ToggleMessages (void);
void M_SizeDisplay (int choice);
void M_ChangeBrightness(int choice);
void M_ChangeSensitivity(int choice, int* sensitivity);

enum
{
    messages,
    scrnsize,
    brightness,
    mousesens,
    togglerun,
    togglemlook,
    invmouse,
    mlooksens,
    soundvol,
    videomode,
    control,
    opt_end
} options_e;


//added:10-02-98: note: alphaKey member is the y offset
menuitem_t OptionsMenu[]=
{
    {3,"",M_HandleOptions,0},
    {3,"",M_HandleOptions,10},
    {3,"",M_HandleOptions,20},
    {3,"",M_HandleOptions,30},
    {3,"",M_HandleOptions,40},
    {3,"",M_HandleOptions,50},
    {3,"",M_HandleOptions,60},
    {3,"",M_HandleOptions,70},
    {3,"",M_HandleOptions,90},
    {3,"",M_HandleOptions,100},
    {3,"",M_HandleOptions,110}
};


menu_t	OptionsDef =
{
    opt_end,
    &MainDef,
    OptionsMenu,
    M_DrawOptions,
    60,40,
    0
};


//
// M_Options
//
static int    options_cursor;
static char   onoff[2][4] = {"Off","On"};

#define SLIDER_RANGE	10

//
//  A smaller 'Thermo', with range given as percents (0-100)
//
void M_DrawSlider (int x, int y, int range)
{
    int i;

    if (range < 0)
	range = 0;
    if (range > 100)
	range = 100;

    V_DrawScaledPatch (x-8, y, 0, W_CacheLumpName( "M_SLIDEL" ,PU_CACHE) );

    for (i=0 ; i<SLIDER_RANGE ; i++)
	V_DrawScaledPatch (x+i*8, y, 0,
			   W_CacheLumpName( "M_SLIDEM" ,PU_CACHE) );

    V_DrawScaledPatch (x+SLIDER_RANGE*8, y, 0,
		       W_CacheLumpName( "M_SLIDER" ,PU_CACHE) );

    // draw the slider cursor
    V_DrawMappedPatch (x + ((SLIDER_RANGE-1)*8*range)/100, y, 0,
		       W_CacheLumpName( "M_SLIDEC" ,PU_CACHE),
		       whitemap);
}

void M_DrawOptions(void)
{
    int 	r;
    int 	mx,my;
    patch_t*	p;
    menuitem_t* it;
    int 	w[2];

    w[0] = V_StringWidth (onoff[0]);
    w[1] = V_StringWidth (onoff[1]);

    mx = OptionsDef.x;
    my = OptionsDef.y;

    p = W_CacheLumpName("M_OPTTTL",PU_CACHE);
    V_DrawScaledPatch ((BASEVIDWIDTH-p->width)/2,15,0,p);

    V_DrawString (mx,my,    "Messages:");
    V_DrawStringWhite (260-w[showMessages&1],my, onoff[showMessages&1]);

    V_DrawString (mx,my+10, "Screen Size");
    r = (screenSize * 100) / 8;
    M_DrawSlider (260-86,my+10,r);

    V_DrawString (mx,my+20, "Brightness");
    r = (usegamma * 100) / 4;
    M_DrawSlider (260-86,my+20,r);

    V_DrawString (mx,my+30, "Mouse Speed");
    r = (mouseSensitivity * 100) / 19;
    M_DrawSlider (260-86,my+30,r);

    V_DrawString (mx,my+40, "Always Run");
    V_DrawStringWhite (260-w[speed_locked&1],my+40, onoff[speed_locked&1]);

    V_DrawString (mx,my+50, "Always MouseLook");
    V_DrawStringWhite (260-w[mouseaiming_lock&1],my+50, onoff[mouseaiming_lock&1]);

    V_DrawString (mx,my+60, "Invert Mouse");
    V_DrawStringWhite (260-w[invert_mouse&1],my+60, onoff[invert_mouse&1]);

    V_DrawString (mx,my+70, "Mlook Speed");
    r = (mlookSensitivity * 100) / 19;
    M_DrawSlider (260-86,my+70,r);

    V_DrawStringWhite (mx,my+90, "Sound Volume...");
    V_DrawStringWhite (mx,my+100, "Video Options...");
    V_DrawStringWhite (mx,my+110,"Setup Controls...");


// Draw the cursor for the Options menu
    drawskullcursor = false;
    if (skullAnimCounter<4)   //blink cursor
    {
	V_DrawMappedPatch(mx - 10,
			  my + OptionsMenu[options_cursor].alphaKey, 0,
			  W_CacheLumpName( "STCFN042" ,PU_CACHE),
			  whitemap);
    }
}


void M_Options(int choice)
{
    options_cursor = OptionsDef.lastOn;
    M_SetupNextMenu(&OptionsDef);
}


//added:10-02-98:
menu_t	VidModeDef;
menu_t	ControlDef;
menu_t	SoundDef;
void M_HandleOptions (int ch)
{
    switch( ch )
    {
      case KEY_DOWNARROW:
	S_StartSound(NULL,sfx_pstop);
	if (options_cursor+1 >= opt_end)
	    options_cursor = 0;
	else options_cursor++;
	break;

      case KEY_UPARROW:
	S_StartSound(NULL,sfx_pstop);
	if (!options_cursor)
	    options_cursor = opt_end-1;
	else options_cursor--;
	break;

      case KEY_LEFTARROW:
	S_StartSound(NULL,sfx_stnmov);
	if (options_cursor==scrnsize)
	    M_SizeDisplay(0);
	else if (options_cursor==brightness)
	    M_ChangeBrightness(0);
	else if (options_cursor==mousesens)
	    M_ChangeSensitivity(0,&mouseSensitivity);
	else if (options_cursor==mlooksens)
	    M_ChangeSensitivity(0,&mlookSensitivity);
	break;

      case KEY_RIGHTARROW:
	S_StartSound(NULL,sfx_stnmov);
	if (options_cursor==scrnsize)
	    M_SizeDisplay(1);
	else if (options_cursor==brightness)
	    M_ChangeBrightness(1);
	else if (options_cursor==mousesens)
	    M_ChangeSensitivity(1,&mouseSensitivity);
	else if (options_cursor==mlooksens)
	    M_ChangeSensitivity(1,&mlookSensitivity);
	break;

      case KEY_ENTER:
	S_StartSound(NULL,sfx_pstop);
	switch (options_cursor)
	{
	  case messages:
	    M_ToggleMessages ();
	    return;
	  case togglerun:
	    // toggle always run
	    speed_locked ^= 1;
	    return;
	  case togglemlook:
	    // toggle always mlook
	    mouseaiming_lock ^= 1;
	    return;
	  case invmouse:
	    // toggle invert mouse
	    invert_mouse ^= 1;
	    return;
	  case soundvol:
	    M_SetupNextMenu (&SoundDef);
	    return;
	  case videomode:
	    M_SetupNextMenu (&VidModeDef);
	    return;
	  case control:
	    M_SetupNextMenu (&ControlDef);
	    return;
	  default:
	    break;
	}
	break;

      case KEY_ESCAPE:
	S_StartSound(NULL,sfx_swtchx);
	if (currentMenu->prevMenu)
	{
	    OptionsDef.lastOn = options_cursor;
	    M_SetupNextMenu (currentMenu->prevMenu);
	}
	else
	    M_ClearMenus ();
	return;

      default:
	break;
    }

}


//
//	Toggle messages on/off
//
void M_ToggleMessages (void)
{
    showMessages = 1 - showMessages;

    if (!showMessages)
	players[consoleplayer].message = MSGOFF;
    else
	players[consoleplayer].message = MSGON ;

    message_dontfuckwithme = true;
}

void M_SizeDisplay(int choice)
{
    switch(choice)
    {
      case 0:
	if (screenSize > 0)
	{
	    screenblocks--;
	    screenSize--;
	}
	break;
      case 1:
	if (screenSize < 8)
	{
	    screenblocks++;
	    screenSize++;
	}
	break;
    }

    R_SetViewSize (screenblocks, detailLevel);
}

//added:10-02-98: now change gamma from the menu
void M_ChangeBrightness(int choice)
{
    int oldgamma = usegamma;

    if (!choice)
    {
	if (usegamma > 0)
	    usegamma--;
    }
    else
    {
	if (++usegamma > 4)
	    usegamma = 4;
    }

    if (usegamma!=oldgamma)
    {
	players[consoleplayer].message = gammamsg[usegamma];
	I_SetPalette (W_CacheLumpName ("PLAYPAL",PU_CACHE));
    }
}


//
//  used for both mouseSensitivity and mlookSensitivity
//  (they share the same range of values)
//
void M_ChangeSensitivity(int choice, int *sensitivity)
{
    if (!choice)
    {
	if (*sensitivity)
	    (*sensitivity)--;
    }
    else
	if (*sensitivity < 19)
	    (*sensitivity)++;
}


/*
void M_ChangeDetail(int choice)
{
    choice = 0;

    // FIXME - does not work. Remove anyway?
    //fprintf( stderr, "M_ChangeDetail: low detail mode n.a.\n");

    return;

    detailLevel = 1 - detailLevel; // added 3-1-98
    //R_SetViewSize (screenblocks, detailLevel);

    //if (!detailLevel)
    //	  players[consoleplayer].message = DETAILHI;
    //else
    //	  players[consoleplayer].message = DETAILLO;
}
*/


//===========================================================================
//			    Read This! MENU 1 & 2
//===========================================================================

void M_DrawReadThis1(void);
void M_DrawReadThis2(void);
void M_FinishReadThis(int choice);

enum
{
    rdthsempty1,
    read1_end
} read_e;

menuitem_t ReadMenu1[] =
{
    {1,"",M_ReadThis2,0}
};

menu_t	ReadDef1 =
{
    read1_end,
    &MainDef,
    ReadMenu1,
    M_DrawReadThis1,
    280,185,
    0
};

enum
{
    rdthsempty2,
    read2_end
} read_e2;

menuitem_t ReadMenu2[]=
{
    {1,"",M_FinishReadThis,0}
};

menu_t	ReadDef2 =
{
    read2_end,
    &ReadDef1,
    ReadMenu2,
    M_DrawReadThis2,
    330,175,
    0
};


//===========================================================================
//			  SOUND VOLUME MENU
//===========================================================================
void M_DrawSound(void);

void M_SfxVol(int choice);
void M_MusicVol(int choice);

enum
{
    sfx_vol,
    sfx_empty1,
    music_vol,
    sfx_empty2,
    sound_end
} sound_e;

menuitem_t SoundMenu[]=
{
    {2,"M_SFXVOL",M_SfxVol,'s'},
    {-1,"",0},
    {2,"M_MUSVOL",M_MusicVol,'m'},
    {-1,"",0}
};

menu_t	SoundDef =
{
    sound_end,
    &OptionsDef,
    SoundMenu,
    M_DrawSound,
    80,64,
    0
};


//===========================================================================
//			    CONTROLS MENU
//===========================================================================
void M_DrawControl(void);		// added 3-1-98
void M_HandleControlsMenu (int ch);	//added:08-02-98:

typedef struct {
    char name[30];
    byte num;
} controlkey_t;

controlkey_t controlkey[]=
{
  {"Fire"	    ,gc_fire},
  {"Use/Open"	    ,gc_use},
  {"Forward"	    ,gc_forward},
  {"Backpedal"	    ,gc_backward},
  {"Turn Left"	    ,gc_turnleft},
  {"Turn Right"     ,gc_turnright},
  {"Run"	    ,gc_speed},
//  {"Run Lock"       ,gc_},
  {"Strafe On"	    ,gc_strafe},
  {"Strafe Left"    ,gc_strafeleft},
  {"Strafe Right"   ,gc_straferight},
  {"Look Up"	    ,gc_lookup},
  {"Look Down"	    ,gc_lookdown},
  {"Center View"    ,gc_centerview},
  {"Mouselook"	    ,gc_mouseaiming}
};

menuitem_t ControlMenu[]=
{
    {3,"",M_HandleControlsMenu,'\0'}   // dummy menuitem for our control func
};

#define control_end (sizeof(controlkey)/sizeof(controlkey_t))

menu_t	ControlDef =
{
    1,
    &OptionsDef,
    ControlMenu,
    M_DrawControl,
    50,40,
    0
};

//item active for the Controls menu
static int keys_cursor=0;




//
//  Search into the keys for max. two keys mapped to the given control
//
void M_FindKeysForCommand (int control, int *twokeys)
{
    int     count;
    int     j;

    twokeys[0] = twokeys[1] = -1;
    count = 0;

    for (j=0; j<NUMINPUTS; j++)
    {
	if (controlmap[j]==control)
	{
	    twokeys[count] = j;     //number of key,mouse,or joy button
	    count++;
	    if (count == 2)
		break;
	}
    }

}


//
//  Draws the Cutomise Controls menu
//
void M_DrawControl(void)
{
    char     tmp[50];
    int      i,j;
    int      y;
    patch_t* p;
    char     *name;
    int      keys[2];

    M_CentreText (ControlDef.y-20, "PRESS ENTER TO CHANGE, BACKSPACE TO CLEAR");

    p = W_CacheLumpName("M_CONTRO",PU_CACHE);
    V_DrawScaledPatch ((BASEVIDWIDTH-p->width)/2,2,0,p);

    for(i=0;i<control_end;i++)
    {
	y = ControlDef.y + i*8;

	V_DrawString (ControlDef.x, y, controlkey[i].name);

	M_FindKeysForCommand (controlkey[i].num, keys);

	if (keys[0] == -1)
	{
	    V_DrawStringWhite (ControlDef.x+220-3*6, y, "---");
	}
	else
	{
	    name = G_KeynumToString (keys[0]);
	    sprintf (tmp,"%s", name);

	    if (keys[1] != -1)
	    {
		name = G_KeynumToString (keys[1]);
		sprintf (tmp+strlen(tmp)," or %s", name);
	    }

	    V_DrawStringWhite(ControlDef.x+220-V_StringWidth(tmp), y, tmp);
	}

    }

// Draw the cursor for the control menu
    drawskullcursor = false;
    if (skullAnimCounter<4)   //blink cursor
    {
	V_DrawMappedPatch(currentMenu->x - 10,
			  currentMenu->y + keys_cursor*8, 0,
			  W_CacheLumpName( "STCFN042" ,PU_CACHE),
			  whitemap);
    }

}

static int controltochange;

void M_ChangecontrolResponse(int ch)
{
    int        i,control,nkeys,samekey;
    event_t*   ev;
    int        keys[2];

    // this was set just before calling us
    messageNeedsEvent = false;
    ev = messageEvent;

    // ESCAPE cancels
    if (ch!=KEY_ESCAPE && ch!=KEY_PAUSE)
    {

	// THIS CODE SUCKS... mouse and key events may be converted
	// to virtual keys at the interface level...
	switch (ev->type)
	{
	  case ev_mouse:
	    if (ev->data1&1)
	       ch = KEY_MOUSE1;
	    else if (ev->data1&2)
	       ch = KEY_MOUSE1+1;
	    else if (ev->data1&4)
	       ch = KEY_MOUSE1+2;
	    else
	       ch = 1;	    // no key
	    break;
	  case ev_joystick:
	    if (ev->data1&1)
	       ch = KEY_JOY1;
	    else if (ev->data1&2)
	       ch = KEY_JOY1+1;
	    else if (ev->data1&4)
	       ch = KEY_JOY1+2;
	    else if (ev->data1&8)
	       ch = KEY_JOY1+3;
	    else
	       ch = 1;	    // no key
	    break;
	  default:
	    break;
	}

	control = controltochange;

	// ensure that there's max two keys mapped to the given control
	// (we display only two in the menu) user friendly IS NOT coder friendly
	samekey = -1;
	for (i=0,nkeys=0; i<NUMINPUTS; i++)
	    if (controlmap[i]==control)
		if (i!=ch)
		    ++nkeys;
		else
		    samekey = i;

	// clear the two keys
	if (nkeys>1)
	{
	    for (i=0; i<NUMINPUTS; i++)
		 if (controlmap[i]==control)
		     controlmap[i] = gc_null;
	}
	else if (samekey>=0)
	{
	    // replace mouse and joy clicks by double clicks
	    if (ch>=KEY_MOUSE1 && ch<=KEY_MOUSE1+2)
	    {
		controlmap[ch] = gc_null;
		ch -= KEY_MOUSE1;
		controlmap[ch+KEY_DBLMOUSE1] = control;
	    }
	    else
	    if (ch>=KEY_JOY1   && ch<=KEY_JOY1+3)
	    {
		controlmap[ch] = gc_null;
		ch -= KEY_JOY1;
		controlmap[ch+KEY_DBLJOY1] = control;
	    }

	}
	else
	    // map the key to the selected control
	    controlmap[ch] = control;
    }

    M_SetupNextMenu(&ControlDef);
    menuactive = 1;
}

void M_ChangeControl(int choice)
{
    static char tmp[80];

    controltochange = controlkey[choice].num;
    sprintf (tmp,"Hit the new key for\n%s\nESC for Cancel",controlkey[choice].name);

    //added:07-02-98:dirty hack : means mouse and joystick responds only
    //			   with buttons, not moves
    messageNeedsEvent = true;
    M_StartMessage (tmp,M_ChangecontrolResponse,false);
}


//added:08-02-98: special menuitem key handler for customise controls
void M_HandleControlsMenu (int ch)
{
    switch( ch )
    {
      case KEY_DOWNARROW:
      case KEY_RIGHTARROW:
	S_StartSound(NULL,sfx_pstop);
	if (keys_cursor+1 >= control_end)
	    keys_cursor = 0;
	else keys_cursor++;
	break;

      case KEY_UPARROW:
      case KEY_LEFTARROW:
	S_StartSound(NULL,sfx_pstop);
	if (!keys_cursor)
	    keys_cursor = control_end-1;
	else keys_cursor--;
	break;

      case KEY_ENTER:
	S_StartSound(NULL,sfx_pstop);
	M_ChangeControl (keys_cursor);
	break;

      case KEY_ESCAPE:
	S_StartSound(NULL,sfx_swtchx);
	if (currentMenu->prevMenu)
	    M_SetupNextMenu (currentMenu->prevMenu);
	else
	    M_ClearMenus ();
	return;

      case KEY_BACKSPACE:
	S_StartSound(NULL,sfx_stnmov);
	// detach any keys associated to the game control
	G_ClearControlKeys (controlkey[keys_cursor].num);
	return;

      default:
	break;
    }

}



//===========================================================================
//			  VIDEO MODE MENU
//===========================================================================
void M_DrawVideoMode(void);		//added:30-01-98:

void M_HandleVideoMode (int ch);

menuitem_t VideoModeMenu[]=
{
    {3,"", M_HandleVideoMode, '\0'}	// dummy menuitem for the control func
};

menu_t	VidModeDef =
{
    1,			// # of menu items
    &OptionsDef,	// previous menu
    VideoModeMenu,	// menuitem_t ->
    M_DrawVideoMode,	// drawing routine ->
    48,60,		// x,y
    0			// lastOn
};

//added:30-01-98:
#define MAXCOLUMNMODES	 10	//max modes displayed in one column
#define MAXMODEDESCS	 (MAXCOLUMNMODES*3)

// shhh... what am I doing... nooooo!
int   VID_NumModes(void);
char  *VID_GetModeName(int modenum);
extern int   vid_modenum;

static int vidm_testingmode=0;
static int vidm_current=0;
static int vidm_nummodes;
static int vidm_column_size;

typedef struct
{
    int     modenum;	//video mode number in the vidmodes list
    char    *desc;	//XXXxYYY
    int     iscur;	//1 if it is the current active mode
} modedesc_t;

static modedesc_t   modedescs[MAXMODEDESCS];


//
// Draw the video modes list, -la-Quake
//
void M_DrawVideoMode(void)
{
    int     i,j,dup,row,col,nummodes;
    char    *desc;
    char    temp[80];
    patch_t* p;

    p = (patch_t*) W_CacheLumpName("M_VIDEO",PU_CACHE);
    V_DrawScaledPatch ((BASEVIDWIDTH-p->width)/2,2,0,p);

    vidm_nummodes = 0;
    nummodes = VID_NumModes ();

    for (i=0 ; i<nummodes && vidm_nummodes<MAXMODEDESCS ; i++)
    {
	desc = VID_GetModeName (i);
	if (desc)
	{
	    dup = 0;

	    //when a resolution exists both under VGA and VESA, keep the
	    // VESA mode, which is always a higher modenum
	    for (j=0 ; j<vidm_nummodes ; j++)
	    {
		if (!strcmp (modedescs[j].desc, desc))
		{
		    //mode(0): 320x200 is always standard VGA, not vesa
		    if (modedescs[j].modenum != 0)
		    {
			modedescs[j].modenum = i;
			dup = 1;

			if (i == vid_modenum)
			    modedescs[j].iscur = 1;
		    }
		    else
		    {
			dup = 1;
		    }

		    break;
		}
	    }

	    if (!dup)
	    {
		modedescs[vidm_nummodes].modenum = i;
		modedescs[vidm_nummodes].desc = desc;
		modedescs[vidm_nummodes].iscur = 0;

		if (i == vid_modenum)
		    modedescs[vidm_nummodes].iscur = 1;

		vidm_nummodes++;
	    }
	}
    }

    vidm_column_size = (vidm_nummodes+2) / 3;


    row = 16;
    col = 36;
    for(i=0; i<vidm_nummodes; i++)
    {
	if (modedescs[i].iscur)
	    V_DrawStringWhite (row, col, modedescs[i].desc);
	else
	    V_DrawString (row, col, modedescs[i].desc);

	col += 8;
	if((i % vidm_column_size) == (vidm_column_size-1))
	{
	    row += 8*13;
	    col = 36;
	}
    }

    if (vidm_testingmode>0)
    {
	sprintf(temp, "TESTING MODE %s", modedescs[vidm_current].desc );
	M_CentreText(VidModeDef.y+80, temp );
	M_CentreText(VidModeDef.y+90, "Please wait 5 seconds..." );
    }
    else
    {
	M_CentreText(VidModeDef.y+60,"Press ENTER to set mode");

	M_CentreText(VidModeDef.y+70,"T to test mode for 5 seconds");

	sprintf(temp, "D to make %s the default", VID_GetModeName(vid_modenum));
	M_CentreText(VidModeDef.y+80,temp);

	sprintf(temp, "Current default is %dx%d", scr_width, scr_height ); //VID_GetModeName(scr_default_mode)
	M_CentreText(VidModeDef.y+90,temp);

	M_CentreText(VidModeDef.y+100,"Press ESC to exit");
    }

// Draw the cursor for the VidMode menu
    drawskullcursor = false;
    if (skullAnimCounter<4)    //use the Skull anim counter to blink the cursor
    {
	i = 16 - 10 + ((vidm_current / vidm_column_size)*8*13);
	j = 36 + ((vidm_current % vidm_column_size)*8);
	V_DrawMappedPatch(i, j, 0, W_CacheLumpName( "STCFN042" ,PU_CACHE), whitemap);
    }
}


//added:30-01-98: special menuitem key handler for video mode list
void M_HandleVideoMode (int ch)
{
    if (vidm_testingmode>0)
       return;

    switch( ch )
    {
      case KEY_DOWNARROW:
	S_StartSound(NULL,sfx_pstop);
	vidm_current++;
	if (vidm_current>=vidm_nummodes)
	    vidm_current = 0;
	break;

      case KEY_UPARROW:
	S_StartSound(NULL,sfx_pstop);
	vidm_current--;
	if (vidm_current<0)
	    vidm_current = vidm_nummodes-1;
	break;

      case KEY_LEFTARROW:
	S_StartSound(NULL,sfx_pstop);
	vidm_current -= vidm_column_size;
	if (vidm_current<0)
	    vidm_current %= vidm_column_size;
	if (vidm_current>=vidm_nummodes)
	    vidm_current = vidm_nummodes-1;
	break;

      case KEY_RIGHTARROW:
	S_StartSound(NULL,sfx_pstop);
	vidm_current += vidm_column_size;
	if (vidm_current>=(vidm_column_size*3))
	    vidm_current %= vidm_column_size;
	if (vidm_current>=vidm_nummodes)
	    vidm_current = vidm_nummodes-1;
	break;

      case KEY_ENTER:
	S_StartSound(NULL,sfx_pstop);
	if (!setmodeneeded) //in case the previous setmode was not finished
	    setmodeneeded = modedescs[vidm_current].modenum+1;
	break;

      case KEY_ESCAPE:	    //this one same as M_Responder
	S_StartSound(NULL,sfx_swtchx);
	if (currentMenu->prevMenu)
	    M_SetupNextMenu (currentMenu->prevMenu);
	else
	    M_ClearMenus ();
	return;

      case 'T':
      case 't':
	S_StartSound(NULL,sfx_swtchx);
	vidm_testingmode = TICRATE*5;
	return;

      case 'D':
      case 'd':
	// current active mode becomes the default mode.
	S_StartSound(NULL,sfx_swtchx);
	SCR_SetDefaultMode ();
	return;

      default:
	break;
    }

}


//===========================================================================
//LOAD GAME MENU
//===========================================================================
void M_DrawLoad(void);

void M_LoadSelect(int choice);

enum
{
    load1,
    load2,
    load3,
    load4,
    load5,
    load6,
    load_end
} load_e;

menuitem_t LoadMenu[]=
{
    {1,"", M_LoadSelect,'1'},
    {1,"", M_LoadSelect,'2'},
    {1,"", M_LoadSelect,'3'},
    {1,"", M_LoadSelect,'4'},
    {1,"", M_LoadSelect,'5'},
    {1,"", M_LoadSelect,'6'}
};

menu_t	LoadDef =
{
    load_end,
    &MainDef,
    LoadMenu,
    M_DrawLoad,
    80,54,
    0
};

//===========================================================================
//SAVE GAME MENU
//===========================================================================
void M_DrawSave(void);

void M_SaveSelect(int choice);

menuitem_t SaveMenu[]=
{
    {1,"", M_SaveSelect,'1'},
    {1,"", M_SaveSelect,'2'},
    {1,"", M_SaveSelect,'3'},
    {1,"", M_SaveSelect,'4'},
    {1,"", M_SaveSelect,'5'},
    {1,"", M_SaveSelect,'6'}
};

menu_t	SaveDef =
{
    load_end,
    &MainDef,
    SaveMenu,
    M_DrawSave,
    80,54,
    0
};


//
// M_ReadSaveStrings
//  read the strings from the savegame files
//
void M_ReadSaveStrings(void)
{
    int 	    handle;
    int 	    count;
    int 	    i;
    char    name[256];

    for (i = 0;i < load_end;i++)
    {
	if (M_CheckParm("-cdrom"))
	    sprintf(name,"c:\\doomdata\\"SAVEGAMENAME"%d.dsg",i);
	else
	    sprintf(name,SAVEGAMENAME"%d.dsg",i);

	handle = open (name, O_RDONLY | 0, 0666);
	if (handle == -1)
	{
	    strcpy(&savegamestrings[i][0],EMPTYSTRING);
	    LoadMenu[i].status = 0;
	    continue;
	}
	count = read (handle, &savegamestrings[i], SAVESTRINGSIZE);
	close (handle);
	LoadMenu[i].status = 1;
    }
}


//
// M_LoadGame & Cie.
//
void M_DrawLoad(void)
{
    int 	    i;

    patch_t* p = W_CacheLumpName("M_LOADG",PU_CACHE);
    V_DrawScaledPatch ((BASEVIDWIDTH-p->width)/2,28,0,p);

    for (i = 0;i < load_end; i++)
    {
	M_DrawSaveLoadBorder(LoadDef.x,LoadDef.y+LINEHEIGHT*i);
	V_DrawString(LoadDef.x,LoadDef.y+LINEHEIGHT*i,savegamestrings[i]);
    }
}



//
// Draw border for the savegame description
//
void M_DrawSaveLoadBorder(int x,int y)
{
    int 	    i;

    V_DrawScaledPatch (x-8,y+7,0,W_CacheLumpName("M_LSLEFT",PU_CACHE));

    for (i = 0;i < 24;i++)
    {
	V_DrawScaledPatch (x,y+7,0,W_CacheLumpName("M_LSCNTR",PU_CACHE));
	x += 8;
    }

    V_DrawScaledPatch (x,y+7,0,W_CacheLumpName("M_LSRGHT",PU_CACHE));
}



//
// User wants to load this game
//
void M_LoadSelect(int choice)
{
    char    name[256];

    if (M_CheckParm("-cdrom"))
	sprintf(name,"c:\\doomdata\\"SAVEGAMENAME"%d.dsg",choice);
    else
	sprintf(name,SAVEGAMENAME"%d.dsg",choice);
    G_LoadGame (name);
    M_ClearMenus ();
}

//
// Selected from DOOM menu
//
void M_LoadGame (int choice)
{
    if (netgame)
    {
	M_StartMessage(LOADNET,NULL,false);
	return;
    }

    M_SetupNextMenu(&LoadDef);
    M_ReadSaveStrings();
}


//
//  M_SaveGame & Cie.
//
void M_DrawSave(void)
{
    int 	    i;

    patch_t* p = W_CacheLumpName("M_SAVEG",PU_CACHE);
    V_DrawScaledPatch ((BASEVIDWIDTH-p->width)/2,28,0,p);
    for (i = 0;i < load_end; i++)
    {
	M_DrawSaveLoadBorder(LoadDef.x,LoadDef.y+LINEHEIGHT*i);
	V_DrawString(LoadDef.x,LoadDef.y+LINEHEIGHT*i,savegamestrings[i]);
    }

    if (saveStringEnter)
    {
	i = V_StringWidth(savegamestrings[saveSlot]);
	V_DrawString(LoadDef.x + i,LoadDef.y+LINEHEIGHT*saveSlot,"_");
    }
}

//
// M_Responder calls this when user is finished
//
void M_DoSave(int slot)
{
    G_SaveGame (slot,savegamestrings[slot]);
    M_ClearMenus ();

    // PICK QUICKSAVE SLOT YET?
    if (quickSaveSlot == -2)
	quickSaveSlot = slot;
}

//
// User wants to save. Start string input for M_Responder
//
void M_SaveSelect(int choice)
{
    // we are going to be intercepting all chars
    saveStringEnter = 1;

    saveSlot = choice;
    strcpy(saveOldString,savegamestrings[choice]);
    if (!strcmp(savegamestrings[choice],EMPTYSTRING))
	savegamestrings[choice][0] = 0;
    saveCharIndex = strlen(savegamestrings[choice]);
}

//
// Selected from DOOM menu
//
void M_SaveGame (int choice)
{
    if (!usergame)
    {
	M_StartMessage(SAVEDEAD,NULL,false);
	return;
    }

    if (gamestate != GS_LEVEL)
	return;

    M_SetupNextMenu(&SaveDef);
    M_ReadSaveStrings();
}



//
//	M_QuickSave
//
char	tempstring[80];

void M_QuickSaveResponse(int ch)
{
    if (ch == 'y')
    {
	M_DoSave(quickSaveSlot);
	S_StartSound(NULL,sfx_swtchx);
    }
}

void M_QuickSave(void)
{
    if (!usergame)
    {
	S_StartSound(NULL,sfx_oof);
	return;
    }

    if (gamestate != GS_LEVEL)
	return;

    if (quickSaveSlot < 0)
    {
	M_StartControlPanel();
	M_ReadSaveStrings();
	M_SetupNextMenu(&SaveDef);
	quickSaveSlot = -2;	// means to pick a slot now
	return;
    }
    sprintf(tempstring,QSPROMPT,savegamestrings[quickSaveSlot]);
    M_StartMessage(tempstring,M_QuickSaveResponse,true);
}



//
// M_QuickLoad
//
void M_QuickLoadResponse(int ch)
{
    if (ch == 'y')
    {
	M_LoadSelect(quickSaveSlot);
	S_StartSound(NULL,sfx_swtchx);
    }
}


void M_QuickLoad(void)
{
    if (netgame)
    {
	M_StartMessage(QLOADNET,NULL,false);
	return;
    }

    if (quickSaveSlot < 0)
    {
	M_StartMessage(QSAVESPOT,NULL,false);
	return;
    }
    sprintf(tempstring,QLPROMPT,savegamestrings[quickSaveSlot]);
    M_StartMessage(tempstring,M_QuickLoadResponse,true);
}


//
// M_EndGame
//
void M_EndGameResponse(int ch)
{
    if (ch != 'y')
	return;

    currentMenu->lastOn = itemOn;
    M_ClearMenus ();
    D_StartTitle ();
}

void M_EndGame(int choice)
{
    choice = 0;
    if (!usergame)
    {
	S_StartSound(NULL,sfx_oof);
	return;
    }

    if (netgame)
    {
	M_StartMessage(NETEND,NULL,false);
	return;
    }

    M_StartMessage(ENDGAME,M_EndGameResponse,true);
}


//
// Read This Menus
// Had a "quick hack to fix romero bug"
//
void M_DrawReadThis1(void)
{
    inhelpscreens = true;
    switch ( gamemode )
    {
      case commercial:
	V_DrawScaledPatch (0,0,0,W_CacheLumpName("HELP",PU_CACHE));
	break;
      case shareware:
      case registered:
      case retail:
	V_DrawScaledPatch (0,0,0,W_CacheLumpName("HELP1",PU_CACHE));
	break;
      default:
	break;
    }
    return;
}



//
// Read This Menus - optional second page.
//
void M_DrawReadThis2(void)
{
    inhelpscreens = true;
    switch ( gamemode )
    {
      case retail:
      case commercial:
	// This hack keeps us from having to change menus.
	V_DrawScaledPatch (0,0,0,W_CacheLumpName("CREDIT",PU_CACHE));
	break;
      case shareware:
      case registered:
	V_DrawScaledPatch (0,0,0,W_CacheLumpName("HELP2",PU_CACHE));
	break;
      default:
	break;
    }
    return;
}


//
// Change Sfx & Music volumes
//
void M_DrawSound(void)
{
    patch_t* p = W_CacheLumpName("M_SVOL",PU_CACHE);
    V_DrawScaledPatch ((BASEVIDWIDTH-p->width)/2,38,0,p);

    M_DrawThermo(SoundDef.x,SoundDef.y+LINEHEIGHT*(sfx_vol+1),
		 16,snd_SfxVolume);

    M_DrawThermo(SoundDef.x,SoundDef.y+LINEHEIGHT*(music_vol+1),
		 16,snd_MusicVolume);
}


void M_SfxVol(int choice)
{
    switch(choice)
    {
      case 0:
	if (snd_SfxVolume)
	    snd_SfxVolume--;
	break;
      case 1:
	if (snd_SfxVolume < 31)
	    snd_SfxVolume++;
	break;
    }

    S_SetSfxVolume(snd_SfxVolume /* *8 */);
}

void M_MusicVol(int choice)
{
    switch(choice)
    {
      case 0:
	if (snd_MusicVolume)
	    snd_MusicVolume--;
	break;
      case 1:
	if (snd_MusicVolume < 31)
	    snd_MusicVolume++;
	break;
    }

    S_SetMusicVolume(snd_MusicVolume );
}


//
// M_NewGame
//
void M_DrawNewGame(void)
{
    patch_t* p;
    p = W_CacheLumpName("M_NEWG",PU_CACHE);
    V_DrawScaledPatch ((BASEVIDWIDTH-p->width)/2,14,0,p);
//    V_DrawScaledPatch (96,14,0,W_CacheLumpName("M_NEWG",PU_CACHE));
    p = W_CacheLumpName("M_SKILL",PU_CACHE);
    V_DrawScaledPatch ((BASEVIDWIDTH-p->width)/2,38,0,p);
//    V_DrawScaledPatch (54,38,0,W_CacheLumpName("M_SKILL",PU_CACHE));
}

void M_NewGame(int choice)
{
    if (netgame && !demoplayback)
    {
	M_StartMessage(NEWGAME,NULL,false);
	return;
    }

    if ( gamemode == commercial )
	M_SetupNextMenu(&NewDef);
    else
	M_SetupNextMenu(&EpiDef);
}


//
//	M_Episode
//
int	epi;

void M_DrawEpisode(void)
{
    patch_t* p = W_CacheLumpName("M_EPISOD",PU_CACHE);
    V_DrawScaledPatch ((BASEVIDWIDTH-p->width)/2,38,0,p);
}

void M_VerifyNightmare(int ch)
{
    if (ch != 'y')
	return;

    G_DeferedInitNew(nightmare,epi+1,1);
    M_ClearMenus ();
}

void M_ChooseSkill(int choice)
{
    if (choice == nightmare)
    {
	M_StartMessage(NIGHTMARE,M_VerifyNightmare,true);
	return;
    }

    G_DeferedInitNew(choice,epi+1,1);
    M_ClearMenus ();
}

void M_Episode(int choice)
{
    if ( (gamemode == shareware)
	 && choice)
    {
	M_StartMessage(SWSTRING,NULL,false);
	M_SetupNextMenu(&ReadDef1);
	return;
    }

    // Yet another hack...
    if ( (gamemode == registered)
	 && (choice > 2))
    {
      fprintf( stderr,
	       "M_Episode: 4th episode requires UltimateDOOM\n");
      choice = 0;
    }

    epi = choice;
    M_SetupNextMenu(&NewDef);
}


//
// M_ReadThis
//
void M_ReadThis(int choice)
{
    choice = 0;
    M_SetupNextMenu(&ReadDef1);
}

void M_ReadThis2(int choice)
{
    choice = 0;
    M_SetupNextMenu(&ReadDef2);
}

void M_FinishReadThis(int choice)
{
    choice = 0;
    M_SetupNextMenu(&MainDef);
}




//
// M_QuitDOOM
//
int	quitsounds[8] =
{
    sfx_pldeth,
    sfx_dmpain,
    sfx_popain,
    sfx_slop,
    sfx_telept,
    sfx_posit1,
    sfx_posit3,
    sfx_sgtatk
};

int	quitsounds2[8] =
{
    sfx_vilact,
    sfx_getpow,
    sfx_boscub,
    sfx_slop,
    sfx_skeswg,
    sfx_kntdth,
    sfx_bspact,
    sfx_sgtatk
};



void M_QuitResponse(int ch)
{
    if (ch != 'y')
	return;
    if (!netgame)
    {
	if (gamemode == commercial)
	    S_StartSound(NULL,quitsounds2[(gametic>>2)&7]);
	else
	    S_StartSound(NULL,quitsounds[(gametic>>2)&7]);
	I_WaitVBL(105);
    }
    I_Quit ();
}




void M_QuitDOOM(int choice)
{
  // We pick index 0 which is language sensitive,
  //  or one at random, between 1 and maximum number.
  if (language != english )
    sprintf(endstring,"%s\n\n"DOSY, endmsg[0] );
  else
    sprintf(endstring,"%s\n\n"DOSY, endmsg[ (gametic%(NUM_QUITMESSAGES-2))+1 ]);

  M_StartMessage(endstring,M_QuitResponse,true);
}



//
//	Menu Functions
//
void
M_DrawThermo
( int	x,
  int	y,
  int	thermWidth,
  int	thermDot )
{
    int 	xx;
    int 	i;

    xx = x;
    V_DrawScaledPatch (xx,y,0,W_CacheLumpName("M_THERML",PU_CACHE));
    xx += 8;
    for (i=0;i<thermWidth;i++)
    {
	V_DrawScaledPatch (xx,y,0,W_CacheLumpName("M_THERMM",PU_CACHE));
	xx += 8;
    }
    V_DrawScaledPatch (xx,y,0,W_CacheLumpName("M_THERMR",PU_CACHE));

    V_DrawScaledPatch ((x+8) + thermDot*4,y,
		       0,W_CacheLumpName("M_THERMO",PU_CACHE));
}



void
M_DrawEmptyCell
( menu_t*	menu,
  int		item )
{
    V_DrawScaledPatch (menu->x - 10,	    menu->y+item*LINEHEIGHT - 1, 0,
		       W_CacheLumpName("M_CELL1",PU_CACHE));
}

void
M_DrawSelCell
( menu_t*	menu,
  int		item )
{
    V_DrawScaledPatch (menu->x - 10,	    menu->y+item*LINEHEIGHT - 1, 0,
		       W_CacheLumpName("M_CELL2",PU_CACHE));
}


//
//  Draw a textbox, like Quake does, because sometimes it's difficult
//  to read the text with all the stuff in the background...
//
//added:06-02-98:
void M_DrawTextBox (int x, int y, int width, int lines)
{
    patch_t  *p;
    int      cx, cy;
    int      n;

    // draw left side
    cx = x;
    cy = y;
    p = W_CacheLumpName("brdr_tl",PU_CACHE);
    V_DrawScaledPatch (cx, cy, 0, p);
    p = W_CacheLumpName ("brdr_l",PU_CACHE);
    for (n = 0; n < lines; n++)
    {
	cy += 8;
	V_DrawScaledPatch (cx, cy, 0, p);
    }
    p = W_CacheLumpName ("brdr_bl",PU_CACHE);
    V_DrawScaledPatch (cx, cy+8, 0, p);

    // draw middle
    V_DrawFlatFill (x+8,y+8,width*8,lines*8,"RROCK17");
    cx += 8;
    while (width > 0)
    {
	cy = y;
	p = W_CacheLumpName ("brdr_t",PU_CACHE);
	V_DrawScaledPatch (cx, cy, 0, p);

	//p = W_CacheLumpName ("brdr_mt",PU_CACHE);
	//cy += 8;
	//V_DrawScaledPatch (cx, cy, 0, p);

	//p = W_CacheLumpName ("brdr_mm",PU_CACHE); //middle ombr dessus
	//for (n = 1; n < lines; n++)
	//{
	//    cy += 8;
	//    V_DrawScaledPatch (cx, cy, 0, p);
	//}

	p = W_CacheLumpName ("brdr_b",PU_CACHE);
	V_DrawScaledPatch (cx, y+8+lines*8, 0, p);
	width --;
	cx += 8;
    }

    // draw right side
    cy = y;
    p = W_CacheLumpName ("brdr_tr",PU_CACHE);
    V_DrawScaledPatch (cx, cy, 0, p);
    p = W_CacheLumpName ("brdr_r",PU_CACHE);
    for (n = 0; n < lines; n++)
    {
	cy += 8;
	V_DrawScaledPatch (cx, cy, 0, p);
    }
    p = W_CacheLumpName ("brdr_br",PU_CACHE);
    V_DrawScaledPatch (cx, cy+8, 0, p);
}


void M_StartMessage ( char*	    string,
		      void*	    routine,
		      boolean	    input )
{
    messageLastMenuActive = menuactive;
    messageToPrint = 1;
    messageString = string;
    messageRoutine = routine;
    messageNeedsInput = input;
    menuactive = true;
    return;
}



void M_StopMessage(void)
{
    menuactive = messageLastMenuActive;
    messageToPrint = 0;
}


//added:30-01-98:
//
//  Write a string centered using the hu_font
//
void M_CentreText (int y, char* string)
{
    int x;
    //added:02-02-98:centre on 320, because V_DrawString centers on vid.width...
    x = (BASEVIDWIDTH - V_StringWidth(string))>>1;
    V_DrawString(x,y,string);
}


//
// CONTROL PANEL
//

//
// M_Responder
//
boolean M_Responder (event_t* ev)
{
    int 	    ch;
    int 	    i;
    static  int     joywait = 0;
    static  int     mousewait = 0;
    static  int     mousey = 0;
    static  int     lasty = 0;
    static  int     mousex = 0;
    static  int     lastx = 0;

    ch = -1;

    if (ev->type == ev_joystick && joywait < I_GetTime())
    {
	if (ev->data3 == -1)
	{
	    ch = KEY_UPARROW;
	    joywait = I_GetTime() + 5;
	}
	else if (ev->data3 == 1)
	{
	    ch = KEY_DOWNARROW;
	    joywait = I_GetTime() + 5;
	}

	if (ev->data2 == -1)
	{
	    ch = KEY_LEFTARROW;
	    joywait = I_GetTime() + 2;
	}
	else if (ev->data2 == 1)
	{
	    ch = KEY_RIGHTARROW;
	    joywait = I_GetTime() + 2;
	}

	if (ev->data1&(1+4+8))
	{
	    ch = KEY_ENTER;
	    joywait = I_GetTime() + 5;
	}
	if (ev->data1&2)
	{
	    ch = KEY_BACKSPACE;
	    joywait = I_GetTime() + 5;
	}
    }
    else
    {
	if (ev->type == ev_mouse && mousewait < I_GetTime())
	{
	    mousey += ev->data3;
	    if (mousey < lasty-30)
	    {
		ch = KEY_DOWNARROW;
		mousewait = I_GetTime() + 5;
		mousey = lasty -= 30;
	    }
	    else if (mousey > lasty+30)
	    {
		ch = KEY_UPARROW;
		mousewait = I_GetTime() + 5;
		mousey = lasty += 30;
	    }

	    mousex += ev->data2;
	    if (mousex < lastx-30)
	    {
		ch = KEY_LEFTARROW;
		mousewait = I_GetTime() + 5;
		mousex = lastx -= 30;
	    }
	    else if (mousex > lastx+30)
	    {
		ch = KEY_RIGHTARROW;
		mousewait = I_GetTime() + 5;
		mousex = lastx += 30;
	    }

	    if (ev->data1&(1+4))
	    {
		ch = KEY_ENTER;
		mousewait = I_GetTime() + 15;
	    }

	    if (ev->data1&2)
	    {
		ch = KEY_BACKSPACE;
		mousewait = I_GetTime() + 15;
	    }
	}
	else
	    if (ev->type == ev_keydown)
	    {
		ch = ev->data1;
	    }
    }

    if (ch == -1)
	return false;


    // Save Game string input
    if (saveStringEnter)
    {
	switch(ch)
	{
	  case KEY_BACKSPACE:
	    if (saveCharIndex > 0)
	    {
		saveCharIndex--;
		savegamestrings[saveSlot][saveCharIndex] = 0;
	    }
	    break;

	  case KEY_ESCAPE:
	    saveStringEnter = 0;
	    strcpy(&savegamestrings[saveSlot][0],saveOldString);
	    break;

	  case KEY_ENTER:
	    saveStringEnter = 0;
	    if (savegamestrings[saveSlot][0])
		M_DoSave(saveSlot);
	    break;

	  default:
	    ch = toupper(ch);
	    if (ch != 32)
		if (ch-HU_FONTSTART < 0 || ch-HU_FONTSTART >= HU_FONTSIZE)
		    break;
	    if (ch >= 32 && ch <= 127 &&
		saveCharIndex < SAVESTRINGSIZE-1 &&
		V_StringWidth(savegamestrings[saveSlot]) <
		(SAVESTRINGSIZE-2)*8)
	    {
		savegamestrings[saveSlot][saveCharIndex++] = ch;
		savegamestrings[saveSlot][saveCharIndex] = 0;
	    }
	    break;
	}
	return true;
    }

    // Take care of any messages that need input
    if (messageToPrint)
    {
	if (messageNeedsInput == true &&
	    !(ch == ' ' || ch == 'n' || ch == 'y' || ch == KEY_ESCAPE))
	    return false;

	//added:07-02-98:dirty hak:for the customise controls, I want only
	//	buttons/keys, not moves
	if (messageNeedsEvent == true)
	{
	    if (ev->type==ev_mouse && !(ev->data1&7))
	       return false;
	    if (ev->type==ev_joystick && !(ev->data1&0xf))
	       return false;
	}

	menuactive = messageLastMenuActive;
	messageToPrint = 0;
	if (messageRoutine)
	{
	    //added:07-02-98:quick hack for controls menu: I need the event
	    messageEvent = ev;
	    messageRoutine(ch);
	}

	if (messageRoutine!=M_ChangecontrolResponse)
	{
	    menuactive = false;
	    S_StartSound(NULL,sfx_swtchx);
	}
	return true;
    }

    if (devparm && ch == KEY_F1)
    {
	G_ScreenShot ();
	return true;
    }


    // F-Keys
    if (!menuactive)
	switch(ch)
	{
	  case KEY_MINUS:	  // Screen size down
	    if (automapactive || chat_on)
		return false;
	    M_SizeDisplay(0);
	    S_StartSound(NULL,sfx_stnmov);
	    return true;

	  case KEY_EQUALS:	  // Screen size up
	    if (automapactive || chat_on)
		return false;
	    M_SizeDisplay(1);
	    S_StartSound(NULL,sfx_stnmov);
	    return true;

	  case KEY_F1:		  // Help key
	    M_StartControlPanel ();

	    if ( gamemode == retail )
	      currentMenu = &ReadDef2;
	    else
	      currentMenu = &ReadDef1;

	    itemOn = 0;
	    S_StartSound(NULL,sfx_swtchn);
	    return true;

	  case KEY_F2:		  // Save
	    M_StartControlPanel();
	    S_StartSound(NULL,sfx_swtchn);
	    M_SaveGame(0);
	    return true;

	  case KEY_F3:		  // Load
	    M_StartControlPanel();
	    S_StartSound(NULL,sfx_swtchn);
	    M_LoadGame(0);
	    return true;

	  case KEY_F4:		  // Sound Volume
	    M_StartControlPanel ();
	    currentMenu = &SoundDef;
	    itemOn = sfx_vol;
	    S_StartSound(NULL,sfx_swtchn);
	    return true;

	  //case KEY_F5:	    // Detail toggle
	    //M_ChangeDetail(0);
	    //S_StartSound(NULL,sfx_swtchn);
	    //return true;

	  case KEY_F6:		  // Quicksave
	    S_StartSound(NULL,sfx_swtchn);
	    M_QuickSave();
	    return true;

	  case KEY_F7:		  // End game
	    S_StartSound(NULL,sfx_swtchn);
	    M_EndGame(0);
	    return true;

	  case KEY_F8:		  // Toggle messages
	    M_ToggleMessages ();
	    S_StartSound(NULL,sfx_swtchn);
	    return true;

	  case KEY_F9:		  // Quickload
	    S_StartSound(NULL,sfx_swtchn);
	    M_QuickLoad();
	    return true;

	  case KEY_F10: 	  // Quit DOOM
	    S_StartSound(NULL,sfx_swtchn);
	    M_QuitDOOM(0);
	    return true;

	  //added:10-02-98: the gamma toggle has been moved
	  //		    into the Options menu
	  //case KEY_F11:
	}


    // Pop-up menu?
    if (!menuactive)
    {
	if (ch == KEY_ESCAPE)
	{
	    M_StartControlPanel ();
	    S_StartSound(NULL,sfx_swtchn);
	    return true;
	}
	return false;
    }


    //added:30-01-98:
    // Handle menuitems which need a specific key handling
    if( currentMenu->menuitems[itemOn].status == 3 )
    {
	currentMenu->menuitems[itemOn].routine(ch);
	return true;
    }

    // Keys usable within menu
    switch (ch)
    {
      case KEY_DOWNARROW:
	do
	{
	    if (itemOn+1 > currentMenu->numitems-1)
		itemOn = 0;
	    else itemOn++;
	} while(currentMenu->menuitems[itemOn].status==-1);
	S_StartSound(NULL,sfx_pstop);
	return true;

      case KEY_UPARROW:
	do
	{
	    if (!itemOn)
		itemOn = currentMenu->numitems-1;
	    else itemOn--;
	} while(currentMenu->menuitems[itemOn].status==-1);
	S_StartSound(NULL,sfx_pstop);
	return true;

      case KEY_LEFTARROW:
	if (currentMenu->menuitems[itemOn].routine &&
	    currentMenu->menuitems[itemOn].status == 2)
	{
	    S_StartSound(NULL,sfx_stnmov);
	    currentMenu->menuitems[itemOn].routine(0);
	}
	return true;

      case KEY_RIGHTARROW:
	if (currentMenu->menuitems[itemOn].routine &&
	    currentMenu->menuitems[itemOn].status == 2)
	{
	    S_StartSound(NULL,sfx_stnmov);
	    currentMenu->menuitems[itemOn].routine(1);
	}
	return true;

      case KEY_ENTER:
	if (currentMenu->menuitems[itemOn].routine &&
	    currentMenu->menuitems[itemOn].status)
	{
	    currentMenu->lastOn = itemOn;
	    if (currentMenu->menuitems[itemOn].status == 2)
	    {
		currentMenu->menuitems[itemOn].routine(1);	// right arrow
		S_StartSound(NULL,sfx_stnmov);
	    }
	    else
	    {
		currentMenu->menuitems[itemOn].routine(itemOn);
		S_StartSound(NULL,sfx_pistol);
	    }
	}
	return true;

      case KEY_ESCAPE:
	currentMenu->lastOn = itemOn;
	M_ClearMenus ();
	S_StartSound(NULL,sfx_swtchx);
	return true;

      case KEY_BACKSPACE:
	currentMenu->lastOn = itemOn;
	if (currentMenu->prevMenu)
	{
	    currentMenu = currentMenu->prevMenu;
	    itemOn = currentMenu->lastOn;
	    S_StartSound(NULL,sfx_swtchn);
	}
	return true;

      default:
	for (i = itemOn+1;i < currentMenu->numitems;i++)
	    if (currentMenu->menuitems[i].alphaKey == ch)
	    {
		itemOn = i;
		S_StartSound(NULL,sfx_pstop);
		return true;
	    }
	for (i = 0;i <= itemOn;i++)
	    if (currentMenu->menuitems[i].alphaKey == ch)
	    {
		itemOn = i;
		S_StartSound(NULL,sfx_pstop);
		return true;
	    }
	break;

    }

    return false;
}



//
// M_StartControlPanel
//
void M_StartControlPanel (void)
{
    // intro might call this repeatedly
    if (menuactive)
	return;

    menuactive = 1;
    currentMenu = &MainDef;	    // JDC
    itemOn = currentMenu->lastOn;   // JDC
    //added:08-02-98: set the Skull cursor by default
    drawskullcursor = true;

}


//
//	Find string height from hu_font chars
//
int M_StringHeight(char* string)
{
    int      i;
    int      h;
    int      height = 8; //SHORT(hu_font[0]->height);

    h = height;
    for (i = 0;i < strlen(string);i++)
	if (string[i] == '\n')
	    h += height;

    return h;
}


//
// M_Drawer
// Called after the view has been rendered,
// but before it has been blitted.
//
void M_Drawer (void)
{
    static short	x;
    static short	y;
    short		i;
    short		max;
    char		string[40];
    int 		start,lines;

    inhelpscreens = false;

    // now that's more readable with a faded background (yeah like Quake...)
    if (menuactive)
	V_DrawFadeScreen ();

    // Horiz. & Vertically center string and print it.
    if (messageToPrint)
    {
	//added:06-02-98: now draw a textbox around the message
	max = 0;
	start = 0;
	for (lines=0; *(messageString+start); lines++)
	{
	    for (i = 0;i < strlen(messageString+start);i++)
	    {
		if (*(messageString+start+i) == '\n')
		{
		    if (i*8 > max)
			max = i*8;
		    start += i+1;
		    i = -1; //added:07-02-98:damned!
		    break;
		}
	    }

	    if (i == strlen(messageString+start))
		start += i;
	}

	start = 0;
	y = (BASEVIDHEIGHT - M_StringHeight(messageString))/2;
	M_DrawTextBox ((BASEVIDWIDTH-max-16)/2,y-8,(max+7)>>3,lines);
	while(*(messageString+start))
	{
	    for (i = 0;i < strlen(messageString+start);i++)
	    {
		if (*(messageString+start+i) == '\n')
		{
		    memset(string,0,40);
		    strncpy(string,messageString+start,i);
		    start += i+1;
		    i = -1; //added:07-02-98:damned!
		    break;
		}
	    }

	    if (i == strlen(messageString+start))
	    {
		strcpy(string,messageString+start);
		start += i;
	    }

	    x = (BASEVIDWIDTH - V_StringWidth(string))/2;
	    V_DrawString(x,y,string);
	    y += 8; //SHORT(hu_font[0]->height);
	}
	return;
    }

    if (!menuactive)
	return;

    if (currentMenu->routine)
	currentMenu->routine(); 	// call Draw routine

    // DRAW MENU
    x = currentMenu->x;
    y = currentMenu->y;
    max = currentMenu->numitems;

    for (i=0;i<max;i++)
    {
	if (currentMenu->menuitems[i].name[0] )
	    V_DrawScaledPatch (x,y,0,
			       W_CacheLumpName(currentMenu->menuitems[i].name ,PU_CACHE));
	y += LINEHEIGHT;
    }

    // DRAW THE SKULL CURSOR
    if (drawskullcursor) {
	V_DrawScaledPatch(currentMenu->x + SKULLXOFF,
			  currentMenu->y - 5 + itemOn*LINEHEIGHT,
			  0,
			  W_CacheLumpName(skullName[whichSkull],PU_CACHE) );
    }
}


//
// M_ClearMenus
//
void M_ClearMenus (void)
{
    menuactive = 0;
    // if (!netgame && usergame && paused)
    //	     sendpause = true;
}




//
// M_SetupNextMenu
//
void M_SetupNextMenu(menu_t *menudef)
{
    currentMenu = menudef;
    itemOn = currentMenu->lastOn;
    //added:08-02-98: use the Skull cursor as default
    drawskullcursor = true;
}


//
// M_Ticker
//
void M_Ticker (void)
{
    if (--skullAnimCounter <= 0)
    {
	whichSkull ^= 1;
	skullAnimCounter = 8;
    }

    //added:30-01-98:test mode for five seconds
    if( vidm_testingmode>0 )
	vidm_testingmode--;
}


//added:03-02-98: create a single colormap which remaps the hu_font colors
//		  (red) to white colors. 256bytes for a colormap is
//		  certainly much less than adding a new white font into the
//		  Iwad... simple, dirty I know, but simple.
//  Used by M_PrintWhite()
void M_SetupWhiteColormap (void)
{
    int i;

    // this one doesn't need to be aligned, unless you convert the
    // V_DrawMappedPatch() into optimised asm.
    if ( !(whitemap = (byte *) malloc(256)) )
       I_Error("Not enough mem for whitemap\n");

    for(i=0; i<256; i++)
	*(whitemap+i)=i;	//remap each color to itself...

    for(i=168;i<192;i++)
	*(whitemap+i)=i-88;	//remaps reds(168-192) to whites(80-104)
}


//
// M_Init
//
void M_Init (void)
{
    currentMenu = &MainDef;
    menuactive = 0;
    itemOn = currentMenu->lastOn;

    whichSkull = 0;
    skullAnimCounter = 10;
    //added:08-02-98: setup Skull cursor by default
    drawskullcursor = true;

    screenSize = screenblocks - 3;
    messageToPrint = 0;
    messageString = NULL;
    messageLastMenuActive = menuactive;
    quickSaveSlot = -1;

    // Here we could catch other version dependencies,
    //	like HELP1/2, and four episodes.


    switch ( gamemode )
    {
      case commercial:
	// This is used because DOOM 2 had only one HELP
	//  page. I use CREDIT as second page now, but
	//  kept this hack for educational purposes.
	MainMenu[readthis] = MainMenu[quitdoom];
	MainDef.numitems--;
	MainDef.y += 8;
	NewDef.prevMenu = &MainDef;
	ReadDef1.routine = M_DrawReadThis1;
	ReadDef1.x = 330;
	ReadDef1.y = 165;
	ReadMenu1[0].routine = M_FinishReadThis;
	break;
      case shareware:
	// Episode 2 and 3 are handled,
	//  branching to an ad screen.
      case registered:
	// We need to remove the fourth episode.
	EpiDef.numitems--;
	break;
      case retail:
	// We are fine.
      default:
	break;
    }

    //added:03-02-98:need a colormap to write text in brighter colors
    M_SetupWhiteColormap();

}

