/*
  DoomCE.c - A WinCE/Win32 port of Doom
  by Bruce Lewis
*/

#include <windows.h>
#include <malloc.h>
#include <wingdi.h>
#include "resource.h"
#include "v_video.h"
#include "wcedoom.h"
#include "console.h"

#include "d_main.h"
#include "m_misc.h"
#include "w_wad.h"
#include "z_zone.h"

void D_DoomLoop (void);

windata_t   WinData;
devinfo_t   DevInfo;

#ifdef PROFILE
int         blittick = 0, blitnumb = 0, spantick = 0, colmtick = 0, progtick = 0;
char        tstr[128];
#endif

#ifndef _WIN32_WCE
//HANDLE      hConsole;

#define wcsicmp  stricmp
#define wcslen   strlen
#define wcscpy   strcpy
#define wcstok   strtok
#define _wcsicmp strcmp
#define _wtoi    atoi
#endif

typedef struct
   {
    int OriginX;
    int OriginY;
    int Width;
    int Height;
   }blitdata_t;

blitdata_t  Src, Dest;
DWORD       dwRop = SRCCOPY;

extern int      myargc;
extern TCHAR  **myargv;

int         ProgramState = STARTUP;

SYSTEMTIME  now, then;
int         t1, t2, mytimer, bytes;

TCHAR     WadFileName[256];
#ifdef _WIN32_WCE
   TCHAR    *szAppName = TEXT("DOOMCE");
#else
   TCHAR    *szAppName = TEXT("WinDoom");
#endif
TCHAR     szFatal[24], TestMessage[256];

#ifdef _WIN32_WCE
TCHAR    *DebugFile = TEXT("/DOOMCE/DOOMCE.DBG");
#else
TCHAR    *DebugFile = TEXT("WinDoom.DBG");
#endif

BOOL             CreateMainWindow( int width, int height, int bpp, BOOL fullscreen);
LRESULT CALLBACK WndProc (HWND, UINT, WPARAM, LPARAM);

int WINAPI WinMain (HINSTANCE hInstance, HINSTANCE hPrevInstance, LPTSTR pCmdLine, int iCmdShow)
   {
    MSG         msg;
    HDC         hdc;
    static DWORD      time1, time2;
#ifndef _WIN32_WCE
    POINT       mPoint;
    int         i;  // temporary variable for numeric parameters

    //AllocConsole();
    //hConsole = GetStdHandle(STD_OUTPUT_HANDLE);

    //cprintf("This is a test of the console output...\n");
#endif

    CreateDebug();

    // Setup the unicode strings -- unicode sucks
    LoadString(hInstance, IDS_FATAL, (LPTSTR)szFatal, 24 );

    // Not changable by user
    WinData.hInstance    = hInstance;
    WinData.iCmdShow     = iCmdShow;
    WinData.wndproc      = (WNDPROC)WndProc;
    WinData.pResolutions = 0;
    WinData.NearClip     = 2.0f;
    WinData.FarClip      = 512.0f;

    // User definable
    WinData.fov          = 65.0f;
    WinData.bAllowSoft   = TRUE;
    WinData.bFullScreen  = TRUE;

    // Get the current display device info
    hdc = GetDC( NULL );
    if (GetDeviceCaps(hdc, RASTERCAPS) & RC_PALETTE)
       {
        WriteDebug("Device can support a palette...\n");
       }
    DevInfo.bpp    = GetDeviceCaps(hdc, BITSPIXEL);
    DevInfo.width  = GetSystemMetrics(SM_CXSCREEN);
    DevInfo.height = GetSystemMetrics(SM_CYSCREEN);
    ReleaseDC(NULL, hdc);

    // Parse the command line if there is one
    WinData.argc = 0;
    if (wcslen((TCHAR *)pCmdLine) > 0)
       {
        WinData.szCmdLine = (TCHAR *)pCmdLine;
        GetParameters();
       }

    WinData.hMemDC = 0;

    WinData.bpp = DevInfo.bpp;
#ifdef _WIN32_WCE
    WinData.width = DevInfo.width;
    WinData.height = DevInfo.height;
    WinData.bStretch = FALSE;
#else
    WinData.width = 640;
    WinData.height = 480;
    WinData.bStretch = TRUE;

    // Check for windowed rendering
    if (FindParameter(TEXT("-window")))
       {
        WinData.bFullScreen = FALSE;
       }
/*
    if (WinData.bFullScreen == TRUE)
       {
        WinData.width = DevInfo.width;
        WinData.height = DevInfo.height;
        WinData.bpp = DevInfo.bpp;
       }
    else
       {
*/
        // Default to 640 pixels wide
        if ((i = FindNumParameter(TEXT("-width"))) != -1)
           {
            WinData.width = i;
           }

        // Default to 480 pixels high
        if ((i = FindNumParameter(TEXT("-height"))) != -1)
           {
            WinData.height = i;
           }

    if (FindParameter(TEXT("-nostretch")))
       {
        WinData.bStretch = FALSE;
       }
//       }
#endif

#ifndef _WIN32_WCE
    if (WinData.bFullScreen == TRUE)
       {
        BuildModeList();
        if (!SetVideoMode())
           {
            Cleanup();
            return 0;
           }
       }
#endif

    // Create the main program window
    if (CreateMainWindow( WinData.width, WinData.height, WinData.bpp, WinData.bFullScreen) != TRUE)
       {
        FatalError(IDS_NOMAIN);
        Cleanup();
        return 0;
       }

#ifndef _WIN32_WCE
    ShowCursor(FALSE);
    mPoint.x = WinData.width/2;
    mPoint.y = WinData.height/2;
    ClientToScreen( WinData.hWnd, &mPoint);
    SetCursorPos(mPoint.x, mPoint.y);
    SetCapture(WinData.hWnd);
#endif

#ifdef _WIN32_WCE
    if ((WinData.iwad = FileOpen(TEXT("DoomCE/DOOM1.WP1"))) == INVALID_HANDLE_VALUE)
       {
        FatalError(IDS_NOIWAD);
        Cleanup();
        return 0;
       }
    FileClose(WinData.iwad);
#endif

    WinData.hDC = GetDC(WinData.hWnd);
    Dest.OriginX = (WinData.width-320)/2;
    Dest.OriginY = (WinData.height-200)/2;
    Dest.Width = Src.Width = 320;
    Dest.Height = Src.Height = 200;
    Src.OriginX = 0;
    Src.OriginY = 0;

    con_printf(TEXT("DoomCE - Win32/Windows95/WindowsNT/WindowsCE Doom\n"));
    
    // We're live now
    WinData.bActive = TRUE;

    // Begin the main program loop
    t1 = GetTickCount();
    t2 = t1;

    mytimer = 0;
    while (WinData.bActive == TRUE)
       {
        while (PeekMessage(&msg, NULL, 0, 0, PM_REMOVE))
           {
            TranslateMessage (&msg);
            DispatchMessage (&msg);
           }
        // put game loop here
        switch(ProgramState)
           {
            case STARTUP:
                 con_printf(TEXT("Calling D_DoomMain to setup the program...\n"));
                 D_DoomMain();
                 con_printf(TEXT("Setup complete...\n"));
                 time1 = GetTickCount();
                 con_clear();
                 break;
            case RUN:
                 D_DoomLoop();
                 break;
            case ABORT:
            case SHUTDOWN:
                 SendMessage(WinData.hWnd, WM_CLOSE, 0, 0);
                 break;
           }
       }

    // Release the parameter and video resolution lists
#ifdef PROFILE
    time2 = GetTickCount();
    progtick = time2 - time1;
#endif
    Cleanup();

    WriteDebug("DoomCE Shutdown.\n");

    return msg.wParam;
   }

void GetParameters()
   {
    int    count;
    TCHAR *s, *tstring;

    // Make a copy of the command line to count the parameters - strtok is destructive
    tstring = (TCHAR *)malloc(sizeof(TCHAR)*(wcslen(WinData.szCmdLine)+1));
    wcscpy(tstring, WinData.szCmdLine);

    // Count the parameters
    s = wcstok(tstring, TEXT(" "));
    count = 1;
    while (wcstok(NULL, TEXT(" ")) != NULL)
       {
        count++;
       }
    free(tstring);

    // Allocate "pockets" for the parameters
    WinData.argv = (TCHAR **)malloc(sizeof(TCHAR*)*(count+1));

    // Copy first parameter into the "pockets"
    WinData.argc = 0;
    s = wcstok(WinData.szCmdLine, TEXT(" "));
    WinData.argv[WinData.argc] = (TCHAR *)malloc(sizeof(TCHAR)*(wcslen(s)+1));
    wcscpy(WinData.argv[WinData.argc], s);
    WinData.argc++;

    // Copy the rest of the parameters
    do
       {
        // get the next token
        s = wcstok(NULL, TEXT(" "));
        if (s != NULL)
           { 
            // add it to the list
            WinData.argv[WinData.argc] = (TCHAR *)malloc(sizeof(TCHAR)*(wcslen(s)+1));
            wcscpy(WinData.argv[WinData.argc], s);
            WinData.argc++;
           }
       }
    while (s != NULL);

    myargc = WinData.argc;
    myargv = WinData.argv;
   }

void Cleanup()
   {
    int i;

#ifdef PROFILE
    int fps;

    WriteDebug("Blit info: ticks ");
    itoa(blittick, tstr, 10);
    WriteDebug(tstr);
    WriteDebug(" : Blit count : ");
    itoa(blitnumb, tstr, 10);
    WriteDebug(tstr);
    WriteDebug("\n");

    WriteDebug("Column info: ticks ");
    itoa(colmtick, tstr, 10);
    WriteDebug(tstr);
    WriteDebug("\n");

    WriteDebug("Span info: ticks ");
    itoa(spantick, tstr, 10);
    WriteDebug(tstr);
    WriteDebug("\n");

    WriteDebug("Total ticks ");
    itoa(progtick, tstr, 10);
    WriteDebug(tstr);
    WriteDebug("\n");

    fps = blitnumb/(progtick/1000);
    WriteDebug("Frames per second : ");
    itoa(fps, tstr, 10);
    WriteDebug(tstr);
    WriteDebug("\n");
#endif

#ifndef _WIN32_WCE
    ShowCursor(TRUE);
    ReleaseCapture();
    ChangeDisplaySettings(0, 0);
#endif

    if (WinData.bmi != 0)
       {
        free(WinData.bmi);
        WinData.bmi = 0;
       }

    con_shutdown();

    if (WinData.scrnbuff != 0)
       {
        LocalFree(WinData.scrnbuff);
       }

    // Free the command line holder memory
    if (WinData.argc > 0)
       {
        // Free in reverse order of allocation
        for (i = (WinData.argc-1); i >= 0; i--)
           {
            free(WinData.argv[i]);
           }
        // Free the parameter "pockets"
        free(WinData.argv);
       }
    // Free the memory that holds the video resolution list
    if (WinData.pResolutions)
        free(WinData.pResolutions);

    if (WinData.hMemDC)
       {
        ReleaseDC(WinData.hWnd, WinData.hMemDC);
       }
    if (WinData.hDC)
       {
        ReleaseDC(WinData.hWnd, WinData.hDC);
       }
   }

BOOL isdigits( TCHAR *s )
   {
    int i;

    for (i = 0; s[i]; i++)
       {
        if (!iswdigit(s[i]))
           {
            return FALSE;
           }
       }
    return TRUE;
   }

int strnicmp(const char *a, const char *b, size_t n)
   {
    char s1[256], s2[256];

    strncpy(s1, a, n);
    s1[n] = '\0';
    strupr(s1);
    strncpy(s2, b, n);
    s2[n] = '\0';
    strupr(s2);
    return (strncmp(s1, s2, n));
   }

#ifdef _WIN32_WCE
char toupper(char c)
   {
    if ((c >= 'a') && (c <= 'z'))
       return (c - 32);
    else
       return c;
   }

int stricmp(char *s, char *d)
   {
    int i;

    i = 0;
    while (s[i] && d[i])
       {
        if (s[i] != d[i])
           return s[i]-d[i];
        i++;
       }
    return s[i]-d[i];
   }

int atoi(TCHAR *);

#endif

int FindNumParameter(TCHAR *s)
   {
    int i, notfound = -1;

    for (i = 0; i < (WinData.argc-1); i++)
       {
        if (_wcsicmp(WinData.argv[i], s) == 0)
           {
            if (isdigits(WinData.argv[i+1]) == TRUE)
               {
                return(_wtoi(WinData.argv[i+1]));
               }
            else
               {
                return notfound;
               }
           }
       }
     return notfound;
   }

BOOL FindParameter(TCHAR *s)
   {
    int i;

    for (i = 0; i < WinData.argc; i++)
       {
        if (wcsicmp(WinData.argv[i], s) == 0)
           {
            return TRUE;
           }
       }
    return FALSE;
   }

BOOL CreateMainWindow(int width, int height, int bpp, BOOL fullscreen)
   {
    HWND        hwnd;
    WNDCLASS    wndclass;
    DWORD       dwStyle, dwExStyle;
    int         x, y, sx, sy, ex, ey, ty;

    if ((hwnd = FindWindow(szAppName, szAppName)) != NULL)
       {
        SetForegroundWindow(hwnd);
        return 0;
       }

    wndclass.style         = CS_HREDRAW | CS_VREDRAW;
    wndclass.lpfnWndProc   = (WNDPROC)WinData.wndproc;
    wndclass.cbClsExtra    = 0;
    wndclass.cbWndExtra    = 0;
    wndclass.hInstance     = WinData.hInstance;
    wndclass.hIcon         = 0;
    wndclass.hCursor       = NULL;
    wndclass.hbrBackground = (void *)COLOR_GRAYTEXT;
    wndclass.lpszMenuName  = NULL;
    wndclass.lpszClassName = (LPCTSTR)szAppName;

    if (!RegisterClass(&wndclass))
       {
        FatalError(IDS_NOMAIN);
        return FALSE;
       }

    if (fullscreen)
       {
        dwExStyle = WS_EX_TOPMOST;
        dwStyle = WS_POPUP | WS_VISIBLE;
        x = y = 0;
        //x = (DevInfo.width - WinData.width)/2;
        sx = width;
        sy = height;
       }
    else
       {
        dwExStyle = 0;
        dwStyle = WS_CAPTION;
        ex = GetSystemMetrics(SM_CXEDGE);
        ey = GetSystemMetrics(SM_CYEDGE);
#ifdef _WIN32_WCE
        ty = 0;
#else
        ty = GetSystemMetrics(SM_CYSIZE)+4;
#endif
        // Center the window on the screen
        x = (DevInfo.width / 2) - ((WinData.width+(2*ex)) / 2);
        y = (DevInfo.height / 2) - ((WinData.height+(2*ey)+ty) / 2);
        sx = WinData.width+(2*ex);
        sy = WinData.height+(2*ey)+ty;
        /*
           Check to be sure the requested window size fits on the screen and
           adjust each dimension to fit if the requested size does not fit.
        */
        if (sx >= DevInfo.width)
           {
            x = 0;
            sx = DevInfo.width-(2*ex);
           }
        if (sy >= DevInfo.height)
           {
            y = 0;
            sy = DevInfo.height-((2*ey)+ty);
           }
       }

    if ((hwnd = CreateWindowEx (dwExStyle,
                    szAppName,               // window class name
		            szAppName,               // window caption
                    dwStyle,                 // window style
                    x,                       // initial x position
                    y,                       // initial y position
                    sx,                      // initial x size
                    sy,                      // initial y size
                    NULL,                    // parent window handle
                    NULL,                    // window menu handle
                    WinData.hInstance,       // program instance handle
		            NULL))                   // creation parameters
                    == NULL)
       { 
        FatalError(IDS_NOCREATE);
        return FALSE;
       }

    WinData.hWnd = hwnd;

    ShowWindow(WinData.hWnd, WinData.iCmdShow);
    UpdateWindow(WinData.hWnd);

    SetForegroundWindow(WinData.hWnd);
    SetFocus(WinData.hWnd);

    return TRUE;
   }

static int	lastmousex = 0;
static int	lastmousey = 0;
boolean		mousemoved = false;

LRESULT CALLBACK WndProc (HWND hwnd, UINT iMsg, WPARAM wParam, LPARAM lParam)
   {
    static event_t  event;
	POINT           pt;
#ifndef _WIN32_WCE
    POINT           mPoint;
#endif

    switch (iMsg)
       {
        case WM_CREATE:
             con_setup(hwnd, WinData.width, WinData.height);
             break;

		case WM_LBUTTONDOWN:
		case WM_RBUTTONDOWN:
	         event.type  = ev_mouse;
             event.data1 =  (wParam & MK_LBUTTON ? 1 : 0)
                          | (wParam & MK_MBUTTON ? 2 : 0)
                          | (wParam & MK_RBUTTON ? 4 : 0);
			 event.data2 = event.data3 = 0;
			 D_PostEvent(&event);
			 break;

		case WM_LBUTTONUP:
		case WM_RBUTTONUP:
             event.type  = ev_mouse;
             event.data1 =  (wParam & MK_LBUTTON ? 1 : 0)
                          | (wParam & MK_MBUTTON ? 2 : 0)
                          | (wParam & MK_RBUTTON ? 4 : 0);
             event.data2 = event.data3 = 0;
             D_PostEvent(&event);
             break;

		case WM_MOUSEMOVE:
	         pt.x = (int)LOWORD(lParam);
	         pt.y = (int)HIWORD(lParam);
             event.type  = ev_mouse;
#ifdef _WIN32_WCE
             event.data1 = 1; // UNDER_CE will always be left button
#else
             event.data1 =  (wParam & MK_LBUTTON ? 1 : 0)
                          | (wParam & MK_MBUTTON ? 2 : 0)
                          | (wParam & MK_RBUTTON ? 4 : 0);
#endif
             event.data2 = (pt.x - lastmousex) << 2;
             event.data3 = (lastmousey - pt.y) << 2;
             if (event.data2 || event.data3)
                {
                 lastmousex = pt.x;
                 lastmousey = pt.y;
                 if ((pt.x != WinData.width/2) && (pt.y != WinData.height/2))
                    {
                     D_PostEvent(&event);
                     mousemoved = false;
#ifndef _WIN32_WCE
                     mPoint.x = WinData.width/2;
                     mPoint.y = WinData.height/2;
                     ClientToScreen( WinData.hWnd, &mPoint);
                     SetCursorPos(mPoint.x, mPoint.y);
#endif
                    }
                 else
                    {
                     mousemoved = true;
                    }
                }
             break;		

        case WM_KEYUP:
#ifdef _WIN32_WCE
             if (wParam == 'G')  // had to use 'G' because there ISN'T an F11 key...
                {
                 event.type = ev_keyup;
                 event.data1 = KEY_F11;
                 D_PostEvent(&event);
                 break;
                }
             else
#endif
             if ((wParam >= 'A') && (wParam <= 'Z'))
                {
                 event.type = ev_keyup;
                 event.data1 = wParam + 32;
                 D_PostEvent(&event);
                 break;
                }
             else
             if ((wParam >= '0') && (wParam <= '9'))
                {
                 event.type = ev_keyup;
                 event.data1 = wParam;
                 D_PostEvent(&event);
                 break;
                }
             else
                {
             switch(wParam)
                {
                 case VK_MENU:
                      event.type = ev_keyup;
                      event.data1 = KEY_RALT;
                      D_PostEvent(&event);
                      break;
                 case VK_TAB:
                      event.type = ev_keyup;
                      event.data1 = KEY_TAB;
                      D_PostEvent(&event);
                      break;
                 case '=':
                      event.type = ev_keydown;
                      event.data1 = KEY_EQUALS;
                      D_PostEvent(&event);
                      break;
                 case '-':
                      event.type = ev_keydown;
                      event.data1 = KEY_MINUS;
                      D_PostEvent(&event);
                      break;
                 case ' ':
                 case '[':
                 case ']':
                 case ',':
                 case '.':
                 case '/':
                 case '\\':
                 case '`':
                 case ';':
                      event.type = ev_keyup;
                      event.data1 = wParam;
                      D_PostEvent(&event);
                      break;
                 case VK_SHIFT:
                      event.type = ev_keyup;
                      event.data1 = KEY_RSHIFT;
                      D_PostEvent(&event);
                      break;
                 case VK_CONTROL:
                      event.type = ev_keyup;
                      event.data1 = KEY_RCTRL;
                      D_PostEvent(&event);
                      break;
                 case VK_BACK:
                      event.type = ev_keyup;
                      event.data1 = KEY_BACKSPACE;
                      D_PostEvent(&event);
                      break;
                 case VK_F1:
                      event.type = ev_keyup;
                      event.data1 = KEY_F1;
                      D_PostEvent(&event);
                      break;
                 case VK_F2:
                      event.type = ev_keyup;
                      event.data1 = KEY_F2;
                      D_PostEvent(&event);
                      break;
                 case VK_F3:
                      event.type = ev_keyup;
                      event.data1 = KEY_F3;
                      D_PostEvent(&event);
                      break;
                 case VK_F4:
                      event.type = ev_keyup;
                      event.data1 = KEY_F4;
                      D_PostEvent(&event);
                      break;
                 case VK_F5:
                      event.type = ev_keyup;
                      event.data1 = KEY_F5;
                      D_PostEvent(&event);
                      break;
                 case VK_F6:
                      event.type = ev_keyup;
                      event.data1 = KEY_F6;
                      D_PostEvent(&event);
                      break;
                 case VK_F7:
                      event.type = ev_keyup;
                      event.data1 = KEY_F7;
                      D_PostEvent(&event);
                      break;
                 case VK_F8:
                      event.type = ev_keyup;
                      event.data1 = KEY_F8;
                      D_PostEvent(&event);
                      break;
                 case VK_F9:
                      event.type = ev_keyup;
                      event.data1 = KEY_F9;
                      D_PostEvent(&event);
                      break;
                 case VK_F10:
                      event.type = ev_keyup;
                      event.data1 = KEY_F10;
                      D_PostEvent(&event);
                      break;
                 case VK_F11:
                      event.type = ev_keyup;
                      event.data1 = KEY_F11;
                      D_PostEvent(&event);
                      break;
                 case VK_RETURN:
                      event.type = ev_keyup;
                      event.data1 = KEY_ENTER;
                      D_PostEvent(&event);
                      break;
                 case VK_UP:
                      event.type = ev_keyup;
                      event.data1 = KEY_UPARROW;
                      D_PostEvent(&event);
                      break;
                 case VK_DOWN:
                      event.type = ev_keyup;
                      event.data1 = KEY_DOWNARROW;
                      D_PostEvent(&event);
                      break;
                 case VK_LEFT:
                      event.type = ev_keyup;
                      event.data1 = KEY_LEFTARROW;
                      D_PostEvent(&event);
                      break;
                 case VK_RIGHT:
                      event.type = ev_keyup;
                      event.data1 = KEY_RIGHTARROW;
                      D_PostEvent(&event);
                      break;
                 case VK_ESCAPE:
                      event.type = ev_keyup;
                      event.data1 = KEY_ESCAPE;
                      D_PostEvent(&event);
                      break;
                }
                }
             break;

        case WM_KEYDOWN:
             if (lParam & 0x4000)  // debounce the key - if it's already down ignore it
                {
                 break;
                }
#ifdef _WIN32_WCE
             if (wParam == 'G')  // had to use 'G' because there ISN'T an F11 key...
                {
                 event.type = ev_keydown;
                 event.data1 = KEY_F11;
                 D_PostEvent(&event);
                 break;
                }
             else
#endif
             if ((wParam >= 'A') && (wParam <= 'Z'))
                {
                 event.type = ev_keydown;
                 event.data1 = wParam + 32;
                 D_PostEvent(&event);
                 break;
                }
             else
             if ((wParam >= '0') && (wParam <= '9'))
                {
                 event.type = ev_keydown;
                 event.data1 = wParam;
                 D_PostEvent(&event);
                 break;
                }
             else
                {
             switch(wParam)
                {
                 case VK_MENU:
                      event.type = ev_keydown;
                      event.data1 = KEY_RALT;
                      D_PostEvent(&event);
                      break;
                 case VK_TAB:
                      event.type = ev_keydown;
                      event.data1 = KEY_TAB;
                      D_PostEvent(&event);
                      break;
                 case '=':
                      event.type = ev_keydown;
                      event.data1 = KEY_EQUALS;
                      D_PostEvent(&event);
                      break;
                 case '-':
                      event.type = ev_keydown;
                      event.data1 = KEY_MINUS;
                      D_PostEvent(&event);
                      break;
                 case ' ':
                 case '[':
                 case ']':
                 case ',':
                 case '.':
                 case '/':
                 case '\\':
                 case '`':
                 case ';':
                      event.type = ev_keydown;
                      event.data1 = wParam;
                      D_PostEvent(&event);
                      break;
                 case VK_SHIFT:
                      event.type = ev_keydown;
                      event.data1 = KEY_RSHIFT;
                      D_PostEvent(&event);
                      break;
                 case VK_CONTROL:
                      event.type = ev_keydown;
                      event.data1 = KEY_RCTRL;
                      D_PostEvent(&event);
                      break;
                 case VK_BACK:
                      event.type = ev_keydown;
                      event.data1 = KEY_BACKSPACE;
                      D_PostEvent(&event);
                      break;
                 case VK_F1:
                      event.type = ev_keydown;
                      event.data1 = KEY_F1;
                      D_PostEvent(&event);
                      break;
                 case VK_F2:
                      event.type = ev_keydown;
                      event.data1 = KEY_F2;
                      D_PostEvent(&event);
                      break;
                 case VK_F3:
                      event.type = ev_keydown;
                      event.data1 = KEY_F3;
                      D_PostEvent(&event);
                      break;
                 case VK_F4:
                      event.type = ev_keydown;
                      event.data1 = KEY_F4;
                      D_PostEvent(&event);
                      break;
                 case VK_F5:
                      event.type = ev_keydown;
                      event.data1 = KEY_F5;
                      D_PostEvent(&event);
                      break;
                 case VK_F6:
                      event.type = ev_keydown;
                      event.data1 = KEY_F6;
                      D_PostEvent(&event);
                      break;
                 case VK_F7:
                      event.type = ev_keydown;
                      event.data1 = KEY_F7;
                      D_PostEvent(&event);
                      break;
                 case VK_F8:
                      event.type = ev_keydown;
                      event.data1 = KEY_F8;
                      D_PostEvent(&event);
                      break;
                 case VK_F9:
                      event.type = ev_keydown;
                      event.data1 = KEY_F9;
                      D_PostEvent(&event);
                      break;
                 case VK_F10:
                      event.type = ev_keydown;
                      event.data1 = KEY_F10;
                      D_PostEvent(&event);
                      break;
                 case VK_F11:
                      event.type = ev_keydown;
                      event.data1 = KEY_F11;
                      D_PostEvent(&event);
                      break;
                 case VK_RETURN:
                      event.type = ev_keydown;
                      event.data1 = KEY_ENTER;
                      D_PostEvent(&event);
                      break;
                 case VK_UP:
                      event.type = ev_keydown;
                      event.data1 = KEY_UPARROW;
                      D_PostEvent(&event);
                      break;
                 case VK_DOWN:
                      event.type = ev_keydown;
                      event.data1 = KEY_DOWNARROW;
                      D_PostEvent(&event);
                      break;
                 case VK_LEFT:
                      event.type = ev_keydown;
                      event.data1 = KEY_LEFTARROW;
                      D_PostEvent(&event);
                      break;
                 case VK_RIGHT:
                      event.type = ev_keydown;
                      event.data1 = KEY_RIGHTARROW;
                      D_PostEvent(&event);
                      break;
                 case VK_ESCAPE:
                      event.type = ev_keydown;
                      event.data1 = KEY_ESCAPE;
                      D_PostEvent(&event);
                      break;
                }
             }
             break;

        case WM_ACTIVATE:
             if ((LOWORD(wParam) != WA_INACTIVE) && ((HWND)lParam == NULL))
                {
                 ShowWindow(WinData.hWnd, SW_SHOW);
                 SetForegroundWindow(WinData.hWnd);
                }
             else
                {
                 if (WinData.bFullScreen)
                    {
                     ShowWindow(WinData.hWnd, SW_HIDE);
                    }
                }
             return 0;

        case WM_CLOSE:
             //ShutdownGame();
             WinData.bActive = FALSE;
             break;

        case WM_DESTROY:
             PostQuitMessage (0);
             return 0;
       }

    return DefWindowProc (hwnd, iMsg, wParam, lParam);
   }

void FatalError(int Message)
   {
    TCHAR szMessage[256];

    LoadString(WinData.hInstance, Message, szMessage, 256);
    MessageBox(NULL, szMessage, szFatal, MB_OK);
   }

void AbortProgram()
   {
    //SendMessage(WinData.hWnd, WM_CLOSE, 0, 0 );
    ProgramState = ABORT;
   }

//LOGPALETTE   *plGamePal;
//HPALETTE      hPalette;
char          errstr[256];
DWORD         errval;

void CreateBlitSurface()
   {
#ifndef _WIN32_WCE
    int             i;
    unsigned char  *tpalette;

#endif

    WinData.bmi = (BITMAPINFO *)malloc(sizeof(BITMAPINFO)+(sizeof(RGBQUAD)*256));

    WinData.bmi->bmiHeader.biSize = sizeof(BITMAPINFOHEADER);
    WinData.bmi->bmiHeader.biWidth = 320;
    WinData.bmi->bmiHeader.biHeight = 200;
    WinData.bmi->bmiHeader.biPlanes = 1;
    WinData.bmi->bmiHeader.biBitCount = 8;
    WinData.bmi->bmiHeader.biCompression   = BI_RGB;
    WinData.bmi->bmiHeader.biSizeImage     = 320*200;
    WinData.bmi->bmiHeader.biXPelsPerMeter = 0;
    WinData.bmi->bmiHeader.biYPelsPerMeter = 0;
    WinData.bmi->bmiHeader.biClrUsed       = 256;
    WinData.bmi->bmiHeader.biClrImportant  = 256;

#ifndef _WIN32_WCE
    tpalette = W_CacheLumpName("PLAYPAL",PU_CACHE);
    for (i = 0; i < 256; i++)
       {
        WinData.bmi->bmiColors[i].rgbRed = gammatable[usegamma][*tpalette++];
        WinData.bmi->bmiColors[i].rgbGreen = gammatable[usegamma][*tpalette++];
        WinData.bmi->bmiColors[i].rgbBlue = gammatable[usegamma][*tpalette++];
        WinData.bmi->bmiColors[i].rgbReserved = 0;
       }

#endif

#ifdef DIRECT_COLOR
    if ((WinData.hBitmap = CreateDIBSection( WinData.hDC, WinData.bmi, DIB_PAL_COLORS, (void *)&WinData.blitbuff, NULL, 0)) == NULL)
#else
    if ((WinData.hBitmap = CreateDIBSection( WinData.hDC, WinData.bmi, DIB_RGB_COLORS, (void *)&WinData.blitbuff, NULL, 0)) == NULL)
#endif
       {
        errval = GetLastError();
        _itoa(errval, errstr, 16);
        WriteDebug("CreateDIBSection failed : ");
        WriteDebug(errstr);
        WriteDebug("\n");
        MessageBox(WinData.hWnd, TEXT("CreateDIBSection FAILED"), TEXT("DOOMCE Error"), MB_OK);

        SendMessage(WinData.hWnd, WM_CLOSE, 0, 0);
       }
    else
    if ((WinData.hMemDC = CreateCompatibleDC(WinData.hDC)) == NULL)
       {
        errval = GetLastError();
        WriteDebug("CreateCompatibleDC failed : ");
        _itoa(errval, errstr, 16);
        WriteDebug(errstr);
        WriteDebug("\n");
        SendMessage(WinData.hWnd, WM_CLOSE, 0, 0);
       }
#ifndef _WIN32_WCE
    SelectObject(WinData.hMemDC, WinData.hBitmap);
    SetDIBColorTable(WinData.hMemDC, 0, 256, WinData.bmi->bmiColors);
#endif

	WinData.scrnbuff = (LPBYTE)LocalAlloc(LMEM_FIXED, 320 * 200);
	screens[0] = (unsigned char *)WinData.scrnbuff;

   }

void BlitScreen()
   {
    int            y;
    unsigned char *lp1, *lp2;
#ifdef PROFILE
    DWORD   time1, time2;

    time1 = GetTickCount();
#endif

    if (WinData.blitbuff == 0)
       return;

    lp1 = WinData.scrnbuff;
    lp2 = WinData.blitbuff + (SCREENHEIGHT - 1) * SCREENWIDTH;
    for (y = 0; y < SCREENHEIGHT; y++)
       {
        memcpy(lp2, lp1, SCREENWIDTH);
        lp1 += SCREENWIDTH;
        lp2 -= SCREENWIDTH;
       }

    SelectObject(WinData.hMemDC, WinData.hBitmap);

    if (WinData.bStretch == FALSE)
       {
        if (!BitBlt(WinData.hDC, Dest.OriginX, Dest.OriginY, Dest.Width, Dest.Height,
                    WinData.hMemDC, Src.OriginX, Src.OriginY, SRCCOPY))
           {
            WriteDebug("BitBlt failed.\n");
           }
       }
    else
       {
        if (!StretchBlt(WinData.hDC, 0, 0, WinData.width, WinData.height,
                        WinData.hMemDC, Src.OriginX, Src.OriginY, Src.Width, Src.Height, SRCCOPY))
           {
            WriteDebug("StretchBlt failed.\n");
           }
       }

#ifdef PROFILE
    time2 = GetTickCount();
    blittick += (time2-time1);
    blitnumb++;
#endif
   }

int FileAccess(LPCTSTR FileName, DWORD mode)
   {
    HANDLE hFile;

    hFile = CreateFile( FileName, GENERIC_READ, FILE_SHARE_READ, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL );

    if (hFile == INVALID_HANDLE_VALUE)
        return -1;
    else
       {
        FileClose(hFile);
        return 0;
       }
   }

HANDLE FileCreate(LPCTSTR FileName)
   {
    HANDLE hFile;

    if (FileAccess( FileName, 0) == 0)
        hFile = CreateFile( FileName, GENERIC_READ | GENERIC_WRITE, FILE_SHARE_READ, NULL, TRUNCATE_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL );
    else
        hFile = CreateFile( FileName, GENERIC_READ | GENERIC_WRITE, FILE_SHARE_READ, NULL, OPEN_ALWAYS, FILE_ATTRIBUTE_NORMAL, NULL );
    
    return hFile;
   }

void FileClose(HANDLE hFile)
   {
    CloseHandle(hFile);
   }

DWORD FileLength(HANDLE hFile)
   {
    return GetFileSize(hFile, NULL);
   }

HANDLE FileOpen(LPCTSTR FileName)
   {
    HANDLE hFile;

    hFile = CreateFile( FileName, GENERIC_READ, FILE_SHARE_READ, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL );

    return hFile;
   }

HANDLE FileAppend(LPCTSTR FileName)
   {
    HANDLE hFile;

    hFile = CreateFile( FileName, GENERIC_READ | GENERIC_WRITE, FILE_SHARE_READ, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL );
    FileSeek(hFile, 0, FILE_END );

    return hFile;
   }

DWORD FileRead(HANDLE hFile, LPVOID data, DWORD size)
   {
    DWORD readin = 0;

    ReadFile(hFile, data, size, &readin, NULL);

    return readin;
   }

DWORD FileSeek(HANDLE hFile, LONG distance, DWORD method)
   {
    DWORD position;

    position = SetFilePointer(hFile, distance, NULL, method);

    return position;
   }

DWORD FileWrite(HANDLE hFile, LPCVOID data, DWORD size)
   {
    DWORD written = 0;

    WriteFile(hFile, data, size, &written, NULL);

    return written;
   }

unsigned char digits[] = "0123456789ABCDEF";

#ifdef _WIN32_WCE
void itoa(int  i, char *s, int base)
   {
    char t[32];
    int  n, x, negative = 0;

    x = 31;
    t[x] = 0;

    if (i == 0)
       {
        t[0] = '\0';
        s[0] = '0';
        s[1] = '\0';
        return;
       }
    else
    if (i < 0)
       {
        negative = 1;
        i = 0 - i;
       }

    n = i;
    while (n > 0)
       {
        t[--x] = digits[n%base];
        n /= base;
       }
    if (negative)
       t[--x] = '-';
    strcpy(s, &t[x]);
   }
#endif

void CreateDebug()
   {
    HANDLE dbghandle;

    dbghandle = FileCreate(DebugFile);

    FileClose(dbghandle);

#ifdef _WIN32_WCE
    WriteDebug("Windows CE Doom - Debug File\n");
#else
    WriteDebug("Windows Doom - Debug File\n");
#endif
   }

#ifndef _WIN32_WCE
// This function builds a list of the screen resolutions supported by the display driver
void BuildModeList()
   {
    DEVMODE  dm;
    int      mode;

    mode = 0;
    while(EnumDisplaySettings(NULL, mode, &dm))
       {
        mode++;
       }

    WinData.pResolutions = (screen_res_t *)malloc(sizeof(screen_res_t)*mode);
    mode = 0;
    while(EnumDisplaySettings(NULL, mode, &dm))
       {
        WinData.pResolutions[mode].width = dm.dmPelsWidth;
        WinData.pResolutions[mode].height = dm.dmPelsHeight;
        WinData.pResolutions[mode].bpp = dm.dmBitsPerPel;
        WinData.pResolutions[mode].flags = dm.dmDisplayFlags;
        WinData.pResolutions[mode].frequency = dm.dmDisplayFrequency;
        mode++;
       }
    WinData.iResCount = mode;
   }

BOOL SetVideoMode()
   {
    OSVERSIONINFO   vinfo;
    int             mode;
    DEVMODE         dm;

    vinfo.dwOSVersionInfoSize = sizeof(vinfo);

    WinData.bChangeBPP = FALSE;

    if ( GetVersionEx( &vinfo) )
       {
        if ( vinfo.dwMajorVersion > 4 )
           {
            WinData.bChangeBPP = TRUE;
           }
        else
        if ( vinfo.dwMajorVersion == 4 )
           {
            if ( vinfo.dwPlatformId == VER_PLATFORM_WIN32_NT )
               {
                WinData.bChangeBPP = TRUE;
               }
            else
            if ( vinfo.dwPlatformId == VER_PLATFORM_WIN32_WINDOWS )
               {
                if ( LOWORD( vinfo.dwBuildNumber ) >= OSR2_BUILD_NUMBER )
                   {
                    WinData.bChangeBPP = TRUE;
                   }
               }
           }
       }
    else
       {
        MessageBox(NULL, "SetVideoMode - GetVersionEx failed\n", "FATAL ERROR", MB_OK);
        return FALSE;
       }

    if (WinData.bFullScreen)
       {
        if ((WinData.bpp    == DevInfo.bpp) &&
            (WinData.width  == DevInfo.width) &&
            (WinData.height == DevInfo.height))
           {
            WinData.iVidMode = 0;
            return TRUE;
           }
        if ((WinData.bChangeBPP == FALSE) && (DevInfo.bpp != WinData.bpp))
           {
            MessageBox(NULL, "This version of Windows cannot change color depth.\n"
                             "Please request different video mode settings or adjust\n"
                             "your desktop color depth.", "FATAL ERROR", MB_OK);
            return FALSE;
           }
        for (mode = 0; mode < WinData.iResCount; mode++)
           {
            if ((WinData.pResolutions[mode].width == WinData.width) &&
                (WinData.pResolutions[mode].height == WinData.height) &&
                (WinData.pResolutions[mode].bpp == WinData.bpp))
               {
                WinData.iVidMode = mode;

                memset(&dm, 0, sizeof(dm));
                dm.dmSize = sizeof(dm);

                dm.dmPelsWidth  = WinData.width;
                dm.dmPelsHeight = WinData.height;
                dm.dmFields     = DM_PELSWIDTH | DM_PELSHEIGHT;

                if (WinData.bpp != DevInfo.bpp)
                   {
                    dm.dmBitsPerPel = WinData.bpp;
                    dm.dmFields |= DM_BITSPERPEL;
                   }

                if ( ChangeDisplaySettings( &dm, CDS_FULLSCREEN ) != DISP_CHANGE_SUCCESSFUL )
                   {
                    MessageBox(NULL, "SetVideoMode - ChangeDisplaySettings failed.\n"
                                     "Switching to windowed mode.", "WARNING", MB_OK);
                    WinData.bFullScreen = FALSE;
                    return TRUE;
                   }
                return TRUE;
               }
           }
        MessageBox(NULL, "Your requested video mode is unavailable.\n"
                         "Please request different video mode settings.", "FATAL ERROR", MB_OK);
        return FALSE;
       }
    else
       {
        if (DevInfo.bpp != WinData.bpp)
           {
            MessageBox(NULL, "Your requested color depth and desktop do not match.\n"
                             "Using your current desktop color depth.", "WARNING", MB_OK);
           }
       }
    return TRUE;
   }
#endif

void WriteDebug(char *s)
   {
    HANDLE dbghandle;

    dbghandle = FileAppend(DebugFile);

    FileWrite(dbghandle, s, strlen(s));

    FileClose(dbghandle);
   }

#ifdef _WIN32_WCE  // For WindowsCE we need to convert

char    tempstr[256];
TCHAR  wtempstr[256];

TCHAR *StrToUnicode(char *string)
   {
    MultiByteToWideChar(CP_ACP, MB_COMPOSITE, string, -1, wtempstr, 255);
    return wtempstr;
   }

char *StrFromUnicode(TCHAR *string)
   {
    WideCharToMultiByte(CP_ACP, WC_COMPOSITECHECK | WC_DEFAULTCHAR | WC_SEPCHARS, string, -1, tempstr, sizeof(tempstr)-1, NULL, NULL);
    return tempstr;
   }

#else  // For standard Win32 no conversion is necessary

TCHAR *StrToUnicode(char *string)
   {
    return string;
   }

char *StrFromUnicode(TCHAR *string)
   {
    return string;
   }

#endif

/*
#ifndef _WIN32_WCE

void cprintf(char *message, ... )
   {
    va_list	argptr;
    DWORD   written;
    TCHAR   cmsg[1024];

    va_start (argptr,message);
    vsprintf (cmsg,message,argptr);
    va_end (argptr);

    WriteConsole(hConsole, cmsg, strlen(cmsg), &written, NULL ); 
   }
#endif
*/