/*
 * qstat.h
 * by Steve Jankowski
 * steve@activesw.com
 * http://www.activesw.com/people/steve/qstat.html
 *
 * Copyright 1996,1997,1998,1999 by Steve Jankowski
 */

#ifdef unix
#define _ISUNIX
#endif
#ifdef __unix
#define _ISUNIX
#endif
#ifdef VMS
#define _ISUNIX
#endif
#ifdef _AIX
#define _ISUNIX
#endif

#ifdef _ISUNIX
#include <sys/time.h>
#endif
#ifdef _WIN32
#include <winsock.h>
#endif

/* Various magic numbers.
 */

#define MAXFD_DEFAULT			20
#define DEFAULT_PORT			26000
#define HEXEN2_DEFAULT_PORT		26900
#define QUAKE2_DEFAULT_PORT		27910
#define Q2_MASTER_DEFAULT_PORT		27900
#define QW_DEFAULT_PORT			27500
#define QW_MASTER_DEFAULT_PORT		27000
#define HW_DEFAULT_PORT			26950
#define UNREAL_DEFAULT_PORT		7778
#define UNREAL_MASTER_DEFAULT_PORT	28900
#define HALFLIFE_DEFAULT_PORT		27015
#define SIN_DEFAULT_PORT		22450

#define DEFAULT_RETRIES			3
#define DEFAULT_RETRY_INTERVAL		500	/* milli-seconds */

#define SORT_GAME		1
#define SORT_PING		2

/* Structures for keeping information about Quake servers, server
 * rules, and players.
 */

struct player;

#define Q_UNKNOWN_TYPE 0
#define MASTER_SERVER 0x40000000

#define Q_SERVER 1
#define QW_SERVER 2
#define QW_MASTER (4|MASTER_SERVER)
#define H2_SERVER 8
#define Q2_SERVER 16
#define Q2_MASTER (32|MASTER_SERVER)
#define HW_SERVER 64
#define UN_SERVER 128
#define UN_MASTER (256|MASTER_SERVER)
#define HL_SERVER 512
#define SIN_SERVER 1024

#define FLAG_BROADCAST	0x1

struct qserver {
    char *arg;
    char *host_name;
    unsigned long ipaddr;
    int flags;
    int type;
    int fd;
    char *outfilename;
    unsigned short port;
    int retry1;
    int retry2;
    int n_retries;
    struct timeval packet_time1;
    struct timeval packet_time2;
    int ping_total;		/* average is ping_total / n_requests */
    int n_requests;
    int n_packets;

    int n_servers;
    int master_pkt_len;
    char *master_pkt;
    char *error;

    char *server_name;
    char *address;
    char *map_name;
    char *game;
    int max_players;
    int num_players;
    int protocol_version;

    int next_player_info;
    int n_player_info;
    struct player *players;

    char *next_rule;
    int n_rules;
    struct rule *rules;
    int missing_rules;

    struct qserver *next;
};

struct player  {
    int number;
    char *name;
    int frags;
    int team;		/* unreal only */
    int connect_time;
    int shirt_color;
    int pants_color;
    char *address;
    int ping;
    char *skin;
    char *mesh;		/* unreal only */
    struct player *next;
};

struct rule  {
    char *name;
    char *value;
    struct rule *next;
};

extern char *qstat_version;
extern char *DOWN;
extern char *SYSERROR;
extern char *TIMEOUT;
extern char *MASTER;
extern char *SERVERERROR;
extern char *HOSTNOTFOUND;

extern int first_sort_key;
extern int second_sort_key;

/* Definitions for the original Quake network protocol.
 */

#define PACKET_LEN 0xffff

/* Quake packet formats and magic numbers
 */
struct qheader  {
    unsigned char flag1;
    unsigned char flag2;
    unsigned short length;
    unsigned char op_code;
};

struct qpacket  {
    unsigned char flag1;
    unsigned char flag2;
    unsigned short length;
    unsigned char op_code;
    unsigned char data[1500];
};

#define Q_FLAG1			0x80
#define Q_FLAG2			0x00
#define Q_NET_PROTOCOL_VERSION	3
#define HEXEN2_NET_PROTOCOL_VERSION	4
#define Q_HEADER_SIZE		5

#define Q_CCREQ_CONNECT		0x01
#define Q_CCREP_ACCEPT		0x81
#define Q_CCREP_REJECT		0x82

#define Q_CCREQ_SERVER_INFO	0x02
#define Q_CCREP_SERVER_INFO	0x83

#define Q_CCREQ_PLAYER_INFO	0x03
#define Q_CCREP_PLAYER_INFO	0x84

#define Q_CCREQ_RULE_INFO	0x04
#define Q_CCREP_RULE_INFO	0x85

#define Q_DEFAULT_SV_MAXSPEED	"320"
#define Q_DEFAULT_SV_FRICTION	"4"
#define Q_DEFAULT_SV_GRAVITY	"800"
#define Q_DEFAULT_NOEXIT	"0"
#define Q_DEFAULT_TEAMPLAY	"0"
#define Q_DEFAULT_TIMELIMIT	"0"
#define Q_DEFAULT_FRAGLIMIT	"0"


/* Definitions for the QuakeWorld network protocol
 */


#define QW_GET_SERVERS    'c'
#define QW_SERVERS        'd'
#define QW_GET_USERINFO   'o'
#define QW_USERINFO       'p'
#define QW_GET_SEENINFO   'u'
#define QW_SEENINFO       'v'
#define QW_NACK           'm'
#define QW_NEWLINE        '\n'
#define QW_RULE_SEPARATOR '\\'

#define QW_REQUEST_LENGTH 20
#define QW_REQUEST_PREFIX '\377', '\377', '\377', '\377'

/* Output and formatting functions
 */
 
void display_server( struct qserver *server);
void display_qwmaster( struct qserver *server);
void display_server_rules( struct qserver *server);
void display_player_info( struct qserver *server);
void display_q_player_info( struct qserver *server);
void display_qw_player_info( struct qserver *server);
void display_q2_player_info( struct qserver *server);
void display_unreal_player_info( struct qserver *server);
 
void raw_display_server( struct qserver *server);
void raw_display_server_rules( struct qserver *server);
void raw_display_player_info( struct qserver *server);
void raw_display_q_player_info( struct qserver *server);
void raw_display_qw_player_info( struct qserver *server);
void raw_display_q2_player_info( struct qserver *server);
void raw_display_unreal_player_info( struct qserver *server);
 
int is_default_rule( struct rule *rule);
char *escape(unsigned char*);
char *quake_color( int color);
char *play_time( int seconds, int show_seconds);
char *ping_time( int ms);
char *get_qw_game( struct qserver *server);


/* Query status and packet handling functions
 */

void cleanup_qserver( struct qserver *server, int force);
 
void deal_with_packet( struct qserver *server, char *pkt, int pktlen);
void deal_with_q_packet( struct qserver *server, char *pkt, int pktlen);
void deal_with_qw_packet( struct qserver *server, char *pkt, int pktlen);
void deal_with_q1qw_packet( struct qserver *server, char *pkt, int pktlen);
void deal_with_q2_packet( struct qserver *server, char *pkt, int pktlen);
void deal_with_qwmaster_packet( struct qserver *server, char *pkt, int pktlen);
int server_info_packet( struct qserver *server, struct qpacket *pkt,
        int datalen);
int player_info_packet( struct qserver *server, struct qpacket *pkt,
        int datalen);
int rule_info_packet( struct qserver *server, struct qpacket *pkt, int datalen);
 
int time_delta( struct timeval *later, struct timeval *past);
char * strherror( int h_err);
int connection_refused();
 
void add_file( char *filename);
int add_qserver( char *arg, int type, char *outfilename);
struct qserver* add_qserver_byaddr( unsigned long ipaddr, unsigned short port,
	int type);
void init_qserver( struct qserver *server);
int bind_qserver( struct qserver *server);
void bind_sockets();
void send_packets();
int send_server_request_packet( struct qserver *server);
int send_qserver_request_packet( struct qserver *server);
int send_unreal_request_packet( struct qserver *server);
int send_qwmaster_request_packet( struct qserver *server);
int send_player_request_packet( struct qserver *server);
int send_rule_request_packet( struct qserver *server);
 
int set_fds( fd_set *fds);
void get_next_timeout( struct timeval *timeout);

void print_packet( char *buf, int buflen);

void qstatsort(struct qserver **array, int size);


/*
 * Output template stuff
 */

int read_qserver_template( char *filename);
int read_header_template( char *filename);
int read_trailer_template( char *filename);
int read_player_template( char *filename);
int have_server_template();
int have_header_template();
int have_trailer_template();

void template_display_server( struct qserver *server);
void template_display_header();
void template_display_trailer();
void template_display_players( struct qserver *server);
void template_display_player( struct qserver *server, struct player *player);



/*
 * Host cache stuff
 */

int hcache_open( char *filename, int update);
void hcache_write( char *filename);
void hcache_invalidate();
void hcache_validate();
unsigned long hcache_lookup_hostname( char *hostname);
char * hcache_lookup_ipaddr( unsigned long ipaddr);
void hcache_write_file( char *filename);
void hcache_update_file();

