/*
 * hldsfw.c
 * Barry Alcock <alcockba@educk.globalnet.co.uk>
 * 18/Jul/2000
 *
 * v2.00 - 18/Jul/2000
 *     Listens for a request and sends the replies based on the config file (done)
 *     Better config file format (done)
 *     Builds the reply packets itself (done)
 *     Keeps stats on requests serviced (done)
 *     Puts messages in the syslog (done)
 *     Execs hlds_run itself (todo)
 *     Re-execs crashed hlds_run instances (todo)
 *     Hopefully fixes a non-responsiveness bug when under heavy load (done, maybe)
 * 
 * v1.00 - 12/Oct/1999
 *     Did basic packet forwarding between clients and servers.
 * 
 * This software is distributed under the GPL V2 and comes with no warranty what
 * so ever.
 */

#include <stdio.h>
#include <time.h>
#include <signal.h>
#include <syslog.h>
#include <unistd.h>

#include "hldsfw.h"

/* command line modifiable */
char *g_config_file = CONFIG_FILE;
int g_debug = 0;

/* config modifiable */
int g_syslog;

/* internal */
struct list g_servers = { NULL, NULL };
int g_config_line;
int g_requests = 0;
int g_replies = 0;
time_t g_started;

int main (int argc, char **argv)
{
	// put global vars into a known state
	init_globals ();

	// set up environment
	if (parse_cmdline (argc, argv)) return -1;
	if (parse_config ()) return -1;

	if (!g_debug) {
		// detach process from terminal unless debugging
		if (fork ()) exit (0);
		setsid ();
	}

	// install signal handlers
	signal (SIGHUP, sig_handler);
	signal (SIGTERM, sig_handler);
	signal (SIGSEGV, sig_handler);

	// start handling requests
	handle_requests ();

	return 0;
}

void init_globals (void)
{
	if (g_debug) fprintf (stderr, "init_globals: entered\n");

	g_syslog = 1;

	while (g_servers.next) remove_server ( (struct hlds_data *)g_servers.next);
}

void sig_handler (int s)
{
	switch (s) {
	case SIGHUP:
		// reload the config
		if (g_debug) fprintf (stderr, "sig_handler: SIGHUP caught\n");
		init_globals ();
		parse_config ();
		signal (SIGHUP, sig_handler);
		break;

	case SIGTERM:
		// shut down nicely
		if (g_debug) fprintf (stderr, "sig_handler: SIGTERM caught\n");
		signal (SIGCHLD, SIG_IGN);
		if (g_syslog) {
			unsigned long int d, h, m;
			time_t u = time (NULL) - g_started;

			d = u / 86400;
			u %= 86400;
			h = u / 3600;
			u %= 3600;
			m = u / 60;
			u %= 60;

			syslog (LOG_INFO, "shutting down");
			syslog (LOG_INFO, "total uptime = %d day%s, %d hour%s, %d minute%s, %d second%s",
				d, (d == 1) ? "" : "s",
				h, (h == 1) ? "" : "s",
				m, (m == 1) ? "" : "s",
				u, (u == 1) ? "" : "s");
			syslog (LOG_INFO, "requests accepted = %d", g_requests);
			syslog (LOG_INFO, "replies sent = %d", g_replies);
		}
		init_globals ();       // be nice and free memory
		exit (0);
		break;

	case SIGSEGV:
		if (g_debug) fprintf (stderr, "sig_handler: SIGSEGV caught\n");
		if (g_syslog) syslog (LOG_ERR, "segmentation violation - shutting down");
		exit (-1);
		break;
	}
}

