/*
 * cfgcode.c
 * Barry Alcock <alcockba@educk.globalnet.co.uk>
 * 18/Jul/2000
 *
 * This software is distributed under the GPL V2 and comes with no warranty what
 * so ever.
 */

#include <stdio.h>
#include <stdlib.h>
#include <syslog.h>
#include <arpa/inet.h>

#include "hldsfw.h"

int parse_cmdline (int argc, char **argv)
{
	int i, pb;
	char o;

	for (i=1; i<argc; i++) {
		if (argv[i][0] == '-') {
			o = argv[i][1];
			pb = 2;
			if (argv[i][2] == 0) {
				i++;
				pb = 0;
			}

			switch (o) {
			case 'c':
				g_config_file = &argv[i][pb];
				break;

			case 'd':
				g_debug = 1;
				i--;
				break;

			case 'h':
				printf ("hldsfw " VERSION "\n"
					"Barry Alcock <alcockba@educk.globalnet.co.uk>\n\n"
					"Options\n========================================================\n"
					"   -c config_file        Specify a different config file\n"
					"   -d                    Use debug mode\n"
					"   -h                    Display this help\n"
					"\n\n");
				exit (0);
			
			default:
				fprintf (stderr, "Unknown option: %c\n", o);
				return -1;
			}
		} else {
			fprintf (stderr, "Bad command line parameter: %s\n", argv[i]);
			return -1;
		}
	}

	if (g_debug) {
		printf ("g_debug = 1\ng_config_file = %s\n", g_config_file);
	}

	return 0;
}

int parse_config (void)
{
	int t, s = 0;
	char *ident = NULL;
	FILE *cfg;

	if (g_debug) fprintf (stderr, "parse_config: entered\n");

	if (! (cfg = freopen (g_config_file, "r", stdin))) {
		fprintf (stderr, "Error opening config file %s\n", g_config_file);
		return -1;
	}
	
	g_config_line = 1;

	while ( (t = yylex ()) > 0) {
		switch (s) {
		case 0:
			// Blank pad, nothings happened yet
			if (t != TOK_IDENT) {
				fprintf (stderr, "Error in config file %s line %d: Identifier expected\n",
					 g_config_file, g_config_line);
				return -1;
			}
			ident = yylval.string;
			s = 1;
			break;

		case 1:
			// We have an identifier, variable or function?
			if (t == '=') {
				s = 2;
				break;
			} else if (t == '(') {
				// function opened
				if (handle_function (ident)) return -1;
				s = 5;
				break;
			} else {
				fprintf (stderr, "Error in config file %s line %d: Unexpected token\n",
					 g_config_file, g_config_line);
				return -1;
			}
			break;

		case 2:
			// We have a variable identifier, numeric or text?
			if (t == TOK_NUM) {
				if (assign_numeric (ident, yylval.number)) {
					fprintf (stderr, "Error in config file %s line %d: Unknown variable \"%s\"\n",
						 g_config_file, g_config_line, ident);
					return -1;
				}
			} else {
				if (assign_text (ident, yylval.string)) {
					fprintf (stderr, "Error in config file %s line %d: Unknown variable \"%s\"\n",
						 g_config_file, g_config_line, ident);
					return -1;
				}
			}
			s = 4;
			break;

		case 4:
			// statement done, terminate with ';'
			if (t != ';') {
				fprintf (stderr, "Error in config file %s line %d: Missing semicolon\n",
					 g_config_file, g_config_line);
				return -1;
			}
			s = 0;
			break;

		case 5:
			// function done, should close now
			if (t != ')') {
				fprintf (stderr, "Error in config file %s line %d: Missing closing parenthesis\n",
					 g_config_file, g_config_line);
				return -1;
			}
			s = 4;
			break;
		}
	}
	
	fclose (cfg);

	if (g_debug) {
		fprintf (stderr, "g_syslog = %d\n", g_syslog);

		dump_servers ();
	}

	if (g_syslog) {
		openlog ("hldsfw", LOG_PID, LOG_DAEMON);
		syslog (LOG_INFO, "hldsfw v2.00 started");
	}

	if (!g_servers.next) {
		fprintf (stderr, "Error in config file %s line %d: No server() statements found\n",
			 g_config_file, g_config_line);

		if (g_syslog) syslog (LOG_ERR, "shutting down - no servers configured");
		return -1;
	}

	if (g_syslog) syslog (LOG_INFO, "configuration successfully loaded");

	return 0;
}

int cfg_server (void)
{
	struct hlds_svr tmp;
	struct func_param params[] = {
		{ TOK_STRING, &tmp.hs_ident },
		{ TOK_STRING, &tmp.hs_game },
		{ TOK_NUM, &tmp.hs_maxplayers },
		{ TOK_STRING, &tmp.hs_host },
		{ TOK_NUM, &tmp.hs_port },
		{ TOK_STRING, &tmp.hs_startmap },
		{ 0, NULL },
	};

	if (g_debug) fprintf (stderr, "cfg_server: entered\n");

	if (collect_params (params)) return -1;
	if (add_server (&tmp)) return -1;

	return 0;
}

int collect_params (struct func_param *params)
{
	int t, s = 0, c = 0;

	if (g_debug) fprintf (stderr, "collect_params: entered\n");

	while ( (params[s].fp_token) && ( (t = yylex ()) > 0)) {
		if (c) {
			if (t != ',') {
				fprintf (stderr, "Error in config file %s line %d: Missing parameter seperator\n",
					 g_config_file, g_config_line);
				return -1;
			} else {
				c = 0;
				continue;
			}
		}

		if (t != params[s].fp_token) {
			fprintf (stderr, "Error in config file %s line %d: Expected ",
				 g_config_file, g_config_line);
			if (params[s].fp_token == TOK_NUM) {
				fprintf (stderr, "string");
			} else if (params[s].fp_token == TOK_STRING) {
				fprintf (stderr, "number");
			} else {
				fprintf (stderr, "unknown (please report)");
			}
			fprintf (stderr, " for parameter %d\n", s+1);
			return -1;
		} else {
			if (t == TOK_NUM) {
				( (int *)params[s].fp_data)[0] = yylval.number;
			} else if (t == TOK_STRING) {
				int l;
				
				l = strlen (yylval.string) - 2;
				( (char **)params[s].fp_data)[0] = (char *)calloc (1, l + 1);
				memcpy (( (char **)params[s].fp_data)[0], yylval.string + 1, l);
				free (yylval.string);
			}
			s++;
			c = 1;
		}
	}

	return 0;
}

int assign_numeric (char *id, int v)
{
	ASSIGN_NUM_VAR ("use_syslog", g_syslog);

	return -1;
}

int assign_text (char *id, char *v)
{
	return -1;
}

int handle_function (char *id)
{
	HANDLE_CFG_FUNC ("server", cfg_server);

	return -1;
}

int add_server (struct hlds_svr *tmp)
{
	struct hlds_data *n;
	struct in_addr addr;

	if (g_debug) fprintf (stderr, "add_server: entered\n");

	if (dns_lookup (tmp->hs_host, &addr)) {
		fprintf (stderr, "add_server: dns_lookup failed\n");
		return -1;
	}

	if (! (n = (struct hlds_data *)calloc (1, sizeof (struct hlds_data) + 1024))) {
		fprintf (stderr, "add_server: calloc failed\n");
		return -1;
	}

	memcpy (&n->hd_svr, tmp, sizeof (struct hlds_svr));
	n->hd_reply = ( (char *)n) + (sizeof (struct hlds_data));
	n->hd_ipaddr = addr;

        ((unsigned long *)n->hd_reply)[0] = 0xFFFFFFFF;
        n->hd_reply[4] = 'm';

        n->hd_reply_len = sprintf (n->hd_reply + 5,
				   "%s:%d%c%s%c%s%c%s%c%s%c%c%c\x29\x64\x6C%c\x01",
				   inet_ntoa (addr),
				   tmp->hs_port, 0,
				   tmp->hs_game, 0,
				   tmp->hs_startmap, 0,
				   tmp->hs_game, 0,
				   "", 0,
				   0, tmp->hs_maxplayers, 0);

        memset (n->hd_reply + n->hd_reply_len + 5, 0x00, 13);
        n->hd_reply_len += 18;

	if (g_servers.prev) {
		n->hd_list.next = g_servers.next;
		n->hd_list.prev = g_servers.prev;
		g_servers.prev->next = (struct list *)n;
		g_servers.next->prev = (struct list *)n;
		g_servers.prev = (struct list *)n;
	} else {
		n->hd_list.next = (struct list *)n;
		n->hd_list.prev = (struct list *)n;
		g_servers.prev = (struct list *)n;
		g_servers.next = (struct list *)n;
	}

	return 0;
}

void remove_server (struct hlds_data *s)
{
	if ( (!s) || (!g_servers.next)) return;

	if (s->hd_list.next == (struct list *)s) {
		// last one in list
		g_servers.prev = NULL;
		g_servers.next = NULL;
		free_svr_mem (s);
	} else {
		if ( (struct list *)s == g_servers.next) {
			// first in the list
			g_servers.next = g_servers.next->next;
			g_servers.prev->next = g_servers.next;
			g_servers.next->prev = g_servers.prev;
		} else if ( (struct list *)s == g_servers.prev) {
			// last in the list
			g_servers.prev = g_servers.prev->prev;
			g_servers.next->prev = g_servers.prev;
			g_servers.prev->next = g_servers.next;
		} else {
			// somewhere in the middle
			( (struct list *)s)->prev->next = ( (struct list *)s)->next;
			( (struct list *)s)->next->prev = ( (struct list *)s)->prev;
		}
		free_svr_mem (s);
	}
}

void free_svr_mem (struct hlds_data *s)
{
	free (s->hd_svr.hs_ident);
	free (s->hd_svr.hs_game);
	free (s->hd_svr.hs_host);
	free (s->hd_svr.hs_startmap);
	free (s);
}

void dump_servers (void)
{
	struct hlds_data *c;
	int i = 1;
	int j;

	if (!g_servers.next) {
		printf ("No servers configured\n");
		return;
	}

	c = (struct hlds_data *)g_servers.next;
	do {
		printf ("Server %d\n==================================================\n", i++);
		printf ("hd_svr.hs_ident = %s\n", c->hd_svr.hs_ident);
		printf ("hd_svr.hs_game = %s\n", c->hd_svr.hs_game);
		printf ("hd_svr.hs_maxplayers = %d\n", c->hd_svr.hs_maxplayers);
		printf ("hd_svr.hs_host = %s\n", c->hd_svr.hs_host);
		printf ("hd_svr.hs_port = %d\n", c->hd_svr.hs_port);
		printf ("hd_svr.hs_startmap = %s\n", c->hd_svr.hs_startmap);

		if (g_debug) {
			printf ("hd_reply_len = %d\n", c->hd_reply_len);
			printf ("hd_reply = \"");
			for (j=0; j<c->hd_reply_len; j++) {
				printf ("%c", c->hd_reply[j]);
			}
			printf ("\"\n");
		}

		printf ("\n\n");

		c = (struct hlds_data *)c->hd_list.next;
	} while (c != (struct hlds_data *)g_servers.next);
}
