/*==========================================================================
//  x_mad.c -- by Patrick Martin                Last updated:  2-23-1999
//--------------------------------------------------------------------------
//  This file contains code that dictates the properties of the
//  various rockets in the MAD patch.
//========================================================================*/

#include "g_local.h"
#include "x_ctf.h"
#include "x_fire.h"
#include "x_hook.h"
#include "x_mad.h"


/*=========================/  Generic MAD Code  /=========================*/

/*-------------------------------------------------------- Mirror Code -----
//  This is a copy of 'check_dodge' found in 'g_weapons.c'.
//  Because that function is static, and I do not want to
//  modify that file, a copy is placed here.
//------------------------------------------------------------------------*/
void check_rocketdodge (edict_t *self, vec3_t start, vec3_t dir, int speed)
{
	vec3_t	end;
	vec3_t	v;
	trace_t	tr;
	float	eta;

	/* easy mode only ducks one quarter the time */
	if (skill->value == 0)
	{
		if (random() > 0.25)
			return;
	}
	VectorMA (start, 8192, dir, end);
	tr = gi.trace (start, NULL, NULL, end, self, MASK_SHOT);
	if ((tr.ent) && (tr.ent->svflags & SVF_MONSTER) && (tr.ent->health > 0) && (tr.ent->monsterinfo.dodge) && infront(tr.ent, self))
	{
		VectorSubtract (tr.endpos, start, v);
		eta = (VectorLength(v) - tr.ent->maxs[0]) / speed;
		tr.ent->monsterinfo.dodge (tr.ent, self, eta);
	}
}

/*-------------------------------------------------------- New Code --------
//  This purges all the ailments a player has.
//------------------------------------------------------------------------*/
void Coven_CureAll (edict_t *ent)
{
/* Purge the germs. */
        ent->infect_time = 0;
        ent->infect_end = 0;
        ent->infector = NULL;

/* Neutralize the drug. */
        ent->buzz_time = 0;
        ent->buzz_power = 0;
        VectorClear (ent->buzz_color);
        VectorClear (ent->buzz_angle);
        if (ent->client)
        {
                if (deathmatch->value && ((int)dmflags->value & DF_FIXED_FOV))
                        ent->client->ps.fov = 90;
                else
                {       ent->client->ps.fov = atoi(Info_ValueForKey(ent->client->pers.userinfo, "fov"));
                        if (ent->client->ps.fov < 1)
                                ent->client->ps.fov = 90;
                        else if (ent->client->ps.fov > 160)
                                ent->client->ps.fov = 160;
                }
        }

/* Put out the fire. */
	ent->burnout = 0;
}

/*-------------------------------------------------------- New Code --------
//  This gets rid of leftover flashes and blackouts from stray nukes
//  and black holes.
//------------------------------------------------------------------------*/
void Coven_FixVision (edict_t *ent)
{
        ent->nuke_alpha  = 0;
        ent->nuke_fade   = 0;
        ent->black_alpha = 0;
        ent->black_fade  = 0;
}

/*-------------------------------------------------------- New Code --------
//  This eradicates fire and sickness from EVERYONE on the level.
//  This should always be called whenever a client changes levels.
//
//  NOTE:  This causes the flames to simply disappear.
//------------------------------------------------------------------------*/
void Coven_CureLevel (void)
{
        edict_t *ent;

        for (ent = g_edicts; ent < &g_edicts[globals.num_edicts]; ent++)
        {
                if (!ent->inuse)
			continue;

                if (!strcmp(ent->classname, "grapple"))
                {       Coven_KillGrapple (ent);
                        continue;
                }

                Coven_CureAll (ent);
                Coven_FixVision (ent);

                ent->burnout = 0;
                if (ent->burner)
                {       ent->burner->enemy = NULL;
                        G_FreeEdict (ent->burner);
                        ent->burner = NULL;
                }

                ent->stormlink = NULL;  // In case firestorms are in...
        }
}


/*==========================/  Blast Rockets  /==========================*/
/*
   DESCRIPTION:  These are little more than ordinary rockets.
   In addition to normal rocket effects, the explosions caused by
   blast rockets send nearby entities away from the blast at super
   high velocities.
*/
/*========================================================================*/

/*-------------------------------------------------------- Mirror Code -----
//  This tosses a creature through the air against its will.
//------------------------------------------------------------------------*/
void Coven_ThrowThing (edict_t *ent, vec3_t dir, int knockback)
{
        float   mass;

/* Don't throw parts of the map (such as shootable buttons/walls) away. */
        if (ent->solid == SOLID_BSP)
                return;

/* Don't throw crucified soldiers or turret drivers. */
        if (!deathmatch->value)
        {
                if (!strcmp(ent->classname, "misc_insane"))
                        if (ent->spawnflags & 8)
                                if (ent->health > ent->gib_health)
                                        return;

                if (!strcmp(ent->classname, "turret_driver"))
                        return;
        }

/* Insure some mimimum mass. */
	if (ent->mass < 50)
		mass = 50;
	else
		mass = ent->mass;

/* Calculate entity's new velocity. */
	VectorNormalize (dir);
	VectorScale (dir, (knockback / mass), ent->velocity);

/* If entity is on the ground, lift it off the ground. */
	if (ent->groundentity)
	{       if (ent->velocity[2] < 200)
			ent->velocity[2] = 200;
		ent->groundentity = NULL;
	}
}

/*-------------------------------------------------------- New Code --------
//  This is a 'T_RadiusDamage' plus options for burning and
//  controllable knockback.
//------------------------------------------------------------------------*/
void Coven_BlastRadius
(edict_t *inflictor, edict_t *attacker, float damage, edict_t *ignore, float radius, qboolean burn, int knockback, int mod)
{
	float	points;
	edict_t	*ent = NULL;
	vec3_t	v;
	vec3_t	dir;

	while ((ent = findradius(ent, inflictor->s.origin, radius)) != NULL)
	{
		if (ent == ignore)
			continue;
		if (!ent->takedamage)
			continue;

		VectorAdd (ent->mins, ent->maxs, v);
		VectorMA (ent->s.origin, 0.5, v, v);
		VectorSubtract (inflictor->s.origin, v, v);
		points = damage - 0.5 * VectorLength (v);
                if (ent == attacker)
                        points = points * 0.5;

                if (CanDamage (ent, inflictor))
		{
                        if (burn)
                                Coven_Ignite(ent, attacker, inflictor->s.origin);

                        VectorSubtract (ent->s.origin, inflictor->s.origin, dir);
                        if (points > 0)
                                T_Damage (ent, inflictor, attacker, dir, inflictor->s.origin, vec3_origin, (int)points, (int)points, DAMAGE_RADIUS, mod);

                        if (knockback)
                                Coven_ThrowThing (ent, dir, knockback);
		}
	}
}

/*-------------------------------------------------------- New Code --------
//  Do this when a blast rocket touches a target.
//------------------------------------------------------------------------*/
void Coven_BlastTouch (edict_t *ent, edict_t *other, cplane_t *plane, csurface_t *surf)
{
	vec3_t		origin;

	if (other == ent->owner)
		return;

	if (surf && (surf->flags & SURF_SKY))
	{
		G_FreeEdict (ent);
		return;
	}

	if (ent->owner->client)
		PlayerNoise(ent->owner, ent->s.origin, PNOISE_IMPACT);

	// calculate position for the explosion entity
	VectorMA (ent->s.origin, -0.02, ent->velocity, origin);

	if (other->takedamage)
	{
		vec3_t	dir;

		VectorSubtract (other->s.origin, ent->s.origin, dir);
                T_Damage (other, ent, ent->owner, ent->velocity, ent->s.origin, plane->normal, ent->dmg, 0, 0, MOD_BLAST);
		Coven_ThrowThing (other, dir, ent->count);
	}
	else
	{
		// don't throw any debris in net games
		if (!deathmatch->value)
		{
                        if ((surf) && !(surf->flags & (SURF_WARP|SURF_TRANS33|SURF_TRANS66|SURF_FLOWING)))
			{
                                int     n;

				n = rand() % 5;
				while(n--)
					ThrowDebris (ent, "models/objects/debris2/tris.md2", 2, ent->s.origin);
			}
		}
	}

        Coven_BlastRadius(ent, ent->owner, ent->radius_dmg, other, ent->dmg_radius, false, ent->count, MOD_BLAST);

	gi.WriteByte (svc_temp_entity);
	if (ent->waterlevel)
		gi.WriteByte (TE_ROCKET_EXPLOSION_WATER);
	else
		gi.WriteByte (TE_ROCKET_EXPLOSION);
	gi.WritePosition (origin);
	gi.multicast (ent->s.origin, MULTICAST_PVS);

	G_FreeEdict (ent);
}

/*-------------------------------------------------------- New Code --------
//  Entity launches a blast rocket.
//------------------------------------------------------------------------*/
void Coven_FireBlast (edict_t *self, vec3_t start, vec3_t dir, int damage, int speed, float damage_radius, int radius_damage, int knockback)
{
	edict_t	*rocket;

	rocket = G_Spawn();
	VectorCopy (start, rocket->s.origin);
	VectorCopy (dir, rocket->movedir);
	vectoangles (dir, rocket->s.angles);
	VectorScale (dir, speed, rocket->velocity);
        rocket->movetype        = MOVETYPE_FLYMISSILE;
        rocket->clipmask        = MASK_SHOT;
        rocket->solid           = SOLID_BBOX;
        if (damage < 60)
                rocket->s.effects       |= EF_GRENADE;
        else
                rocket->s.effects       |= EF_ROCKET;
	VectorClear (rocket->mins);
	VectorClear (rocket->maxs);
        rocket->s.modelindex    = gi.modelindex ("models/objects/rocket/tris.md2");
        rocket->owner           = self;
        rocket->touch           = Coven_BlastTouch;
        rocket->nextthink       = level.time + 8000/speed;
        rocket->think           = G_FreeEdict;
        rocket->count           = knockback;
        rocket->dmg             = damage;
        rocket->radius_dmg      = radius_damage;
        rocket->dmg_radius      = damage_radius;
        rocket->s.sound         = gi.soundindex ("weapons/rockfly.wav");

	if (self->client)
		check_rocketdodge (self, rocket->s.origin, dir, speed);

	gi.linkentity (rocket);
}


/*===========================/  Gas Rockets  /===========================*/
/*
   DESCRIPTION:  Upon impact, the rocket explodes normally, then
   produces a cloud of gas to hinder those that stumble into it.
   Gas effects cannot be blocked by armor.  Even walls cannot
   completely block it.  However, an invulnerability or bio-suit will
   stop the gas.  A breather may offer partial resistance to gas.

   MAD features three types of gas:  poison, germ, and mind.
   - POISON gas merely damages the victim.
   - GERM gas infects the victim.  While infected, victim
      loses 1 hp per second.  Health or invulnerability will
      cure the victim.
   - MIND gas confuses its victims for up to 30 seconds.
      During this time, the victim's vision is colored and
      tilted.  Strong health or invulnerability will cure
      the victim.

   Model used:  gas.md2 (model by Andrew Eglington)

   Frames:
      0 -  3  Expanding cloud (used for newly spawned gas)
      4 - 23  Full-sized cloud (used for fully formed clouds)
     24 - 27  Shrinking cloud (used for dying clouds)
     28 - 47  Small undulating cloud (NOT used)
          48  Base frame (NOT used)
          49  Skin frame (NOT used)
 
   Skins:
      0  Green (poison)
      1  Purple (germ)
      2  Blue (mind)
*/
/*========================================================================*/

extern qboolean CheckTeamDamage (edict_t *targ, edict_t *attacker);

/*-------------------------------------------------------- New Code --------
//  Returns TRUE if the entity is vulnerable to gas.  FALSE if not.
//------------------------------------------------------------------------*/
qboolean Coven_CanGas (edict_t *ent)
{
/* An entity that is immune to damage is immune to gas as well. */
	if (!ent->takedamage)  return false;

/* OPTIONAL:  Only players and monsters are vulnerable. */
	if ((!ent->client) && (!(ent->svflags & SVF_MONSTER)))
		return false;

/* OPTIONAL:  Dead entities are not affected by gas. */
	if (ent->health <= 0)
		return false;

/* Check if creature is immune to gas. */
	if (ent->client)
	{
	/* Invulnerability protects entity from gas completely. */
		if (ent->client->invincible_framenum > level.framenum)
			return false;

	/* Ditto for the Bio-suit. */
		if (ent->client->enviro_framenum > level.framenum)
			return false;
	}
        else if (!deathmatch->value)
	{
	/* The following monster types are immune to gas. */
		if ( (!strcmp(ent->classname, "monster_tank")) ||
                     (!strcmp(ent->classname, "monster_tank_commander")) ||
                     (!strcmp(ent->classname, "monster_supertank")) ||
                     (!strcmp(ent->classname, "monster_boss2")) ||
                     (!strcmp(ent->classname, "monster_makron")) ||
                     (!strcmp(ent->classname, "monster_jorg")) )
			return false;
	}

	return true;
}

/*-------------------------------------------------------- New Code --------
//  This makes the gas fade away and disappear.
//------------------------------------------------------------------------*/
void Coven_GasFade (edict_t *self)
{
        if (++self->s.frame > 27)
        {       G_FreeEdict (self);
                return;
        }

	self->nextthink = level.time + FRAMETIME;
}

/*-------------------------------------------------------- New Code --------
//  This makes the gas fade away and disappear.
//------------------------------------------------------------------------*/
void Coven_KillCloud (edict_t *self)
{
        self->s.frame   = 24;
        self->classname = "fading_gas";
        self->think     = Coven_GasFade;
        self->nextthink = level.time + FRAMETIME;
}

/*-------------------------------------------------------- New Code --------
//  This is where gas inflicts damage.  Armor does not stop gas.
//  Even walls do not stop the gas fully.  However, an invulnerability
//  or bio-suit will stop the gas.  Breathers offer some protextion.
//------------------------------------------------------------------------*/
void Coven_AcidDamage (edict_t *self)
{
	float   points;
	edict_t *ent = NULL;
	int gasflags = DAMAGE_RADIUS | DAMAGE_NO_ARMOR | DAMAGE_NO_KNOCKBACK;

/* Disappear when duration elapses. */
	if (self->timestamp < level.time)
        {       Coven_KillCloud (self);
		return;
	}

/* Animate cloud.  (FIX) */
        if (++self->s.frame > 23)
                self->s.frame = 4;

/* Check for victims. */
	while ((ent = findradius(ent, self->s.origin, self->dmg_radius)) != NULL)
	{
                /* Can't gas teammates. */
                if (CheckTeamDamage (ent, self->owner))
                        continue;

		if (Coven_CanGas(ent))
		{
			points = self->dmg;

			/* Check if attacker is pumped up with quad damage. */
			if (self->owner->client)
				if (self->owner->client->quad_framenum > level.framenum)
					points *= 4;

			/* Walls and other barriers stop up to 75% of cloud effects. */
			if (!(CanDamage (ent, self)))
				points *= 0.25;

			/* Check for gas resistance. */
			if (ent->client)
			{	/* Breather offers some protection. */
				if (ent->client->breather_framenum > level.framenum)
					points *= 0.5;
			}

			/* Attacker takes half damage from the cloud. */
                        if (ent == self->owner)
				points *= 0.5;

			/* Cloud will inflict at least one point of damage. */
			if (points < 1)
				points = 1;

                        T_Damage (ent, self, self->owner, vec3_origin, ent->s.origin, vec3_origin, (int)points, 0, gasflags, MOD_GAS_POISON);
		}
	}
	self->nextthink = level.time + FRAMETIME;
}

/*-------------------------------------------------------- New Code --------
//  This is where germ cloud infects entities.  If entity is not
//  fully protected, infection is guaranteed.  Germ damage is
//  handled in 'p_view.c' and 'g_monster.c'.
//------------------------------------------------------------------------*/
void Coven_GermDamage (edict_t *self)
{
	edict_t *ent = NULL;
	int gasflags = DAMAGE_RADIUS | DAMAGE_NO_ARMOR | DAMAGE_NO_KNOCKBACK;

/* Disappear when duration elapses. */
	if (self->timestamp < level.time)
        {       Coven_KillCloud (self);
		return;
	}

/* Animate cloud. */
        if (++self->s.frame > 23)
                self->s.frame = 4;

/* Check for victims. */
	while ((ent = findradius(ent, self->s.origin, self->dmg_radius)) != NULL)
	{
                /* Regeneration tech protects player completely. */
                if (ent->client && CTFHasRegeneration(ent))
                        continue;

                /* Can't infect teammates. */
                if (CheckTeamDamage (ent, self->owner))
                        continue;

                if (Coven_CanGas(ent))
		{
			if (!ent->infect_time)
                        {
                                int     damage = rand() % 11 + 10;

                                if (CanDamage (ent, self))
                                        T_Damage (ent, self, self->owner, vec3_origin, ent->s.origin, vec3_origin, damage, 0, gasflags, MOD_GAS_GERM);
				ent->infect_time = level.time + 1;
                        }

			ent->infector = self->owner;

                /* If victim is a monster, germs last for 2 minutes only. */
                        if (ent->svflags & SVF_MONSTER)
                                ent->infect_end = ent->infect_time + 120;
		}
	}
	self->nextthink = level.time + FRAMETIME;
}

/*-------------------------------------------------------- New Code --------
//  This is where mind gas affects entities.  If entity is not
//  fully protected, a dose is guaranteed.  Confusion effects are
//  handled in 'p_view.c' and 'g_monster.c'.
//------------------------------------------------------------------------*/
void Coven_AlterMind (edict_t *self)
{
	edict_t *ent = NULL;

/* Disappear when duration elapses. */
	if (self->timestamp < level.time)
        {       Coven_KillCloud (self);
		return;
	}

/* Animate cloud. */
        if (++self->s.frame > 23)
                self->s.frame = 4;

/* Check for victims. */
	while ((ent = findradius(ent, self->s.origin, self->dmg_radius)) != NULL)
	{
                /* Can't confuse teammates. */
                if (CheckTeamDamage (ent, self->owner))
                        continue;

		if (Coven_CanGas(ent))
		{
                        float   dose = random() * 5 + 10;

			/* Walls and other barriers stop up to 75% of cloud effects. */
                        if (!(CanDamage (ent, self)))
                                dose *= 0.25;

			/* Check for gas resistance. */
			if (ent->client)
                        {       /* Breather reduces effects of gas. */
				if (ent->client->breather_framenum > level.framenum)
                                        dose *= 0.5;

                                /* Resistance tech also reduces the dose. */
                                if (ent->client->pers.inventory[ITEM_INDEX(FindItem("Disruptor Shield"))])
                                        dose *= 0.5;
			}

                        /* Insure minimum dose. */
                        if (dose < 1)
                                dose = 1;

			if (!ent->buzz_time)
				ent->buzz_time = level.time + FRAMETIME;

			if ((ent->buzz_time + dose) > ent->buzz_end)
				ent->buzz_end = ent->buzz_time + dose;
		}
	}
	self->nextthink = level.time + FRAMETIME;
}

/*-------------------------------------------------------- New Code --------
//  Make a cloud.
//------------------------------------------------------------------------*/
void Coven_SpawnGas
(edict_t *self, vec3_t start, vec3_t dir, float damage, int speed, float duration, float damage_radius, int gastype)
{
        edict_t *gas = NULL;
        float   cycles = duration / FRAMETIME;   /* FRAMETIME > 0 */
/*-----/ Kyle's stuff plus small tweaks. /-----/ Persistent gas. /-----*/
        float   absorb_radius = damage_radius * 0.5;
        vec3_t  v;

/* Had to use G_Find because FindRadius ignores SOLID_NOT edicts. */
        while ((gas = G_Find (gas, FOFS(classname), "gas")) != NULL)
        {       VectorSubtract(gas->s.origin, start, v);
                if ((VectorLength(v) < absorb_radius) && (gas->style == gastype))
                {
                        gas->timestamp  += duration + 2;
                        gas->owner      = self;   /* Steal cloud. */

                        /* If poison gas.  Make it stronger. */
                        if (gastype == GAS_POISON)
                                (gas->dmg)++;

                        return;
                }
        }
/*---------------------------------------------------------------------*/

	if (cycles < 1)   cycles = 1;

	gas = G_Spawn();
        gas->s.modelindex = gi.modelindex ("models/mad/gas/tris.md2");
        VectorSet (gas->mins, -96, -96, -96);
        VectorSet (gas->maxs, 96, 96, 96);
	VectorCopy (start, gas->s.origin);
	if (speed)
        {       vectoangles (dir, gas->s.angles);
                VectorScale (dir, speed, gas->velocity);
	}
	else
        {       VectorClear (gas->s.angles);
                VectorClear (gas->velocity);
	}
        gas->movetype     = MOVETYPE_FLYMISSILE;
        gas->clipmask     = MASK_SHOT;
	gas->solid        = SOLID_NOT;
	gas->takedamage   = DAMAGE_NO;
        gas->s.effects    = EF_COLOR_SHELL;
	gas->owner        = self;
	gas->classname    = "gas";
	gas->touch        = NULL;
	gas->timestamp    = level.time + duration;
	gas->nextthink    = level.time + FRAMETIME;
/*-----/ Kyle's stuff. /-----/ Remember gas type. /-----*/
        gas->style        = gastype;
/*------------------------------------------------------*/
        if (gastype == GAS_MIND)
        {
                gas->s.renderfx   = RF_SHELL_BLUE;
		gas->think        = Coven_AlterMind;
                gas->s.frame      = 0;
                gas->s.skinnum    = 2;
        }
        else if (gastype == GAS_BIO)
        {
                gas->s.renderfx   = (RF_SHELL_RED | RF_SHELL_BLUE);
		gas->think        = Coven_GermDamage;
                gas->s.frame      = 0;
                gas->s.skinnum    = 1;
        }
	else
        {
                gas->s.renderfx   = RF_SHELL_GREEN;
		gas->think        = Coven_AcidDamage;
                gas->s.frame      = 0;
                gas->s.skinnum    = 0;
        }
	gas->dmg          = (int)(damage / cycles);
	gas->dmg_radius   = damage_radius;

	gi.linkentity(gas);
}

/*-------------------------------------------------------- New Code --------
//  Do this when a gas bomb touches a target.
//------------------------------------------------------------------------*/
void Coven_GasbombTouch (edict_t *ent, edict_t *other, cplane_t *plane, csurface_t *surf)
{
        vec3_t  origin;
        float   gas_damage, gas_radius, gas_time;

	if (other == ent->owner)
		return;

	if (surf && (surf->flags & SURF_SKY))
	{
		G_FreeEdict (ent);
		return;
	}

	if (ent->owner->client)
		PlayerNoise(ent->owner, ent->s.origin, PNOISE_IMPACT);

	// calculate position for the explosion entity
	VectorMA (ent->s.origin, -0.02, ent->velocity, origin);

	if (other->takedamage)
	{
                T_Damage (other, ent, ent->owner, ent->velocity, ent->s.origin, plane->normal, ent->dmg, 0, 0, MOD_ROCKET);
	}
	else
	{
		// don't throw any debris in net games
		if (!deathmatch->value)
		{
			if ((surf) && !(surf->flags & (SURF_WARP|SURF_TRANS33|SURF_TRANS66|SURF_FLOWING)))
			{
                                int     n;

				n = rand() % 5;
				while(n--)
					ThrowDebris (ent, "models/objects/debris2/tris.md2", 2, ent->s.origin);
			}
		}
	}

        /* Spawn a gas cloud. */
        if (ent->style == GAS_POISON)
        {       gas_damage = 160;
                gas_radius = 128;
                gas_time   = 2;
        }
        else
        {       gas_damage = 0;
                gas_radius = 128;
                gas_time   = 2;
        }

        T_RadiusDamage(ent, ent->owner, ent->radius_dmg, other, ent->dmg_radius, MOD_R_SPLASH);
        Coven_SpawnGas(ent->owner, ent->s.origin, vec3_origin, gas_damage, 0, gas_time, gas_radius, ent->style);

        /* Explosion noise. */
        if (ent->waterlevel)
                gi.sound(ent, CHAN_AUTO, gi.soundindex("weapons/xpld_wat.wav"), 1, ATTN_NORM, 0);
        else
                gi.sound(ent, CHAN_AUTO, gi.soundindex("weapons/grenlx1a.wav"), 1, ATTN_NORM, 0);

	G_FreeEdict (ent);
}

/*-------------------------------------------------------- New Code --------
//  Entity launches a gas bomb of any type.
//------------------------------------------------------------------------*/
void Coven_FireGasbomb (edict_t *self, vec3_t start, vec3_t dir, int damage, int speed, float damage_radius, int radius_damage, int gastype)
{
	edict_t	*rocket;

	rocket = G_Spawn();
	VectorCopy (start, rocket->s.origin);
	VectorCopy (dir, rocket->movedir);
	vectoangles (dir, rocket->s.angles);
	VectorScale (dir, speed, rocket->velocity);
        rocket->movetype        = MOVETYPE_FLYMISSILE;
        rocket->clipmask        = MASK_SHOT;
        rocket->solid           = SOLID_BBOX;
        rocket->s.effects       |= EF_GRENADE;
	VectorClear (rocket->mins);
	VectorClear (rocket->maxs);
        rocket->s.modelindex    = gi.modelindex ("models/objects/rocket/tris.md2");
        rocket->owner           = self;
        rocket->touch           = Coven_GasbombTouch;
        rocket->nextthink       = level.time + 8000/speed;
        rocket->think           = G_FreeEdict;
        rocket->style           = gastype;
        rocket->dmg             = damage;
        rocket->radius_dmg      = radius_damage;
        rocket->dmg_radius      = damage_radius;
        rocket->s.sound         = gi.soundindex ("weapons/rockfly.wav");

	if (self->client)
		check_rocketdodge (self, rocket->s.origin, dir, speed);

	gi.linkentity (rocket);
}


/*===========================/  Napalm Bombs  /===========================*/
/*
   DESCRIPTION:  In addition to standard rocket effects, those
   caught in the explosion will ignite and burn.  Also, a fiery
   cloud will form and rain down fire to burn anyone below.
   Once on fire, a victim will burn until (s)he finds water
   or health.

        NOTE:  Most of the fire code is in 'x_fire*.c'.
*/
/*========================================================================*/

/*-------------------------------------------------------- New Code --------
//  Do this when a napalm bomb touches a target.
//------------------------------------------------------------------------*/
void Coven_NapalmTouch (edict_t *ent, edict_t *other, cplane_t *plane, csurface_t *surf)
{
	vec3_t		origin;
        vec3_t          cloud = {3, 120, 60};
        vec3_t          timer = {0, 0.5, 0};
	vec3_t		damage = {6, 1, 100};
	vec3_t		radius_damage = {4, 1, 50};

	if (other == ent->owner)
		return;

	if (surf && (surf->flags & SURF_SKY))
	{
		G_FreeEdict (ent);
		return;
	}

	if (ent->owner->client)
		PlayerNoise(ent->owner, ent->s.origin, PNOISE_IMPACT);

	// calculate position for the explosion entity
	VectorMA (ent->s.origin, -0.02, ent->velocity, origin);

	if (other->takedamage)
	{
                Coven_Ignite (other, ent->owner, other->s.origin);
                T_Damage (other, ent, ent->owner, ent->velocity, ent->s.origin, plane->normal, ent->dmg, 0, 0, MOD_NAPALM);
	}
	else
	{
		// don't throw any debris in net games
		if (!deathmatch->value)
		{
			if ((surf) && !(surf->flags & (SURF_WARP|SURF_TRANS33|SURF_TRANS66|SURF_FLOWING)))
			{
                                int     n;

				n = rand() % 5;
				while(n--)
					ThrowDebris (ent, "models/objects/debris2/tris.md2", 2, ent->s.origin);
			}
		}
	}

        /* Set entities on fire then make an fiery cloud. */
        Coven_BlastRadius (ent, ent->owner, ent->radius_dmg, other, ent->dmg_radius, true, 0, MOD_NAPALM);
        Coven_FireDrop (ent->owner, origin, damage, radius_damage, FLAME_TIME, 50, MOD_CLOUD_FIRE, MOD_CLOUD_FIRE);
        Coven_FlameCloud (ent->owner, ent->s.origin, cloud, timer, damage, radius_damage, 100, 100);

	gi.WriteByte (svc_temp_entity);
	if (ent->waterlevel)
		gi.WriteByte (TE_ROCKET_EXPLOSION_WATER);
	else
		gi.WriteByte (TE_ROCKET_EXPLOSION);
	gi.WritePosition (origin);
	gi.multicast (ent->s.origin, MULTICAST_PVS);

	G_FreeEdict (ent);
}

/*-------------------------------------------------------- New Code --------
//  Entity launches a napalm rocket.
//------------------------------------------------------------------------*/
void Coven_FireNapalm (edict_t *self, vec3_t start, vec3_t dir, int damage, int speed, float damage_radius, int radius_damage)
{
	edict_t	*rocket;

	rocket = G_Spawn();
	VectorCopy (start, rocket->s.origin);
	VectorCopy (dir, rocket->movedir);
	vectoangles (dir, rocket->s.angles);
	VectorScale (dir, speed, rocket->velocity);
        rocket->movetype        = MOVETYPE_FLYMISSILE;
        rocket->clipmask        = MASK_SHOT;
        rocket->solid           = SOLID_BBOX;
        rocket->s.effects       |= EF_ROCKET;
	VectorClear (rocket->mins);
	VectorClear (rocket->maxs);
        rocket->s.modelindex    = gi.modelindex ("models/objects/rocket/tris.md2");
        rocket->owner           = self;
        rocket->touch           = Coven_NapalmTouch;
        rocket->nextthink       = level.time + 8000/speed;
        rocket->think           = G_FreeEdict;
        rocket->dmg             = damage;
        rocket->radius_dmg      = radius_damage;
        rocket->dmg_radius      = damage_radius;
        rocket->s.sound         = gi.soundindex ("weapons/rockfly.wav");

	if (self->client)
		check_rocketdodge (self, rocket->s.origin, dir, speed);

	gi.linkentity (rocket);
}


/*===========================/  Smart Bombs  /===========================*/
/*
   DESCRIPTION:  Smart bombs are like miniature BFG-shots.
   When launched, smart bombs glow blue and emit a blue exhaust.
   Entities close to the rocket (except its owner), are zapped
   by lasers.  After the rocket explodes, any entities 'seen'
   by the explosion and its owner will suffer more damage.

   Models used:  fire.md2 (see x_fire*.c)
                 smart.md2 (model by Patrick Martin, skin by Jake Bielanski)

   (SMART.MD2)
   Frames:
      0 - 14  Smart bomb explosion
          15  Base frame (NOT used)
 
   Skin:
      0  Blue cloud design
*/
/*========================================================================*/

/*-------------------------------------------------------- New Code --------
//  This animates the flame that scorches those caught by the
//  smart bomb blast.
//------------------------------------------------------------------------*/
void Coven_SmartFlameThink (edict_t *self)
{
        if (++self->s.frame > FRAME_FIRE_flare6)
	{	G_FreeEdict (self);
		return;
	}

	self->nextthink = level.time + FRAMETIME;
}

/*-------------------------------------------------------- New Code --------
//  This creates a colored flame that appears on each
//  smart bomb blast victim.
//------------------------------------------------------------------------*/
void Coven_SmartFlame (vec3_t origin, int color)
{
	edict_t *fire;

	fire = G_Spawn();
	VectorCopy(origin, fire->s.origin);
	VectorClear(fire->s.angles);
	fire->s.angles[1]  = rand() % 360;
	VectorClear(fire->velocity);
        fire->movetype     = MOVETYPE_NONE;
        fire->clipmask     = 0;
        fire->solid        = SOLID_NOT;
        fire->s.effects    = 0;
        fire->s.renderfx   = RF_FULLBRIGHT;
	VectorClear (fire->mins);
	VectorClear (fire->maxs);
        fire->s.modelindex = MD2_FIRE;
        fire->s.frame      = FRAME_FIRE_flare1;
        fire->s.skinnum    = color;
	fire->touch        = NULL;
	fire->nextthink    = level.time + FRAMETIME;
        fire->think        = Coven_SmartFlameThink;
	gi.linkentity (fire);
}

/*-------------------------------------------------------- New Code --------
//  This animates the blue cloud that results from an exploding smart
//  bomb.  This also creates a minor BFG-like blast effect that damages
//  entities within a wide area.
//------------------------------------------------------------------------*/
void Coven_SmartExplode (edict_t *self)
{
        if (++self->s.frame > 14)
        {       G_FreeEdict (self);
                return;
        }

        if (self->s.frame == 2)
	{
                edict_t *ent;
                float   points;
                vec3_t  v;
                float   dist;
                float   min_damage = 5;

                /* The BFG-like effect. */
		ent = NULL;
                while ((ent = findradius(ent, self->s.origin, self->dmg_radius)) != NULL)
		{
			if (!ent->takedamage)
				continue;
			if (ent == self->owner)
				continue;
                        if (CheckTeamDamage (ent, self->owner))
                                continue;
			if (!CanDamage (ent, self))
				continue;
			if (!CanDamage (ent, self->owner))
				continue;

			VectorAdd (ent->mins, ent->maxs, v);
			VectorMA (ent->s.origin, 0.5, v, v);
			VectorSubtract (self->s.origin, v, v);
			dist = VectorLength(v);
                        points = self->radius_dmg * (1.0 - dist/self->dmg_radius);
			if (points < min_damage)
				points = min_damage;

                        if (self->style == CTF_TEAM1)
                                Coven_SmartFlame (ent->s.origin, SKIN_FIRE_red);
                        else
                                Coven_SmartFlame (ent->s.origin, SKIN_FIRE_blue);
                        T_Damage (ent, self, self->owner, self->velocity, ent->s.origin, vec3_origin, (int)points, 0, DAMAGE_ENERGY, MOD_SMART_BLAST);
		}
	}
        else if (self->s.frame == 11)
                self->s.effects &= ~(EF_FLAG1 | EF_FLAG2);

        self->nextthink = level.time + FRAMETIME;
}

/*-------------------------------------------------------- New Code --------
//  Entity (smart bomb) becomes a bluish explosion.  The explosion
//  will damage entities like the BFG.
//------------------------------------------------------------------------*/
void Coven_BecomeSmartExplosion (edict_t *ent)
{
        int     damage = 50;

/* Check if attacker is pumped up with quad damage. */
        if (ent->owner->client)
                if (ent->owner->client->quad_framenum > level.framenum)
                        damage *= 4;

/* Turn entity into bluish explosion. */
	if (ent->waterlevel)
                gi.sound(ent, CHAN_AUTO, gi.soundindex("weapons/xpld_wat.wav"), 1, ATTN_NORM, 0);
        else
                gi.sound(ent, CHAN_AUTO, gi.soundindex("weapons/grenlx1a.wav"), 1, ATTN_NORM, 0);
        ent->s.modelindex  = gi.modelindex("models/mad/smart/tris.md2");
        ent->s.frame       = 0;
        ent->touch         = NULL;
        ent->solid         = SOLID_NOT;
        ent->takedamage    = DAMAGE_NO;
        ent->clipmask      = 0;
        if (ent->style == CTF_TEAM1)
        {       ent->s.effects     |= (EF_PENT | EF_FLAG1);
                ent->s.skinnum     = 1;
                ent->classname     = "red_explosion";
        }
        else
        {       ent->s.effects     |= (EF_QUAD | EF_FLAG2);
                ent->s.skinnum     = 0;
                ent->classname     = "blue_explosion";
        }
        ent->s.renderfx    = RF_FULLBRIGHT;
        ent->movetype      = MOVETYPE_NONE;
        VectorClear(ent->velocity);
        VectorClear(ent->s.angles);
        ent->s.angles[YAW] = rand() % 360;
        ent->s.sound       = 0;
        ent->radius_dmg    = damage;
        ent->dmg_radius    = 1000;
        ent->nextthink     = level.time + FRAMETIME;
        ent->think         = Coven_SmartExplode;
        gi.linkentity (ent);
}

/*-------------------------------------------------------- New Code --------
//  Entity (smart bomb) becomes a bluish explosion.  The explosion
//  will damage entities like the BFG.
//------------------------------------------------------------------------*/
void Coven_KillSmartLaser (edict_t *ent)
{
	edict_t	*now;
	edict_t	*next;

/* Clean up laser entities. */
	for (now = ent->teamchain; now; now = next)
	{
		next = now->teamchain;
		G_FreeEdict (now);
	}
}

/*-------------------------------------------------------- New Code --------
//  This happens when a smart bomb touches a target.  Lasers
//  are cleaned up here.
//------------------------------------------------------------------------*/
void Coven_SmartTouch (edict_t *ent, edict_t *other, cplane_t *plane, csurface_t *surf)
{
        vec3_t  origin;

	if (other == ent->owner)
		return;

        /* Clean up laser entities. */
        Coven_KillSmartLaser (ent);

	if (surf && (surf->flags & SURF_SKY))
	{
		G_FreeEdict (ent);
		return;
	}

	if (ent->owner->client)
		PlayerNoise(ent->owner, ent->s.origin, PNOISE_IMPACT);

	// calculate position for the explosion entity
	VectorMA (ent->s.origin, -0.02, ent->velocity, origin);

	if (other->takedamage)
                T_Damage (other, ent, ent->owner, ent->velocity, ent->s.origin, plane->normal, ent->dmg, 0, 0, MOD_SMART_BOMB);

        T_RadiusDamage (ent, ent->owner, ent->radius_dmg, other, ent->dmg_radius, MOD_SMART_BLAST);

        /* Make green particle explosion. */
	gi.WriteByte (svc_temp_entity);
	gi.WriteByte (TE_BFG_BIGEXPLOSION);
        gi.WritePosition (ent->s.origin);
        gi.multicast (ent->s.origin, MULTICAST_PVS);

        /* Turn into a bluish explosion. */
        Coven_BecomeSmartExplosion (ent);
}

/*-------------------------------------------------------- New Code --------
//  This creates thin colored exhaust smoke.
//------------------------------------------------------------------------*/
void Coven_ColorSmoke (vec3_t origin, vec3_t smokedir, int color)
{
        gi.WriteByte (svc_temp_entity);
        gi.WriteByte (TE_LASER_SPARKS);
        gi.WriteByte (4);
        gi.WritePosition (origin);
        gi.WriteDir (smokedir);
        gi.WriteByte (color);
        gi.multicast (origin, MULTICAST_PVS);
}

/*----------------------------------------------------/ New Code /--------//
//  Laser think.  Laser damages anything in the way.
//
//  s.origin = beam start.
//  s.oldorigin = beam end.
//------------------------------------------------------------------------*/
void Coven_SmartLaserThink (edict_t *self)
{
	edict_t	*ignore;
	vec3_t	start;
	vec3_t	end;
	trace_t	tr;
	vec3_t	point;

	if (self->enemy)
	{
		VectorMA (self->enemy->absmin, 0.5, self->enemy->size, point);
		VectorSubtract (point, self->s.origin, self->movedir);
		VectorNormalize (self->movedir);
	}

	ignore = self;
	VectorCopy (self->s.origin, start);
	VectorMA (start, 2048, self->movedir, end);
	while(1)
	{
		tr = gi.trace (start, NULL, NULL, end, ignore, CONTENTS_SOLID|CONTENTS_MONSTER|CONTENTS_DEADMONSTER);

		if (!tr.ent)
			break;

		// hurt it if we can
		if ((tr.ent->takedamage) && !(tr.ent->flags & FL_IMMUNE_LASER))
                        T_Damage (tr.ent, self, self->activator, self->movedir, tr.endpos, vec3_origin, self->dmg, 1, DAMAGE_ENERGY, MOD_SMART_LASER);

                // if we hit something that's not a monster or player, we're done
		if (!(tr.ent->svflags & SVF_MONSTER) && (!tr.ent->client))
                {
//                        Coven_ColorSmoke (tr.endpos, tr.plane.normal, self->s.skinnum);
			break;
                }

		ignore = tr.ent;
		VectorCopy (tr.endpos, start);
	}

	VectorCopy (tr.endpos, self->s.old_origin);

	self->nextthink = level.time + FRAMETIME;
}

/*----------------------------------------------------/ New Code /--------//
//  Create and turn on smart laser.
//
//  s.frame = beam diameter.
//       4 = normal
//      16 = fat
//
//  s.skinnum = beam color.
//      0xf2f2f0f0 = RED
//      0xd0d1d2d3 = GREEN
//      0xf3f3f1f1 = BLUE
//      0xdcdddedf = YELLOW
//      0xe0e1e2e3 = ORANGE
//------------------------------------------------------------------------*/
void Coven_SpawnSmartLaser (edict_t *self, edict_t *targ, int damage)
{
	edict_t	*beam;

        beam = G_Spawn();
        beam->flags             |= FL_TEAMSLAVE;
        beam->teamchain         = self->teamchain;
        beam->teammaster        = self;
        self->teamchain         = beam;
        beam->owner             = self->owner;
        beam->enemy             = targ;
        beam->classname         = "smart laser";

        beam->movetype          = MOVETYPE_FLYMISSILE;
        beam->solid             = SOLID_NOT;
        beam->s.renderfx        |= RF_BEAM|RF_TRANSLUCENT;
        beam->s.modelindex      = 1;    // must be non-zero
        beam->s.frame           = 4;
        if (self->style == CTF_TEAM1)
                beam->s.skinnum         = 0xf2f2f0f0;
        else
                beam->s.skinnum         = 0xf3f3f1f1;
        beam->dmg               = damage;
        VectorSet (beam->mins, -8, -8, -8);
        VectorSet (beam->maxs, 8, 8, 8);

        if (!beam->enemy)
                G_SetMovedir (beam->s.angles, beam->movedir);

        beam->activator         = self->owner;  // Must have this!
        beam->think             = Coven_SmartLaserThink;
        gi.linkentity (beam);

        Coven_SmartLaserThink (beam);
}

/*-------------------------------------------------------- New Code --------
//  Zap nearby targets.
//------------------------------------------------------------------------*/
void Coven_SmartThink (edict_t *self)
{
	edict_t	*ent;
	edict_t	*next;
	edict_t	*check;
	vec3_t	v;
	float	distance;

        /* Disappear when duration elapses. */
	if (self->timestamp < level.time)
	{
                Coven_KillSmartLaser (self);
        	G_FreeEdict (self);
		return;
	}

        /* Make green smoke trail. */
        Coven_ColorSmoke (self->s.origin, self->movedir, 0xd0d1d2d3);

	ent = self->teamchain;
	self->teamchain = NULL;

	// go through the old list; re-add good ones and free one the have gone out of range
	while (ent)
	{
		next = ent->teamchain;

		VectorSubtract(ent->enemy->s.origin, self->s.origin, v);
		distance = VectorLength(v);
		if (distance <= 256)
		{
			ent->teamchain = self->teamchain;
			self->teamchain = ent;
		}
		else
		{
			G_FreeEdict (ent);
		}

		ent = next;
	}

	while ((ent = findradius(ent, self->s.origin, 256)) != NULL)
	{
		if (ent == self)
			continue;

		if (ent == self->owner)
			continue;

		if (!ent->takedamage)
			continue;

		if (!(ent->svflags & SVF_MONSTER) && (!ent->client) && (strcmp(ent->classname, "misc_explobox") != 0))
			continue;

		if (ctf->value && ent->client && self->owner->client &&
                    ent->client->resp.ctf_team == self->owner->client->resp.ctf_team)
			continue;

		// see if this is already on the list
		for (check = self->teamchain; check; check = check->teamchain)
			if (check->enemy == ent)
				break;
		if (check)
			continue;

		// create a new laser
                Coven_SpawnSmartLaser (self, ent, 5);
	}

	self->nextthink = level.time + FRAMETIME;
}

/*-------------------------------------------------------- New Code --------
//  Launch a smart bomb.
//------------------------------------------------------------------------*/
void Coven_FireSmart (edict_t *self, vec3_t start, vec3_t dir, int damage, int speed, float damage_radius, int radius_damage)
{
	edict_t	*rocket;

	rocket = G_Spawn();
	VectorCopy (start, rocket->s.origin);
	VectorCopy (dir, rocket->movedir);
	vectoangles (dir, rocket->s.angles);
	VectorScale (dir, speed, rocket->velocity);
        rocket->movetype        = MOVETYPE_FLYMISSILE;
        rocket->clipmask        = MASK_SHOT;
        rocket->solid           = SOLID_BBOX;
        if (self->client && (self->client->resp.ctf_team == CTF_TEAM1))
        {       rocket->s.effects       |= EF_FLAG1;    /* Red glow */
                rocket->style           = CTF_TEAM1;
        }
        else
        {       rocket->s.effects       |= EF_FLAG2;    /* Blue glow */
                rocket->style           = CTF_TEAM2;
        }
	VectorClear (rocket->mins);
	VectorClear (rocket->maxs);
        rocket->s.modelindex    = gi.modelindex ("models/objects/rocket/tris.md2");
        rocket->owner           = self;
        rocket->classname       = "smart_bomb";
        rocket->touch           = Coven_SmartTouch;
        rocket->timestamp       = level.time + 8000/speed;
        rocket->nextthink       = level.time + FRAMETIME;
        rocket->think           = Coven_SmartThink;
        rocket->dmg             = damage;
        rocket->radius_dmg      = radius_damage;
        rocket->dmg_radius      = damage_radius;
        rocket->s.sound         = gi.soundindex ("weapons/rockfly.wav");
        rocket->teammaster      = rocket;
        rocket->teamchain       = NULL;

        /* Don't bother dodging the smart bomb because of lasers. */

	gi.linkentity (rocket);
}


/*=========================/  Tactical Nukes  /=========================*/
/*
   DESCRIPTION:  This is like a (much) more powerful version of the
   napalm bomb.  Upon impact, a nuke will flash and vaporize any
   projectile in its LOS (line-of-sight).  Anyone within the explosion
   radius, which is very large, suffers extreme damage and ignites.
   A large cloud of fire will erupt and rain down fire.

   Models used:  fire.md2 (see x_fire*.c)
                 ripple.md2 (model by Patrick Martin)

   Frames:
      0 -  8  Expanding shockwave
           9  Base frame (NOT used)
 
   Skins:
      0  Orange-red (nuke)
      1  Green (black hole bomb)
*/
/*========================================================================*/

/*-------------------------------------------------------- New Code --------
//  This finds and returns the next entity.  Used to find EVERYONE.
//------------------------------------------------------------------------*/
static edict_t *findall (edict_t *from)
{
	if (!from)
		from = g_edicts;
	else
		from++;
	for ( ; from < &g_edicts[globals.num_edicts]; from++)
	{
		if (!from->inuse)
			continue;
                if ((from->solid == SOLID_NOT) &&
                    (strcmp (from->classname, "gas")) &&
                    (strcmp (from->classname, "grapple")))
			continue;
		return from;
	}

	return NULL;
}

/*-------------------------------------------------------- New Code --------
//  This animates the mushroom clouds.  The magic numbers in this
//  function represent the frame numbers in id's r_explode md2.
//------------------------------------------------------------------------*/
void Coven_MushExplodeThink (edict_t *self)
{
	if (++self->s.frame > 29)
	{	G_FreeEdict (self);
		return;
	}

        if (self->s.frame == 27)
	{	self->s.skinnum++;
		self->s.renderfx |= RF_TRANSLUCENT;
	}
	else if (self->s.frame == 25)
		self->s.skinnum++;
	else if (self->s.frame == 24)
		self->s.skinnum++;
	else if (self->s.frame == 23)
		self->s.skinnum++;
	else if (self->s.frame == 21)
		self->s.skinnum++;
        else if (self->s.frame == 17)
		self->s.skinnum++;

	self->nextthink = level.time + FRAMETIME;
}

/*-------------------------------------------------------- New Code --------
//  This creates the flaming mushroom clouds.
//------------------------------------------------------------------------*/
void Coven_MakeMushExplosion (vec3_t origin)
{
	edict_t *fire;

	fire = G_Spawn();
	VectorCopy(origin, fire->s.origin);
	VectorClear(fire->s.angles);
        fire->s.angles[YAW]     = rand() % 360;
	VectorClear(fire->velocity);
        fire->movetype          = MOVETYPE_NONE;
        fire->clipmask          = MASK_SHOT;
        fire->solid             = SOLID_NOT;
        fire->s.effects         = 0;
        fire->s.renderfx        = RF_FULLBRIGHT;
	VectorClear (fire->mins);
	VectorClear (fire->maxs);
        fire->s.modelindex      = gi.modelindex("models/objects/r_explode/tris.md2");
        fire->s.frame           = 15;
        fire->s.skinnum         = 0;
        fire->touch             = NULL;
        fire->nextthink         = level.time + FRAMETIME;
        fire->think             = Coven_MushExplodeThink;
	gi.linkentity (fire);
}

/*-------------------------------------------------------- New Code --------
//  This adds more explosions to create a mushroom cloud appearance.
//------------------------------------------------------------------------*/
void Coven_Mushroom (vec3_t origin)
{
	origin[2] += 40;
        Coven_MakeMushExplosion(origin);
	origin[2] += 40;
        Coven_MakeMushExplosion(origin);
}

/*-------------------------------------------------------- New Code --------
//  This animates the shockwave.
//------------------------------------------------------------------------*/
void Coven_WaveThink (edict_t *self)
{
        if (++self->s.frame > 8)
	{	G_FreeEdict (self);
		return;
	}

        if (self->s.frame == 5)
                self->s.effects  &= ~EF_BFG;

	self->nextthink = level.time + FRAMETIME;
}

/*-------------------------------------------------------- New Code --------
//  This transforms the object into a shockwave.
//------------------------------------------------------------------------*/
void Coven_BecomeWave (edict_t *self, int type)
{
/* Make some steam. */
        self->s.modelindex      = gi.modelindex("models/mad/ripple/tris.md2");
        self->s.frame           = 0;
        self->s.skinnum         = type;
        self->touch             = NULL;
        self->solid             = SOLID_NOT;
        self->clipmask          = 0;
        self->s.effects         = 0;
        self->s.renderfx        = RF_FULLBRIGHT;
        self->movetype          = MOVETYPE_NONE;
        VectorClear(self->velocity);
        VectorClear(self->s.angles);
        self->s.angles[YAW]     = rand() % 360;
        self->nextthink         = level.time + 0.1;
        self->think             = Coven_WaveThink;

        gi.linkentity (self);
}

/*-------------------------------------------------------- New Code --------
//  Nuke vaporizes projectiles and burns other targets.
//------------------------------------------------------------------------*/
void Coven_NukeDamage (edict_t *inflictor, edict_t *attacker, float damage, float radius)
{
	float	points;
	edict_t	*ent = NULL;
	vec3_t	v;
	vec3_t	dir;
        float   dist;
        //float min_damage = damage * 0.05;

        while ((ent = findall(ent)) != NULL)
	{
                if (!CanDamage (ent, inflictor))
                        continue;

        /* If projectile, vaporize it! */
                if ((ent->clipmask == MASK_SHOT) && (ent != inflictor))
                {
                        if (!strcmp (ent->classname, "grapple"))
                        {       Coven_KillGrapple (ent);
                                continue;
                        }

                        if (!strcmp (ent->classname, "smart_bomb"))
                                Coven_KillSmartLaser (ent);

                        G_FreeEdict(ent);
                        continue;
                }

		if (!ent->takedamage)
			continue;

		VectorSubtract(ent->s.origin, inflictor->s.origin, dir);
		dist = VectorLength(dir);

        /* If player, blind him or her. */
		if (ent->client)
		{
			float fade = dist * 0.0002;
			if (fade < 0.04)
				fade = 0.04;
			if ((fade < ent->nuke_fade) || (ent->nuke_fade <= 0))
				ent->nuke_fade = fade;
			ent->nuke_alpha = 1;
		}

        /* If entity is within the blast, burn it. */
		if (dist <= radius)
		{
			VectorAdd (ent->mins, ent->maxs, v);
			VectorMA (ent->s.origin, 0.5, v, v);
			VectorSubtract (inflictor->s.origin, v, v);

                        points = damage * (1.0 - sqrt(dist/radius));
                        if (points < 1)
                                points = 1;

			VectorSubtract (ent->s.origin, inflictor->s.origin, dir);
                        T_Damage (ent, inflictor, attacker, dir, inflictor->s.origin, vec3_origin, (int)points, (int)points, DAMAGE_RADIUS, MOD_NUKE);

                        if (dist <= NUKE_BURNRADIUS)
                                if (ent)
                                        if (ent->health > ent->gib_health)
                                        {       ent->pain_debounce_time = 0;
                                                Coven_Ignite(ent, attacker, ent->s.origin);
                                        }
		}
	}
}

/*-------------------------------------------------------- New Code --------
//  Do this upon nuke impact.
//------------------------------------------------------------------------*/
void Coven_NukeTouch (edict_t *ent, edict_t *other, cplane_t *plane, csurface_t *surf)
{
	vec3_t		origin;
        vec3_t          cloud = {5, 160, 80};
        vec3_t          timer = {0.2, 0.8, 0};
	vec3_t		damage = {6, 1, 100};
	vec3_t		radius_damage = {4, 1, 50};
//        vec3_t          fstorm = {0, 1, 128};

	if (other == ent->owner)
		return;

	if (surf && (surf->flags & SURF_SKY))
	{
		G_FreeEdict (ent);
		return;
	}

	if (ent->owner->client)
		PlayerNoise(ent->owner, ent->s.origin, PNOISE_IMPACT);

	// calculate position for the explosion entity
	VectorMA (ent->s.origin, -0.02, ent->velocity, origin);

	if (other->takedamage)
	{
                T_Damage (other, ent, ent->owner, ent->velocity, ent->s.origin, plane->normal, ent->dmg, 0, 0, MOD_NUKE);
	}
	else
	{
		// don't throw any debris in net games
		if (!deathmatch->value)
		{
			if ((surf) && !(surf->flags & (SURF_WARP|SURF_TRANS33|SURF_TRANS66|SURF_FLOWING)))
			{
                                int     n;

				n = rand() % 5;
				while(n--)
					ThrowDebris (ent, "models/objects/debris2/tris.md2", 2, ent->s.origin);
			}
		}
	}

        Coven_NukeDamage (ent, ent->owner, ent->radius_dmg, ent->dmg_radius);
        Coven_FireDrop (ent->owner, origin, damage, radius_damage, FLAME_TIME, 50, MOD_CLOUD_FIRE, MOD_CLOUD_FIRE);
        Coven_FlameCloud (ent->owner, ent->s.origin, cloud, timer, damage, radius_damage, 100, 75);
//        Coven_CreateFireStorm (ent->owner, ent->s.origin, damage, radius_damage, fstorm);

        Coven_FlashEffects (origin, ent->s.origin);

        VectorCopy (origin, ent->s.origin);
        gi.linkentity (ent);

        Coven_Mushroom (origin);
        Coven_BecomeWave (ent, 0);
}

/*-------------------------------------------------------- New Code --------
//  Launch a nuke.
//------------------------------------------------------------------------*/
void Coven_FireNuke (edict_t *self, vec3_t start, vec3_t dir, int damage, int speed, float damage_radius, int radius_damage)
{
	edict_t	*rocket;

	rocket = G_Spawn();
	VectorCopy (start, rocket->s.origin);
	VectorCopy (dir, rocket->movedir);
	vectoangles (dir, rocket->s.angles);
	VectorScale (dir, speed, rocket->velocity);
        rocket->movetype        = MOVETYPE_FLYMISSILE;
        rocket->clipmask        = MASK_SHOT;
        rocket->solid           = SOLID_BBOX;
        rocket->s.effects       |= EF_ROCKET;
	VectorClear (rocket->mins);
	VectorClear (rocket->maxs);
        rocket->s.modelindex    = gi.modelindex ("models/objects/rocket/tris.md2");
        rocket->owner           = self;
        rocket->touch           = Coven_NukeTouch;
        rocket->nextthink       = level.time + 8000/speed;
        rocket->think           = G_FreeEdict;
        rocket->dmg             = damage;
        rocket->radius_dmg      = radius_damage;
        rocket->dmg_radius      = damage_radius;
        rocket->s.sound         = gi.soundindex ("weapons/rockfly.wav");

	if (self->client)
		check_rocketdodge (self, rocket->s.origin, dir, speed);

	gi.linkentity (rocket);
}


/*=========================/  Black Hole Bombs  /=========================*/
/*
   DESCRIPTION:  Upon impact, a black hole bomb explodes like a rocket
   and immediately inflicts similar damage.  In addition, a hole is
   formed that sucks in light and many other mortal entities.  After
   a brief period of time, the hole explodes into a green flash that
   inflicts very heavy damage and sends all surviving entities at
   very high velocities -- all within a LARGE radius.

   Models used:  hole.md2 (model by Patrick Martin)
                 ripple.md2 (model by Patrick Martin)

   Frames:
      0 -  4  Expanding black hole
 
   Skin:
      0  Full black
*/
/*========================================================================*/

/*-------------------------------------------------------- New Code --------
//  This animates the BFG-style explosion.
//------------------------------------------------------------------------*/
void Coven_CoreThink (edict_t *self)
{
        if (++self->s.frame > 4)
		self->think = G_FreeEdict;

	self->nextthink = level.time + FRAMETIME;
}

/*-------------------------------------------------------- New Code --------
//  This creates a BFG-style explosion.
//------------------------------------------------------------------------*/
void Coven_BecomeCore (edict_t *self)
{
        gi.sound (self, CHAN_VOICE, gi.soundindex ("weapons/bfg__x1b.wav"), 1, ATTN_NORM, 0);
        self->solid             = SOLID_NOT;
        self->touch             = NULL;
        VectorMA (self->s.origin, -1 * FRAMETIME, self->velocity, self->s.origin);
        VectorClear (self->velocity);
        self->s.modelindex      = gi.modelindex ("sprites/s_bfg3.sp2");
        self->s.frame           = 0;
        self->s.sound           = 0;
        self->think             = Coven_CoreThink;
        self->nextthink         = level.time + FRAMETIME;

	gi.WriteByte (svc_temp_entity);
	gi.WriteByte (TE_BFG_BIGEXPLOSION);
	gi.WritePosition (self->s.origin);
	gi.multicast (self->s.origin, MULTICAST_PVS);
}

/*-------------------------------------------------------- New Code --------
//  This creates a shockwave centered on the black hole.
//------------------------------------------------------------------------*/
void Coven_MakeWave (vec3_t origin)
{
        edict_t *wave;

        wave = G_Spawn();
        VectorCopy(origin, wave->s.origin);
        VectorClear(wave->s.angles);
        wave->s.angles[YAW]  = rand() % 360;
        VectorClear(wave->velocity);

        wave->s.modelindex = gi.modelindex("models/mad/ripple/tris.md2");
        wave->s.frame      = 0;
        wave->s.skinnum    = 1;
        wave->movetype     = MOVETYPE_NONE;
        wave->clipmask     = 0;
        wave->solid        = SOLID_NOT;
        wave->s.effects    = EF_BFG;
        wave->s.renderfx   = RF_FULLBRIGHT;
        VectorClear (wave->mins);
        VectorClear (wave->maxs);
        wave->touch        = NULL;
        wave->nextthink    = level.time + FRAMETIME;
        wave->think        = Coven_WaveThink;
        gi.linkentity(wave);
}

/*-------------------------------------------------------- New Code --------
//  Check if entity is close enough to the hole for crushing damage.
//------------------------------------------------------------------------*/
void Coven_GravityDamage (edict_t *targ, edict_t *hole, float dist)
{
        if (dist <= 100)
        {
                int     damage;

                damage = (int)(ceil((100-dist) * 0.2));
                if (damage < 1)
                        damage = 1;

                T_Damage (targ, hole, hole->owner, vec3_origin, targ->s.origin, vec3_origin, damage, damage, DAMAGE_NO_KNOCKBACK|DAMAGE_RADIUS, MOD_BH_GRAVITY);

                /* If client, gib it to stop it from getting drawn through the ceiling. */
                if ((targ->client) && (targ->health <= 0))
                        T_Damage (targ, hole, hole->owner, vec3_origin, targ->s.origin, vec3_origin, 999, 999, DAMAGE_NO_KNOCKBACK|DAMAGE_RADIUS, MOD_BH_GRAVITY);
        }
}

/*-------------------------------------------------------- New Code --------
//  Here, the black hole draws in light and many other objects.
//------------------------------------------------------------------------*/
void Coven_HoleThink (edict_t *self)
{
        edict_t *ent = NULL;
        vec3_t  dir;
        float   dist;

	if (self->timestamp < level.time)
	{
                Coven_BlastRadius (self, self->owner, self->radius_dmg, NULL, self->dmg_radius, false, 500000, MOD_BH_BLAST);
                Coven_MakeWave(self->s.origin);
                Coven_BecomeCore(self);

		while ((ent = findall(ent)) != NULL)
		{
			if (ent->client)
			{
				ent->black_alpha = 0;
				if (CanDamage (ent, self))
				{
					ent->nuke_alpha = 1;
					ent->nuke_fade = 0.25;
				}
			}
		}
		return;
	}

/* Animate hole.  Hole is static once it grows big enough. */
	if (++self->s.frame > 4)
		self->s.frame = 4;

/* Cycle through all entities and apply any effects. */
	while ((ent = findall(ent)) != NULL)
	{
        /* Affect only creatures, projectiles, and barrels. */
		if ( (ent->client) || (ent->svflags & SVF_MONSTER) ||
                     (ent->clipmask & MASK_SHOT) || (!strcmp(ent->classname, "misc_explobox")) )
		{
			VectorSubtract(ent->s.origin, self->s.origin, dir);
			dist = VectorLength(dir);

			if (ent->client)
			{
                        /* Blackout player vision if hole is in LOS. */
				if (CanDamage (ent, self))
				{
                                        /* Could turn off prediction here, but not necessary. */
                                        if (dist <= self->dmg_radius)
                                                ent->black_alpha += 0.1;
                                        else
                                                ent->black_alpha += 0.03;
					if (ent->black_alpha > 1)
						ent->black_alpha = 1;
				}
				else
				{
                                        ent->black_alpha -= 0.1;
					if (ent->black_alpha < 0)
						ent->black_alpha = 0;
				}
			}

                /* Check if entity is close enough to get sucked in. */
                        if (dist <= self->dmg_radius)
			{
				float mass;
				vec3_t dir;

                                /* Some projectiles require special handling. */
                                if (!strcmp (ent->classname, "gas"))
                                {       Coven_KillCloud (ent);
                                        continue;
                                }
                                if (!strcmp (ent->classname, "grapple"))
                                {       Coven_KillGrapple (ent);
                                        continue;
                                }
                                if (!strcmp (ent->classname, "smart_bomb"))
                                {       ent->classname  = "";
                                        ent->nextthink  = ent->timestamp;
                                        ent->think      = G_FreeEdict;
                                        Coven_KillSmartLaser (ent);
                                        continue;
                                }

                                /* If not in DM, check entity for possible special case. */
                                if (!deathmatch->value)
                                {
                                        /* Don't suck in crucified soldiers. */
                                        if (!strcmp(ent->classname, "misc_insane"))
                                                if (ent->spawnflags & 8)
                                                {       Coven_GravityDamage (ent, self, dist);
                                                        continue;
                                                }

                                        /* Ditto for turret drivers. */
                                        if (!strcmp(ent->classname, "turret_driver"))
                                        {       Coven_GravityDamage (ent, self, dist);
                                                continue;
                                        }
                                        
                                        /* Make dead soldiers moveable. */
                                        if (!strcmp(ent->classname, "misc_deadsoldier"))
                                                if (CanDamage (ent, self))
                                                        ent->movetype = MOVETYPE_TOSS;
                                }

                                /* Assume all objects have some mass. */
				if (ent->mass < 50)
					mass = 50;
				else
					mass = ent->mass;

                                /* Suck in the target. */
				VectorSubtract (self->s.origin, ent->s.origin, dir);
				VectorNormalize (dir);
				VectorScale (dir, (100000 / mass), dir);
				VectorMA (dir, 0.2, ent->velocity, ent->velocity);

				if (ent->groundentity)
                                {       /* Lift off the ground if neceesary. */
                                        if (ent->velocity[2] < 200)
                                                ent->velocity[2] = 200;
                                        ent->groundentity = NULL;
                                }

                                Coven_GravityDamage (ent, self, dist);
			}
		}
	}

	self->nextthink = level.time + FRAMETIME;
}

/*-------------------------------------------------------- New Code --------
//  Black hole creation.
//------------------------------------------------------------------------*/
void Coven_BlackHole
(edict_t *attacker, vec3_t start, vec3_t dir, float damage, int speed, float duration, float damage_radius, int radius_damage)
{
	edict_t *hole;
	float  cycles = duration / FRAMETIME;   /* FRAMETIME > 0 */

	if (cycles < 1)   cycles = 1;

/* Check if attacker is pumped up with quad damage. */
        if (attacker->client)
                if (attacker->client->quad_framenum > level.framenum)
                        radius_damage *= 4;

/* Create the black hole. */
	hole = G_Spawn();
        hole->s.modelindex = gi.modelindex("models/mad/hole/tris.md2");
	hole->s.frame      = 0;
	hole->s.skinnum    = 0;
	VectorClear (hole->mins);
	VectorClear (hole->maxs);
	VectorCopy (start, hole->s.origin);
	vectoangles (dir, hole->s.angles);
	if (speed)
	{	VectorScale (dir, speed, hole->velocity);
		hole->movetype     = MOVETYPE_FLYMISSILE;
	}
	else
	{	VectorClear (hole->velocity);
		VectorClear (hole->s.angles);
		hole->movetype     = MOVETYPE_NONE;
	}
	hole->clipmask     = 0;
	hole->solid        = SOLID_NOT;
	hole->takedamage   = DAMAGE_NO;
	hole->s.effects    = 0;
	hole->s.renderfx   = RF_FULLBRIGHT;
        hole->owner        = attacker;
	hole->classname    = "hole";
	hole->touch        = NULL;
	hole->timestamp    = level.time + duration;
	hole->nextthink    = level.time + FRAMETIME;
	hole->think        = Coven_HoleThink;
	hole->dmg_radius   = damage_radius;
        hole->radius_dmg   = radius_damage;

	gi.linkentity(hole);
}

/*-------------------------------------------------------- New Code --------
//  Hole touch
//------------------------------------------------------------------------*/
void Coven_HoleBombTouch
(edict_t *ent, edict_t *other, cplane_t *plane, csurface_t *surf)
{
	vec3_t		origin;

	if (other == ent->owner)
		return;

	if (surf && (surf->flags & SURF_SKY))
	{
		G_FreeEdict (ent);
		return;
	}

	if (ent->owner->client)
		PlayerNoise(ent->owner, ent->s.origin, PNOISE_IMPACT);

	// calculate position for the explosion entity
	VectorMA (ent->s.origin, -0.02, ent->velocity, origin);

	if (other->takedamage)
	{
                T_Damage (other, ent, ent->owner, ent->velocity, ent->s.origin, plane->normal, ent->dmg, 0, 0, MOD_BH_BOMB);
	}
	else
	{
		// don't throw any debris in net games
		if (!deathmatch->value)
		{
			if ((surf) && !(surf->flags & (SURF_WARP|SURF_TRANS33|SURF_TRANS66|SURF_FLOWING)))
			{
                                int     n;

				n = rand() % 5;
				while(n--)
					ThrowDebris (ent, "models/objects/debris2/tris.md2", 2, ent->s.origin);
			}
		}
	}

        T_RadiusDamage(ent, ent->owner, ent->radius_dmg, other, ent->dmg_radius, MOD_BH_SPLASH);
        Coven_BlackHole(ent->owner, origin, vec3_origin, 0, 0, 1.5, NUKE_BURNRADIUS, HOLE_DAMAGE);

	gi.WriteByte (svc_temp_entity);
	if (ent->waterlevel)
		gi.WriteByte (TE_ROCKET_EXPLOSION_WATER);
	else
		gi.WriteByte (TE_ROCKET_EXPLOSION);
	gi.WritePosition (origin);
	gi.multicast (ent->s.origin, MULTICAST_PVS);

        G_FreeEdict (ent);
}

/*-------------------------------------------------------- New Code --------
//  Launch a black hole bomb.
//------------------------------------------------------------------------*/
void Coven_FireHole
(edict_t *self, vec3_t start, vec3_t dir, int damage, int speed, float damage_radius, int radius_damage)
{
	edict_t	*rocket;

	rocket = G_Spawn();
	VectorCopy (start, rocket->s.origin);
	VectorCopy (dir, rocket->movedir);
	vectoangles (dir, rocket->s.angles);
	VectorScale (dir, speed, rocket->velocity);
        rocket->movetype        = MOVETYPE_FLYMISSILE;
        rocket->clipmask        = MASK_SHOT;
        rocket->solid           = SOLID_BBOX;
        rocket->s.effects       |= EF_ROCKET;
	VectorClear (rocket->mins);
	VectorClear (rocket->maxs);
        rocket->s.modelindex    = gi.modelindex ("models/objects/rocket/tris.md2");
        rocket->owner           = self;
        rocket->touch           = Coven_HoleBombTouch;
        rocket->nextthink       = level.time + 8000/speed;
        rocket->think           = G_FreeEdict;
        rocket->dmg             = damage;
        rocket->radius_dmg      = radius_damage;
        rocket->dmg_radius      = damage_radius;
        rocket->s.sound         = gi.soundindex ("weapons/rockfly.wav");

	if (self->client)
		check_rocketdodge (self, rocket->s.origin, dir, speed);

	gi.linkentity (rocket);
}

/*===========================/  END OF FILE  /===========================*/
