/*==========================================================================
//  x_hook.c -- by Patrick Martin               Last updated:  3-8-1999
//--------------------------------------------------------------------------
//  This file contains code that lets the player use the grapple.
//========================================================================*/

#include "g_local.h"
#include "x_hook.h"
#include "x_mad.h"


/*=============================/  Grapple  /=============================*/
/*
   DESCRIPTION:  This fires a projectile with a cable attached.
   If it strikes a shootable target, the target is damaged.  If
   the target is a client or monster, the grapple will stay in
   it to damage it even more.  If it strikes an obstacle, the
   player has the option of pulling himself in.
*/
/*========================================================================*/

/*----------------------------------------------------/ New Code /--------//
//  This is a copy of 'P_ProjectSource'.
//------------------------------------------------------------------------*/
static void Coven_ProjectHookSource
(gclient_t *client, vec3_t point, vec3_t distance, vec3_t forward, vec3_t right, vec3_t result)
{
	vec3_t	_distance;

	VectorCopy (distance, _distance);
	if (client->pers.hand == LEFT_HANDED)
		_distance[1] *= -1;
	else if (client->pers.hand == CENTER_HANDED)
		_distance[1] = 0;
	G_ProjectSource (point, _distance, forward, right, result);
}

/*----------------------------------------------------/ New Code /--------//
//  This destroys the grapple.
//------------------------------------------------------------------------*/
void Coven_KillGrapple (edict_t *self)
{
        if (self->owner)
        {
                if (self->owner->hook == self)
                        self->owner->hook = NULL;

                if (!self->owner->hook)
                        self->owner->client->ps.pmove.pm_flags &= ~PMF_NO_PREDICTION;
        }
        G_FreeEdict (self);
}

/*----------------------------------------------------/ New Code /--------//
//  This checks if the grapple cable should be cut.
//------------------------------------------------------------------------*/
qboolean Coven_CutCable (edict_t *self)
{
        trace_t tr;
        vec3_t  start, dir;
        vec3_t  forward, right;
        vec3_t  offset;
        int     dist;

/* Cut cable if owner died. */
        if (self->owner->deadflag)
                return true;

/* Cut cable if link between hook and its owner is broken. */
        if (!self->owner->hook)
                return true;

/* Cut cable if owner released fire button and is holding grapple. */
        if (!(self->owner->client->buttons & BUTTON_ATTACK))
        {
                if ((!self->owner->client->newweapon) &&
                    (!strcmp (self->owner->client->pers.weapon->pickup_name, "Grapple")))
                        return true;
        }

/* Calculate origin of grapple. */
        VectorCopy (self->pos1, offset);
        if (self->style)
                offset[1] = 0 - offset[1];
        AngleVectors (self->owner->client->v_angle, forward, right, NULL);
        offset[2] += self->owner->viewheight;
        Coven_ProjectHookSource (self->owner->client, self->owner->s.origin, offset, forward, right, start);

/* Cut cable if a solid crosses it. */
        tr = gi.trace (start, NULL, NULL, self->s.origin, self->owner, MASK_SHOT);
        if (tr.fraction < 1)
        {       if (!self->enemy)
                        return true;
                if (tr.ent != self->enemy)
                        return true;
                if (self->enemy->solid == SOLID_BSP)
                        return true;
        }

/* Calculate distance between hook and its owner. */
	VectorSubtract (start, self->s.origin, dir);
        dist = VectorLength (dir);

/* Cut cable if it stretches beyond its range. */
        if (dist > ((int)self->dmg_radius))
                return true;

/* Nothing is interfering with the grapple, leave the cable intact. */
        if (dist >= 64)
        {       /* Draw cable if player is not too close to hook. */
                gi.WriteByte (svc_temp_entity);
                gi.WriteByte (TE_PARASITE_ATTACK);
                gi.WriteShort (self - g_edicts);
                gi.WritePosition (start);
                gi.WritePosition (self->s.origin);
                gi.multicast (start, MULTICAST_PVS);
        }

        return false;
}

/*----------------------------------------------------/ New Code /--------//
//  Here, the grapple attacks and flays a target.
//------------------------------------------------------------------------*/
qboolean Coven_GrappleAttack (edict_t *self)
{
        edict_t *moveent;
        vec3_t  dir;
        vec3_t  spot;
        int     height;
        int     drag = 400;

        /* Grind the target. */
        T_Damage (self->enemy, self, self->owner, self->velocity, self->s.origin, vec3_origin, self->radius_dmg, 0, DAMAGE_NO_KNOCKBACK, MOD_GRAPPLE);

        if (!self->enemy)
                return false;   // FIXME:  Do we need this check?

        /* If target is a client, check if hook should be destroyed. */
        if (self->enemy->client)
        {
                if (self->enemy->client->invincible_framenum > level.framenum)
                        return false;

                /* If client is killed, gib it at the frame before respawning. */
                if ((self->enemy->health <= 0) &&
                    (level.time >= self->enemy->client->respawn_time))
                        T_Damage (self->enemy, self, self->owner, self->velocity, self->s.origin, vec3_origin, 999, 0, DAMAGE_NO_KNOCKBACK, MOD_GRAPPLE);
        }

        /* Remove hook once target is destroyed. */
        if (self->enemy->health <= self->enemy->gib_health)
                return false;

        /* Stay inside the target. */
        VectorCopy (self->enemy->maxs, spot);
        if (self->enemy->health <= 0)
        {       /* Go lower if target is dead. */
                height = spot[2] - self->enemy->mins[2];
                if (height > 32)
                        spot[2] = self->enemy->mins[2] + (height / 2);
                else if (height > 16)
                        spot[2] = self->enemy->mins[2] + 16;
        }
        VectorAdd (self->enemy->mins, spot, spot);
        VectorMA (self->enemy->s.origin, 0.5, spot, self->s.origin);
        gi.linkentity (self);

        /* Pull in the owner or target (whoever has the least mass). */
        VectorSubtract (self->s.origin, self->owner->s.origin, dir);
        VectorNormalize (dir);

        if (self->enemy->mass > self->owner->mass)
        {       /* Pull grapple owner. */
                self->owner->client->ps.pmove.pm_flags |= PMF_NO_PREDICTION;
                moveent = self->owner;
        }
        else
        {       /* Pull target. */
                VectorInverse (dir);
                moveent = self->enemy;

                /* (KISS) Drag victim at medium speed regardless of mass. */
                drag = 600;

                /* If not in DM, check entity for possible special case. */
                if (!deathmatch->value)
                {
                        /* Don't pull in crucified soldiers. */
                        if (!strcmp(moveent->classname, "misc_insane"))
                                if (moveent->spawnflags & 8)
                                        if (moveent->health > moveent->gib_health)
                                                drag = 0;

                        /* Don't pull in turret drivers either. */
                        if (!strcmp(moveent->classname, "turret_driver"))
                                if (moveent->health > moveent->gib_health)
                                        drag = 0;
                }
        }

        if (drag)
        {       /* NOTE:  Total pull speed is ~20% greater than 'drag'. */
                VectorScale (dir, drag, dir);
                VectorMA (dir, 0.2, moveent->velocity, moveent->velocity);
        }

        return true;
}

/*----------------------------------------------------/ New Code /--------//
//  Here, the grapple pulls in its owner.
//------------------------------------------------------------------------*/
qboolean Coven_GrapplePull (edict_t *self)
{
        vec3_t  end, dir;
        trace_t tr;

        /* Stay on target. */
        VectorCopy (self->enemy->velocity, self->velocity);

        /* Remove hook if surface is no longer there (e.g., door). */
        VectorMA (self->s.origin, 8, self->movedir, end);
        tr = gi.trace (self->s.origin, NULL, NULL, end, self->owner, MASK_SHOT);
        if (tr.fraction == 1)
                return false;

        /* Pull in the owner. */
        self->owner->client->ps.pmove.pm_flags |= PMF_NO_PREDICTION;
        VectorSubtract (self->s.origin, self->owner->s.origin, dir);
        VectorNormalize (dir);
        VectorScale (dir, 500, dir);
        VectorMA (dir, 0.2, self->owner->velocity, self->owner->velocity);

        return true;
}

/*----------------------------------------------------/ New Code /--------//
//  This returns the target's type based on solidity and mortality.
//  Used to determine if grapple should disappear, attack, or pull.
//------------------------------------------------------------------------*/
int Coven_TargetType (edict_t *targ)
{
/* Null Target? */
/*
// NOTE:
//    This is commented out because the current code never passes
//    a null target to this.  If later, through code modification,
//    a null target might make it here, uncomment the next two
//    lines of code.

        if (!targ)
                return TARGET_NONE;
*/

/* BSPs of any sort are treated as the world, even if shootable. */
        if (targ->solid == SOLID_BSP)
                return TARGET_WORLD;

/* Attack solid targets that are mortal and not part of the world. */
        if (targ->solid == SOLID_BBOX)
        {       if (targ->takedamage)
                        return TARGET_MORTAL;

                /* Treat indestructable objects as part of the world. */
                return TARGET_WORLD;
        }

/* Non-solid objects, including triggers, are not valid targets. */
        return TARGET_NONE;
}

/*----------------------------------------------------/ New Code /--------//
//  This checks if the grapple is done and should be removed.
//------------------------------------------------------------------------*/
qboolean Coven_GrappleDone (edict_t *self)
{
        if (self->enemy)
        {
                int     targtype = Coven_TargetType (self->enemy);

                if (targtype == TARGET_MORTAL)
                {       if (!Coven_GrappleAttack (self))
                                return true;
                }
                else if (targtype == TARGET_WORLD)
                {       if (!Coven_GrapplePull (self))
                                return true;
                }
                else
                        return true;
        }
        else if (!self->solid)
        {       /* Remove hook if target disappears from Q2. */
                return true;
        }

/* Check cable integrity. */
        return Coven_CutCable (self);
}

/*----------------------------------------------------/ New Code /--------//
//  The grapple reels in its owner until cut.
//------------------------------------------------------------------------*/
void Coven_CheckGrapple (edict_t *self)
{
        if (Coven_GrappleDone (self))
        {       Coven_KillGrapple (self);
                return;
        }

        self->nextthink = level.time + FRAMETIME;
}

/*----------------------------------------------------/ New Code /--------//
//  The grappling projectile touched something.
//------------------------------------------------------------------------*/
void Coven_GrappleTouch
(edict_t *self, edict_t *other, cplane_t *plane, csurface_t *surf)
{
        int     targtype;

	if (surf && (surf->flags & SURF_SKY))
        {
                Coven_KillGrapple (self);
		return;
	}

        targtype = Coven_TargetType (other);

        if (targtype == TARGET_MORTAL)
        {       /* Use medic cable hit noise. */
                gi.sound (self, CHAN_BODY, gi.soundindex("medic/medatck3.wav"), 1, ATTN_NORM, 0);
        }
        else if (targtype == TARGET_WORLD)
        {       /* Play custom clank noise. */
                gi.sound (self, CHAN_BODY, gi.soundindex("weapons/hook/hookhit.wav"), 1, ATTN_NORM, 0);
                VectorClear (self->avelocity);
        }
        else
        {       /* Not a shootable solid?!  Cut the cable. */
                if (self->owner->hook)
                        self->owner->hook = NULL;
        }

        if (other->takedamage)
        {
                T_Damage (other, self, self->owner, self->velocity, self->s.origin, vec3_origin, self->dmg, self->dmg * 2, 0, MOD_GRAPPLE);

                if (targtype == TARGET_MORTAL)
                {
                        if (other->health <= other->gib_health)
                        {       /* Target destroyed -- sever cable. */
                                if (self->owner->hook)
                                        self->owner->hook = NULL;
                        }
                        else
                                self->s.sound = gi.soundindex ("weapons/hook/hookrip.wav");
                }
        }
        else
        {       /* Draw a few sparks. */
                gi.WriteByte (svc_temp_entity);
                gi.WriteByte (TE_SPARKS);
                gi.WritePosition (self->s.origin);
                if (!plane)
                        gi.WriteDir (vec3_origin);
                else
                        gi.WriteDir (plane->normal);
                gi.multicast (self->s.origin, MULTICAST_PVS);
        }

/* Remove grapple if cable was cut. */
        if (self->owner->hook != self)
        {       Coven_KillGrapple (self);
                return;
        }

/* Grapple has attached to something. */
        VectorClear (self->velocity);
        self->touch     = NULL;
        self->enemy     = other;
        self->solid     = SOLID_NOT;
        self->nextthink = level.time + FRAMETIME;
        self->think     = Coven_CheckGrapple;

        gi.linkentity (self);
}

/*----------------------------------------------------/ New Code /--------//
//  This checks if the grapple should disappear or its cable cut.
//------------------------------------------------------------------------*/
void Coven_GrappleThink (edict_t *self)
{
/* Disappear if grapple reached its range limit. */
        if (self->timestamp <= level.time)
        {       Coven_KillGrapple (self);
                return;
        }

/* Check cable integrity. */
        if (Coven_CutCable (self))
        {       /* Sever cable and let grapple run its course. */
                self->owner->hook = NULL;
                self->nextthink = self->timestamp;
                self->think     = Coven_KillGrapple;
                return;
        }

        self->nextthink = level.time + FRAMETIME;
}

/*----------------------------------------------------/ New Code /--------//
//  This creates the grapple projectile.
//
//  Only a client can use this!
//------------------------------------------------------------------------*/
void Coven_FireGrapple
(edict_t *self, vec3_t start, vec3_t dir, vec3_t offset, int damage, int hold_damage, int speed, float hook_range, float cable_range)
{
        edict_t *hook;

        if (!self->client)
                return;

        hook = G_Spawn();
        VectorCopy (start, hook->s.origin);
        VectorCopy (start, hook->s.old_origin);
        VectorCopy (dir, hook->movedir);
        vectoangles (dir, hook->s.angles);
        VectorScale (dir, speed, hook->velocity);
        VectorSet (hook->avelocity, 0, 0, 500);
        hook->movetype        = MOVETYPE_FLYMISSILE;
        hook->clipmask        = MASK_SHOT;
        hook->solid           = SOLID_BBOX;
        hook->s.effects       = 0;
        VectorClear (hook->mins);
        VectorClear (hook->maxs);
        hook->s.modelindex    = gi.modelindex ("models/weapons/v_grapple/grapple.md2");
        hook->s.frame         = 0;
        hook->s.skinnum       = 0;
        hook->style           = CABLE_NORMAL;
        VectorCopy (offset, hook->pos1);
        hook->owner           = self;
        hook->enemy           = NULL;
        hook->classname       = "grapple";
        hook->touch           = Coven_GrappleTouch;
        hook->radius_dmg      = hold_damage;
        hook->dmg_radius      = cable_range;
        hook->dmg             = damage;
        hook->nextthink       = level.time + (hook_range/speed);
        hook->think           = Coven_KillGrapple;

        if (self->client->buttons & BUTTON_ATTACK)
        {       /* Launch hook with cable attached. */
                hook->timestamp       = level.time + (hook_range/speed);
                hook->nextthink       = level.time + FRAMETIME;
                hook->think           = Coven_GrappleThink;
                self->hook            = hook;
        }

        check_rocketdodge (self, hook->s.origin, dir, speed);

        gi.linkentity (hook);
}


/*===========================/  END OF FILE  /===========================*/
