/*==========================================================================
//  x_config.c -- by Patrick Martin             Last updated:  3-4-1999
//--------------------------------------------------------------------------
//  This file contains code for adjustable MOTD.
//========================================================================*/

#include "g_local.h"
#include "x_config.h"

cvar_t  *gamedir;
cvar_t  *file_motd;
cvar_t  *file_maps;


/*===========================/  Map Cycling  /===========================*/

/*------------------------------------------------------/ New Code /--------
//  This selects the next map below the current map.
//------------------------------------------------------------------------*/
void Coven_CycleMap (char *newmap)
{
        FILE    *infile;
        char    filename[MAX_OSPATH];
        char    buffer[MAX_QPATH];
        int     no_of_maps = 0;
        int     maplength;

/* Get path of game directory, and the name of the maps file. */
        sprintf (filename, "%s/%s.txt", gamedir->string, file_maps->string);

/* If map list file does not exist, exit function. */
        if ( (infile = fopen(filename, "r")) == NULL )
                return;

/* Find current map, then select the next map after the current map. */
        while ((fgets(buffer, MAX_QPATH, infile)) && (no_of_maps < MAX_MAPS))
        {
                no_of_maps++;
                sscanf (buffer, "%s", newmap);
                if (!stricmp (level.mapname, newmap))
                {       /* Match found!  Get the next map. */
                        if ((!fgets(buffer, MAX_QPATH, infile)) || (no_of_maps == MAX_MAPS))
                        {       /* Current map is the last, go to first. */
                                rewind (infile);
                                fgets (buffer, MAX_QPATH, infile);
                        }
                        fclose (infile);
                        sscanf (buffer, "%s", newmap);

                        /* If map name is too long, assume the entry is
                           bogus and ignore the map cycler's results.
                        */
                        maplength = strlen(newmap);
                        if (maplength > 8)
                                newmap[0] = 0;

                        return;
                }
        }

/* No match was found. */
        fclose (infile);
}

/*------------------------------------------------------/ New Code /--------
//  This randomly selects the next map.
//------------------------------------------------------------------------*/
void Coven_RandomMap (char *newmap)
{
        FILE    *infile;
        char    filename[MAX_OSPATH];
        char    buffer[MAX_QPATH];
        int     no_of_maps = 0;
        int     new_map_no;
        int     maplength;

/* Get path of game directory, and the name of the maps file. */
        sprintf (filename, "%s/%s.txt", gamedir->string, file_maps->string);

/* If map list file does not exist, exit function. */
        if ( (infile = fopen(filename, "r")) == NULL )
                return;

/* Assign each map name a number. */
        while ((fgets(buffer, MAX_QPATH, infile)) && (no_of_maps < MAX_MAPS))
                no_of_maps++;

/* Do NOT use random selection if less than two map names are found. */
        if (no_of_maps < 2)
                return;

/* Random select a map. */
        new_map_no = ((rand() % no_of_maps) + 1);

/* Check if we can reach the selected map. */
        rewind (infile);

        while (new_map_no--)
        {
                if (!fgets(buffer, MAX_QPATH, infile))
                {       fclose (infile);
                        return;
                }
        }

/* Map found and selected. */
        fclose (infile);
        sscanf (buffer, "%s", newmap);

/* If map name is too long, assume the entry is bogus and ignore
   the map cycler's results.
*/
        maplength = strlen(newmap);
        if (maplength > 8)
                newmap[0] = 0;
}


/*===============================/  MOTD  /===============================*/

static char     motd[MAX_MOTD];

/*------------------------------------------------------/ New Code /--------
//  This checks if the character value is useful and desired.
//------------------------------------------------------------------------*/
qboolean Coven_LegalChar (char c)
{
        int     i = (int)c;

/* Newline character (ascii: 10) is legal, but handled elsewhere. */

/* If value is negative (signed), convert it to positive (unsigned). */
        if (i < 0)
                i += 256;

/* Any character whose value is or between 32 and 254 is legal. */
        if ((i > 31) && (i < 255))
                return true;

/* All others are forbidden.  (Value 255 will CRASH Quake!) */
        return false;
}

/*------------------------------------------------------/ New Code /--------
//  This opens the "motd.txt" file if available, then puts the file's
//  contents into a buffer for later use.
//------------------------------------------------------------------------*/
void Coven_GetMOTD (void)
{
        FILE    *infile;
        char    filename[MAX_OSPATH];
        char    c;
        int     i = 0;
        int     filepos = 0;
        int     linepos = 0;
        int     no_of_lines = 0;
        int     textlimit = MAX_MOTD - 1;

/* Initialize MOTD. */
        motd[0] = 0;

/* Get path of game directory, and the name of the motd file. */
        sprintf (filename, "%s/%s.txt", gamedir->string, file_motd->string);

/* If MOTD file does not exist, exit function.
   NOTE:  File is opened in binary mode because the file may contain
   displayable characters whose value is above the ASCII value 127.
*/
        if ((infile = fopen(filename, "rb")) == NULL)
                return;

/* Read file until EOF or end of MOTD buffer or many lines are read,
   whichever comes first.
*/
        while ( (filepos < textlimit) && (no_of_lines < 16) &&
                ((c = fgetc(infile)) != EOF) )
        {
                if (c == '\n')
                {       motd[i] = c;
                        i++;
                        filepos++;
                        linepos = 0;
                        no_of_lines++;
                }
                else if ( (Coven_LegalChar (c)) && (linepos < 36) )
                {       motd[i] = c;
                        i++;
                        filepos++;
                        linepos++;
                }
                else
                        filepos++;
        }

/* We don't need the file anymore, so close it! */
        fclose (infile);

/* If no message was read, exit function. */
        if (i < 1)
                return;

/* Make sure the rest of the buffer has no garbage. */
        motd[i] = 0;
}

/*------------------------------------------------------/ New Code /--------
//  This centerprints the MOTD if any.
//------------------------------------------------------------------------*/
void Coven_ShowMOTD (edict_t *ent)
{
/* If no MOTD, nothing to show. */
        if (!motd[0])
                return;

/* Display the MOTD. */
        gi.centerprintf (ent, motd);
}


/*==========================/  Initialization  /==========================*/

/*------------------------------------------------------/ New Code /--------
//  This initializes some variables and the MOTD.
//------------------------------------------------------------------------*/
void Coven_ConfigInit (void)
{
        gamedir = gi.cvar ("game", "", CVAR_SERVERINFO | CVAR_LATCH);
        file_motd = gi.cvar ("file_motd", "motd", CVAR_SERVERINFO);
        file_maps = gi.cvar ("file_maps", "maps", CVAR_SERVERINFO);

        Coven_GetMOTD ();
}


/*===========================/  END OF FILE  /===========================*/
