// g_weapon.c

#include "g_local.h"
#include "m_player.h"
// COVEN:  New header files and definitions.
#include "x_ctf.h"
#include "x_fire.h"
#include "x_hook.h"
#include "x_mad.h"

#define  CELL_COST_SMART        15
#define  CELL_COST_BLACK        50

#define  SPEED_FASTROCKET     1000
#define  SPEED_ROCKET          650
#define  SPEED_HEAVYROCKET     550
// COVEN


static qboolean	is_quad;
static byte		is_silenced;


void weapon_grenade_fire (edict_t *ent, qboolean held);


static void P_ProjectSource (gclient_t *client, vec3_t point, vec3_t distance, vec3_t forward, vec3_t right, vec3_t result)
{
	vec3_t	_distance;

	VectorCopy (distance, _distance);
	if (client->pers.hand == LEFT_HANDED)
		_distance[1] *= -1;
	else if (client->pers.hand == CENTER_HANDED)
		_distance[1] = 0;
	G_ProjectSource (point, _distance, forward, right, result);
}


/*
===============
PlayerNoise

Each player can have two noise objects associated with it:
a personal noise (jumping, pain, weapon firing), and a weapon
target noise (bullet wall impacts)

Monsters that don't directly see the player can move
to a noise in hopes of seeing the player from there.
===============
*/
void PlayerNoise(edict_t *who, vec3_t where, int type)
{
	edict_t		*noise;

	if (type == PNOISE_WEAPON)
	{
		if (who->client->silencer_shots)
		{
			who->client->silencer_shots--;
			return;
		}
	}

	if (deathmatch->value)
		return;

	if (who->flags & FL_NOTARGET)
		return;


	if (!who->mynoise)
	{
		noise = G_Spawn();
		noise->classname = "player_noise";
		VectorSet (noise->mins, -8, -8, -8);
		VectorSet (noise->maxs, 8, 8, 8);
		noise->owner = who;
		noise->svflags = SVF_NOCLIENT;
		who->mynoise = noise;

		noise = G_Spawn();
		noise->classname = "player_noise";
		VectorSet (noise->mins, -8, -8, -8);
		VectorSet (noise->maxs, 8, 8, 8);
		noise->owner = who;
		noise->svflags = SVF_NOCLIENT;
		who->mynoise2 = noise;
	}

	if (type == PNOISE_SELF || type == PNOISE_WEAPON)
	{
		noise = who->mynoise;
		level.sound_entity = noise;
		level.sound_entity_framenum = level.framenum;
	}
	else // type == PNOISE_IMPACT
	{
		noise = who->mynoise2;
		level.sound2_entity = noise;
		level.sound2_entity_framenum = level.framenum;
	}

	VectorCopy (where, noise->s.origin);
	VectorSubtract (where, noise->maxs, noise->absmin);
	VectorAdd (where, noise->maxs, noise->absmax);
	noise->teleport_time = level.time;
	gi.linkentity (noise);
}


qboolean Pickup_Weapon (edict_t *ent, edict_t *other)
{
	int			index;
	gitem_t		*ammo;

	index = ITEM_INDEX(ent->item);

	if ( ( ((int)(dmflags->value) & DF_WEAPONS_STAY) || coop->value) 
		&& other->client->pers.inventory[index])
	{
		if (!(ent->spawnflags & (DROPPED_ITEM | DROPPED_PLAYER_ITEM) ) )
			return false;	// leave the weapon for others to pickup
	}

	other->client->pers.inventory[index]++;

	if (!(ent->spawnflags & DROPPED_ITEM) )
	{
		// give them some ammo with it
		ammo = FindItem (ent->item->ammo);
		if ( (int)dmflags->value & DF_INFINITE_AMMO )
			Add_Ammo (other, ammo, 1000);
		else
			Add_Ammo (other, ammo, ammo->quantity);

		if (! (ent->spawnflags & DROPPED_PLAYER_ITEM) )
		{
			if (deathmatch->value)
			{
				if ((int)(dmflags->value) & DF_WEAPONS_STAY)
					ent->flags |= FL_RESPAWN;
				else
					SetRespawn (ent, 30);
			}
			if (coop->value)
				ent->flags |= FL_RESPAWN;
		}
	}

	if (other->client->pers.weapon != ent->item && 
		(other->client->pers.inventory[index] == 1) &&
		( !deathmatch->value || other->client->pers.weapon == FindItem("blaster") ) )
		other->client->newweapon = ent->item;

	return true;
}


/*
===============
ChangeWeapon

The old weapon has been dropped all the way, so make the new one
current
===============
*/
void ChangeWeapon (edict_t *ent)
{
	int i;

	if (ent->client->grenade_time)
	{
		ent->client->grenade_time = level.time;
		ent->client->weapon_sound = 0;
// COVEN:  Don't double detonate hand grenades.
                if (!ent->client->grenade_blew_up)
                        weapon_grenade_fire (ent, false);
// COVEN
		ent->client->grenade_time = 0;
	}

	ent->client->pers.lastweapon = ent->client->pers.weapon;
	ent->client->pers.weapon = ent->client->newweapon;
	ent->client->newweapon = NULL;
	ent->client->machinegun_shots = 0;

	// set visible model
	if (ent->s.modelindex == 255) {
		if (ent->client->pers.weapon)
			i = ((ent->client->pers.weapon->weapmodel & 0xff) << 8);
		else
			i = 0;
		ent->s.skinnum = (ent - g_edicts - 1) | i;
	}

	if (ent->client->pers.weapon && ent->client->pers.weapon->ammo)
		ent->client->ammo_index = ITEM_INDEX(FindItem(ent->client->pers.weapon->ammo));
	else
		ent->client->ammo_index = 0;

	if (!ent->client->pers.weapon)
	{	// dead
		ent->client->ps.gunindex = 0;
		return;
	}

	ent->client->weaponstate = WEAPON_ACTIVATING;
	ent->client->ps.gunframe = 0;
	ent->client->ps.gunindex = gi.modelindex(ent->client->pers.weapon->view_model);
// COVEN:  Show weapon.
        ent->client->ps.stats[STAT_PICKUP_ICON] = gi.imageindex(ent->client->pers.weapon->icon);
        ent->client->ps.stats[STAT_PICKUP_STRING] = CS_ITEMS+ITEM_INDEX(ent->client->pers.weapon);
        ent->client->pickup_msg_time = level.time + 2.0;
// COVEN

	ent->client->anim_priority = ANIM_PAIN;
	if(ent->client->ps.pmove.pm_flags & PMF_DUCKED)
	{
			ent->s.frame = FRAME_crpain1;
			ent->client->anim_end = FRAME_crpain4;
	}
	else
	{
			ent->s.frame = FRAME_pain301;
			ent->client->anim_end = FRAME_pain304;
			
	}
}

/*
=================
NoAmmoWeaponChange
=================
*/
void NoAmmoWeaponChange (edict_t *ent)
{
// COVEN:  Switch to safer rockets or the blaster.
        if ( ent->client->pers.inventory[ITEM_INDEX(FindItem("Blast-Rockets"))] )
        {
                if ( ent->client->pers.inventory[ITEM_INDEX(FindItem("Cells"))] >= CELL_COST_SMART
                        &&  ent->client->pers.inventory[ITEM_INDEX(FindItem("Smart Bomb"))] )
                {
                        ent->client->newweapon = FindItem ("Smart Bomb");
                        return;
                }
                if ( ent->client->pers.inventory[ITEM_INDEX(FindItem("Blast-Rockets"))] > 1
                        &&  ent->client->pers.inventory[ITEM_INDEX(FindItem("Blast Bomb II"))] )
                {
                        ent->client->newweapon = FindItem ("Blast Bomb II");
                        return;
                }
                if ( ent->client->pers.inventory[ITEM_INDEX(FindItem("Blast Bomb I"))] )
                {
                        ent->client->newweapon = FindItem ("Blast Bomb I");
                        return;
                }
        }
// COVEN
	ent->client->newweapon = FindItem ("blaster");
}

/*
=================
Think_Weapon

Called by ClientBeginServerFrame and ClientThink
=================
*/
void Think_Weapon (edict_t *ent)
{
	// if just died, put the weapon away
	if (ent->health < 1)
	{
		ent->client->newweapon = NULL;
		ChangeWeapon (ent);
	}

	// call active weapon think routine
	if (ent->client->pers.weapon && ent->client->pers.weapon->weaponthink)
	{
		is_quad = (ent->client->quad_framenum > level.framenum);
		if (ent->client->silencer_shots)
			is_silenced = MZ_SILENCED;
		else
			is_silenced = 0;
		ent->client->pers.weapon->weaponthink (ent);
	}
}


/*
================
Use_Weapon

Make the weapon ready if there is ammo
================
*/
void Use_Weapon (edict_t *ent, gitem_t *item)
{
	int			ammo_index;
	gitem_t		*ammo_item;
// COVEN:  Secondary ammo vars.
        int     br_index, cr_index, cell_index;
        gitem_t *br_item, *cr_item, *cell_item;
// COVEN

	// see if we're already using it
	if (item == ent->client->pers.weapon)
// COVEN:  Show weapon, but only if not spectating.
        {
                if (!ent->client->resp.spectator)
                {
                        ent->client->ps.stats[STAT_PICKUP_ICON] = gi.imageindex(ent->client->pers.weapon->icon);
                        ent->client->ps.stats[STAT_PICKUP_STRING] = CS_ITEMS+ITEM_INDEX(ent->client->pers.weapon);
                        ent->client->pickup_msg_time = level.time + 2.0;
                }
		return;
        }
// COVEN

// COVEN:  Don't use weapon if disabled.
        if (no_grapple)
        {
                if (!strcmp(item->pickup_name, "Grapple"))
                {       gi.cprintf (ent, PRINT_HIGH, "Grapple is disabled.\n");
                        if (ent->hook)
                        {       gi.cprintf (ent, PRINT_HIGH, "Cable severed.\n");
                                ent->hook = NULL;
                        }
                        return;
                }
        }
        if (no_nukes)
        {
                if ( (!strcmp(item->pickup_name, "Nuclear Bomb")) ||
                     (!strcmp(item->pickup_name, "Black Hole Bomb")) )
                {       gi.cprintf (ent, PRINT_HIGH, "Nukes are disabled.\n");
                        return;
                }
        }
// COVEN

	if (item->ammo && !g_select_empty->value && !(item->flags & IT_AMMO))
	{
		ammo_item = FindItem(item->ammo);
		ammo_index = ITEM_INDEX(ammo_item);

		if (!ent->client->pers.inventory[ammo_index])
		{
			gi.cprintf (ent, PRINT_HIGH, "No %s for %s.\n", ammo_item->pickup_name, item->pickup_name);
			return;
		}

		if (ent->client->pers.inventory[ammo_index] < item->quantity)
		{
			gi.cprintf (ent, PRINT_HIGH, "Not enough %s for %s.\n", ammo_item->pickup_name, item->pickup_name);
			return;
		}

// COVEN:  Check for 2nd ammo used by some bombs.
                if (!strcmp(item->pickup_name, "Mind Gas Bomb"))
                {
                        cr_item  = FindItem("Chem-Rockets");
                        cr_index = ITEM_INDEX(cr_item);
                        if (!ent->client->pers.inventory[cr_index])
                        {
                                gi.cprintf (ent, PRINT_HIGH, "No %s for %s.\n", "Chem-Rockets", item->pickup_name);
                                return;
                        }
                        if (ent->client->pers.inventory[cr_index] < 1)
                        {
                                gi.cprintf (ent, PRINT_HIGH, "Not enough %s for %s.\n", "Chem-Rockets", item->pickup_name);
                                return;
                        }
                }
                if (!strcmp(item->pickup_name, "Napalm Bomb"))
                {
                        br_item  = FindItem("Blast-Rockets");
                        br_index = ITEM_INDEX(br_item);
                        if (!ent->client->pers.inventory[br_index])
                        {
                                gi.cprintf (ent, PRINT_HIGH, "No %s for %s.\n", "Blast-Rockets", item->pickup_name);
                                return;
                        }
                        if (ent->client->pers.inventory[br_index] < 1)
                        {
                                gi.cprintf (ent, PRINT_HIGH, "Not enough %s for %s.\n", "Blast-Rockets", item->pickup_name);
                                return;
                        }
                }
                cell_item  = FindItem("Cells");
                cell_index = ITEM_INDEX(cell_item);

                if (!strcmp(item->pickup_name, "Smart Bomb"))
                {
                        if (!ent->client->pers.inventory[cell_index])
                        {
                                gi.cprintf (ent, PRINT_HIGH, "No %s for %s.\n", "Cells", item->pickup_name);
                                return;
                        }
                        if (ent->client->pers.inventory[cell_index] < CELL_COST_SMART)
                        {
                                gi.cprintf (ent, PRINT_HIGH, "Not enough %s for %s.\n", "Cells", item->pickup_name);
                                return;
                        }
                }
                if (!strcmp(item->pickup_name, "Black Hole Bomb"))
                {
                        if (!ent->client->pers.inventory[cell_index])
                        {
                                gi.cprintf (ent, PRINT_HIGH, "No %s for %s.\n", "Cells", item->pickup_name);
                                return;
                        }
                        if (ent->client->pers.inventory[cell_index] < CELL_COST_BLACK)
                        {
                                gi.cprintf (ent, PRINT_HIGH, "Not enough %s for %s.\n", "Cells", item->pickup_name);
                                return;
                        }
                }
// COVEN
	}

	// change to this weapon when down
	ent->client->newweapon = item;
}



/*
================
Drop_Weapon
================
*/
void Drop_Weapon (edict_t *ent, gitem_t *item)
{
	int		index;

	if ((int)(dmflags->value) & DF_WEAPONS_STAY)
		return;

	index = ITEM_INDEX(item);
	// see if we're already using it
	if ( ((item == ent->client->pers.weapon) || (item == ent->client->newweapon))&& (ent->client->pers.inventory[index] == 1) )
	{
		gi.cprintf (ent, PRINT_HIGH, "Can't drop current weapon\n");
		return;
	}

	Drop_Item (ent, item);
	ent->client->pers.inventory[index]--;
}


//======================================================================//
// COVEN:  Support functions!

// Some of the replacement weapons require two ammo types.
// Check if weapon has all the required ammo.
static qboolean Coven_HasAmmo (edict_t *ent)
{
        int     sec_index;
        gitem_t *sec_item;

        // In no nukes are flagged, jam the weapon if player tries to fire.
        if (no_nukes)
        {
                if ( (!strcmp(ent->client->pers.weapon->pickup_name, "Nuclear Bomb")) ||
                     (!strcmp(ent->client->pers.weapon->pickup_name, "Black Hole Bomb")) )
                {
                        gi.cprintf (ent, PRINT_HIGH, "Weapon jammed!\n");
                        return false;
                }
        }

        // Check if weapon is unlimited.  (e.g., blaster.)
        if (!ent->client->ammo_index)
                return true;

        // Check if weapon has enough ammo.  Check multi-ammo first.
        if (!strcmp(ent->client->pers.weapon->pickup_name, "Mind Gas Bomb"))
        {
                sec_item  = FindItem("Chem-Rockets");
                sec_index = ITEM_INDEX(sec_item);
                if (ent->client->pers.inventory[sec_index] < 1)
                        return false;
        }
        if (!strcmp(ent->client->pers.weapon->pickup_name, "Napalm Bomb"))
        {
                sec_item  = FindItem("Blast-Rockets");
                sec_index = ITEM_INDEX(sec_item);
                if (ent->client->pers.inventory[sec_index] < 1)
                        return false;
        }
        sec_item  = FindItem("Cells");
        sec_index = ITEM_INDEX(sec_item);
        if (!strcmp(ent->client->pers.weapon->pickup_name, "Smart Bomb"))
        {
                if (ent->client->pers.inventory[sec_index] < CELL_COST_SMART)
                        return false;
        }
        if (!strcmp(ent->client->pers.weapon->pickup_name, "Black Hole Bomb"))
        {
                if (ent->client->pers.inventory[sec_index] < CELL_COST_BLACK)
                        return false;
        }


        if (ent->client->pers.inventory[ent->client->ammo_index] >= ent->client->pers.weapon->quantity)
                return true;

        return false;
}

// Play powered-up attack sound.
static void Coven_PowerupSfx (edict_t *ent)
{
        if (CTFApplyStrengthSound (ent))
                return;

        if ((ent->client->quad_framenum > level.framenum) && (ent->client->quadsndtime < level.time))
        {       ent->client->quadsndtime = level.time + 1;
                gi.sound (ent, CHAN_ITEM, gi.soundindex("items/damage3.wav"), 1, ATTN_NORM, 0);
                return;
        }

        CTFApplyHasteSound (ent);
}

// Return vector to where the crosshairs point exactly.  Used by grapple.
static void Coven_CrosshairVector
(vec3_t forward, vec3_t start, edict_t *ent)
{
        trace_t tr;
        vec3_t  v;

        VectorCopy (ent->s.origin, v);
        v[2] += ent->viewheight;
        VectorMA (v, 8192, forward, forward);
        tr = gi.trace (v, NULL, NULL, forward, ent, MASK_SHOT);
        VectorSubtract (tr.endpos, start, forward);
        VectorNormalize (forward);
}

// COVEN
//======================================================================//


/*
================
Weapon_Generic

A generic function to handle the basics of weapon thinking
================
*/
#define FRAME_FIRE_FIRST		(FRAME_ACTIVATE_LAST + 1)
#define FRAME_IDLE_FIRST		(FRAME_FIRE_LAST + 1)
#define FRAME_DEACTIVATE_FIRST	(FRAME_IDLE_LAST + 1)

// COVEN:  Functions heavily modified.
void Weapon_Generic2 (edict_t *ent, int FRAME_ACTIVATE_LAST, int FRAME_FIRE_LAST, int FRAME_IDLE_LAST, int FRAME_DEACTIVATE_LAST, int *pause_frames, int *fire_frames, void (*fire)(edict_t *ent))
{
	int		n;

	if (ent->client->weaponstate == WEAPON_DROPPING)
	{
		if (ent->client->ps.gunframe == FRAME_DEACTIVATE_LAST)
		{
			ChangeWeapon (ent);
			return;
		}
		else if ((FRAME_DEACTIVATE_LAST - ent->client->ps.gunframe) == 4)
		{
			ent->client->anim_priority = ANIM_REVERSE;
			if(ent->client->ps.pmove.pm_flags & PMF_DUCKED)
			{
				ent->s.frame = FRAME_crpain4+1;
				ent->client->anim_end = FRAME_crpain1;
			}
			else
			{
				ent->s.frame = FRAME_pain304+1;
				ent->client->anim_end = FRAME_pain301;
				
			}
		}

		ent->client->ps.gunframe++;
		return;
	}

	if (ent->client->weaponstate == WEAPON_ACTIVATING)
	{
                if ( (ent->client->ps.gunframe == FRAME_ACTIVATE_LAST) ||
                     ((int)(dmflags->value) & DF_FAST_SWITCH) )
		{
			ent->client->weaponstate = WEAPON_READY;
			ent->client->ps.gunframe = FRAME_IDLE_FIRST;
			return;
		}

		ent->client->ps.gunframe++;
		return;
	}

	if ((ent->client->newweapon) && (ent->client->weaponstate != WEAPON_FIRING))
	{
                // Check for instant switch.  Pain/switch anim is bypassed as well.
                if ((int)(dmflags->value) & DF_FAST_SWITCH)
		{       // Bypass pain/switch anim as well.
                        ChangeWeapon (ent);
			return;
		}

		ent->client->weaponstate = WEAPON_DROPPING;
		ent->client->ps.gunframe = FRAME_DEACTIVATE_FIRST;

		if ((FRAME_DEACTIVATE_LAST - FRAME_DEACTIVATE_FIRST) < 4)
		{
			ent->client->anim_priority = ANIM_REVERSE;
			if(ent->client->ps.pmove.pm_flags & PMF_DUCKED)
			{
				ent->s.frame = FRAME_crpain4+1;
				ent->client->anim_end = FRAME_crpain1;
			}
			else
			{
				ent->s.frame = FRAME_pain304+1;
				ent->client->anim_end = FRAME_pain301;
				
			}
		}
		return;
	}

	if (ent->client->weaponstate == WEAPON_READY)
	{
		if ( ((ent->client->latched_buttons|ent->client->buttons) & BUTTON_ATTACK) )
		{
			ent->client->latched_buttons &= ~BUTTON_ATTACK;
                        if (Coven_HasAmmo (ent))
			{
				ent->client->ps.gunframe = FRAME_FIRE_FIRST;
				ent->client->weaponstate = WEAPON_FIRING;

				// start the animation
				ent->client->anim_priority = ANIM_ATTACK;
				if (ent->client->ps.pmove.pm_flags & PMF_DUCKED)
				{
					ent->s.frame = FRAME_crattak1-1;
					ent->client->anim_end = FRAME_crattak9;
				}
				else
				{
					ent->s.frame = FRAME_attack1-1;
					ent->client->anim_end = FRAME_attack8;
				}
			}
			else
			{
				if (level.time >= ent->pain_debounce_time)
				{
					gi.sound(ent, CHAN_VOICE, gi.soundindex("weapons/noammo.wav"), 1, ATTN_NORM, 0);
					ent->pain_debounce_time = level.time + 1;
				}
				NoAmmoWeaponChange (ent);
			}
		}
		else
		{
			if (ent->client->ps.gunframe == FRAME_IDLE_LAST)
			{
				ent->client->ps.gunframe = FRAME_IDLE_FIRST;
				return;
			}

			if (pause_frames)
			{
				for (n = 0; pause_frames[n]; n++)
				{
					if (ent->client->ps.gunframe == pause_frames[n])
					{
						if (rand()&15)
							return;
					}
				}
			}

			ent->client->ps.gunframe++;
			return;
		}
	}

	if (ent->client->weaponstate == WEAPON_FIRING)
	{
		for (n = 0; fire_frames[n]; n++)
		{
			if (ent->client->ps.gunframe == fire_frames[n])
			{
                                // Quad/tech sounds played elsewhere...
				fire (ent);
				break;
			}
		}

		if (!fire_frames[n])
			ent->client->ps.gunframe++;

		if (ent->client->ps.gunframe == FRAME_IDLE_FIRST+1)
			ent->client->weaponstate = WEAPON_READY;
	}
}

void Weapon_Generic (edict_t *ent, int FRAME_ACTIVATE_LAST, int FRAME_FIRE_LAST, int FRAME_IDLE_LAST, int FRAME_DEACTIVATE_LAST, int *pause_frames, int *fire_frames, void (*fire)(edict_t *ent))
{
	int oldstate = ent->client->weaponstate;

        // VWep animations screw up corpses
        if (ent->deadflag || ent->s.modelindex != 255)
		return;

	Weapon_Generic2 (ent, FRAME_ACTIVATE_LAST, FRAME_FIRE_LAST, 
		FRAME_IDLE_LAST, FRAME_DEACTIVATE_LAST, pause_frames, 
		fire_frames, fire);

	// run the weapon frame again if hasted
        if (CTFApplyHaste (ent) && (oldstate == ent->client->weaponstate))
        {
		Weapon_Generic2 (ent, FRAME_ACTIVATE_LAST, FRAME_FIRE_LAST,
                        FRAME_IDLE_LAST, FRAME_DEACTIVATE_LAST, pause_frames,
			fire_frames, fire);
	}

        // Apply super coolant.
        if ((ent->client->cool_framenum > level.framenum) &&
            (stricmp(ent->client->pers.weapon->pickup_name, "Blaster")) &&
            (ent->client->weaponstate == WEAPON_FIRING))
        {
		Weapon_Generic2 (ent, FRAME_ACTIVATE_LAST, FRAME_FIRE_LAST, 
			FRAME_IDLE_LAST, FRAME_DEACTIVATE_LAST, pause_frames, 
			fire_frames, fire);
	}
}
// COVEN


/*========================================================================*/
/*  ARMOURBACK GRAPPLE */

void Coven_Hook
(edict_t *ent, int damage, int hold_damage, int speed, int hook_range, int cable_range)
{
	vec3_t	forward, right;
	vec3_t	start;
	vec3_t	offset;
        vec3_t  base_offset = {8, 8, -8};

	if (is_quad)
        {       damage *= 4;
                hold_damage *= 4;
        }
        Coven_PowerupSfx (ent);

	AngleVectors (ent->client->v_angle, forward, right, NULL);
        VectorCopy (base_offset, offset);
        offset[2] += ent->viewheight;
	P_ProjectSource (ent->client, ent->s.origin, offset, forward, right, start);

	VectorScale (forward, -2, ent->client->kick_origin);
	ent->client->kick_angles[0] = -1;

        Coven_CrosshairVector (forward, start, ent);
        Coven_FireGrapple (ent, start, forward, base_offset, damage, hold_damage, speed, hook_range, cable_range);

        gi.sound (ent, CHAN_WEAPON, gi.soundindex("weapons/hook/hookfire.wav"), 1, ATTN_NORM, 0);

	PlayerNoise(ent, start, PNOISE_WEAPON);
}

// 0-8 / 9-16 / 17-18 / 19-22
void Weapon_Grapple2 (edict_t *ent)
{
        // Check if player is putting away the grapple.
	if (ent->client->weaponstate == WEAPON_DROPPING)
        {       if (ent->client->ps.gunframe == 22)
		{       ChangeWeapon (ent);
			return;
		}
		ent->client->ps.gunframe++;
		return;
	}

        // Check if player is pulling out the grapple.
	if (ent->client->weaponstate == WEAPON_ACTIVATING)
        {
                if (ent->hook)
                {       ent->client->weaponstate = WEAPON_FIRING;
                        ent->client->ps.gunframe = 11;
			return;
		}
                if ( (ent->client->ps.gunframe == 8) ||
                     ((int)(dmflags->value) & DF_FAST_SWITCH) )
		{       ent->client->weaponstate = WEAPON_READY;
                        ent->client->ps.gunframe = 17;
			return;
		}
		ent->client->ps.gunframe++;
		return;
	}

        // Check if player wants (or is forced) to put away the grapple.
        if (ent->client->newweapon && (ent->client->weaponstate != WEAPON_FIRING))
        {       if ((int)(dmflags->value) & DF_FAST_SWITCH)
		{       ChangeWeapon (ent);
			return;
		}
                ent->client->weaponstate = WEAPON_DROPPING;
                ent->client->ps.gunframe = 19;
		return;
	}

        // Check if grapple is ready and idle.
	if (ent->client->weaponstate == WEAPON_READY)
	{
		if ( ((ent->client->latched_buttons|ent->client->buttons) & BUTTON_ATTACK) )
		{       ent->client->latched_buttons &= ~BUTTON_ATTACK;
                        if (no_grapple)
                        {       // Jam grapple if it is disabled.
                                gi.cprintf (ent, PRINT_HIGH, "Weapon jammed!\n");
				if (level.time >= ent->pain_debounce_time)
				{       gi.sound (ent, CHAN_VOICE, gi.soundindex("weapons/noammo.wav"), 1, ATTN_NORM, 0);
					ent->pain_debounce_time = level.time + 1;
				}
				NoAmmoWeaponChange (ent);
                                return;
                        }
                        ent->client->ps.gunframe = 9;
                        ent->client->weaponstate = WEAPON_FIRING;

                        // Start animation.
                        ent->client->anim_priority = ANIM_ATTACK;
                        if (ent->client->ps.pmove.pm_flags & PMF_DUCKED)
                        {       ent->s.frame = FRAME_crattak1-1;
                                ent->client->anim_end = FRAME_crattak9;
                        }
                        else
                        {       ent->s.frame = FRAME_attack1-1;
                                ent->client->anim_end = FRAME_attack8;
                        }
			return;
		}

                // Animate idle frames
                ent->client->ps.gunframe = 17;
		return;
	}

        // Check if grapple is firing.
	if (ent->client->weaponstate == WEAPON_FIRING)
	{
                if (ent->client->ps.gunframe == 9)
                {       Coven_Hook (ent, 20, 2, 1000, 2000, 2000);
                        ent->client->ps.gunframe++;
                        return;
                }

                if (ent->client->ps.gunframe == 11)
                {
                        if (ent->hook)
                        {
                                if (ent->client->newweapon)
                                        ChangeWeapon (ent);
                                return;
                        }
                }

		ent->client->ps.gunframe++;

                if (ent->client->ps.gunframe == 17)
			ent->client->weaponstate = WEAPON_READY;
	}
}

void Weapon_Grapple (edict_t *ent)
{
	int oldstate = ent->client->weaponstate;

        // VWep animations screw up corpses
        if (ent->deadflag || ent->s.modelindex != 255)
		return;

        Weapon_Grapple2 (ent);

	// run the weapon frame again if hasted
        if (CTFApplyHaste (ent) && (oldstate == ent->client->weaponstate))
                Weapon_Grapple2 (ent);
}


/*========================================================================*/
/*  MAD WEAPONARY */

/*----------------//
//  BLAST BOMB I  //
//----------------*/
void Coven_BlastRocket_Fire (edict_t *ent)
{
	vec3_t	offset, start;
	vec3_t	forward, right;
	int		damage;
	float	damage_radius;
	int		radius_damage;

        damage = 30 + (int)(random() * 10.0);
        radius_damage = 40;
	damage_radius = 120;
	if (is_quad)
	{
		damage *= 4;
		radius_damage *= 4;
	}
        Coven_PowerupSfx (ent);

	AngleVectors (ent->client->v_angle, forward, right, NULL);

	VectorScale (forward, -2, ent->client->kick_origin);
	ent->client->kick_angles[0] = -1;

	VectorSet(offset, 8, 8, ent->viewheight-8);
	P_ProjectSource (ent->client, ent->s.origin, offset, forward, right, start);
        Coven_FireBlast (ent, start, forward, damage, (SPEED_FASTROCKET * 2), damage_radius, radius_damage, 150000);

	// send muzzle flash
	gi.WriteByte (svc_muzzleflash);
	gi.WriteShort (ent-g_edicts);
	gi.WriteByte (MZ_ROCKET | is_silenced);
	gi.multicast (ent->s.origin, MULTICAST_PVS);

	ent->client->ps.gunframe++;

	PlayerNoise(ent, start, PNOISE_WEAPON);

        if (! ( (int)dmflags->value & DF_INFINITE_AMMO ) )
                ent->client->pers.inventory[ent->client->ammo_index] -= ent->client->pers.weapon->quantity;
}

void Weapon_BlastBomb (edict_t *ent)
{
	static int	pause_frames[]	= {25, 33, 42, 50, 0};
	static int	fire_frames[]	= {5, 0};

        Weapon_Generic (ent, 4, 12, 50, 54, pause_frames, fire_frames, Coven_BlastRocket_Fire);
}

/*-----------------//
//  BLAST BOMB II  //
//-----------------*/
void Coven_BlastRocket2_Fire (edict_t *ent)
{
	vec3_t	offset, start;
	vec3_t	forward, right;
	int		damage;
	float	damage_radius;
	int		radius_damage;

	damage = 100 + (int)(random() * 20.0);
	radius_damage = 120;
        damage_radius = 160;
	if (is_quad)
	{
		damage *= 4;
		radius_damage *= 4;
	}
        Coven_PowerupSfx (ent);

	AngleVectors (ent->client->v_angle, forward, right, NULL);

	VectorScale (forward, -2, ent->client->kick_origin);
	ent->client->kick_angles[0] = -1;

	VectorSet(offset, 8, 8, ent->viewheight-8);
	P_ProjectSource (ent->client, ent->s.origin, offset, forward, right, start);
        Coven_FireBlast (ent, start, forward, damage, SPEED_FASTROCKET, damage_radius, radius_damage, 300000);

	// send muzzle flash
	gi.WriteByte (svc_muzzleflash);
	gi.WriteShort (ent-g_edicts);
	gi.WriteByte (MZ_ROCKET | is_silenced);
	gi.multicast (ent->s.origin, MULTICAST_PVS);

	ent->client->ps.gunframe++;

	PlayerNoise(ent, start, PNOISE_WEAPON);

        if (! ( (int)dmflags->value & DF_INFINITE_AMMO ) )
                ent->client->pers.inventory[ent->client->ammo_index] -= ent->client->pers.weapon->quantity;
}

void Weapon_BlastBomb2 (edict_t *ent)
{
	static int	pause_frames[]	= {25, 33, 42, 50, 0};
	static int	fire_frames[]	= {5, 0};

        Weapon_Generic (ent, 4, 12, 50, 54, pause_frames, fire_frames, Coven_BlastRocket2_Fire);
}

/*-------------------//
//  POISON GAS BOMB  //
//-------------------*/
void Coven_PoisonGas_Fire (edict_t *ent)
{
	vec3_t	offset, start;
	vec3_t	forward, right;
	int		damage;
	float	damage_radius;
	int		radius_damage;

        damage = 30 + (int)(random() * 10.0);
        radius_damage = 40;
        damage_radius = 80;
	if (is_quad)
	{
		damage *= 4;
		radius_damage *= 4;
	}
        Coven_PowerupSfx (ent);

	AngleVectors (ent->client->v_angle, forward, right, NULL);

	VectorScale (forward, -2, ent->client->kick_origin);
	ent->client->kick_angles[0] = -1;

	VectorSet(offset, 8, 8, ent->viewheight-8);
	P_ProjectSource (ent->client, ent->s.origin, offset, forward, right, start);
        Coven_FireGasbomb (ent, start, forward, damage, SPEED_ROCKET, damage_radius, radius_damage, GAS_POISON);

	// send muzzle flash
	gi.WriteByte (svc_muzzleflash);
	gi.WriteShort (ent-g_edicts);
	gi.WriteByte (MZ_ROCKET | is_silenced);
	gi.multicast (ent->s.origin, MULTICAST_PVS);

	ent->client->ps.gunframe++;

	PlayerNoise(ent, start, PNOISE_WEAPON);

        if (! ( (int)dmflags->value & DF_INFINITE_AMMO ) )
                ent->client->pers.inventory[ent->client->ammo_index] -= ent->client->pers.weapon->quantity;
}

void Weapon_PoisonGasBomb (edict_t *ent)
{
	static int	pause_frames[]	= {25, 33, 42, 50, 0};
	static int	fire_frames[]	= {5, 0};

        Weapon_Generic (ent, 4, 12, 50, 54, pause_frames, fire_frames, Coven_PoisonGas_Fire);
}

/*--------------------//
//  SPORE CLOUD BOMB  //
//--------------------*/
void Coven_GermGas_Fire (edict_t *ent)
{
	vec3_t	offset, start;
	vec3_t	forward, right;
	int		damage;
	float	damage_radius;
	int		radius_damage;

        damage = 30 + (int)(random() * 10.0);
        radius_damage = 40;
        damage_radius = 80;
	if (is_quad)
	{
		damage *= 4;
		radius_damage *= 4;
	}
        Coven_PowerupSfx (ent);

	AngleVectors (ent->client->v_angle, forward, right, NULL);

	VectorScale (forward, -2, ent->client->kick_origin);
	ent->client->kick_angles[0] = -1;

	VectorSet(offset, 8, 8, ent->viewheight-8);
	P_ProjectSource (ent->client, ent->s.origin, offset, forward, right, start);
        Coven_FireGasbomb (ent, start, forward, damage, SPEED_ROCKET, damage_radius, radius_damage, GAS_BIO);

	// send muzzle flash
	gi.WriteByte (svc_muzzleflash);
	gi.WriteShort (ent-g_edicts);
	gi.WriteByte (MZ_ROCKET | is_silenced);
	gi.multicast (ent->s.origin, MULTICAST_PVS);

	ent->client->ps.gunframe++;

	PlayerNoise(ent, start, PNOISE_WEAPON);

        if (! ( (int)dmflags->value & DF_INFINITE_AMMO ) )
                ent->client->pers.inventory[ent->client->ammo_index] -= ent->client->pers.weapon->quantity;
}

void Weapon_SporeCloudBomb (edict_t *ent)
{
	static int	pause_frames[]	= {25, 33, 42, 50, 0};
	static int	fire_frames[]	= {5, 0};

        Weapon_Generic (ent, 4, 12, 50, 54, pause_frames, fire_frames, Coven_GermGas_Fire);
}

/*-----------------//
//  MIND GAS BOMB  //
//-----------------*/
void Coven_MindGas_Fire (edict_t *ent)
{
	vec3_t	offset, start;
	vec3_t	forward, right;
	int		damage;
	float	damage_radius;
	int		radius_damage;

        damage = 30 + (int)(random() * 10.0);
        radius_damage = 40;
        damage_radius = 80;
	if (is_quad)
	{
		damage *= 4;
		radius_damage *= 4;
	}
        Coven_PowerupSfx (ent);

	AngleVectors (ent->client->v_angle, forward, right, NULL);

	VectorScale (forward, -2, ent->client->kick_origin);
	ent->client->kick_angles[0] = -1;

	VectorSet(offset, 8, 8, ent->viewheight-8);
	P_ProjectSource (ent->client, ent->s.origin, offset, forward, right, start);
        Coven_FireGasbomb (ent, start, forward, damage, SPEED_ROCKET, damage_radius, radius_damage, GAS_MIND);

	// send muzzle flash
	gi.WriteByte (svc_muzzleflash);
	gi.WriteShort (ent-g_edicts);
	gi.WriteByte (MZ_ROCKET | is_silenced);
	gi.multicast (ent->s.origin, MULTICAST_PVS);

	ent->client->ps.gunframe++;

	PlayerNoise(ent, start, PNOISE_WEAPON);

        if (! ( (int)dmflags->value & DF_INFINITE_AMMO ) )
        {
                int     sec_index;
                gitem_t *sec_item;

                ent->client->pers.inventory[ent->client->ammo_index] -= ent->client->pers.weapon->quantity;

                // Consume secondary ammo.
                sec_item  = FindItem("Chem-Rockets");
                sec_index = ITEM_INDEX(sec_item);
                ent->client->pers.inventory[sec_index]--;
        }
}

void Weapon_MindGasBomb (edict_t *ent)
{
	static int	pause_frames[]	= {25, 33, 42, 50, 0};
	static int	fire_frames[]	= {5, 0};

	Weapon_Generic (ent, 4, 12, 50, 54, pause_frames, fire_frames, Coven_MindGas_Fire);
}

/*---------------//
//  NAPALM BOMB  //
//---------------*/
void Coven_NapalmRocket_Fire (edict_t *ent)
{
	vec3_t	offset, start;
	vec3_t	forward, right;
	int		damage;
	float	damage_radius;
	int		radius_damage;

	damage = 100 + (int)(random() * 20.0);
	radius_damage = 120;
	damage_radius = 120;
	if (is_quad)
	{
		damage *= 4;
		radius_damage *= 4;
	}
        Coven_PowerupSfx (ent);

	AngleVectors (ent->client->v_angle, forward, right, NULL);

	VectorScale (forward, -2, ent->client->kick_origin);
	ent->client->kick_angles[0] = -1;

	VectorSet(offset, 8, 8, ent->viewheight-8);
	P_ProjectSource (ent->client, ent->s.origin, offset, forward, right, start);
        Coven_FireNapalm (ent, start, forward, damage, SPEED_ROCKET, damage_radius, radius_damage);

	// send muzzle flash
	gi.WriteByte (svc_muzzleflash);
	gi.WriteShort (ent-g_edicts);
	gi.WriteByte (MZ_ROCKET | is_silenced);
	gi.multicast (ent->s.origin, MULTICAST_PVS);

	ent->client->ps.gunframe++;

	PlayerNoise(ent, start, PNOISE_WEAPON);

        if (! ( (int)dmflags->value & DF_INFINITE_AMMO ) )
        {
                int     sec_index;
                gitem_t *sec_item;

                ent->client->pers.inventory[ent->client->ammo_index] -= ent->client->pers.weapon->quantity;

                // Consume secondary ammo.
                sec_item  = FindItem("Blast-Rockets");
                sec_index = ITEM_INDEX(sec_item);
                ent->client->pers.inventory[sec_index]--;
        }
}

void Weapon_NapalmBomb (edict_t *ent)
{
	static int	pause_frames[]	= {25, 33, 42, 50, 0};
	static int	fire_frames[]	= {5, 0};

        Weapon_Generic (ent, 4, 12, 50, 54, pause_frames, fire_frames, Coven_NapalmRocket_Fire);
}

/*---------------------------------//
//  SMART BOMB (a.k.a. tech bomb)  //
//---------------------------------*/
void Coven_SmartBomb_Fire (edict_t *ent)
{
	vec3_t	offset, start;
	vec3_t	forward, right;
	int		damage;
	float	damage_radius;
	int		radius_damage;

	damage = 100 + (int)(random() * 20.0);
	radius_damage = 120;
	damage_radius = 120;
	if (is_quad)
	{
		damage *= 4;
		radius_damage *= 4;
	}
        Coven_PowerupSfx (ent);

	AngleVectors (ent->client->v_angle, forward, right, NULL);

	VectorScale (forward, -2, ent->client->kick_origin);
	ent->client->kick_angles[0] = -1;

	VectorSet(offset, 8, 8, ent->viewheight-8);
	P_ProjectSource (ent->client, ent->s.origin, offset, forward, right, start);
        Coven_FireSmart (ent, start, forward, damage, SPEED_HEAVYROCKET, damage_radius, radius_damage);

	// send muzzle flash
	gi.WriteByte (svc_muzzleflash);
	gi.WriteShort (ent-g_edicts);
	gi.WriteByte (MZ_ROCKET | is_silenced);
	gi.multicast (ent->s.origin, MULTICAST_PVS);

	ent->client->ps.gunframe++;

	PlayerNoise(ent, start, PNOISE_WEAPON);

        if (! ( (int)dmflags->value & DF_INFINITE_AMMO ) )
        {
                int     sec_index;
                gitem_t *sec_item;

                ent->client->pers.inventory[ent->client->ammo_index] -= ent->client->pers.weapon->quantity;

                // Consume secondary ammo.
                sec_item  = FindItem("Cells");
                sec_index = ITEM_INDEX(sec_item);
                ent->client->pers.inventory[sec_index] -= CELL_COST_SMART;
        }
}

void Weapon_SmartBomb (edict_t *ent)
{
	static int	pause_frames[]	= {25, 33, 42, 50, 0};
	static int	fire_frames[]	= {5, 0};

        Weapon_Generic (ent, 4, 12, 50, 54, pause_frames, fire_frames, Coven_SmartBomb_Fire);
}

/*----------------//
//  NUCLEAR BOMB  //
//----------------*/
void Coven_Nuke_Fire (edict_t *ent)
{
	vec3_t	offset, start;
	vec3_t	forward, right;
	int		damage;
	float	damage_radius;
	int		radius_damage;

	damage = 100 + (int)(random() * 20.0);
        radius_damage = NUKE_DAMAGE;
        damage_radius = NUKE_RADIUS;
	if (is_quad)
	{
		damage *= 4;
		radius_damage *= 4;
	}
        Coven_PowerupSfx (ent);

	AngleVectors (ent->client->v_angle, forward, right, NULL);

	VectorScale (forward, -2, ent->client->kick_origin);
	ent->client->kick_angles[0] = -1;

	VectorSet(offset, 8, 8, ent->viewheight-8);
	P_ProjectSource (ent->client, ent->s.origin, offset, forward, right, start);
        Coven_FireNuke (ent, start, forward, damage, SPEED_HEAVYROCKET, damage_radius, radius_damage);

	// send muzzle flash
	gi.WriteByte (svc_muzzleflash);
	gi.WriteShort (ent-g_edicts);
	gi.WriteByte (MZ_ROCKET | is_silenced);
	gi.multicast (ent->s.origin, MULTICAST_PVS);

	ent->client->ps.gunframe++;

	PlayerNoise(ent, start, PNOISE_WEAPON);

        if (! ( (int)dmflags->value & DF_INFINITE_AMMO ) )
                ent->client->pers.inventory[ent->client->ammo_index] -= ent->client->pers.weapon->quantity;
}

void Weapon_NuclearBomb (edict_t *ent)
{
	static int	pause_frames[]	= {25, 33, 42, 50, 0};
	static int	fire_frames[]	= {5, 0};

        Weapon_Generic (ent, 4, 12, 50, 54, pause_frames, fire_frames, Coven_Nuke_Fire);
}

/*-------------------//
//  BLACK HOLE BOMB  //
//-------------------*/
void Coven_BlackHole_Fire (edict_t *ent)
{
	vec3_t	offset, start;
	vec3_t	forward, right;
	int		damage;
	float	damage_radius;
	int		radius_damage;

	damage = 100 + (int)(random() * 20.0);
	radius_damage = 120;
	damage_radius = 120;
	if (is_quad)
	{
		damage *= 4;
		radius_damage *= 4;
	}
        Coven_PowerupSfx (ent);

	AngleVectors (ent->client->v_angle, forward, right, NULL);

	VectorScale (forward, -2, ent->client->kick_origin);
	ent->client->kick_angles[0] = -1;

	VectorSet(offset, 8, 8, ent->viewheight-8);
	P_ProjectSource (ent->client, ent->s.origin, offset, forward, right, start);
        Coven_FireHole (ent, start, forward, damage, SPEED_HEAVYROCKET, damage_radius, radius_damage);

	// send muzzle flash
	gi.WriteByte (svc_muzzleflash);
	gi.WriteShort (ent-g_edicts);
	gi.WriteByte (MZ_ROCKET | is_silenced);
	gi.multicast (ent->s.origin, MULTICAST_PVS);

	ent->client->ps.gunframe++;

	PlayerNoise(ent, start, PNOISE_WEAPON);

        if (! ( (int)dmflags->value & DF_INFINITE_AMMO ) )
        {
                int     sec_index;
                gitem_t *sec_item;

                ent->client->pers.inventory[ent->client->ammo_index] -= ent->client->pers.weapon->quantity;

                // Consume secondary ammo.
                sec_item  = FindItem("Cells");
                sec_index = ITEM_INDEX(sec_item);
                ent->client->pers.inventory[sec_index] -= CELL_COST_BLACK;
        }
}

void Weapon_BlackHoleBomb (edict_t *ent)
{
	static int	pause_frames[]	= {25, 33, 42, 50, 0};
	static int	fire_frames[]	= {5, 0};

        Weapon_Generic (ent, 4, 12, 50, 54, pause_frames, fire_frames, Coven_BlackHole_Fire);
}

/*========================================================================*/


/*
======================================================================

GRENADE

======================================================================
*/

#define GRENADE_TIMER		3.0
#define GRENADE_MINSPEED	400
#define GRENADE_MAXSPEED	800

void weapon_grenade_fire (edict_t *ent, qboolean held)
{
	vec3_t	offset;
	vec3_t	forward, right;
	vec3_t	start;
	int		damage = 125;
	float	timer;
	int		speed;
	float	radius;

	radius = damage+40;
	if (is_quad)
		damage *= 4;
        Coven_PowerupSfx (ent);

	VectorSet(offset, 8, 8, ent->viewheight-8);
	AngleVectors (ent->client->v_angle, forward, right, NULL);
	P_ProjectSource (ent->client, ent->s.origin, offset, forward, right, start);

	timer = ent->client->grenade_time - level.time;
	speed = GRENADE_MINSPEED + (GRENADE_TIMER - timer) * ((GRENADE_MAXSPEED - GRENADE_MINSPEED) / GRENADE_TIMER);
	fire_grenade2 (ent, start, forward, damage, speed, timer, radius, held);

	ent->client->grenade_time = level.time + 1.0;

        if (ent->deadflag || ent->s.modelindex != 255) // VWep animations screw up corpses
		return;

	if (ent->health <= 0)
		return;

	if (ent->client->ps.pmove.pm_flags & PMF_DUCKED)
	{
		ent->client->anim_priority = ANIM_ATTACK;
		ent->s.frame = FRAME_crattak1-1;
		ent->client->anim_end = FRAME_crattak3;
	}
	else
	{
		ent->client->anim_priority = ANIM_REVERSE;
		ent->s.frame = FRAME_wave08;
		ent->client->anim_end = FRAME_wave01;
	}
}

void Weapon_Grenade2 (edict_t *ent)
{
	if ((ent->client->newweapon) && (ent->client->weaponstate == WEAPON_READY))
	{
		ChangeWeapon (ent);
		return;
	}

	if (ent->client->weaponstate == WEAPON_ACTIVATING)
	{
		ent->client->weaponstate = WEAPON_READY;
		ent->client->ps.gunframe = 16;
		return;
	}

	if (ent->client->weaponstate == WEAPON_READY)
	{
// COVEN:  Switch weapons if no more grenades.
                if (ent->client->pers.inventory[ent->client->ammo_index] < 1)
                {       NoAmmoWeaponChange (ent);
                        return;
                }
// COVEN
		if ( ((ent->client->latched_buttons|ent->client->buttons) & BUTTON_ATTACK) )
		{
			ent->client->latched_buttons &= ~BUTTON_ATTACK;
			if (ent->client->pers.inventory[ent->client->ammo_index])
			{
				ent->client->ps.gunframe = 1;
				ent->client->weaponstate = WEAPON_FIRING;
				ent->client->grenade_time = 0;
			}
			else
			{
				if (level.time >= ent->pain_debounce_time)
				{
					gi.sound(ent, CHAN_VOICE, gi.soundindex("weapons/noammo.wav"), 1, ATTN_NORM, 0);
					ent->pain_debounce_time = level.time + 1;
				}
				NoAmmoWeaponChange (ent);
			}
			return;
		}

		if ((ent->client->ps.gunframe == 29) || (ent->client->ps.gunframe == 34) || (ent->client->ps.gunframe == 39) || (ent->client->ps.gunframe == 48))
		{
			if (rand()&15)
				return;
		}

		if (++ent->client->ps.gunframe > 48)
			ent->client->ps.gunframe = 16;
		return;
	}

	if (ent->client->weaponstate == WEAPON_FIRING)
	{
		if (ent->client->ps.gunframe == 5)
			gi.sound(ent, CHAN_WEAPON, gi.soundindex("weapons/hgrena1b.wav"), 1, ATTN_NORM, 0);

		if (ent->client->ps.gunframe == 11)
		{
			if (!ent->client->grenade_time)
			{
// COVEN:  Subtract ammo here.
                                if (! ( (int)dmflags->value & DF_INFINITE_AMMO ) )
                                        ent->client->pers.inventory[ent->client->ammo_index]--;
// COVEN
				ent->client->grenade_time = level.time + GRENADE_TIMER + 0.2;
				ent->client->weapon_sound = gi.soundindex("weapons/hgrenc1b.wav");
			}

			// they waited too long, detonate it in their hand
			if (!ent->client->grenade_blew_up && level.time >= ent->client->grenade_time)
			{
				ent->client->weapon_sound = 0;
// COVEN:  Flag 'blew_up' first so if player dies, only one grenade is fired.
				ent->client->grenade_blew_up = true;
				weapon_grenade_fire (ent, true);
// COVEN
			}

			if (ent->client->buttons & BUTTON_ATTACK)
				return;

			if (ent->client->grenade_blew_up)
			{
				if (level.time >= ent->client->grenade_time)
				{
					ent->client->ps.gunframe = 15;
					ent->client->grenade_blew_up = false;
				}
				else
				{
					return;
				}
			}
		}

		if (ent->client->ps.gunframe == 12)
		{
			ent->client->weapon_sound = 0;
			weapon_grenade_fire (ent, false);
		}

// COVEN:  Switch weapons if no more grenades.
                if (ent->client->ps.gunframe == 15)
                {
                        if (ent->client->pers.inventory[ent->client->ammo_index] < 1)
                        {       NoAmmoWeaponChange (ent);
                                ent->client->grenade_time = 0;
                                ent->client->weaponstate = WEAPON_READY;
                                return;
                        }
                        if (level.time < ent->client->grenade_time)
                                return;
                }
// COVEN

		ent->client->ps.gunframe++;

		if (ent->client->ps.gunframe == 16)
		{
			ent->client->grenade_time = 0;
			ent->client->weaponstate = WEAPON_READY;
		}
	}
}

void Weapon_Grenade (edict_t *ent)
{
	int oldstate = ent->client->weaponstate;

        // VWep animations screw up corpses
        if (ent->deadflag || ent->s.modelindex != 255)
                return;

        Weapon_Grenade2 (ent);

	// run the weapon frame again if hasted
        if (CTFApplyHaste (ent) && (oldstate == ent->client->weaponstate))
                Weapon_Grenade2 (ent);
}


/*
======================================================================

BLASTER

======================================================================
*/

void Blaster_Fire (edict_t *ent, vec3_t g_offset, int damage)
{
	vec3_t	forward, right;
	vec3_t	start;
	vec3_t	offset;

	if (is_quad)
		damage *= 4;
        Coven_PowerupSfx (ent);
	AngleVectors (ent->client->v_angle, forward, right, NULL);
        VectorSet(offset, 8, 8, ent->viewheight-8);
	VectorAdd (offset, g_offset, offset);
	P_ProjectSource (ent->client, ent->s.origin, offset, forward, right, start);

	VectorScale (forward, -2, ent->client->kick_origin);
	ent->client->kick_angles[0] = -1;

        fire_blaster (ent, start, forward, damage, 1000, EF_BLASTER, false);

	// send muzzle flash
	gi.WriteByte (svc_muzzleflash);
	gi.WriteShort (ent-g_edicts);
        gi.WriteByte (MZ_BLASTER | is_silenced);
	gi.multicast (ent->s.origin, MULTICAST_PVS);

	PlayerNoise(ent, start, PNOISE_WEAPON);
}

void Weapon_Blaster_Fire (edict_t *ent)
{
        int     damage;

	if (deathmatch->value)
		damage = 15;
	else
		damage = 10;
        Blaster_Fire (ent, vec3_origin, damage);
	ent->client->ps.gunframe++;
}

void Weapon_Blaster (edict_t *ent)
{
	static int	pause_frames[]	= {19, 32, 0};
	static int	fire_frames[]	= {5, 0};

	Weapon_Generic (ent, 4, 8, 52, 55, pause_frames, fire_frames, Weapon_Blaster_Fire);
}


//======================================================================
