/*==========================================================================
//  x_team.c -- by various                      Last updated:  4-19-1999
//--------------------------------------------------------------------------
//  This file contains code that regulates various teams.
//========================================================================*/

#include "g_local.h"
#include "x_team.h"


/*======================/  General Team Functions  /======================*/

/*----------------------------------------------------/ New Code /--------//
//  This checks if the two entities 'targ' and 'attacker' are on
//  the same team.  For that to be true, the following conditions
//  must be met:
//
//  * Game must be deathmatch.  (duh!)
//  * Teams must be enabled.  (duh!)
//  * Both entities must be battlesuits.  (i.e., absolutely NO PPMs!)
//  * Both entities are non-gray and are the same color.
//------------------------------------------------------------------------*/
qboolean Coven_OnSameSide (edict_t *targ, edict_t *attacker)
{
// We could check for coop here, but AB is a DM mod so we don't.

// Check for teams.
        if (!deathmatch->value || (ab_teams < 2))
                return false;

// Attacker must be a client.
        if (!attacker->client)
                return false;

// Attacker must be on a team.
        if (!attacker->client->resp.suitcolor)
                return false;

// If colors match, both are the same alignment.
        if (targ->client)
        {       // Target is a player.
                if (attacker->client->resp.suitcolor == targ->client->resp.suitcolor)
                        return true;
        }
        else
        {       // Target may be a dead body of a former player.
                if (attacker->client->resp.suitcolor == targ->style)
                        return true;
        }

// Target is an enemy!
        return false;
}

/*----------------------------------------------------/ New Code /--------//
//  Given a number, this returns the name of a team.
//------------------------------------------------------------------------*/
char *Coven_TeamName (int team)
{
        switch (team)
        {
        case SKIN_BLUE:
		return "BLUE";
        case SKIN_RED:
                return "RED";
        case SKIN_GREEN:
                return "GREEN";
        case SKIN_GOLD:
                return "GOLD";
        case SKIN_PURPLE:
                return "PURPLE";
	}
        return "UNKNOWN";
}

/*----------------------------------------------------/ New Code /--------//
//  Find the team with the most points, and declare it the winner.
//------------------------------------------------------------------------*/
void Coven_WinningTeam (void)
{
        gclient_t       *cl;
        int     teamfrags[MAX_AB_SKINS];
        int     hiscore = 0;
        int     winner = 0;
        int     i;

/* Check if enough teams are available. */
        if (ab_teams < 2)
                return;

/* Initialize skin/team array. */
        for (i = 0; i < MAX_AB_SKINS; i++)
                teamfrags[i] = 0;

/* Count frags. */
        for (i=0 ; i<maxclients->value ; i++)
        {
                cl = game.clients + i;
                if (!g_edicts[i+1].inuse)
                        continue;

                teamfrags[cl->resp.suitcolor] += cl->resp.score;
        }

/* Find winning team. */
        for (i = 1; i < MAX_AB_SKINS; i++)
        {
                if (teamfrags[i] > hiscore)
                {       hiscore = teamfrags[i];
                        winner = i;
                }
                else if (teamfrags[i] == hiscore)
                        winner = 0;
        }

        if (winner)
        {       winning_team = winner;
                gi.bprintf (PRINT_HIGH, "%s team wins!\n", Coven_TeamName(winner));
        }
}


/*============================/  Force Team  /============================*/

/*----------------------------------------------------/ New Code /--------//
//  Find the team with the least number of players and return its
//  number.  Ties go to the lowest number.  If all teams are full,
//  zero (no team) is returned.
//
//  Originally written by KBM; modified by PM.
//------------------------------------------------------------------------*/
int Coven_AssignToTeam (int *team_players, int team_no)
{
        int less_team = 0, min_players = 999, t;
        int team_size, temp_teams[MAX_AB_SKINS];
        int team_count;   // PM:  Max possible teams.
        int max_teams = (MAX_AB_SKINS - 1);

	if (!team_players) {
		for (t=0; t<maxclients->value; t++)
			if (g_edicts[t+1].inuse)
                                temp_teams[g_edicts[t+1].client->resp.suitcolor]++;
		team_players = &temp_teams[0];
	}

	// team_size of 0 is no team-size limit, all clients play
	// if team_size != 0, then extra clients observe and wait for a turn
	team_size = AB_teamsize->value;
        if (team_size < 1)
		team_size = 999;

        // PM:  Safety check -- do NOT exceed maximum number of teams.
        team_count = ab_teams;
        if (team_count > max_teams)
                team_count = max_teams;
        team_count++;

        // Find team with least number of players.
        for (t=1; t<team_count; t++)
        {       if (team_players[t] < min_players)
                {       min_players = team_players[t];
			less_team = t;
		}
	}

        // Add new player to team if it is not full.
	if (team_players[less_team] < team_size)
		return less_team;
	else
		return 0;
}

/*----------------------------------------------------/ New Code /--------//
//  This forces renegade gray players into some team if possible.
//------------------------------------------------------------------------*/
void Coven_ForceTeam (edict_t *ent, int teamindex[MAX_AB_SKINS])
{
        int     team;

/* Don't bother if not enough teams. */
        if (ab_teams < 2)
                return;

/* Not a client?  Abort. */
        if (!ent->client)
                return;

/* Don't change teams if already aligned. */
        if (ent->client->resp.suitcolor)
                return;

/* Change team if possible. */
        team = Coven_AssignToTeam(teamindex, ent->client->resp.suitcolor);
        if (team != ent->client->resp.suitcolor)
        {       teamindex[ent->client->resp.suitcolor]--;
                teamindex[team]++;
                ent->client->resp.suitcolor = team;
                if ( (Coven_IsPlayer (ent)) && (ent->battlesuit) )
                        ent->s.skinnum = ent->client->resp.suitcolor;
                if (team)
                {
                        if ((int)(dmflags->value) & DF_LESS_MESSAGES)
                                gi.cprintf (ent, PRINT_HIGH, "You have been assigned to team %s.\n", Coven_TeamName(team));
                        else
                                gi.bprintf (PRINT_HIGH, "%s has been assigned to team %s.\n", ent->client->pers.netname, Coven_TeamName(team));
                }
        }
}

/*----------------------------------------------------/ New Code /--------//
//  First, this checks if the player's team skin is legal.  If not,
//  force the player into one of the existing teams if possible.
//------------------------------------------------------------------------*/
void Coven_CheckTeamSkin (edict_t *ent)
{
/* Do nothing if not enough teams. */
        if (ab_teams < 2)
                return;

/* Not a client?  Abort. */
        if (!ent->client)
                return;

/* Neutralize team if invalid. */
        if (ent->client->resp.suitcolor > ab_teams)
        {       ent->client->resp.suitcolor = 0;
                gi.cprintf (ent, PRINT_HIGH, "Skin changed to default.\n");
        }

/* Change skin/team if neutral. */
        if (!ent->client->resp.suitcolor)
        {
                int     i;
                int     teamindex[MAX_AB_SKINS];

                /* Initialize skin/team array. */
                for (i = 0; i < MAX_AB_SKINS; i++)
                        teamindex[i] = 0;

                /* Find out how many players are in each team. */
                for (i=0; i<maxclients->value; i++)
                        if (g_edicts[i+1].inuse && g_edicts[i+1].client)
                                teamindex[g_edicts[i+1].client->resp.suitcolor]++;

                /* Assign player to some team if possible. */
                Coven_ForceTeam (ent, teamindex);
        }
}

/*----------------------------------------------------/ New Code /--------//
//  This forces a player to join in on the team with the least player.
//  This is called when forcejoin is on and player is joining in.
//  Unlike other force team functions above, this provides no messages.
//------------------------------------------------------------------------*/
void Coven_StartingTeam (edict_t *ent)
{
        int     i;
        int     teamindex[MAX_AB_SKINS];
        int     team;

/* Do nothing if not enough teams. */
        if (ab_teams < 2)
                return;

/* Not a client?  Abort. */
        if (!ent->client)
                return;

/* Neutralize team. */
        ent->client->resp.suitcolor = 0;

/* Players who join server during intermission are neutrals. */
        if (level.intermissiontime)
                return;

/* Change skin/team. */
        /* Initialize skin/team array. */
        for (i = 0; i < MAX_AB_SKINS; i++)
                teamindex[i] = 0;

        /* Find out how many players are in each team. */
        for (i=0; i<maxclients->value; i++)
                if (g_edicts[i+1].inuse && g_edicts[i+1].client)
                        teamindex[g_edicts[i+1].client->resp.suitcolor]++;

        /* Assign player to some team if possible. */
        team = Coven_AssignToTeam (teamindex, ent->client->resp.suitcolor);
        if (team != ent->client->resp.suitcolor)
        {       teamindex[ent->client->resp.suitcolor]--;
                teamindex[team]++;
                ent->client->resp.suitcolor = team;
                if ( (Coven_IsPlayer (ent)) && (ent->battlesuit) )
                        ent->s.skinnum = ent->client->resp.suitcolor;
        }
}


/*========================/  Team Size Checking  /========================*/

/*----------------------------------------------------/ New Code /--------//
//  This displays all the teams available to the player.
//------------------------------------------------------------------------*/
int Coven_OpenTeams (edict_t *ent)
{
        int     i;
        int     teamindex[MAX_AB_SKINS];
        int     open = 0;
        int     last1 = 0;
        int     last2 = 0;
        char    text[256];   /* Only 5 teams... */

/* Ignore if no teams. */
        if (ab_teams < 2)
                return -1;

/* Ignore if no team limit. */
        if (!AB_teamsize->value)
                return -1;

/* Not a client?  Abort. */
        if (!ent->client)
                return -1;

/* Initialize skin/team array. */
        for (i = 0; i < MAX_AB_SKINS; i++)
                teamindex[i] = 0;

/* Find out how many players are in each team. */
        for (i = 0; i < maxclients->value; i++)
                if (g_edicts[i+1].inuse && g_edicts[i+1].client)
                        teamindex[g_edicts[i+1].client->resp.suitcolor]++;

/* Check if the team is full. */
        for (i = 1; i <= ab_teams; i++)
        {       if (teamindex[i] < AB_teamsize->value)
                {       open++;
                        last2 = last1;
                        last1 = i;
                }
        }

        switch (open)
	{
        case 0:
                gi.cprintf (ent, PRINT_HIGH, "All teams are full.\n");
		break;
        case 1:
                gi.cprintf (ent, PRINT_HIGH, "Team %s is open.\n", Coven_TeamName(last1));
		break;
        case 2:
                gi.cprintf (ent, PRINT_HIGH, "Teams %s and %s are open.\n", Coven_TeamName(last2), Coven_TeamName(last1));
		break;
	default:
                strcpy (text, "Teams ");
                for (i = 1; i < last2; i++)
                {       if (teamindex[i] < AB_teamsize->value)
                        {       strcat (text, Coven_TeamName(i));
                                strcat (text, ", ");
                        }
                }
                gi.cprintf (ent, PRINT_HIGH, "%s%s, and %s are open.\n", text, Coven_TeamName(last2), Coven_TeamName(last1));
		break;
	}

        return open;
}

/*----------------------------------------------------/ New Code /--------//
//  This checks if the team given is full.
//------------------------------------------------------------------------*/
qboolean Coven_TeamFull (int teamskin)
{
        int     i;
        int     teamindex[MAX_AB_SKINS];

/* Ignore if no teams. */
        if (ab_teams < 2)
                return false;

/* If skin is neutral, quit. */
        if (!teamskin)
                return false;

/* Ignore if no team limit. */
        if (!AB_teamsize->value)
                return false;

/* Initialize skin/team array. */
        for (i = 0; i < MAX_AB_SKINS; i++)
                teamindex[i] = 0;

/* Find out how many players are in each team. */
        for (i = 0; i < maxclients->value; i++)
                if (g_edicts[i+1].inuse && g_edicts[i+1].client)
                        teamindex[g_edicts[i+1].client->resp.suitcolor]++;

/* Check if the team is full. */
        if (teamindex[teamskin] >= AB_teamsize->value)
                return true;

        return false;
}

/*----------------------------------------------------/ New Code /--------//
//  This checks if all the teams in ArmourBack are full.
//------------------------------------------------------------------------*/
qboolean Coven_AllFull (void)
{
        int     i;
        int     teamindex[MAX_AB_SKINS];

/* Ignore if no teams. */
        if (ab_teams < 2)
                return false;

/* Ignore if no team limit. */
        if (!AB_teamsize->value)
                return false;

/* Initialize skin/team array. */
        for (i = 0; i < MAX_AB_SKINS; i++)
                teamindex[i] = 0;

/* Find out how many players are in each team. */
        for (i = 0; i < maxclients->value; i++)
                if (g_edicts[i+1].inuse && g_edicts[i+1].client)
                        teamindex[g_edicts[i+1].client->resp.suitcolor]++;

/* Check if the team is full. */
        for (i = 1; i <= ab_teams; i++)
        {       if (teamindex[i] < AB_teamsize->value)
                        return false;
        }

/* All teams full. */
        return true;
}


/*==========================/  Team Selection  /==========================*/

/*----------------------------------------------------/ New Code /--------//
//  This command allows a player to change skins.  If teams are
//  enabled, the switch is allowed only if the player is an observer,
//  and not waiting in line.  Switching teams resets the player's score.
//------------------------------------------------------------------------*/
void Coven_SetSuitSkin (edict_t *ent)
{
        int     team;
        char    *color;

/* Not a client?  Abort. */
        if (!ent->client)
                return;

/* Only in DM. */
        if (!deathmatch->value)
                return;

/* Show player skin. */
        if (gi.argc() == 1)
        {       if (ab_teams >= 2)
                {       if (ent->client->resp.suitcolor)
                                gi.cprintf (ent, PRINT_HIGH, "You are on team %s.\n", Coven_TeamName(ent->client->resp.suitcolor));
                        else
                                gi.cprintf (ent, PRINT_HIGH, "You are not on any team.\n");
                }
                else
                {       if (ent->client->resp.suitcolor)
                                gi.cprintf (ent, PRINT_HIGH, "Your color is %s.\n", Coven_TeamName(ent->client->resp.suitcolor));
                        else
                                gi.cprintf (ent, PRINT_HIGH, "Your color is GRAY.\n");
                }
                return;
        }

/* Get a number for the player's input. */
        team = floor (atoi (gi.argv(1)));

/* If player gave text instead of a number (or the number given was zero),
   convert text to team number.
*/
        if (!team)
        {
                color = gi.args();

                if (Q_stricmp(color, "blue") == 0)
                        team = SKIN_BLUE;
                else if (Q_stricmp(color, "red") == 0)
                        team = SKIN_RED;
                else if (Q_stricmp(color, "green") == 0)
                        team = SKIN_GREEN;
                else if (Q_stricmp(color, "gold") == 0)
                        team = SKIN_GOLD;
                else if (Q_stricmp(color, "purple") == 0)
                        team = SKIN_PURPLE;
                else
                        team = SKIN_GRAY;
	}

/* Check new color. */
        switch (team)
	{
	case 1:
	case 2:
	case 3:
	case 4:
        case 5:
		break;
	default:
                team = 0;
		break;
	}

/* Check for teams. */
        if (ab_teams >= 2)
        {
                if ((int)(dmflags->value) & DF_TEAM_LOCK)
                {       if (ent->client->resp.suitcolor)
                        {       gi.cprintf (ent, PRINT_HIGH, "Can't -- not allowed to change teams.\n");
                                return;
                        }
                }
                if (ent->client->resp.suitcolor)
                {       // NOTE:  This is here for flood protection.
                        gi.cprintf (ent, PRINT_HIGH, "Can't -- use menu to change teams.\n");
                        return;
                }
                if (!ent->client->resp.spectator || ent->RA_quepos)
                {       if (ent->RA_quepos)
                                gi.cprintf (ent, PRINT_HIGH, "Can't -- you need to get out of line.\n");
                        else
                                gi.cprintf (ent, PRINT_HIGH, "Can't -- you must become a spectator first.\n");
                        return;
                }
                if (!team)
                {       gi.cprintf (ent, PRINT_HIGH, "Not a team.\n");
                        return;
                }
                if (ab_teams < team)
                {       gi.cprintf (ent, PRINT_HIGH, "Team %s is not available.\n", Coven_TeamName(team));
                        return;
                }
                if (ent->client->resp.suitcolor == team)
                {       gi.cprintf (ent, PRINT_HIGH, "You are already on team %s.\n", Coven_TeamName(team));
                        return;
                }
                if (Coven_TeamFull(team))
                {       gi.cprintf (ent, PRINT_HIGH, "Team %s is full.\n", Coven_TeamName(team));
                        return;
                }

                /* Let everyone know what team 'ent' is on. */
                if ( team && (!((int)(dmflags->value) & DF_LESS_MESSAGES)) )
                {       if (ent->client->resp.suitcolor)
                                gi.bprintf (PRINT_HIGH, "%s changed to team %s.\n", ent->client->pers.netname, Coven_TeamName(team));
                        else
                                gi.bprintf (PRINT_HIGH, "%s joined team %s.\n", ent->client->pers.netname, Coven_TeamName(team));
                }

                /* Reset score. */
                if (ent->client->resp.score)
                {       ent->client->resp.score = 0;
                        gi.cprintf (ent, PRINT_HIGH, "Your score has been reset to zero.\n");
                }
        }

/* Select new color. */
        ent->client->resp.suitcolor = team;

/* Change model skin if possible. */
        if (Coven_IsPlayer (ent))
                ent->s.skinnum = ent->client->resp.suitcolor;
}

/*----------------------------------------------------/ New Code /--------//
//  This changes the player's team/skin.  If teams are enabled,
//  score is reset to zero if player changes teams.
//------------------------------------------------------------------------*/
void Coven_MenuTeamChange (edict_t *ent, int teamskin)
{
/* Check for teams. */
        if (ab_teams >= 2)
        {       if (ent->client->resp.suitcolor != teamskin)
                {       /* Tell everyone who joined what. */
                        if ( teamskin && (!((int)(dmflags->value) & DF_LESS_MESSAGES)) )
                        {       if (ent->client->resp.suitcolor)
                                        gi.bprintf (PRINT_HIGH, "%s changed to team %s.\n", ent->client->pers.netname, Coven_TeamName(teamskin));
                                else
                                        gi.bprintf (PRINT_HIGH, "%s joined team %s.\n", ent->client->pers.netname, Coven_TeamName(teamskin));
                        }

                        /* Reset score. */
                        if (ent->client->resp.score)
                        {       ent->client->resp.score = 0;
                                gi.cprintf (ent, PRINT_HIGH, "Your score has been reset to zero.\n");
                        }
                }
        }

/* Set new color. */
        ent->client->resp.suitcolor = teamskin;
}


/*===========================/  END OF FILE  /===========================*/
