//===================================================
// Coven RA Rules for Disposable Heroes MOD
// Copyright 1998, Kyle Mallory. All rights reserved.
//
// Use without express written permission is strictly
// prohibited and punishable for local and
// international copyright laws.
//===================================================

//===================================================
// *Heavily* modified by Patrick Martin (PM).
// Last updated:  4-20-1999
//===================================================

#include "g_local.h"
#include "x_rules.h"
#include "x_team.h"

qboolean        jetpack_on;
qboolean        alt_armor;
qboolean        secret_on;

int             ab_arms;
int             ab_rules;
int             ab_teams;

qboolean        exitnow;
float		game_nofire;
int             rune_scale;
int             winning_team;
edict_t		*RA_PlayerQue;
cvar_t		*AB_teamsize;
cvar_t          *AB_maxdelay;

static cvar_t   *AB_arms;
static cvar_t   *AB_rules;
static cvar_t   *AB_numteams;
static cvar_t   *AB_maxrunes;

static float    countdown;

static int      survival_bonus;
static int      capture_bonus;


//=========================================================================
// INITIALIZATION

/*----------------------------------------------------/ New Code /--------//
//  This initializes new ArmourBack console variables.
//------------------------------------------------------------------------*/
void Coven_AB_CvarInit (void)
{
        AB_arms = gi.cvar ("arms", "0", CVAR_SERVERINFO);
        AB_rules = gi.cvar ("rules", "0", CVAR_SERVERINFO);
        AB_teamsize = gi.cvar ("teamsize", "0", CVAR_SERVERINFO|CVAR_LATCH);
        AB_numteams = gi.cvar ("numteams", "0", CVAR_SERVERINFO);
        AB_maxrunes = gi.cvar ("maxrunes", "20", CVAR_SERVERINFO|CVAR_LATCH);
        AB_maxdelay = gi.cvar ("maxdelay", "30", CVAR_SERVERINFO|CVAR_LATCH);
}

/*----------------------------------------------------/ New Code /--------//
//  This checks if the cvar values are legit.
//------------------------------------------------------------------------*/
void Coven_AB_CvarCheck (void)
{
        float   var;

// PM:  Make sure weapons are set properly.
        var = floor (AB_arms->value);

        if ((var != ARMS_NORMAL) && (var != ARMS_JOUST) && (var != ARMS_ALL))
                var = 0;

        if (AB_arms->value != var)
                gi.cvar_set ("arms", va("%d", (int)var));

// PM:  Make sure rules are set properly.
        var = floor (AB_rules->value);

        if ((var != RULES_DUEL) && (var != RULES_ELIMINATION) &&
            (var != RULES_CAPTURE) && (var != RULES_CTF))
                var = 0;

        if (AB_rules->value != var)
                gi.cvar_set ("rules", va("%d", (int)var));

// PM:  Do NOT exceed number of possible teams and don't use fractions.
        var = floor (AB_numteams->value);

        if ((AB_rules->value == RULES_CTF) && (var != 2))
        {       // In CTF, teams are always set to two.
                gi.dprintf("Number of teams is set to 2.\n");
                var = 2;
        }

        if (var > 5)
                var = 5;
        else if (var < 2)
                var = 0;

        if (AB_numteams->value != var)
                gi.cvar_set ("numteams", va("%d", (int)var));

// PM:  Check teamsize limit.
        var = floor (AB_teamsize->value);

        if (var < 1)
                var = 0;

        if (AB_teamsize->value != var)
                gi.cvar_set ("teamsize", va("%d", (int)var));

// PM:  Do NOT allow extreme values and don't use fractions.
        var = floor (AB_maxrunes->value);

        if (var > 100)
                var = 100;
        else if (var < 1)
                var = 0;

        if (AB_maxrunes->value != var)
                gi.cvar_set ("maxrunes", va("%d", (int)var));

// PM:  Do NOT allow extreme delays.
        if (AB_maxdelay->value > 60)
                gi.cvar_set ("maxdelay", "60");
        else if (AB_maxdelay->value < 15)
                gi.cvar_set ("maxdelay", "15");
}

/*----------------------------------------------------/ New Code /--------//
//  This initializes new ArmourBack global variables.
//------------------------------------------------------------------------*/
void ArmourBack_Init (void)
{
        exitnow = false;

	RA_PlayerQue = NULL;
        countdown = 5;
        rune_scale = 5;
        game_nofire = 0;
        winning_team = 0;

        survival_bonus = 0;
        capture_bonus = 0;

        if ((int)(dmflags->value) & DF_NO_JETPACK)
                jetpack_on = false;
        else
                jetpack_on = true;

        if ((int)(dmflags->value) & DF_ALTERNATE_ARMOR)
                alt_armor = true;
        else
                alt_armor = false;

        if ((int)(dmflags->value) & DF_SECRET)
                secret_on = true;
        else
                secret_on = false;

        Coven_AB_CvarCheck ();

        ab_arms  = AB_arms->value;
        ab_rules = AB_rules->value;
        ab_teams = AB_numteams->value;
}


/*=======================================================================*/
// NOFIRE CHECKS

/*----------------------------------------------------/ New Code /--------//
//  This kills all the flames on players, and sever the cables of
//  all the players' grapples.
//------------------------------------------------------------------------*/
void Coven_KillInfection (void)
{
        edict_t *ent;
        int     i;

        for (i=0; i<maxclients->value; i++)
        {
                if (g_edicts[i+1].inuse && g_edicts[i+1].client)
                {
                        ent = &g_edicts[i+1];

                        // Snuff out fires.
                        if (ent->burnout)
                                ent->burnout = 0;

                        // Sever any grapple cables.
                        if (ent->hook)
                                ent->hook = NULL;
                }
        }
}

/*----------------------------------------------------/ New Code /--------//
//  Check if player can fire weapon.
//------------------------------------------------------------------------*/
qboolean Coven_NoFire (void)
{
        if (deathmatch->value)
                if (Coven_ArenaOn() || (ab_rules == RULES_CAPTURE))
                        if (game_nofire > level.time)
                                return true;

        return false;
}


/*=======================================================================*/
// RA RULES

//==================================================
// PM:  Check if arena is on.
//
qboolean Coven_ArenaOn (void)
{
        if (ab_rules == RULES_DUEL)
                return true;
        if (ab_rules == RULES_ELIMINATION)
                return true;

        return false;
}

//==================================================
// PM:  Update the players' position in line.
//
void Coven_QuePosition (void)
{
        edict_t *ent;
        int     t;

	// Set players que position for HUD.
        if (ab_rules == RULES_ELIMINATION)
        {       // Order is irrelevent in elimination.
                for (ent = RA_PlayerQue; ent != NULL; ent = ent->RA_nextinque)
                        ent->RA_quepos = 1;
                return;
        }

        if (ab_teams < 2)
        {       // Players in que.
                for (t = 1, ent = RA_PlayerQue; ent != NULL; ent = ent->RA_nextinque, t++)
                        ent->RA_quepos = t;
        }
        else
        {       // Teams in que.
                int     team = -1;

                for (t = 0, ent = RA_PlayerQue; ent != NULL; ent = ent->RA_nextinque)
                {       if (ent->client->resp.suitcolor != team)
                        {       // Advance team que position.
                                team = ent->client->resp.suitcolor;
                                t++;
                        }
                        ent->RA_quepos = t;
                }
        }
}

//==================================================
// PM:  This removes a player from the que.
//      Then the que position for everyone in
//      line is updated.
//
void Coven_RemoveFromPlayerQue (edict_t *ent)
{
        edict_t *prev, *last;

	// Remove player from the PlayerQue
        if (Coven_ArenaOn ()) {
		prev = last = RA_PlayerQue;
		while ((last != NULL) && (last != ent)) {
			prev = last;
			last = last->RA_nextinque;
		}
		if (last == ent) {
			if (last != RA_PlayerQue) prev->RA_nextinque = ent->RA_nextinque;
			else last = RA_PlayerQue = ent->RA_nextinque;
			ent->RA_nextinque = NULL;
                        ent->RA_quepos = 0;
		}

                // PM:  Update arena que.
                Coven_QuePosition ();
	}
}

//==================================================
// RA_nextinque == NULL implies a new player,
// or someone who has just come from playing.
// Either way, we move them to the end of the que
// Anyone on the que *should* be an observer
//
// (PM:  Update)  In team arena, teams wait in
// line.  Individual players entering the que are
// placed behind the last teammate, not necessarily
// at the end of the que.
//
void Coven_AddToPlayerQue (edict_t *ent)
{
        edict_t *last;

        // Do nothing if no player is here.
        if (!ent || !ent->client)
                return;

	// See if client is already in the que
	for (last = RA_PlayerQue; (last != NULL) && (last != ent); last = last->RA_nextinque);
        if (!last)
        {       // Player is not in the que yet, so put player in.
                if (ab_teams < 2)
                {       // Find last entry in que.
                        for (last = RA_PlayerQue; (last != NULL) && (last->RA_nextinque != NULL); last = last->RA_nextinque);

                        if (last == NULL) last = RA_PlayerQue = ent;
                        else last->RA_nextinque = ent;

                        ent->RA_nextinque = NULL;
                }
                else
                {       // Find last teammate in que.
                        for (last = RA_PlayerQue; (last != NULL) && (last->RA_nextinque != NULL); last = last->RA_nextinque)
                        {
                                if ((last->client->resp.suitcolor == ent->client->resp.suitcolor) &&
                                    (last->RA_nextinque->client->resp.suitcolor != ent->client->resp.suitcolor))
                                {       // Place player between last teammate and opponent.
                                        ent->RA_nextinque = last->RA_nextinque;
                                        last->RA_nextinque = ent;
                                        Coven_QuePosition ();
                                        return;
                                }
                        }

                        // No teammate was found in que.
                        if (last == NULL) last = RA_PlayerQue = ent;
                        else last->RA_nextinque = ent;

                        ent->RA_nextinque = NULL;
                }

                // PM:  Update arena que.
                Coven_QuePosition ();
	}
}

//==================================================
// Remove the client from the que, and respawn them
// into the game as a player.
//
void Coven_PutPlayerInGame (edict_t *ent)
{
        // Remove player from Que.
        Coven_RemoveFromPlayerQue (ent);

        // Prepare player for entry.
	ent->RA_nextinque = NULL;
	ent->RA_playing = true;

        // Trigger spectator respawn.
        ent->client->pers.spectator = false;
}

//==================================================
// Reset elimination bonus.
//
void Coven_ResetEliminationBonus (void)
{
        edict_t *ent;
        int     i;

        survival_bonus = 0;

        for (i=0; i<maxclients->value; i++)
        {
                if (g_edicts[i+1].inuse && g_edicts[i+1].client)
                {
                        ent = &g_edicts[i+1];
                        if (ent->RA_playing)
                                survival_bonus++;
                }
        }
}

//==================================================
// Give elimination bonus.
//
void Coven_GiveEliminationBonus
(edict_t *survivor, int winteam, int winners)
{
        edict_t *ent;
        int     bonus = survival_bonus;
        int     t;

        if (ab_teams < 2)
        {       // PM:  There can only be one.
                if (survivor)
                        gi.bprintf (PRINT_HIGH, "%s survived.\n", survivor->client->pers.netname);
        }
        else if (ab_teams >= 2)
        {       // PM:  Show which team survived.
                if (winteam && winners)
                {       // PM:  Teams bonus is divided among its surviving members.
                        gi.bprintf (PRINT_HIGH, "%s team survived.\n", Coven_TeamName(winteam));
                        bonus = bonus / winners;
                        if (bonus < 1)
                                bonus = 1;
                }
        }

        // PM:  Give points to all survivors.
        for (t=0; t<maxclients->value; t++)
        {       if (g_edicts[t+1].inuse && g_edicts[t+1].client)
                {
                        ent = &g_edicts[t+1];
                        if (ent->RA_playing)
                        {
                                ent->client->resp.score += bonus;
                                gi.cprintf (ent, PRINT_HIGH, "Survival bonus:  %d\n", bonus);
                        }
                }
        }
}

/*
=================
PrepNextRAMatch
=================
*/
void Coven_PrepNextRAMatch (int ta_players, int team1, int team2)
{
        edict_t  *ent;
        qboolean newplayer = true;

        if (ab_rules == RULES_ELIMINATION)
        {       // Elimination.
                survival_bonus = ta_players;
                while (newplayer)
                {       // Move all players from the que and into the arena.
                        newplayer = false;
                        ent = RA_PlayerQue;
                        if (ent != NULL)
                        {       Coven_PutPlayerInGame (ent);
                                newplayer = true;
                                survival_bonus++;
                        }
		}
	}
        else if (ab_teams < 2)
        {       // 1-on-1.
                while (newplayer && (ta_players < 2))
                {       // Move player from the que and into the arena.
                        // Always take players from the front of the que.
                        newplayer = false;
                        ent = RA_PlayerQue;
                        if (ent != NULL)
                        {       ta_players++;
                                Coven_PutPlayerInGame (ent);
                                gi.bprintf (PRINT_HIGH, "%s entered the arena\n", ent->client->pers.netname);
                                newplayer = true;
                        }
                }
        }
        else
        {       // 'team1' vs. 'team2'.
                while (newplayer)
                {       // Move players from two teams from the que and into the arena.
                        newplayer = false;
                        ent = RA_PlayerQue;
                        while (ent != NULL)
                        {       if ((ent->client->resp.suitcolor == team1) || (ent->client->resp.suitcolor == team2))
                                {       Coven_PutPlayerInGame (ent);
                                        newplayer = true;
                                        break;
                                }
                                ent = ent->RA_nextinque;
                        }
                }
                gi.bprintf (PRINT_HIGH, "Team %s vs. Team %s\n", Coven_TeamName(team1), Coven_TeamName(team2));
        }
}

/*
=================
CheckRARules
=================
*/
void Coven_CheckRARules (void)
{
        edict_t *ent;
        edict_t *survivor;
        int     t, numclients = 0, ta_players = 0;
        int     RA_all[MAX_AB_SKINS], RA_players[MAX_AB_SKINS], RA_teams = 0;
        int     winteam = 0, team1 = 0, team2 = 0;

        if (!Coven_ArenaOn ())
		return;

	// No players in the waiting que - usually means empty game
	if (RA_PlayerQue == NULL)
		return;

        // Can't start a match if game has not started yet.
        if (game_nofire > level.time)
                return;

        for (t = 0; t < MAX_AB_SKINS; t++)
        {       RA_all[t] = 0;
                RA_players[t] = 0;
        }
        survivor = NULL;

        // First, count the number of active players for each team.
        // PM:  Also count every player in the game.
        for (t=0; t<maxclients->value; t++)
        {       if (g_edicts[t+1].inuse && g_edicts[t+1].client)
                {       numclients++;
			ent = &g_edicts[t+1];
                        RA_all[ent->client->resp.suitcolor]++;
                        if (ent->RA_playing)
                        {       RA_players[ent->client->resp.suitcolor]++;
                                ta_players++;
                                survivor = ent;
                        }
                        else if (!ent->RA_quepos)
                        {       // Give player time to get in line.
                                if (ent->stall_time > level.time)
                                        return;
                        }
		}
        }

        // Then count number of teams w/ active players.
        // PM:  If teams are not enabled, then each player is a team.
        if (ab_teams < 2)
        {
                if (numclients < 2)
                        return;   // Not enough players to start a game.
                if (ta_players >= 2)
                        return;   // 2 or more players are left, no winners.
                if (!ta_players && (RA_PlayerQue->RA_nextinque == NULL))
                        return;   // Empty arena?  Need 2+ players in line.
        }
        else
        {
                // Count number of teams and number of players in teams.
                numclients = 0;
                for (t=1; t<MAX_AB_SKINS; t++)
                {       if (RA_all[t])
                        {       numclients += RA_all[t];
                                RA_teams++;
                        }
                }

                if (numclients < 2)
                        return;   // Not enough players to start a game.

                if (RA_teams < 2)
                        return;   // Not enough teams to start a game.

                // Now count all the teams left in the arena.
                RA_teams = RA_players[0];
                for (t=1; t<MAX_AB_SKINS; t++)
                {       if (RA_players[t])
                        {       winteam = t;
                                RA_teams++;
                        }
                }

                // Don't continue if two or more sides are left.
                if (RA_teams >= 2)
                        return;

                // Check if enough players/teams are in que.
                ent = RA_PlayerQue;
                team1 = winteam;
                if (!team1)
                {       // If no survivors, get team first in que.
                        team1 = ent->client->resp.suitcolor;
                        ent = ent->RA_nextinque;
                        // Must have at least two teams in line.
                        if (!ent)
                                return;
                }
                // Normally, the team first in que is placed into the
                // arena to battle the surviving team.  If those first
                // in que are part of the surviving team, look for the
                // team second in que and put them into the arena.
                if (ent->client->resp.suitcolor == team1)
                {       // First team in que already has some players in arena.
                        ent = ent->RA_nextinque;
                        while (ent)
                        {       // Look for team second in que.
                                if (ent->client->resp.suitcolor != team1)
                                {       // Found it!  Done looking.
                                        team2 = ent->client->resp.suitcolor;
                                        break;
                                }
                                ent = ent->RA_nextinque;
                        }
                        // Must have two teams to have a game.
                        if (!team2)
                                return;
                }
                else
                        team2 = ent->client->resp.suitcolor;
        }

        // Only one (or no) side is left, start a new match if possible.
        // If elimination, award bonus points to survivors.
        Coven_KillInfection ();
        game_nofire = level.time + countdown;
        if (ab_rules == RULES_ELIMINATION)
                Coven_GiveEliminationBonus (survivor, winteam, RA_players[winteam]);
        Coven_PrepNextRAMatch (ta_players, team1, team2);
}


/*=======================================================================*/
// CAPTURE RULES

// Reset and redistribute powerups for all players.
void Coven_ResetRunes (edict_t *ent, int runes)
{
        int     index;
        float   rnd;

        // PM:  Reset max health.
        ent->max_health = DEFAULT_MAXHEALTH;

        // Take away all the old runes.
        ent->client->pers.inventory[ITEM_INDEX(FindItem("Armour Rune"))] = 0;
        ent->client->pers.inventory[ITEM_INDEX(FindItem("Damage Rune"))] = 0;
        ent->client->pers.inventory[ITEM_INDEX(FindItem("Health Rune"))] = 0;

        // Don't give runes to inactive players.
        if (ent->deadflag || (ent->client->resp.spectator && !ent->RA_quepos) )
                return;

        // Choose randomly one of three runes.
        if ((rnd = random()) > 0.6666)
                index = ITEM_INDEX(FindItem("Armour Rune"));
        else if (rnd > 0.3333)
                index = ITEM_INDEX(FindItem("Damage Rune"));
        else
        {
                index = ITEM_INDEX(FindItem("Health Rune"));

                // PM:  Add health.
                ent->max_health += (rune_scale * runes);
                if ((ent->health < ent->max_health) && (!ent->deadflag))
                {       ent->health += (rune_scale * runes);
                        if (ent->health > ent->max_health)
                                ent->health = ent->max_health;
                }
        }

        ent->client->pers.inventory[index] = runes;
}

/*
=================
PrepNextCAMatch
=================
*/
void Coven_PrepNextCAMatch
(edict_t *survivor, int winteam, int winners, int mortal)
{
        edict_t *ent;
        int     maxbonus = capture_bonus;
        int     bonus;   // Capture bonus.
        int     runes;   // # of runes players start with.
        int     t;

        // Show who won.
        if (ab_teams < 2)
        {       // Show everyone who captured all the runes.
                if (survivor)
                        gi.bprintf (PRINT_HIGH, "%s captured all the runes.\n", survivor->client->pers.netname);
        }
        else
        {       // Show everyone which team won capture bonus.
                if (winteam && winners)
                {       // Team bonus is divided among members with runes.
                        gi.bprintf (PRINT_HIGH, "%s team captured all the runes.\n", Coven_TeamName(winteam));
                        maxbonus = maxbonus / winners;
                        if (maxbonus < 1)
                                maxbonus = 1;
                }
        }

        // PM:  Reset next capture bonus.
        capture_bonus = mortal;

        // Don't allow division by zero.  (Shouldn't happen anyway.)
        if (mortal < 1)
                mortal = 1;

        // PM:  Calculate number of runes players start with.
        runes = AB_maxrunes->value / mortal;
        if (runes < 1)
                runes = 1;

        // Award points then reset runes.
        for (t=0; t<maxclients->value; t++)
        {       if (g_edicts[t+1].inuse && g_edicts[t+1].client)
                {       ent = &g_edicts[t+1];
                        // Award points to players with powerups.
                        bonus = (ent->client->pers.inventory[ITEM_INDEX(FindItem("Armour Rune"))]) +
                                (ent->client->pers.inventory[ITEM_INDEX(FindItem("Damage Rune"))]) +
                                (ent->client->pers.inventory[ITEM_INDEX(FindItem("Health Rune"))]);
                        if (bonus)
                        {
                                // Give points equal to no. of players
                                // or all held runes whichever is lower.
                                if (bonus > maxbonus)
                                        bonus = maxbonus;

                                ent->client->resp.score += bonus;
                                gi.cprintf (ent, PRINT_HIGH, "Capture bonus:  %d\n", bonus);
                        }

                        // Reset and redistribute powerups for all players.
                        Coven_ResetRunes (ent, runes);
                }
        }
}

/*
=================
CheckCARules
=================
*/
void Coven_CheckCARules (void)
{
        edict_t *ent;
        edict_t *survivor;
        int     t, numclients = 0, ta_players = 0;
        int     CA_players[MAX_AB_SKINS], CA_teams = 0;
        int     CA_mortal[MAX_AB_SKINS], mortal = 0;
        int     CA_all[MAX_AB_SKINS];
        int     winteam = 0;

        if (ab_rules != RULES_CAPTURE)
		return;

        // Can't capture if game has not started yet.
        if (game_nofire > level.time)
                return;

        for (t = 0; t < MAX_AB_SKINS; t++)
        {       CA_all[t] = 0;
                CA_mortal[t] = 0;
		CA_players[t] = 0;
        }
        survivor = NULL;

        // Count the number of players who have runes.
        // Also count every player in the game.
        for (t=0; t<maxclients->value; t++)
        {       if (g_edicts[t+1].inuse && g_edicts[t+1].client)
                {       numclients++;
			ent = &g_edicts[t+1];
                        CA_all[ent->client->resp.suitcolor]++;
                        if (ent->client->pers.inventory[ITEM_INDEX(FindItem("Armour Rune"))] ||
                                ent->client->pers.inventory[ITEM_INDEX(FindItem("Damage Rune"))] ||
                                ent->client->pers.inventory[ITEM_INDEX(FindItem("Health Rune"))])
                        {       // Player has runes.
                                CA_players[ent->client->resp.suitcolor]++;
                                ta_players++;
                                survivor = ent;
			}

                        if (!ent->deadflag && (!ent->client->resp.spectator || ent->RA_quepos))
                        {       // Player is in game or waiting in line.
                                CA_mortal[ent->client->resp.suitcolor]++;
                                mortal++;
                        }
                        else if (ent->stall_time > level.time)
                        {       // A player is not ready.  Don't check rules!
                                return;
                        }
		}
	}

        // Count number of teams w/ active players (players w/ runes)
        // If not enough teams, then use total number of players.
        if (ab_teams < 2)
        {
                if (mortal < 2)
                        return;   // Not enough players alive to start a game.
		CA_teams = ta_players;
        }
        else
        {
                // Count number of teams and number of players in teams.
                mortal = numclients = 0;
                for (t=1; t<MAX_AB_SKINS; t++)
                {       if (CA_all[t])
                                numclients += CA_all[t];
                        if (CA_mortal[t])
                        {       mortal += CA_mortal[t];
                                CA_teams++;
                        }
                }

                if (mortal < 2)
                        return;   // Not enough players alive to start a game.

                if (CA_teams < 2)
                        return;   // Not enough active teams to start a game.

                // Now count number of teams with runes.
                CA_teams = CA_players[0];
                for (t=1; t<MAX_AB_SKINS; t++)
                {       if (CA_players[t])
                        {       winteam = t;
				CA_teams++;
                        }
                }
	}

        // Don't continue if two or more sides have runes.
        if (CA_teams >= 2)
                return;

        // Check for unclaimed "runes".  As long as free runes
        // exist, the game will continue to play.  i.e., players MUST
        // collect all runes in order to win.
        ent = G_Find(NULL, FOFS(classname), "capture_rune");
        if (ent)
                return;

        // We have a winner?!  Cease fire for a moment.
        Coven_KillInfection ();
        game_nofire = level.time + countdown;
        Coven_PrepNextCAMatch (survivor, winteam, CA_players[winteam], mortal);
}


/*=======================================================================*/
// RUNES

/*----------------------------------------------------/ New Code /--------//
//  Player touches and picks up a rune.
//------------------------------------------------------------------------*/
qboolean Coven_PickupRune (edict_t *ent, edict_t *other)
{
        int     runes;

// Pick up armour rune?
        if (ent->CA_apups)
        {       runes = MAX_CLIENTS - other->client->pers.inventory[ITEM_INDEX(FindItem("Armour Rune"))];
                if (runes < 0)  runes = 0;
                if (runes > ent->CA_apups)  runes = ent->CA_apups;
                other->client->pers.inventory[ITEM_INDEX(FindItem("Armour Rune"))] += runes;
        }

// Pick up damage rune?
        if (ent->CA_dpups)
        {       runes = MAX_CLIENTS - other->client->pers.inventory[ITEM_INDEX(FindItem("Damage Rune"))];
                if (runes < 0)  runes = 0;
                if (runes > ent->CA_dpups)  runes = ent->CA_dpups;
                other->client->pers.inventory[ITEM_INDEX(FindItem("Damage Rune"))] += runes;
        }

// Pick up health rune?
        if (ent->CA_hpups)
        {       int     cure = ent->CA_hpups * rune_scale;

                runes = MAX_CLIENTS - other->client->pers.inventory[ITEM_INDEX(FindItem("Health Rune"))];
                if (runes < 0)  runes = 0;
                if (runes > ent->CA_hpups)  runes = ent->CA_hpups;
                other->client->pers.inventory[ITEM_INDEX(FindItem("Health Rune"))] += runes;

                // Boost health values if possible.
                if (runes)
                        other->max_health += runes * rune_scale;
                if (other->health < other->max_health)
                {       other->health += cure;
                        if (other->health > other->max_health)
                                other->health = other->max_health;
                }

                // Extinguish (Napalm2) flames burning on player.
                if (cure >= 10)
                        other->burnout = 0;
        }

// Can always pickup rune, even if worthless.
        return true;
}

/*----------------------------------------------------/ New Code /--------//
//  This animates the atom model.  Atom is used by health rune.
//------------------------------------------------------------------------*/
void Coven_HealthOrbit (edict_t *self)
{
	if (self->timestamp < level.time)
        {       G_FreeEdict (self);
                return;
        }

        if (++self->s.frame > 31)
                self->s.frame = 0;

	self->nextthink = level.time + FRAMETIME;
}

/*----------------------------------------------------/ New Code /--------//
//  This is a temporary function that animates the health rune.
//------------------------------------------------------------------------*/
static void Coven_TempHealthOrbit (edict_t *self)
{
        if (++self->s.frame > 31)
                self->s.frame = 0;

	self->nextthink = level.time + FRAMETIME;

        if (self->delay < level.time)
        {       self->touch      = Touch_Item;
                self->think      = Coven_HealthOrbit;
        }
}

/*----------------------------------------------------/ New Code /--------//
//  This is a temporary function that lets anyone except the
//  rune's owner pickup the rune.
//------------------------------------------------------------------------*/
static void Coven_DropTempRuneTouch
(edict_t *self, edict_t *other, cplane_t *plane, csurface_t *surf)
{
        if (other == self->owner)
		return;

        Touch_Item (self, other, plane, surf);
}

/*----------------------------------------------------/ New Code /--------//
//  This makes the rune touchable by anyone.
//------------------------------------------------------------------------*/
static void Coven_MakeRuneTouchable (edict_t *self)
{
        self->touch     = Touch_Item;
        self->nextthink = self->timestamp;
        self->think     = G_FreeEdict;
}

/*----------------------------------------------------/ New Code /--------//
//  This creates a rune and throws it out at a random velocity.
//------------------------------------------------------------------------*/
void Coven_TossRune (edict_t *self, gitem_t *item, int cnt)
{
        edict_t *rune;

        rune = G_Spawn();
        rune->classname   = item->classname;
        rune->item        = item;
        rune->spawnflags  = DROPPED_ITEM;
        rune->s.effects   = item->world_model_flags;
        rune->s.renderfx  = RF_GLOW;
        VectorSet (rune->mins, -15, -15, -15);
        VectorSet (rune->maxs, 15, 15, 15);
        gi.setmodel (rune, rune->item->world_model);
        rune->s.frame     = 0;
        rune->solid       = SOLID_TRIGGER;
        rune->movetype    = MOVETYPE_TOSS;  
        rune->touch       = Coven_DropTempRuneTouch;
        rune->owner       = self;
        rune->think       = Coven_MakeRuneTouchable;
        rune->nextthink   = level.time + 1;
        rune->timestamp   = level.time + 60 + (cnt * 0.5);

        VectorCopy (self->s.origin, rune->s.origin);
        rune->velocity[0] = (random() * 400 - 200);
        rune->velocity[1] = (random() * 400 - 200);
        rune->velocity[2] = 400 + (random() * 200);
        rune->CA_hpups    = 0;
        rune->CA_apups    = 0;
        rune->CA_dpups    = 0;

        if (!stricmp(item->pickup_name, "Armour Rune"))
                rune->CA_apups += cnt;
        else if (!stricmp(item->pickup_name, "Damage Rune"))
                rune->CA_dpups += cnt;
        else //if (!stricmp(item->pickup_name, "Health Rune"))
        {       rune->CA_hpups += cnt;
                rune->s.frame   = rand() % 32;
                rune->delay     = level.time + 1;
                rune->nextthink = level.time + FRAMETIME;
                rune->think     = Coven_TempHealthOrbit;
        }

        if (cnt >= 50)
                rune->s.skinnum  = 5;
        else if (cnt >= 20)
                rune->s.skinnum  = 4;
        else if (cnt >= 10)
                rune->s.skinnum  = 3;
        else if (cnt >= 5)
                rune->s.skinnum  = 2;
        else if (cnt >= 2)
                rune->s.skinnum  = 1;
        else
                rune->s.skinnum  = 0;

        gi.linkentity (rune);
}

/*----------------------------------------------------/ New Code /--------//
//  This returns how powerful the rune should be based on 'cnt'.
//------------------------------------------------------------------------*/
int Coven_RuneCount (int cnt)
{
        if (cnt > 50)  return 50;
        if (cnt > 20)  return 20;
        if (cnt > 10)  return 10;
        if (cnt > 5)   return 5;
        if (cnt > 2)   return 2;
        return 1;
}

/*----------------------------------------------------/ New Code /--------//
//  This causes the player to drop all runes.
//------------------------------------------------------------------------*/
void Coven_DropAllRunes (edict_t *self)
{
        gitem_t *item;
        int     index, n;

// Nothing to drop if not in capture mode.
        if (!deathmatch->value || (ab_rules != RULES_CAPTURE))
		return;

// Don't spawn runes during intermission.
        if (level.intermissiontime)
        {       self->client->pers.inventory[ITEM_INDEX(FindItem("Armour Rune"))] = 0;
                self->client->pers.inventory[ITEM_INDEX(FindItem("Damage Rune"))] = 0;
                self->client->pers.inventory[ITEM_INDEX(FindItem("Health Rune"))] = 0;
                return;
        }

// Drop armour runes.
        item = FindItem ("Armour Rune");
        index = ITEM_INDEX (item);
        while (self->client->pers.inventory[index] > 0)
        {       n = Coven_RuneCount (self->client->pers.inventory[index]);
                Coven_TossRune (self, item, n);
                self->client->pers.inventory[index] -= n;
        }

// Drop damage runes.
        item = FindItem ("Damage Rune");
        index = ITEM_INDEX (item);
        while (self->client->pers.inventory[index] > 0)
        {       n = Coven_RuneCount (self->client->pers.inventory[index]);
                Coven_TossRune (self, item, n);
                self->client->pers.inventory[index] -= n;
        }

// Drop health runes.
        item = FindItem ("Health Rune");
        index = ITEM_INDEX (item);
        while (self->client->pers.inventory[index] > 0)
        {       n = Coven_RuneCount (self->client->pers.inventory[index]);
                Coven_TossRune (self, item, n);
                self->client->pers.inventory[index] -= n;
                self->max_health = DEFAULT_MAXHEALTH + (self->client->pers.inventory[index] * rune_scale);
        }
}


/*=======================================================================*/
// MISCELLANEOUS

/*----------------------------------------------------/ New Code /--------//
//  This adjusts the damage based on runes possessed by attacker and
//  target.
//------------------------------------------------------------------------*/
int Coven_ApplyCaptureRunes
(edict_t *targ, edict_t *attacker, int damage, int dflags)
{
        if (deathmatch->value && (ab_rules == RULES_CAPTURE))
        {
                int     n;
                float   runepower = rune_scale * 0.01;

                if (attacker->client)
                {       // Increase damage for attacker's damage powerups.
                        n = attacker->client->pers.inventory[ITEM_INDEX(FindItem("Damage Rune"))];

                        if (n > 0)
                        {
                                float   muscle = ((n*runepower)+1);

                                damage = (ceil(damage * muscle));
                        }
                }
                if (targ->client)
                {       // Reduce damage for target's armor powerups.
                        n = targ->client->pers.inventory[ITEM_INDEX(FindItem("Armour Rune"))];

                        if (dflags & DAMAGE_NO_PROTECTION)
                                n = 0;

                        if (n > 0)
                        {
                                float   armour = ((n*runepower)+1);

                                if (damage > 1)
                                {       damage = floor(damage / armour);
                                        if (damage < 1)
                                                damage = 1;
                                }
                        }
                }
        }

        return damage;
}


/*===========================/  END OF FILE  /===========================*/
